/*                                                                            *
 *   This file is part of the ESPRESSO Pipeline                               *
 *   Copyright (C) 2006 European Southern Observatory                         *
 *                                                                            *
 *   This library is free software; you can redistribute it and/or modify     *
 *   it under the terms of the GNU General Public License as published by     *
 *   the Free Software Foundation; either version 2 of the License, or        *
 *   (at your option) any later version.                                      *
 *                                                                            *
 *   This program is distributed in the hope that it will be useful,          *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 *   GNU General Public License for more details.                             *
 *                                                                            *
 *   You should have received a copy of the GNU General Public License        *
 *   along with this program; if not, write to the Free Software              *
 *   Foundation, 51 Franklin St, Fifth Floor, Boston, MA  02111-1307  USA     *
 *                                                                            */

/*
 * $Author: dsosnows $
 * $Date: 2013-08-30 15:13:41 $
 * $Revision: 1.3 $
 * $Name: not supported by cvs2svn $
 */


#include <espdr_dark.h>
#include <espdr_overscan.h>
#include <espdr_hdrl_func.h>
/*----------------------------------------------------------------------------
 Functions code
 ----------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/**
 @brief     Init recipe structires
 @param
 @return    CPL_ERROR_NONE iff OK
 */
/*---------------------------------------------------------------------------*/
static cpl_error_code espdr_mdark_init_structures(cpl_parameterlist *parameters,
                                                  cpl_frameset *frameset,
												  const char* recipe_id,
                                                  espdr_OVSC_param **OVSC_param,
                                                  espdr_CCD_geometry **CCD_geom,
                                                  espdr_inst_config **inst_config,
                                                  espdr_DARK_param **DARK_param) {

	/* AMO Changed way to read CCD geom frame and to init relevant recipe
	 * internal parameters structures to simplify this calling function
	 */

	cpl_frame* CCD_geom_frame = espdr_frame_find(frameset, ESPDR_CCD_GEOM);

	*CCD_geom = espdr_CCD_geom_init(parameters, CCD_geom_frame);

	cpl_frame* inst_config_frame = espdr_get_inst_config(frameset);

	*inst_config = espdr_inst_config_init(parameters,
                                          (*CCD_geom)->ext_nb, inst_config_frame);

    *OVSC_param = espdr_parameters_OVSC_init(recipe_id, parameters);

	*DARK_param = espdr_DARK_param_init(recipe_id, parameters);

	return (cpl_error_get_code());
}
/*---------------------------------------------------------------------------*/
/**
 @brief     Prepare input frames
 @param
 @return    CPL_ERROR_NONE iff OK
 */
/*---------------------------------------------------------------------------*/
static cpl_error_code espdr_mdark_prepare_input_frames(cpl_frameset *set,
                                                       espdr_inst_config *inst_config,
                                                       cpl_frameset **dark_frames,
                                                       cpl_frameset **all_mbias_frames,
                                                       cpl_frameset** used_frames) {

	cpl_error_code my_error;
	/* Extract the raw dark frames and put them into dark_frames frameset */
	/* AMO: there is no need to have a frameset to load a frame */
	*dark_frames = cpl_frameset_new();
	my_error = espdr_frame_extract_by_tag(set, ESPDR_DARK_RAW, *dark_frames);

	espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
			"DARK frames extraction failed.");

	*used_frames = cpl_frameset_duplicate(*dark_frames);

	cpl_frame* CCD_geom_frame = espdr_frame_find(set, ESPDR_CCD_GEOM);
	my_error = cpl_frameset_insert(*used_frames, cpl_frame_duplicate(CCD_geom_frame));
	cpl_frame* inst_config_frame = espdr_get_inst_config(set);
	my_error = cpl_frameset_insert(*used_frames, cpl_frame_duplicate(inst_config_frame));

	double tolerance = 0.1;
	int same_Texp_flag = 1;

    my_error = espdr_verify_texp(*dark_frames, inst_config->Texp_kw,
                                 tolerance, &same_Texp_flag);

    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_verify_texp failed: %s",
                 cpl_error_get_message_default(my_error));

	if (same_Texp_flag != 1) {
		/* free memory */
		espdr_parameters_inst_config_delete(inst_config);
		cpl_frameset_delete(*dark_frames);
		cpl_frameset_delete(*used_frames);
	}

	espdr_ensure(same_Texp_flag != 1, CPL_ERROR_ILLEGAL_INPUT,
			"Dark frames don't have the same exposure time, exiting");

    *all_mbias_frames = cpl_frameset_new();
    if (inst_config->raw_bias_limit_nb > 0) {
        my_error = espdr_frame_extract_by_tag(set, ESPDR_PRO_CATG_MBIAS_RES,
                                              *all_mbias_frames);

        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "MASTER BIAS RES frame extraction failed");

        espdr_ensure(cpl_frameset_get_size(*all_mbias_frames) < 1,
                     CPL_ERROR_ILLEGAL_INPUT,
                     "There should be at least one master bias residuals frame available");

        my_error = cpl_frameset_insert(*used_frames,
                    cpl_frame_duplicate(cpl_frameset_get_position(*all_mbias_frames, 0)));
    }

	return (cpl_error_get_code());
}

/*---------------------------------------------------------------------------*/
/**
 *  @brief     Cleanup the frameset structure
 *   @param     set     frameset structure to be cleaned
 *    @param     nb      number of framesets in the structure
 *     @return   CPL_ERROR_NONE iff OK
 *      */
/*---------------------------------------------------------------------------*/
//TODO: Why is this needed??
cpl_error_code espdr_dark_cleanup(cpl_frameset* set) {

    const char* fname = NULL;
    cpl_frame* frame = NULL;
    int size = 0;
    char cmd[80];
    size=cpl_frameset_get_size(set);
    for (int i = 0; i < size; i++) {
        frame = cpl_frameset_get_position(set, i);
        fname = cpl_frame_get_filename(frame);
        sprintf(cmd, "rm  %s", fname);
        system(cmd);
    }

    return (cpl_error_get_code());
}



/*---------------------------------------------------------------------------*/
/**
 @brief     Add extra QC keywords
 @param     keywords    property list to add the extra KWs
 @param     ext_nb      number of extensions
 @param     kext        property list of the extensions (?)
 @return    0 if everything is ok
 */
/*---------------------------------------------------------------------------*/
static cpl_error_code espdr_add_extra_qc(cpl_propertylist* keywords,
                                         const int ext_nb,
                                         cpl_propertylist*** kext) {

    const char* ksuff[2] = {"", ""};
    cpl_propertylist_append_int(keywords,"ESO QC PROC VERSION", 2);
    espdr_add_qc_key_stat_ext(keywords, ext_nb, "DARK MEAN", ksuff, CPL_TRUE, CPL_TRUE,
                              CPL_FALSE, CPL_FALSE, CPL_FALSE, kext);
    espdr_add_qc_key_stat_ext(keywords, ext_nb, "HOTPIX NB", ksuff, CPL_TRUE, CPL_TRUE,
                              CPL_FALSE, CPL_FALSE, CPL_TRUE, kext);
    espdr_copy_qc_ext_key_to_ext(keywords, kext, ext_nb);

    return cpl_error_get_code();
}

/*---------------------------------------------------------------------------*/
/**
 @brief
 @param
 @return    CPL_ERROR_NONE iff OK
 */
/*---------------------------------------------------------------------------*/
static cpl_error_code espdr_mdark_save_products(cpl_imagelist *hot_pix_mask,
                                                cpl_imagelist *master_dark,
                                                cpl_imagelist *master_dark_pxl_nb,
												const char* recipe_id,
                                                cpl_frameset *frameset,
                                                cpl_parameterlist *parameters,
                                                cpl_frameset *used_frames,
                                                cpl_frameset* dark_frames,
                                                espdr_inst_config *inst_config,
                                                espdr_CCD_geometry *CCD_geom,
                                                cpl_propertylist* keywords) {

	cpl_error_code my_error;

    my_error = cpl_propertylist_update_string(keywords, PRO_CATG_KW,
                                              ESPDR_PRO_CATG_HOT_PIXELS);
    if (my_error != CPL_ERROR_NONE) {
        espdr_msg_error("Error adding product category is %s",
                        cpl_error_get_message());
    }

	/* keywords_ext contains keywords of all the extension headers
	 of the first input frame */
	cpl_propertylist **keywords_ext =
	(cpl_propertylist**)cpl_malloc(CCD_geom->ext_nb * sizeof(cpl_propertylist*));
	const char* input_filename =
	    		cpl_frame_get_filename(cpl_frameset_get_position(dark_frames, 0));
	int i = 0;
	for (i = 0; i < CCD_geom->ext_nb; i++) {
		keywords_ext[i] = cpl_propertylist_load(input_filename, i+1);
	}

	/* Save the HOT PIXELS fits frame */
	if (strcmp(inst_config->instrument, "HARPN") != 0 ) {
        my_error = espdr_add_extra_qc(keywords, CCD_geom->ext_nb, &keywords_ext);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_add_extra_qc failed: %s ",
                     cpl_error_get_message_default(my_error));
	}

    // CORALIE DARKS have ESO OBS TEXP instead of EXPTIME KW
    if (!cpl_propertylist_has(keywords, "EXPTIME")) {
        if (cpl_propertylist_has(keywords, "ESO OBS TEXP")) {
            double texp = cpl_propertylist_get_double(keywords, "ESO OBS TEXP");
            cpl_propertylist_update_double(keywords, "EXPTIME", texp);
        }
    }

    my_error = espdr_dfs_image_save(frameset, parameters, used_frames,
                                    recipe_id, keywords, keywords_ext,
                                    inst_config->hot_pixels_filename,
                                    hot_pix_mask, CPL_TYPE_USHORT, CCD_geom);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_dfs_image_save failed: %s for file %s",
                 cpl_error_get_message_default(my_error),
                 inst_config->hot_pixels_filename);

	/* Change the PRO.CATG KW to the master DARK and save the same FITS
	 * header as hot pixels */
    my_error = cpl_propertylist_set_string(keywords, PRO_CATG_KW,
                                           ESPDR_PRO_CATG_MDARK);

	/* Save the Master DARK fits frame */
    //my_error = espdr_dfs_image_save(frameset, parameters, used_frames,
    //                                recipe_id, keywords, keywords_ext,
    //                                inst_config->master_dark_filename,
    //                                master_dark, CPL_TYPE_FLOAT, CCD_geom);
    my_error = espdr_dfs_image_save_with_pxl_nb(frameset, parameters, used_frames,
                                                recipe_id, keywords, keywords_ext,
                                                inst_config->master_dark_filename,
                                                master_dark, master_dark_pxl_nb,
                                                CPL_TYPE_FLOAT, CCD_geom);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_dfs_image_save failed: %s for file %s",
                 cpl_error_get_message_default(my_error),
                 inst_config->master_dark_filename);

#if SAVE_STATIC_PRODUCT_DARK
    my_error = cpl_propertylist_update_double(keywords, "MJD-OBS", 52518.0);
    my_error = cpl_propertylist_update_string(keywords, PRO_CATG_KW,
                                              ESPDR_PRO_CATG_HOT_PIXELS);
    char *ins_mode = NULL;
    if (cpl_propertylist_has(keywords, "ESO INS MODE")) {
        ins_mode = cpl_propertylist_get_string(keywords, "ESO INS MODE");
    } else {
        ins_mode = (char *)cpl_malloc(sizeof(char));
        ins_mode[1] = '\0';
    }
    char static_filename[64];
    sprintf(static_filename, "%s_%s_hot_pixels_2002-09-01.fits",
            inst_config->instrument, ins_mode);
    
    // This code is to produce the hot pixels mask larger by 1 pixel in both dimensions for early CORALIE frames from 1998-11-01 to 2002-09-01
    //cpl_image *hot_pix_mask_2048 = cpl_image_new(2148, 2048, CPL_TYPE_INT);
    //cpl_image_copy(hot_pix_mask_2048, cpl_imagelist_get(hot_pix_mask, 0), 2, 1);
    //cpl_imagelist *hot_pix_iml_2048 = cpl_imagelist_new();
    //cpl_imagelist_set(hot_pix_iml_2048, hot_pix_mask_2048, 0);
    
    my_error = espdr_dfs_image_save(frameset, parameters, used_frames,
                                    recipe_id, keywords, keywords_ext,
                                    static_filename,
                                    hot_pix_mask, CPL_TYPE_USHORT, CCD_geom);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_dfs_image_save failed: %s for file %s",
                 cpl_error_get_message_default(my_error), static_filename);

    //my_error = cpl_propertylist_save(keywords, static_filename, CPL_IO_CREATE);
    //for (int i = 0; i < CCD_geom->ext_nb; i++) {
    //    my_error = cpl_image_save(cpl_imagelist_get(hot_pix_mask, i), static_filename,
    //                              CPL_TYPE_USHORT, keywords_ext[i], CPL_IO_EXTEND);
    //}
    //my_error = espdr_dfs_image_save(frameset, parameters, used_frames,
    //                                RECIPE_ID, keywords, keywords_ext,
    //                                static_filename, hot_pix_mask,
    //                                CPL_TYPE_USHORT, CCD_geom);
    //espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
    //             "cpl_image_save failed: %s for file %s",
    //             cpl_error_get_message_default(my_error),
    //             static_filename);

    my_error = cpl_propertylist_update_string(keywords, PRO_CATG_KW,
                                              ESPDR_PRO_CATG_MDARK);
    sprintf(static_filename, "%s_%s_master_dark_2002-09-01.fits",
            inst_config->instrument, ins_mode);
    
    my_error = espdr_dfs_image_save_with_pxl_nb(frameset, parameters, used_frames,
                                                recipe_id, keywords, keywords_ext,
                                                static_filename,
                                                master_dark, master_dark_pxl_nb,
                                                CPL_TYPE_FLOAT, CCD_geom);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_dfs_image_save_with_pxl_nb failed: %s for file %s",
                 cpl_error_get_message_default(my_error), static_filename);

    //my_error = cpl_propertylist_save(keywords, static_filename, CPL_IO_CREATE);
    //for (int i = 0; i < CCD_geom->ext_nb; i++) {
    //    my_error = cpl_image_save(cpl_imagelist_get(master_dark, i), static_filename,
    //                              CPL_TYPE_FLOAT, keywords_ext[i], CPL_IO_EXTEND);
    //}
    //my_error = espdr_dfs_image_save(frameset, parameters, used_frames,
    //                                RECIPE_ID, keywords, keywords_ext,
    //                                static_filename, master_dark,
    //                                CPL_TYPE_FLOAT, CCD_geom);
    //espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
    //             "cpl_image_save failed: %s for file %s",
    //             cpl_error_get_message_default(my_error),
    //             static_filename);
#endif
    
	for (int i = 0; i < CCD_geom->ext_nb; i++) {
			cpl_propertylist_delete(keywords_ext[i]);
		}
	cpl_free(keywords_ext);

	return (cpl_error_get_code());
}



/*---------------------------------------------------------------------------*/
/**
 @brief    Interpret the command line options and execute the data processing
 @param    parameters     the parameters list
 @param    frameset   the frames list

 In case of failure the cpl_error_code is set.
 */
/*---------------------------------------------------------------------------*/
int espdr_mdark(cpl_parameterlist *parameters,
                       cpl_frameset *frameset,
					   const char* recipe_id){

	int use_hdrl = 0;
	cpl_error_code my_error = CPL_ERROR_NONE;
	cpl_msg_set_level(CPL_MSG_INFO);
    const int rec_ntags = 3;
    const char* rec_tags[3] = {ESPDR_DARK_RAW, ESPDR_CCD_GEOM,
        ESPDR_PRO_CATG_MBIAS_RES};
    int is_required[3] = {1, 1, 0};

    espdr_msg("Starting dark");
    espdr_ensure(espdr_check_input_tags(frameset,
                                        rec_tags, is_required,
                                        rec_ntags) != CPL_ERROR_NONE,
                 cpl_error_get_code(), "Wrong input tag!");
    espdr_ensure(espdr_check_input_inst_config(frameset) != CPL_ERROR_NONE,
                cpl_error_get_code(), "Wrong input tag!");
	/* Identify the RAW and CALIB frames in the input frameset */
	espdr_ensure(espdr_dfs_set_groups(frameset) != CPL_ERROR_NONE,
                 cpl_error_get_code(),
                 "DFS setting groups failed. Expected inputs are:\n"
                 "raw dark images, tagged as DARK\n"
                 "master bias residuals image, tagged as MASTER_BIAS_RES\n"
                 "CCD geometry table, tagged as CCD_GEOM\n"
                 "instrument configuration table, tagged as INST_CONFIG");


	espdr_OVSC_param *OVSC_param = NULL;
	cpl_frame* CCD_geom_frame = NULL;
	cpl_frame* inst_config_frame = NULL;
	espdr_DARK_param *DARK_param = NULL;
	espdr_inst_config *inst_config = NULL;
	espdr_CCD_geometry *CCD_geom = NULL;
    espdr_mdark_init_structures(parameters, frameset, recipe_id, &OVSC_param,
    		&CCD_geom, &inst_config, &DARK_param);

	cpl_frameset *all_mbias_frames = NULL;
	cpl_frameset *dark_frames = cpl_frameset_new();
	cpl_frameset *used_frames = NULL;
    cpl_frameset *bias_corr_darks = NULL;
	int ndarks;

    my_error = espdr_frame_extract_by_tag(frameset, ESPDR_DARK_RAW, dark_frames);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "DARK frames extraction failed.");

    ndarks = cpl_frameset_get_size(dark_frames);
    espdr_msg("The input frameset contains %d DARK frames", ndarks);
    if (ndarks < inst_config->raw_dark_limit_nb) {
        espdr_msg_warning("The number of input raw DARK frames is not enough (%d), should be minimum %d, exiting.",
                          ndarks, inst_config->raw_dark_limit_nb);
        cpl_frameset_delete(dark_frames);
        espdr_parameters_OVSC_delete(OVSC_param);
        espdr_parameters_DARK_delete(DARK_param);
        espdr_parameters_CCD_geometry_delete(CCD_geom);
        espdr_parameters_inst_config_delete(inst_config);
        cpl_frame_delete(CCD_geom_frame);
        cpl_frame_delete(inst_config_frame);
        return(CPL_ERROR_ILLEGAL_INPUT);
    }

    my_error = espdr_mdark_prepare_input_frames(frameset, inst_config, &dark_frames,
                                                &all_mbias_frames, &used_frames);

    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_mdark_prepare_input_frames failed: %s",
                 cpl_error_get_message_default(my_error));

    /* Extract orders masking frame */
    cpl_frame* orders_mask_frame = espdr_frame_find(frameset,
                                                    ESPDR_ORDERS_MASK);
    if (orders_mask_frame != NULL) {
        my_error = cpl_frameset_insert(used_frames,
                                       cpl_frame_duplicate(orders_mask_frame));
    }
    cpl_imagelist *orders_mask_list = NULL;
    /* Extract all the outputs from the orders masks frame if provided */
    if (orders_mask_frame != NULL) {
        orders_mask_list = cpl_imagelist_new();
        my_error = espdr_extract_extensions(orders_mask_frame,
                                            CPL_TYPE_INT,
                                            CCD_geom->ext_nb,
                                            &orders_mask_list);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_extract_real_outputs failed: %s",
                     cpl_error_get_message_default(my_error));
    }

    /* Filling up the DRS QC KWs structure */
    espdr_qc_keywords *qc_kws = NULL;
    qc_kws = (espdr_qc_keywords *)cpl_malloc( sizeof(espdr_qc_keywords));
    my_error = espdr_fill_qc_keywords(inst_config, qc_kws);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Filling QC KWs failed: %s",
                 cpl_error_get_message_default(my_error));
    espdr_ensure(qc_kws == NULL, CPL_ERROR_ILLEGAL_OUTPUT,
                 "QC KWs structure is NULL");

    if (inst_config->inst_type == NIR) {

        espdr_msg("Preprocessing");

        my_error = espdr_preprocess(dark_frames,
                                    orders_mask_list,
                                    CCD_geom, inst_config,
                                    &bias_corr_darks);

        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "preprocesing failed: %s",
                     cpl_error_get_message_default(my_error));

    } else {

        // The bias/overscan correction is done only for visual spectrographs
        /* keywords contains all the keywords read from
         the first input frame header */
        espdr_msg("size mbias_frames = %lld",
                  cpl_frameset_get_size(all_mbias_frames));

        espdr_msg("BIAS correction on %d DARKS input frames (%d extensions each)",
                  ndarks, CCD_geom->ext_nb);

        /* AMO: The following may be implemented using less RAM */
        int remove_bias_res_flag = 1;
        if (inst_config->raw_bias_limit_nb == 0) remove_bias_res_flag = 0;
        my_error = espdr_correct_bias_frames(dark_frames, all_mbias_frames,
                                             CCD_geom, inst_config,
                                             remove_bias_res_flag,
                                             &bias_corr_darks, use_hdrl);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error, "Bias correction failed");
    }
    
    espdr_msg("Taking off cosmics from all the darks via sigma clipping");
    int total_output_nb = CCD_geom->total_output_nb;

    double RON_DARK[total_output_nb];
    double mean_DARK[total_output_nb];
    int total_cosmics[CCD_geom->ext_nb];
    /* AMO: The following may be implemented using less RAM */
    cpl_imagelist *master_dark_list = cpl_imagelist_new();
    cpl_imagelist *master_dark_pxl_nb_list = cpl_imagelist_new();
    /*
    my_error = espdr_stack_sigma(bias_corr_darks,
                                 CCD_geom, inst_config->dark_ksigma,
                                 inst_config->dark_sigma_clipping_method,
                                 ndarks-1, &master_dark_list,
                                 mean_DARK, RON_DARK,
                                 total_cosmics,use_hdrl);
     */
    my_error = espdr_stack_sigma_with_pxl_nb(bias_corr_darks,
                                             CCD_geom, inst_config->dark_ksigma,
                                             inst_config->dark_sigma_clipping_method,
                                             ndarks-1, &master_dark_list,
                                             &master_dark_pxl_nb_list,
                                             mean_DARK, RON_DARK,
                                             total_cosmics,use_hdrl);
    
    espdr_dark_cleanup(bias_corr_darks);
    cpl_frameset_delete(bias_corr_darks);

    int master_dark_size = cpl_imagelist_get_size(master_dark_list);
    int master_dark_pxl_nb_size = cpl_imagelist_get_size(master_dark_pxl_nb_list);
    espdr_msg("master dark size = %d, master dark pxl nb size = %d",
              master_dark_size, master_dark_pxl_nb_size);

	espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
			   "Master dark image computation failed");
    
    espdr_msg("Master DARK created");
    for (int i = 0; i < CCD_geom->ext_nb; i++) {
        espdr_msg("cosmics nb for ext %d: %d", i, total_cosmics[i]);
	}

    double exposure_time = 0.0;
    const char* input_filename =
    		cpl_frame_get_filename(cpl_frameset_get_position(dark_frames, 0));
    /* Load keywords from the first image (primary header headers) */
    cpl_propertylist* keywords = cpl_propertylist_load(input_filename, 0);
    cpl_propertylist **keywords_ext =
    		(cpl_propertylist**)cpl_malloc(CCD_geom->ext_nb*sizeof(cpl_propertylist*));
    for (int i = 0; i < CCD_geom->ext_nb; i++) {
           keywords_ext[i] = cpl_propertylist_load(input_filename, i+1);
       }
	exposure_time = cpl_propertylist_get_double(keywords, inst_config->Texp_kw);
	espdr_msg_debug("exposure time = %lf", exposure_time);

    int recipes[10] = {1, 0, 0, 0, 0, 0, 0, 0, 0, 0};
    my_error = espdr_compute_calibrations_intervals(frameset, inst_config, keywords,
                                                    NULL, NULL, NULL, "DARK", recipes);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_compute_calibrations_intervals failed: %s",
                 cpl_error_get_message_default(my_error));
    
	my_error = espdr_cosmics_rates_prepare(total_cosmics, CCD_geom, qc_kws,
                                           ndarks, exposure_time, &keywords);
	/* AMO: The following may be implemented using less RAM */
	cpl_imagelist *hot_pixels_list = cpl_imagelist_new();
	my_error = espdr_hot_pixels_prepare(master_dark_list, CCD_geom,
                                        qc_kws, inst_config, exposure_time, RON_DARK,
                                        &hot_pixels_list, &keywords, use_hdrl);
	espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Hot pixels detection failed");

	/* Merge all master dark output images into one extension,
	 all the merged extensions are in the master_dark */
	espdr_msg("Merging the MASTER DARK image");
	cpl_imagelist *master_dark = cpl_imagelist_new();
    my_error = espdr_image_merge_2_dim(master_dark_list, CCD_geom,
                                       CPL_TYPE_DOUBLE, &master_dark);
    cpl_imagelist_delete(master_dark_list);

    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_image_merge_2_dim on master dark failed: %s",
                 cpl_error_get_message_default(my_error));

    espdr_msg("Merging the MASTER DARK PXL NB image");
    cpl_imagelist *master_dark_pxl_nb = cpl_imagelist_new();
    my_error = espdr_image_merge_2_dim(master_dark_pxl_nb_list, CCD_geom,
                                       CPL_TYPE_INT, &master_dark_pxl_nb);
    cpl_imagelist_delete(master_dark_pxl_nb_list);

    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_image_merge_2_dim on master dark pxl nb failed: %s",
                 cpl_error_get_message_default(my_error));
    
	/* Merge all hot pixels output images into one extension,
	 all the merged extensions are in the hot_pix_mask */
	espdr_msg("Merging the HOT PIXELS image");
	cpl_imagelist *hot_pix_mask = cpl_imagelist_new();
    my_error = espdr_image_merge_2_dim(hot_pixels_list, CCD_geom,
                                       CPL_TYPE_INT, &hot_pix_mask);

    cpl_imagelist_delete(hot_pixels_list);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_image_merge_2_dim hot pixels failed: %s",
                 cpl_error_get_message_default(my_error));

    my_error = espdr_dark_QC(CCD_geom, exposure_time, inst_config,
                             qc_kws, &keywords);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_dark_QC failed: %s",
                 cpl_error_get_message_default(my_error));

    espdr_mdark_save_products(hot_pix_mask, master_dark, master_dark_pxl_nb,
                              recipe_id, frameset, parameters, used_frames,
                              dark_frames, inst_config, CCD_geom, keywords);

    espdr_msg("Cleaning memory");

	cpl_propertylist_delete(keywords);
    cpl_free(qc_kws);

	cpl_imagelist_delete(master_dark);
	cpl_imagelist_delete(hot_pix_mask);
	cpl_frameset_delete(dark_frames);
	cpl_frameset_delete(all_mbias_frames);
    cpl_frameset_delete(used_frames);

    espdr_parameters_OVSC_delete(OVSC_param);
    espdr_parameters_DARK_delete(DARK_param);
    espdr_parameters_CCD_geometry_delete(CCD_geom);
    espdr_parameters_inst_config_delete(inst_config);

	return (0);
}




/*----------------------------------------------------------------------------*/
/**
 @brief    Create the structure for DARK input parameters
 @param    recipe_id	recipe identifier
 @param    list			list of input parameters
 @param    p			input DARK parameters structure
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_parameters_DARK_create(const char* recipe_id,
										  cpl_parameterlist *list,
										  espdr_DARK_param *p) {
	
    espdr_ensure(recipe_id == NULL,
                 CPL_ERROR_NULL_INPUT, "The recipe ID is NULL");
    
	/* check parameters */
	espdr_ensure(list == NULL, CPL_ERROR_NULL_INPUT, "The parameters list NULL");
	
	/* Fill the parameter list */
	cpl_error_code error_got;
	char comment[COMMENT_LENGTH];
	double ksigma_min = KSIGMA_MIN;
	double ksigma_max = KSIGMA_MAX;
	int sig_clip_max_iter_min = SIGMA_CLIP_MAX_ITER_MIN;
	int sig_clip_max_iter_max = SIGMA_CLIP_MAX_ITER_MAX;

	/* check parameters */
	espdr_ensure(list == NULL, CPL_ERROR_NULL_INPUT, "The parameters list NULL");

	/* Fill the parameter list */
	error_got = espdr_parameters_new_string(recipe_id, list,
			"dark_sig_clip_method",
			p->dark_sigma_clipping_method,
			"method for sigma clipping in master DARK, can be: mean or median");
	espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
			"Error adding parameter dark_sigma_clipping_method to the list");

	sprintf(comment,
			"ksigma for sigma clipping in MDARK, must be between: %.2lf and %.2lf",
			ksigma_min, ksigma_max);
	error_got = espdr_parameters_new_range_double(recipe_id, list,
			"dark_ksigma", p->dark_ksigma,
			KSIGMA_MIN, KSIGMA_MAX,
			comment);
	espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
			"Error adding parameter dark_ksigma to the list");

	error_got = espdr_parameters_new_string(recipe_id, list,
			"hot_pix_sig_clip_method",
			p->hotpixels_sigma_clipping_method,
			"method for sigma clipping in HOT PIXELS, can be mean or median");
	espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
			"Error adding parameter hotpixels_sigma_clipping_method to the list");

	sprintf(comment,
			"ksigma for sigma clipping in HOT PIXELS, must be between: %.2lf and %.2lf",
			ksigma_min, ksigma_max);
	error_got = espdr_parameters_new_range_double(recipe_id, list,
			"hot_pix_ksigma",
			p->hotpixels_ksigma,
			KSIGMA_MIN, KSIGMA_MAX,
			comment);
	espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
			"Error adding parameter hotpixels_ksigma to the list");

	sprintf(comment,
			"maximal number of iterations in HOT PIXELS, must be between: %d and %d",
			sig_clip_max_iter_min, sig_clip_max_iter_max);
	error_got = espdr_parameters_new_range_int(recipe_id, list,
			"hot_pix_max_iter",
			p->hotpixels_max_iter,
			SIGMA_CLIP_MAX_ITER_MIN,
			SIGMA_CLIP_MAX_ITER_MAX,
			comment);
	espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
			"Error adding parameter hotpixels_max_iter to the list");
	return(CPL_ERROR_NONE);
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Delete the structure for DARK input parameters
 @param    p	input DARK parameters structure
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_parameters_DARK_delete(espdr_DARK_param* p) {
	
    cpl_free((void *)p->dark_sigma_clipping_method);
    cpl_free((void *)p->hotpixels_sigma_clipping_method);
    cpl_free(p);
    p = NULL;
    
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief     Get the mdark recipe parameters
 @param     recipe_id   recipe ID
 @param     param_list  parameters list
 @param     DARK_param  DARK parameters structure
 @return    CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_parameters_DARK_get(const char* recipe_id,
                                         cpl_parameterlist* param_list,
                                         espdr_DARK_param *DARK_param) {
	
    espdr_ensure(recipe_id == NULL,
                 CPL_ERROR_NULL_INPUT, "The recipe ID is NULL");
    
	/* check parameters */
	espdr_ensure(param_list == NULL, CPL_ERROR_NULL_INPUT,
				 "Parameters list is NULL");
	
	/* Fill the structure */
    DARK_param->dark_sigma_clipping_method = espdr_parameters_get_string(recipe_id,
                                                                         param_list,
                                                            "dark_sig_clip_method");
    
    DARK_param->dark_ksigma = espdr_parameters_get_double(recipe_id,
                                                          param_list,
                                                          "dark_ksigma");
    
    DARK_param->hotpixels_sigma_clipping_method = espdr_parameters_get_string(recipe_id,
                                                                              param_list,
                                                            "hot_pix_sig_clip_method");
    
    DARK_param->hotpixels_ksigma = espdr_parameters_get_double(recipe_id,
                                                               param_list,
                                                               "hot_pix_ksigma");
    
    DARK_param->hotpixels_max_iter = espdr_parameters_get_int(recipe_id,
                                                              param_list,
                                                              "hot_pix_max_iter");
    
	return cpl_error_get_code();
}


/*---------------------------------------------------------------------------*/
/**
 @brief     print the DARK parameters
 @param     DARK_param  DARK parameters structure
 @return    CPL_ERROR_NONE iff OK
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_parameters_DARK_print(espdr_DARK_param *DARK_param) {
	
	espdr_msg("\tDARK parameters:");
    espdr_msg("\t\tDARK stacking sigma clipping method = %s",
              DARK_param->dark_sigma_clipping_method);
    espdr_msg("\t\tDARK stacking ksigma = %.2f",
              DARK_param->dark_ksigma);
    espdr_msg("\t\tDARK hotpixels sigma clipping method = %s",
              DARK_param->hotpixels_sigma_clipping_method);
    espdr_msg("\t\tDARK hotpixels ksigma = %.2f",
              DARK_param->hotpixels_ksigma);
    espdr_msg("\t\tDARK hotpixels max iter = %d",
              DARK_param->hotpixels_max_iter);
    
	return (CPL_ERROR_NONE);
}


/*----------------------------------------------------------------------------*/
/**
 @brief    Init DARK_param structure
 @param    recipe_id    recipe name
 @param    parameters   recipe input parameters
 @return   allocated structure iff OK
 */
/*----------------------------------------------------------------------------*/

espdr_DARK_param *espdr_DARK_param_init(const char *recipe_id,
                                        cpl_parameterlist *parameters) {
    
    cpl_error_code my_error;
    
    if (recipe_id == NULL) return NULL;
    
    espdr_DARK_param *DARK_param =
    (espdr_DARK_param *)cpl_malloc(sizeof(espdr_DARK_param));
    
    /* Read the dark parameters */
    my_error = espdr_parameters_DARK_get(recipe_id, parameters, DARK_param);
    my_error = espdr_parameters_DARK_print(DARK_param);

    if(my_error != CPL_ERROR_NONE) {
        return NULL;
    } else {
        return DARK_param;
    }
}

/*---------------------------------------------------------------------------*/
/**
 @brief     Verify if all the frames in a frameset have the same exposure time
 @param         dark_frameset   the frames list
 @param         Texp_KW         exposure time KW name
 @param         tolerance       tolerance in comparing Texps
 @param[out]    Texp_flag       1 if time are the same, 0 if not
 @return   CPL_ERROR_NONE iff OK
 
 TODO:
 1) To prevent usage of deprecated CPL functions one can do the loop like:
    nframes = cpl_frameset_get_size(dark_frameset);
    for(i = 0; i < nframes; i++) {
        frame = cpl_frameset_get_frame(dark_frameset, i);
        ....
    }
 --> DSO to AMO: OK, thanks for the indication
 
 2) Not all functions need to return cpl_error_code. One may also return
 directly the exit of the verification (Texp_flag)
 --> DSO to AMO: I prefer to have the codew homogenous, to keep like it is.
 */
/*---------------------------------------------------------------------------*/
cpl_error_code espdr_verify_texp(const cpl_frameset *dark_frameset,
                                 const char *Texp_KW,
                                 const double tolerance,
                                 int *Texp_flag) {
	
	double exposure_time = 0.0;
	double ref_exp_time = -1.0;
	cpl_frame *curr_frame = NULL;
	cpl_propertylist *keywords = NULL;
    	cpl_error_code my_error = CPL_ERROR_NONE;
	
	espdr_msg_debug("Starting espdr_verify_texp()");
	espdr_msg_debug("dark frameset size: %lld", 
					cpl_frameset_get_size(dark_frameset));
	
	espdr_ensure(dark_frameset == NULL, CPL_ERROR_NULL_INPUT, 
				 "Input frameset is NULL");
	*Texp_flag = 1;

    int nset = cpl_frameset_get_size(dark_frameset);
    cpl_frameset_iterator* iter = cpl_frameset_iterator_new(dark_frameset);
    curr_frame = cpl_frameset_iterator_get(iter);
    
    for ( int iter_frame = 0; iter_frame < nset; iter_frame++ ) {
        
        keywords = cpl_propertylist_load(cpl_frame_get_filename(curr_frame), 0);
        exposure_time = cpl_propertylist_get_double(keywords, Texp_KW);
        
        if (ref_exp_time == -1.0) {
            ref_exp_time = exposure_time;
        } else {
            //if (exposure_time != ref_exp_time) {
            if (fabs(exposure_time - ref_exp_time) > tolerance) {
                Texp_flag = 0;
            }
        }
        
        cpl_propertylist_delete(keywords);
        
	cpl_frameset_iterator_advance(iter, 1);
	curr_frame = cpl_frameset_iterator_get(iter);
	}
	cpl_frameset_iterator_delete(iter);
    
    my_error = cpl_error_get_code();
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_verify_texp failed: %s",
                 cpl_error_get_message_default(my_error));
    
	return (CPL_ERROR_NONE);
}

/*---------------------------------------------------------------------------*/
/**
 @brief    Subtract cosmics via stacking and sigma-clipping
 @param         input_set            input image frames list
 @param         CCD_geom             CCD geometry layout structure
 @param         ksigma               kappa value for sigma-clipping
 @param         sigma_clipping_method    'mean' or 'median'
 @param         max_iter             max number of iterations
 @param[out]    master_imagelist_RE  imagelist holding master images
 @param[out]    master_mean_RE       array holding master mean values
 @param[out]    RON_RE               array holding RON results
 @param[out]    total_cosmics_RE     array with the total number of cosmics
 @param         use_hdrl             HDRL use flag
 @return   CPL_ERROR_NONE iff OK
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_stack_sigma(const cpl_frameset *input_set,
                                 const espdr_CCD_geometry *CCD_geom,
                                 const double ksigma,
                                 const char *sigma_clipping_method,
                                 const int max_iter,
                                 cpl_imagelist **master_imagelist_RE,
                                 double *master_mean_RE,
                                 double *RON_RE,
                                 int *total_cosmics_RE,
                                 const int use_hdrl) {
    
    /* Loop  indices */
    int i, j, k, l, index, imagelist_place;
    cpl_error_code my_error = CPL_ERROR_NONE;
    int frames_nb = 0;
    int input_set_size = 0;
    cpl_image *curr_image = NULL;
    cpl_image *master_image = NULL;
    cpl_imagelist *images_stack = NULL;
    int cosmics = 0;
    int total_output_nb = CCD_geom->total_output_nb;
    
    int real_llx, real_lly, real_urx, real_ury;
    int used_real_nx, used_real_ny;
    /* AMO added */
    const cpl_frame* frame = NULL;
    const char *fname = NULL;
    espdr_msg_debug("Starting espdr_stack_sigma");
    
    espdr_ensure(input_set == NULL, CPL_ERROR_NULL_INPUT,
                 "Input frameset is NULL");
    
    input_set_size = cpl_frameset_get_size(input_set);
    espdr_msg("STACK SIGMA: input frameset size is %d",
                    input_set_size);

     if(master_mean_RE == NULL) {
             /* ASE: Do nothing, Just to document this parameter is not used*/
     }
    
    frames_nb = input_set_size/CCD_geom->total_output_nb;
    espdr_ensure(((input_set_size % total_output_nb) != 0),
                 CPL_ERROR_INCOMPATIBLE_INPUT,
                 "The nb of images doesn't correspond to the nb of frames x nb of extensions x nb of outputs");
    /* AMO: make checks
     espdr_msg("input_set_size=%d",input_set_size);
     espdr_msg("total_output_nb=%d",CCD_geom->total_output_nb);
     espdr_msg("frames_nb=%d",frames_nb);
     */
    
    imagelist_place = 0;
    for (j = 0; j < CCD_geom->ext_nb; j++) {
        total_cosmics_RE[j] = 0;
        for (k = 0; k < CCD_geom->exts[j].out_nb_x; k++) {
            for (l = 0; l < CCD_geom->exts[j].out_nb_y; l++) {
                index = 0;
                images_stack = cpl_imagelist_new();
                for (i = 0; i < frames_nb; i++) {
                    /* loops over frames nb */
                    frame = cpl_frameset_get_position_const(input_set,
                                                      i * total_output_nb +
                                                      j * CCD_geom->exts[j].out_nb_x * CCD_geom->exts[j].out_nb_y +
                                                      k * CCD_geom->exts[j].out_nb_y + l);
                    //frame = cpl_frameset_get_position(input_set,
                    //                                  i*total_output_nb+j);
                    fname = cpl_frame_get_filename(frame);
                    //espdr_msg("fname2=%s",fname);
                    curr_image = cpl_image_load(fname,CPL_TYPE_DOUBLE,0,0);
                    
                    /* AMO added to check results
                     sprintf(tmp_name,"stack2_dark_ext_%d_port_x%d_y%d_inxed_%d.fits",j,k,l,index);
                     cpl_image_save(curr_image, tmp_name, CPL_TYPE_DOUBLE,
                     NULL, CPL_IO_DEFAULT);
                     */
                    /* AMO: we have replaced an imagelist with a frameset
                     curr_image = cpl_imagelist_get(input_imagelist,
                     i * total_output_nb + j);
                     
                     
                     my_error = cpl_error_get_code();
                     espdr_ensure(my_error == CPL_ERROR_NULL_INPUT, my_error,
                     "Image in the input list nb %d is NULL",
                     i * total_output_nb + j);
                     espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "Getting image from the imagelist failed");
                     */
                    my_error = cpl_imagelist_set(images_stack,
                                                 cpl_image_duplicate(curr_image),
                                                 index);
                    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                                 "Insert into imagelist failed");
                    cpl_image_delete(curr_image);
                    index++;
                }
                
                real_llx = CCD_geom->exts[j].outputs[k][l].real_llx;
                real_lly = CCD_geom->exts[j].outputs[k][l].real_lly;
                real_urx = CCD_geom->exts[j].outputs[k][l].real_urx;
                real_ury = CCD_geom->exts[j].outputs[k][l].real_ury;
                used_real_nx = real_urx - real_llx + 1;
                used_real_ny = real_ury - real_lly + 1;
                /* AMO: we have replaced an imagelist with a frameset
                 cpl_imagelist_save(images_stack,cpl_sprintf("iml2_ext%d.fits",j),CPL_TYPE_FLOAT,
                 NULL,CPL_IO_DEFAULT);
                 cpl_image_save(master_image,cpl_sprintf("master2_ext%d.fits",j),CPL_TYPE_FLOAT,
                 NULL,CPL_IO_DEFAULT);
                 */
                espdr_msg("real_nx=%d real_ny=%d ksigma=%g method=%s max_iter=%d",
                          used_real_nx,used_real_ny,ksigma,sigma_clipping_method,max_iter);


                if(use_hdrl) {
                      // HDRL based stacking function
                      my_error = espdr_hdrl_master_port(images_stack,
                                                        ksigma, ksigma,
                                                        max_iter,
                                                        &master_image,
                                                        &RON_RE[imagelist_place],
                                                        &cosmics);

                  } else {

                	  if (strcmp(sigma_clipping_method, "median") == 0) {
                		  espdr_stack_sigma_one_output_median(images_stack,
                				  used_real_nx,
								  used_real_ny,
								  ksigma,
								  max_iter,
								  &master_image,
								  &RON_RE[imagelist_place],
								  &cosmics);
                	  } else {
                   		  espdr_stack_sigma_one_output_mean(images_stack,
                    				  used_real_nx,
    								  used_real_ny,
    								  ksigma,
    								  max_iter,
    								  &master_image,
    								  &RON_RE[imagelist_place],
    								  &cosmics);
                	  }

                      /*
                      my_error = espdr_stack_sigma_one_output(images_stack,
                                                              used_real_nx,
                                                              used_real_ny,
                                                              ksigma,
                                                              sigma_clipping_method,
                                                              max_iter,
                                                              &master_image,
                                                              &RON_RE[imagelist_place],
                                                              &cosmics);
                                                              */


                  }

                
                espdr_ensure(master_image == NULL, CPL_ERROR_ILLEGAL_OUTPUT,
                             "Resulting master image is NULL");
                
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "Stack sigma on one output failed");
                
                double master_mean = cpl_image_get_mean(master_image);
                espdr_msg("master mean %d[%d, %d] is %lf", j, k, l, master_mean);
                
                total_cosmics_RE[j] = total_cosmics_RE[j] + cosmics;
                cpl_imagelist_set(*master_imagelist_RE,
                                  cpl_image_duplicate(master_image),
                                  imagelist_place);
                imagelist_place++;
                
                my_error = cpl_error_get_code();
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "Imagelist set failed");
                
                cpl_image_delete(master_image);
                cpl_imagelist_delete(images_stack);
            }
        }
    }
    
    for (i = 0; i < CCD_geom->ext_nb; i++) {
        espdr_msg_debug("Total cosmics[ext = %d]: %d", i, total_cosmics_RE[i]);
    }
    
    return cpl_error_get_code();
}


/*---------------------------------------------------------------------------*/
/**
 @brief    Subtract cosmics via stacking and sigma-clipping
 @param         input_set            input image frames list
 @param         CCD_geom             CCD geometry layout structure
 @param         ksigma               kappa value for sigma-clipping
 @param         sigma_clipping_method    'mean' or 'median'
 @param         max_iter             max number of iterations
 @param[out]    master_imagelist_RE  imagelist holding master images
 @param[out]    master_mean_RE       array holding master mean values
 @param[out]    RON_RE               array holding RON results
 @param[out]    total_cosmics_RE     array with the total number of cosmics
 @param         use_hdrl             HDRL use flag
 @return   CPL_ERROR_NONE iff OK
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_stack_sigma_with_pxl_nb(const cpl_frameset *input_set,
                                             const espdr_CCD_geometry *CCD_geom,
                                             const double ksigma,
                                             const char *sigma_clipping_method,
                                             const int max_iter,
                                             cpl_imagelist **master_imagelist_RE,
                                             cpl_imagelist **master_pxl_nb_imagelist_RE,
                                             double *master_mean_RE,
                                             double *RON_RE,
                                             int *total_cosmics_RE,
                                             const int use_hdrl) {

    /* Loop  indices */
    int i, j, k, l, index, imagelist_place;
    cpl_error_code my_error = CPL_ERROR_NONE;
    int frames_nb = 0;
    int input_set_size = 0;
    cpl_image *curr_image = NULL;
    cpl_image *master_image = NULL;
    cpl_image *master_pxl_nb_image = NULL;
    cpl_imagelist *images_stack = NULL;
    int cosmics = 0;
    int total_output_nb = CCD_geom->total_output_nb;
    
    int real_llx, real_lly, real_urx, real_ury;
    int used_real_nx, used_real_ny;
    /* AMO added */
    const cpl_frame* frame = NULL;
    const char *fname = NULL;
    espdr_msg_debug("Starting espdr_stack_sigma");
    
    espdr_ensure(input_set == NULL, CPL_ERROR_NULL_INPUT,
                 "Input frameset is NULL");
    
    input_set_size = cpl_frameset_get_size(input_set);
    espdr_msg("STACK SIGMA: input frameset size is %d",
                    input_set_size);

     if(master_mean_RE == NULL) {
             /* ASE: Do nothing, Just to document this parameter is not used*/
     }
    
    frames_nb = input_set_size/CCD_geom->total_output_nb;
    espdr_ensure(((input_set_size % total_output_nb) != 0),
                 CPL_ERROR_INCOMPATIBLE_INPUT,
                 "The nb of images doesn't correspond to the nb of frames x nb of extensions x nb of outputs");
    /* AMO: make checks
     espdr_msg("input_set_size=%d",input_set_size);
     espdr_msg("total_output_nb=%d",CCD_geom->total_output_nb);
     espdr_msg("frames_nb=%d",frames_nb);
     */
    double DARK_raw[total_output_nb][frames_nb];
    double RON_raw[total_output_nb][frames_nb];
    double DARK_raw_mean[total_output_nb];
    double RON_raw_mean[total_output_nb];

    imagelist_place = 0;
    for (j = 0; j < CCD_geom->ext_nb; j++) {
        total_cosmics_RE[j] = 0;
        for (k = 0; k < CCD_geom->exts[j].out_nb_x; k++) {
            for (l = 0; l < CCD_geom->exts[j].out_nb_y; l++) {
                index = 0;
                images_stack = cpl_imagelist_new();
                DARK_raw_mean[imagelist_place] = 0.0;
                RON_raw_mean[imagelist_place] = 0.0;
                for (i = 0; i < frames_nb; i++) {
                    /* loops over frames nb */
                    frame = cpl_frameset_get_position_const(input_set,
                                                      i * total_output_nb +
                                                      j * CCD_geom->exts[j].out_nb_x * CCD_geom->exts[j].out_nb_y +
                                                      k * CCD_geom->exts[j].out_nb_y + l);
                    //frame = cpl_frameset_get_position(input_set,
                    //                                  i*total_output_nb+j);
                    fname = cpl_frame_get_filename(frame);
                    //espdr_msg("fname2=%s",fname);
                    curr_image = cpl_image_load(fname,CPL_TYPE_DOUBLE,0,0);
                    
                    /* AMO added to check results
                     sprintf(tmp_name,"stack2_dark_ext_%d_port_x%d_y%d_inxed_%d.fits",j,k,l,index);
                     cpl_image_save(curr_image, tmp_name, CPL_TYPE_DOUBLE,
                     NULL, CPL_IO_DEFAULT);
                     */
                    /* AMO: we have replaced an imagelist with a frameset
                     curr_image = cpl_imagelist_get(input_imagelist,
                     i * total_output_nb + j);
                     
                     
                     my_error = cpl_error_get_code();
                     espdr_ensure(my_error == CPL_ERROR_NULL_INPUT, my_error,
                     "Image in the input list nb %d is NULL",
                     i * total_output_nb + j);
                     espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "Getting image from the imagelist failed");
                     */
                    my_error = cpl_imagelist_set(images_stack,
                                                 cpl_image_duplicate(curr_image),
                                                 index);
                    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                                 "Insert into imagelist failed");
                    
                    DARK_raw[imagelist_place][i] = cpl_image_get_mad(curr_image, &RON_raw[imagelist_place][i]);
                    DARK_raw_mean[imagelist_place] += DARK_raw[imagelist_place][i];
                    RON_raw_mean[imagelist_place] += RON_raw[imagelist_place][i];
                    
                    cpl_image_delete(curr_image);
                    index++;
                }
                
                DARK_raw_mean[imagelist_place] = DARK_raw_mean[imagelist_place] / frames_nb;
                RON_raw_mean[imagelist_place] = RON_raw_mean[imagelist_place] / frames_nb;
                //espdr_msg("DARK_raw_mean[%d] = %f, RON_raw_mean[%d] = %f",
                //          imagelist_place, DARK_raw_mean[imagelist_place],
                //          imagelist_place, RON_raw_mean[imagelist_place]);
                
                real_llx = CCD_geom->exts[j].outputs[k][l].real_llx;
                real_lly = CCD_geom->exts[j].outputs[k][l].real_lly;
                real_urx = CCD_geom->exts[j].outputs[k][l].real_urx;
                real_ury = CCD_geom->exts[j].outputs[k][l].real_ury;
                used_real_nx = real_urx - real_llx + 1;
                used_real_ny = real_ury - real_lly + 1;
                /* AMO: we have replaced an imagelist with a frameset
                 cpl_imagelist_save(images_stack,cpl_sprintf("iml2_ext%d.fits",j),CPL_TYPE_FLOAT,
                 NULL,CPL_IO_DEFAULT);
                 cpl_image_save(master_image,cpl_sprintf("master2_ext%d.fits",j),CPL_TYPE_FLOAT,
                 NULL,CPL_IO_DEFAULT);
                 */
                espdr_msg("real_nx=%d real_ny=%d ksigma=%g method=%s max_iter=%d",
                          used_real_nx,used_real_ny,ksigma,sigma_clipping_method,max_iter);


                if(use_hdrl) {
                      // HDRL based stacking function
                      my_error = espdr_hdrl_master_port(images_stack,
                                                        ksigma, ksigma,
                                                        max_iter,
                                                        &master_image,
                                                        &RON_RE[imagelist_place],
                                                        &cosmics);

                  } else {

                      if (strcmp(sigma_clipping_method, "median") == 0) {
                          espdr_stack_sigma_one_output_median_with_pxl_nb(images_stack,
                                  used_real_nx,
                                  used_real_ny,
                                  ksigma,
                                  max_iter,
                                  &master_image,
                                  &master_pxl_nb_image,
                                  &RON_RE[imagelist_place],
                                  &cosmics);
                      } else {
                             espdr_stack_sigma_one_output_mean_with_pxl_nb(images_stack,
                                      used_real_nx,
                                      used_real_ny,
                                      ksigma,
                                      max_iter,
                                      &master_image,
                                      &master_pxl_nb_image,
                                      &RON_RE[imagelist_place],
                                      &cosmics);
                      }

                      /*
                      my_error = espdr_stack_sigma_one_output(images_stack,
                                                              used_real_nx,
                                                              used_real_ny,
                                                              ksigma,
                                                              sigma_clipping_method,
                                                              max_iter,
                                                              &master_image,
                                                              &RON_RE[imagelist_place],
                                                              &cosmics);
                                                              */


                  }

                
                espdr_ensure(master_image == NULL, CPL_ERROR_ILLEGAL_OUTPUT,
                             "Resulting master image is NULL");
                
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "Stack sigma on one output failed");
                
                RON_RE[imagelist_place] = RON_raw_mean[imagelist_place];
                double master_mean = cpl_image_get_mean(master_image);
                espdr_msg("master mean %d[%d, %d] is %lf", j, k, l, master_mean);
                espdr_msg("RON[%d] %d[%d, %d] is %lf",
                          imagelist_place, j, k, l, RON_RE[imagelist_place]);
                
                total_cosmics_RE[j] = total_cosmics_RE[j] + cosmics;
                cpl_imagelist_set(*master_imagelist_RE,
                                  cpl_image_duplicate(master_image),
                                  imagelist_place);
                cpl_imagelist_set(*master_pxl_nb_imagelist_RE,
                                  cpl_image_duplicate(master_pxl_nb_image),
                                  imagelist_place);
                imagelist_place++;
                
                my_error = cpl_error_get_code();
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "Imagelist set failed");
                
                cpl_image_delete(master_image);
                cpl_image_delete(master_pxl_nb_image);
                cpl_imagelist_delete(images_stack);
            }
        }
    }
    
    for (i = 0; i < CCD_geom->ext_nb; i++) {
        espdr_msg_debug("Total cosmics[ext = %d]: %d", i, total_cosmics_RE[i]);
    }
    
    return cpl_error_get_code();
}


/*---------------------------------------------------------------------------*/
/**
 @brief    Computes master frame, RON, CRHs, via sigma-clipping for 1 output
 @param         input_imagelist the input images list
 @param         nx              x size of image (can be removed)
 @param         ny              y size of image (acn be removed)
 @param         ksigma          kappa value in kappa-sigma clip
 @param         sigma_clipping_method    'mean' or 'median'
 @param         max_iter        maximum number of iterations
 @param[out]    master_image_RE master image frame
 @param[out]    RON_RE          Read Out Noise of master frame
 @param[out]    totalCosmics_RE total number of detected cosmics hits
 @return   CPL_ERROR_NONE iff OK

 TODO:
 2) nx,ny can be removed as they are the size of any input image of the list
 3) The parameters ksigma,sigma_clipping_method,max_iter should be part of a
 single parameter structure, controlling sigma clipping. the method should be
 a typedef
 4) This function does not seems to deal with possible (static) bad pixels.
 5) This function may be replaced by HDRL corresponding function to stack frames
 6) This function should be unit tested
 7) commented-out code should be removed
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_stack_sigma_one_output(const cpl_imagelist *input_imagelist,
                                            const int nx,
                                            const int ny,
                                            const double ksigma,
                                            const char *sigma_clipping_method,
                                            const int max_iter,
                                            cpl_image **master_image_RE,
                                            double *RON_RE,
                                            int *totalCosmics_RE) {
	
	/* Loop indices */
	int i = 0;
	int point = 0;
	int image_size = nx*ny;
	
	/* Flag indicating to reject (or not) the 0, when computing the MAD */
	/* Not used anymore */
	int reject_zeroMAD = 1;
	/* Instead w force the MAD to another value if it is == 0.0 */
	double forced_MAD = 1.0;
	
	/* error code */
	cpl_error_code my_error;
	
	espdr_msg_debug("Starting espdr_stack_sigma with kSigma = %lf, sigma_clipping_method = %s", 
			ksigma, sigma_clipping_method);
	
	espdr_msg_debug("Images size: nx = %d, ny = %d", nx, ny);
	
	espdr_ensure(input_imagelist == NULL, CPL_ERROR_NULL_INPUT,
			   "Input images list is NULL");
	
	/* Number of frames (FITS files) */
	int frames_nb = cpl_imagelist_get_size(input_imagelist);
	
	espdr_msg_debug("On %d frames", frames_nb);
	
	for (i = 0; i < frames_nb; i++) {
		espdr_ensure(cpl_imagelist_get_const(input_imagelist, i) == NULL,
				   CPL_ERROR_NULL_INPUT, 
				   "Input image %d in the input list is NULL", i);
	}
	
	/* data tables extracted from the input images list */
	const double *images_table[frames_nb];
	
	/* vector of the same pixel in all the frames */
	//double *data_vector = (double *)cpl_calloc(frames_nb, sizeof(double));
	
	/* master image data table */
	double *master_vector = (double *)cpl_calloc(nx*ny, sizeof(double));
    
    /* master image crated */
    cpl_image *master_image;
    
	/* center of data: mean or MAD */
	double CoD = 0.0;
	
	/* number of cosmics rejected */
	int cosmics = 0;
	int total_cosmics = 0;
	int total_cosmics_tmp = 0;
	/* sigma value for rejecting cosmics */
	double sigma = 0.0;
	
	/* mean of the vector */
	double mean = 0.0;
	
	/* standard deviation of the vector */
	double stddev = 0.0;
	
	/* bounds for rejecting outliers */
	double reject_low = -1.0;
	double reject_high = -1.0;
	
	/* RON for each point of the CCD */
	double *RON_image = (double *)cpl_calloc(nx*ny, sizeof(double));
	/* resulting RON */
	double RON_mean;
	
	
	/* get the data from images list */
	//espdr_msg("New output mean:");
	for (i = 0; i < frames_nb; i++) {
		images_table[i] = 
		cpl_image_get_data_double_const(cpl_imagelist_get_const(input_imagelist, i));
		//espdr_msg("Frame %d mean: %f", i, 
		//	cpl_image_get_mean(cpl_imagelist_get(input_imagelist, i)));
	}
	
	/* Calculate the master bias for all the image points with cosmics rejection 
	 by sigma clipping */
	double *data_vector;
    HDRL_OMP(omp parallel private(data_vector,total_cosmics_tmp,point,i,my_error,CoD,sigma,cosmics,mean,stddev,reject_low,reject_high))
	{
	/* vector of the same pixel in all the frames */
	data_vector = (double *)cpl_calloc(frames_nb, sizeof(double));
	cpl_vector *data_vector_wrapped = cpl_vector_wrap(frames_nb, data_vector);
	total_cosmics_tmp = 0;
    HDRL_OMP(omp for)
	for (point = 0; point < image_size; point++) {
	    /* vector of the same pixel in all the frames */
	    //data_vector = (double *)cpl_calloc(frames_nb, sizeof(double));
		for (i = 0; i < frames_nb; i++) {
			data_vector[i] = images_table[i][point];
		}

		//cpl_vector *data_vector_wrapped = cpl_vector_wrap(frames_nb, data_vector);
		my_error = espdr_sig_clip(data_vector_wrapped, ksigma, ksigma, 
                                  sigma_clipping_method,
                                  max_iter, reject_zeroMAD, forced_MAD,
                                  &CoD, &sigma, &cosmics,
                                  &mean, &stddev,
                                  &reject_low, &reject_high);
		/*
		espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_sig_clip failed");
		*/

		master_vector[point] = mean;
		RON_image[point] = stddev;
		total_cosmics_tmp = total_cosmics_tmp + cosmics;
	}
	cpl_vector_unwrap(data_vector_wrapped);
	cpl_free(data_vector);
    HDRL_OMP(omp critical)
	{

        //HDRL_OMP(omp atomic)
		total_cosmics = total_cosmics + total_cosmics_tmp;
	}

		/* test on ESO sigma clipping same speed as my sigma_clipping,
		 even slower and rejected over 5 mln cosmics */
		/*
		 double mean_ks = 0.0;
		 double mean_ks_err = 0.0;
		 cpl_size naccepted = 0;
		 double reject_low = 0.0;
		 double reject_high = 0.0;
		 cpl_vector *my_vector = cpl_vector_wrap(frames_nb, data_vector);
		 cpl_vector *my_vector_err = cpl_vector_wrap(frames_nb, error_vector);
		 my_error = drl_kappa_sigma_clip(my_vector, my_vector_err, 4.0, 4.0, frames_nb, 
		 &mean_ks, &mean_ks_err, &naccepted, &reject_low, &reject_high);
		 master_bias_vector[point] = mean_ks;
		 RON_image[point] = mean_ks_err;
		 total_cosmics = total_cosmics + frames_nb - naccepted;
		 */
	}
	
	/* Not sure if the image is copied into the list or
	 only the pointer is inserted: image is copied */
	master_image = cpl_image_wrap_double(nx, ny, master_vector);
    *master_image_RE = cpl_image_duplicate(master_image);
    cpl_image_unwrap(master_image);
    
    cpl_vector *RON_vector = cpl_vector_wrap(nx*ny, RON_image);
    RON_mean = cpl_vector_get_mean(RON_vector);
    cpl_vector_unwrap(RON_vector);
	//espdr_msg("Mean of RON: %lf", RON_mean);
	
	*totalCosmics_RE = total_cosmics;
	espdr_msg_debug("total cosmics (RE) = %d", *totalCosmics_RE);
	
	*RON_RE = RON_mean;
	espdr_msg_debug("RON_RE = %lf", *RON_RE);
	
	//espdr_msg_debug("total cosmics: %d", total_cosmics);
	
	//cpl_free(data_vector);
    cpl_free(master_vector);
	
	cpl_free(RON_image);
	
	return cpl_error_get_code();
}

cpl_error_code espdr_stack_sigma_one_output_mean(const cpl_imagelist *input_imagelist,
                                            const int nx,
                                            const int ny,
                                            const double ksigma,
                                            const int max_iter,
                                            cpl_image **master_image_RE,
                                            double *RON_RE,
                                            int *totalCosmics_RE) {

    /* Loop indices */
    int i = 0;
    int point = 0;
    int image_size = nx*ny;

    /* Flag indicating to reject (or not) the 0, when computing the MAD */
    /* Not used anymore */
    int reject_zeroMAD = 1;
    /* Instead w force the MAD to another value if it is == 0.0 */
    double forced_MAD = 1.0;

    /* error code */
    cpl_error_code my_error;

    espdr_msg_debug("Starting espdr_stack_sigma with kSigma = %lf, sigma_clipping_method = %s",
            ksigma, "mean");

    espdr_msg_debug("Images size: nx = %d, ny = %d", nx, ny);

    espdr_ensure(input_imagelist == NULL, CPL_ERROR_NULL_INPUT,
               "Input images list is NULL");

    /* Number of frames (FITS files) */
    int frames_nb = cpl_imagelist_get_size(input_imagelist);

    espdr_msg_debug("On %d frames", frames_nb);

    for (i = 0; i < frames_nb; i++) {
        espdr_ensure(cpl_imagelist_get_const(input_imagelist, i) == NULL,
                   CPL_ERROR_NULL_INPUT,
                   "Input image %d in the input list is NULL", i);
    }

    /* data tables extracted from the input images list */
    const double *images_table[frames_nb];

    /* vector of the same pixel in all the frames */
    //double *data_vector = (double *)cpl_calloc(frames_nb, sizeof(double));

    /* master image data table */
    double *master_vector = (double *)cpl_calloc(nx*ny, sizeof(double));

    /* master image crated */
    cpl_image *master_image;

    /* center of data: mean or MAD */
    double CoD = 0.0;

    /* number of cosmics rejected */
    int cosmics = 0;
    int total_cosmics = 0;
    int total_cosmics_tmp = 0;
    /* sigma value for rejecting cosmics */
    double sigma = 0.0;

    /* mean of the vector */
    double mean = 0.0;

    /* standard deviation of the vector */
    double stddev = 0.0;

    /* bounds for rejecting outliers */
    double reject_low = -1.0;
    double reject_high = -1.0;

    /* RON for each point of the CCD */
    double *RON_image = (double *)cpl_calloc(nx*ny, sizeof(double));
    /* resulting RON */
    double RON_mean;


    /* get the data from images list */
    //espdr_msg("New output mean:");
    for (i = 0; i < frames_nb; i++) {
        images_table[i] =
        cpl_image_get_data_double_const(cpl_imagelist_get_const(input_imagelist, i));
        //espdr_msg("Frame %d mean: %f", i,
        //    cpl_image_get_mean(cpl_imagelist_get(input_imagelist, i)));
    }

    /* Calculate the master bias for all the image points with cosmics rejection
     by sigma clipping */
    double *data_vector;
    HDRL_OMP(omp parallel private(data_vector,total_cosmics_tmp,point,i,my_error,CoD,sigma,cosmics,mean,stddev,reject_low,reject_high))
    {
    /* vector of the same pixel in all the frames */
    data_vector = (double *)cpl_calloc(frames_nb, sizeof(double));
    cpl_vector *data_vector_wrapped = cpl_vector_wrap(frames_nb, data_vector);
    total_cosmics_tmp = 0;
    HDRL_OMP(omp for)
    for (point = 0; point < image_size; point++) {
        /* vector of the same pixel in all the frames */
        //data_vector = (double *)cpl_calloc(frames_nb, sizeof(double));
        for (i = 0; i < frames_nb; i++) {
            data_vector[i] = images_table[i][point];
        }

        //cpl_vector *data_vector_wrapped = cpl_vector_wrap(frames_nb, data_vector);
        my_error = espdr_sig_clip_method_mean(data_vector_wrapped, ksigma, ksigma,
                                  max_iter, reject_zeroMAD, forced_MAD,
                                  &CoD, &sigma, &cosmics,
                                  &mean, &stddev,
                                  &reject_low, &reject_high);
        /*
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_sig_clip failed");
        */

        master_vector[point] = mean;
        RON_image[point] = stddev;
        total_cosmics_tmp = total_cosmics_tmp + cosmics;
    }
    cpl_vector_unwrap(data_vector_wrapped);
    cpl_free(data_vector);
    HDRL_OMP(omp critical)
    {
        //HDRL_OMP(omp atomic)
        total_cosmics = total_cosmics + total_cosmics_tmp;
    }
        /* test on ESO sigma clipping same speed as my sigma_clipping,
         even slower and rejected over 5 mln cosmics */
        /*
         double mean_ks = 0.0;
         double mean_ks_err = 0.0;
         cpl_size naccepted = 0;
         double reject_low = 0.0;
         double reject_high = 0.0;
         cpl_vector *my_vector = cpl_vector_wrap(frames_nb, data_vector);
         cpl_vector *my_vector_err = cpl_vector_wrap(frames_nb, error_vector);
         my_error = drl_kappa_sigma_clip(my_vector, my_vector_err, 4.0, 4.0, frames_nb,
         &mean_ks, &mean_ks_err, &naccepted, &reject_low, &reject_high);
         master_bias_vector[point] = mean_ks;
         RON_image[point] = mean_ks_err;
         total_cosmics = total_cosmics + frames_nb - naccepted;
         */
    }

    /* Not sure if the image is copied into the list or
     only the pointer is inserted: image is copied */
    master_image = cpl_image_wrap_double(nx, ny, master_vector);
    *master_image_RE = cpl_image_duplicate(master_image);
    cpl_image_unwrap(master_image);

    cpl_vector *RON_vector = cpl_vector_wrap(nx*ny, RON_image);
    RON_mean = cpl_vector_get_mean(RON_vector);
    cpl_vector_unwrap(RON_vector);
    //espdr_msg("Mean of RON: %lf", RON_mean);

    *totalCosmics_RE = total_cosmics;
    espdr_msg_debug("total cosmics (RE) = %d", *totalCosmics_RE);

    *RON_RE = RON_mean;
    espdr_msg_debug("RON_RE = %lf", *RON_RE);

    //espdr_msg_debug("total cosmics: %d", total_cosmics);

    //cpl_free(data_vector);
    cpl_free(master_vector);

    cpl_free(RON_image);

    return cpl_error_get_code();
}


cpl_error_code espdr_stack_sigma_one_output_mean_with_pxl_nb(const cpl_imagelist *input_imagelist,
                                            const int nx,
                                            const int ny,
                                            const double ksigma,
                                            const int max_iter,
                                            cpl_image **master_image_RE,
                                            cpl_image **master_pxl_nb_image_RE,
                                            double *RON_RE,
                                            int *totalCosmics_RE) {

    /* Loop indices */
    int i = 0;
    int point = 0;
    int image_size = nx*ny;

    /* Flag indicating to reject (or not) the 0, when computing the MAD */
    /* Not used anymore */
    int reject_zeroMAD = 1;
    /* Instead w force the MAD to another value if it is == 0.0 */
    double forced_MAD = 1.0;

    /* error code */
    cpl_error_code my_error;

    espdr_msg_debug("Starting espdr_stack_sigma with kSigma = %lf, sigma_clipping_method = %s",
            ksigma, "mean");

    espdr_msg_debug("Images size: nx = %d, ny = %d", nx, ny);

    espdr_ensure(input_imagelist == NULL, CPL_ERROR_NULL_INPUT,
               "Input images list is NULL");

    /* Number of frames (FITS files) */
    int frames_nb = cpl_imagelist_get_size(input_imagelist);

    espdr_msg_debug("On %d frames", frames_nb);

    for (i = 0; i < frames_nb; i++) {
        espdr_ensure(cpl_imagelist_get_const(input_imagelist, i) == NULL,
                   CPL_ERROR_NULL_INPUT,
                   "Input image %d in the input list is NULL", i);
    }

    /* data tables extracted from the input images list */
    const double *images_table[frames_nb];

    /* vector of the same pixel in all the frames */
    //double *data_vector = (double *)cpl_calloc(frames_nb, sizeof(double));

    /* master image data table */
    double *master_vector = (double *)cpl_calloc(nx*ny, sizeof(double));
    /* master image valid (non cosmics) pixels number data table */
    int *master_pxl_nb_vector = (int *)cpl_calloc(nx*ny, sizeof(int));
    
    /* master image created */
    cpl_image *master_image;
    /* master valid (non cosmics) pixels number image created */
    cpl_image *master_pxl_nb_image;
    
    /* center of data: mean or MAD */
    double CoD = 0.0;

    /* number of cosmics rejected */
    int cosmics = 0;
    int total_cosmics = 0;
    int total_cosmics_tmp = 0;
    /* sigma value for rejecting cosmics */
    double sigma = 0.0;

    /* mean of the vector */
    double mean = 0.0;

    /* standard deviation of the vector */
    double stddev = 0.0;

    /* bounds for rejecting outliers */
    double reject_low = -1.0;
    double reject_high = -1.0;

    /* RON for each point of the CCD */
    double *RON_image = (double *)cpl_calloc(nx*ny, sizeof(double));
    /* resulting RON */
    double RON_mean;


    /* get the data from images list */
    //espdr_msg("New output mean:");
    for (i = 0; i < frames_nb; i++) {
        images_table[i] =
        cpl_image_get_data_double_const(cpl_imagelist_get_const(input_imagelist, i));
        //espdr_msg("Frame %d mean: %f", i,
        //    cpl_image_get_mean(cpl_imagelist_get(input_imagelist, i)));
    }

    /* Calculate the master bias for all the image points with cosmics rejection
     by sigma clipping */
    double *data_vector;
    HDRL_OMP(omp parallel private(data_vector,total_cosmics_tmp,point,i,my_error,CoD,sigma,cosmics,mean,stddev,reject_low,reject_high))
    {
    /* vector of the same pixel in all the frames */
    data_vector = (double *)cpl_calloc(frames_nb, sizeof(double));
    cpl_vector *data_vector_wrapped = cpl_vector_wrap(frames_nb, data_vector);
    total_cosmics_tmp = 0;
    HDRL_OMP(omp for)
    for (point = 0; point < image_size; point++) {
        /* vector of the same pixel in all the frames */
        //data_vector = (double *)cpl_calloc(frames_nb, sizeof(double));
        for (i = 0; i < frames_nb; i++) {
            data_vector[i] = images_table[i][point];
        }

        //cpl_vector *data_vector_wrapped = cpl_vector_wrap(frames_nb, data_vector);
        my_error = espdr_sig_clip_method_mean(data_vector_wrapped, ksigma, ksigma,
                                  max_iter, reject_zeroMAD, forced_MAD,
                                  &CoD, &sigma, &cosmics,
                                  &mean, &stddev,
                                  &reject_low, &reject_high);
        /*
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_sig_clip failed");
        */

        master_vector[point] = mean;
        master_pxl_nb_vector[point] = frames_nb-cosmics;
        RON_image[point] = stddev;
        total_cosmics_tmp = total_cosmics_tmp + cosmics;
    }
    cpl_vector_unwrap(data_vector_wrapped);
    cpl_free(data_vector);
    HDRL_OMP(omp critical)
    {
        //HDRL_OMP(omp atomic)
        total_cosmics = total_cosmics + total_cosmics_tmp;
    }
        /* test on ESO sigma clipping same speed as my sigma_clipping,
         even slower and rejected over 5 mln cosmics */
        /*
         double mean_ks = 0.0;
         double mean_ks_err = 0.0;
         cpl_size naccepted = 0;
         double reject_low = 0.0;
         double reject_high = 0.0;
         cpl_vector *my_vector = cpl_vector_wrap(frames_nb, data_vector);
         cpl_vector *my_vector_err = cpl_vector_wrap(frames_nb, error_vector);
         my_error = drl_kappa_sigma_clip(my_vector, my_vector_err, 4.0, 4.0, frames_nb,
         &mean_ks, &mean_ks_err, &naccepted, &reject_low, &reject_high);
         master_bias_vector[point] = mean_ks;
         RON_image[point] = mean_ks_err;
         total_cosmics = total_cosmics + frames_nb - naccepted;
         */
    }

    /* Not sure if the image is copied into the list or
     only the pointer is inserted: image is copied */
    master_image = cpl_image_wrap_double(nx, ny, master_vector);
    *master_image_RE = cpl_image_duplicate(master_image);
    cpl_image_unwrap(master_image);
    master_pxl_nb_image = cpl_image_wrap_int(nx, ny, master_pxl_nb_vector);
    *master_pxl_nb_image_RE = cpl_image_duplicate(master_pxl_nb_image);
    cpl_image_unwrap(master_pxl_nb_image);
    
    cpl_vector *RON_vector = cpl_vector_wrap(nx*ny, RON_image);
    RON_mean = cpl_vector_get_mean(RON_vector);
    cpl_vector_unwrap(RON_vector);
    //espdr_msg("Mean of RON: %lf", RON_mean);

    *totalCosmics_RE = total_cosmics;
    espdr_msg_debug("total cosmics (RE) = %d", *totalCosmics_RE);

    *RON_RE = RON_mean;
    espdr_msg_debug("RON_RE = %lf", *RON_RE);

    //espdr_msg_debug("total cosmics: %d", total_cosmics);

    //cpl_free(data_vector);
    cpl_free(master_vector);

    cpl_free(RON_image);

    return cpl_error_get_code();
}


cpl_error_code espdr_stack_sigma_one_output_median(const cpl_imagelist *input_imagelist,
                                            const int nx,
                                            const int ny,
                                            const double ksigma,
                                            const int max_iter,
                                            cpl_image **master_image_RE,
                                            double *RON_RE,
                                            int *totalCosmics_RE) {

	/* Loop indices */
	int i = 0;
	int point = 0;
	int image_size = nx*ny;

	/* Flag indicating to reject (or not) the 0, when computing the MAD */
	/* Not used anymore */
	int reject_zeroMAD = 1;
	/* Instead w force the MAD to another value if it is == 0.0 */
	double forced_MAD = 1.0;

	/* error code */
	cpl_error_code my_error;

	espdr_msg_debug("Starting espdr_stack_sigma with kSigma = %lf, sigma_clipping_method = %s",
			ksigma, "median");

	espdr_msg_debug("Images size: nx = %d, ny = %d", nx, ny);

	espdr_ensure(input_imagelist == NULL, CPL_ERROR_NULL_INPUT,
			   "Input images list is NULL");

	/* Number of frames (FITS files) */
	int frames_nb = cpl_imagelist_get_size(input_imagelist);

	espdr_msg_debug("On %d frames", frames_nb);

	for (i = 0; i < frames_nb; i++) {
		espdr_ensure(cpl_imagelist_get_const(input_imagelist, i) == NULL,
				   CPL_ERROR_NULL_INPUT,
				   "Input image %d in the input list is NULL", i);
	}

	/* data tables extracted from the input images list */
	const double *images_table[frames_nb];

	/* vector of the same pixel in all the frames */
	//double *data_vector = (double *)cpl_calloc(frames_nb, sizeof(double));

	/* master image data table */
	double *master_vector = (double *)cpl_calloc(nx*ny, sizeof(double));

    /* master image crated */
    cpl_image *master_image;

	/* center of data: mean or MAD */
	double CoD = 0.0;

	/* number of cosmics rejected */
	int cosmics = 0;
	int total_cosmics = 0;
	int total_cosmics_tmp = 0;
	/* sigma value for rejecting cosmics */
	double sigma = 0.0;

	/* mean of the vector */
	double mean = 0.0;

	/* standard deviation of the vector */
	double stddev = 0.0;

	/* bounds for rejecting outliers */
	double reject_low = -1.0;
	double reject_high = -1.0;

	/* RON for each point of the CCD */
	double *RON_image = (double *)cpl_calloc(nx*ny, sizeof(double));
	/* resulting RON */
	double RON_mean;


	/* get the data from images list */
	//espdr_msg("New output mean:");
	for (i = 0; i < frames_nb; i++) {
		images_table[i] =
		cpl_image_get_data_double_const(cpl_imagelist_get_const(input_imagelist, i));
		//espdr_msg("Frame %d mean: %f", i,
		//	cpl_image_get_mean(cpl_imagelist_get(input_imagelist, i)));
	}

	/* Calculate the master bias for all the image points with cosmics rejection
	 by sigma clipping */
	double *data_vector;
    HDRL_OMP(omp parallel private(data_vector,total_cosmics_tmp,point,i,my_error,CoD,sigma,cosmics,mean,stddev,reject_low,reject_high))
	{
	/* vector of the same pixel in all the frames */
	data_vector = (double *)cpl_calloc(frames_nb, sizeof(double));
	cpl_vector *data_vector_wrapped = cpl_vector_wrap(frames_nb, data_vector);
	total_cosmics_tmp = 0;
    HDRL_OMP(omp for)
	for (point = 0; point < image_size; point++) {
	    /* vector of the same pixel in all the frames */
	    //data_vector = (double *)cpl_calloc(frames_nb, sizeof(double));
		for (i = 0; i < frames_nb; i++) {
			data_vector[i] = images_table[i][point];
		}

		//cpl_vector *data_vector_wrapped = cpl_vector_wrap(frames_nb, data_vector);
		my_error = espdr_sig_clip_method_median(data_vector_wrapped, ksigma, ksigma,
                                  max_iter, reject_zeroMAD, forced_MAD,
                                  &CoD, &sigma, &cosmics,
                                  &mean, &stddev,
                                  &reject_low, &reject_high);
		/*
		espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_sig_clip failed");
		*/

		master_vector[point] = mean;
		RON_image[point] = stddev;
		total_cosmics_tmp = total_cosmics_tmp + cosmics;
	}
	cpl_vector_unwrap(data_vector_wrapped);
	cpl_free(data_vector);
    HDRL_OMP(omp critical)
	{
        //HDRL_OMP(omp atomic)
		total_cosmics = total_cosmics + total_cosmics_tmp;
	}

		/* test on ESO sigma clipping same speed as my sigma_clipping,
		 even slower and rejected over 5 mln cosmics */
		/*
		 double mean_ks = 0.0;
		 double mean_ks_err = 0.0;
		 cpl_size naccepted = 0;
		 double reject_low = 0.0;
		 double reject_high = 0.0;
		 cpl_vector *my_vector = cpl_vector_wrap(frames_nb, data_vector);
		 cpl_vector *my_vector_err = cpl_vector_wrap(frames_nb, error_vector);
		 my_error = drl_kappa_sigma_clip(my_vector, my_vector_err, 4.0, 4.0, frames_nb,
		 &mean_ks, &mean_ks_err, &naccepted, &reject_low, &reject_high);
		 master_bias_vector[point] = mean_ks;
		 RON_image[point] = mean_ks_err;
		 total_cosmics = total_cosmics + frames_nb - naccepted;
		 */
	}

	/* Not sure if the image is copied into the list or
	 only the pointer is inserted: image is copied */
	master_image = cpl_image_wrap_double(nx, ny, master_vector);
    *master_image_RE = cpl_image_duplicate(master_image);
    cpl_image_unwrap(master_image);

    cpl_vector *RON_vector = cpl_vector_wrap(nx*ny, RON_image);
    RON_mean = cpl_vector_get_mean(RON_vector);
    cpl_vector_unwrap(RON_vector);
	//espdr_msg("Mean of RON: %lf", RON_mean);

	*totalCosmics_RE = total_cosmics;
	espdr_msg_debug("total cosmics (RE) = %d", *totalCosmics_RE);

	*RON_RE = RON_mean;
	espdr_msg_debug("RON_RE = %lf", *RON_RE);

	//espdr_msg_debug("total cosmics: %d", total_cosmics);

	//cpl_free(data_vector);
    cpl_free(master_vector);

	cpl_free(RON_image);

	return cpl_error_get_code();
}


cpl_error_code espdr_stack_sigma_one_output_median_with_pxl_nb(const cpl_imagelist *input_imagelist,
                                            const int nx,
                                            const int ny,
                                            const double ksigma,
                                            const int max_iter,
                                            cpl_image **master_image_RE,
                                            cpl_image **master_pxl_nb_image_RE,
                                            double *RON_RE,
                                            int *totalCosmics_RE) {

    /* Loop indices */
    int i = 0;
    int point = 0;
    int image_size = nx*ny;

    /* Flag indicating to reject (or not) the 0, when computing the MAD */
    /* Not used anymore */
    int reject_zeroMAD = 1;
    /* Instead w force the MAD to another value if it is == 0.0 */
    double forced_MAD = 1.0;

    /* error code */
    cpl_error_code my_error;

    espdr_msg_debug("Starting espdr_stack_sigma with kSigma = %lf, sigma_clipping_method = %s",
            ksigma, "mean");

    espdr_msg_debug("Images size: nx = %d, ny = %d", nx, ny);

    espdr_ensure(input_imagelist == NULL, CPL_ERROR_NULL_INPUT,
               "Input images list is NULL");

    /* Number of frames (FITS files) */
    int frames_nb = cpl_imagelist_get_size(input_imagelist);

    espdr_msg_debug("On %d frames", frames_nb);

    for (i = 0; i < frames_nb; i++) {
        espdr_ensure(cpl_imagelist_get_const(input_imagelist, i) == NULL,
                   CPL_ERROR_NULL_INPUT,
                   "Input image %d in the input list is NULL", i);
    }

    /* data tables extracted from the input images list */
    const double *images_table[frames_nb];

    /* vector of the same pixel in all the frames */
    //double *data_vector = (double *)cpl_calloc(frames_nb, sizeof(double));

    /* master image data table */
    double *master_vector = (double *)cpl_calloc(nx*ny, sizeof(double));
    /* master image of valid (non cosmics) pixels number data table */
    int *master_pxl_nb_vector = (int *)cpl_calloc(nx*ny, sizeof(int));
    
    /* master image created */
    cpl_image *master_image;
    /* master valid (non cosmics) pixels number image created */
    cpl_image *master_pxl_nb_image;
    
    /* center of data: mean or MAD */
    double CoD = 0.0;

    /* number of cosmics rejected */
    int cosmics = 0;
    int total_cosmics = 0;
    int total_cosmics_tmp = 0;
    /* sigma value for rejecting cosmics */
    double sigma = 0.0;

    /* mean of the vector */
    double mean = 0.0;

    /* standard deviation of the vector */
    double stddev = 0.0;

    /* bounds for rejecting outliers */
    double reject_low = -1.0;
    double reject_high = -1.0;

    /* RON for each point of the CCD */
    double *RON_image = (double *)cpl_calloc(nx*ny, sizeof(double));
    /* resulting RON */
    double RON_mean;


    /* get the data from images list */
    //espdr_msg("New output mean:");
    for (i = 0; i < frames_nb; i++) {
        images_table[i] =
        cpl_image_get_data_double_const(cpl_imagelist_get_const(input_imagelist, i));
        //espdr_msg("Frame %d mean: %f", i,
        //    cpl_image_get_mean(cpl_imagelist_get(input_imagelist, i)));
    }

    /* Calculate the master bias for all the image points with cosmics rejection
     by sigma clipping */
    double *data_vector;
    HDRL_OMP(omp parallel private(data_vector,total_cosmics_tmp,point,i,my_error,CoD,sigma,cosmics,mean,stddev,reject_low,reject_high))
    {
    /* vector of the same pixel in all the frames */
    data_vector = (double *)cpl_calloc(frames_nb, sizeof(double));
    cpl_vector *data_vector_wrapped = cpl_vector_wrap(frames_nb, data_vector);
    total_cosmics_tmp = 0;
    HDRL_OMP(omp for)
    for (point = 0; point < image_size; point++) {
        /* vector of the same pixel in all the frames */
        //data_vector = (double *)cpl_calloc(frames_nb, sizeof(double));
        for (i = 0; i < frames_nb; i++) {
            data_vector[i] = images_table[i][point];
        }

        //cpl_vector *data_vector_wrapped = cpl_vector_wrap(frames_nb, data_vector);
        my_error = espdr_sig_clip_method_median(data_vector_wrapped, ksigma, ksigma,
                                  max_iter, reject_zeroMAD, forced_MAD,
                                  &CoD, &sigma, &cosmics,
                                  &mean, &stddev,
                                  &reject_low, &reject_high);
        /*
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_sig_clip failed");
        */

        master_vector[point] = mean;
        master_pxl_nb_vector[point] = frames_nb-cosmics;
        RON_image[point] = stddev;
        total_cosmics_tmp = total_cosmics_tmp + cosmics;
    }
    cpl_vector_unwrap(data_vector_wrapped);
    cpl_free(data_vector);
    HDRL_OMP(omp critical)
    {
        //HDRL_OMP(omp atomic)
        total_cosmics = total_cosmics + total_cosmics_tmp;
    }
        /* test on ESO sigma clipping same speed as my sigma_clipping,
         even slower and rejected over 5 mln cosmics */
        /*
         double mean_ks = 0.0;
         double mean_ks_err = 0.0;
         cpl_size naccepted = 0;
         double reject_low = 0.0;
         double reject_high = 0.0;
         cpl_vector *my_vector = cpl_vector_wrap(frames_nb, data_vector);
         cpl_vector *my_vector_err = cpl_vector_wrap(frames_nb, error_vector);
         my_error = drl_kappa_sigma_clip(my_vector, my_vector_err, 4.0, 4.0, frames_nb,
         &mean_ks, &mean_ks_err, &naccepted, &reject_low, &reject_high);
         master_bias_vector[point] = mean_ks;
         RON_image[point] = mean_ks_err;
         total_cosmics = total_cosmics + frames_nb - naccepted;
         */
    }

    /* Not sure if the image is copied into the list or
     only the pointer is inserted: image is copied */
    master_image = cpl_image_wrap_double(nx, ny, master_vector);
    *master_image_RE = cpl_image_duplicate(master_image);
    cpl_image_unwrap(master_image);
    master_pxl_nb_image = cpl_image_wrap_int(nx, ny, master_pxl_nb_vector);
    *master_pxl_nb_image_RE = cpl_image_duplicate(master_pxl_nb_image);
    cpl_image_unwrap(master_pxl_nb_image);
    
    cpl_vector *RON_vector = cpl_vector_wrap(nx*ny, RON_image);
    RON_mean = cpl_vector_get_mean(RON_vector);
    cpl_vector_unwrap(RON_vector);
    //espdr_msg("Mean of RON: %lf", RON_mean);

    *totalCosmics_RE = total_cosmics;
    espdr_msg_debug("total cosmics (RE) = %d", *totalCosmics_RE);

    *RON_RE = RON_mean;
    espdr_msg_debug("RON_RE = %lf", *RON_RE);

    //espdr_msg_debug("total cosmics: %d", total_cosmics);

    //cpl_free(data_vector);
    cpl_free(master_vector);

    cpl_free(RON_image);

    return cpl_error_get_code();
}


/*---------------------------------------------------------------------------*/
/**
 @brief    Compute cosmics occurence rates and save the QC keywords
 @param         total_cosmic_nb total number of cosmics
 @param         CCD_geom        CCD geometry
 @param         qc_kws          KWs names
 @param         frames_nb       number of frames
 @param         exposure_time   exposure time
 @param[out]    keywords_RE     propertylist containing computed QC
 @return   CPL_ERROR_NONE iff OK

 TODO:
 1) the constant 512 should be set as a global
 --> DSO to AMO: I see no point to do this
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_cosmics_rates_prepare(const int *total_cosmics_nb,
                                           const espdr_CCD_geometry *CCD_geom,
                                           espdr_qc_keywords *qc_kws,
                                           const int frames_nb,
                                           const double exposure_time,
                                           cpl_propertylist **keywords_RE) {
    
    cpl_error_code my_error = CPL_ERROR_NONE;
    double rate_hit_pixel[CCD_geom->ext_nb], rate_hit_cm2[CCD_geom->ext_nb];
    char *new_keyword = NULL;
    int i;
    char comment[COMMENT_LENGTH];
    
    for (i = 0; i < CCD_geom->ext_nb; i++) {
        my_error = espdr_cosmics_rates(total_cosmics_nb[i],
                                       CCD_geom->exts[i].CCD_nx,
                                       CCD_geom->exts[i].CCD_ny,
                                       CCD_geom->exts[i].pxl_size,
                                       frames_nb, exposure_time,
                                       &(rate_hit_pixel[i]),
                                       &(rate_hit_cm2[i]));
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_cosmics_rates failed: %s",
                     cpl_error_get_message_default(my_error));
        
        espdr_msg("Cosmics rate per pixel: %lf", rate_hit_pixel[i]);
        espdr_msg("Cosmics rate per cm2: %lf", rate_hit_cm2[i]);
    }
    
    for (i = 0; i < CCD_geom->ext_nb; i++) {
        new_keyword =
            espdr_add_output_index_to_keyword(qc_kws->qc_cosmic_rate_pix_kw_first,
                                              qc_kws->qc_cosmic_rate_pix_kw_last,
                                              i);
        sprintf(comment, "[hits|pixel|hour] for ext %d", i);
        my_error = espdr_keyword_add_double(new_keyword,
                                            rate_hit_pixel[i],
                                            comment,
                                            keywords_RE);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "Add keyword QC_COSMIC_RATE_PIX_KW failed: %s",
                     cpl_error_get_message_default(my_error));
    
        cpl_free(new_keyword);
        
        new_keyword =
        espdr_add_output_index_to_keyword(qc_kws->qc_cosmic_rate_cm2_kw_first,
                                          qc_kws->qc_cosmic_rate_cm2_kw_last,
                                          i);
        sprintf(comment, "[hits|cm2|hour] for ext %d", i);
        my_error = espdr_keyword_add_double(new_keyword,
                                            rate_hit_cm2[i],
                                            comment,
                                            keywords_RE);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "Add keyword QC_COSMIC_RATE_CM2_KW failed: %s",
                     cpl_error_get_message_default(my_error));
        
        cpl_free(new_keyword);
    }
    
    return cpl_error_get_code();
}


/*---------------------------------------------------------------------------*/
/**
 @brief    Compute cosmics occurence rates.
 @param         total_cosmics_nb    total number of cosmics
 @param         nx                  x image size
 @param         ny                  y image size
 @param         pixel_size          size of a pixel
 @param         Texp                exposure time
 @param[out]    rate_hit_pixel_RE   hit rate per pixel
 @param[out]    rate_hit_cm2_RE     hit rate per cm2
 @return   CPL_ERROR_NONE iff OK
 
 TODO: do we really need a function with 8 input parameters to compute two numbers?
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_cosmics_rates(const int total_cosmics_nb,
                                   const int nx, const int ny,
								   const int pixel_size,
                                   const int frames_nb,
                                   const double Texp,
                                   double *rate_hit_pixel_RE,
                                   double *rate_hit_cm2_RE) {
	
	espdr_msg_debug("total cosmics nb = %d", total_cosmics_nb);
	espdr_msg_debug("Texp = %lf", Texp);
	
	int total_pixels_nb = nx * ny * frames_nb;
	
	*rate_hit_pixel_RE = (total_cosmics_nb/(Texp/NB_SEC_IN_HOUR))
                            /total_pixels_nb;
	*rate_hit_cm2_RE = *rate_hit_pixel_RE *
                        (NB_MC_IN_CM/pixel_size)*(NB_MC_IN_CM/pixel_size);
	
    espdr_msg_debug("Cosmics rate per pixel: %lf", *rate_hit_pixel_RE);
    espdr_msg_debug("Cosmics rate per cm2: %lf", *rate_hit_cm2_RE);
    
	return cpl_error_get_code();
}



/*---------------------------------------------------------------------------*/
/**
 @brief    Prepare the computation of hot pixels and save the QC keywords
 @param         master_dark_list        input dark images list
 @param         CCD_geom                CCD geometry structure
 @param         DARK_param              DARK parameter structure
 @param         qc_kws                  KWs names
 @param         inst_config             instrument config
 @param         exposure_time           master dark exposure time
 @param[out]    hot_pixel_mask_list_RE  list of computed hot pixel masks
 @param[out]    keywords_RE             fits header to hold QC
 @param         use_hdrl                HDRL use flag
 @return   CPL_ERROR_NONE iff OK

 TODO:
 1) parameter keywords_RE can be passed as a single pointer parameter
 2) 512 could be a general key. comment max length (in FITS header) should be 80
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_hot_pixels_prepare(cpl_imagelist *master_dark_list,
                                        espdr_CCD_geometry *CCD_geom,
                                        espdr_qc_keywords *qc_kws,
                                        espdr_inst_config *inst_config,
                                        const double exposure_time,
                                        double *RON_DARK,
                                        cpl_imagelist **hot_pixel_mask_list_RE,
                                        cpl_propertylist **keywords_RE,
                                        const int use_hdrl) {

    int total_output_nb = CCD_geom->total_output_nb;
    double mean[total_output_nb];
    int hot_pixels[total_output_nb];
    double sigma[total_output_nb];
        cpl_error_code my_error = CPL_ERROR_NONE;
    int i, j, k, index;
    char *new_keyword = NULL;
    char comment[COMMENT_LENGTH];

    if(use_hdrl) {
      my_error = espdr_hdrl_hot_pixels(master_dark_list,
                                       CCD_geom, inst_config,
                                       mean, hot_pixels, sigma,
                                       hot_pixel_mask_list_RE);
    } else {
        my_error = espdr_hot_pixels(master_dark_list,
                                    CCD_geom, inst_config,
                                    mean, hot_pixels, sigma,
                                    hot_pixel_mask_list_RE);
    }
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_hot_pixels failed: %s",
                 cpl_error_get_message_default(my_error));

    int total_hot_pixels = 0;
    for (i = 0; i < total_output_nb; i++) {
        espdr_msg_debug("mean[%d] = %lf", i, mean[i]);
        espdr_msg_debug("hot_pixels[%d] = %d", i, hot_pixels[i]);
        espdr_msg_debug("sigma[%d] = %lf", i, sigma[i]);
        total_hot_pixels = total_hot_pixels + hot_pixels[i];
    }
    espdr_msg("Total number of hot pixels: %d", total_hot_pixels);

    index = 0;
    for (i = 0; i < CCD_geom->ext_nb; i++) {
        for (j = 0; j < CCD_geom->exts[i].out_nb_x; j++) {
            for (k = 0; k < CCD_geom->exts[i].out_nb_y; k++) {
                new_keyword = espdr_add_ext_output_index_to_keyword(qc_kws->qc_out_dark_mean_kw_part,
                                                                    inst_config->prefix, i, j, k);
                sprintf(comment, "Mean[ADU|px|hour] for ext %d, out %d, %d", i, j, k);
                my_error = espdr_keyword_add_double(new_keyword,
                                                    mean[index]/(exposure_time/NB_SEC_IN_HOUR),
                                                    comment, keywords_RE);
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "Add keyword %s to the propertylist failed: %s",
                             new_keyword, cpl_error_get_message_default(my_error));
                cpl_free(new_keyword);

                new_keyword = espdr_add_ext_output_index_to_keyword(qc_kws->qc_out_dark_ron_kw_part,
                                                                    inst_config->prefix, i, j, k);
                sprintf(comment, "RON[ADU] for ext %d, out %d, %d", i, j, k);
                my_error = espdr_keyword_add_double(new_keyword, RON_DARK[index], comment, keywords_RE);
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "Add keyword %s to the propertylist failed: %s",
                             new_keyword, cpl_error_get_message_default(my_error));
                cpl_free(new_keyword);

                new_keyword = espdr_add_ext_output_index_to_keyword(qc_kws->qc_out_hotpix_nb_kw_part,
                                                                    inst_config->prefix, i, j, k);
                sprintf(comment, "Hot pixels nb for ext %d, out %d, %d", i, j, k);
                my_error = espdr_keyword_add_int(new_keyword, hot_pixels[index], comment, keywords_RE);
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "Add keyword %s to the propertylist failed: %s",
                             new_keyword, cpl_error_get_message_default(my_error));
                cpl_free(new_keyword);

                index++;
            }
        }
    }

    return cpl_error_get_code();
}

/*---------------------------------------------------------------------------*/
/**
 @brief    Compute hot pixels
 @param         master_dark_list        input dark images list
 @param         CCD_geom                CCD geometry structure
 @param         DARK_param              DARK parameter structure
 @param[out]    mean_RE                 master dark mean array
 @param[out]    hot_pixels_RE           array of computed hot pixels number
 @param[out]    sigma_RE                computed sigma array
 @param[out]    hot_pixel_mask_list_RE  list of computed hot pixel masks
 @return   CPL_ERROR_NONE iff OK

 TODO: esdr_sigma_clip should be unit tests (both using images and vectors)
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_hot_pixels(cpl_imagelist *master_dark_list,
                                espdr_CCD_geometry *CCD_geom,
                                espdr_inst_config *inst_config,
                                double *mean_RE,
                                int *hot_pixels_RE,
                                double *sigma_RE,
                                cpl_imagelist **hot_pixel_mask_list_RE) {
	
	int i, j, k, index;
	cpl_image *curr_image;
	cpl_image *hot_pixel_image;
	cpl_vector *curr_image_vector = NULL;
    int real_llx, real_lly, real_urx, real_ury;
    int used_real_nx, used_real_ny;
	//int *hot_pixel_mask = NULL;
	cpl_error_code my_error = CPL_ERROR_NONE;
	double CoD = 0.0, sigma = 0.0, mean = 0.0, stddev = 0.0;
	double low_limit = 0.0, high_limit = 0.0;
	int hot_pixels;
    /* forced MAD value in case MAD == 0.0 */
    double forced_MAD = 0.01;
	
    index = 0;
    for (i = 0; i < CCD_geom->ext_nb; i++) {
        for (j = 0; j < CCD_geom->exts[i].out_nb_x; j++) {
            for (k = 0; k < CCD_geom->exts[i].out_nb_y; k++) {
                curr_image = cpl_imagelist_get(master_dark_list, index);
                
                real_llx = CCD_geom->exts[i].outputs[j][k].real_llx;
                real_lly = CCD_geom->exts[i].outputs[j][k].real_lly;
                real_urx = CCD_geom->exts[i].outputs[j][k].real_urx;
                real_ury = CCD_geom->exts[i].outputs[j][k].real_ury;
                used_real_nx = real_urx - real_llx + 1;
                used_real_ny = real_ury - real_lly + 1;
                curr_image_vector = cpl_vector_wrap(used_real_nx * used_real_ny,
                                                    cpl_image_get_data_double(curr_image));

                my_error = espdr_sig_clip(curr_image_vector,
                                          inst_config->hotpixels_ksigma,
                                          inst_config->hotpixels_ksigma,
                                          inst_config->hotpixels_sigma_clipping_method,
                                          inst_config->hotpixels_max_iter,
                                          1, forced_MAD,
                                          &CoD, &sigma, &hot_pixels,
                                          &mean, &stddev,
                                          &low_limit, &high_limit);
                                         /*
                if (strcmp(inst_config->hotpixels_sigma_clipping_method, "median") == 0) {
                    	my_error = espdr_sig_clip_method_median(curr_image_vector,
                    			inst_config->stat_ksigma,
                				inst_config->stat_ksigma,
                				inst_config->stat_max_iter,
                				1., forced_MAD,
                				&CoD, &sigma, &hot_pixels,
                				&mean, &stddev,
                				&low_limit, &high_limit);
                    } else {
                    	my_error = espdr_sig_clip_method_mean(curr_image_vector,
                    			inst_config->stat_ksigma,
                				inst_config->stat_ksigma,
                				inst_config->stat_max_iter,
                				1., forced_MAD,
                				&CoD, &sigma, &hot_pixels,
                				&mean, &stddev,
                				&low_limit, &high_limit);

                    }
                   */


                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "espdr_sig_clip() failed");
                cpl_vector_unwrap(curr_image_vector);
                
                hot_pixel_image = cpl_image_new(used_real_nx, used_real_ny,
                                                CPL_TYPE_INT);
                
                my_error = espdr_create_mask(curr_image,
                                             used_real_nx * used_real_ny,
                                             low_limit, high_limit, HOT_PIXEL,
                                             &hot_pixel_image);
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error, "espdr_create_mask() failed");
                
                mean_RE[index] = mean;
                hot_pixels_RE[index] = hot_pixels;
                sigma_RE[index] = stddev;
                
                cpl_imagelist_set(*hot_pixel_mask_list_RE,
                                  cpl_image_duplicate(hot_pixel_image), index);
                cpl_image_delete(hot_pixel_image);
                index++;
            }
        }
    }
	
	return cpl_error_get_code();
	
}

/*---------------------------------------------------------------------------*/
/**
 @brief    Check the dark QC
 @param         CCD_geom        CCD geometry
 @param         exposure_time   exposure time
 @param         inst_config     instrument configuration structure
 @param         qc_kws          structure holdin quality control, keys
 @param[out]    keywords_RE     fits header to hold QC
 @return   CPL_ERROR_NONE iff OK

 TODO: keywords_RE does not need to be passed as double pointer parameter
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_dark_QC(const espdr_CCD_geometry *CCD_geom,
                             const double exposure_time,
                             espdr_inst_config *inst_config,
                             espdr_qc_keywords *qc_kws,
                             cpl_propertylist **keywords_RE) {
    
    cpl_error_code my_error = CPL_ERROR_NONE;
    int total_outputs = CCD_geom->total_output_nb;
    int i, j, k,  index;
    double mean[total_outputs];
    int hot_pixels[total_outputs];
    double rate_hit_pixel[CCD_geom->ext_nb];
	int cosmics_level_QC = 1;
    int dark_level_QC = 1;
	int hot_pixels_QC = 1;
    char *new_keyword = NULL;

    
	for (i = 0; i < CCD_geom->ext_nb; i++) {
        new_keyword =
            espdr_add_output_index_to_keyword(qc_kws->qc_cosmic_rate_pix_kw_first,
                                              qc_kws->qc_cosmic_rate_pix_kw_last,
                                              i);
        rate_hit_pixel[i] = cpl_propertylist_get_double(*keywords_RE,
                                                        new_keyword);
        cpl_free(new_keyword);
        
        if (rate_hit_pixel[i] > inst_config->cosmics_limit_per_pixel) {
            cosmics_level_QC = 0;
        }
    }
    
    
    my_error = espdr_keyword_add_int(qc_kws->qc_dark_cosmic_check_kw,
                                     cosmics_level_QC,
                                     "Cosmics rate QC",
                                     keywords_RE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
            "Add keyword qc_dark_cosmic_check_kw to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));
    
    index = 0;
	for (i = 0; i < CCD_geom->ext_nb; i++) {
		for (j = 0; j < CCD_geom->exts[i].out_nb_x; j++) {
			for (k = 0; k < CCD_geom->exts[i].out_nb_y; k++) {
				new_keyword =
            espdr_add_ext_output_index_to_keyword(qc_kws->qc_out_dark_mean_kw_part,
                                                  inst_config->prefix,
                                                          i, j, k);
                mean[index] = cpl_propertylist_get_double(*keywords_RE,
                                                          new_keyword);
                cpl_free(new_keyword);
                
 				new_keyword =
            espdr_add_ext_output_index_to_keyword(qc_kws->qc_out_hotpix_nb_kw_part,
                                                  inst_config->prefix,
                                                          i, j, k);
                hot_pixels[index] = cpl_propertylist_get_int(*keywords_RE,
                                                             new_keyword);
                cpl_free(new_keyword);
                index++;
            }
        }
    }
    
	for (i = 0; i < total_outputs; i++) {
        if (mean[i]/(exposure_time/NB_SEC_IN_HOUR) >
                            inst_config->dark_current_limit_el) {
			dark_level_QC = 0;
		}
        if (mean[i] < 0.0) {
            dark_level_QC = 0;
        }
		if (hot_pixels[i] > inst_config->hot_pixels_limit) {
			hot_pixels_QC = 0;
		}
	}
	
    my_error = espdr_keyword_add_int(qc_kws->qc_dark_mean_check_kw,
                                     dark_level_QC,
                                     "QC on the mean dark level",
                                     keywords_RE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
            "Add keyword qc_dark_mean_check_kw to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));
    
    my_error = espdr_keyword_add_int(qc_kws->qc_dark_hotpix_check_kw,
                                     hot_pixels_QC,
                                     "QC on the hot pixels number",
                                     keywords_RE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
            "Add keyword qc_dark_hotpix_check_kw to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));
    
	int dark_QC = 1;
	if ((cosmics_level_QC == 0) || (dark_level_QC == 0) || (hot_pixels_QC == 0)) {
		dark_QC = 0;
	}
	
    my_error = espdr_keyword_add_int(qc_kws->qc_dark_check_kw,
                                     dark_QC,
                                     "QC on the DARK",
                                     keywords_RE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add keyword qc_dark_check_kw to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));
	
    return cpl_error_get_code();
}
