/*                                                                            *
 *   This file is part of the ESPRESSO Pipeline                               *
 *   Copyright (C) 2006 European Southern Observatory                         *
 *                                                                            *
 *   This library is free software; you can redistribute it and/or modify     *
 *   it under the terms of the GNU General Public License as published by     *
 *   the Free Software Foundation; either version 2 of the License, or        *
 *   (at your option) any later version.                                      *
 *                                                                            *
 *   This program is distributed in the hope that it will be useful,          *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 *   GNU General Public License for more details.                             *
 *                                                                            *
 *   You should have received a copy of the GNU General Public License        *
 *   along with this program; if not, write to the Free Software              *
 *   Foundation, 51 Franklin St, Fifth Floor, Boston, MA  02111-1307  USA     *
 *                                                                            */

/*
 * $Author: dsosnows $
 * $Date: 2013-08-30 15:13:41 $
 * $Revision: 1.3 $
 * $Name: not supported by cvs2svn $
 */


#include <espdr_detector_signature.h>

#include <hdrl.h>
#include <sys/time.h>

static struct timeval tv1, tv2;

/*----------------------------------------------------------------------------
 Functions code
 ----------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/**
 @brief    Check flat saturation
 @param
 @param
 @return   CPL_ERROR_NONE iff OK
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_check_saturation(cpl_imagelist *input_iml,
                                      espdr_CCD_geometry *CCD_geom,
                                      espdr_inst_config *inst_config,
                                      cpl_imagelist *pixels_mask,
                                      double cosmics_part,
                                      double *max_flux,
                                      int print_max_flux,
                                      cpl_imagelist **sat_mask_RE) {
    
    cpl_error_code my_error = CPL_ERROR_NONE;
    
    /* Computing max flux on raw images */
    my_error = espdr_get_max_flux_ignoring_cosmics(input_iml,
                                                   pixels_mask,
                                                   cosmics_part,
                                                   max_flux);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Getting max flux failed: %s",
                 cpl_error_get_message_default(my_error));
    
    if (print_max_flux == 1) {
        espdr_msg("MAX FLUX:");
        int index = 0;
        int i, j, k;
        for (i = 0; i < CCD_geom->ext_nb; i++) {
            for (j = 0; j < CCD_geom->exts[i].out_nb_x; j++) {
                for (k = 0; k < CCD_geom->exts[i].out_nb_y; k++) {
                    espdr_msg("Flux of extension %d output [%d,%d] is %lf",
                              i, j, k, max_flux[index]);
                    if (max_flux[index] >= inst_config->satur_limit_adu) {
                        espdr_msg_warning(ANSI_COLOR_RED
                                          "Extension %d ouptput [%d, %d] is saturated, max flux = %lf"
                                          ANSI_COLOR_RESET, i, j, k, max_flux[index]);
                    }
                    index++;
                }
            }
        }
    }
    
    cpl_imagelist *saturation_mask = cpl_imagelist_new();
    cpl_imagelist *merged_saturation_mask = cpl_imagelist_new();
    cpl_imagelist *geometry_corrected_saturation_mask = cpl_imagelist_new();
    
    my_error = espdr_create_saturation_mask(input_iml,
                                            pixels_mask,
                                            inst_config->satur_limit_adu,
                                            &saturation_mask);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_create_saturation_mask failed: %s",
                 cpl_error_get_message_default(my_error));
        
    my_error = espdr_image_merge_2_dim(saturation_mask, CCD_geom, CPL_TYPE_INT,
                                       &merged_saturation_mask);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_image_merge_2_dim failed: %s",
                 cpl_error_get_message_default(my_error));
    
    my_error = espdr_correct_geometry(merged_saturation_mask, CCD_geom,
                                      geometry_corrected_saturation_mask);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_correct_geometry failed: %s",
                 cpl_error_get_message_default(my_error));
    
    *sat_mask_RE = cpl_imagelist_duplicate(geometry_corrected_saturation_mask);
    
    cpl_imagelist_delete(saturation_mask);
    cpl_imagelist_delete(merged_saturation_mask);
    cpl_imagelist_delete(geometry_corrected_saturation_mask);
    
    return (cpl_error_get_code());
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Create the saturation mask
 @param         input_imagelist         input images to be check for saturation
 @param         hot_bad_pixels_mask     mask of hot and bad pixels
 @param         satur_limit_adu         detector saturation limit
 @param[out]    saturation_mask_RE      computed saturation mask
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_create_saturation_mask(cpl_imagelist *input_imagelist,
                                            cpl_imagelist *hot_bad_pixels_mask,
                                            double satur_limit_adu,
                                            cpl_imagelist **saturation_mask_RE) {
    
    cpl_error_code my_error = CPL_ERROR_NONE;
    int i = 0, j = 0;
    cpl_image *curr_image = NULL;
    cpl_image *curr_mask = NULL;
    cpl_image *satur_mask = NULL;
    double *curr_image_data = NULL;
    int *curr_mask_data = NULL;
    int *satur_data = NULL;
    int size_x = 0, size_y = 0;
    
    espdr_msg("Creating saturation mask");
    
    int input_size_x = cpl_image_get_size_x(cpl_imagelist_get(input_imagelist, 0));
    int input_size_y = cpl_image_get_size_y(cpl_imagelist_get(input_imagelist, 0));
    int HB_size_x = cpl_image_get_size_x(cpl_imagelist_get(hot_bad_pixels_mask, 0));
    int HB_size_y = cpl_image_get_size_y(cpl_imagelist_get(hot_bad_pixels_mask, 0));
    espdr_msg("input size: %d x %d", input_size_x, input_size_y);
    espdr_msg("HB size: %d x %d", HB_size_x, HB_size_y);
    
    espdr_ensure(input_imagelist == NULL, CPL_ERROR_NULL_INPUT,
                 "Input imagelist is NULL");
    espdr_ensure(hot_bad_pixels_mask == NULL, CPL_ERROR_NULL_INPUT,
                 "Input imagelist is NULL");
    espdr_ensure(cpl_imagelist_get_size(input_imagelist) !=
                 cpl_imagelist_get_size(hot_bad_pixels_mask),
                 CPL_ERROR_INCOMPATIBLE_INPUT,
                 "The input imagelist and hot_bad_pixels_mask have different sizes.");
    
    size_x = cpl_image_get_size_x(cpl_imagelist_get(input_imagelist, 0));
    size_y = cpl_image_get_size_y(cpl_imagelist_get(input_imagelist, 0));
    satur_data = (int *) cpl_calloc (size_x * size_y, sizeof(int));
    
    for (i = 0; i < cpl_imagelist_get_size(input_imagelist); i++) {
        curr_image = cpl_imagelist_get(input_imagelist, i);
        curr_mask = cpl_imagelist_get(hot_bad_pixels_mask, i);
        
        curr_image_data = cpl_image_get_data_double(curr_image);
        curr_mask_data = cpl_image_get_data_int(curr_mask);
        
        for (j = 0; j < size_x * size_y; j++) {
            if (curr_mask_data[j] == 0) {
                if (curr_image_data[j] >= satur_limit_adu) {
                    satur_data[j] = AD_CONVERTER_SATURATION;
                } else {
                    satur_data[j] = 0;
                }
            } else {
                satur_data[j] = 0;
            }
        }
        
        satur_mask = cpl_image_wrap_int(size_x, size_y, satur_data);
        my_error = cpl_imagelist_set(*saturation_mask_RE,
                                     cpl_image_duplicate(satur_mask), i);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "cpl_imagelist_set failed");
        cpl_image_unwrap(satur_mask);
    }
    
    cpl_free(satur_data);
    
    return cpl_error_get_code();
}





/*----------------------------------------------------------------------------*/
/**
 @brief Remove the detector signature one output
 @param         input_image         input raw images
 @param         master_bias_image   master bias list
 @param         keywords            raw image keywords
 @param         keywords_MB         master bias keywords
 @param         keywords_HP         hot pixels keywords
 @param         keywords_HP         bad pixels keywords
 @param         qc_kws              KWs names
 @param         inst_config         instrument config
 @param         ext_no              extension number
 @param         out_x               horizontal output number
 @param         out_y               vertical output number
 @param         CCD_geom            CCD geometry structure
 @param         OVSC_param          overscan parameter structure
 @param[out]    CONAD_RE            returned CONAD
 @param[out]    DARK_RE             returned dark current
 @param[out]    RON_RE              returned RON
 @param[out]    input_corrected_RE  CCD corrected frames
 @return    CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_remove_det_signature_one_output(cpl_image *input_image,
        const cpl_image *orders_mask_image,
		const cpl_image *master_bias_res_image,
        cpl_image *master_dark_image,
		cpl_propertylist *keywords,
		cpl_propertylist *keywords_HP,
		cpl_propertylist *keywords_BP,
		espdr_qc_keywords *qc_kws,
		espdr_inst_config *inst_config,
		const int remove_bias_res,
		const int ext_no,
		const int out_x,
		const int out_y,
		const espdr_CCD_geometry *CCD_geom,
		double *CONAD_RE,
		double *DARK_RE,
        double *DARK_RON_RE,
		double *RON_RE,
		cpl_image **input_corrected_RE) {

	cpl_error_code my_error = CPL_ERROR_NONE;
	cpl_image *bias_corrected_input = NULL;
    cpl_image *cte_corrected_input = NULL;
	cpl_image *dark_corrected_input = NULL;
	cpl_image *conad_applied_input = NULL;
	double exp_time_hour = 0.0;
	double conad, dark, dark_ron, ron;
    int nx, ny;
    
    espdr_ensure(input_image == NULL, CPL_ERROR_NULL_INPUT,
                 "input image is NULL");
    espdr_ensure(keywords == NULL, CPL_ERROR_NULL_INPUT,
                 "raw frame keywords list is NULL");
    espdr_ensure(keywords_HP == NULL, CPL_ERROR_NULL_INPUT,
                 "hot pixels keywords list is NULL");
    espdr_ensure(keywords_BP == NULL, CPL_ERROR_NULL_INPUT,
                 "bad pixels keywords list is NULL");
    espdr_ensure(CCD_geom == NULL, CPL_ERROR_NULL_INPUT,
                 "CCD geometry parameter is NULL");

    if (inst_config->inst_type == NIR) {
        my_error = espdr_preprocess_img(input_image,
                                        orders_mask_image,
                                        keywords,
                                        CCD_geom, inst_config,
                                        &ron, &conad,
                                        &bias_corrected_input);
        
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "preprocesing on imagelist failed: %s",
                     cpl_error_get_message_default(my_error));
    } else {
        my_error = espdr_correct_bias_one_output(input_image,
                                                 master_bias_res_image,
                                                 ext_no, out_x, out_y,
                                                 CCD_geom, inst_config,
                                                 remove_bias_res, &ron,
                                                 &bias_corrected_input);
        
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_correct_bias_one_output failed: %s",
                     cpl_error_get_message_default(my_error));
    }
    
    if (inst_config->cte_flag) {
        nx = CCD_geom->exts[ext_no].outputs[out_x][out_y].real_nx;
        ny = CCD_geom->exts[ext_no].outputs[out_x][out_y].real_ny;
        cte_corrected_input = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
        
        my_error = espdr_cte_one_output(bias_corrected_input, ron,
                                        ext_no, out_x, out_y,
                                        CCD_geom, inst_config,
                                        cte_corrected_input);
        
#if SAVE_DEBUG_PRODUCT_CTE
        cpl_image *diff_img = cpl_image_subtract_create(bias_corrected_input,
                                                        cte_corrected_input);
        const char *dpr_type = cpl_propertylist_get_string(keywords,
                                                           inst_config->dpr_type_kw);
        
        char diff_filename[64];
        sprintf(diff_filename, "%s_CORR_BIAS_%s.fits",
                inst_config->instrument, dpr_type);
        my_error = cpl_image_save(bias_corrected_input, diff_filename,
                                  CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "cpl_image_save of %s failed: %s",
                     diff_filename, cpl_error_get_message_default(my_error));
        espdr_msg("Image %s saved", diff_filename);
        
        sprintf(diff_filename, "%s_CORR_CTE_%s.fits",
                inst_config->instrument, dpr_type);
        my_error = cpl_image_save(cte_corrected_input, diff_filename,
                                  CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "cpl_image_save of %s failed: %s",
                     diff_filename, cpl_error_get_message_default(my_error));
        espdr_msg("Image %s saved", diff_filename);
        
        sprintf(diff_filename, "%s_DIFF_BIAS_CTE_%s.fits",
                inst_config->instrument, dpr_type);
        my_error = cpl_image_save(diff_img, diff_filename,
                                  CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "cpl_image_save of %s failed: %s",
                     diff_filename, cpl_error_get_message_default(my_error));
        espdr_msg("Image %s saved", diff_filename);
        
        cpl_image_delete(diff_img);
#endif
    } else {
        cte_corrected_input = bias_corrected_input;
    }

	/* Extract exp_time from keywords */
	exp_time_hour = cpl_propertylist_get_double(keywords,
                                                inst_config->Texp_kw) / NB_SEC_IN_HOUR;

    my_error = espdr_remove_dark_current_one_output(cte_corrected_input,
                                                    master_dark_image,
                                                    keywords_HP, exp_time_hour,
                                                    ext_no, out_x, out_y,
                                                    CCD_geom, qc_kws, inst_config,
                                                    &dark, &dark_ron, &dark_corrected_input);
    
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_remove_dark_current_one_output failed: %s",
                 cpl_error_get_message_default(my_error));
    
    if (inst_config->inst_type == VIS) {
        // for NIRPS the multiplying by conad is done within the preprocesing
        my_error = espdr_apply_gain_one_output(dark_corrected_input,
                                               keywords_BP,
                                               ext_no, out_x, out_y,
                                               CCD_geom, qc_kws, inst_config,
                                               &conad, &conad_applied_input);
        
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_apply_gain_one_output failed: %s",
                     cpl_error_get_message_default(my_error));
        
        *input_corrected_RE = cpl_image_duplicate(conad_applied_input);
    } else {
        *input_corrected_RE = cpl_image_duplicate(dark_corrected_input);
    }
    *CONAD_RE = conad;
    *DARK_RE = dark;
    *DARK_RON_RE = dark_ron;
    *RON_RE = ron;
    
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "conad_applied_input duplication into input_corrected_RE failed: %s",
                 cpl_error_get_message_default(my_error));
    
    cpl_image_delete(bias_corrected_input);
    if (inst_config->cte_flag) {
        cpl_image_delete(cte_corrected_input);
    }
    cpl_image_delete(dark_corrected_input);
    cpl_image_delete(conad_applied_input);
    
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief Remove the detector signature
 @param         input_imagelist         input raw images
 @param         master_bias_imagelist   master bias list
 @param         keywords                raw frame keywords
 @param         keywords_MB             master bias keywords
 @param         keywords_HP             hot pixels keywords
 @param         keywords_HP             bad pixels keywords
 @param         qc_kws                  KWs names
 @param         inst_config             instrument config
 @param         CCD_geom                CCD geometry structure
 @param         OVSC_param              overscan parameter structure
 @param[out]    RON                     returned RON
 @param[out]    DARK                    returned dark current
 @param[out]    CONAD                   returned conversion factor
 @param[out]    input_corrected_RE      CCD corrected frames
 @return    CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_remove_det_signature(cpl_imagelist *input_imagelist,
        const cpl_imagelist *orders_mask_imagelist,
		const cpl_imagelist *master_bias_imagelist,
        cpl_imagelist *master_dark_imagelist,
		cpl_propertylist *keywords,
		cpl_propertylist *keywords_HP,
		cpl_propertylist *keywords_BP,
		espdr_qc_keywords *qc_kws,
		espdr_inst_config *inst_config,
		const int remove_bias_res,
		const espdr_CCD_geometry *CCD_geom,
		const cpl_type image_type,
		double *RON,
		double *DARK,
        double *DARK_RON,
		double *CONAD,
		cpl_imagelist *input_corrected_RE) {


	cpl_error_code my_error = CPL_ERROR_NONE;
	cpl_imagelist *bias_corrected_input = NULL;
    cpl_imagelist *cte_corrected_input = NULL;
	cpl_imagelist *dark_corrected_input = NULL;
	cpl_imagelist *conad_applied_input = NULL;
	cpl_imagelist *ext_components_images = NULL;
	cpl_imagelist *full_ext_images = NULL;
	cpl_imagelist *all_extensions = NULL;
	double exp_time_hour = 0.0;
	int i, j, k, l, frames_nb;
	int index_all, index_one_image;
	int all_size;

    espdr_ensure(input_imagelist == NULL, CPL_ERROR_NULL_INPUT,
                 "input imagelist is NULL");
    espdr_ensure(keywords == NULL, CPL_ERROR_NULL_INPUT,
                 "raw frame keywords list is NULL");
    espdr_ensure(keywords_HP == NULL, CPL_ERROR_NULL_INPUT,
                 "hot pixels keywords list is NULL");
    espdr_ensure(keywords_BP == NULL, CPL_ERROR_NULL_INPUT,
                 "bad pixels keywords list is NULL");
    espdr_ensure(CCD_geom == NULL, CPL_ERROR_NULL_INPUT,
                 "CCD geometry parameter is NULL");

	bias_corrected_input = cpl_imagelist_new();
    cte_corrected_input = cpl_imagelist_new();
	dark_corrected_input = cpl_imagelist_new();
	conad_applied_input = cpl_imagelist_new();

	/*
    cpl_image *curr_image = NULL;

    espdr_msg("Medians of input raw images");
    for (i = 0; i < cpl_imagelist_get_size(input_imagelist); i++) {
        curr_image = cpl_imagelist_get(input_imagelist, i);
        espdr_msg("\t%d: %lf", i, cpl_image_get_median(curr_image));
    }
    espdr_msg("\n");
	 */
    
    if (inst_config->inst_type == NIR) {
        espdr_msg("Preprocessing");
        double exp_time = cpl_propertylist_get_double(keywords,
                                                      inst_config->Texp_kw);
        char *keyword_to_get = espdr_add_output_index_to_keyword(inst_config->conad_kw_first_part,
                                                                 inst_config->conad_kw_last_part,
                                                                 1);
        CONAD[0] = cpl_propertylist_get_double(keywords, keyword_to_get);
        if (CONAD[0] == 0.0) CONAD[0] = 1.0;
        
        RON[0] = sqrt(400.0/(exp_time/5.57) + 36.0) / CONAD[0];
        espdr_msg("RON = %f [ADU]", RON[0]);
        
        my_error = espdr_preprocess_iml(input_imagelist,
                                        orders_mask_imagelist,
                                        CCD_geom, inst_config,
                                        exp_time, CONAD[0],
                                        &bias_corrected_input);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "preprocesing on imagelist failed: %s",
                     cpl_error_get_message_default(my_error));
        
#if SAVE_DEBUG_PRODUCT_PREPROCESSING
        for (int i = 0; i < cpl_imagelist_get_size(bias_corrected_input); i++) {
            cpl_image *curr_image = cpl_imagelist_get(bias_corrected_input, i);
            char *filename_pp2 = (char *)cpl_malloc(64*sizeof(char));
            char *filename_arc = (char *)cpl_malloc(64*sizeof(char));
            if (cpl_propertylist_has(keywords, "ARCFILE")) {
                filename_arc = cpl_propertylist_get_string(keywords, "ARCFILE");
                filename_arc[strlen(filename_arc) - 5] = '\0';
                espdr_msg("Filename of raw image: %s", filename_arc);
                sprintf(filename_pp2, "%s_prep.fits", filename_arc);
                espdr_msg("Filename of preprocessed image: %s", filename_pp2);
                my_error = cpl_image_save(curr_image,
                                          filename_pp2,
                                          CPL_TYPE_DOUBLE,
                                          keywords,
                                          CPL_IO_CREATE);
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "Image saving %s failed: %s",
                             filename_pp2, cpl_error_get_message_default(my_error));
                espdr_msg("Preprocessed image %s saved", filename_pp2);
            } else {
                espdr_msg("Can't save the preprocessed image, because the original filename is not present in the header");
            }
            cpl_free(filename_pp2);
        }
#endif
        espdr_msg("Preprocessing finished");
    } else {
        my_error = espdr_correct_bias(input_imagelist, master_bias_imagelist,
                                      inst_config, CCD_geom, remove_bias_res, RON,
                                      bias_corrected_input);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_correct_bias failed: %s",
                     cpl_error_get_message_default(my_error));
        /*
         espdr_msg("Medians of bias corrected images");
         for (i = 0; i < cpl_imagelist_get_size(bias_corrected_input); i++) {
         curr_image = cpl_imagelist_get(bias_corrected_input, i);
         espdr_msg("\t%d: %lf", i, cpl_image_get_median(curr_image));
         }
         espdr_msg("\n");
         */
    }
    
    if (inst_config->cte_flag) {
        my_error = espdr_cte_compute(bias_corrected_input, RON, CCD_geom, inst_config,
                                     cte_corrected_input);
        
#if SAVE_DEBUG_PRODUCT_CTE
        cpl_image *bias_corr_img = cpl_imagelist_get(bias_corrected_input, 0);
        cpl_image *cte_corr_img = cpl_imagelist_get(cte_corrected_input, 0);
        cpl_image *diff_img = cpl_image_subtract_create(bias_corr_img,
                                                        cte_corr_img);
        const char *dpr_type = cpl_propertylist_get_string(keywords,
                                                           inst_config->dpr_type_kw);
        
        char diff_filename[64];
        sprintf(diff_filename, "%s_CORR_BIAS_%s.fits",
                inst_config->instrument, dpr_type);
        my_error = cpl_image_save(bias_corr_img, diff_filename,
                                  CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "cpl_image_save of %s failed: %s",
                     diff_filename, cpl_error_get_message_default(my_error));
        espdr_msg("Image %s saved", diff_filename);
        
        sprintf(diff_filename, "%s_CORR_CTE_%s.fits",
                inst_config->instrument, dpr_type);
        my_error = cpl_image_save(cte_corr_img, diff_filename,
                                  CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "cpl_image_save of %s failed: %s",
                     diff_filename, cpl_error_get_message_default(my_error));
        espdr_msg("Image %s saved", diff_filename);
        
        sprintf(diff_filename, "%s_DIFF_BIAS_CTE_%s.fits",
                inst_config->instrument, dpr_type);
        my_error = cpl_image_save(diff_img, diff_filename,
                                  CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "cpl_image_save of %s failed: %s",
                     diff_filename, cpl_error_get_message_default(my_error));
        espdr_msg("Image %s saved", diff_filename);
        
        cpl_image_delete(diff_img);
#endif
    } else {
        cpl_imagelist_delete(cte_corrected_input);
        cte_corrected_input = bias_corrected_input;
    }
    

	/* Extract exp_time from keywords */
	exp_time_hour = cpl_propertylist_get_double(keywords,
			inst_config->Texp_kw) / NB_SEC_IN_HOUR;

	my_error = espdr_remove_dark_current(cte_corrected_input,
            master_dark_imagelist,
			keywords_HP, exp_time_hour,
			CCD_geom, qc_kws, inst_config, DARK, DARK_RON,
			dark_corrected_input);
	espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
			"espdr_remove_dark_current failed: %s",
			cpl_error_get_message_default(my_error));

	/*
    espdr_msg("Medians of dark corrected images");
    for (i = 0; i < cpl_imagelist_get_size(dark_corrected_input); i++) {
        curr_image = cpl_imagelist_get(dark_corrected_input, i);
        espdr_msg("\t%d: %lf", i, cpl_image_get_median(curr_image));
    }
    espdr_msg("\n");
	 */

    if (inst_config->inst_type == VIS) {
        // for NIRPS the multiplying by conversion factor is done within the preprocesing
        my_error = espdr_apply_gain(dark_corrected_input,
                                    keywords_BP,
                                    CCD_geom, qc_kws, inst_config,
                                    CONAD,
                                    conad_applied_input);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_apply_gain failed: %s",
                     cpl_error_get_message_default(my_error));
        
        /*
         espdr_msg("Medians of conad applied images");
         for (i = 0; i < cpl_imagelist_get_size(conad_applied_input); i++) {
         curr_image = cpl_imagelist_get(conad_applied_input, i);
         espdr_msg("\t%d: %lf", i, cpl_image_get_median(curr_image));
         }
         espdr_msg("\n");
         */
    } else {
        conad_applied_input = cpl_imagelist_duplicate(dark_corrected_input);
    }
	/* Merge images to have all the outputs in one,
        to correct the geometry.
	 */

	frames_nb = cpl_imagelist_get_size(conad_applied_input) /
			CCD_geom->total_output_nb;
	all_extensions = cpl_imagelist_new();
	index_all = 0;
	for (i = 0; i < frames_nb; i++) {
		ext_components_images = cpl_imagelist_new();
		full_ext_images = cpl_imagelist_new();
		index_one_image = 0;
		for (j = 0; j < CCD_geom->ext_nb; j++) {
			for (k = 0; k < CCD_geom->exts[i].out_nb_x; k++) {
				for (l = 0; l < CCD_geom->exts[i].out_nb_y; l++) {
					my_error = cpl_imagelist_set(ext_components_images,
							cpl_image_duplicate(cpl_imagelist_get(conad_applied_input,
									index_all)),
									index_one_image);
					index_all++;
					index_one_image++;
				}
			}
		}

		my_error = espdr_image_merge_2_dim(ext_components_images, CCD_geom,
				image_type, &full_ext_images);
		espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
				"espdr_image_merge_2_dim failed: %s",
				cpl_error_get_message_default(my_error));

		all_size = cpl_imagelist_get_size(all_extensions);
		for (j = 0; j < CCD_geom->ext_nb; j++) {
			my_error = cpl_imagelist_set(all_extensions,
					cpl_image_duplicate(cpl_imagelist_get(full_ext_images,
							j)),
							all_size+j);
			espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
					"Adding merged image to the imagelist failed: %s",
					cpl_error_get_message_default(my_error));
		}

		cpl_imagelist_delete(ext_components_images);
		cpl_imagelist_delete(full_ext_images);
	}
    
	// No need to trim the images, the bias correction does it
    
	my_error = espdr_correct_geometry(all_extensions,
			CCD_geom,
			input_corrected_RE);
	espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
			"espdr_correct_geometry failed: %s",
			cpl_error_get_message_default(my_error));

    if (inst_config->cte_flag) {
        cpl_imagelist_delete(cte_corrected_input);
    }
    cpl_imagelist_delete(bias_corrected_input);
    cpl_imagelist_delete(dark_corrected_input);
	cpl_imagelist_delete(conad_applied_input);
	cpl_imagelist_delete(all_extensions);

	return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief     Remove the dark current one output
 @param         input_image         input raw image
 @param         keywords            hot pixels keywords
 @param         exp_time            exposure time of raw orderdef frames
 @param         ext_no              extension number
 @param         out_x               horizontal output number
 @param         out_y               vertical output number
 @param         CCD_geom            CCD geometry structure
 @param         qc_kws              KWs names
 @param         inst_config         instrument config
 @param[out]    DARK_RE             returned dark current
 @param[out]    input_corrected_RE  dark corrected frames
 @return    CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_remove_dark_current_one_output(cpl_image *input_image,
        cpl_image *master_dark_img,
		cpl_propertylist *keywords,
		const double exp_time,
		const int ext_no,
		const int out_x,
		const int out_y,
		const espdr_CCD_geometry *CCD_geom,
		espdr_qc_keywords *qc_kws,
		espdr_inst_config *inst_config,
		double *DARK_RE,
        double *DARK_RON_RE,
		cpl_image **input_corrected_RE) {

	cpl_image *curr_image = NULL;
	cpl_image *corrected_image = NULL;
	char *dark_current_KW;
    char *dark_ron_KW;

	espdr_ensure(input_image == NULL, CPL_ERROR_NULL_INPUT,
			"Input image is NULL");
	espdr_ensure(keywords == NULL, CPL_ERROR_NULL_INPUT,
			"keywords parameter is NULL");
	espdr_ensure(CCD_geom == NULL, CPL_ERROR_NULL_INPUT,
			"CCD geometry parameter is NULL");

	dark_current_KW = espdr_add_ext_output_index_to_keyword(qc_kws->qc_out_dark_mean_kw_part,
                                                            inst_config->prefix, ext_no, out_x, out_y);
	*DARK_RE = cpl_propertylist_get_double(keywords, dark_current_KW);

    dark_ron_KW = espdr_add_ext_output_index_to_keyword(qc_kws->qc_out_dark_ron_kw_part,
                                                        inst_config->prefix, ext_no, out_x, out_y);
    if (cpl_propertylist_has(keywords, dark_ron_KW)) {
        *DARK_RON_RE = cpl_propertylist_get_double(keywords, dark_ron_KW);
    } else {
        espdr_msg_warning("No DARK RON KW in the mdark recipe products, setting the DARK RON to 0.0");
        *DARK_RON_RE = 0.0;
    }

	curr_image = cpl_image_duplicate(input_image);
    if (master_dark_img == NULL) {
        corrected_image = cpl_image_subtract_scalar_create(curr_image, exp_time*(*DARK_RE));
    } else {
        //espdr_msg("Subtracting MASTER DARK for ext %d output %dx%d",
        //          ext_no, out_x, out_y);
        double dark_time = cpl_propertylist_get_double(keywords, inst_config->Texp_kw);
        cpl_image_multiply_scalar(master_dark_img, exp_time * NB_SEC_IN_HOUR / dark_time);
        //espdr_msg("MDARK:::::: dark_time: %f, exp_time: %f, factor: %f",
        //          dark_time, exp_time*NB_SEC_IN_HOUR,
        //          exp_time * NB_SEC_IN_HOUR / dark_time);
        corrected_image = cpl_image_subtract_create(curr_image, master_dark_img);
    }
	*input_corrected_RE = cpl_image_duplicate(corrected_image);

	cpl_image_delete(curr_image);
	cpl_image_delete(corrected_image);
	cpl_free(dark_current_KW);

	return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief     Remove the dark current
 @param         input_imagelist     input raw images
 @param         keywords            hot pixels keywords
 @param         exp_time            exposure time of raw orderdef frames
 @param         CCD_geom            CCD geometry structure
 @param         qc_kws              KWs names
 @param         inst_config         instrument config
 @param[out]    DARK_RE             dark current returned
 @param[out]    input_corrected_RE  dark corrected frames
 @return    CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_remove_dark_current(cpl_imagelist *input_imagelist,
        cpl_imagelist *master_dark_iml,
		cpl_propertylist *keywords,
		const double exp_time,
		const espdr_CCD_geometry *CCD_geom,
		espdr_qc_keywords *qc_kws,
		espdr_inst_config *inst_config,
		double *DARK_RE,
        double *DARK_RON_RE,
		cpl_imagelist *input_corrected_RE) {

	const cpl_image *curr_image = NULL;
    cpl_image *curr_mdark = NULL;
	cpl_image *corrected_image = NULL;
	int i, j, k, index, fn, index_dark;
	char *dark_current_KW, *dark_ron_KW;
	int input_size = 0;
	int frames_nb = 0;

	espdr_ensure(input_imagelist == NULL, CPL_ERROR_NULL_INPUT,
			"Input imagelist is NULL");
	espdr_ensure(keywords == NULL, CPL_ERROR_NULL_INPUT,
			"keywords parameter is NULL");
	espdr_ensure(CCD_geom == NULL, CPL_ERROR_NULL_INPUT,
			"CCD geometry parameter is NULL");
	input_size = cpl_imagelist_get_size(input_imagelist);
	espdr_ensure((input_size % CCD_geom->total_output_nb) != 0,
			CPL_ERROR_INCOMPATIBLE_INPUT,
			"The number of input images is wrong: %d", input_size);

	frames_nb = (int)(input_size / CCD_geom->total_output_nb);

	//espdr_msg("Dark current & exp_time:");
	index = 0;
	for (fn = 0; fn < frames_nb; fn++) {
		index_dark = 0;
		for (i = 0; i < CCD_geom->ext_nb; i++) {
			for (j = 0; j < CCD_geom->exts[i].out_nb_x; j++) {
				for (k = 0; k < CCD_geom->exts[i].out_nb_y; k++) {
					dark_current_KW = espdr_add_ext_output_index_to_keyword(qc_kws->qc_out_dark_mean_kw_part,
                                                                            inst_config->prefix, i, j, k);
                    DARK_RE[index_dark] = cpl_propertylist_get_double(keywords, dark_current_KW);
					//espdr_msg("\t%d: %lf, %lf", index, DARK_RE[index_dark], exp_time);
                    dark_ron_KW = espdr_add_ext_output_index_to_keyword(qc_kws->qc_out_dark_ron_kw_part,
                                                                        inst_config->prefix, i, j, k);
                    if (cpl_propertylist_has(keywords, dark_ron_KW)) {
                        DARK_RON_RE[index_dark] = cpl_propertylist_get_double(keywords, dark_ron_KW);
                        //espdr_msg("\t%d: %lf, %lf", index, DARK_RON_RE[index_dark], exp_time);
                    } else {
                        espdr_msg_warning("No DARK RON KW in the mdark recipe products, setting the DARK RON to 0.0");
                        DARK_RON_RE[index_dark] = 0.0;
                    }
					curr_image = cpl_imagelist_get_const(input_imagelist, index);
                    if (master_dark_iml == NULL) {
                        corrected_image = cpl_image_subtract_scalar_create(curr_image, exp_time*DARK_RE[index_dark]);
                    } else {
                        double dark_time = cpl_propertylist_get_double(keywords, inst_config->Texp_kw);
                        curr_mdark = cpl_imagelist_get(master_dark_iml, index_dark);
                        cpl_image_multiply_scalar(curr_mdark, exp_time * NB_SEC_IN_HOUR / dark_time);
                        //espdr_msg("MDARK:::::: dark_time: %f, exp_time: %f, factor: %f",
                        //          dark_time, exp_time*NB_SEC_IN_HOUR,
                        //          exp_time * NB_SEC_IN_HOUR / dark_time);
                        corrected_image = cpl_image_subtract_create(curr_image, curr_mdark);
                    }
					cpl_imagelist_set(input_corrected_RE, cpl_image_duplicate(corrected_image), index);
					cpl_image_delete(corrected_image);

					cpl_free(dark_current_KW);
					index++;
					index_dark++;
				}
			}
		}
	}

	// output pixel = input pixel - exp_time*dark_current (QC.OUT_i_j_k.DARK.MEAN)

	return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief     Apply gain
 @param         input_image         input raw image
 @param         keywords            bad pixels keywords
 @param         ext_no              extension number
 @param         out_x               horizontal output number
 @param         out_y               vertical output number
 @param         CCD_geom            CCD geometry structure
 @param         qc_kws              KWs names
 @param         inst_config         instrument config
 @param[out]    CONAD_RE            returned conversion factor
 @param[out]    input_corrected_RE  conad applied frames
 @return    CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_apply_gain_one_output(const cpl_image *input_image,
		cpl_propertylist *keywords,
		const int ext_no,
		const int out_x,
		const int out_y,
		const espdr_CCD_geometry *CCD_geom,
		espdr_qc_keywords *qc_kws,
		espdr_inst_config *inst_config,
		double *CONAD_RE,
		cpl_image **input_corrected_RE) {

	char *conad_KW;
	double conad = 0.0;
	cpl_image *curr_image = NULL;
	cpl_error_code my_error = CPL_ERROR_NONE;

	espdr_ensure(input_image == NULL, CPL_ERROR_NULL_INPUT,
			"Input image is NULL");
	espdr_ensure(keywords == NULL, CPL_ERROR_NULL_INPUT,
			"keywords parameter is NULL");
	espdr_ensure(CCD_geom == NULL, CPL_ERROR_NULL_INPUT,
			"CCD geometry parameter is NULL");

	conad_KW = espdr_add_ext_output_index_to_keyword(qc_kws->qc_out_gain_kw_part,
			inst_config->prefix,
			ext_no, out_x, out_y);
	conad = cpl_propertylist_get_double(keywords, conad_KW);

	*CONAD_RE = conad;

	curr_image = cpl_image_duplicate(input_image);
	my_error = cpl_image_multiply_scalar(curr_image, conad);

	espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
			"cpl_image_multiply_scalar failed: %s",
			cpl_error_get_message_default(my_error));

	*input_corrected_RE = cpl_image_duplicate(curr_image);

	cpl_image_delete(curr_image);
	cpl_free(conad_KW);

	return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief     Apply gain
 @param         input_imagelist     input raw images
 @param         keywords            bad pixels keywords
 @param         CCD_geom            CCD geometry structure
 @param         qc_kws              KWs names
 @param         inst_config         instrument config
 @param[out]    CONAD_RE            returned conversion factor
 @param[out]    input_corrected_RE  conad applied frames
 @return    CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_apply_gain(const cpl_imagelist *input_imagelist,
		cpl_propertylist *keywords,
		const espdr_CCD_geometry *CCD_geom,
		espdr_qc_keywords *qc_kws,
		espdr_inst_config *inst_config,
		double *CONAD_RE,
		cpl_imagelist *input_corrected_RE) {

	char *conad_KW;
	double conad = 0.0;
	int index;
	int fn, i, j, k;
	int frames_nb = 0;
	cpl_image *curr_image = NULL;
	cpl_error_code my_error = CPL_ERROR_NONE;

	espdr_ensure(input_imagelist == NULL, CPL_ERROR_NULL_INPUT,
			"Input imagelist is NULL");
	espdr_ensure(keywords == NULL, CPL_ERROR_NULL_INPUT,
			"keywords parameter is NULL");
	espdr_ensure(CCD_geom == NULL, CPL_ERROR_NULL_INPUT,
			"CCD geometry parameter is NULL");

	frames_nb = (int)(cpl_imagelist_get_size(input_imagelist) /
			CCD_geom->total_output_nb);

	index = 0;
	for (fn = 0; fn < frames_nb; fn++) {
		for (i = 0; i < CCD_geom->ext_nb; i++) {
			for (j = 0; j < CCD_geom->exts[i].out_nb_x; j++) {
				for (k = 0; k < CCD_geom->exts[i].out_nb_y; k++) {
					conad_KW = espdr_add_ext_output_index_to_keyword(qc_kws->qc_out_gain_kw_part,
									inst_config->prefix,
									i, j, k);
					conad = cpl_propertylist_get_double(keywords, conad_KW);
                    //espdr_msg("in espdr_apply_gain, conad = %f", conad);
                    
					curr_image = (cpl_image*) cpl_imagelist_get_const(input_imagelist, index);

					my_error = cpl_image_multiply_scalar(curr_image, conad);

					espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
							"cpl_image_multiply_scalar failed: %s",
							cpl_error_get_message_default(my_error));

					cpl_imagelist_set(input_corrected_RE,
							cpl_image_duplicate(curr_image),
							index);

					cpl_free(conad_KW);
					CONAD_RE[index] = conad;
					index++;
				}
			}
		}
	}
    
	return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief     Correct geometry
 @param         input_imagelist     input raw images
 @param         CCD_geom            CCD geometry structure
 @param[out]    input_corrected_RE  corrected frames
 @return    CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_correct_geometry(const cpl_imagelist *input_imagelist,
		const espdr_CCD_geometry *CCD_geom,
		cpl_imagelist *input_corrected_RE) {

	espdr_ensure(input_imagelist == NULL, CPL_ERROR_NULL_INPUT,
			"Input imagelist is NULL");
	espdr_ensure(CCD_geom == NULL, CPL_ERROR_NULL_INPUT,
			"CCD geometry parameter is NULL");

	int i;
	const cpl_image *curr_image = NULL;
	cpl_image *curr_corrected = NULL;
	cpl_error_code my_error = CPL_ERROR_NONE;
	int ext_no = 0;
    
	for (i = 0; i < cpl_imagelist_get_size(input_imagelist); i++) {
		curr_image = cpl_imagelist_get_const(input_imagelist, i);

		curr_corrected = cpl_image_duplicate(curr_image);

		my_error = cpl_image_turn(curr_corrected,
				(int)(CCD_geom->exts[ext_no].rot_angle/
						RIGHT_ANGLE_DEG));
		espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
				"cpl_image_turn failed: %s",
				cpl_error_get_message_default(my_error));

		if (CCD_geom->exts[ext_no].ver_flip == 1) {
			my_error = cpl_image_flip(curr_corrected, 2);
			espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
					"cpl_image_flip vertical failed: %s",
					cpl_error_get_message_default(my_error));
		}

		if (CCD_geom->exts[ext_no].hor_flip == 1) {
			my_error = cpl_image_flip(curr_corrected, 0);
			espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
					"cpl_image_flip horizontal failed: %s",
					cpl_error_get_message_default(my_error));
		}

		my_error = cpl_imagelist_set(input_corrected_RE,
				cpl_image_duplicate(curr_corrected), i);
		cpl_image_delete(curr_corrected);

		ext_no = (ext_no + 1) % CCD_geom->ext_nb;
	}

	return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief     Get the max flux ignoring the cosmic part
 @param     input_imagelist     input CCD corrected raw images
 @param     pixel_mask          hot & bad pixels mask
 @param     max_flux_RE         estimated cosmics part
 @param     total_mask_iml      mask with cosmics
 @return    CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_get_max_flux_ignoring_cosmics(const cpl_imagelist *input_iml,
                                                   const cpl_imagelist *mask_iml,
                                                   const double rejected_part,
                                                   double *max_fluxes_RE) {

	espdr_ensure(input_iml == NULL, CPL_ERROR_NULL_INPUT,
			"Input imagelist is NULL");
	espdr_ensure(mask_iml == NULL, CPL_ERROR_NULL_INPUT,
			"Mask imagelist is NULL");

	int input_size = cpl_imagelist_get_size(input_iml);
	int mask_size = cpl_imagelist_get_size(mask_iml);
    double *mean_img, *median_img, *max_img;
    
	//espdr_msg("input iml size = %lld", cpl_imagelist_get_size(input_iml));
	//espdr_msg("mask iml size = %lld", cpl_imagelist_get_size(mask_iml));
	espdr_ensure(input_size != mask_size,
                 CPL_ERROR_INCOMPATIBLE_INPUT,
                 "input imagelist and/or mask imagelist has wrong size");

	const cpl_image *curr_image = NULL;
	const cpl_image *curr_mask = NULL;
    
    mean_img = (double *)cpl_calloc(input_size, sizeof(double));
    median_img = (double *)cpl_calloc(input_size, sizeof(double));
    max_img = (double *)cpl_calloc(input_size, sizeof(double));

    gettimeofday(&tv1, NULL);
    
    //HDRL_OMP(omp parallel shared(input_iml, mask_iml, rejected_part, max_fluxes_RE,input_size), private(curr_mask,curr_image))
    //{
    //HDRL_OMP(omp for)
#pragma omp parallel for private(curr_mask,curr_image) \
shared(input_iml,mask_iml,mean_img,median_img,max_img,max_fluxes_RE,input_size)

	for (int i = 0; i < input_size; i++) {
		curr_image = cpl_imagelist_get_const(input_iml, i);
		curr_mask = cpl_imagelist_get_const(mask_iml, i);
        
        double bound = 0.0;
		espdr_get_high_flux_rejection_bound(curr_image, curr_mask,
                                                       rejected_part, &bound);
        
		max_fluxes_RE[i] = bound;
        
        mean_img[i] = cpl_image_get_mean(curr_image);
        median_img[i] = cpl_image_get_median(curr_image);
        max_img[i] = cpl_image_get_max(curr_image);

	}
    //}
    
    gettimeofday(&tv2, NULL);
    
    espdr_msg("Wall time for parallel loop was %f seconds",
              (double) (tv2.tv_usec - tv1.tv_usec) / 1000000 +
              (double) (tv2.tv_sec - tv1.tv_sec));
    
    for (int i = 0; i < input_size; i++) {
        espdr_msg("image/output %d: mean: %lf, median: %lf, max: %lf", i,
                  mean_img[i], median_img[i], max_img[i]);
    }
    
    cpl_free(mean_img);
    cpl_free(median_img);
    cpl_free(max_img);

	return (cpl_error_get_code());
}



/*----------------------------------------------------------------------------*/
/**
 @brief     Get the cosmics rejection bound
 @param         input_image     input CCD corrected raw image
 @param         pixel_mask      hot & bad pixels mask
 @param         rejected_part   estimated cosmics part
 @param[out]    bound           computed rejected bound
 @return    CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_get_high_flux_rejection_bound(const cpl_image *input_image,
		const cpl_image *mask,
		const double rejected_part,
		double *bound) {

	espdr_ensure(input_image == NULL, CPL_ERROR_NULL_INPUT,
			"Input image is NULL");
	espdr_ensure((rejected_part < 0.0) || (rejected_part > 0.2),
			CPL_ERROR_INCOMPATIBLE_INPUT,
			"rejected_part parameter should be between 0.0 and 0.2");
	int data_size_x = cpl_image_get_size_x(input_image);
	int data_size_y = cpl_image_get_size_y(input_image);
	int mask_size_x = cpl_image_get_size_x(mask);
	int mask_size_y = cpl_image_get_size_y(mask);
	espdr_msg_debug("data size: %d x %d", data_size_x, data_size_y);
	espdr_msg_debug("mask size: %d x %d", mask_size_x, mask_size_y);

	espdr_ensure(data_size_x != mask_size_x, CPL_ERROR_INCOMPATIBLE_INPUT,
			"Input data and mask sizes X differ");
	espdr_ensure(data_size_y != mask_size_y, CPL_ERROR_INCOMPATIBLE_INPUT,
			"Input data and mask sizes Y differ");

	cpl_error_code my_error = CPL_ERROR_NONE;
	const double *input_data = cpl_image_get_data_double_const(input_image);
	const int *mask_data = cpl_image_get_data_int_const(mask);
	int i, index, good_pixels_nb = 0;
	double *good_data_vector = NULL;
	int rejection_index;


	for (i = 0; i < data_size_x*data_size_y; i++) {
		if (mask_data[i] == 0) {
			good_pixels_nb++;
		}
	}

	good_data_vector = (double *)cpl_calloc(good_pixels_nb, sizeof(double));

	index = 0;
	for (i = 0; i < data_size_x*data_size_y; i++) {
		if (mask_data[i] == 0) {
			good_data_vector[index] = input_data[i];
			index++;
		}
	}

	my_error = espdr_quicksort(good_data_vector, good_pixels_nb);

	espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
			"espdr_quicksort failed: %s",
			cpl_error_get_message_default(my_error));

	rejection_index = (int)((1.0 - rejected_part) * good_pixels_nb);
	*bound = good_data_vector[rejection_index];
	espdr_msg_debug("rejected high flux pxls number: %d",
			good_pixels_nb - rejection_index);

	cpl_free(good_data_vector);

	return (cpl_error_get_code());
}

