/*                                                                            *
 *   This file is part of the ESPRESSO Pipeline                               *
 *   Copyright (C) 2006 European Southern Observatory                         *
 *                                                                            *
 *   This library is free software; you can redistribute it and/or modify     *
 *   it under the terms of the GNU General Public License as published by     *
 *   the Free Software Foundation; either version 2 of the License, or        *
 *   (at your option) any later version.                                      *
 *                                                                            *
 *   This program is distributed in the hope that it will be useful,          *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 *   GNU General Public License for more details.                             *
 *                                                                            *
 *   You should have received a copy of the GNU General Public License        *
 *   along with this program; if not, write to the Free Software              *
 *   Foundation, 51 Franklin St, Fifth Floor, Boston, MA  02111-1307  USA     *
 *                                                                            */

/*
 * $Author: asegovia $
 * $Date: 2015-05-01 15:13:41 $
 * $Revision:     $
 * $Name: not supported by cvs2svn $
 */


#include <espdr_eff_ab.h>
#include <espdr_overscan.h>
#include <espdr_instrument.h>
#include <espdr_background.h>
#include <espdr_blaze.h>
#include <espdr_wave_cal.h>
#include <espdr_orders.h>
#include <espdr_pixels.h>
#include <stdlib.h>

/*----------------------------------------------------------------------------
 Functions code
 ----------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/**
 @brief     Add extra QC keywords
 @param     keywords    property list to add the extra KWs
 @param     ext_nb      number of extensions
 @return    0 if everything is ok
 */
/*---------------------------------------------------------------------------*/

//TODO: missing keywords
static cpl_error_code espdr_add_extra_qc(cpl_propertylist** keywords,
                                         const int ext_nb) {

    const char* ksuff[2] = {" BLUE", " RED"};
    cpl_propertylist_append_int(*keywords,"ESO QC PROC VERSION", 2);
    int* ord_ref = espdr_get_ord_ref(*keywords, "espdr_cal_eff_ab");

    espdr_add_qc_key_stat_pri(keywords, ext_nb, "MAX FLUX", ksuff, CPL_FALSE, CPL_FALSE,
                              CPL_FALSE, CPL_TRUE, CPL_FALSE);

    espdr_add_qc_key_effab_ord_ref(keywords, ext_nb, ord_ref, "SNR", ksuff,
                                   CPL_TRUE, CPL_TRUE, CPL_FALSE, CPL_FALSE, CPL_FALSE);

    espdr_add_qc_key_stat_ord_pri(keywords,ext_nb, "SNR", "DUMMY", CPL_TRUE, CPL_TRUE,
                                  CPL_TRUE, CPL_TRUE, CPL_FALSE);



    cpl_free(ord_ref);
    return cpl_error_get_code();
}


/*---------------------------------------------------------------------------*/
/**
 @brief    Interpret the command line options and execute the data processing
 @param    parameters     the parameters list
 @param    frameset   the frames list

 In case of failure the cpl_error_code is set.
 */
/*---------------------------------------------------------------------------*/

int espdr_cal_eff_ab(cpl_parameterlist *parameters, cpl_frameset *frameset,
		const char* recipe_id)
{
    cpl_frameset *used_frames = NULL;
	cpl_error_code my_error = CPL_ERROR_NONE;

    const int rec_ntags = 8;
    const char* rec_tags[8] = {ESPDR_EFF_AB_RAW,
        ESPDR_PRO_CATG_HOT_PIXELS, ESPDR_PRO_CATG_MDARK,
        ESPDR_PRO_CATG_BAD_PIXELS, ESPDR_CCD_GEOM,
        ESPDR_PRO_CATG_ORDERS_A, ESPDR_PRO_CATG_ORDERS_B,
        ESPDR_ORDERS_MASK
    };
    int is_required[8] = {1, 1, 0, 1, 1, 1, 1, 0};

	cpl_msg_set_level(CPL_MSG_INFO);
	//cpl_msg_set_level(CPL_MSG_DEBUG);

    espdr_CCD_geometry *CCD_geom = NULL;
    espdr_inst_config *inst_config = NULL;
    espdr_qc_keywords *qc_kws = NULL;
    espdr_OVSC_param *OVSC_param = NULL;
    espdr_CAL_EFF_AB_param *CAL_EFF_AB_param = NULL;
    cpl_frame* CCD_geom_frame = NULL;
    cpl_frame* inst_config_frame = NULL;

    espdr_msg("Starting cal_eff_ab");

    espdr_ensure(espdr_check_input_tags(frameset,
                                        rec_tags, is_required,
                                        rec_ntags) != CPL_ERROR_NONE,
                 cpl_error_get_code(), "Wrong input tag!");
    espdr_ensure(espdr_check_input_inst_config(frameset) != CPL_ERROR_NONE,
                 cpl_error_get_code(), "Wrong input tag!");
    espdr_ensure(espdr_check_input_flat(frameset, 0) != CPL_ERROR_NONE,
                 cpl_error_get_code(), "Wrong flat input tag!");

    /* Identify the RAW and CALIB frames in the input frameset */
    espdr_ensure(espdr_dfs_set_groups(frameset) != CPL_ERROR_NONE,
                 cpl_error_get_code(),
                 "DFS setting groups failed. Expected inputs are:\n"
                 "raw efficiency (sky flat) image, tagged as EFF_AB\n"
                 "CCD geometry table, tagged as CCD_GEOM\n"
                 "instrument configuration table, tagged as MASTER_INST_CONFIG or INST_CONFIG\n"
                 "ThAr lines reference table, tagged as REF_LINE_TABLEA or REF_LINE_TABLE_B\n"
                 "hot pixel mask image, tagged as HOT_PIXEL_MASK\n"
                 "bad pixel mask image, tagged as BAD_PIXEL_MASK\n"
                 "orders definition images for fibres A and B, tagged as ORDER_TABLE_A & ORDER_TABLE_B\n"
                 "orders profile images for fibres A and B, tagged as ORDER_PROFILE_A & ORDER_PROFILE_B\n"
                 "flat images for fibres A and B, tagged as FLAT_A & FLAT_B\n");

    /* AMO Changed way to read CCD geom frame and to init relevant recipe
     * internal parameters structures to simplify this calling function
     */
    OVSC_param        = espdr_parameters_OVSC_init(recipe_id, parameters);
    CCD_geom_frame    = espdr_frame_find(frameset, ESPDR_CCD_GEOM);
    espdr_ensure(CCD_geom_frame == NULL,cpl_error_get_code(),
                 "CCD geometry frame not found!");
    CCD_geom          = espdr_CCD_geom_init(parameters, CCD_geom_frame);
    inst_config_frame = espdr_get_inst_config(frameset);
    espdr_ensure(inst_config_frame == NULL,cpl_error_get_code(),
                 "Instrument config frame not found!");
    inst_config       = espdr_inst_config_init(parameters,
                                               CCD_geom->ext_nb,
                                               inst_config_frame);
    CAL_EFF_AB_param  = espdr_CAL_EFF_AB_param_init(recipe_id, parameters);

    used_frames = cpl_frameset_new();
    my_error = cpl_frameset_insert(used_frames,
                                   cpl_frame_duplicate(CCD_geom_frame));
    my_error = cpl_frameset_insert(used_frames,
                                   cpl_frame_duplicate(inst_config_frame));

    /* Filling up the DRS QC KWs structure */
    qc_kws = (espdr_qc_keywords *)cpl_malloc( sizeof(espdr_qc_keywords));
    my_error = espdr_fill_qc_keywords(inst_config, qc_kws);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Filling QC KWs failed: %s",
                 cpl_error_get_message_default(my_error));
    espdr_ensure(qc_kws == NULL, CPL_ERROR_ILLEGAL_OUTPUT,
                 "QC KWs structure is NULL");

    cpl_frame *eff_ab_frame = espdr_frame_find(frameset, ESPDR_EFF_AB_RAW);
    espdr_ensure(eff_ab_frame == NULL, CPL_ERROR_NULL_INPUT,
                 "No input raw EFF_AB frame");

    my_error = cpl_frameset_insert(used_frames, cpl_frame_duplicate(eff_ab_frame));

    cpl_propertylist *keywords = NULL;
    /* Load eff_ab keywords from the first image (primary header) */
    const char *input_filename = NULL;
    input_filename = cpl_frame_get_filename(eff_ab_frame);
    espdr_msg("KEYWORDS for CAL_EFF_AB input filename: %s", input_filename);
    keywords = cpl_propertylist_load(input_filename, 0);
    espdr_ensure(keywords == NULL, CPL_ERROR_ILLEGAL_OUTPUT, "keywords are NULL");

#if SAVE_DEBUG_PRODUCT_PREPROCESSING
    // Adding artificially the ARCFILE if not present - final filename - to be removed when using ESO DFS
    if (cpl_propertylist_has(keywords, "ARCFILE") == 0) {
        char *arc_filename = strrchr(input_filename, '/')+1;
        espdr_msg("ARCFILE not existing - updating the filename: %s", arc_filename);
        my_error = cpl_propertylist_update_string(keywords, "ARCFILE", arc_filename);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "Updating the ARCFILE KW (%s) failed: %s",
                     arc_filename, cpl_error_get_message_default(my_error));
    }
#endif

    /* Getting the physical orders id */
    int *orders_nb_per_fibre = (int *)cpl_calloc(inst_config->fibres_nb, sizeof(int));
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        orders_nb_per_fibre[i] = 0;
        for (int j = 0; j < CCD_geom->ext_nb; j++) {
            orders_nb_per_fibre[i] += inst_config->orders_nb[i*CCD_geom->ext_nb+j];
        }
        espdr_msg("Orders nb for fibre %c: %d", fibre_name[i], orders_nb_per_fibre[i]);
    }

    int **physical_orders_id = (int**) cpl_calloc(inst_config->fibres_nb,sizeof(int *));
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        physical_orders_id[i] = (int *)cpl_calloc(orders_nb_per_fibre[i],sizeof(int));
    }
    for (int fibre = 0; fibre < inst_config->fibres_nb; fibre++) {
        espdr_get_order_phys_numbers(CCD_geom, inst_config, fibre,
                                     physical_orders_id[fibre]);
    }

    double *RON = (double *)cpl_calloc(CCD_geom->total_output_nb, sizeof(double));
    int **cosmics_nb = (int **)cpl_calloc(inst_config->fibres_nb, sizeof(int *));
    double **snr_per_fibre = (double **) cpl_calloc(inst_config->fibres_nb,sizeof(int *));
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        snr_per_fibre[i] = (double *) cpl_calloc(orders_nb_per_fibre[i],sizeof(double));
    }
    cpl_imagelist *eff_ab_bkgr_subs = NULL;
    cpl_image **flat_corr_flux_table = (cpl_image **)cpl_malloc(inst_config->fibres_nb * sizeof(cpl_image *));
    cpl_image **flat_corr_err_table = (cpl_image **)cpl_malloc(inst_config->fibres_nb * sizeof(cpl_image *));
    cpl_image **flat_corr_qual_table = (cpl_image **)cpl_malloc(inst_config->fibres_nb * sizeof(cpl_image *));
    my_error = espdr_process_calib_till_extraction(frameset, parameters, recipe_id,
                                                   eff_ab_frame, used_frames,
                                                   inst_config, CCD_geom,
                                                   qc_kws, keywords,
                                                   orders_nb_per_fibre,
                                                   0, // remove_bias_res_flag
                                                   "SKY",
                                                   "on", // background_sw
                                                   RON, cosmics_nb, snr_per_fibre,
                                                   &eff_ab_bkgr_subs,
                                                   flat_corr_flux_table,
                                                   flat_corr_err_table,
                                                   flat_corr_qual_table);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_process_calib_till_extraction failed: %s",
                 cpl_error_get_message_default(my_error));

	/* Check physical order numbers calculation */

	/*espdr_msg("Physical orders numbers (slices per phys: %d):",
	      inst_config->slices_nb_per_phys_order);
	for (int fibre = 0; fibre < inst_config->fibres_nb; fibre++) {
	for (int i = 0; i < inst_config->orders_nb[fibre]; i++) {
	espdr_msg("Fibre %d Order %d: %d",fibre, i+1, physical_orders_id[fibre][i]);
	}
	}*/

    cpl_image *s2d_flux = cpl_image_new(cpl_image_get_size_x(flat_corr_flux_table[0]),
                                        cpl_image_get_size_y(flat_corr_flux_table[1]),
                                        CPL_TYPE_DOUBLE);
    espdr_msg("Computing the flux ratio (fibre B / fibre A) ...");
    my_error = espdr_relative_efficiency(inst_config->eff_ab_poly_deg,
                                         physical_orders_id,
                                         inst_config, CCD_geom,
                                         flat_corr_flux_table,
                                         &s2d_flux);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_relative_efficiency failed: %s",
                 cpl_error_get_message_default(my_error));
    espdr_msg("Flux ratio computed and smoothed");

    cpl_propertylist **keywords_fibre = (cpl_propertylist **)cpl_malloc(inst_config->fibres_nb *
                                                                        sizeof(cpl_propertylist *));
    for (int t = 0; t < inst_config->fibres_nb; t++) {
        /* QC function */
        keywords_fibre[t] = cpl_propertylist_duplicate(keywords);
        my_error = espdr_cal_eff_ab_QC(CCD_geom, inst_config, qc_kws,
                                       cosmics_nb[t],
                                       snr_per_fibre[t],
                                       orders_nb_per_fibre[t],
                                       keywords_fibre[t]);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_cal_eff_ab_QC failed: %s",
                     cpl_error_get_message_default(my_error));
    }

    /* QC for relative efficiency */
    my_error = espdr_rel_eff_QC(inst_config, qc_kws, s2d_flux, keywords_fibre[1]);
	espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_rel_eff_QC failed: %s",
                 cpl_error_get_message_default(my_error));

    int global_QC_A = cpl_propertylist_get_int(keywords_fibre[0], qc_kws->qc_cal_eff_ab_check_kw);
    int global_QC_B = cpl_propertylist_get_int(keywords_fibre[1], qc_kws->qc_cal_eff_ab_check_kw);
    char comment[COMMENT_LENGTH];
    if ((global_QC_A == 0) || (global_QC_B == 0)) {
        if (global_QC_A == 1) {
            sprintf(comment, "Overall QC set to 0, because QC on B = 0");
            my_error = espdr_keyword_add_int(qc_kws->qc_cal_eff_ab_check_kw,
                                             0,
                                             comment,
                                             &keywords_fibre[0]);
        }
        if (global_QC_B == 1) {
            sprintf(comment, "Overall QC set to 0, because QC on A = 0");
            my_error = espdr_keyword_add_int(qc_kws->qc_cal_eff_ab_check_kw,
                                             0,
                                             comment,
                                             &keywords_fibre[1]);
        }
    }

    for (int t = 0; t < inst_config->fibres_nb; t++) {
        char s2d_eff_name[64];
        char pro_catg_string[64];
        sprintf(s2d_eff_name, "%s_S2D_BLAZE_EFF_%c.fits", inst_config->instrument, fibre_name[t]);
        sprintf(pro_catg_string,"%s_%c", "S2D_BLAZE_EFF",fibre_name[t]);
        cpl_propertylist_erase(keywords_fibre[t], "RADECSYS");
        my_error = espdr_dfs_save_S2D(flat_corr_flux_table[t],
                                      flat_corr_err_table[t],
                                      flat_corr_qual_table[t],
                                      frameset, used_frames, parameters,
                                      keywords_fibre[t], s2d_eff_name,
                                      pro_catg_string, "espdr_cal_eff_ab");
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_dfs_save_S2D failed: %s for file %s",
                     cpl_error_get_message_default(my_error), s2d_eff_name);
    }

    /* saving product */
    char rel_eff_name[64];
    cpl_frame* product_frame;
    sprintf(rel_eff_name,"%s_REL_EFF_B.fits", inst_config->instrument);
    my_error = cpl_propertylist_update_string(keywords_fibre[1], PRO_CATG_KW,
                                              ESPDR_PRO_CATG_REL_EFF);
    cpl_frameset *frameset_orig = cpl_frameset_duplicate(frameset);

    if ((strcmp(inst_config->instrument, "HARPN") != 0) &&
        (strcmp(inst_config->instrument, "CORALIE") != 0)) {
        my_error = espdr_add_extra_qc(&keywords_fibre[1], CCD_geom->ext_nb);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_add_extra_qc failed: %s ",
                     cpl_error_get_message_default(my_error));
    }
    my_error = espdr_dfs_image_save_one_ext(frameset_orig,
                                            parameters,
                                            used_frames,
											recipe_id,
                                            keywords_fibre[1],
                                            rel_eff_name,
                                            s2d_flux,
                                            CPL_TYPE_FLOAT);

    espdr_frame_new(&product_frame,
                    rel_eff_name,
                    CPL_FRAME_GROUP_PRODUCT,
                    CPL_FRAME_LEVEL_FINAL,
                    CPL_FRAME_TYPE_IMAGE,
                    ESPDR_PRO_CATG_REL_EFF);
    cpl_frameset_insert(frameset, product_frame);

    /* cleaning memory */
    espdr_msg("Cleaning memory");
	for (int i = 0; i < inst_config->fibres_nb; i++) {
		cpl_image_delete(flat_corr_flux_table[i]);
		cpl_image_delete(flat_corr_err_table[i]);
		cpl_image_delete(flat_corr_qual_table[i]);
        cpl_free(physical_orders_id[i]);
        cpl_free(cosmics_nb[i]);
        cpl_free(snr_per_fibre[i]);
        cpl_propertylist_delete(keywords_fibre[i]);
	}
    cpl_free(flat_corr_flux_table);
    cpl_free(flat_corr_err_table);
    cpl_free(flat_corr_qual_table);
    cpl_imagelist_delete(eff_ab_bkgr_subs);

    cpl_free(physical_orders_id);
    cpl_free(orders_nb_per_fibre);
    cpl_free(cosmics_nb);
    cpl_free(snr_per_fibre);
    cpl_free(RON);

    cpl_propertylist_delete(keywords);
    cpl_free(keywords_fibre);
    cpl_free(qc_kws);

    cpl_frameset_delete(used_frames);
    cpl_frameset_delete(frameset_orig);

    espdr_parameters_OVSC_delete(OVSC_param);
    espdr_parameters_CAL_EFF_AB_delete(CAL_EFF_AB_param);
    espdr_parameters_CCD_geometry_delete(CCD_geom);
    espdr_parameters_inst_config_delete(inst_config);

	return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Create the structure for CAL_EFF_AB input parameters
 @param    recipe_id	recipe identifier
 @param    list			list of input parameters
 @param    p			input CAL_EFF_AB parameters structure
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_parameters_CAL_EFF_AB_create(const char* recipe_id,
                                            cpl_parameterlist *list,
                                            espdr_CAL_EFF_AB_param *p) {
    espdr_ensure(recipe_id == NULL,
                 CPL_ERROR_NULL_INPUT, "The recipe ID is NULL");
    
    /* check parameters */
    espdr_ensure(list == NULL, CPL_ERROR_NULL_INPUT,
                 "The parameters list NULL");
    
    cpl_error_code error_got;
    char comment[COMMENT_LENGTH];
    int bkgr_x_min = BKGR_X_MIN;
    int bkgr_x_max = BKGR_X_MAX;
    int bkgr_y_min = BKGR_Y_MIN;
    int bkgr_y_max = BKGR_Y_MAX;
    
	/* Fill the parameter list */
    sprintf(comment,
            "Grid size in x used to calculate the background, between: %d and %d",
            bkgr_x_min,bkgr_x_max);
    error_got = espdr_parameters_new_range_int(recipe_id, list,
                                               "eff_ab_bkgr_grid_size_x",
                                               p->bkgr_grid_size_x,
                                               bkgr_x_min, bkgr_x_max,
                                               comment);
    espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
                 "Error adding parameter bkgr_grid_size_x to the list");
    
    sprintf(comment,
            "Grid size in y used to calculate the background, between: %d and %d",
            bkgr_y_min,bkgr_y_max);
    error_got = espdr_parameters_new_range_int(recipe_id, list,
                                               "eff_ab_bkgr_grid_size_y",
                                               p->bkgr_grid_size_y,
                                               bkgr_y_min, bkgr_y_max,
                                               comment);
    espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
                 "Error adding parameter bkgr_grid_size_y to the list");
    
    error_got = espdr_parameters_new_string(recipe_id, list,
                                            "eff_ab_extraction_method",
                                            p->extraction_method,
                                            "Method used to extract orders");
    espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
                 "Error adding parameter extraction_method to the list");
    
    error_got = espdr_parameters_new_double(recipe_id, list,
                                            "eff_ab_extraction_ksigma",
                                            p->extraction_ksigma,
                                            "ksigma used to extract orders");
    espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
                 "Error adding parameter extraction_method to the list");
    
    error_got = espdr_parameters_new_range_int(recipe_id, list,
                                               "eff_ab_poly_deg",
                                               p->poly_deg, 1, 89,
                                               "Efficiency computation fit polynomial degree.");
    espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
                 "Error adding parameter poly_deg to the list");
    
	return (CPL_ERROR_NONE);
}


/*----------------------------------------------------------------------------*/
/**
 @brief    Delete the structure for CAL_EFF_AB input parameters
 @param    p	input CAL_EFF_AB parameters structure
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_parameters_CAL_EFF_AB_delete(espdr_CAL_EFF_AB_param* p) {
	
    cpl_free((void *)p->extraction_method);
    cpl_free(p);
    p = NULL;
    
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief     Get the cal_eff_ab recipe parameters
 @param     recipe_id   recipe ID
 @param     param_list  parameters list
 @param     CAL_EFF_AB_param  CAL_EFF_AB parameters structure
 @return    CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_parameters_CAL_EFF_AB_get(const char* recipe_id,
                                         cpl_parameterlist* param_list,
                                         espdr_CAL_EFF_AB_param *CAL_EFF_AB_param) {
	
    espdr_ensure(recipe_id == NULL,
                 CPL_ERROR_NULL_INPUT, "The recipe ID is NULL");
    
	/* check parameters */
	espdr_ensure(param_list == NULL, CPL_ERROR_NULL_INPUT,
				 "Parameters list is NULL");
	
	/* Fill the structure */
    CAL_EFF_AB_param->bkgr_grid_size_x = espdr_parameters_get_int(recipe_id,
                                                                  param_list,
                                                        "eff_ab_bkgr_grid_size_x");
    
    CAL_EFF_AB_param->bkgr_grid_size_y = espdr_parameters_get_int(recipe_id,
                                                                  param_list,
                                                        "eff_ab_bkgr_grid_size_y");
    
    CAL_EFF_AB_param->extraction_method = espdr_parameters_get_string(recipe_id,
                                                                      param_list,
                                                        "eff_ab_extraction_method");
    
    CAL_EFF_AB_param->extraction_ksigma = espdr_parameters_get_double(recipe_id,
                                                                      param_list,
                                                        "eff_ab_extraction_ksigma");
    
    CAL_EFF_AB_param->poly_deg = espdr_parameters_get_int(recipe_id,
                                                          param_list,
                                                          "eff_ab_poly_deg");
    
	return cpl_error_get_code();
}


/*---------------------------------------------------------------------------*/
/**
 @brief     print the CAL_EFF_AB parameters
 @param     CAL_EFF_AB_param  CAL_EFF_AB parameters structure
 @return    CPL_ERROR_NONE iff OK
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_parameters_CAL_EFF_AB_print(
				espdr_CAL_EFF_AB_param *CAL_EFF_AB_param) {
	
	espdr_msg("\tCAL_EFF_AB parameters:");
    espdr_msg("\t\tCAL_EFF_AB background grid size x = %d",
              CAL_EFF_AB_param->bkgr_grid_size_x);
    espdr_msg("\t\tCAL_EFF_AB background grid size y = %d",
              CAL_EFF_AB_param->bkgr_grid_size_y);
    espdr_msg("\t\tCAL_EFF_AB extraction method = %s",
              CAL_EFF_AB_param->extraction_method);
    espdr_msg("\t\tCAL_EFF_AB extraction ksigma = %.2f",
              CAL_EFF_AB_param->extraction_ksigma);
    espdr_msg("\t\tCAL_EFF_AB polynomial degree = %d",
              CAL_EFF_AB_param->poly_deg);
    
	return (CPL_ERROR_NONE);
}


/*----------------------------------------------------------------------------*/
/**
 @brief    Init CAL_EFF_AB_param structure
 
 @param    recipe_id    recipe name
 @param    parameters   recipe input parameters
 
 @return   allocated structure iff OK
 */
/*----------------------------------------------------------------------------*/

espdr_CAL_EFF_AB_param * espdr_CAL_EFF_AB_param_init(const char *recipe_id, 
					 cpl_parameterlist *parameters) {
    
    espdr_CAL_EFF_AB_param *CAL_EFF_AB_param =
    (espdr_CAL_EFF_AB_param *)cpl_malloc(sizeof(espdr_CAL_EFF_AB_param));
    
    /* Read the cal_eff_ab parameters */
	espdr_parameters_CAL_EFF_AB_get(recipe_id, parameters, CAL_EFF_AB_param);
	espdr_parameters_CAL_EFF_AB_print(CAL_EFF_AB_param);
    if(cpl_error_get_code() != CPL_ERROR_NONE) {
        return NULL;
    } else {
        return CAL_EFF_AB_param;
    }

}

/*----------------------------------------------------------------------------*/
/**
 @brief    Computing the relative efficiency
 @param
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_relative_efficiency(int pol_deg,
                                         int **physical_orders_id,
                                         espdr_inst_config *inst_config,
                                         espdr_CCD_geometry *CCD_geom,
                                         cpl_image **s2d_flux,
                                         cpl_image **s2d_eff_flux_RE) {
    
    cpl_error_code my_error = CPL_ERROR_NONE;
    
    int order_a = 0;
    int order_b = 0;
    int last_order_a_matched = -1;
    int ext_a;
    int ext_b;
    bool found = false;
    int fibre_A = 0;
    int fibre_B = 1;
    int pix;
    
    int sz_order_x = cpl_image_get_size_x(s2d_flux[fibre_A]);
    int orders_nb_a = cpl_image_get_size_y(s2d_flux[fibre_A]);
    int orders_nb_b = cpl_image_get_size_y(s2d_flux[fibre_B]);
    
    cpl_image *s2d_flux_A = cpl_image_duplicate(s2d_flux[fibre_A]);
    cpl_image *s2d_flux_B = cpl_image_duplicate(s2d_flux[fibre_B]);
    
    double snr_threshold = 30.0;
    if (inst_config->inst_type == NIR) {
        snr_threshold = 300.0;
    }
    //CLO: Set to 30 -electrons- in order to have sufficiently high SNR
    my_error = cpl_image_threshold(s2d_flux_A, snr_threshold, DBL_MAX, -1.0, 0.0);
    my_error = cpl_image_threshold(s2d_flux_B, snr_threshold, DBL_MAX, 1.0, 0.0);
    
    cpl_image *order_flux_A = NULL;
    cpl_image *order_flux_B = NULL;
    cpl_image *order_ratio = NULL;
    
    for (order_b = 0; order_b < orders_nb_b; order_b++) {
        ext_b = espdr_get_ext_index_for_order(order_b+1, fibre_B,
                                              inst_config, CCD_geom);
        found = false;
        
        for (order_a = 0; order_a < orders_nb_a; order_a++) {
            ext_a = espdr_get_ext_index_for_order(order_a+1, fibre_A,
                                                  inst_config, CCD_geom);
            
            if ((physical_orders_id[fibre_A][order_a] == physical_orders_id[fibre_B][order_b]) &&
                (ext_a == ext_b) && (!found) && (order_a > last_order_a_matched)) {
                
                last_order_a_matched = order_a; // in order not to much twice the same order from fibre B
                found = true;                   // once order found, we don't look for another one
                
                /* order in A & B fibres ==> calculates ratio */
                
                //espdr_msg("Computing ratio for order %d on A and order %d on B",
                //          order_a, order_b);
                
                order_flux_A = cpl_image_extract(s2d_flux_A, 1, order_a+1, sz_order_x, order_a+1);
                order_flux_B = cpl_image_extract(s2d_flux_B, 1, order_b+1, sz_order_x, order_b+1);
                order_ratio = cpl_image_divide_create(order_flux_B, order_flux_A);
                
                my_error = cpl_image_copy(*s2d_eff_flux_RE, order_ratio, 1, order_b+1);
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "cpl_image_copy with s2d_eff_flux_RE failed: %s",
                             cpl_error_get_message_default(my_error));
                
                cpl_image_delete(order_flux_A);
                cpl_image_delete(order_flux_B);
                cpl_image_delete(order_ratio);
                
                /*
                for (int i = 0; i < sz_order_x; i++ ) {
                    
                    pix = sz_order_x*order_a+i;
                    pix_a = cpl_image_get(s2d_flux[fibre],i+1,order_a+1,&pir);
                    pix_b = cpl_image_get(s2d_flux[fibre+1],i+1,order_b+1,&pir);
                    
                    if ((pix_a > 30.) && (pix_b > 30.)) { //CLO: Set to 30 -electrons- in oder to have sufficiently high SNR.
                        ratio_data[pix]=pix_b/pix_a;
                    } else {
                        ratio_data[pix]=-1.0;
                    }
                }
                */
                
            }
        }
        
        if (!found) { // order present in fibre B, but not in fibre A ==> 1.0
            
            order_ratio = cpl_image_new(sz_order_x, 1, CPL_TYPE_DOUBLE);
            my_error = cpl_image_add_scalar(order_ratio, 1.0);
            
            my_error = cpl_image_copy(*s2d_eff_flux_RE, order_ratio, 1, order_b+1);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "cpl_image_copy with s2d_eff_flux_RE failed: %s",
                         cpl_error_get_message_default(my_error));
            
            cpl_image_delete(order_ratio);
            
            /*
            for (int i = 0; i < sz_order_x; i++ ) {
                pix = sz_order_x*order_a+i;
                ratio_data[pix]=1.0;
            }
            */
        }
    }
    
    cpl_image_delete(s2d_flux_A);
    cpl_image_delete(s2d_flux_B);
    
    double *ratio_data = cpl_image_get_data_double(*s2d_eff_flux_RE);
    
    /* smoothing ratio ...*/
    espdr_msg("Flux ratio computed, smoothing...");
    espdr_msg("orders nb: %d", orders_nb_b);
    
    if (pol_deg == 1) {
        espdr_msg("Polynomial degree is 1, taking the median");
        double ratio_median = cpl_image_get_median(*s2d_eff_flux_RE);
        for (int i = 0; i < sz_order_x * orders_nb_b; i++) {
            ratio_data[i] = ratio_median;
        }
    } else {
        if (inst_config->slices_nb_per_phys_order == 2) {
            
            espdr_poly_data *p1_data = (espdr_poly_data *)cpl_malloc(sizeof(espdr_poly_data));
            
            p1_data->x = (double *)cpl_calloc(sz_order_x * orders_nb_b / 2.0, sizeof(double));
            p1_data->y = (double *)cpl_calloc(sz_order_x * orders_nb_b / 2.0, sizeof(double));
            p1_data->err = (double *)cpl_calloc(sz_order_x * orders_nb_b / 2.0, sizeof(double));
            
            double *fit1 = NULL;
            double *coeffs1 = NULL;
            double *coeffs_err1 = NULL;
            double chisq1 = 0.0;
            int index = 0;
            
            // smoothing of the 1st slice
            for (order_b = 0; order_b < orders_nb_b; order_b += 2) {
                for (int i = 0; i < sz_order_x; i++) {
                    pix = sz_order_x*order_b+i;
                    p1_data->x[index] = (double)index;
                    p1_data->y[index] = ratio_data[pix];
                    
                    if (ratio_data[pix] < 0.) {
                        p1_data->err[index] = DBL_MAX;
                    } else {
                        p1_data->err[index] = 1.;
                    }
                    index++;
                }
            }
            
            p1_data->n = sz_order_x * orders_nb_b / 2.0;
            
            fit1 = (double *)cpl_calloc(sz_order_x * orders_nb_b / 2.0, sizeof(double));
            coeffs1 = (double *)cpl_calloc(pol_deg, sizeof(double));
            coeffs_err1 = (double *)cpl_calloc(pol_deg, sizeof(double));
            
            my_error = espdr_fit_poly(p1_data, pol_deg, fit1,
                                      coeffs1, coeffs_err1, &chisq1, 0);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_fit_poly failed: %s",
                         cpl_error_get_message_default(my_error));
            
            espdr_msg("1st slice flux ratio smoothing coeffs:");
            for (int i = 0; i < pol_deg; i++) {
                espdr_msg("\t%.15f", coeffs1[i]);
            }
            espdr_msg("CHI2 = %f", chisq1);
            
            index = 0;
            for (order_b = 0; order_b < orders_nb_b; order_b += 2) {
                for (int j = 0; j < sz_order_x; j++) {
                    pix = sz_order_x*order_b+j;
                    ratio_data[pix] = 0.;
                    for (int c = pol_deg - 1; c >= 0; c--) {
                        ratio_data[pix] = ratio_data[pix] * (double)index + coeffs1[c];
                    }
                    index++;
                }
            }
            
            cpl_free(p1_data->x);
            cpl_free(p1_data->y);
            cpl_free(p1_data->err);
            
            cpl_free(fit1);
            cpl_free(coeffs1);
            cpl_free(coeffs_err1);
            cpl_free(p1_data);
            
            // smoothing of the 2nd slice
            espdr_poly_data *p2_data = (espdr_poly_data *)cpl_malloc(sizeof(espdr_poly_data));
            
            p2_data->x = (double *)cpl_calloc(sz_order_x * orders_nb_b / 2.0, sizeof(double));
            p2_data->y = (double *)cpl_calloc(sz_order_x * orders_nb_b / 2.0, sizeof(double));
            p2_data->err = (double *)cpl_calloc(sz_order_x * orders_nb_b / 2.0, sizeof(double));
            
            double *fit2 = NULL;
            double *coeffs2 = NULL;
            double *coeffs_err2 = NULL;
            double chisq2 = 0.0;
            
            index = 0;
            for (order_b = 1; order_b < orders_nb_b; order_b += 2) {
                for (int i = 0; i < sz_order_x; i++) {
                    pix = sz_order_x*order_b+i;
                    p2_data->x[index] = (double)index;
                    p2_data->y[index] = ratio_data[pix];
                    
                    if (ratio_data[pix] < 0.) {
                        p2_data->err[index] = DBL_MAX;
                    } else {
                        p2_data->err[index] = 1.;
                    }
                    index++;
                }
            }
            
            p2_data->n = sz_order_x * orders_nb_b / 2.0;
            
            fit2 = (double *)cpl_calloc(sz_order_x * orders_nb_b / 2.0, sizeof(double));
            coeffs2 = (double *)cpl_calloc(pol_deg, sizeof(double));
            coeffs_err2 = (double *)cpl_calloc(pol_deg, sizeof(double));
            
            my_error = espdr_fit_poly(p2_data, pol_deg, fit2,
                                      coeffs2, coeffs_err2, &chisq2, 0);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_fit_poly failed: %s",
                         cpl_error_get_message_default(my_error));
            
            espdr_msg("2nd slice flux ratio smoothing coeffs:");
            for (int i = 0; i < pol_deg; i++) {
                espdr_msg("\t%.15f", coeffs2[i]);
            }
            espdr_msg("CHI2 = %f", chisq2);
            
            index = 0;
            for (order_b = 1; order_b < orders_nb_b; order_b += 2) {
                for (int j = 0; j < sz_order_x; j++) {
                    pix = sz_order_x*order_b+j;
                    ratio_data[pix] = 0.;
                    for (int c = pol_deg - 1; c >= 0; c--) {
                        ratio_data[pix] = ratio_data[pix] * (double)index + coeffs2[c];
                    }
                    index++;
                }
            }
            
            cpl_free(p2_data->x);
            cpl_free(p2_data->y);
            cpl_free(p2_data->err);
            
            cpl_free(fit2);
            cpl_free(coeffs2);
            cpl_free(coeffs_err2);
            cpl_free(p2_data);
            
        } else {
            
            espdr_poly_data *p_data = (espdr_poly_data *)cpl_malloc(sizeof(espdr_poly_data));
            
            p_data->x = (double *)cpl_calloc(sz_order_x * orders_nb_b, sizeof(double));
            p_data->y = (double *)cpl_calloc(sz_order_x * orders_nb_b, sizeof(double));
            p_data->err = (double *)cpl_calloc(sz_order_x * orders_nb_b, sizeof(double));
            
            double *fit = NULL;
            double *coeffs = NULL;
            double *coeffs_err = NULL;
            double chisq = 0.0;
            int index = 0;
            
            // smoothing of the 1st slice
            for (order_b = 0; order_b < orders_nb_b; order_b++) {
                for (int i = 0; i < sz_order_x; i++) {
                    pix = sz_order_x*order_b+i;
                    p_data->x[index] = (double)index;
                    p_data->y[index] = ratio_data[pix];
                    
                    if (ratio_data[pix] < 0.) {
                        p_data->err[index] = DBL_MAX;
                    } else {
                        p_data->err[index] = 1.;
                    }
                    index++;
                }
            }
            
            p_data->n = sz_order_x * orders_nb_b;
            
            fit = (double *)cpl_calloc(sz_order_x * orders_nb_b, sizeof(double));
            coeffs = (double *)cpl_calloc(pol_deg, sizeof(double));
            coeffs_err = (double *)cpl_calloc(pol_deg, sizeof(double));
            
            my_error = espdr_fit_poly(p_data, pol_deg, fit,
                                      coeffs, coeffs_err, &chisq, 0);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_fit_poly failed: %s",
                         cpl_error_get_message_default(my_error));
            
            espdr_msg("Flux ratio smoothing coeffs:");
            for (int i = 0; i < pol_deg; i++) {
                espdr_msg("\t%.15f", coeffs[i]);
            }
            espdr_msg("CHI2 = %f", chisq);
            
            index = 0;
            for (order_b = 0; order_b < orders_nb_b; order_b++) {
                for (int j = 0; j < sz_order_x; j++) {
                    pix = sz_order_x*order_b+j;
                    ratio_data[pix] = 0.;
                    for (int c = pol_deg - 1; c >= 0; c--) {
                        ratio_data[pix] = ratio_data[pix] * (double)index + coeffs[c];
                    }
                    index++;
                }
            }
            
            cpl_free(p_data->x);
            cpl_free(p_data->y);
            cpl_free(p_data->err);
            
            cpl_free(fit);
            cpl_free(coeffs);
            cpl_free(coeffs_err);
            cpl_free(p_data);
        }
    }
    
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief    Perform EFF_AB QC
 @param     inst_config instrument config
 @param     qc_kw       QC KWs names
 @param     contam      resulting image (?)
 @param     keywords_RE completed KWs list
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/


cpl_error_code espdr_cal_eff_ab_QC(espdr_CCD_geometry *CCD_geom,
                                   espdr_inst_config *inst_config,
                                   espdr_qc_keywords *qc_kws,
                                   int *cosmics_nb,
                                   double *snr,
                                   int orders_nr,
                                   cpl_propertylist *keywords_RE) {
    
    cpl_error_code my_error = CPL_ERROR_NONE;
    
    int i;
    char *new_keyword;
    int saturation_QC = 1;
    int snr_QC = 1;
    
    int global_QC = 1;
    
    saturation_QC = cpl_propertylist_get_int(keywords_RE, qc_kws->qc_saturation_check_kw);
    if (saturation_QC == 0) { global_QC = 0; }
    
    // Ading cosmics KWs
    for (i = 0; i < orders_nr; i++) {
        new_keyword = espdr_add_output_index_to_keyword(qc_kws->qc_cal_eff_ab_order_cosmic_nb_kw_first,
                                                        qc_kws->qc_cal_eff_ab_order_cosmic_nb_kw_last,
                                                        i+1);
        my_error = espdr_keyword_add_int(new_keyword, cosmics_nb[i],
                                         "Cosmics number in the order", &keywords_RE);
        cpl_free(new_keyword);
    }
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Adding one of the consmic keyword to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));
    
    // Adding SNR KWs
    double SNR_mean = 0.0;
    for (i = 0; i < orders_nr; i++) {
        new_keyword = espdr_add_output_index_to_keyword(qc_kws->qc_cal_eff_ab_order_snr_kw_first,
                                                        qc_kws->qc_cal_eff_ab_order_snr_kw_last,
                                                        i+1);
        my_error = espdr_keyword_add_double(new_keyword, snr[i],
                                            "SNR in the order", &keywords_RE);
        cpl_free(new_keyword);
        
        SNR_mean += snr[i];
    }
    SNR_mean = SNR_mean / orders_nr;
    if (SNR_mean < inst_config->eff_ab_snr_limit) {
        espdr_msg("SNR mean is %f", SNR_mean);
        snr_QC = 0;
        global_QC = 0;
    }
    
    my_error = espdr_keyword_add_int(qc_kws->qc_cal_eff_ab_snr_check_kw, snr_QC,
                                     "SNR QC", &keywords_RE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Adding one of the SNR keywords to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));
    
    my_error = espdr_keyword_add_int(qc_kws->qc_cal_eff_ab_check_kw, global_QC,
                                     "EFF_AB global QC", &keywords_RE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add keyword QC_EFF_AB_CHECK to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));
    
    return cpl_error_get_code();
    
}


/*----------------------------------------------------------------------------*/
/**
 @brief    Perform relative efficiency QC
 @param     inst_config instrument config
 @param     qc_kw       QC KWs names
 @param     contam      resulting image (?)
 @param     keywords_RE completed KWs list
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/


cpl_error_code espdr_rel_eff_QC(espdr_inst_config *inst_config,
                                espdr_qc_keywords *qc_kws,
                                cpl_image *eff_s2d,
                                cpl_propertylist *keywords_RE) {
    
    cpl_error_code my_error = CPL_ERROR_NONE;
    
    int min_eff_QC = 1;
    int max_eff_QC = 1;
    double min_eff = 0.0;
    double max_eff = 0.0;
    double avg_eff = 0.0;
    double rms_eff = 0.0;
    
    int global_QC = cpl_propertylist_get_int(keywords_RE, qc_kws->qc_cal_eff_ab_check_kw);
    
    espdr_ensure(eff_s2d == NULL, CPL_ERROR_NULL_INPUT,
                 "espdr_rel_eff_QC failed: NULL relative efficiency S2D");
    
    min_eff = cpl_image_get_min(eff_s2d);
    max_eff = cpl_image_get_max(eff_s2d);
    avg_eff = cpl_image_get_mean(eff_s2d);
    rms_eff = cpl_image_get_stdev(eff_s2d);
    
    my_error = espdr_keyword_add_double(qc_kws->qc_rel_eff_min_kw, min_eff,
                                        "Min relative efficiency", &keywords_RE);
    my_error = espdr_keyword_add_double(qc_kws->qc_rel_eff_max_kw, max_eff,
                                        "Max relative efficiency", &keywords_RE);
    
    if (min_eff < inst_config->rel_eff_min_limit) { min_eff_QC = 0; global_QC = 0; }
    if (max_eff > inst_config->rel_eff_max_limit) { max_eff_QC = 0; global_QC = 0; }
    my_error = espdr_keyword_add_int(qc_kws->qc_rel_eff_min_check_kw, min_eff_QC,
                                     "Min rel eff QC", &keywords_RE);
    my_error = espdr_keyword_add_int(qc_kws->qc_rel_eff_max_check_kw, max_eff_QC,
                                     "Max rel eff QC", &keywords_RE);
    
    my_error = espdr_keyword_add_double(qc_kws->qc_rel_eff_avg_kw,
                                        avg_eff,
                                        "Avg relative efficiency",
                                        &keywords_RE);
    
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add keyword QC_REL_EFF_AVG to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));
    
    my_error = espdr_keyword_add_double(qc_kws->qc_rel_eff_rms_kw,
                                        rms_eff,
                                        "RMS relative efficiency",
                                        &keywords_RE);
    
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add keyword QC_REL_EFF_RMS to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));
    
    my_error = espdr_keyword_add_int(qc_kws->qc_cal_eff_ab_check_kw, global_QC,
                                     "EFF_AB global QC", &keywords_RE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Adding one of the efficiency keywords to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));
    
    return cpl_error_get_code();
}



