/*
 * $Author: dsosnows $
 * $Date: 2013-08-05 17:38:28 $
 * $Revision: 1.4 $
 * $Name: not supported by cvs2svn $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*----------------------------------------------------------------------------
                                 Includes
 ----------------------------------------------------------------------------*/
#include <espdr_parameters.h>
#include <espdr_utils.h>
#include <espdr_msg.h>
#include <strings.h>


/*---------------------------------------------------------------------------*/
/**
  @brief Add a double parameter to the recipe parameters list

  @param recipe_id the current recipe in use
  @param list the parameters list to update
  @param name parameter name
  @param value parameter value
  @param comment parameter comment
 @return   CPL_ERROR_NONE iff OK
*/
/*---------------------------------------------------------------------------*/
cpl_error_code espdr_parameters_new_double(const char* recipe_id,
                                           cpl_parameterlist* list,
                                           const char* name,
                                           double value,
                                           const char* comment) {
	char paramname[PARAM_NAME_LENGTH];
	char recipename[PARAM_NAME_LENGTH];
	cpl_parameter *p = NULL;
    cpl_error_code my_error = CPL_ERROR_NONE;

    espdr_ensure(recipe_id == NULL, CPL_ERROR_NULL_INPUT, "recipe_id is NULL");
	espdr_ensure(list == NULL, CPL_ERROR_NULL_INPUT, "Parameters list is NULL");

	
	sprintf(recipename, "espdr.%s", recipe_id);
	sprintf(paramname, "%s.%s", recipename, name);
	espdr_ensure(name == NULL, CPL_ERROR_NULL_INPUT, "Parameter name is NULL");
	espdr_ensure(comment == NULL, CPL_ERROR_NULL_INPUT, "Parameter comment is NULL");

	p = cpl_parameter_new_value(paramname, CPL_TYPE_DOUBLE, 
								comment, recipename, value);
    espdr_ensure(p == NULL, cpl_error_get_code(),
                 "cpl_parameters_new_value failed: %s",
                 cpl_error_get_message());
    
	my_error = cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, name);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "cpl_parameters_set_alias failed: %s",
                 cpl_error_get_message_default(my_error));
    
	my_error = cpl_parameterlist_append(list, p);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "cpl_parameterlist_append failed: %s",
                 cpl_error_get_message_default(my_error));

	return (cpl_error_get_code());
}


/*---------------------------------------------------------------------------*/
/**
 @brief Add a double parameter to the recipe parameters list, checking
        the value range
 
 @param recipe_id the current recipe in use
 @param list the parameters list to update
 @param name parameter name
 @param value parameter value
 @param min allowed value
 @param max allowed value
 @param comment parameter comment
 @return   CPL_ERROR_NONE iff OK
 */
/*---------------------------------------------------------------------------*/
cpl_error_code espdr_parameters_new_range_double(const char* recipe_id,
                                                 cpl_parameterlist* list,
                                                 const char* name,
                                                 double value,
                                                 double min, double max,
                                                 const char* comment) {
	char paramname[PARAM_NAME_LENGTH];
	char recipename[PARAM_NAME_LENGTH];
	cpl_parameter *p = NULL;
    cpl_error_code my_error = CPL_ERROR_NONE;
    
	espdr_ensure(list == NULL, CPL_ERROR_NULL_INPUT, "Parameters list is NULL");
	
	sprintf(recipename, "espdr.%s", recipe_id);
	sprintf(paramname, "%s.%s", recipename, name);
	
	p = cpl_parameter_new_range(paramname, CPL_TYPE_DOUBLE,
								comment, recipename, value, min, max);
    espdr_ensure(p == NULL, cpl_error_get_code(),
                 "cpl_parameter_new_range failed: %s",
                 cpl_error_get_message());
    
	my_error = cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, name);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "cpl_parameter_set_alias failed: %s",
                 cpl_error_get_message_default(my_error));
    
	my_error = cpl_parameterlist_append(list, p);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "cpl_parameterlist_append failed: %s",
                 cpl_error_get_message_default(my_error));
    
	return (CPL_ERROR_NONE);
}


/*---------------------------------------------------------------------------*/
/**
 @brief Add a boolean parameter to the recipe parameters list
 
 @param recipe_id the current recipe in use
 @param list the parameters list to update
 @param name parameter name
 @param value parameter value
 @param comment parameter comment
 @return   CPL_ERROR_NONE iff OK
 */
/*---------------------------------------------------------------------------*/
cpl_error_code espdr_parameters_new_bool(const char* recipe_id,
                                         cpl_parameterlist* list,
                                         const char* name,
                                         cpl_boolean value,
                                         const char* comment) {
    char paramname[PARAM_NAME_LENGTH];
    char recipename[PARAM_NAME_LENGTH];
    cpl_parameter *p = NULL;
    cpl_error_code my_error = CPL_ERROR_NONE;
    
    espdr_ensure(recipe_id == NULL, CPL_ERROR_NULL_INPUT, "recipe_id is NULL");
    espdr_ensure(list == NULL, CPL_ERROR_NULL_INPUT, "Parameters list is NULL");
    espdr_ensure(name == NULL, CPL_ERROR_NULL_INPUT, "Parameter name is NULL");
    espdr_ensure(comment == NULL, CPL_ERROR_NULL_INPUT, "Parameter comment is NULL");
    
    sprintf(recipename, "espdr.%s", recipe_id);
    sprintf(paramname, "%s.%s", recipename, name);
    
    p = cpl_parameter_new_value(paramname, CPL_TYPE_BOOL, comment,
                                recipename, value);
    espdr_ensure(p == NULL, cpl_error_get_code(),
                 "cpl_parameters_new_value failed: %s",
                 cpl_error_get_message());
    
    my_error = cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, name);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "cpl_parameters_set_alias failed: %s",
                 cpl_error_get_message_default(my_error));
    
    my_error = cpl_parameterlist_append(list, p);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "cpl_parameterlist_append failed: %s",
                 cpl_error_get_message_default(my_error));
    
    return (CPL_ERROR_NONE);
}


/*---------------------------------------------------------------------------*/
/**
 @brief Add a int parameter to the recipe parameters list
 
 @param recipe_id the current recipe in use
 @param list the parameters list to update
 @param name parameter name
 @param value parameter value
 @param comment parameter comment
 @return   CPL_ERROR_NONE iff OK
 */
/*---------------------------------------------------------------------------*/
cpl_error_code espdr_parameters_new_int(const char* recipe_id, 
									  cpl_parameterlist* list,
									  const char* name, 
									  int value, 
									  const char* comment) {
	char paramname[PARAM_NAME_LENGTH];
	char recipename[PARAM_NAME_LENGTH];
	cpl_parameter *p = NULL;
    cpl_error_code my_error = CPL_ERROR_NONE;

    espdr_ensure(recipe_id == NULL, CPL_ERROR_NULL_INPUT, "recipe_id is NULL");
	espdr_ensure(list == NULL, CPL_ERROR_NULL_INPUT, "Parameters list is NULL");
	espdr_ensure(name == NULL, CPL_ERROR_NULL_INPUT, "Parameter name is NULL");
	espdr_ensure(comment == NULL, CPL_ERROR_NULL_INPUT, "Parameter comment is NULL");

	sprintf(recipename, "espdr.%s", recipe_id);
	sprintf(paramname, "%s.%s", recipename, name);
	
	p = cpl_parameter_new_value(paramname, CPL_TYPE_INT, comment, 
								recipename, value);
    espdr_ensure(p == NULL, cpl_error_get_code(),
                 "cpl_parameters_new_value failed: %s",
                 cpl_error_get_message());
    
	my_error = cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, name);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "cpl_parameters_set_alias failed: %s",
                 cpl_error_get_message_default(my_error));
    
	my_error = cpl_parameterlist_append(list, p);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "cpl_parameterlist_append failed: %s",
                 cpl_error_get_message_default(my_error));
	
	return (CPL_ERROR_NONE);
}


/*---------------------------------------------------------------------------*/
/**
 @brief Add a int parameter to the recipe parameters list, checking 
        the value range
 
 @param recipe_id the current recipe in use
 @param list the parameters list to update
 @param name parameter name
 @param value parameter value
 @param min allowed value
 @param max allowed value
 @param comment parameter comment
 @return   CPL_ERROR_NONE iff OK
 */
/*---------------------------------------------------------------------------*/
cpl_error_code espdr_parameters_new_range_int(const char* recipe_id,
                                              cpl_parameterlist* list,
                                              const char* name,
                                              int value,
                                              int min, int max,
                                              const char* comment) {
	char paramname[PARAM_NAME_LENGTH];
	char recipename[PARAM_NAME_LENGTH];
	cpl_parameter *p = NULL;
    cpl_error_code my_error = CPL_ERROR_NONE;
	
	espdr_ensure(list == NULL, CPL_ERROR_NULL_INPUT, "Parameters list is NULL");
    
	sprintf(recipename, "espdr.%s", recipe_id);
	sprintf(paramname, "%s.%s", recipename, name);
	
	p = cpl_parameter_new_range(paramname, CPL_TYPE_INT, comment,
								recipename, value, min, max);
    espdr_ensure(p == NULL, cpl_error_get_code(),
                 "cpl_parameters_new_range failed: %s",
                 cpl_error_get_message());
    
	my_error = cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, name);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "cpl_parameters_set_alias failed: %s",
                 cpl_error_get_message_default(my_error));
    
	my_error = cpl_parameterlist_append(list, p);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "cpl_parameterlist_append failed: %s",
                 cpl_error_get_message_default(my_error));
	
	return (CPL_ERROR_NONE);
}


/*---------------------------------------------------------------------------*/
/**
 @brief Add a string parameter to the recipe parameters list
 
 @param recipe_id the current recipe in use
 @param list the parameters list to update
 @param name parameter name
 @param value parameter value
 @param comment parameter comment
 @return   CPL_ERROR_NONE iff OK
 */
/*---------------------------------------------------------------------------*/
cpl_error_code espdr_parameters_new_string(const char* recipe_id,
										 cpl_parameterlist* list,
										 const char* name,
										 const char* value,
										 const char* comment) {
	char paramname[PARAM_NAME_LENGTH];
	char recipename[PARAM_NAME_LENGTH];
	cpl_parameter *p = NULL;
    cpl_error_code my_error = CPL_ERROR_NONE;

    espdr_ensure(recipe_id == NULL, CPL_ERROR_NULL_INPUT, "recipe_id is NULL");
	espdr_ensure(list == NULL, CPL_ERROR_NULL_INPUT, "Parameters list is NULL");
	espdr_ensure(name == NULL, CPL_ERROR_NULL_INPUT, "Parameter name is NULL");
	espdr_ensure(comment == NULL, CPL_ERROR_NULL_INPUT, "Parameter comment is NULL");
		
	sprintf(recipename, "espdr.%s", recipe_id);
	sprintf(paramname, "%s.%s", recipename, name);
	
	p = cpl_parameter_new_value(paramname, CPL_TYPE_STRING, comment,
								recipename, value);
    espdr_ensure(p == NULL, cpl_error_get_code(),
                 "cpl_parameters_new_value failed: %s",
                 cpl_error_get_message());
    
	my_error = cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, name);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "cpl_parameters_set_alias failed: %s",
                 cpl_error_get_message_default(my_error));
    
	my_error = cpl_parameterlist_append(list, p);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "cpl_parameterlist_append failed: %s",
                 cpl_error_get_message_default(my_error));
	
	return (CPL_ERROR_NONE);
}


/*---------------------------------------------------------------------------*/
/**
 @brief Add a string parameter to the recipe parameters list
 
 @param recipe_id the current recipe in use
 @param list the parameters list to update
 @param name parameter name
 @param value parameter value
 @param comment parameter comment
 @return   CPL_ERROR_NONE iff OK
 */
/*---------------------------------------------------------------------------*/
cpl_error_code espdr_parameters_new_method(const char* recipe_id,
                                           cpl_parameterlist* list,
                                           const char* name,
                                           const char* value,
                                           const char* comment) {
	char paramname[PARAM_NAME_LENGTH];
	char recipename[PARAM_NAME_LENGTH];
	cpl_parameter *p = NULL;
    cpl_error_code my_error = CPL_ERROR_NONE;
	
	espdr_ensure(list == NULL, CPL_ERROR_NULL_INPUT, "Parameters list is NULL");
    
	sprintf(recipename, "espdr.%s", recipe_id);
	sprintf(paramname, "%s.%s", recipename, name);
	
    if (strcmp(value, "mean") == 0) {
        p = cpl_parameter_new_value(paramname, CPL_TYPE_INT, comment,
                                    recipename, ESPDR_SIG_CLIP_METHOD_MEAN);
        espdr_ensure(p == NULL, cpl_error_get_code(),
                     "cpl_parameters_new_value failed: %s",
                     cpl_error_get_message());
    } else {
        if (strcmp(value, "median") == 0) {
            p = cpl_parameter_new_value(paramname, CPL_TYPE_INT, comment,
                                        recipename,
                                        ESPDR_SIG_CLIP_METHOD_MEDIAN);
            espdr_ensure(p == NULL, cpl_error_get_code(),
                         "cpl_parameters_new_value failed: %s",
                         cpl_error_get_message());
        } else {
            espdr_ensure(0, CPL_ERROR_ILLEGAL_INPUT,
                         "Unknown sigma clipping method: %s", value);
        }
    }
    
	my_error = cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, name);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "cpl_parameters_set_alias failed: %s",
                 cpl_error_get_message_default(my_error));
    
	my_error = cpl_parameterlist_append(list, p);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "cpl_parameterlist_append failed: %s",
                 cpl_error_get_message_default(my_error));
	
	return (CPL_ERROR_NONE);
}


/*---------------------------------------------------------------------------*/
/**
  @brief Get a double parameter from the recipe parameters list

  @param recipe_id the current recipe in use
  @param list the parameters list to update
  @param name parameter name
  @return parameter value
*/
/*---------------------------------------------------------------------------*/
double espdr_parameters_get_double(const char* recipe_id, 
								 cpl_parameterlist* list, 
								 const char* name)
{
	char paramname[PARAM_NAME_LENGTH];
	char recipename[PARAM_NAME_LENGTH];
	cpl_parameter* p = NULL;
	double result = 0.0;

	espdr_ensure(recipe_id == NULL, CPL_ERROR_NULL_INPUT, "recipe_id is NULL");
	espdr_ensure(list == NULL, CPL_ERROR_NULL_INPUT, "Parameters list is NULL");
	espdr_ensure(name == NULL, CPL_ERROR_NULL_INPUT, "Parameter name is NULL");
	
	sprintf(recipename,"espdr.%s",recipe_id);
	sprintf(paramname,"%s.%s",recipename,name);

	p = cpl_parameterlist_find(list,paramname);
	result = cpl_parameter_get_double(p);

	return result;
}


/*---------------------------------------------------------------------------*/
/**
 @brief Get a bool parameter from the recipe parameters list
 
 @param recipe_id the current recipe in use
 @param list the parameters list to update
 @param name parameter name
 @return parameter value
 */
/*---------------------------------------------------------------------------*/
cpl_boolean espdr_parameters_get_bool(const char* recipe_id,
                                      cpl_parameterlist* list,
                                      const char* name)
{
    char paramname[PARAM_NAME_LENGTH];
    char recipename[PARAM_NAME_LENGTH];
    cpl_parameter* p =NULL;
    cpl_boolean result = 0;
    
    espdr_ensure(recipe_id == NULL, CPL_ERROR_NULL_INPUT, "recipe_id is NULL");
    espdr_ensure(list == NULL, CPL_ERROR_NULL_INPUT, "Parameters list is NULL");
    espdr_ensure(name == NULL, CPL_ERROR_NULL_INPUT, "Parameter name is NULL");
    
    sprintf(recipename,"espdr.%s",recipe_id);
    sprintf(paramname,"%s.%s",recipename,name);
    
    p = cpl_parameterlist_find(list,paramname);
    result = cpl_parameter_get_bool(p);
    
    return result;
}


/*---------------------------------------------------------------------------*/
/**
 @brief Get a int parameter from the recipe parameters list
 
 @param recipe_id the current recipe in use
 @param list the parameters list to update
 @param name parameter name
 @return parameter value
 */
/*---------------------------------------------------------------------------*/
int espdr_parameters_get_int(const char* recipe_id, 
						   cpl_parameterlist* list, 
						   const char* name)
{
	char paramname[PARAM_NAME_LENGTH];
	char recipename[PARAM_NAME_LENGTH];
	cpl_parameter* p =NULL;
	int result = 0;
	
	espdr_ensure(recipe_id == NULL, CPL_ERROR_NULL_INPUT, "recipe_id is NULL");
	espdr_ensure(list == NULL, CPL_ERROR_NULL_INPUT, "Parameters list is NULL");
	espdr_ensure(name == NULL, CPL_ERROR_NULL_INPUT, "Parameter name is NULL");
	
	sprintf(recipename,"espdr.%s",recipe_id);
	sprintf(paramname,"%s.%s",recipename,name);
	
	p = cpl_parameterlist_find(list,paramname);
	result = cpl_parameter_get_int(p);
	
	return result;
}


/*---------------------------------------------------------------------------*/
/**
  @brief Get a string parameter from the recipe parameters list
  @param recipe_id the current recipe in use
  @param list the parameters list to update
  @param name parameter name
  @return The string parameter or NULL on error
*/
/*---------------------------------------------------------------------------*/
const char *espdr_parameters_get_string(const char        * recipe_id,
                                      cpl_parameterlist * list,
                                      const char        * name)
{
    char * paramname = NULL;
    char * recipename = NULL;
    cpl_parameter* p = NULL;
    char *result = NULL;

	//espdr_ensure(list == NULL, CPL_ERROR_NULL_INPUT, "Parameters list is NULL");
    if (list == NULL) {
        espdr_msg("Parameters list is NULL");
        cpl_error_set(__FUNCTION__, CPL_ERROR_NULL_INPUT);
        return (NULL);
    }
    
    if (recipe_id == NULL) {
        espdr_msg("recipe_id is NULL");
        cpl_error_set(__FUNCTION__, CPL_ERROR_NULL_INPUT);
        return (NULL);
    }
    
    if (name == NULL) {
        espdr_msg("Parameters name is NULL");
        cpl_error_set(__FUNCTION__, CPL_ERROR_NULL_INPUT);
        return (NULL);
    }
    
    //espdr_ensure(recipe_id == NULL, CPL_ERROR_NULL_INPUT, "recipe_id is NULL");
    //espdr_ensure(name == NULL, CPL_ERROR_NULL_INPUT, "Parameter name is NULL");

    recipename = cpl_sprintf("espdr.%s", recipe_id);
    paramname = cpl_sprintf("%s.%s", recipename, name);

    p = cpl_parameterlist_find(list, paramname);
    result = cpl_sprintf("%s", cpl_parameter_get_string(p));

    cpl_free(paramname);
    cpl_free(recipename);

    return result;
}



