/*
 * This file is part of the ESO Common Pipeline Library
 * Copyright (C) 2014 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifndef ESPDR_SDP_SPECTRUM_H
#define ESPDR_SDP_SPECTRUM_H

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include <cpl.h>
#include <math.h>

CPL_BEGIN_DECLS

/*-----------------------------------------------------------------------------
                                   Defines
 -----------------------------------------------------------------------------*/

/* Define a default value for NAN if not already defined.
 * Note: one should add a test to configure.ac to check if the compiler can
 * handle NAN defined as (0.0/0.0) and define espdr_USE_ZERO_BY_ZERO_FOR_NAN
 * appropriately when compiling this file. */
#ifndef NAN
#  ifdef espdr_USE_ZERO_BY_ZERO_FOR_NAN

#    define NAN (0.0/0.0)

#  else /* espdr_USE_ZERO_BY_ZERO_FOR_NAN */

#    include <assert.h>
#    include <stdint.h>
#    include <string.h>

     inline static float _espdr_nan_const(void)
     {
       /* Perform type-punning to convert binary representation to the floating
        * point format. We use memcpy since this should be the most portable.
        * Unions are OK only from C99, though in principle most older compilers
        * should anyway behave in the expected manner. The _espdr_nan_const is
        * actually a workaround for the better alternative of using 0.0/0.0
        * directly in the code, which should work with a modern compiler. Thus,
        * we just stick to memcpy to avoid strict aliasing rule violations.
        */
       uint32_t i = 0xFFC00000;
       float f;
       assert(sizeof(i) == sizeof(f));
       memcpy(&f, &i, sizeof(f));
       return f;
     }

#    define NAN _espdr_nan_const()

#  endif /* espdr_USE_ZERO_BY_ZERO_FOR_NAN */
#endif /* NAN */

#define ESPDR_ANCILLARY_CCF         "ANCILLARY.CCF"
#define ESPDR_ANCILLARY_DRIFT       "ANCILLARY.DRIFT"
#define ESPDR_ANCILLARY_SPECTRUM    "ANCILLARY.SPECTRUM"

/*-----------------------------------------------------------------------------
                                   Type declarations
 -----------------------------------------------------------------------------*/

/**
 * @ingroup espdr_sdp_spectrum
 * @brief Data type for a Science Data Product 1D spectrum.
 */
typedef struct _espdr_sdp_spectrum_ espdr_sdp_spectrum;

/**
 * @ingroup espdr_sdp_spectrum
 * @brief Bitfield flags for the column update function.
 */
typedef enum _espdr_sdp_spectrum_update_flags_ {
  /* Indicates the units should be copied. */
  ESPDR_COLUMN_UNIT    = 1 << 1,

  /* Indicates the format string should be copied. */
  ESPDR_COLUMN_FORMAT  = 1 << 2,

  /* Indicates the data should be copied. */
  ESPDR_COLUMN_DATA    = 1 << 3

} espdr_sdp_spectrum_update_flags;

/*-----------------------------------------------------------------------------
                                   Prototypes
 -----------------------------------------------------------------------------*/

/*
 * Create, copy and destroy operations.
 */

espdr_sdp_spectrum * espdr_sdp_spectrum_new(void) CPL_ATTR_ALLOC;

espdr_sdp_spectrum *
espdr_sdp_spectrum_duplicate(const espdr_sdp_spectrum *other) CPL_ATTR_ALLOC;

void espdr_sdp_spectrum_delete(espdr_sdp_spectrum *self);

/*
 * Method to compare two spectra.
 */

cpl_boolean espdr_sdp_spectrum_equal(const espdr_sdp_spectrum *a,
                                      const espdr_sdp_spectrum *b,
                                      cpl_boolean only_intersect);

/*
 * Methods to count number of elements for keyword series.
 */
cpl_size espdr_sdp_spectrum_count_obid(const espdr_sdp_spectrum *self);
cpl_size espdr_sdp_spectrum_count_prov(const espdr_sdp_spectrum *self);
cpl_size espdr_sdp_spectrum_count_asson(const espdr_sdp_spectrum *self);
cpl_size espdr_sdp_spectrum_count_assoc(const espdr_sdp_spectrum *self);
cpl_size espdr_sdp_spectrum_count_assom(const espdr_sdp_spectrum *self);

/*
 * Methods to copy keyword values from property objects.
 */
cpl_error_code espdr_sdp_spectrum_copy_keyword(espdr_sdp_spectrum *self,
                                                const cpl_propertylist *plist,
                                                const char *name);

cpl_error_code espdr_sdp_spectrum_copy_property(espdr_sdp_spectrum *self,
                                                 const cpl_property *prop);

cpl_error_code espdr_sdp_spectrum_copy_property_regexp(
                                                espdr_sdp_spectrum *self,
                                                const cpl_propertylist *plist,
                                                const char *regexp,
                                                int invert);

/*
 * Get/set methods for keywords.
 */

double espdr_sdp_spectrum_get_ra(const espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_reset_ra(espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_set_ra(espdr_sdp_spectrum *self,
                                          double value);
cpl_error_code espdr_sdp_spectrum_copy_ra(espdr_sdp_spectrum *self,
                                           const cpl_propertylist *plist,
                                           const char *name);

double espdr_sdp_spectrum_get_dec(const espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_reset_dec(espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_set_dec(espdr_sdp_spectrum *self,
                                           double value);
cpl_error_code espdr_sdp_spectrum_copy_dec(espdr_sdp_spectrum *self,
                                            const cpl_propertylist *plist,
                                            const char *name);

double espdr_sdp_spectrum_get_exptime(const espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_reset_exptime(espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_set_exptime(espdr_sdp_spectrum *self,
                                               double value);
cpl_error_code espdr_sdp_spectrum_copy_exptime(espdr_sdp_spectrum *self,
                                                const cpl_propertylist *plist,
                                                const char *name);

double espdr_sdp_spectrum_get_texptime(const espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_reset_texptime(espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_set_texptime(espdr_sdp_spectrum *self,
                                                double value);
cpl_error_code espdr_sdp_spectrum_copy_texptime(espdr_sdp_spectrum *self,
                                                 const cpl_propertylist *plist,
                                                 const char *name);

double espdr_sdp_spectrum_get_mjdobs(const espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_reset_mjdobs(espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_set_mjdobs(espdr_sdp_spectrum *self,
                                              double value);
cpl_error_code espdr_sdp_spectrum_copy_mjdobs(espdr_sdp_spectrum *self,
                                               const cpl_propertylist *plist,
                                               const char *name);

const char * espdr_sdp_spectrum_get_timesys(const espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_reset_timesys(espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_set_timesys(espdr_sdp_spectrum *self,
                                               const char *value);
cpl_error_code espdr_sdp_spectrum_copy_timesys(espdr_sdp_spectrum *self,
                                                const cpl_propertylist *plist,
                                                const char *name);

double espdr_sdp_spectrum_get_mjdend(const espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_reset_mjdend(espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_set_mjdend(espdr_sdp_spectrum *self,
                                              double value);
cpl_error_code espdr_sdp_spectrum_copy_mjdend(espdr_sdp_spectrum *self,
                                               const cpl_propertylist *plist,
                                               const char *name);

int espdr_sdp_spectrum_get_prodlvl(const espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_reset_prodlvl(espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_set_prodlvl(espdr_sdp_spectrum *self,
                                               int value);
cpl_error_code espdr_sdp_spectrum_copy_prodlvl(espdr_sdp_spectrum *self,
                                                const cpl_propertylist *plist,
                                                const char *name);

const char * espdr_sdp_spectrum_get_procsoft(const espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_reset_procsoft(espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_set_procsoft(espdr_sdp_spectrum *self,
                                                const char *value);
cpl_error_code espdr_sdp_spectrum_copy_procsoft(espdr_sdp_spectrum *self,
                                                 const cpl_propertylist *plist,
                                                 const char *name);

const char * espdr_sdp_spectrum_get_prodcatg(const espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_reset_prodcatg(espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_set_prodcatg(espdr_sdp_spectrum *self,
                                                const char *value);
cpl_error_code espdr_sdp_spectrum_copy_prodcatg(espdr_sdp_spectrum *self,
                                                 const cpl_propertylist *plist,
                                                 const char *name);

const char * espdr_sdp_spectrum_get_origin(const espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_reset_origin(espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_set_origin(espdr_sdp_spectrum *self,
                                              const char *value);
cpl_error_code espdr_sdp_spectrum_copy_origin(espdr_sdp_spectrum *self,
                                               const cpl_propertylist *plist,
                                               const char *name);

cpl_boolean espdr_sdp_spectrum_get_extobj(const espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_reset_extobj(espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_set_extobj(espdr_sdp_spectrum *self,
                                              cpl_boolean value);
cpl_error_code espdr_sdp_spectrum_copy_extobj(espdr_sdp_spectrum *self,
                                               const cpl_propertylist *plist,
                                               const char *name);

const char * espdr_sdp_spectrum_get_dispelem(const espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_reset_dispelem(espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_set_dispelem(espdr_sdp_spectrum *self,
                                                const char *value);
cpl_error_code espdr_sdp_spectrum_copy_dispelem(espdr_sdp_spectrum *self,
                                                 const cpl_propertylist *plist,
                                                 const char *name);

const char * espdr_sdp_spectrum_get_specsys(const espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_reset_specsys(espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_set_specsys(espdr_sdp_spectrum *self,
                                               const char *value);
cpl_error_code espdr_sdp_spectrum_copy_specsys(espdr_sdp_spectrum *self,
                                                const cpl_propertylist *plist,
                                                const char *name);

const char * espdr_sdp_spectrum_get_progid(const espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_reset_progid(espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_set_progid(espdr_sdp_spectrum *self,
                                              const char *value);
cpl_error_code espdr_sdp_spectrum_copy_progid(espdr_sdp_spectrum *self,
                                               const cpl_propertylist *plist,
                                               const char *name);

int espdr_sdp_spectrum_get_obid(const espdr_sdp_spectrum *self,
                                 cpl_size index);
cpl_error_code espdr_sdp_spectrum_reset_obid(espdr_sdp_spectrum *self,
                                              cpl_size index);
cpl_error_code espdr_sdp_spectrum_set_obid(espdr_sdp_spectrum *self,
                                            cpl_size index, int value);
cpl_error_code espdr_sdp_spectrum_copy_obid(espdr_sdp_spectrum *self,
                                             cpl_size index,
                                             const cpl_propertylist *plist,
                                             const char *name);

cpl_boolean espdr_sdp_spectrum_get_mepoch(const espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_reset_mepoch(espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_set_mepoch(espdr_sdp_spectrum *self,
                                              cpl_boolean value);
cpl_error_code espdr_sdp_spectrum_copy_mepoch(espdr_sdp_spectrum *self,
                                               const cpl_propertylist *plist,
                                               const char *name);

const char * espdr_sdp_spectrum_get_obstech(const espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_reset_obstech(espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_set_obstech(espdr_sdp_spectrum *self,
                                               const char *value);
cpl_error_code espdr_sdp_spectrum_copy_obstech(espdr_sdp_spectrum *self,
                                                const cpl_propertylist *plist,
                                                const char *name);

const char * espdr_sdp_spectrum_get_fluxcal(const espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_reset_fluxcal(espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_set_fluxcal(espdr_sdp_spectrum *self,
                                               const char *value);
cpl_error_code espdr_sdp_spectrum_copy_fluxcal(espdr_sdp_spectrum *self,
                                                const cpl_propertylist *plist,
                                                const char *name);

cpl_boolean espdr_sdp_spectrum_get_contnorm(const espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_reset_contnorm(espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_set_contnorm(espdr_sdp_spectrum *self,
                                                cpl_boolean value);
cpl_error_code espdr_sdp_spectrum_set_noesodat(espdr_sdp_spectrum *self,
                                                cpl_boolean value);
cpl_error_code espdr_sdp_spectrum_copy_contnorm(espdr_sdp_spectrum *self,
                                                 const cpl_propertylist *plist,
                                                 const char *name);

double espdr_sdp_spectrum_get_wavelmin(const espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_reset_wavelmin(espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_set_wavelmin(espdr_sdp_spectrum *self,
                                                double value);
cpl_error_code espdr_sdp_spectrum_copy_wavelmin(espdr_sdp_spectrum *self,
                                                 const cpl_propertylist *plist,
                                                 const char *name);

double espdr_sdp_spectrum_get_wavelmax(const espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_reset_wavelmax(espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_set_wavelmax(espdr_sdp_spectrum *self,
                                                double value);
cpl_error_code espdr_sdp_spectrum_copy_wavelmax(espdr_sdp_spectrum *self,
                                                 const cpl_propertylist *plist,
                                                 const char *name);

double espdr_sdp_spectrum_get_specbin(const espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_reset_specbin(espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_set_specbin(espdr_sdp_spectrum *self,
                                               double value);
cpl_error_code espdr_sdp_spectrum_copy_specbin(espdr_sdp_spectrum *self,
                                                const cpl_propertylist *plist,
                                                const char *name);

cpl_boolean espdr_sdp_spectrum_get_totflux(const espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_reset_totflux(espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_set_totflux(espdr_sdp_spectrum *self,
                                               cpl_boolean value);
cpl_error_code espdr_sdp_spectrum_copy_totflux(espdr_sdp_spectrum *self,
                                                const cpl_propertylist *plist,
                                                const char *name);

double espdr_sdp_spectrum_get_fluxerr(const espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_reset_fluxerr(espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_set_fluxerr(espdr_sdp_spectrum *self,
                                               double value);
cpl_error_code espdr_sdp_spectrum_copy_fluxerr(espdr_sdp_spectrum *self,
                                                const cpl_propertylist *plist,
                                                const char *name);

const char * espdr_sdp_spectrum_get_referenc(const espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_reset_referenc(espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_set_referenc(espdr_sdp_spectrum *self,
                                                const char *value);
cpl_error_code espdr_sdp_spectrum_copy_referenc(espdr_sdp_spectrum *self,
                                                 const cpl_propertylist *plist,
                                                 const char *name);

double espdr_sdp_spectrum_get_specres(const espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_reset_specres(espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_set_specres(espdr_sdp_spectrum *self,
                                               double value);
cpl_error_code espdr_sdp_spectrum_copy_specres(espdr_sdp_spectrum *self,
                                                const cpl_propertylist *plist,
                                                const char *name);

double espdr_sdp_spectrum_get_specerr(const espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_reset_specerr(espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_set_specerr(espdr_sdp_spectrum *self,
                                               double value);
cpl_error_code espdr_sdp_spectrum_copy_specerr(espdr_sdp_spectrum *self,
                                                const cpl_propertylist *plist,
                                                const char *name);

double espdr_sdp_spectrum_get_specsye(const espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_reset_specsye(espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_set_specsye(espdr_sdp_spectrum *self,
                                               double value);
cpl_error_code espdr_sdp_spectrum_copy_specsye(espdr_sdp_spectrum *self,
                                                const cpl_propertylist *plist,
                                                const char *name);

int espdr_sdp_spectrum_get_lamnlin(const espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_reset_lamnlin(espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_set_lamnlin(espdr_sdp_spectrum *self,
                                               int value);
cpl_error_code espdr_sdp_spectrum_copy_lamnlin(espdr_sdp_spectrum *self,
                                                const cpl_propertylist *plist,
                                                const char *name);

double espdr_sdp_spectrum_get_lamrms(const espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_reset_lamrms(espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_set_lamrms(espdr_sdp_spectrum *self,
                                              double value);
cpl_error_code espdr_sdp_spectrum_copy_lamrms(espdr_sdp_spectrum *self,
                                               const cpl_propertylist *plist,
                                               const char *name);

double espdr_sdp_spectrum_get_gain(const espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_reset_gain(espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_set_gain(espdr_sdp_spectrum *self,
                                            double value);
cpl_error_code espdr_sdp_spectrum_copy_gain(espdr_sdp_spectrum *self,
                                             const cpl_propertylist *plist,
                                             const char *name);

double espdr_sdp_spectrum_get_detron(const espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_reset_detron(espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_set_detron(espdr_sdp_spectrum *self,
                                              double value);
cpl_error_code espdr_sdp_spectrum_copy_detron(espdr_sdp_spectrum *self,
                                               const cpl_propertylist *plist,
                                               const char *name);

double espdr_sdp_spectrum_get_effron(const espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_reset_effron(espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_set_effron(espdr_sdp_spectrum *self,
                                              double value);
cpl_error_code espdr_sdp_spectrum_copy_effron(espdr_sdp_spectrum *self,
                                               const cpl_propertylist *plist,
                                               const char *name);

double espdr_sdp_spectrum_get_snr(const espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_reset_snr(espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_set_snr(espdr_sdp_spectrum *self,
                                           double value);
cpl_error_code espdr_sdp_spectrum_copy_snr(espdr_sdp_spectrum *self,
                                            const cpl_propertylist *plist,
                                            const char *name);

int espdr_sdp_spectrum_get_ncombine(const espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_reset_ncombine(espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_set_ncombine(espdr_sdp_spectrum *self,
                                                int value);
cpl_error_code espdr_sdp_spectrum_copy_ncombine(espdr_sdp_spectrum *self,
                                                 const cpl_propertylist *plist,
                                                 const char *name);

const char * espdr_sdp_spectrum_get_prov(const espdr_sdp_spectrum *self,
                                          cpl_size index);
cpl_error_code espdr_sdp_spectrum_reset_prov(espdr_sdp_spectrum *self,
                                              cpl_size index);
cpl_error_code espdr_sdp_spectrum_set_prov(espdr_sdp_spectrum *self,
                                            cpl_size index, const char *value);
cpl_error_code espdr_sdp_spectrum_copy_prov(espdr_sdp_spectrum *self,
                                             cpl_size index,
                                             const cpl_propertylist *plist,
                                             const char *name);
cpl_error_code espdr_sdp_spectrum_append_prov(espdr_sdp_spectrum *self,
                                               cpl_size firstindex,
                                               const cpl_frameset *frames);

const char * espdr_sdp_spectrum_get_asson(const espdr_sdp_spectrum *self,
                                           cpl_size index);
cpl_error_code espdr_sdp_spectrum_reset_asson(espdr_sdp_spectrum *self,
                                               cpl_size index);
cpl_error_code espdr_sdp_spectrum_set_asson(espdr_sdp_spectrum *self,
                                             cpl_size index, const char *value);
cpl_error_code espdr_sdp_spectrum_copy_asson(espdr_sdp_spectrum *self,
                                              cpl_size index,
                                              const cpl_propertylist *plist,
                                              const char *name);

const char * espdr_sdp_spectrum_get_assoc(const espdr_sdp_spectrum *self,
                                           cpl_size index);
cpl_error_code espdr_sdp_spectrum_reset_assoc(espdr_sdp_spectrum *self,
                                               cpl_size index);
cpl_error_code espdr_sdp_spectrum_set_assoc(espdr_sdp_spectrum *self,
                                             cpl_size index, const char *value);
cpl_error_code espdr_sdp_spectrum_copy_assoc(espdr_sdp_spectrum *self,
                                              cpl_size index,
                                              const cpl_propertylist *plist,
                                              const char *name);

const char * espdr_sdp_spectrum_get_assom(const espdr_sdp_spectrum *self,
                                           cpl_size index);
cpl_error_code espdr_sdp_spectrum_reset_assom(espdr_sdp_spectrum *self,
                                               cpl_size index);
cpl_error_code espdr_sdp_spectrum_set_assom(espdr_sdp_spectrum *self,
                                             cpl_size index, const char *value);
cpl_error_code espdr_sdp_spectrum_copy_assom(espdr_sdp_spectrum *self,
                                              cpl_size index,
                                              const cpl_propertylist *plist,
                                              const char *name);

const char * espdr_sdp_spectrum_get_voclass(const espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_reset_voclass(espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_set_voclass(espdr_sdp_spectrum *self,
                                               const char *value);
cpl_error_code espdr_sdp_spectrum_copy_voclass(espdr_sdp_spectrum *self,
                                                const cpl_propertylist *plist,
                                                const char *name);

const char * espdr_sdp_spectrum_get_vopub(const espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_reset_vopub(espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_set_vopub(espdr_sdp_spectrum *self,
                                             const char *value);
cpl_error_code espdr_sdp_spectrum_copy_vopub(espdr_sdp_spectrum *self,
                                              const cpl_propertylist *plist,
                                              const char *name);

const char * espdr_sdp_spectrum_get_title(const espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_reset_title(espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_set_title(espdr_sdp_spectrum *self,
                                             const char *value);
cpl_error_code espdr_sdp_spectrum_copy_title(espdr_sdp_spectrum *self,
                                              const cpl_propertylist *plist,
                                              const char *name);

const char * espdr_sdp_spectrum_get_object(const espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_reset_object(espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_set_object(espdr_sdp_spectrum *self,
                                              const char *value);
cpl_error_code espdr_sdp_spectrum_copy_object(espdr_sdp_spectrum *self,
                                               const cpl_propertylist *plist,
                                               const char *name);

double espdr_sdp_spectrum_get_aperture(const espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_reset_aperture(espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_set_aperture(espdr_sdp_spectrum *self,
                                                double value);
cpl_error_code espdr_sdp_spectrum_copy_aperture(espdr_sdp_spectrum *self,
                                                 const cpl_propertylist *plist,
                                                 const char *name);

double espdr_sdp_spectrum_get_telapse(const espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_reset_telapse(espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_set_telapse(espdr_sdp_spectrum *self,
                                               double value);
cpl_error_code espdr_sdp_spectrum_copy_telapse(espdr_sdp_spectrum *self,
                                                const cpl_propertylist *plist,
                                                const char *name);

double espdr_sdp_spectrum_get_tmid(const espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_reset_tmid(espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_set_tmid(espdr_sdp_spectrum *self,
                                            double value);
cpl_error_code espdr_sdp_spectrum_copy_tmid(espdr_sdp_spectrum *self,
                                             const cpl_propertylist *plist,
                                             const char *name);

double espdr_sdp_spectrum_get_specval(const espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_reset_specval(espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_set_specval(espdr_sdp_spectrum *self,
                                               double value);
cpl_error_code espdr_sdp_spectrum_copy_specval(espdr_sdp_spectrum *self,
                                                const cpl_propertylist *plist,
                                                const char *name);

double espdr_sdp_spectrum_get_specbw(const espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_reset_specbw(espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_set_specbw(espdr_sdp_spectrum *self,
                                              double value);
cpl_error_code espdr_sdp_spectrum_copy_specbw(espdr_sdp_spectrum *self,
                                               const cpl_propertylist *plist,
                                               const char *name);

const char * espdr_sdp_spectrum_get_extname(const espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_reset_extname(espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_set_extname(espdr_sdp_spectrum *self,
                                               const char *value);
cpl_error_code espdr_sdp_spectrum_copy_extname(espdr_sdp_spectrum *self,
                                                const cpl_propertylist *plist,
                                                const char *name);

cpl_boolean espdr_sdp_spectrum_get_inherit(const espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_reset_inherit(espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_set_inherit(espdr_sdp_spectrum *self,
                                               cpl_boolean value);
cpl_error_code espdr_sdp_spectrum_copy_inherit(espdr_sdp_spectrum *self,
                                                const cpl_propertylist *plist,
                                                const char *name);

cpl_size espdr_sdp_spectrum_get_nelem(const espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_reset_nelem(espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_set_nelem(espdr_sdp_spectrum *self,
                                             cpl_size value);
cpl_error_code espdr_sdp_spectrum_copy_nelem(espdr_sdp_spectrum *self,
                                              const cpl_propertylist *plist,
                                              const char *name);

double espdr_sdp_spectrum_get_tdmin(const espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_reset_tdmin(espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_set_tdmin(espdr_sdp_spectrum *self,
                                             double value);
cpl_error_code espdr_sdp_spectrum_copy_tdmin(espdr_sdp_spectrum *self,
                                              const cpl_propertylist *plist,
                                              const char *name);

double espdr_sdp_spectrum_get_tdmax(const espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_reset_tdmax(espdr_sdp_spectrum *self);
cpl_error_code espdr_sdp_spectrum_set_tdmax(espdr_sdp_spectrum *self,
                                             double value);
cpl_error_code espdr_sdp_spectrum_copy_tdmax(espdr_sdp_spectrum *self,
                                              const cpl_propertylist *plist,
                                              const char *name);

/*
 * Method to replace the default keyword comment with a custom comment
 */

cpl_error_code
espdr_sdp_spectrum_replace_column_comment(espdr_sdp_spectrum *self,
                                           const char *name,
                                           const char *keyword,
                                           const char *comment);

/*
 * Column manipulation functions.
 */

cpl_size espdr_sdp_spectrum_get_ncol(const espdr_sdp_spectrum *self);

cpl_boolean espdr_sdp_spectrum_has_column(const espdr_sdp_spectrum *self,
                                           const char* name);

cpl_array *
espdr_sdp_spectrum_get_column_names(const espdr_sdp_spectrum *self);

cpl_error_code
espdr_sdp_spectrum_new_column(espdr_sdp_spectrum *self, const char *name,
                               cpl_type type);

cpl_error_code
espdr_sdp_spectrum_add_column(espdr_sdp_spectrum *self, const char *name,
                               cpl_type type, const char *unit,
                               const char *format, const char *tutyp,
                               const char *tucd, const cpl_array *data);

cpl_error_code
espdr_sdp_spectrum_copy_column(espdr_sdp_spectrum *self,
                                const cpl_table* table, const char *name);
cpl_error_code
espdr_sdp_spectrum_divide_columns(espdr_sdp_spectrum *self,const char *dividend_name,
                                const char *divisor_name);

cpl_error_code
espdr_sdp_spectrum_duplicate_column(espdr_sdp_spectrum *self,const char *to,
                                const char *from);
cpl_error_code
espdr_sdp_spectrum_copy_column_regexp(espdr_sdp_spectrum *self,
                                       const cpl_table* table,
                                       const char *regexp, int invert);

cpl_error_code
espdr_sdp_spectrum_update_column(espdr_sdp_spectrum *self, const char *name,
                                  const cpl_table* table, const char *colname,
                                  int flags);

cpl_error_code
espdr_sdp_spectrum_delete_column(espdr_sdp_spectrum *self, const char *name);

cpl_type espdr_sdp_spectrum_get_column_type(const espdr_sdp_spectrum *self,
                                             const char *name);

const char *
espdr_sdp_spectrum_get_column_unit(const espdr_sdp_spectrum *self,
                                    const char *name);

cpl_error_code
espdr_sdp_spectrum_set_column_unit(espdr_sdp_spectrum *self,
                                    const char *name, const char *unit);

cpl_error_code
espdr_sdp_spectrum_copy_column_unit(espdr_sdp_spectrum *self,
                                     const char *name,
                                     const cpl_propertylist *plist,
                                     const char *key);

const char *
espdr_sdp_spectrum_get_column_format(const espdr_sdp_spectrum *self,
                                      const char *name);

cpl_error_code
espdr_sdp_spectrum_set_column_format(espdr_sdp_spectrum *self,
                                      const char *name, const char *format);

const char *
espdr_sdp_spectrum_get_column_tutyp(const espdr_sdp_spectrum *self,
                                     const char *name);

cpl_error_code
espdr_sdp_spectrum_set_column_tutyp(espdr_sdp_spectrum *self,
                                     const char *name, const char *tutyp);

cpl_error_code
espdr_sdp_spectrum_copy_column_tutyp(espdr_sdp_spectrum *self,
                                      const char *name,
                                      const cpl_propertylist *plist,
                                      const char *key);

const char *
espdr_sdp_spectrum_get_column_tucd(const espdr_sdp_spectrum *self,
                                    const char *name);

cpl_error_code
espdr_sdp_spectrum_set_column_tucd(espdr_sdp_spectrum *self,
                                    const char *name, const char *tucd);

cpl_error_code
espdr_sdp_spectrum_copy_column_tucd(espdr_sdp_spectrum *self,
                                     const char *name,
                                     const cpl_propertylist *plist,
                                     const char *key);

const char *
espdr_sdp_spectrum_get_column_tcomm(const espdr_sdp_spectrum *self,
                                     const char *name);

cpl_error_code
espdr_sdp_spectrum_set_column_tcomm(espdr_sdp_spectrum *self,
                                     const char *name, const char *tcomm);

cpl_error_code
espdr_sdp_spectrum_copy_column_tcomm(espdr_sdp_spectrum *self,
                                      const char *name,
                                      const cpl_propertylist *plist,
                                      const char *key);

const cpl_array *
espdr_sdp_spectrum_get_column_data(const espdr_sdp_spectrum *self,
                                    const char *name);

cpl_error_code
espdr_sdp_spectrum_set_column_data(espdr_sdp_spectrum *self,
                                    const char *name, const cpl_array *array);

/*
 * Methods for loading and saving a spectrum object.
 */

espdr_sdp_spectrum * espdr_sdp_spectrum_load(const char *filename);

cpl_error_code espdr_sdp_spectrum_save(const espdr_sdp_spectrum *self,
                                        const char *filename,
                                        const cpl_propertylist *extra_pheader,
                                        const cpl_propertylist *extra_header);

cpl_error_code espdr_dfs_save_spectrum(cpl_frameset              * allframes,
                                        cpl_propertylist          * header,
                                        const cpl_parameterlist   * parlist,
                                        const cpl_frameset        * usedframes,
                                        const cpl_frame           * inherit,
                                        const espdr_sdp_spectrum * spectrum,
                                        const char                * recipe,
                                        const cpl_propertylist    * applist,
                                        const cpl_propertylist    * tablelist,
                                        const char                * remregexp,
                                        const char                * pipe_id,
                                        const char                * dict_id,
                                        const char                * filename,
										const cpl_boolean trim_keys);

cpl_error_code
espdr_add_sdp_product_spectrum2(const cpl_frame *product_frame,
                                  cpl_frameset *frameset,
                                  const cpl_frameset *used_frames,
                                  const cpl_parameterlist *parameters,
                                  const char *recipe_id,
								  cpl_boolean is_fluxcal);
cpl_error_code
espdr_add_sdp_product_merged_spectrum2(const cpl_frame *product_frame,
                                       cpl_frameset *frameset,
                                       const cpl_frameset *used_frames,
                                       const cpl_parameterlist *parameters,
                                       const char *recipe_id,
                                       const cpl_table* s1d_fluxcal,
                                       const cpl_table* s1d_fluxcal_skysub,
                                       const cpl_table* s1d_skysub,
                                       const cpl_boolean has_ccf);

/*
 * Dump function for debugging.
 */

void espdr_sdp_spectrum_dump(const espdr_sdp_spectrum *self, FILE *stream);

/*
 * Additional useful helper functions.
 */

#  ifdef ESPDR_USE_FITS_UPDATE_CHECKSUM

/* NOTE: if one wants to use the following function the preprocessor macro
 * espdr_USE_FITS_UPDATE_CHECKSUM must be defined when including this file and
 * when compiling the libirplib.a library. In addition, there will be a
 * dependency introduced to cfitsio in libirplib.a and any binary linking
 * against it. This will require adding the appropriate linker flags when
 * building such libraries.
 * The easiest way to add the espdr_USE_FITS_UPDATE_CHECKSUM macro is to add
 * the following lines to either the acinclude.m4 or configure.ac files of the
 * parent pipeline project:
 *   espdr_CPPFLAGS='-Despdr_USE_FITS_UPDATE_CHECKSUM'
 *   AC_SUBST(espdr_CPPFLAGS)
 */
cpl_error_code espdr_fits_update_checksums(const char* filename);

#  endif /* espdr_USE_FITS_UPDATE_CHECKSUM */

CPL_END_DECLS

#endif /* ESPDR_SDP_SPECTRUM_H */
