/*                                                                            *
 *   This file is part of the ESPRESSO Pipeline                               *
 *   Copyright (C) 2006 European Southern Observatory                         *
 *                                                                            *
 *   This library is free software; you can redistribute it and/or modify     *
 *   it under the terms of the GNU General Public License as published by     *
 *   the Free Software Foundation; either version 2 of the License, or        *
 *   (at your option) any later version.                                      *
 *                                                                            *
 *   This program is distributed in the hope that it will be useful,          *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 *   GNU General Public License for more details.                             *
 *                                                                            *
 *   You should have received a copy of the GNU General Public License        *
 *   along with this program; if not, write to the Free Software              *
 *   Foundation, 51 Franklin St, Fifth Floor, Boston, MA  02111-1307  USA     *
 *                                                                            */

/*
 */


#include <espdr_shifted_extraction_cal.h>


/*----------------------------------------------------------------------------
 Functions code
 ----------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/**
 @brief    Interpret the command line options and execute the data processing
 @param    parameters     the parameters list
 @param    frameset   the frames list

 In case of failure the cpl_error_code is set.
 */
/*---------------------------------------------------------------------------*/

int espdr_shifted_extraction(cpl_parameterlist *parameters, cpl_frameset *frameset,
                             const char* recipe_id) {

    cpl_frameset *ghost_frames_to_reduce = NULL;
    cpl_frame *ghost_frame = NULL;
    cpl_frameset *used_frames = NULL;

	cpl_error_code my_error = CPL_ERROR_NONE;

	cpl_msg_set_level(CPL_MSG_INFO);

	espdr_msg("Starting shifted extraction");

    /* Identify the RAW and CALIB frames in the input frameset */
    espdr_ensure(espdr_dfs_set_groups(frameset) != CPL_ERROR_NONE,
                 cpl_error_get_code(),
                 "DFS setting groups failed. Expected inputs are:\n"
                 "raw OBJECT image, tagged as GHOSTS_FP, GHOSTS_THAR, GHOSTS_SKY or GHOSTS_DARK\n"
                 "CCD geometry table, tagged as CCD_GEOM\n"
                 "instrument configuration table, tagged as MASTER_INST_CONFIG or INST_CONFIG\n"
                 "hot pixel mask image, tagged as HOT_PIXEL_MASK\n"
                 "bad pixel mask image, tagged as BAD_PIXEL_MASK\n"
                 "orders definition images for fibres A and B, tagged as ORDER_TABLE_A & ORDER_TABLE_B\n"
                 "orders profile images for fibres A and B, tagged as ORDER_PROFILE_A & ORDER_PROFILE_B\n"
                 "flat images for fibres A and B, tagged as FLAT_A & FLAT_B\n"
                 "blaze images for fibres A and B, tagged as BLAZE_A & BLAZE_B\n");

    espdr_CCD_geometry *CCD_geom            = NULL;
    espdr_inst_config *inst_config          = NULL;
    espdr_qc_keywords *qc_kws               = NULL;
    espdr_OVSC_param *OVSC_param            = NULL;
    espdr_shifted_extraction_param *shifted_extraction_param  = NULL;
    cpl_frame* CCD_geom_frame               = NULL;
    cpl_frame* inst_config_frame            = NULL;

    CCD_geom_frame    = espdr_frame_find(frameset, ESPDR_CCD_GEOM);
    espdr_ensure(CCD_geom_frame == NULL,cpl_error_get_code(),
                 "CCD geometry frame not found!");
    CCD_geom          = espdr_CCD_geom_init(parameters, CCD_geom_frame);
    inst_config_frame = espdr_get_inst_config(frameset);
    espdr_ensure(inst_config_frame == NULL,cpl_error_get_code(),
                 "Instrument config frame not found!");
    inst_config       = espdr_inst_config_init(parameters,
                                               CCD_geom->ext_nb,
                                               inst_config_frame);
    OVSC_param        = espdr_parameters_OVSC_init(recipe_id, parameters);
    shifted_extraction_param   = espdr_shifted_extraction_param_init(recipe_id, parameters);

    used_frames = cpl_frameset_new();
    my_error = cpl_frameset_insert(used_frames, cpl_frame_duplicate(CCD_geom_frame));
    my_error = cpl_frameset_insert(used_frames, cpl_frame_duplicate(inst_config_frame));

    /* Filling up the DRS QC KWs structure */
    qc_kws = (espdr_qc_keywords *)cpl_malloc(sizeof(espdr_qc_keywords));
    my_error = espdr_fill_qc_keywords(inst_config, qc_kws);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Filling QC KWs failed: %s",
                 cpl_error_get_message_default(my_error));
    espdr_ensure(qc_kws == NULL, CPL_ERROR_ILLEGAL_OUTPUT,
                 "QC KWs structure is NULL");
    
    /* Getting the raw frame */
    ghost_frame = espdr_frame_find(frameset, ESPDR_SHIFTED_EXTRACTION_RAW_DARK);
    if (ghost_frame == NULL) {
        ghost_frame = espdr_frame_find(frameset, ESPDR_SHIFTED_EXTRACTION_RAW_SKY);
        if (ghost_frame == NULL) {
            ghost_frame = espdr_frame_find(frameset, ESPDR_SHIFTED_EXTRACTION_RAW_FP);
            if (ghost_frame == NULL) {
                ghost_frame = espdr_frame_find(frameset, ESPDR_SHIFTED_EXTRACTION_RAW_THAR);
                if (ghost_frame == NULL) {
                    ghost_frame = espdr_frame_find(frameset, ESPDR_SHIFTED_EXTRACTION_RAW_LFC);
                    if (ghost_frame == NULL) {
                        espdr_msg_error("No raw GHOSTS frame, exiting");
                        return(CPL_ERROR_NULL_INPUT);
                    }
                }
            }
        }
    }
    my_error = cpl_frameset_insert(used_frames, cpl_frame_duplicate(ghost_frame));
    const char *GHOST_tag = cpl_frame_get_tag(ghost_frame);
    espdr_msg("GHOST_tag: %s", GHOST_tag);
    /* Load science keywords from the first image (primary header) */
    const char *input_filename = cpl_frame_get_filename(ghost_frame);
    espdr_msg("KEYWORDS for shifted_extraction input filename: %s", input_filename);
    cpl_propertylist *keywords = cpl_propertylist_load(input_filename, 0);
    espdr_ensure(keywords == NULL, CPL_ERROR_ILLEGAL_OUTPUT, "keywords are NULL");
    
    char **fibre_GHOST = (char **)cpl_malloc(inst_config->fibres_nb * sizeof(char *));
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        fibre_GHOST[i] = (char *)cpl_malloc(TAG_LENGTH*sizeof(char));
    }
    int pos;
    for (pos = 0; pos < strlen(GHOST_tag); pos++) {
        if (GHOST_tag[pos] == '_') {
            break;
        }
    }
    strncpy(fibre_GHOST[0], GHOST_tag, pos);
    fibre_GHOST[0][pos]='\0';
    strncpy(fibre_GHOST[1], GHOST_tag+pos+1, strlen(GHOST_tag)-1);
    espdr_msg("fibre_GHOST: %s & %s", fibre_GHOST[0], fibre_GHOST[1]);
    
    char *bias_res_removal_sw = (char *)cpl_malloc(16 * sizeof(char));
    strcpy(bias_res_removal_sw, "on");
    int remove_bias_res_flag = 1;
    char *background_sw = (char *)cpl_malloc(16 * sizeof(char));
    strcpy(background_sw, "on");
    int bkgr_grid_size_x = inst_config->sci_bkgr_grid_size_x;
    int bkgr_grid_size_y = inst_config->sci_bkgr_grid_size_y;
    int extraction_offset = 0;
    
    my_error = espdr_get_shifted_extraction_params(inst_config, parameters,
                                                   shifted_extraction_param,
                                                   fibre_GHOST[1],
                                                   bias_res_removal_sw,
                                                   &remove_bias_res_flag,
                                                   background_sw,
                                                   &bkgr_grid_size_x,
                                                   &bkgr_grid_size_y,
                                                   &extraction_offset);
    
    cpl_image *flat_corr_spectrum, *flat_corr_error, *flat_corr_qual;
    cpl_imagelist *CCD_corrected_image = cpl_imagelist_new();
    double *RON = (double *)cpl_calloc(CCD_geom->total_output_nb, sizeof(double));
    
    int *orders_nb_per_fibre = NULL;
    orders_nb_per_fibre = (int *)cpl_calloc(inst_config->fibres_nb, sizeof(int));
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        orders_nb_per_fibre[i] = 0;
        for (int j = 0; j < CCD_geom->ext_nb; j++) {
            orders_nb_per_fibre[i] += inst_config->orders_nb[i*CCD_geom->ext_nb+j];
        }
    }
    
    int *cosmics_nb = (int *) cpl_calloc(orders_nb_per_fibre[0],sizeof(int));
    double *snr_per_fibre = (double *) cpl_calloc(orders_nb_per_fibre[0],sizeof(double));

    my_error = espdr_process_ghost_till_extraction(frameset, parameters, recipe_id,
                                                   ghost_frame, used_frames,
                                                   inst_config, CCD_geom, qc_kws,
                                                   keywords,
                                                   remove_bias_res_flag,
                                                   fibre_GHOST[1],
                                                   background_sw,
                                                   bkgr_grid_size_x,
                                                   bkgr_grid_size_y,
                                                   extraction_offset,
                                                   RON,
                                                   cosmics_nb,
                                                   snr_per_fibre,
                                                   &flat_corr_spectrum,
                                                   &flat_corr_error,
                                                   &flat_corr_qual);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_process_wave_till_extraction failed: %s",
                 cpl_error_get_message_default(my_error));
    
    espdr_msg("Creating the QC keywords");
    my_error = espdr_shifted_extraction_QC(CCD_geom, inst_config, qc_kws,
                                           fibre_GHOST[1], cosmics_nb, snr_per_fibre,
                                           orders_nb_per_fibre[0], RON,
                                           extraction_offset,
                                           &keywords);
    
    espdr_msg("Saving products");
    my_error = espdr_save_ghost_S2D_products(frameset, parameters, recipe_id,
                                             used_frames, GHOST_tag, extraction_offset,
                                             keywords, inst_config,
                                             flat_corr_spectrum,
                                             flat_corr_error,
                                             flat_corr_qual);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_save_wave_S2D_products failed: %s for fibre A",
                 cpl_error_get_message_default(my_error));

    espdr_msg("Cleaning the memory");

	// Cleaning memory.
    espdr_msg("Cleaning keywords lists");
    cpl_propertylist_delete(keywords);

	cpl_free(qc_kws);

    espdr_msg("Cleaning calibration products");
	cpl_imagelist_delete(CCD_corrected_image);
    cpl_frameset_delete(used_frames);

    espdr_msg("Cleaning products");
    cpl_image_delete(flat_corr_spectrum);
    cpl_image_delete(flat_corr_error);
    cpl_image_delete(flat_corr_qual);

    espdr_msg("Cleaning global variables");
    cpl_free(RON);

    espdr_msg("Cleaning parameters");
    espdr_parameters_OVSC_delete(OVSC_param);
    espdr_parameters_shifted_extraction_delete(shifted_extraction_param);
    espdr_parameters_CCD_geometry_delete(CCD_geom);
    espdr_parameters_inst_config_delete(inst_config);

    espdr_msg("End of the recipe espdr_shifted_extraction");
	return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Create the structure for WAVE input parameters
 @param    recipe_id	recipe identifier
 @param    list			list of input parameters
 @param    p			input WAVE parameters structure
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_parameters_shifted_extraction_create(const char* recipe_id,
                                                          cpl_parameterlist *list,
                                                          espdr_shifted_extraction_param *p) {
    cpl_error_code error_got;
    char comment[COMMENT_LENGTH];
    int bkgr_x_min = BKGR_X_MIN;
    int bkgr_x_max = BKGR_X_MAX;
    int bkgr_y_min = BKGR_Y_MIN;
    int bkgr_y_max = BKGR_Y_MAX;
    
    espdr_ensure(recipe_id == NULL,
                 CPL_ERROR_NULL_INPUT, "The recipe ID is NULL");
    
	/* check parameters */
	espdr_ensure(list == NULL, CPL_ERROR_NULL_INPUT, "The parameters list NULL");
	
	/* Fill the parameter list */
    error_got = espdr_parameters_new_string(recipe_id, list,
                                            "background_sw",
                                            p->background_sw,
                                            "Background subtraction activation (on/on_using_B/off)");
    espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
                 "Error adding parameter background_sw to the list");
    
    sprintf(comment,
            "Grid size in x used to calculate the background, between: %d and %d",
            bkgr_x_min,bkgr_x_max);
    error_got = espdr_parameters_new_range_int(recipe_id, list,
                                               "shifted_extraction_bkgr_grid_size_x",
                                               p->bkgr_grid_size_x,
                                               bkgr_x_min, bkgr_x_max,
                                               comment);
    espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
                 "Error adding parameter bkgr_grid_size_x to the list");
    
    sprintf(comment,
            "Grid size in y used to calculate the background, between: %d and %d",
            bkgr_y_min,bkgr_y_max);
    error_got = espdr_parameters_new_range_int(recipe_id, list,
                                               "shifted_extraction_bkgr_grid_size_y",
                                               p->bkgr_grid_size_y,
                                               bkgr_y_min, bkgr_y_max,
                                               comment);
    espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
                 "Error adding parameter bkgr_grid_size_y to the list");
    
    error_got = espdr_parameters_new_string(recipe_id, list,
                                            "bias_res_removal_sw",
                                            p->bias_res_removal_sw,
                                            "Flag indicating to remove or not MB residuals");
    espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
                 "Error adding parameter bias_res_removal_sw to the list");
    
    error_got = espdr_parameters_new_int(recipe_id, list,
                                        "extraction_offset",
                                         p->extraction_offset,
                                         "Extraction offset to extract outside the orders");
    espdr_ensure(error_got != CPL_ERROR_NONE, error_got,
                 "Error adding parameter extraction_offset to the list");
    

    return(CPL_ERROR_NONE);

}

/*----------------------------------------------------------------------------*/
/**
 @brief    Delete the structure for ORDERDEF input parameters
 @param    p	input ORDERDEF parameters structure
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_parameters_shifted_extraction_delete(espdr_shifted_extraction_param* p) {
	
    cpl_free((void *)p->background_sw);
    cpl_free((void *)p->bias_res_removal_sw);
    cpl_free(p);
    p = NULL;
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief     Get the shifted_extraction recipe parameters
 @param     recipe_id   recipe ID
 @param     param_list  parameters list
 @param     shifted_extraction_param  shifted_extraction parameters structure
 @return    CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_parameters_shifted_extraction_get(const char* recipe_id,
                                              cpl_parameterlist* param_list,
                                              espdr_shifted_extraction_param *shifted_extraction_param) {
	
    /* check parameters */
    espdr_ensure(recipe_id == NULL,
                 CPL_ERROR_NULL_INPUT, "The recipe ID is NULL");
    
    /* Fill the structure */
	espdr_ensure(param_list == NULL, CPL_ERROR_NULL_INPUT,
				 "Parameters list is NULL");
	
    shifted_extraction_param->background_sw =
                    espdr_parameters_get_string(recipe_id,
                                                param_list,
                                                "background_sw");

    shifted_extraction_param->bkgr_grid_size_x =
                    espdr_parameters_get_int(recipe_id,
                                             param_list,
                                             "shifted_extraction_bkgr_grid_size_x");
    
    shifted_extraction_param->bkgr_grid_size_y =
                    espdr_parameters_get_int(recipe_id,
                                             param_list,
                                             "shifted_extraction_bkgr_grid_size_y");
    
    shifted_extraction_param->bias_res_removal_sw =
                    espdr_parameters_get_string(recipe_id,
                                                param_list,
                                                "bias_res_removal_sw");
    
    shifted_extraction_param->extraction_offset =
                    espdr_parameters_get_int(recipe_id,
                                             param_list,
                                             "extraction_offset");
    
	return cpl_error_get_code();
}


/*---------------------------------------------------------------------------*/
/**
 @brief     print the shifted_extraction parameters
 @param     shifted_extraction_param shifted_extraction parameters structure
 @return    CPL_ERROR_NONE iff OK
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_parameters_shifted_extraction_print(espdr_shifted_extraction_param *shifted_extraction_param) {
	
	espdr_msg("\tshifted_extraction parameters:");
    espdr_msg("\t\tshifted_extraction background subtraction = %s",
              shifted_extraction_param->background_sw);
    espdr_msg("\t\tshifted_extraction background box size: = %d x %d",
              shifted_extraction_param->bkgr_grid_size_x,
              shifted_extraction_param->bkgr_grid_size_y);
    espdr_msg("\t\tshifted_extraction bias_res_removal_sw = %s",
              shifted_extraction_param->bias_res_removal_sw);
    espdr_msg("\t\tshifted_extraction extraction offset = %d",
              shifted_extraction_param->extraction_offset);

	return (cpl_error_get_code());
}


/*----------------------------------------------------------------------------*/
/**
 @brief    Init shifted_extraction_param structure
 
 @param    recipe_id    recipe name
 @param    parameters   recipe input parameters
 
 @return   allocated structure iff OK
 */
/*----------------------------------------------------------------------------*/

espdr_shifted_extraction_param *espdr_shifted_extraction_param_init(const char *recipe_id,
                                                                    cpl_parameterlist *parameters) {
    
    cpl_error_code my_error;
    
    espdr_shifted_extraction_param *shifted_extraction_param =
                (espdr_shifted_extraction_param *)cpl_malloc(sizeof(espdr_shifted_extraction_param));
    
    /* Read the wave parameters */
    my_error = espdr_parameters_shifted_extraction_get(recipe_id, parameters, shifted_extraction_param);
    my_error = espdr_parameters_shifted_extraction_print(shifted_extraction_param);
    //if(cpl_error_get_code() != CPL_ERROR_NONE) {
    if(my_error != CPL_ERROR_NONE) {
        return NULL;
    } else {
        return shifted_extraction_param;
    }
}


/*----------------------------------------------------------------------------*/
/**
 @brief    get recipe user parameters
 
 @param
 @param
 
 @return   cpl_error_code
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_get_shifted_extraction_params(espdr_inst_config *inst_config,
                                                   cpl_parameterlist *parameters,
                                                   espdr_shifted_extraction_param *shifted_extraction_param,
                                                   char *fibre_b,
                                                   char *bias_res_removal_sw,
                                                   int *remove_bias_res_flag,
                                                   char *background_sw,
                                                   int *bkgr_grid_size_x,
                                                   int *bkgr_grid_size_y,
                                                   int *extraction_offset) {
    
    espdr_msg("Getting params");
    
    //espdr_msg("bias_removal_sw");
    cpl_parameter *par_bias_res_removal_sw =
                    cpl_parameterlist_find(parameters, "espdr.espdr_shifted_extraction.bias_res_removal_sw");
    int bias_res_removal_sw_set = cpl_parameter_get_default_flag(par_bias_res_removal_sw);
    if (bias_res_removal_sw_set == 1) {
        sprintf(bias_res_removal_sw, "%s", shifted_extraction_param->bias_res_removal_sw);
        if (strcmp(bias_res_removal_sw, "off") == 0) {
            *remove_bias_res_flag = 0;
        }
    }
    cpl_parameter_set_string(par_bias_res_removal_sw, bias_res_removal_sw);
    if (inst_config->raw_bias_limit_nb == 0) {
        *remove_bias_res_flag = 0;
    }
    if (*remove_bias_res_flag == 1) {
        cpl_parameter_set_string(par_bias_res_removal_sw, "on");
    } else {
        cpl_parameter_set_string(par_bias_res_removal_sw, "off");
    }
    
    //espdr_msg("background_sw");
    cpl_parameter *par_background_sw =
                    cpl_parameterlist_find(parameters,"espdr.espdr_shifted_extraction.background_sw");
    int background_sw_set = cpl_parameter_get_default_flag(par_background_sw);
    int background_sw_has_changed = espdr_param_has_changed(par_background_sw);
    if ((background_sw_set == 1) && (background_sw_has_changed == 1)) {
        sprintf(background_sw, "%s", shifted_extraction_param->background_sw);
    }
    /* By default ON for any case, just OFF in case of SKY, unless overwritten
     * by command line by --background_sw=on */
    if ((strcmp(fibre_b, "SKY") == 0) && (background_sw_set == 0)) {
        sprintf(background_sw, "%s", "off");
    }
    cpl_parameter_set_string(par_background_sw, background_sw);
    
    //espdr_msg("bkgr grid sizes");
    cpl_parameter *par_bkgr_grid_size_x =
                    cpl_parameterlist_find(parameters,
                                           "espdr.espdr_shifted_extraction.shifted_extraction_bkgr_grid_size_x");
    cpl_parameter_set_int(par_bkgr_grid_size_x, inst_config->sci_bkgr_grid_size_x);
    int bkgr_grid_size_x_set = cpl_parameter_get_default_flag(par_bkgr_grid_size_x);
    int bkgr_grid_size_x_has_changed = espdr_param_has_changed(par_bkgr_grid_size_x);
    if ((bkgr_grid_size_x_set == 1) && (bkgr_grid_size_x_has_changed == 1)) {
        *bkgr_grid_size_x = shifted_extraction_param->bkgr_grid_size_x;
    }
    cpl_parameter_set_int(par_bkgr_grid_size_x, shifted_extraction_param->bkgr_grid_size_x);
    
    cpl_parameter *par_bkgr_grid_size_y =
                    cpl_parameterlist_find(parameters,
                                           "espdr.espdr_shifted_extraction.shifted_extraction_bkgr_grid_size_y");
    cpl_parameter_set_int(par_bkgr_grid_size_y, inst_config->sci_bkgr_grid_size_y);
    int bkgr_grid_size_y_set = cpl_parameter_get_default_flag(par_bkgr_grid_size_y);
    int bkgr_grid_size_y_has_changed = espdr_param_has_changed(par_bkgr_grid_size_y);
    if ((bkgr_grid_size_y_set == 1) && (bkgr_grid_size_y_has_changed == 1)) {
        *bkgr_grid_size_y = shifted_extraction_param->bkgr_grid_size_y;
    }
    cpl_parameter_set_int(par_bkgr_grid_size_y, shifted_extraction_param->bkgr_grid_size_y);

    //espdr_msg("extraction_offset");
    cpl_parameter *par_extraction_offset =
                    cpl_parameterlist_find(parameters,"espdr.espdr_shifted_extraction.extraction_offset");
    cpl_parameter_set_int(par_extraction_offset, *extraction_offset);
    int extraction_offset_set = cpl_parameter_get_default_flag(par_extraction_offset);
    espdr_msg("offset set: %d", extraction_offset_set);
    if (extraction_offset_set == 1) {
        espdr_msg("Offset set: %d", shifted_extraction_param->extraction_offset);
        *extraction_offset = shifted_extraction_param->extraction_offset;
    }
    cpl_parameter_set_int(par_extraction_offset, shifted_extraction_param->extraction_offset);
    
    
    espdr_msg(ANSI_COLOR_GREEN"background_sw = %s"ANSI_COLOR_RESET,
              background_sw);
    espdr_msg(ANSI_COLOR_GREEN"background box size = %d x %d"ANSI_COLOR_RESET,
              *bkgr_grid_size_x, *bkgr_grid_size_y);
    espdr_msg(ANSI_COLOR_GREEN"bias_res_removal_sw = %s"ANSI_COLOR_RESET,
              bias_res_removal_sw);
    espdr_msg(ANSI_COLOR_GREEN"extraction_offset = %d"ANSI_COLOR_RESET,
              *extraction_offset);
    
    return cpl_error_get_code();
}



/*----------------------------------------------------------------------------*/
/**
 @brief
 @param
 @param
 @param
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_process_ghost_till_extraction(cpl_frameset *frameset,
                                                   cpl_parameterlist *parameters,
                                                   const char *RECIPE_ID,
                                                   cpl_frame *raw_frame,
                                                   cpl_frameset *used_frames,
                                                   espdr_inst_config *inst_config,
                                                   espdr_CCD_geometry *CCD_geom,
                                                   espdr_qc_keywords *qc_kws,
                                                   cpl_propertylist *keywords,
                                                   int remove_bias_res_flag,
                                                   char *fibre_b,
                                                   char *background_sw,
                                                   int bkgr_grid_size_x,
                                                   int bkgr_grid_size_y,
                                                   int extraction_offset,
                                                   double *ron,
                                                   int *cosmics_nb,
                                                   double *snr_per_fibre,
                                                   cpl_image **flat_corr_flux_RE,
                                                   cpl_image **flat_corr_err_RE,
                                                   cpl_image **flat_corr_qual_RE) {

    cpl_error_code my_error = CPL_ERROR_NONE;
    
    espdr_msg("Start processing %s", RECIPE_ID);
    
    cpl_frame* master_bias_frame;
    if ((inst_config->raw_bias_limit_nb > 0) && (remove_bias_res_flag == 1)) {
        master_bias_frame = espdr_get_mbias_from_set(frameset, used_frames);
    }

    
    /* Extract hot pixels mask frame */
    cpl_frame* hot_pixels_frame = espdr_get_hpixmap_from_set(frameset,
                                                             used_frames);
    
    /* Extract master dark frame */
    cpl_frame* master_dark_frame = espdr_get_mdark_from_set(frameset,
                                                            used_frames);
    
    /* Extract bad pixels mask frame */
    cpl_frame* bad_pixels_frame = espdr_get_bpixmap_from_set(frameset,
                                                             used_frames);
    
    /* Extract orders mask frame */
    cpl_frame* orders_mask_frame = espdr_get_orders_mask_from_set(frameset,
                                                                  used_frames);
    
    /* Extract orders maps frame */
    cpl_table ***orders_coeffs = NULL;
    espdr_get_orders_coeff_from_set(frameset, used_frames,
                                    inst_config->fibres_nb,
                                    CCD_geom->ext_nb,
                                    &orders_coeffs);
    
    /* Extract order profile frame per fibre */
    cpl_frame **order_profile_fibre_frame = NULL;
    espdr_get_order_profile_from_set(frameset, used_frames,
                                     inst_config->fibres_nb,
                                     CCD_geom->ext_nb,
                                     &order_profile_fibre_frame);
    
    /* Extract flat frame per fibre */
    cpl_frame **flat_fibre_frame = NULL;
    espdr_get_flat_fibre_from_set(frameset, used_frames,
                                  inst_config->fibres_nb,
                                  CCD_geom->ext_nb,
                                  &flat_fibre_frame);
    
    /* Extract BLAZE frame per fibre */
    cpl_frame **blaze_fibre_frame = NULL;
    espdr_get_blaze_fibre_from_set(frameset, used_frames,
                                   inst_config->fibres_nb,
                                   CCD_geom->ext_nb,
                                   &blaze_fibre_frame);
    
    cpl_imagelist *master_bias_list = NULL;
    if ((inst_config->raw_bias_limit_nb > 0) && (remove_bias_res_flag == 1)) {
        /* Extract all the outputs from the master bias frame */
        master_bias_list = cpl_imagelist_new();
        my_error = espdr_extract_real_outputs(master_bias_frame, CPL_TYPE_DOUBLE,
                                              CCD_geom, 0, &master_bias_list);
    }
    
    /* Extract all the outputs from the hot pixels frame */
    cpl_imagelist *hot_pixels_list = cpl_imagelist_new();
    my_error = espdr_extract_real_outputs(hot_pixels_frame, CPL_TYPE_INT,
                                          CCD_geom, 0, &hot_pixels_list);
    
    /* Extract all the outputs from the master dark frame if provided */
    cpl_imagelist *master_dark_list = NULL;
    if (master_dark_frame != NULL) {
        master_dark_list = cpl_imagelist_new();
        my_error = espdr_extract_real_outputs(master_dark_frame, CPL_TYPE_DOUBLE,
                                              CCD_geom, 0, &master_dark_list);
    }
    
    /* Extract all the outputs from the bad pixels frame */
    cpl_imagelist *bad_pixels_list = cpl_imagelist_new();
    my_error = espdr_extract_real_outputs(bad_pixels_frame, CPL_TYPE_INT,
                                          CCD_geom, 0, &bad_pixels_list);
    
    cpl_imagelist *orders_mask_list = NULL;
    /* Extract all the outputs from the orders mask frame if provided */
    if (orders_mask_frame != NULL) {
        orders_mask_list = cpl_imagelist_new();
        my_error = espdr_extract_extensions(orders_mask_frame,
                                            CPL_TYPE_INT,
                                            CCD_geom->ext_nb,
                                            &orders_mask_list);
    }
    
    /* Extract extensions from order profile frames */
    cpl_imagelist **order_profile_fibre_imagelist = (cpl_imagelist **)cpl_malloc
                                    (inst_config->fibres_nb * sizeof(cpl_imagelist *));
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        order_profile_fibre_imagelist[i] = cpl_imagelist_new();
        my_error = espdr_extract_extensions(order_profile_fibre_frame[i],
                                            CPL_TYPE_DOUBLE,
                                            CCD_geom->ext_nb,
                                            &order_profile_fibre_imagelist[i]);
    }
    
    /* Extract extensions from flat frames */
    cpl_image **flat_table = (cpl_image **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_image *));
    cpl_image **flat_error_table = (cpl_image **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_image *));
    cpl_image **flat_quality_table = (cpl_image **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_image *));
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        const char *flat_filename = cpl_frame_get_filename(flat_fibre_frame[i]);
        flat_table[i] = cpl_image_load(flat_filename, CPL_TYPE_DOUBLE, 0, 1);
        flat_error_table[i] = cpl_image_load(flat_filename, CPL_TYPE_DOUBLE, 0, 2);
        flat_quality_table[i] = cpl_image_load(flat_filename, CPL_TYPE_INT, 0, 3);
    }
    
    /* Extract raw and real images from the raw frame */
    cpl_imagelist *raw_imagelist = cpl_imagelist_new();
    cpl_imagelist *real_imagelist = cpl_imagelist_new();
    if (inst_config->inst_type == NIR) {
        my_error = espdr_extract_extensions(raw_frame, CPL_TYPE_DOUBLE,
                                            CCD_geom->ext_nb, &raw_imagelist);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_extract_extensions failed: %s",
                     cpl_error_get_message_default(my_error));
    } else {
        my_error = espdr_extract_raw_outputs_one_frame(raw_frame, CPL_TYPE_DOUBLE,
                                                       CCD_geom, &raw_imagelist);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_extract_raw_outputs_one_frame failed: %s",
                     cpl_error_get_message_default(my_error));
    }
    my_error = espdr_extract_real_outputs_one_frame(raw_frame, CPL_TYPE_DOUBLE,
                                                    CCD_geom, &real_imagelist);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_extract_real_outputs_one_frame failed: %s",
                 cpl_error_get_message_default(my_error));
    
    /* hot & bad pixel mask creation */
    espdr_msg("Creating hot & bad pixels mask");
    
    // Update bad pixel mask with all the nans from raw frames
    if (inst_config->inst_type == NIR) {
        cpl_frameset *raw_frames_to_reduce = cpl_frameset_new();
        cpl_frameset_insert(raw_frames_to_reduce, cpl_frame_duplicate(raw_frame));
        my_error = espdr_update_bad_pixel_mask(raw_frames_to_reduce, CCD_geom,
                                               cpl_imagelist_get(bad_pixels_list, 0));
        cpl_frameset_delete(raw_frames_to_reduce);
    }
    
    cpl_imagelist *pixels_mask = cpl_imagelist_new();
    my_error = espdr_create_hot_bad_pixels_mask(hot_pixels_list,
                                                bad_pixels_list,
                                                &pixels_mask);
    
    cpl_imagelist *merged_pixels_mask = cpl_imagelist_new();
    my_error = espdr_image_merge_2_dim(pixels_mask, CCD_geom, CPL_TYPE_INT,
                                       &merged_pixels_mask);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_image_merge_2_dim failed: %s",
                 cpl_error_get_message_default(my_error));
    
    cpl_imagelist *geometry_corrected_pixels_mask = cpl_imagelist_new();
    my_error = espdr_correct_geometry(merged_pixels_mask, CCD_geom,
                                      geometry_corrected_pixels_mask);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_correct_geometry failed: %s",
                 cpl_error_get_message_default(my_error));
    
    cpl_imagelist_delete(merged_pixels_mask);
    
    /* Checking for saturation, creating saturation mask */
    
    /* Checking the saturation outside hot and bad pixels and cosmics */
    double cosmics_part = inst_config->image_cosmics_part;
    cpl_imagelist *geometry_corrected_saturation_mask = NULL;
    double *max_flux = (double *)cpl_calloc(cpl_imagelist_get_size(real_imagelist),
                                            sizeof(double));
    int print_max_flux = 1;
    if (strcmp(fibre_b, "THAR") == 0) {
        cosmics_part = inst_config->image_thar_part;
        print_max_flux = 0;
    }
    
    // For NIR we have to multiply the input images by Texp, to get ADUs instead of ADUs/s
    if (inst_config->inst_type == NIR) {
        double texp = cpl_propertylist_get_double(keywords, inst_config->Texp_kw);
        //espdr_msg("--->>>>>  exposure time: %f", texp);
        my_error = cpl_imagelist_multiply_scalar(real_imagelist, texp);
    }
    my_error = espdr_check_saturation(real_imagelist,
                                      CCD_geom, inst_config,
                                      pixels_mask, cosmics_part,
                                      max_flux, print_max_flux,
                                      &geometry_corrected_saturation_mask);
    cpl_imagelist_delete(real_imagelist);
    
    /* Add QC KWs to the header */
    if (strcmp(fibre_b, "THAR") != 0) {
        my_error = espdr_max_flux_QC(inst_config, CCD_geom, qc_kws, max_flux,
                                     qc_kws->qc_shifted_extraction_check_kw, keywords);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_max_flux_QC failed: %s",
                     cpl_error_get_message_default(my_error));
    }
    
    double *dark = (double *)cpl_calloc (CCD_geom->total_output_nb, sizeof(double));
    double *dark_ron = (double *)cpl_calloc (CCD_geom->total_output_nb, sizeof(double));
    double *conad = (double *)cpl_calloc (CCD_geom->total_output_nb, sizeof(double));
    cpl_imagelist *CCD_corrected_raw = cpl_imagelist_new();
    
    espdr_msg("Loading headers");
    /* Load primary header keywords from the hot pixels image */
    const char *input_filename_HP = NULL;
    if (master_dark_frame != NULL) {
        input_filename_HP = cpl_frame_get_filename(master_dark_frame);
        espdr_msg("KEYWORDS for MASTER DARK input filename: %s", input_filename_HP);
    } else {
        input_filename_HP = cpl_frame_get_filename(hot_pixels_frame);
        espdr_msg("KEYWORDS for HOT PIXELS input filename: %s", input_filename_HP);
    }
    cpl_propertylist *keywords_HP = cpl_propertylist_load(input_filename_HP, 0);
    
    /* Load primary header keywords from the bad pixels image */
    const char *input_filename_BP = cpl_frame_get_filename(bad_pixels_frame);
    espdr_msg("KEYWORDS for BAD PIXELS input filename: %s", input_filename_BP);
    cpl_propertylist *keywords_BP = cpl_propertylist_load(input_filename_BP, 0);
    
    
    // Detector signature removal for the science frame process ...
    espdr_msg("Removing the CCD signature ...");
    
    my_error = espdr_remove_det_signature(raw_imagelist,
                                          orders_mask_list,
                                          master_bias_list,
                                          master_dark_list,
                                          keywords, keywords_HP, keywords_BP,
                                          qc_kws, inst_config, remove_bias_res_flag,
                                          CCD_geom, CPL_TYPE_DOUBLE,
                                          ron, dark, dark_ron, conad,
                                          CCD_corrected_raw);
    
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_remove_detector_signature failed: %s",
                 cpl_error_get_message_default(my_error));

    cpl_imagelist_delete(raw_imagelist);
    
    cpl_propertylist **keywords_ext = (cpl_propertylist**)cpl_malloc(
                                            CCD_geom->ext_nb * sizeof(cpl_propertylist*));
    
    const char *input_filename_ext = cpl_frame_get_filename(raw_frame);
    for (int j = 0; j < CCD_geom->ext_nb; j++) {
        keywords_ext[j] = cpl_propertylist_load(input_filename_ext,j+1);
    }
        
#if SAVE_DEBUG_CCD_CLEANED_SHIFTED_EXTRACTIONs
    espdr_msg("Saving cleaned shifted extraction frame");
    /* Save the PRO.CATG */
    char pro_catg[16];
    strcpy(pro_catg, ESPDR_SHIFTED_EXTRACTION_RAW);
    my_error = cpl_propertylist_update_string(keywords, PRO_CATG_KW, pro_catg);
    char ccd_cleaned_filename[FILENAME_LENGTH];
    sprintf(ccd_cleaned_filename, "%s_CCD_cleaned_%s_%s.fits",
            inst_config->instrument, pro_catg, fibre_b);
    
    /* Save the cleaned WAVE fits frame */
    my_error = espdr_save_ccd_cleaned_wave_frame(frameset,
                                                 used_frames,
                                                 parameters,
                                                 RECIPE_ID,
                                                 ccd_cleaned_filename,
                                                 CCD_geom,
                                                 keywords,
                                                 keywords_ext,
                                                 CCD_corrected_raw);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_save_ccd_cleaned_wave_frame failed: %s for file %s",
                 cpl_error_get_message_default(my_error),
                 ccd_cleaned_filename);
#endif
    
    cpl_imagelist *raw_bkgr_measured = cpl_imagelist_new();
    cpl_imagelist *raw_bkgr_subs = cpl_imagelist_new();
    
    /* Background calculation process ... */
    espdr_msg("background switch: <%s>", background_sw);
    if (strstr(background_sw, "on") != NULL) {
        espdr_msg("Background measurement process ...");
        cpl_imagelist *order_profile_fibre_iml_both_fibres = cpl_imagelist_duplicate(order_profile_fibre_imagelist[0]);
        if (strcmp(background_sw, "on") == 0) {
            espdr_msg("Removing fiber B from background pixels");
            cpl_imagelist_add(order_profile_fibre_iml_both_fibres,
                              cpl_imagelist_duplicate(order_profile_fibre_imagelist[1]));
        } else {
            espdr_msg("Using also pixels from fibre B for background calculation");
        }
        cpl_imagelist_add(order_profile_fibre_imagelist[0], order_profile_fibre_imagelist[1]);
        //espdr_msg("Background box size: %d x %d",
        //          inst_config->sci_bkgr_grid_size_x, inst_config->sci_bkgr_grid_size_y);
        espdr_msg("Background box size: %d x %d",
                  bkgr_grid_size_x, bkgr_grid_size_y);
        my_error = espdr_measure_background_science_by_ext(
                                            CCD_corrected_raw,
                                            order_profile_fibre_iml_both_fibres,
                                            conad,
                                            ron,
                                            bkgr_grid_size_x,
                                            bkgr_grid_size_y,
                                            CCD_geom,
                                            raw_bkgr_measured,
                                            raw_bkgr_subs);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_measure_inter_order_background failed");
        cpl_imagelist_delete(order_profile_fibre_iml_both_fibres);
        espdr_msg("Background measurement process finished !");
    } else {
        espdr_msg(ANSI_COLOR_RED"Warning : Background measurement NOT performed !"ANSI_COLOR_RESET);
        raw_bkgr_measured = NULL;
    }
    
    //espdr_msg("Saving BKGR KWs");
    char *new_keyword = NULL;
    double bkgr_mean = -9999.9, bkgr_max = -9999.9, bkgr_min = -9999.9;
    for (int i = 0; i < CCD_geom->ext_nb; i++) {
        if (raw_bkgr_measured != NULL) {
            bkgr_mean = cpl_image_get_mean(cpl_imagelist_get(raw_bkgr_measured,i));
            bkgr_min = cpl_image_get_min(cpl_imagelist_get(raw_bkgr_measured,i));
            bkgr_max = cpl_image_get_max(cpl_imagelist_get(raw_bkgr_measured,i));
        }
        
        new_keyword = espdr_add_output_index_to_keyword(qc_kws->qc_sci_red_bkgr_mean_kw_first,
                                                        qc_kws->qc_sci_red_bkgr_mean_kw_last, i);
        my_error = espdr_keyword_add_double(new_keyword, bkgr_mean, "Bkgr mean [e-] in the extension", &keywords);
        cpl_free(new_keyword);
        
        new_keyword = espdr_add_output_index_to_keyword(qc_kws->qc_sci_red_bkgr_min_kw_first,
                                                        qc_kws->qc_sci_red_bkgr_min_kw_last, i);
        my_error = espdr_keyword_add_double(new_keyword, bkgr_min, "Bkgr min [e-] in the extension", &keywords);
        cpl_free(new_keyword);
        
        new_keyword = espdr_add_output_index_to_keyword(qc_kws->qc_sci_red_bkgr_max_kw_first,
                                                        qc_kws->qc_sci_red_bkgr_max_kw_last, i);
        my_error = espdr_keyword_add_double(new_keyword, bkgr_max, "Bkgr max [e-] in the extension", &keywords);
        cpl_free(new_keyword);
    }
    

    
#if SAVE_DEBUG_PRODUCT_SHIFTED_EXTRACTION
    //-----------------------------------------------------------
    
    if (strcmp(background_sw, "on") == 0) {
        
        char bkgr_map_filename[64];
        char bkgr_subs_filename[64];
        
        sprintf(bkgr_map_filename, "%s_background_map.fits", inst_config->instrument);
        my_error = cpl_propertylist_update_string(keywords, PRO_CATG_KW, ESPDR_PRO_CATG_TEST_PRODUCT);
        my_error = espdr_dfs_image_save(frameset, parameters,
                                        used_frames, "shifted_extraction",
                                        keywords, keywords_ext,
                                        bkgr_map_filename,
                                        raw_bkgr_measured,
                                        CPL_TYPE_FLOAT, CCD_geom);
        
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "background_map image saving has failed: %s",
                     cpl_error_get_message_default(my_error));
        
        sprintf(bkgr_subs_filename, "%s_science_bkgr_subs.fits", inst_config->instrument);
        
        //SCIENCE (backgroud substracted) image saved
        my_error = espdr_dfs_image_save(frameset, parameters,
                                        used_frames, "shifted_extraction",
                                        keywords, keywords_ext,
                                        bkgr_subs_filename,
                                        raw_bkgr_subs,
                                        CPL_TYPE_FLOAT, CCD_geom);
        
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "science_bkgr_subs image saving has failed: %s",
                     cpl_error_get_message_default(my_error));
        
    }
    
#endif
    
    
    double exp_time_hour = cpl_propertylist_get_double(keywords, inst_config->Texp_kw) / 3600.0;
    double exp_time_hour_mdark = cpl_propertylist_get_double(keywords_HP, inst_config->Texp_kw) / 3600.0;
    
    cpl_image *s2d_flux, *s2d_error, *s2d_quality;
    
    /* extraction process */
    espdr_msg("Horne extraction process fibre A ...");
    double extraction_ksigma = inst_config->extraction_ksigma_object;
    espdr_msg("KSIGMA used for extraction: %f", extraction_ksigma);
    
    cpl_imagelist *input_iml = NULL;
    if (strcmp(background_sw, "off") == 0) {
        input_iml = CCD_corrected_raw;
    } else {
        input_iml = raw_bkgr_subs;
    }
    
    my_error = espdr_shifted_extraction_one_fibre(input_iml,
                                                  geometry_corrected_pixels_mask,
                                                  geometry_corrected_saturation_mask,
                                                  NULL, // used CRH map --> not used here
                                                  raw_bkgr_measured,
                                                  order_profile_fibre_imagelist[0],
                                                  master_dark_list,
                                                  orders_coeffs[0], 0,
                                                  inst_config, CCD_geom,
                                                  ron, dark, conad, exp_time_hour,
                                                  exp_time_hour_mdark,
                                                  inst_config->extraction_window_size[0],
                                                  inst_config->tolerance_rejection,
                                                  extraction_ksigma,
                                                  extraction_offset,
                                                  cosmics_nb,
                                                  &s2d_flux,
                                                  &s2d_error,
                                                  &s2d_quality);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_shifted_extraction_one_fibre failed for fibre A: %s",
                 cpl_error_get_message_default(my_error));
    espdr_msg("Shifted extraction process fibre A finished !");
    
    
#if SAVE_DEBUG_PRODUCT_SHIFTED_EXTRACTION
    
    char pro_catg_extr[16];
    char extracted_filename[64];
    strcpy(pro_catg_extr, ESPDR_SHIFTED_EXTRACTION_RAW);
    sprintf(extracted_filename, "%s_spectrum_extracted_%s_A.fits",
            inst_config->instrument, pro_catg_extr);
    espdr_msg("Saving extracted spectrum A in %s", extracted_filename);
    
    my_error = espdr_dfs_save_S2D(s2d_flux, s2d_error, s2d_quality,
                                  frameset, used_frames, parameters,
                                  keywords, extracted_filename,
                                  ESPDR_PRO_CATG_TEST_PRODUCT,
                                  "shifted_extraction");
    
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_dfs_save_S2D failed: %s for file %s",
                 cpl_error_get_message_default(my_error),
                 extracted_filename);
#endif
    
    /* Flat correction */
    espdr_msg("Flat correction process fibre A ...");
    my_error = espdr_correct_flat(s2d_flux,
                                  s2d_error,
                                  s2d_quality,
                                  flat_table[0],
                                  flat_error_table[0],
                                  flat_quality_table[0],
                                  flat_corr_flux_RE,
                                  flat_corr_err_RE,
                                  flat_corr_qual_RE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_correct_flat failed for fiber A: %s",
                 cpl_error_get_message_default(my_error));
    espdr_msg("---->Flat correction process fibre A finished !");
    
    espdr_msg("Compute SNR for fiber A on extracted spectrum");
    my_error = espdr_compute_snr(s2d_flux, s2d_error,
                                 inst_config->snr_averaging_window,
                                 snr_per_fibre);
    


    /* Cleaning some memory */
    cpl_imagelist_delete(CCD_corrected_raw);
    cpl_imagelist_delete(geometry_corrected_saturation_mask);
    cpl_imagelist_delete(raw_bkgr_subs);
    cpl_imagelist_delete(raw_bkgr_measured);

    for (int i = 0; i < CCD_geom->ext_nb; i++) {
        cpl_propertylist_delete(keywords_ext[i]);
    }
    cpl_free(keywords_ext);
    cpl_propertylist_delete(keywords_HP);
    cpl_propertylist_delete(keywords_BP);
    
    cpl_free(conad);
    cpl_free(dark);
    cpl_free(max_flux);
    
    espdr_msg("Cleaning orders coeffs");
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        for (int j = 0; j < CCD_geom->ext_nb; j++) {
            cpl_table_delete(orders_coeffs[i][j]);
        }
        cpl_free(orders_coeffs[i]);
    }
    cpl_free(orders_coeffs);
    
    cpl_image_delete(s2d_flux);
    cpl_image_delete(s2d_error);
    cpl_image_delete(s2d_quality);

    if ((inst_config->raw_bias_limit_nb > 0) && (remove_bias_res_flag == 1)) {
        cpl_imagelist_delete(master_bias_list);
    }
    cpl_imagelist_delete(hot_pixels_list);
    if (master_dark_frame != NULL) {
        cpl_imagelist_delete(master_dark_list);
    }
    cpl_imagelist_delete(bad_pixels_list);
    if (orders_mask_frame != NULL) {
        cpl_imagelist_delete(orders_mask_list);
    }
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        cpl_imagelist_delete(order_profile_fibre_imagelist[i]);
        cpl_image_delete(flat_table[i]);
        cpl_image_delete(flat_error_table[i]);
        cpl_image_delete(flat_quality_table[i]);
    }
    cpl_free(order_profile_fibre_imagelist);
    cpl_free(flat_table);
    cpl_free(flat_error_table);
    cpl_free(flat_quality_table);
    
    cpl_imagelist_delete(pixels_mask);
    cpl_imagelist_delete(geometry_corrected_pixels_mask);
        
    return cpl_error_get_code();
}



/*----------------------------------------------------------------------------*/
/**
 @brief Optimal (Horne) shifted extraction of spectrum on one fiber
 @param         input_images        imagelist of images to extract
 @param         hot_bad_pixels_mask mask of hot & bad pixels
 @param         saturation mask     mask of saturated pixels
 @param         background          background image
 @param         order_profile       order profile image
 @param         contam              contamination image
 @param         orders_coeffs       coefficients of the polynomial describing
                                    orders position
 @param         fibre_nr            number of the fiber to extract
 @param         inst_config         instrument configuration
 @param         CCD_geom            CCD geometry
 @param         RON                 RON for all the outputs
 @param         DARK                DARK for all the outputs
 @param         Texp                input frame exposure time in hours
 @param         extraction_window_size  size of the extraction window = 2*size+1
 @param         tolerance_reject    tolerance of residuals rejection
 @param         ksig_extraction     limit of residuals rejection
 @param         extraction_offset   offset to extract outside the orders
 @param[out]    cosmics_nb_RE   number of cosmics detected per order of each fibre
 @param[out]    extr_spectr_RE  extracted spectrum
 @param[out]    spectr_error_RE extracted spectrum error
 @param[out]    spectr_qual_RE  extracted spectrum quality (saturation & hot/bad
                                pixels)
 @return   CPL_ERROR_NONE iff OK
 
 Comment about cosmics:
 Once we get the cosmics flaggiing from the LA_cosmics algorithm, the flagged
 pixels will be treated the same way as saturated pixels:
 - they will not be included in the bad pixels, so not interpolated or preventing
   the optimal extraction
 - they will be passed to the optimal extraction as masked ones
 What to do to get the extracted flux as close to the one behind the hot/bad pixels,
 as possible?
 - create an artificial profile through a gaussian defined by a FWHM saved in the
   inst_config for each mode
 - use the artificial profile when the spline interpolation is not possible
   (too many hot/bad pixels), but there are still some good pixels
 - of course nothing can be done if all the pixels are hot or bad
 
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_shifted_extraction_one_fibre(cpl_imagelist *input_images,
                                                  cpl_imagelist *hot_bad_pixels_mask,
                                                  cpl_imagelist *saturation_mask,
                                                  cpl_mask **crh_mask_whole,
                                                  cpl_imagelist *background,
                                                  cpl_imagelist *order_profile,
                                                  cpl_imagelist *master_dark,
                                                  cpl_table **orders_coeffs,
                                                  int fibre_nr,
                                                  espdr_inst_config *inst_config,
                                                  espdr_CCD_geometry *CCD_geom,
                                                  double *RON,
                                                  double *DARK,
                                                  double *GAIN,
                                                  double Texp,
                                                  double Texp_master_dark,
                                                  int extraction_window_size,
                                                  double tolerance_reject,
                                                  double ksig_extraction,
                                                  int extraction_offset,
                                                  int *cosmics_nb_RE,
                                                  cpl_image **extr_spectr_RE,
                                                  cpl_image **spectr_error_RE,
                                                  cpl_image **spectr_qual_RE) {

    cpl_error_code my_error = CPL_ERROR_NONE;
    int i, ext, order, coeff, pxl, fibre_ext_index, index;
    double *one_order_coeffs = NULL;
    int coeffs_nb = 0;
    char column_name[KEYWORD_LENGTH];
    double order_pos = 0.0;
    double *extracted_spectrum = NULL;
    cpl_image *extracted_spectrum_image = NULL;
    double *spectrum_error = NULL;
    cpl_image *spectrum_error_image = NULL;
    int *spectrum_quality = NULL;
    cpl_image *spectrum_quality_image = NULL;
    cpl_image *hot_bad_image = NULL;
    int *hot_bad_data = NULL;
    cpl_image *satur_image = NULL;
    int *satur_data = NULL;
    cpl_mask *crh_mask_window = NULL;
    cpl_binary *crh_data = NULL;
    cpl_image *order_profile_image = NULL;
    double *order_profile_data = NULL;
    int order_length, number_of_orders, window_base;
    int max_order_length, min_order_start, max_order_end;
    int identified_number_of_orders, total_number_of_orders;
    int output_index = 0;
    double flux = 0.0;
    int converter_sat_flag = 0, detector_sat_flag = 0;
    int hot_pixel_flag = 0, bad_pixel_flag = 0;
    cpl_image *input_extraction_image = NULL;
    double *input_data = NULL;
    int *mask = NULL;
    int *corr_mask = NULL;
    int *satur_mask = NULL;
    int *crh_mask = NULL;
    int flux_min_index_lower, flux_min_index_upper;
    double flux_min_lower, flux_min_upper, profile_background;
    int hot_bad_pixels_nb = 0;
    int satur_pixels_nb = 0;
    int crh_pixels_nb = 0;
    double *var = NULL, *res = NULL;
    double *background_data = NULL;
    cpl_image *background_image = NULL;
    double *contam_data = NULL;
    cpl_image *contam_image = NULL;
    double *master_dark_data = NULL;
    cpl_image *master_dark_image = NULL;
    double sum_prof, sumPD, sumPP, res_max, variance_flux = 0.0;
    int resmax_index;
    int img_size_x, img_size_y, cosmics_nb;
    double sumC, RON_mean;
    double sumPC, sumCD, sumCC, k;
    int non_masked_pxls_nb;
    int good_pxls_nb;
    int window_fraction_background_compute = extraction_window_size+1;
    double RON_EL[CCD_geom->total_output_nb];
    double DARK_EL[CCD_geom->total_output_nb];
    
    int order_index = 0;
    
    int test_fibre = -1;
    int test_ext = 0;
    int test_order = 2;
    int test_pixel_min = 45;
    int test_pixel_max = 47;

    espdr_ensure(input_images == NULL, CPL_ERROR_NULL_INPUT,
                 "Input images are NULL");
    espdr_ensure(orders_coeffs == NULL, CPL_ERROR_NULL_INPUT,
                 "Orders coefficients table is NULL");
    
    espdr_msg("Starting optimal extraction on fibre %c, with ksig = %.2f, window size: %d, Texp: %f, Texp_mdark: %f",
              fibre_name[fibre_nr], ksig_extraction, extraction_window_size, Texp, Texp_master_dark);
    
    for (int i = 0; i < CCD_geom->total_output_nb; i++) {
        RON_EL[i] = RON[i] * GAIN[i];
        DARK_EL[i] = DARK[i] * GAIN[i];
        //espdr_msg("RON[%d] = %f, DARK[%d] = %f, GAIN[%d] = %f, RON_EL[%d] = %f, DARK_EL[%d] = %f",
        //          i, RON[i], i, DARK[i], i, GAIN[i], i, RON_EL[i], i, DARK_EL[i]);
    }
    
    max_order_length = 0;
    total_number_of_orders = 0;
    identified_number_of_orders = 0;
    min_order_start = 10000;
    max_order_end = 0;
    for (ext = 0; ext < CCD_geom->ext_nb; ext++) {
        fibre_ext_index = CCD_geom->ext_nb * fibre_nr + ext;
        //espdr_msg("ext: %d, fibre ext index: %d", ext, fibre_ext_index);
        
        if (min_order_start > inst_config->order_start[fibre_ext_index]) {
            min_order_start = inst_config->order_start[fibre_ext_index];
        }
        if (max_order_end < inst_config->order_end[fibre_ext_index]) {
            max_order_end = inst_config->order_end[fibre_ext_index];
        }
        
        total_number_of_orders = total_number_of_orders +
                                        inst_config->orders_nb[fibre_ext_index];
        
        identified_number_of_orders = identified_number_of_orders +
                                        cpl_table_get_nrow(orders_coeffs[ext]);
    }

    for (i = 0; i < total_number_of_orders; i++) {
        cosmics_nb_RE[i] = 0;
        //espdr_msg("Cosmics for order %d initialized to 0.0", i);
    }
    
    max_order_length = max_order_end - min_order_start + 1;

    if(identified_number_of_orders < total_number_of_orders) {
        
        espdr_msg(ANSI_COLOR_RED
                  "Warning : nominal number of orders: %d, identified number of orders: %d"
                  ANSI_COLOR_RESET,
                  total_number_of_orders, identified_number_of_orders);
        
        espdr_msg(ANSI_COLOR_RED
                  "Extracting nb of orders: %d, orders length: %d"
                  ANSI_COLOR_RESET,
                  identified_number_of_orders, max_order_length);
        
    } else {
        
        espdr_msg("Nominal number of orders: %d, identified number of orders: %d",
                  total_number_of_orders, identified_number_of_orders);
        espdr_msg("Extracting nb of orders: %d, orders length: %d",
                  identified_number_of_orders, max_order_length);
        
    }
    
    extracted_spectrum = (double *)cpl_calloc(max_order_length *
                                              total_number_of_orders,
                                              sizeof(double));
    spectrum_error = (double *)cpl_calloc(max_order_length *
                                          total_number_of_orders,
                                          sizeof(double));
    spectrum_quality = (int *)cpl_calloc(max_order_length *
                                         total_number_of_orders,
                                         sizeof(int));
    mask = (int *) cpl_calloc ((extraction_window_size*2+1), sizeof(int));
    corr_mask = (int *) cpl_calloc ((extraction_window_size*2+1), sizeof(int));
    satur_mask = (int *) cpl_calloc ((extraction_window_size*2+1), sizeof(int));
    crh_mask = (int *) cpl_calloc ((extraction_window_size*2+1), sizeof(int));
    var = (double *) cpl_calloc ((extraction_window_size*2+1), sizeof(double));
    res = (double *) cpl_calloc ((extraction_window_size*2+1), sizeof(double));
    if (background == NULL) {
        background_data = (double *)cpl_calloc((extraction_window_size*2+1), sizeof(double));
    }
    
    index = 0;
    
    cpl_imagelist *merged_master_dark = cpl_imagelist_new();
    cpl_imagelist *geometry_corrected_master_dark = cpl_imagelist_new();
    if (master_dark != NULL) {
        //espdr_msg("Correcting the master_dark geometry");
        my_error = espdr_image_merge_2_dim(master_dark, CCD_geom, CPL_TYPE_DOUBLE,
                                           &merged_master_dark);
        my_error = espdr_correct_geometry(merged_master_dark, CCD_geom,
                                          geometry_corrected_master_dark);
    }
    
    for (ext = 0; ext < CCD_geom->ext_nb; ext++) {
        fibre_ext_index = CCD_geom->ext_nb * fibre_nr + ext;
        order_length = inst_config->order_end[fibre_ext_index] -
                        inst_config->order_start[fibre_ext_index] + 1;
        number_of_orders = inst_config->orders_nb[fibre_ext_index];
        coeffs_nb = cpl_table_get_ncol(orders_coeffs[ext]);
        one_order_coeffs = (double *)cpl_calloc(coeffs_nb, sizeof(double));
        img_size_x = cpl_image_get_size_x(cpl_imagelist_get(input_images, ext));
        img_size_y = cpl_image_get_size_y(cpl_imagelist_get(input_images, ext));
        
        double RON_mean = 0.0;
        int i_start;
        if (ext == 0) {
            i_start = 0;
        } else {
            i_start = CCD_geom->exts[ext-1].out_nb_x * CCD_geom->exts[ext-1].out_nb_y;
        }
        for (i = i_start; i < i_start + CCD_geom->exts[ext].out_nb_x * CCD_geom->exts[ext].out_nb_y; i++) {
            RON_mean += RON_EL[i];
        }
        RON_mean = RON_mean / (CCD_geom->exts[ext].out_nb_x * CCD_geom->exts[ext].out_nb_y);
        
        // If some orders are not identified, they are not extracted
        for (order = 0; order < cpl_table_get_nrow(orders_coeffs[ext]); order++) {
            //espdr_msg("Extracting order %d", order);

            cosmics_nb = 0;

            for (coeff = 0; coeff < coeffs_nb; coeff++) {
                sprintf(column_name, "COEFF_%d", coeff);
                one_order_coeffs[coeff] = cpl_table_get(orders_coeffs[ext],
                                                        column_name, order, NULL);
            }
            
            if (inst_config->order_start[fibre_ext_index] > min_order_start) {
                for (pxl = min_order_start;
                     pxl < inst_config->order_start[fibre_ext_index];
                     pxl++) {
                    extracted_spectrum[index] = 0.0;
                    spectrum_error[index] = sqrt(extraction_window_size*2+1) * RON_mean;
                    spectrum_quality[index] = OTHER_BAD_PIXEL;
                    index++;
                }
            }
            
            for (pxl = inst_config->order_start[fibre_ext_index];
                 pxl <= inst_config->order_end[fibre_ext_index]; pxl++) {
                
                satur_pixels_nb = 0;
                crh_pixels_nb = 0;
                
                //espdr_msg("Extracting pxl %d", pxl);
                order_pos = 0.0;
                for (i = coeffs_nb-1; i > 0; i--) {
                    order_pos = (order_pos + one_order_coeffs[i]) * (double)pxl;
                }
                order_pos = order_pos + one_order_coeffs[0] + extraction_offset;
                double order_rest = order_pos - (int)order_pos;
                if (order_rest > 0.5) {
                    window_base = (int)order_pos - extraction_window_size + 1;
                } else {
                    window_base = (int)order_pos - extraction_window_size;
                }
                
                
                if ((window_base > 0) && (window_base+extraction_window_size*2 <= img_size_y)) {
                    // It's OK, the extraction can be done
                
                    espdr_ensure(((window_base < 1) || (window_base+extraction_window_size*2 > img_size_y)),
                                 CPL_ERROR_INCOMPATIBLE_INPUT,
                                 "The extraction window is too large (%d), please reduce its size and relaunch the recipe.",
                                 extraction_window_size);
                    
                    int pxl_raw = img_size_x * (window_base + extraction_window_size) + pxl;
                    output_index = espdr_get_output_index_for_pixel(ext, pxl_raw, CCD_geom);
                    espdr_ensure(cpl_error_get_code() != CPL_ERROR_NONE,
                                 cpl_error_get_code(),
                                 "espdr_get_output_index_for_pixel failed: %s",
                                 cpl_error_get_message_default(cpl_error_get_code()));
                    
                    spectrum_quality[index] = 0;
                    converter_sat_flag = 0;
                    detector_sat_flag = 0;
                    hot_pixel_flag = 0;
                    bad_pixel_flag = 0;
                    
                    //espdr_msg("Extracting hot_bad data");
                    hot_bad_image = cpl_image_extract(cpl_imagelist_get(hot_bad_pixels_mask, ext),
                                                      pxl, window_base, pxl,
                                                      window_base + extraction_window_size*2);
                    if (cpl_error_get_code() == CPL_ERROR_ILLEGAL_INPUT) {
                        espdr_msg("BAD coord (order %d): %d, %d, %d, %d",
                                  order, pxl, window_base, pxl,
                                  window_base + extraction_window_size*2);
                    }
                    espdr_ensure(cpl_error_get_code()!=CPL_ERROR_NONE,
                                 cpl_error_get_code(),
                                 "cpl_image_extract failed for hot_bad_pixels_mask: %s",
                                 cpl_error_get_message_default(cpl_error_get_code()));
                    hot_bad_data = cpl_image_get_data_int(hot_bad_image);
                    espdr_ensure(cpl_error_get_code() != CPL_ERROR_NONE,
                                 cpl_error_get_code(),
                                 "cpl_image_get_data_int failed for hot_bad_pixels_mask: %s",
                                 cpl_error_get_message_default(cpl_error_get_code()));
                    
                    if (background != NULL) {
                        //espdr_msg("Extracting background");
                        background_image =  cpl_image_extract(cpl_imagelist_get(background, ext),
                                                              pxl, window_base, pxl,
                                                              window_base + extraction_window_size*2);
                        background_data = cpl_image_get_data_double(background_image);
                        espdr_ensure(cpl_error_get_code() != CPL_ERROR_NONE,
                                     cpl_error_get_code(),
                                     "cpl_image_extract failed for background: %s",
                                     cpl_error_get_message_default(cpl_error_get_code()));
                    }
                    
                    //espdr_msg("Extracting saturation");
                    satur_image = cpl_image_extract(cpl_imagelist_get(saturation_mask, ext),
                                                    pxl, window_base, pxl,
                                                    window_base + extraction_window_size*2);
                    espdr_ensure(cpl_error_get_code() != CPL_ERROR_NONE,
                                 cpl_error_get_code(),
                                 "cpl_image_extract failed for saturation_mask: %s",
                                 cpl_error_get_message_default(cpl_error_get_code()));
                              
                    satur_data = cpl_image_get_data_int(satur_image);
                    for (i = 0; i < extraction_window_size*2+1; i++) {
                        if (satur_data[i] != 0) {
                            satur_pixels_nb ++;
                            satur_mask[i] = 0;
                        } else {
                            satur_mask[i] = 1;
                        }
                    }

                    //espdr_msg("Extracting cosmics");
                    if (crh_mask_whole != NULL) {
                        crh_mask_window = cpl_mask_extract(crh_mask_whole[ext],
                                                           pxl, window_base, pxl,
                                                           window_base + extraction_window_size*2);
                        espdr_ensure(cpl_error_get_code() != CPL_ERROR_NONE,
                                     cpl_error_get_code(),
                                     "cpl_mask_extract failed for crh_mask: %s",
                                     cpl_error_get_message_default(cpl_error_get_code()));
                        
                        crh_data = cpl_mask_get_data(crh_mask_window);
                        for (i = 0; i < extraction_window_size*2+1; i++) {
                            if (crh_data[i] == CPL_BINARY_1) {
                                crh_pixels_nb ++;
                                crh_mask[i] = 0;
                            } else {
                                crh_mask[i] = 1;
                            }
                        }
                    } else {
                        for (i = 0; i < extraction_window_size*2+1; i++) {
                            crh_mask[i] = 1;
                        }
                    }
                    
                    if (master_dark != NULL) {
                        //espdr_msg("Extracting master_dark for master_dark error calculation");
                        master_dark_image =  cpl_image_extract(cpl_imagelist_get(geometry_corrected_master_dark, ext),
                                                               pxl, window_base, pxl,
                                                               window_base + extraction_window_size*2);
                        master_dark_data = cpl_image_get_data_double(master_dark_image);
                        espdr_ensure(cpl_error_get_code() != CPL_ERROR_NONE,
                                     cpl_error_get_code(),
                                     "cpl_image_extract failed for master_dark: %s",
                                     cpl_error_get_message_default(cpl_error_get_code()));
                    }
                    
                    //espdr_msg("Extracting image");
                    input_extraction_image = cpl_image_extract(cpl_imagelist_get(input_images, ext),
                                                               pxl, window_base, pxl,
                                                               window_base + extraction_window_size*2);
                    espdr_ensure(cpl_error_get_code()!=CPL_ERROR_NONE,
                                 cpl_error_get_code(),
                                 "cpl_image_extract failed for input_data: %s",
                                 cpl_error_get_message_default(cpl_error_get_code()));
                    
                    input_data = cpl_image_get_data_double(input_extraction_image);
                    espdr_ensure(cpl_error_get_code()!=CPL_ERROR_NONE,
                                 cpl_error_get_code(),
                                 "cpl_image_get_data_double failed: %s",
                                 cpl_error_get_message_default(cpl_error_get_code()));
                    
                    /* DEBUGGING DANUTA */
                    
                    if ((fibre_nr == test_fibre) && (ext == test_ext) && (order == test_order) &&
                        (pxl >= test_pixel_min) && (pxl <= test_pixel_max)) {
                        espdr_msg("EXTRACTION DEBUG START -------------------------------->");
                        espdr_msg("ext %d, order: %d, pxl: %d, window_base: %d",
                                  ext, order, pxl, window_base);
                    }
                    
                    /********************/
                    
                    espdr_msg_debug("Extracting order profile");
                    order_profile_image =
                    cpl_image_extract(cpl_imagelist_get(order_profile, ext),
                                      pxl, window_base - extraction_offset, pxl,
                                      window_base + extraction_window_size*2 - extraction_offset);
                    espdr_ensure(cpl_error_get_code()!=CPL_ERROR_NONE,
                                 cpl_error_get_code(),
                                 "cpl_image_extract failed for input_data: %s",
                                 cpl_error_get_message_default(cpl_error_get_code()));
                    
                    order_profile_data = cpl_image_get_data_double(order_profile_image);
                    espdr_ensure(cpl_error_get_code()!=CPL_ERROR_NONE,
                                 cpl_error_get_code(),
                                 "cpl_image_get_data_double failed: %s",
                                 cpl_error_get_message_default(cpl_error_get_code()));
                    
                    
                    /* DEBUGGING DANUTA */
                    
                    if ((fibre_nr == test_fibre) && (ext == test_ext) && (order == test_order) &&
                        (pxl >= test_pixel_min) && (pxl <= test_pixel_max)) {
                        espdr_msg("Image read: ext: %d, order: %d, pxl: %d",
                                  ext, order, pxl);
                        if (crh_mask_whole != NULL) {
                            for (i = 0; i < extraction_window_size*2+1; i++) {
                                espdr_msg("order_profile[%d] = %lf\tinput_data[%d] = %lf\thot_bad[%d] = %d\tsatur[%d] = %d\tcrh[%d] = %d",
                                          i, order_profile_data[i],
                                          i, input_data[i],
                                          i, hot_bad_data[i],
                                          i, satur_data[i],
                                          i, crh_data[i]);
                            }
                        } else {
                            for (i = 0; i < extraction_window_size*2+1; i++) {
                                espdr_msg("order_profile[%d] = %lf\tinput_data[%d] = %lf\thot_bad[%d] = %d\tsatur[%d] = %d",
                                          i, order_profile_data[i],
                                          i, input_data[i],
                                          i, hot_bad_data[i],
                                          i, satur_data[i]);
                            }
                        }
                    }
                    
                    
                    
                    /* Compute the background level */
                    espdr_msg_debug("Computing background level");
                    hot_bad_pixels_nb = 0;
                    good_pxls_nb = 0;
                    for (i = 0; i < extraction_window_size*2+1; i++) {
                        if (hot_bad_data[i] > 0) {
                            mask[i] = 0;
                            hot_bad_pixels_nb++;
                        } else {
                            mask[i] = 1;
                        }
                        
                        if ((mask[i] == 1) && (satur_mask[i] == 1) && (crh_mask[i] == 1)) {
                            good_pxls_nb++;
                        }
                        
                        if ((fibre_nr == test_fibre) && (ext == test_ext) && (order == test_order) &&
                            (pxl >= test_pixel_min) && (pxl <= test_pixel_max)) {
                            espdr_msg("order: %d, pxl: %d, window_base: %d, hot_bad[%d]: %d, hot_bad_nb %d, satur[%d] = %d, crh[%d] = %d, qual: %d",
                                      order, pxl, window_base, i, hot_bad_data[i], hot_bad_pixels_nb,
                                      i, satur_mask[i], i, crh_mask[i], spectrum_quality[index]);
                        }
                    }
                    
                    if ((fibre_nr == test_fibre) && (ext == test_ext) && (order == test_order) &&
                        (pxl >= test_pixel_min) && (pxl <= test_pixel_max)) {
                        espdr_msg("good_pxls_nb with all masks: %d", good_pxls_nb);
                    }
                    
                    if (good_pxls_nb == 0) {
                        extracted_spectrum[index] = 0.0;
                        spectrum_error[index] = sqrt(extraction_window_size*2+1) * RON_EL[output_index];
                        for (i = 0; i < extraction_window_size*2+1; i++) {
                            if (hot_bad_data[i] > 0) {
                                spectrum_quality[index] = spectrum_quality[index] | hot_bad_data[i];
                            }
                            if (satur_data[i] > 0) {
                                spectrum_quality[index] = spectrum_quality[index] | satur_data[i];
                            }
                            if (crh_mask_whole != NULL) {
                                if (crh_data[i] > 0) {
                                    spectrum_quality[index] = spectrum_quality[index] | COSMIC_RAY_REMOVED;
                                }
                            }
                        }
                    } else {
                        // at least one good pixel present in the extraction window
                    
                        window_fraction_background_compute = (int)(extraction_window_size/
                                                                   inst_config->slices_nb)+1;
                        // To avoid dust on the order effect, we take just the 30% border of the order
                        // to find the correct background
                        if (inst_config->slices_nb == 1) {
                            window_fraction_background_compute = (int)(extraction_window_size*
                                                                       inst_config->extraction_bkgr_calc_fraction);
                        }
                        flux_min_lower = DBL_MAX;
                        flux_min_index_lower = -1;
                        for (i = 0; i < window_fraction_background_compute; i++) {
                            if (hot_bad_data[i] == 0) {
                                if (order_profile_data[i] < flux_min_lower) {
                                    flux_min_lower = order_profile_data[i];
                                    flux_min_index_lower = i;
                                }
                            }
                        }
                        
                        flux_min_upper = DBL_MAX;
                        flux_min_index_upper = -1;
                        for (i = extraction_window_size * 2;
                             i > extraction_window_size*2 - window_fraction_background_compute;
                             i--) {
                            if (hot_bad_data[i] == 0) {
                                if (order_profile_data[i] < flux_min_upper) {
                                    flux_min_upper = order_profile_data[i];
                                    flux_min_index_upper = i;
                                }
                            }
                        }
                        
                        if ((flux_min_index_lower == -1) && (flux_min_index_upper == -1)) {
                            profile_background = 0.0;
                        } else {
                            if (flux_min_index_lower == -1) {
                                profile_background = order_profile_data[flux_min_index_upper];
                            } else {
                                if (flux_min_index_upper == -1) {
                                    profile_background = order_profile_data[flux_min_index_lower];
                                } else {
                                    if (order_profile_data[flux_min_index_lower] >
                                        order_profile_data[flux_min_index_upper]) {
                                        profile_background = order_profile_data[flux_min_index_lower];
                                    } else {
                                        profile_background = order_profile_data[flux_min_index_upper];
                                    }
                                }
                            }
                        }
                        
                        /* DEBUGGING DANUTA */
                        
                        if ((fibre_nr == test_fibre) && (ext == test_ext) && (order == test_order) &&
                            (pxl >= test_pixel_min) && (pxl <= test_pixel_max)) {
                            espdr_msg("profile_background = %lf", profile_background);
                            espdr_msg("window fraction background compute = %d",
                                      window_fraction_background_compute);
                            espdr_msg("hot bad pixels nb = %d",
                                      hot_bad_pixels_nb);
                            espdr_msg("Quality: %d", spectrum_quality[index]);
                        }
                        
                        /********************/
                        
                        // If the upper index is -1 --> all profile -> 0.0, so we have to put it to the last index in the window
                        if (flux_min_index_upper == -1) {
                            flux_min_index_upper = extraction_window_size*2+1;
                        }
                        for (i = 0; i < extraction_window_size*2+1; i++) {
                            if (mask[i] == 1) {
                                if ((i < flux_min_index_lower) ||
                                    (i > flux_min_index_upper)) {
                                    order_profile_data[i] = 0.0;
                                } else {
                                    if (order_profile_data[i] < profile_background) {
                                        order_profile_data[i] = 0.0;
                                    } else {
                                        order_profile_data[i] -= profile_background;
                                    }
                                }
                            }
                        }
                        
                        /* DEBUGGING DANUTA */
                        
                        if ((fibre_nr == test_fibre) && (ext == test_ext) && (order == test_order) &&
                            (pxl >= test_pixel_min) && (pxl <= test_pixel_max)) {
                            espdr_msg("EXTRACTION DEBUG------>");
                            espdr_msg("flux min lower: %lf at %d, flux min upper: %lf at %d",
                                      flux_min_lower, flux_min_index_lower,
                                      flux_min_upper, flux_min_index_upper);
                        }
                        
                        if ((fibre_nr == test_fibre) && (ext == test_ext) && (order == test_order) &&
                            (pxl >= test_pixel_min) && (pxl <= test_pixel_max)) {
                            espdr_msg("After background level: order: %d, pxl: %d, hot_bad_nb: %d, qual: %d",
                                      order, pxl, hot_bad_pixels_nb, spectrum_quality[index]);
                            if (crh_mask_whole != NULL) {
                                for (i = 0; i < extraction_window_size*2+1; i++) {
                                    espdr_msg("order_profile[%d] = %f\tinput_data[%d] = %f\thot_bad[%d] = %d\tsatur_data[%d] = %d\tsatur_mask[%d] = %d\tcrh_data[%d] = %d",
                                              i, order_profile_data[i],
                                              i, input_data[i],
                                              i, hot_bad_data[i],
                                              i, satur_data[i],
                                              i, satur_mask[i],
                                              i, crh_data[i]);
                                }
                            } else {
                                for (i = 0; i < extraction_window_size*2+1; i++) {
                                    espdr_msg("order_profile[%d] = %f\tinput_data[%d] = %f\thot_bad[%d] = %d\tsatur_data[%d] = %d\tsatur_mask[%d] = %d",
                                              i, order_profile_data[i],
                                              i, input_data[i],
                                              i, hot_bad_data[i],
                                              i, satur_data[i],
                                              i, satur_mask[i]);
                                }
                            }
                        }
                        
                        /********************/
                        
                        /* Extraction */
                        
                        sum_prof = 0.0;
                        flux = 0.0;
                        for(i = 0; i < extraction_window_size*2+1; i++) {
                            sum_prof += order_profile_data[i] * mask[i]; // only good & saturated pixels
                            flux += input_data[i] * mask[i]; // only good & saturated pixels
                        }
                        
                        if (sum_prof == 0) {
                            for (i = 0; i < extraction_window_size*2+1; i++) {
                                extracted_spectrum[index] += input_data[i] * mask[i] * satur_mask[i] * crh_mask[i]; // only good pixels
                                spectrum_quality[index] = spectrum_quality[index] | CALIBRATION_DEFECT;
                            }
                            double MAXSPEC = ESPDR_MAX(0, extracted_spectrum[index];
                            spectrum_error[index] = sqrt(MAXSPEC + (extraction_window_size*2+1) * RON_EL[output_index] * RON_EL[output_index]));
                        } else {
                            
                            for (i = 0; i < extraction_window_size*2+1; i++) {
                                order_profile_data[i] = order_profile_data[i]/sum_prof;
                                var[i] = flux*order_profile_data[i]+background_data[i];
                                if (master_dark == NULL) {
                                    var[i] += DARK_EL[output_index] * Texp;
                                    var[i] = ESPDR_MAX(0,var[i]);
                                    var[i] += RON_EL[output_index] * RON_EL[output_index];
                                } else {
                                    // No MASTER DARK noise propagation, it is negligible for 10 raw DARKs
                                    var[i] += master_dark_data[i] * GAIN[output_index] * Texp / Texp_master_dark;
                                    var[i] = ESPDR_MAX(0,var[i]);
                                    var[i] += RON_EL[output_index] * RON_EL[output_index];
                                }
                            }
                            
                            do {
                                sumPD = 0.0;
                                sumPP = 0.0;
                                res_max = 0.0;
                                resmax_index = 0;
                                
                                /* Compute the flux */
                                for (i = 0; i < extraction_window_size*2+1; i++) {
                                    sumPD += (order_profile_data[i]/var[i]) * mask[i] * satur_mask[i] * input_data[i] * crh_mask[i];
                                    sumPP += (order_profile_data[i] * order_profile_data[i]/var[i]) * mask[i] * satur_mask[i] * crh_mask[i];
                                }
                                flux = sumPD / sumPP;
                                
                                if (sumPP == 0.0) {
                                    flux = 0.0;
                                    for (i = 0; i < extraction_window_size*2+1; i++) {
                                        if (hot_bad_data[i] > 0) {
                                            spectrum_quality[index] = spectrum_quality[index] | hot_bad_data[i];
                                        }
                                        if (satur_data[i] > 0) {
                                            spectrum_quality[index] = spectrum_quality[index] | satur_data[i];
                                        }
                                        if (crh_mask_whole != NULL) {
                                            if (crh_data[i] > 0) {
                                                spectrum_quality[index] = spectrum_quality[index] | COSMIC_RAY_REMOVED;
                                            }
                                        }
                                    }
                                    good_pxls_nb = 0;
                                }
                                
                                /* Compute the variance and residuals */
                                for (i = 0; i < extraction_window_size*2+1; i++) {
                                    var[i] = flux * order_profile_data[i] + background_data[i];
                                    if (master_dark == NULL) {
                                        var[i] += DARK_EL[output_index] * Texp;
                                        var[i] = ESPDR_MAX(0,var[i]);
                                        var[i] += RON_EL[output_index] * RON_EL[output_index];
                                    } else {
                                        // No MASTER DARK noise propagation, it is negligible for 10 raw DARKs
                                        var[i] += master_dark_data[i] * GAIN[output_index] * Texp / Texp_master_dark;
                                        var[i] = ESPDR_MAX(0,var[i]);
                                        var[i] += RON_EL[output_index] * RON_EL[output_index];
                                    }

                                    res[i] = (input_data[i] - flux*order_profile_data[i]) * mask[i] * satur_mask[i] * crh_mask[i] /
                                                sqrt(var[i] + flux * flux * tolerance_reject * tolerance_reject);
                                    if (res[i] > res_max) {
                                        res_max = res[i];
                                        resmax_index = i;
                                    }
                                }
                                
                                /* if ksig_extraction < 0.0 we don't reject cosmics */
                                if ((res_max>ksig_extraction) && (ksig_extraction>0.0)) {
                                    mask[resmax_index] = 0;
                                    cosmics_nb++;
                                    if (good_pxls_nb > 0) { // if all the pixels are already bad - don't remove
                                        good_pxls_nb--;
                                    }
                                    if ((fibre_nr == test_fibre) && (ext == test_ext) && (order == test_order) &&
                                        (pxl >= test_pixel_min) && (pxl <= test_pixel_max)) {
                                        espdr_msg("EXTRACTION DEBUG------> cosmics sigma clipping");
                                        espdr_msg("ext: %d, order: %d, pxl: %d, res_max = %f",
                                                  ext, order+1, pxl, res_max);
                                    }
                                }
                            } while ((res_max>ksig_extraction) && (ksig_extraction>0.0) && (good_pxls_nb>0));
                        } // if sum_prof == 0
                        
                        if ((fibre_nr == test_fibre) && (ext == test_ext) && (order == test_order) &&
                            (pxl >= test_pixel_min) && (pxl <= test_pixel_max)) {
                            espdr_msg("sum_prof: %f, good_pxls_nb: %d, flux: %f, extr_spec: %f, error: %f, qual: %d",
                                      sum_prof, good_pxls_nb, flux, extracted_spectrum[index],
                                      spectrum_error[index], spectrum_quality[index]);
                        }
                        
                        if (sum_prof != 0.0) {
                            if (good_pxls_nb == 0) {
                                extracted_spectrum[index] = 0.0;
                                spectrum_error[index] = sqrt(extraction_window_size*2+1) * RON_EL[output_index];
                                if (sumPP != 0.0) { // if sum_PP == 0.0: vicious case of 1 good pixel on the border, no cosmics
                                    spectrum_quality[index] = spectrum_quality[index] | COSMIC_RAY_REMOVED;
                                }
                            } else {
                                extracted_spectrum[index] = flux;
                                variance_flux = 0.0;
                                for (i = 0; i < extraction_window_size*2+1; i++) {
                                    variance_flux += (order_profile_data[i] * order_profile_data[i] / var[i]) *
                                                    mask[i] * satur_mask[i] * crh_mask[i];
                                }
                                
                                spectrum_error[index] = sqrt(1.0 / variance_flux);
                            }
                        }
                        
                        /* DEBUGGING DANUTA */
                        
                        if ((fibre_nr == test_fibre) && (ext == test_ext) && (order == test_order) &&
                            (pxl >= test_pixel_min) && (pxl <= test_pixel_max)) {
                            espdr_msg("EXTRACTION DEBUG------> FINAL++++++++++++++++++++++++++++");
                            espdr_msg("ext: %d, order: %d, pxl: %d, flux: %lf, err: %f, quality: %d",
                                      ext, order, pxl, extracted_spectrum[index],
                                      spectrum_error[index], spectrum_quality[index]);
                        }
                        
                        if (isnan(extracted_spectrum[index]) || isinf(extracted_spectrum[index])) {
                            espdr_msg_warning("Extracted pixel %d is NAN or INF, fibre: %d, ext: %d, order: %d, pxl: %d",
                                              index, fibre_nr, ext, order, pxl);
                            return(CPL_ERROR_INCOMPATIBLE_INPUT);
                        }
                        
                        if (isinf(spectrum_error[index]) || isnan(spectrum_error[index])) {
                            espdr_msg_warning("Extracted pixel %d error is INF or NAN, fibre: %d, ext: %d, order: %d, pxl: %d",
                                              index, fibre_nr, ext, order, pxl);
                            return(CPL_ERROR_INCOMPATIBLE_INPUT);
                        }
                        
                    } // else if (all_hot_bad || all_saturated)
                    
                    cpl_image_delete(input_extraction_image);
                    cpl_image_delete(hot_bad_image);
                    cpl_image_delete(satur_image);
                    if (crh_mask_whole != NULL) {
                        cpl_mask_delete(crh_mask_window);
                    }
                    if (background != NULL) {
                        cpl_image_delete(background_image);
                    }
                    cpl_image_delete(order_profile_image);
                    
                } else { // if (window_base > 0) && (window_base*2+1 < img_size_y)
                    
                    // Fill the rest of the extracted spectrum with 0.0
                    extracted_spectrum[index] = 0.0;
                    spectrum_error[index] = sqrt(extraction_window_size*2+1) * RON_EL[output_index];
                    spectrum_quality[index] = OTHER_BAD_PIXEL;
                    
                } // if (window_base > 0) && (window_base*2+1 < img_size_y)
                
                index++;
            } // end of the for loop on pixels
            
            if (inst_config->order_end[fibre_ext_index] < max_order_end) {
                for (pxl = inst_config->order_end[fibre_ext_index] + 1;
                     pxl <= max_order_end;
                     pxl++) {
                    extracted_spectrum[index] = 0.0;
                    spectrum_error[index] = sqrt(extraction_window_size*2+1) * RON_mean;
                    spectrum_quality[index] = OTHER_BAD_PIXEL;
                    index++;
                }
            }
            cosmics_nb_RE[order_index] = cosmics_nb;
            order_index++;
            
        } // end of for loop on orders
        cpl_free(one_order_coeffs);
    } // end of for loop on extensions
    
    //espdr_msg("Constructing extracted spectrum image");
    extracted_spectrum_image = cpl_image_wrap_double(max_order_length,
                                                     total_number_of_orders,
                                                     extracted_spectrum);
    *extr_spectr_RE = cpl_image_duplicate(extracted_spectrum_image);
    cpl_image_unwrap(extracted_spectrum_image);
    cpl_free(extracted_spectrum);
    
    spectrum_error_image = cpl_image_wrap_double(max_order_length,
                                                 total_number_of_orders,
                                                 spectrum_error);
    *spectr_error_RE = cpl_image_duplicate(spectrum_error_image);
    cpl_image_unwrap(spectrum_error_image);
    cpl_free(spectrum_error);
    
    spectrum_quality_image = cpl_image_wrap_int(max_order_length,
                                                total_number_of_orders,
                                                spectrum_quality);
    *spectr_qual_RE = cpl_image_duplicate(spectrum_quality_image);
    cpl_image_unwrap(spectrum_quality_image);
    cpl_free(spectrum_quality);
    
    if (background == NULL) {
        cpl_free(background_data);
    }
    cpl_free(mask);
    cpl_free(corr_mask);
    cpl_free(satur_mask);
    cpl_free(crh_mask);
    cpl_free(var);
    cpl_free(res);
    cpl_imagelist_delete(geometry_corrected_master_dark);
    
    
    espdr_msg_debug("EOF: extr_spectr_RE size: %lld x %lld",
                    cpl_image_get_size_x(*extr_spectr_RE),
                    cpl_image_get_size_y(*extr_spectr_RE));
    
    return cpl_error_get_code();
}


/*---------------------------------------------------------------------------*/
/**
 @brief    Check the sci_red QC
 @param    CCD_geom         CCD geometry
 @param    exposure_time    exposure time
 @param    inst_config      instrument configuration structure
 @param    qc_kws           structure holdin quality control, keys
 @param    keywords_RE      fits header to hold QC
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_shifted_extraction_QC(espdr_CCD_geometry *CCD_geom,
                                           espdr_inst_config *inst_config,
                                           espdr_qc_keywords *qc_kws,
                                           char *fibre_B_src,
                                           int *cosmics_nb,
                                           double *snr,
                                           int orders_nr,
                                           double *ron_RE,
                                           int extraction_offset,
                                           cpl_propertylist **keywords_RE) {

    cpl_error_code my_error = CPL_ERROR_NONE;
    int index;
    char *new_keyword = NULL;
    int saturation_QC = 1, global_QC = 1;
    char comment[COMMENT_LENGTH];
    
    //espdr_msg("Saving QC KWs");
    //espdr_msg("Saving RON KWs");
    index = 0;
    for (int i = 0; i < CCD_geom->ext_nb; i++) {
        for (int j = 0; j < CCD_geom->exts[i].out_nb_x; j++) {
            for (int k = 0; k < CCD_geom->exts[i].out_nb_y; k++) {
                new_keyword = espdr_add_ext_output_index_to_keyword(qc_kws->qc_out_bias_ron_kw_part,
                                                                    inst_config->prefix, i, j, k);
                sprintf(comment, "RON[ADU] for ext %d, out %d, %d", i, j, k);
                my_error = espdr_keyword_add_double(new_keyword, ron_RE[index], comment, keywords_RE);
                cpl_free(new_keyword);
                index++;
            }
        }
    }
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add RON keyword to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));
    
    if (cpl_propertylist_has(*keywords_RE, qc_kws->qc_saturation_check_kw)) {
        saturation_QC = cpl_propertylist_get_int(*keywords_RE, qc_kws->qc_saturation_check_kw);
        if ((saturation_QC == 0) && (strcmp(fibre_B_src, "THAR") != 0)) {
            global_QC = 0;
        }
    }
    
    
    //espdr_msg("Saving cosmics KWs");
    for (int i = 0; i < orders_nr; i++) {
        new_keyword = espdr_add_output_index_to_keyword(qc_kws->qc_sci_red_order_cosmic_nb_kw_first,
                                                        qc_kws->qc_sci_red_order_cosmic_nb_kw_last,
                                                        i+1);
        my_error = espdr_keyword_add_int(new_keyword, cosmics_nb[i], "Cosmics number in the order",
                                         keywords_RE);
        cpl_free(new_keyword);
    }
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add QC_ORDER_COSMIC_NB_KW to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));

    //espdr_msg("Saving SNR KWs");
    for (int i = 1; i <= orders_nr; i++) {
        new_keyword = espdr_add_output_index_to_keyword(qc_kws->qc_sci_red_order_snr_kw_first,
                                                        qc_kws->qc_sci_red_order_snr_kw_last,
                                                        i);
        my_error = espdr_keyword_add_double(new_keyword, snr[i-1], "SNR in the order", keywords_RE);
        cpl_free(new_keyword);
    }
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add QC_ORDER_SNR_KW to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));
    
    //espdr_msg("Saving extrcation offset KW");
    my_error = espdr_keyword_add_int(qc_kws->qc_shifted_extraction_offset_kw, extraction_offset,
                                     "Shifted extraction offset", keywords_RE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add keyword QC_SHIFTED_EXTRACTION_OFFSET_KW to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));
    
    //espdr_msg("Saving global shifted extraction QC");
    my_error = espdr_keyword_add_int(qc_kws->qc_shifted_extraction_check_kw, global_QC,
                                     "Shifted extraction global QC", keywords_RE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Add keyword QC_SHIFTED_EXTRACTION_CHECK_KW to the propertylist failed: %s",
                 cpl_error_get_message_default(my_error));
    
    return cpl_error_get_code();
}


/*---------------------------------------------------------------------------*/
/**
 @brief     Save shifted_extraction products
 @param
 @return    CPL_ERROR_NONE iff OK
 
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_save_ghost_S2D_products(cpl_frameset *frameset,
                                             cpl_parameterlist *parameters,
                                             const char *recipe_id,
                                             cpl_frameset *used_frames,
                                             const char *GHOST_tag,
                                             int extraction_offset,
                                             cpl_propertylist *keywords,
                                             espdr_inst_config *inst_config,
                                             cpl_image *flat_corr_spectrum,
                                             cpl_image *flat_corr_error,
                                             cpl_image *flat_corr_qual) {

    cpl_error_code my_error = CPL_ERROR_NONE;
    char filename[FILENAME_LENGTH];
    char *new_keyword =(char *)cpl_malloc(KEYWORD_LENGTH*sizeof(char));
    
    /* Save the ed2s extracted spectra immediately after creation */
    espdr_msg("Saving the S2D extracted spectra");
    cpl_image **images_to_save = (cpl_image**)cpl_malloc(3*sizeof(cpl_image*));
    
    espdr_msg("Saving flat corrected spectra");
    sprintf(new_keyword, "%s_BLAZE_GHOSTS_A", ESPDR_PRO_CATG_S2D);
    espdr_msg("Adding PRO.CATG KW: %s", new_keyword);
    my_error = cpl_propertylist_update_string(keywords, PRO_CATG_KW, new_keyword);
    sprintf(filename, "%s_BLAZE_GHOSTS_A_%d.fits",
            inst_config->extr_spectr_filename_no_fits, extraction_offset);
    
    images_to_save[0] = cpl_image_duplicate(flat_corr_spectrum);
    images_to_save[1] = cpl_image_duplicate(flat_corr_error);
    images_to_save[2] = cpl_image_duplicate(flat_corr_qual);
    my_error = espdr_dfs_save_data_err_qual(frameset, parameters, used_frames, recipe_id,
                                            keywords, filename, images_to_save);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_dfs_image_save failed: %s for file %s",
                 cpl_error_get_message_default(my_error), filename);
    for (int i = 0; i < 3; i++) {
        cpl_image_delete(images_to_save[i]);
    }
    
    cpl_free(images_to_save);
    
    espdr_msg("S2D saved");
    
    return (cpl_error_get_code());
}


