/*                                                                            *
 *   This file is part of the ESPRESSO Pipeline                               *
 *   Copyright (C) 2006 European Southern Observatory                         *
 *                                                                            *
 *   This library is free software; you can redistribute it and/or modify     *
 *   it under the terms of the GNU General Public License as published by     *
 *   the Free Software Foundation; either version 2 of the License, or        *
 *   (at your option) any later version.                                      *
 *                                                                            *
 *   This program is distributed in the hope that it will be useful,          *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 *   GNU General Public License for more details.                             *
 *                                                                            *
 *   You should have received a copy of the GNU General Public License        *
 *   along with this program; if not, write to the Free Software              *
 *   Foundation, 51 Franklin St, Fifth Floor, Boston, MA  02111-1307  USA     *
 *                                                                            */

/*
 * $Author: dsosnows $
 * $Date: 2023-05-22 16:52:15 $
 * $Revision: 1.6 $
 * $Name: not supported by cvs2svn $
 */

#ifndef ESPDR_TELLURICS_H
#define ESPDR_TELLURICS_H

/*----------------------------------------------------------------------------
 Includes
 ----------------------------------------------------------------------------*/
/* DRL functions*/
#include <espdr_instrument.h>
#include <espdr_msg.h>
#include <espdr_CCD.h>
#include <espdr_dfs.h>
#include <espdr_pixels.h>
#include <espdr_fit.h>
#include <espdr_utils.h>
#include <espdr_science.h>
#include <espdr_ccf.h>

/* Library */
#include <cpl.h>
#include <assert.h>
#include <gsl/gsl_multimin.h>
#include <gsl/gsl_rng.h>
#include <gsl/gsl_randist.h>
#include <gsl/gsl_matrix.h>
#include <gsl/gsl_vector.h>
#include <gsl/gsl_blas.h>
#include <gsl/gsl_multifit_nlinear.h>
#include <time.h>

/*----------------------------------------------------------------------------
 Constants
 ----------------------------------------------------------------------------*/
#define C2 1.43880285              // [cm * K] C * h / k
#define NA 6.02214086e23           // [1/mol]
#define KB 1.3806503e-23           // [J/K] or [kg * m2 / s2]
#define N_H2O 3.3427274952610645e22 // [molecules / cm3]
#define N_O2 5.649e18               // [molecules / cm3]
#define N_CO2 9.8185e15             // [molecules / cm3]
#define N_CH4 4.573e13              // [molecules / cm3]
#define M_molH2O 18.01528           // [g / mol]
#define M_molO2 31.9988             // [g / mol]
#define M_molCO2 44.01              // [g / mol]
#define M_molCH4 16.04              // [g / mol]
// PWV_H2O is taken from the header for ESPRESSO and equal to 1.0 otherwise
#define PWV_CH4 2163379.0           // [cm]
#define PWV_CO2 951782.0            // [cm]
#define PWV_O2 660128.0             // [cm]

#define MODEL_RESOLUTION 3000000.0
#define Q_T_296K 174.5813

#define RV_RANGE_TELLURICS 40.0
#define HITRAN_LINE_PROFILE_WIDTH 3000

#define GSL_NAN_NUMBER -9999.9
#define FIT_ITERATIONS_MAX_NB   50

/* Columns of the HITRAN LINES_TABLE */
static const char COL_NAME_WAVE_NUMBER[] = "wave_number";
static const char COL_NAME_INTENSITY[] = "Intensity";
static const char COL_NAME_GAMMA_AIR[] = "gamma_air";
static const char COL_NAME_GAMMA_SELF[] = "gamma_self";
static const char COL_NAME_N_AIR[] = "N";
static const char COL_NAME_DELTA_AIR[] = "delta";
static const char COL_NAME_EPP[] = "Epp";
static const char COL_NAME_TEMP[] = "Temperature";
static const char COL_NAME_QT[] = "Qt";
static const char COL_NAME_CCF_WAVE_NUMBER[] = "CCF_lines_position_wavenumber";
static const char COL_NAME_CCF_WAVELENGTH[] = "CCF_lines_position_wavelength";
static const char COL_NAME_CCF_LAMBDA[] = "lambda";
static const char COL_NAME_CCF_CONTRAST[] = "contrast";


/*----------------------------------------------------------------------------
 Data structures
 ----------------------------------------------------------------------------*/

/* Data structure for HITRAN LINES TABLE */
typedef struct {
    char molecule[5];
    double wave_number;
    double intensity;
    double gamma_air;
    double gamma_self;
    double n_air;
    double delta_air;
    double Epp;
} espdr_hitran_line;

/* Data structure for QT TABLE */
typedef struct {
    char molecule[5];
    double Temp;
    double Qt;
} espdr_qt_temp;

/* Data dtrusture to pass the telluric_fit parameters to the minimizing function */
typedef struct {
    char molecule[5];
    double Temp_offseted;
    double qt_used;
    double airmass;
    double N_mol;
    double M_mol;
    espdr_inst_config *inst_config;
    double **wave;
    double **wavenumber_grid;
    int *wavenumber_len;
    int *selected_orders;
    espdr_hitran_line *hitran_table;
    int hitran_lines_nb;
    cpl_image *res_map;
    cpl_image *wave_matrix;
    cpl_image *dll_matrix;
    cpl_image *s2d_qual;
    cpl_table *telluric_mask;
    double rv_step;
    double berv;
    double berv_max;
    int nx_s2d;
    int ny_s2d;
    int nx_ccf;
    cpl_array *RV_table;
    cpl_image *CCF_flux_norm;
    int save_flag;
    double *CCF_flux_data_RE;
    double *CCF_model_flux_data_RE;
    cpl_image **spectrum_vs_model_RE;
} espdr_fit_params;

/*----------------------------------------------------------------------------
 Functions prototypes
 ----------------------------------------------------------------------------*/

cpl_error_code espdr_correct_tellurics(const cpl_frameset* set,
                                       cpl_frameset* used_frames,
                                       espdr_inst_config *inst_config,
                                       cpl_image *s2d_blaze,
                                       cpl_image *s2d_error,
                                       cpl_image *s2d_qual,
                                       cpl_image *blaze,
                                       cpl_propertylist *keywords,
                                       cpl_image *wave_vacuum_shifted,
                                       cpl_image *dll_matrix,
                                       double rv_step,
                                       double berv,
                                       double berv_max,
                                       cpl_propertylist **tell_keywords_RE,
                                       cpl_image **telluric_spectrum_RE,
                                       cpl_image **s2d_tell_corr_RE,
                                       cpl_image **s2d_err_tell_corr_RE,
                                       cpl_image **s2d_qual_tell_corr_RE);

cpl_error_code espdr_read_hitran_from_frameset(cpl_frameset *hitran_frames,
                                               espdr_hitran_line **hitran_tables,
                                               espdr_qt_temp **hitran_qt,
                                               int *lines_nb_RE,
                                               int *qt_nb_RE,
                                               char **molecule_name,
                                               int qt_read_flag);

cpl_error_code espdr_get_telluric_mask(cpl_frameset *hitran_frames,
                                       cpl_table **mask,
                                       int *lines_nb_RE);

cpl_error_code espdr_create_wavenumber_grid(cpl_image *wave_matrix,
                                            double **wave_RE,
                                            double **wavenumber_RE,
                                            int *wavenumber_len_RE,
                                            double **wavenumber_selected_RE,
                                            int *wavenumber_selected_len_RE);

cpl_error_code espdr_select_orders(double **wavenumber_grid,
                                   int *wavenumber_len,
                                   int ny_s2d,
                                   espdr_hitran_line *hitran_table,
                                   int lines_nb,
                                   int *selected_orders);

double espdr_fit_telluric_model(double Pressure,
                                double PWV_aimass,
                                char *molecule,
                                double Temp_offseted,
                                double qt_used,
                                double airmass,
                                double N_mol,
                                double M_mol,
                                espdr_inst_config *inst_config,
                                double **wave,
                                double **wavenumber_grid,
                                int *wavenumber_len,
                                int *selected_orders,
                                espdr_hitran_line *hitran_table,
                                int hitran_lines_nb,
                                cpl_image *res_map,
                                cpl_image *wave_matrix,
                                cpl_image *dll_matrix,
                                cpl_image *s2d_qual,
                                cpl_table *telluric_mask,
                                double rv_step,
                                double berv,
                                double berv_max,
                                int nx_s2d,
                                int ny_s2d,
                                int nx_ccf_in,
                                cpl_array *RV_table,
                                cpl_image *CCF_flux,
                                int save_flag,
                                double *CCF_flux_data_RE,
                                double *CCF_model_flux_data_RE,
                                cpl_image **spectrum_vs_model_RE);

int espdr_telluric_fit_least_squares(const gsl_vector *v,
                                     void *params,
                                     gsl_vector *f);

void callback(const size_t iter,
              void *params,
              const gsl_multifit_nlinear_workspace *w);

cpl_error_code espdr_compute_telluric_model(double *Pressure,
                                            double *PWV_molecule,
                                            double *Temp_offseted,
                                            double *qt_used,
                                            double airmass,
                                            double *N_mol,
                                            double *M_mol,
                                            int mol_nb,
                                            espdr_inst_config *inst_config,
                                            double **wave,
                                            double **wavenumber_grid,
                                            int *wavenumber_len,
                                            int **selected_orders,
                                            int *selected_orders_all,
                                            espdr_hitran_line **hitran_table,
                                            int *hitran_lines_nb,
                                            cpl_image *res_map,
                                            cpl_image *wave_matrix,
                                            cpl_image *s2d_qual,
                                            int nx_s2d,
                                            int ny_s2d,
                                            int save_flag,
                                            cpl_image **telluric_spectrum_RE);

cpl_error_code espdr_fill_header(cpl_propertylist **tell_keywords,
                                 espdr_inst_config *inst_config,
                                 int mol_nb,
                                 char **molecule_name,
                                 double *PWV_molecule,
                                 double *PWV_molecule_err,
                                 double *pressure_atm_molecule,
                                 double *pressure_atm_molecule_err,
                                 double *Temp_offseted,
                                 int *iterations_nb,
                                 double *chi2_final);

#endif /* ESPDR_TELLURICS_H */


