/*                                                                            *
 *   This file is part of the ESPRESSO Pipeline                               *
 *   Copyright (C) 2006 European Southern Observatory                         *
 *                                                                            *
 *   This library is free software; you can redistribute it and/or modify     *
 *   it under the terms of the GNU General Public License as published by     *
 *   the Free Software Foundation; either version 2 of the License, or        *
 *   (at your option) any later version.                                      *
 *                                                                            *
 *   This program is distributed in the hope that it will be useful,          *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 *   GNU General Public License for more details.                             *
 *                                                                            *
 *   You should have received a copy of the GNU General Public License        *
 *   along with this program; if not, write to the Free Software              *
 *   Foundation, 51 Franklin St, Fifth Floor, Boston, MA  02111-1307  USA     *
 *                                                                            */

/*
 */


#include <espdr_wave_LFC_LFC_cal.h>
#include <espdr_overscan.h>

/*----------------------------------------------------------------------------
 Functions code
 ----------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/**
 @brief    Interpret the command line options and execute the data processing
 @param    parameters     the parameters list
 @param    frameset   the frames list

 In case of failure the cpl_error_code is set.
 */
/*---------------------------------------------------------------------------*/

int espdr_wave_LFC_LFC(cpl_parameterlist *parameters, cpl_frameset *frameset,
		const char* recipe_id) {

    cpl_frameset *wave_frames_to_reduce = NULL;
    cpl_frame *wave_frame = NULL;
    cpl_frameset *used_frames = NULL;

	cpl_error_code my_error = CPL_ERROR_NONE;

	cpl_msg_set_level(CPL_MSG_INFO);

	espdr_msg("Starting wave LFC LFC");

    /* Identify the RAW and CALIB frames in the input frameset */
    espdr_ensure(espdr_dfs_set_groups(frameset) != CPL_ERROR_NONE,
                 cpl_error_get_code(),
                 "DFS setting groups failed. Expected inputs are:\n"
                 "raw wave LFC_LFC image, tagged as LFC_LFC\n"
                 "CCD geometry table, tagged as CCD_GEOM\n"
                 "instrument configuration table, tagged as MASTER_INST_CONFIG or INST_CONFIG\n"
                 "hot pixel mask image, tagged as HOT_PIXEL_MASK\n"
                 "bad pixel mask image, tagged as BAD_PIXEL_MASK\n"
                 "orders definition images for fibres A and B, tagged as ORDER_TABLE_A & ORDER_TABLE_B\n"
                 "orders profile images for fibres A and B, tagged as ORDER_PROFILE_A & ORDER_PROFILE_B\n"
                 "flat images for fibres A and B, tagged as FLAT_A & FLAT_B\n"
                 "blaze images for fibres A and B, tagged as BLAZE_A & BLAZE_B\n");

    espdr_CCD_geometry *CCD_geom            = NULL;
    espdr_inst_config *inst_config          = NULL;
    espdr_qc_keywords *qc_kws               = NULL;
    espdr_OVSC_param *OVSC_param            = NULL;
    espdr_WAVE_LFC_LFC_param *WAVE_LFC_LFC_param  = NULL;
    cpl_frame* CCD_geom_frame               = NULL;
    cpl_frame* inst_config_frame            = NULL;

    CCD_geom_frame    = espdr_frame_find(frameset, ESPDR_CCD_GEOM);
    espdr_ensure(CCD_geom_frame == NULL,cpl_error_get_code(),
                 "CCD geometry frame not found!");
    CCD_geom          = espdr_CCD_geom_init(parameters, CCD_geom_frame);
    inst_config_frame = espdr_get_inst_config(frameset);
    espdr_ensure(inst_config_frame == NULL,cpl_error_get_code(),
                 "Instrument config frame not found!");
    inst_config       = espdr_inst_config_init(parameters,
                                               CCD_geom->ext_nb,
                                               inst_config_frame);
    OVSC_param        = espdr_parameters_OVSC_init(recipe_id, parameters);
    WAVE_LFC_LFC_param   = espdr_WAVE_LFC_LFC_param_init(recipe_id, parameters);

    used_frames = cpl_frameset_new();
    my_error = cpl_frameset_insert(used_frames, cpl_frame_duplicate(CCD_geom_frame));
    my_error = cpl_frameset_insert(used_frames, cpl_frame_duplicate(inst_config_frame));

    /* Filling up the DRS QC KWs structure */
    qc_kws = (espdr_qc_keywords *)cpl_malloc(sizeof(espdr_qc_keywords));
    my_error = espdr_fill_qc_keywords(inst_config, qc_kws);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Filling QC KWs failed: %s",
                 cpl_error_get_message_default(my_error));
    espdr_ensure(qc_kws == NULL, CPL_ERROR_ILLEGAL_OUTPUT,
                 "QC KWs structure is NULL");

    wave_frame = espdr_frame_find(frameset, ESPDR_WAVE_LFC_LFC_RAW);
    espdr_ensure(wave_frame == NULL, CPL_ERROR_NULL_INPUT,
                 "wave (LFC_LFC) frame not found!");
    my_error = cpl_frameset_insert(used_frames, cpl_frame_duplicate(wave_frame));
    const char *WAVE_tag = cpl_frame_get_tag(wave_frame);
    wave_frames_to_reduce = cpl_frameset_new();
    cpl_frameset_insert(wave_frames_to_reduce, cpl_frame_duplicate(wave_frame));
    espdr_msg("WAVE_tag: %s", WAVE_tag);

    espdr_src_type fibre_source[inst_config->fibres_nb];
    char **fibre_WAVE = (char **)cpl_malloc(inst_config->fibres_nb * sizeof(char *));
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        fibre_WAVE[i] = (char *)cpl_malloc(TAG_LENGTH*sizeof(char));
    }
    my_error = espdr_get_fibre_source(WAVE_tag, fibre_source, fibre_WAVE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_get_fibre_source failed: %s",
                 cpl_error_get_message_default(my_error));
    espdr_msg("fibre_WAVE: %s & %s", fibre_WAVE[0], fibre_WAVE[1]);

    int *orders_nb_per_fibre = (int *)cpl_calloc(inst_config->fibres_nb, sizeof(int));
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        orders_nb_per_fibre[i] = 0;
        for (int j = 0; j < CCD_geom->ext_nb; j++) {
            orders_nb_per_fibre[i] += inst_config->orders_nb[i*CCD_geom->ext_nb+j];
        }
    }

    cpl_image **pixel_geom_image = (cpl_image **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_image *));
    cpl_image **pixel_size_image = (cpl_image **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_image *));
    cpl_image **flat_corr_spectrum = (cpl_image **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_image *));
    cpl_image **flat_corr_error = (cpl_image **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_image *));
    cpl_image **flat_corr_qual = (cpl_image **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_image *));
    cpl_image **flat_blaze_corr_spectrum = (cpl_image **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_image *));
    cpl_image **flat_blaze_corr_error = (cpl_image **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_image *));
    cpl_image **flat_blaze_corr_qual = (cpl_image **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_image *));
    cpl_propertylist **keywords_fibre = (cpl_propertylist **)cpl_malloc(inst_config->fibres_nb*sizeof(cpl_propertylist *));
    cpl_imagelist *CCD_corrected_image = cpl_imagelist_new();
    cpl_table ***orders_coeffs = NULL;
    double *RON = (double *)cpl_calloc(CCD_geom->total_output_nb, sizeof(double));
    my_error = espdr_process_wave_till_extraction(frameset, parameters, recipe_id,
                                                  wave_frames_to_reduce,
                                                  wave_frame,
                                                  used_frames,
                                                  inst_config,
                                                  CCD_geom,
                                                  qc_kws,
                                                  orders_nb_per_fibre,
                                                  fibre_source,
                                                  fibre_WAVE,
                                                  keywords_fibre,
                                                  CCD_corrected_image,
                                                  &orders_coeffs,
                                                  RON,
                                                  flat_corr_spectrum,
                                                  flat_corr_error,
                                                  flat_corr_qual,
                                                  flat_blaze_corr_spectrum,
                                                  flat_blaze_corr_error,
                                                  flat_blaze_corr_qual,
                                                  pixel_geom_image,
                                                  pixel_size_image);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_process_wave_till_extraction failed: %s",
                 cpl_error_get_message_default(my_error));

    espdr_msg("Saving products");
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        my_error = espdr_save_wave_S2D_products(frameset, parameters, recipe_id,
                                                used_frames, "LFC_LFC", i,
                                                keywords_fibre[i],
                                                inst_config,
                                                flat_corr_spectrum[i],
                                                flat_corr_error[i],
                                                flat_corr_qual[i],
                                                flat_blaze_corr_spectrum[i],
                                                flat_blaze_corr_error[i],
                                                flat_blaze_corr_qual[i]);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_save_wave_S2D_products failed: %s for fibre %c",
                     cpl_error_get_message_default(my_error), fibre_name[i]);
    }

    espdr_msg("Cleaning the memory");

	// Cleaning memory.
    espdr_msg("Cleaning keywords lists");
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        cpl_propertylist_delete(keywords_fibre[i]);
    }
    cpl_free(keywords_fibre);

	cpl_free(qc_kws);

    espdr_msg("Cleaning calibration products");
	cpl_imagelist_delete(CCD_corrected_image);

    espdr_msg("Cleaning products");
    for (int i = 0; i < inst_config->fibres_nb; i++) {
        cpl_image_delete(flat_corr_spectrum[i]);
        cpl_image_delete(flat_corr_error[i]);
        cpl_image_delete(flat_corr_qual[i]);
    }
    cpl_free(flat_corr_spectrum);
    cpl_free(flat_corr_error);
    cpl_free(flat_corr_qual);

    for (int i = 0; i < inst_config->fibres_nb; i++) {
        cpl_image_delete(flat_blaze_corr_spectrum[i]);
        cpl_image_delete(flat_blaze_corr_error[i]);
        cpl_image_delete(flat_blaze_corr_qual[i]);
    }
    cpl_free(flat_blaze_corr_spectrum);
    cpl_free(flat_blaze_corr_error);
    cpl_free(flat_blaze_corr_qual);

    espdr_msg("Cleaning global variables");
    cpl_free(RON);
    cpl_free(orders_nb_per_fibre);

    espdr_msg("Cleaning parameters");
    espdr_parameters_OVSC_delete(OVSC_param);
    espdr_parameters_WAVE_LFC_LFC_delete(WAVE_LFC_LFC_param);
    espdr_parameters_CCD_geometry_delete(CCD_geom);
    espdr_parameters_inst_config_delete(inst_config);

    espdr_msg("End of the recipe espdr_wave");
	return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Create the structure for WAVE input parameters
 @param    recipe_id	recipe identifier
 @param    list			list of input parameters
 @param    p			input WAVE parameters structure
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_parameters_WAVE_LFC_LFC_create(const char* recipe_id,
                                                 cpl_parameterlist *list,
                                                 espdr_WAVE_LFC_LFC_param *p) {
    espdr_ensure(recipe_id == NULL,
                 CPL_ERROR_NULL_INPUT, "The recipe ID is NULL");
    
	/* check parameters */
	espdr_ensure(list == NULL, CPL_ERROR_NULL_INPUT, "The parameters list NULL");
	
	/* Fill the parameter list */
    p = NULL;
    
    return(CPL_ERROR_NONE);

}

/*----------------------------------------------------------------------------*/
/**
 @brief    Delete the structure for ORDERDEF input parameters
 @param    p	input ORDERDEF parameters structure
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_parameters_WAVE_LFC_LFC_delete(espdr_WAVE_LFC_LFC_param* p) {
	
    cpl_free(p);
    p = NULL;
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief     Get the WAVE recipe parameters
 @param     recipe_id   recipe ID
 @param     param_list  parameters list
 @param     WAVE_param  WAVE parameters structure
 @return    CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_parameters_WAVE_LFC_LFC_get(const char* recipe_id,
                                              cpl_parameterlist* param_list,
                                              espdr_WAVE_LFC_LFC_param *WAVE_LFC_LFC_param) {
	
    espdr_ensure(recipe_id == NULL,
                 CPL_ERROR_NULL_INPUT, "The recipe ID is NULL");
    
	/* check parameters */
	espdr_ensure(param_list == NULL, CPL_ERROR_NULL_INPUT,
				 "Parameters list is NULL");
	
	/* Fill the structure */
    WAVE_LFC_LFC_param = NULL;
    
	return cpl_error_get_code();
}


/*---------------------------------------------------------------------------*/
/**
 @brief     print the WAVE parameters
 @param     WAVE_param  WAVE parameters structure
 @return    CPL_ERROR_NONE iff OK
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_parameters_WAVE_LFC_LFC_print(espdr_WAVE_LFC_LFC_param *WAVE_LFC_LFC_param) {
	
	espdr_msg("\tWAVE parameters: NONE");
    WAVE_LFC_LFC_param = NULL;
    
	return (cpl_error_get_code());
}


/*----------------------------------------------------------------------------*/
/**
 @brief    Init WAVE_param structure
 
 @param    recipe_id    recipe name
 @param    parameters   recipe input parameters
 
 @return   allocated structure iff OK
 */
/*----------------------------------------------------------------------------*/

espdr_WAVE_LFC_LFC_param *espdr_WAVE_LFC_LFC_param_init(const char *recipe_id,
                                             cpl_parameterlist *parameters) {
    
    cpl_error_code my_error;
    
    espdr_WAVE_LFC_LFC_param *WAVE_LFC_LFC_param =
                (espdr_WAVE_LFC_LFC_param *)cpl_malloc(sizeof(espdr_WAVE_LFC_LFC_param));
    
    /* Read the wave parameters */
    my_error = espdr_parameters_WAVE_LFC_LFC_get(recipe_id, parameters, WAVE_LFC_LFC_param);
    my_error = espdr_parameters_WAVE_LFC_LFC_print(WAVE_LFC_LFC_param);
    //if(cpl_error_get_code() != CPL_ERROR_NONE) {
    if(my_error != CPL_ERROR_NONE) {
        return NULL;
    } else {
        return WAVE_LFC_LFC_param;
    }
}

#if 0
/*----------------------------------------------------------------------------*/
/**
 @brief    Read the wave recipe parameters
 @param         parameters   recipe input parameters
 @param         frameset     recipe input frames
 @param         recipe_id    recipe name
 @param[out]    CCD_geom     CCD geometry structure
 @param[out]    inst_config  instrument config
 @param[out]    OVSC_param   overscan parameter structure
 @param[out]    WAVE_param   WAVE parameter structure
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_wave_LFC_LFC_read_parameters(cpl_frameset *frameset,
                                               espdr_CCD_geometry *CCD_geom,
                                               espdr_inst_config *inst_config) {
	
	cpl_frameset *CCD_geom_frames = cpl_frameset_new();
	cpl_frame *CCD_geom_table_frame = NULL;
	const char *CCD_geom_filename = NULL;
    cpl_frameset *inst_config_frames = cpl_frameset_new();
    cpl_frame *inst_config_table_frame = NULL;
    const char *inst_config_filename = NULL;
	cpl_error_code my_error;
	
	/* Extract and read the CCD geometry FITS table */
	my_error = espdr_frame_extract_by_tag(frameset, ESPDR_CCD_GEOM,
                                          CCD_geom_frames);
	espdr_ensure(my_error != CPL_ERROR_NONE, my_error, 
			   "CCD_GEOM parameters frame extraction failed");
	
	espdr_ensure(CCD_geom_frames == NULL, CPL_ERROR_NULL_INPUT, 
			   "No CCD geometry parameters table %s, exiting", 
			   cpl_error_get_message());
	
    /* Extract and read the instrument configuration FITS table */
    my_error = espdr_frame_extract_by_tag(frameset, ESPDR_INST_CONFIG,
                                          inst_config_frames);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Instrument parameters frame extraction failed");
    
    espdr_ensure(inst_config_frames == NULL, CPL_ERROR_NULL_INPUT,
                 "No instrument parameters table %s, exiting",
                 cpl_error_get_message());
    
    /* Read the CCD geometry configuration file provided in SOF */
    CCD_geom_table_frame = cpl_frameset_get_position(CCD_geom_frames, 0);
    CCD_geom_filename = cpl_frame_get_filename(CCD_geom_table_frame);
    my_error = espdr_read_CCD_parameters(CCD_geom_filename, CCD_geom);
    espdr_ensure(my_error != CPL_ERROR_NONE, CPL_ERROR_INCOMPATIBLE_INPUT,
                 "Error getting CCD geometry: %s",
                 cpl_error_get_message_default(my_error));
    
    /* Read the instrument configuration file provided in SOF */
    inst_config_table_frame = cpl_frameset_get_position(inst_config_frames, 0);
    inst_config_filename = cpl_frame_get_filename(inst_config_table_frame);
    my_error = espdr_read_inst_parameters(inst_config_filename,
                                          CCD_geom->ext_nb,
                                          inst_config);
    espdr_ensure(my_error != CPL_ERROR_NONE, CPL_ERROR_INCOMPATIBLE_INPUT,
                 "Error getting instrument configuration: %s",
                 cpl_error_get_message_default(my_error));
    
	cpl_frameset_delete(CCD_geom_frames);
    cpl_frameset_delete(inst_config_frames);
	
	return cpl_error_get_code();
}
#endif

