/*-----------------------------------------------------------------*/
/*!
  \file earth_orientation_parameters.c
  \brief read and interpolate the Earth Orientation Parameters

  \author  H. Manche
           Astronomie et Systemes Dynamiques, IMCCE, CNRS, Observatoire de Paris.

   Copyright 2014, 2015, 2016, CNRS
   email of the author : herve.manche@obspm.fr

  last revision: 05/06/2014

  History:
  \note H. Manche 05/06/2014 : creation
 */
/*-----------------------------------------------------------------*/

/*-----------------------------------------------------------------*/
/* License  of this file :
 This file is "triple-licensed", you have to choose one  of the three licenses
 below to apply on this file.

    CeCILL-C
    	The CeCILL-C license is close to the GNU LGPL.
    	( http://www.cecill.info/licences/Licence_CeCILL-C_V1-en.html )

 or CeCILL-B
        The CeCILL-B license is close to the BSD.
        (http://www.cecill.info/licences/Licence_CeCILL-B_V1-en.txt)

 or CeCILL v2.0
      The CeCILL license is compatible with the GNU GPL.
      ( http://www.cecill.info/licences/Licence_CeCILL_V2-en.html )


This library is governed by the CeCILL-C, CeCILL-B or the CeCILL license under
French law and abiding by the rules of distribution of free software.
You can  use, modify and/ or redistribute the software under the terms
of the CeCILL-C,CeCILL-B or CeCILL license as circulated by CEA, CNRS and INRIA
at the following URL "http://www.cecill.info".

As a counterpart to the access to the source code and  rights to copy,
modify and redistribute granted by the license, users are provided only
with a limited warranty  and the software's author,  the holder of the
economic rights,  and the successive licensors  have only  limited
liability.

In this respect, the user's attention is drawn to the risks associated
with loading,  using,  modifying and/or developing or reproducing the
software by the user in light of its specific status of free software,
that may mean  that it is complicated to manipulate,  and  that  also
therefore means  that it is reserved for developers  and  experienced
professionals having in-depth computer knowledge. Users are therefore
encouraged to load and test the software's suitability as regards their
requirements in conditions enabling the security of their systems and/or
data to be ensured and,  more generally, to use and operate it in the
same conditions as regards security.

The fact that you are presently reading this means that you have had
knowledge of the CeCILL-C,CeCILL-B or CeCILL license and that you accept its terms.
*/
/*-----------------------------------------------------------------*/



#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "earth_orientation_parameters.h"
#include "erfa.h"
#include "erfam.h"
#include "errors.h"
#include "polynomial_interpolation.h"

#define MaxLinesNumber 50000


/*!
  Read all the EOP from a file

  @param FileName (in): file name where EOPs are stored
  @param EOPvalues (out): array of values of all the Earth Orientation Parameters
 */
void extract_EOP_FromFile(const char *FileName, t_ArraysAllEOPs *ptr_EOPvalues )
{

  FILE *file ;
  int year, month, day, MJD ;       /* fields read from a line in EOP file */
  double xp,yp, UT1mUTC,LoD,dX,dY ; /* fields read from a line in EOP file */
  int nLines ; /* number of line containing information */
  const int nArg=10 ; /* number of argument read from a line */
  char StringRead[200] ; /*line of a file converted into a string of caracters */
  double TAI1,TAI2 ; /* integer and decimal parts of a julian day TAI */
  int status ; /* variable for output status of SOFA/ERFA routines */
  double TAImUTC ; /* difference TAI-UTC */



  int tab_MJD_UTC[MaxLinesNumber] ;
  double tab_UT1mUTC[MaxLinesNumber] ;
  double tab_xp[MaxLinesNumber],tab_yp[MaxLinesNumber] ;
  double tab_dX[MaxLinesNumber],tab_dY[MaxLinesNumber] ;

  double tab_JulianDay_TT1[MaxLinesNumber] ;
  double tab_JulianDay_TT2[MaxLinesNumber] ;
  double tab_UT1mTT[MaxLinesNumber] ;
  int iLine ;

  /* ERFA_DASR is defined in erfam.h ( conversion from arcsec to radian) */


  /* Checks if the file exists */
  if (  (file = fopen(FileName,"r"))==NULL  )
  {
    printf("Error: cannot open file '%s'\n",FileName);
    FatalError("Can't open the file of EOPs",__FILE__,__LINE__);
  }

  /* read the file and store the values in a local temporary array */
  /* computes the derived fields */
  nLines=0 ;
  while (fgets(StringRead, sizeof StringRead, file) != NULL)
  {
    if ( sscanf(StringRead,"%d %d %d %d %lf %lf %lf %lf %lf %lf",&year,&month,&day,&MJD,
                &xp,&yp,&UT1mUTC,&LoD,&dX,&dY)==nArg )
    {
      /* fields directly available in EOP file */
      tab_MJD_UTC[nLines] = MJD ;
      tab_xp[nLines] = xp*ERFA_DAS2R ; /* conversion from arcsec to radian */
      tab_yp[nLines] = yp*ERFA_DAS2R ;
      tab_UT1mUTC[nLines] = UT1mUTC ;
      tab_dX[nLines] = dX*ERFA_DAS2R ;
      tab_dY[nLines] = dY*ERFA_DAS2R ;
      /* fields derived from previous ones */
      status = eraUtctai( (double) (MJD+2400000), 0.5 , &TAI1, &TAI2 ) ;
      if ( status !=0 ) FatalError("Problem in transformation UTC->TAI performed by sofa/erfa",__FILE__,__LINE__);
      status = eraTaitt( TAI1, TAI2 , &tab_JulianDay_TT1[nLines],
                         &tab_JulianDay_TT2[nLines]  ) ;
      if ( status !=0 ) FatalError("Problem in transformation TAI->TT performed by sofa/erfa",__FILE__,__LINE__);
      status = eraDat    ( year, month, day, 0.0, &TAImUTC );
      if ( status !=0 ) FatalError("Problem in sofa/erfa computation of TAI-UTC",__FILE__,__LINE__);
      if ( status == 1  ) WarningError("Problem in sofa/erfa computation of TAI-UTC, check the year (dubious)",__FILE__,__LINE__);
      tab_UT1mTT[nLines] = UT1mUTC-TAImUTC-ERFA_TTMTAI ; /* TTmTAI is defined in erfam.h */
      nLines +=1 ;
    }
  }
  fclose(file);

  if ( nLines<=0 ) FatalError("no EOP have been extracted from the file",__FILE__,__LINE__);

  /* allocation of arrays */
  ptr_EOPvalues->MJD_UTC =(int*)malloc(nLines*sizeof(int));
  ptr_EOPvalues->xp=(double*)malloc(nLines*sizeof(double));
  ptr_EOPvalues->yp=(double*)malloc(nLines*sizeof(double));
  ptr_EOPvalues->UT1mUTC=(double*)malloc(nLines*sizeof(double));
  ptr_EOPvalues->dX=(double*)malloc(nLines*sizeof(double));
  ptr_EOPvalues->dY=(double*)malloc(nLines*sizeof(double));
  ptr_EOPvalues->UT1mTT=(double*)malloc(nLines*sizeof(double));
  ptr_EOPvalues->JulianDay_TT1=(double*)malloc(nLines*sizeof(double));
  ptr_EOPvalues->JulianDay_TT2=(double*)malloc(nLines*sizeof(double));

  /* copy from temporay arrays */
  for (iLine=0  ; iLine<nLines ; iLine++)
  {
    ptr_EOPvalues->MJD_UTC[iLine]=tab_MJD_UTC[iLine] ;
    ptr_EOPvalues->xp[iLine]=tab_xp[iLine] ;
    ptr_EOPvalues->yp[iLine]=tab_yp[iLine] ;
    ptr_EOPvalues->UT1mUTC[iLine]=tab_UT1mUTC[iLine] ;
    ptr_EOPvalues->dX[iLine]=tab_dX[iLine] ;
    ptr_EOPvalues->dY[iLine]=tab_dY[iLine] ;
    ptr_EOPvalues->UT1mTT[iLine]=tab_UT1mTT[iLine] ;
    ptr_EOPvalues->JulianDay_TT1[iLine]=tab_JulianDay_TT1[iLine] ;
    ptr_EOPvalues->JulianDay_TT2[iLine]=tab_JulianDay_TT2[iLine] ;
  }
  ptr_EOPvalues->N_EOP =nLines ;
  ptr_EOPvalues->ValMinTT = (tab_JulianDay_TT1[0]-2451545.0)+tab_JulianDay_TT2[0]  ;
  ptr_EOPvalues->ValMaxTT = (tab_JulianDay_TT1[nLines-1]-2451545.0)+tab_JulianDay_TT2[nLines-1]  ;
}


/*!
  Interpolation of EOPs from a tabulated serie

  @param JulianDay_TT (in): Julian Day (in TT) where EOPs are needed
  @param DegreeInterpolation (in): degree of the polynomial interpolation
  @param ActiveEOPs (in): define which EOP are taken into account (according binary decomposition)
  @param TabAllEOP (in): tabulated EOPs (from IERS C04 serie)
  @param EOPvalues (out): structure containing all EOPs values at JulianDay_TT
  @todo: implementer le degre pair (nombre impair de points)
 */

void interpol_EOP(const double JulianDay_TT[2], const int DegreeInterpolation,
                  const int ActiveEOPs,
                  const t_ArraysAllEOPs TabAllEOP,
                  EarthOrientationParameters_t *EOPvalue)
{
  int N1;
  double TT2000 ; /* time from J2000 in days */

  double vecX[DegreeInterpolation+1] ;
  int i ;
  /*double ERFA_DJ00 is defined in erfam.h (J2000.0) */


  TT2000 = (JulianDay_TT[0]-ERFA_DJ00) +JulianDay_TT[1]  ;


  /* Checks if JulianDay_TT is inside the timespan where EOPs are available */
  if ( TT2000<TabAllEOP.ValMinTT) FatalError("TT2000 is outside the interval of EOPs",__FILE__,__LINE__);
  if ( TT2000>TabAllEOP.ValMaxTT) FatalError("TT2000 is outside the interval of EOPs",__FILE__,__LINE__) ;

  /* Since julian days TT (where EOP are given) are "regularly" (but not exactly) spaced,
   * linear approximation is performed to find the time interval for interpolation */

  /* search N1 and N2 */
  if (DegreeInterpolation % 2==0)
  {
    /* search N1 so that abs(JulianDay_EOP(N1)-JulianDay_TT) is minimum */
    N1= (int) ( (TabAllEOP.N_EOP-1)*(TT2000-TabAllEOP.ValMinTT)/(TabAllEOP.ValMaxTT-TabAllEOP.ValMinTT) +0.5) ;
    while (  (JulianDay_TT[0]-TabAllEOP.JulianDay_TT1[N1])
            +(JulianDay_TT[1]-TabAllEOP.JulianDay_TT2[N1]) <-0.5 )
    {
      N1-- ;
    }
    N1 -= DegreeInterpolation/2 ;
  }
  else
  {
    /* search N1 so that JulianDay_EOP(N1)<=JulianDay_TT<JulianDay_EOP(N1+1)  */
    N1= (int) ( (TabAllEOP.N_EOP-1)*(TT2000-TabAllEOP.ValMinTT)/(TabAllEOP.ValMaxTT-TabAllEOP.ValMinTT) ) ;
    while (  (JulianDay_TT[0]-TabAllEOP.JulianDay_TT1[N1+1])
            +(JulianDay_TT[1]-TabAllEOP.JulianDay_TT2[N1+1]) >0.0 ) /* seems to be useless */
    {
      N1++ ;
    }
    while (  (JulianDay_TT[0]-TabAllEOP.JulianDay_TT1[N1])
            +(JulianDay_TT[1]-TabAllEOP.JulianDay_TT2[N1]) <0.0 ) /* is sometimes usefull */
    {
      N1-- ;
    }
    N1 -= (DegreeInterpolation-1)/2 ;
  };

  if (N1<0) FatalError("No EOPs available (date too old)",__FILE__,__LINE__);
  if (N1+DegreeInterpolation>TabAllEOP.N_EOP-1) FatalError("No EOPs available (date too recent)",__FILE__,__LINE__);

  /* build vecX vector */
  for (i=0 ; i<=DegreeInterpolation ; i++)
  {
    vecX[i] =( TabAllEOP.JulianDay_TT1[N1+i]-JulianDay_TT[0] )+TabAllEOP.JulianDay_TT2[N1+i] ;
  }

  /* Interpolation of xp, yp, UT1mTT, dx and dY */
  if ( ActiveEOPs & 0x01 )
  {
    EOPvalue->xp = LagrangianInterpolation( DegreeInterpolation+1 , vecX,
                   &TabAllEOP.xp[N1],JulianDay_TT[1] ) ;
    EOPvalue->yp = LagrangianInterpolation( DegreeInterpolation+1 , vecX,
                   &TabAllEOP.yp[N1],JulianDay_TT[1] ) ;
  }
  if ( ActiveEOPs & 0x02 )
  {
    EOPvalue->UT1mTT = LagrangianInterpolation( DegreeInterpolation+1 , vecX,
                   &TabAllEOP.UT1mTT[N1],JulianDay_TT[1] ) ;
  }
  if ( ActiveEOPs & 0x04 )
  {
    EOPvalue->dX = LagrangianInterpolation( DegreeInterpolation+1 , vecX,
                   &TabAllEOP.dX[N1],JulianDay_TT[1] ) ;
    EOPvalue->dY = LagrangianInterpolation( DegreeInterpolation+1 , vecX,
                   &TabAllEOP.dY[N1],JulianDay_TT[1] ) ;
  }
  /* At this step, EOPs should be corrected by ocean tides and librations effects */
  /* They are neglicted, their effect being small (respectively 3 cm and 2 mm) */
}
