from .hawki_utils import HawkiSetupInfo, HawkiReportMixin
from adari_core.data_libs.master_std_star_img import MasterStdStarImg
from adari_core.plots.text import TextPlot

class HawkiStdStarReport(HawkiReportMixin, MasterStdStarImg):
    """HAWK-I hawki_standard_process recipe report class.

    Attributes
    ----------
    - files_needed: dictionary contaning the categories needed and the file names."""

    detectors = {
        "CHIP1.INT1": [1, 1],
        "CHIP2.INT1": [2, 1],
        "CHIP3.INT1": [2, 2],
        "CHIP4.INT1": [1, 2],
    }  # dictionary values indicate typical (x,y) mosaic position
    files_needed = {
        "calib_std": "BASIC_CALIBRATED_STD",
        "matched_phot": "MATCHSTD_PHOTOM",
        "raw_im": "STD",
    }
    exts = detectors.keys()

    def __init__(self):
        super().__init__("hawki_std_star")

    def parse_sof(self):
        """
        Returns a list of files selected from a set of frames (sof).

        If more than one file fullfills the criteria, the first file
        in the array will be selected.
        """
        file_path, files_category = (
            [elem[0] for elem in self.inputs],
            [elem[1] for elem in self.inputs],
        )
        file_lists = {}
        for required_file in self.files_needed.keys():
            # Check that category matches the requirement
            if self.files_needed[required_file] in files_category:
                file_lists[required_file] = file_path[
                    files_category.index(self.files_needed[required_file])
                ]
            else:
                raise IOError(
                    "[WARNING] {} file not found".format(
                        self.files_needed[required_file]
                    )
                )
        return [file_lists]

    def get_standards(self, data):
        std_xvals = data["X_coordinate"]
        std_yvals = data["Y_coordinate"]
        return [std_xvals, std_yvals]

    def generate_panels(self, hist_clipping="sigma", hist_n_clipping=4):
        panels = {}
        vspace = 0.3

        new_panels = super().generate_panels(
            im_ext=self.exts,
            raw_ext=list(self.exts),
            marking=False,
            marking_centre=True,
        )

        for i, (panel, panel_descr) in enumerate(new_panels.items()):
            calib_std = self.hdus[0]["calib_std"]
            t2 = TextPlot(columns=1, v_space=vspace, xext=1)
            col2 = HawkiSetupInfo.std(calib_std)
            t2.add_data(col2)
            panel.assign_plot(t2, 2, 0, xext=1)

        panels = {**panels, **new_panels}

        return panels


rep = HawkiStdStarReport()
