from edps import QC1_CALIB, QC0, SCIENCE, CALCHECKER, ReportInput
from edps import task, alternative_association, FilterMode, copy_all

from .hawki_datasources import *
from .hawki_hit import hawki_hit
from .hawki_task_functions import *

__title__ = "HAWK-I workflow"

IDP = "idp"

# --- Processing tasks ----------------------------------------------------------------------------

# --- Compute detector linearity
detlin = (task('detlin')
          .with_recipe('detmon_ir_lg')
          .with_report("hawki_detlin", ReportInput.RECIPE_INPUTS_OUTPUTS)
          .with_report("hawki_rawdisp", ReportInput.RECIPE_INPUTS)
          .with_main_input(raw_detlin)
          .with_meta_targets([QC1_CALIB])
          .build())

# --- Process raw dark calibrations
dark = (task('dark')
        .with_recipe('hawki_dark_combine')
        .with_report("hawki_master_dark", ReportInput.RECIPE_INPUTS_OUTPUTS)
        .with_report("hawki_rawdisp", ReportInput.RECIPE_INPUTS)
        .with_main_input(raw_dark)
        .with_associated_input(reference_dark, min_ret=0)
        .with_associated_input(master_bpm, min_ret=0)
        .with_associated_input(master_conf, min_ret=0)
        .with_meta_targets([QC1_CALIB])
        .build())

# --- Process raw flat field calibrations
flat = (task('flat')
        .with_recipe('hawki_twilight_flat_combine')
        .with_report("hawki_sky_flat", ReportInput.RECIPE_INPUTS_OUTPUTS)
        .with_report("hawki_rawdisp", ReportInput.RECIPE_INPUTS)
        .with_main_input(raw_flat)
        .with_associated_input(dark, [MASTER_DARK])
        .with_associated_input(reference_flat, min_ret=0)
        .with_associated_input(master_bpm, min_ret=0)
        .with_associated_input(master_conf, min_ret=0)
        .with_meta_targets([QC1_CALIB, CALCHECKER])
        .build())

# --- Compute detector noise
detector_noise = (task('detector_noise')
                  .with_recipe('hawki_detector_noise')
                  .with_main_input(detnoise_flat)
                  .with_associated_input(detnoise_flat, min_ret=1, max_ret=1)
                  .with_associated_input(detnoise_dark, min_ret=2, max_ret=2)
                  .with_associated_input(flat, [MASTER_BPM, MASTER_CONF], min_ret=0)
                  .with_input_filter(MASTER_BPM, MASTER_CONF)
                  .build())

# --- Select local catalogs for astrometry. ---
# Local astrometry catalogs are selected by setting the workflow parameter cds_astrom_catalog == 'none'
# The workflow parameter astrom_catalog select which local catalogue to use for astrometry. Options are:
#  '2MASS', 'PPMXL', or 'LOCAL' (user provided catalogue).
# If cds_astrom_catalog is not 'none', then the local catalogues are not passed to the standard and object
# tasks. The corresponding recipes will download the catalogue from the CDS server.
astrometric_catalogs = (alternative_association()
                        .with_associated_input(astrom_2mass, condition=astrom_local_2mass, min_ret=0)
                        .with_associated_input(astrom_ppmxl, condition=astrom_local_ppmxl, min_ret=0)
                        .with_associated_input(astrom_custom, condition=astrom_local_custom, min_ret=0))

# --- Select local catalogs for photometry. ---
# Local photometric catalogs are selected by setting the workflow parameter cds_photom_catalog == 'none'
# The workflow parameter photom_catalog select which local catalogue to use for photometry. Options are:
#  '2MASS', 'PPMXL', or 'LOCAL' (user provided catalogue).
# If cds_photom_catalog is not 'none', then the local catalogues are not passed to the standard and object
# tasks. The corresponding recipes will download the catalogue from the CDS server.
photometric_catalogs = (alternative_association()
                        .with_associated_input(photom_2mass, condition=photom_local_2mass, min_ret=0)
                        .with_associated_input(photom_ppmxl, condition=photom_local_ppmxl, min_ret=0)
                        .with_associated_input(photom_custom, condition=photom_local_custom, min_ret=0))

readgain_calibrations = (alternative_association()
                         .with_associated_input(master_readgain, condition=readgain_from_static)
                         .with_associated_input(detector_noise, [MASTER_READGAIN], condition=readgain_from_raw_calib,
                                                min_ret=0))

# --- Reduce standard star raw calibrations
standard = (task('standard')
            .with_recipe('hawki_standard_process')
            .with_report("hawki_std_star", ReportInput.RECIPE_INPUTS_OUTPUTS)
            .with_report("hawki_rawdisp", ReportInput.RECIPE_INPUTS)
            .with_main_input(raw_standard)
            .with_job_processing(catalogs)
            .with_associated_input(dark, [MASTER_DARK])
            .with_associated_input(flat, [MASTER_FLAT, MASTER_CONF])
            .with_alternatives(readgain_calibrations)
            .with_alternatives(astrometric_catalogs)
            .with_alternatives(photometric_catalogs)
            .with_associated_input(schlegel_map_north)
            .with_associated_input(schlegel_map_south)
            .with_associated_input(photcal_tab)
            .with_meta_targets([QC1_CALIB, CALCHECKER])
            .build())

# Filtering products of the standard task. I cannot filter them in the
# object or object_postprocess tasks because some share the same PRO.CATG than
# those produced by the object task.
filter_standard_products = (task("filter_standard_products")
                            .with_main_input(standard)
                            .with_function(copy_all)
                            .with_input_filter(MATCHSTD_PHOTOM, mode=FilterMode.SELECT)
                            .build())

# --- Reduce science observations -----------------------------------------------------------------

# --- Science reduction
science = (task('object')
           .with_recipe('hawki_science_process')
           .with_report('hawki_science', ReportInput.RECIPE_INPUTS_OUTPUTS)
           .with_main_input(raw_science)
           .with_job_processing(catalogs)
           .with_dynamic_parameter("which_band", which_band)
           .with_associated_input(raw_sky, min_ret=0)
           .with_associated_input(flat, [MASTER_FLAT, MASTER_CONF])
           .with_associated_input(dark, [MASTER_DARK])
           .with_alternatives(readgain_calibrations)
           .with_alternatives(astrometric_catalogs)
           .with_alternatives(photometric_catalogs)
           .with_associated_input(schlegel_map_north)
           .with_associated_input(schlegel_map_south)
           .with_associated_input(photcal_tab)
           # Association of standard star is done only for CalSelector or to fulfill the night calibration plan, which
           # consider them mandatory only for observations in broad band filters (Y, J, H, and Ks). These calibrations are
           # not used by the recipe hawki_science_process, therefore they are removed from the inputs.
           .with_associated_input(filter_standard_products, [MATCHSTD_PHOTOM], min_ret=0,
                                  condition=is_narrow_and_requested)
           .with_associated_input(filter_standard_products, [MATCHSTD_PHOTOM], min_ret=1,
                                  condition=is_broad_and_requested)
           .with_input_filter(MATCHSTD_PHOTOM, mode=FilterMode.REJECT)
           .with_meta_targets([QC0, SCIENCE, CALCHECKER, IDP])
           .build())

dark_windowed, flat_windowed, science_hit = hawki_hit(flat, astrometric_catalogs, photometric_catalogs)

# --- Science post-processing and combination of frames

object_postprocess = (task('object_postprocess')
                      .with_recipe('hawki_science_postprocess')
                      .with_report('hawki_science', ReportInput.RECIPE_INPUTS_OUTPUTS)
                      .with_main_input(science)
                      .with_job_processing(catalogs)
                      .with_associated_input(flat, [MASTER_FLAT, MASTER_CONF])
                      .with_associated_input(dark, [MASTER_DARK])
                      .with_alternatives(readgain_calibrations)
                      .with_alternatives(astrometric_catalogs)
                      .with_alternatives(photometric_catalogs)
                      .with_associated_input(schlegel_map_north)
                      .with_associated_input(schlegel_map_south)
                      .with_associated_input(photcal_tab)
                      .with_meta_targets([SCIENCE, IDP, CALCHECKER])
                      # Association of standard star is done only for CalSelector or to fulfill the night calibration plan, which
                      # consider them mandatory only for observations in broad band filters (Y, J, H, and Ks). These calibrations
                      # are not supported by hawki_science_postprocess at the moment, therefore they are removed from the inputs.
                      .with_associated_input(filter_standard_products, [MATCHSTD_PHOTOM], min_ret=0,
                                             condition=is_narrow_and_requested)
                      .with_associated_input(filter_standard_products, [MATCHSTD_PHOTOM], min_ret=1,
                                             condition=is_broad_and_requested)
                      .with_input_filter(MATCHSTD_PHOTOM, mode=FilterMode.REJECT)
                      .build())
# -------------------------------------------------------------------------------------------------
