from edps import Job, JobParameters, List, ClassifiedFitsFile

from . import hawki_keywords as kwd


# Function to select the source of MASTER_READGAIN calibrations (static or from raw data)
def readgain_from_raw_calib(params: JobParameters) -> bool:
    return params.get_workflow_param('use_static_readgain') == "FALSE"


def readgain_from_static(params: JobParameters) -> bool:
    return params.get_workflow_param('use_static_readgain') == "TRUE"


# Function to set recipe parameters to select the CDS on-line catalogues to use.
# The choice is expressed in the workflow parameters file
def catalogs(job: Job):
    job.parameters.recipe_parameters[f'hawki.{job.command}.cdssearch_astrom'] = job.parameters.workflow_parameters[
        'cds_astrom_catalog']
    job.parameters.recipe_parameters[f'hawki.{job.command}.cdssearch_photom'] = job.parameters.workflow_parameters[
        'cds_photom_catalog']

    # Temporary removal of "MATCHSTD_PHOTOM" from sof in it is among the associated files.
    # The input filter in the corresponding jobs take care of removing
    # the file if comes from a parent job.
    if job.task_name in ["object", "object_postprocess"]:
        job.associated_files = [f for f in job.associated_files if f.classification != "MATCHSTD_PHOTOM"]


# --- Functions to select the local copies of the catalogues to associate. ------------------

# The function returns True if astrom_catalog = 2MASS and cds_astrom_catalog = 'none'
def astrom_local_2mass(params: JobParameters) -> bool:
    return params.get_workflow_param('astrom_catalog') == '2MASS' \
        and params.get_workflow_param('cds_astrom_catalog') == 'none'


# The function returns True if astrom_catalog = PPMXL and cds_astrom_catalog = 'none'
def astrom_local_ppmxl(params: JobParameters) -> bool:
    return params.get_workflow_param('astrom_catalog') == 'PPMXL' \
        and params.get_workflow_param('cds_astrom_catalog') == 'none'


# The function returns True if astrom_catalog = LOCAL and cds_astrom_catalog = 'none'
def astrom_local_custom(params: JobParameters) -> bool:
    return params.get_workflow_param('astrom_catalog') == 'LOCAL' \
        and params.get_workflow_param('cds_astrom_catalog') == 'none'


# The function returns True if photom_catalog = 2MASS and cds_photom_catalog = 'none'
def photom_local_2mass(params: JobParameters) -> bool:
    return params.get_workflow_param('photom_catalog') == '2MASS' \
        and params.get_workflow_param('cds_photom_catalog') == 'none'


# The function returns True if photom_catalog = PPMXL and cds_photom_catalog = 'none'
def photom_local_ppmxl(params: JobParameters) -> bool:
    return params.get_workflow_param('photom_catalog') == 'PPMXL' \
        and params.get_workflow_param('cds_photom_catalog') == 'none'


# The function returns True if photom_catalog = LOCAL and cds_photom_catalog = 'none'
def photom_local_custom(params: JobParameters) -> bool:
    return params.get_workflow_param('photom_catalog') == 'LOCAL' \
        and params.get_workflow_param('cds_photom_catalog') == 'none'


# This function determines whether the observation is done with a broad or narrow band


def which_band(files: List[ClassifiedFitsFile]):
    # Check whether the observation type is SIM or OBS nd set the return value accordingly
    filter = files[0].get_keyword_value(kwd.ins_filt1_name, None)
    return "BROAD" if filter in ["Y", "J", "H", "Ks"] else "NARROW"


def is_requested(params: JobParameters) -> bool:
    return params.get_workflow_param("associate_night_standard_star") == "TRUE"


def is_broad_and_requested(params: JobParameters) -> bool:
    return params.get_workflow_param("which_band") == "BROAD" and is_requested(params)


def is_narrow_and_requested(params: JobParameters) -> bool:
    return params.get_workflow_param("which_band") == "NARROW" and is_requested(params)


def num_inputs(files: List[ClassifiedFitsFile]) -> int:
    return len(files)


def should_combine(params: JobParameters) -> bool:
    return params.get_workflow_param("is_idp") == "FALSE" or params.get_workflow_param("num_inputs") > 3
