/*
 * This file is part of the KMOS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 *                              Includes
 *----------------------------------------------------------------------------*/

#include <string.h>
#include <math.h>

#include <cpl.h>

#include "kmo_utils.h"
#include "kmos_pfits.h"
#include "kmo_dfs.h"
#include "kmo_error.h"
#include "kmo_constants.h"
#include "kmo_debug.h"
#include "kmos_oscan.h"

/*-----------------------------------------------------------------------------
 *                          Functions prototypes
 *----------------------------------------------------------------------------*/

static char * kmos_get_root_name(const char * filename) ;
static int kmos_level_correct_create(cpl_plugin *);
static int kmos_level_correct_exec(cpl_plugin *);
static int kmos_level_correct_destroy(cpl_plugin *);
static int kmos_level_correct(cpl_parameterlist *, cpl_frameset *);

/*-----------------------------------------------------------------------------
 *                          Static variables
 *----------------------------------------------------------------------------*/

static char kmos_level_correct_description[] =
"This recipe applies the level correction to the input frames.\n"
"\n"
"---------------------------------------------------------------------------\n"
"Input files:\n"
"   DO CATG          Type   Explanation                     Required #Frames\n"
"   -------          -----  -----------                     -------- -------\n"
"   SCIENCE          RAW    Science frames                     Y       1-n  \n"
"   LCAL              F2D    Wavelength calib. frame           N       0,1  \n"
"   BADPIXEL_DARK     B2D    Bad pixels frame                  N       0,1  \n"
"\n"
"Output files:\n"
"   DO CATG          Type   Explanation\n"
"   -------          -----  -----------\n"
"   LEVEL_CORRECTED  F2D    Level corrected frame\n"
"---------------------------------------------------------------------------"
"\n";

/*----------------------------------------------------------------------------*/
/**
 * @defgroup kmos_level_correct     Create level corrected frames
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*-----------------------------------------------------------------------------
 *                              Functions code
 *----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Build the list of available plugins, for this module. 
  @param    list    the plugin list
  @return   0 if everything is ok, -1 otherwise

  Create the recipe instance and make it available to the application using the 
  interface. This function is exported.
 */
/*----------------------------------------------------------------------------*/
int cpl_plugin_get_info(cpl_pluginlist *list)
{
    cpl_recipe *recipe = cpl_calloc(1, sizeof *recipe);
    cpl_plugin *plugin = &recipe->interface;

    cpl_plugin_init(plugin,
            CPL_PLUGIN_API,
            KMOS_BINARY_VERSION,
            CPL_PLUGIN_TYPE_RECIPE,
            "kmos_level_correct",
            "Create level corrected frames",
            kmos_level_correct_description,
            "Yves Jung",
            "https://support.eso.org/",
            kmos_get_license(),
            kmos_level_correct_create,
            kmos_level_correct_exec,
            kmos_level_correct_destroy);

    cpl_pluginlist_append(list, plugin);

    return 0;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Setup the recipe options    
  @param    plugin  the plugin
  @return   0 if everything is ok

  Defining the command-line/configuration parameters for the recipe.
 */
/*----------------------------------------------------------------------------*/
static int kmos_level_correct_create(cpl_plugin *plugin)
{
    cpl_recipe *recipe;
    cpl_parameter *p;

    /* Check that the plugin is part of a valid recipe */
    if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE) 
        recipe = (cpl_recipe *)plugin;
    else
        return -1;

    /* Create the parameters list in the cpl_recipe object */
    recipe->parameters = cpl_parameterlist_new();

    /* Fill the parameters list */

    /* --lcmethod (level correction method) */
    p = cpl_parameter_new_value("kmos.kmos_level_correct.lcmethod", 
            CPL_TYPE_STRING,
            "Method to use for the level correction "
            "[\"OSCAN\" (overscan), "
            "\"SLICES_MEAN\" (intra slices with average), "
            "\"SLICES_MEDIAN\" (intra slices with median)]",
            "kmos.kmos_level_correct", "OSCAN");
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "lcmethod");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(recipe->parameters, p);

    return 0;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Execute the plugin instance given by the interface
  @param    plugin  the plugin
  @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/
static int kmos_level_correct_exec(cpl_plugin *plugin)
{
    cpl_recipe  *recipe;

    /* Get the recipe out of the plugin */
    if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE) 
        recipe = (cpl_recipe *)plugin;
    else return -1;

    return kmos_level_correct(recipe->parameters, recipe->frames);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Destroy what has been created by the 'create' function
  @param    plugin  the plugin
  @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/
static int kmos_level_correct_destroy(cpl_plugin *plugin)
{
    cpl_recipe *recipe;

    /* Get the recipe out of the plugin */
    if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE) 
        recipe = (cpl_recipe *)plugin;
    else return -1 ;

    cpl_parameterlist_delete(recipe->parameters);
    return 0 ;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Interpret the command line options and execute the data processing
  @param    parlist     the parameters list
  @param    frameset   the frames list
  @return   0 if everything is ok

  Possible _cpl_error_code_ set in this function:
  @li CPL_ERROR_ILLEGAL_INPUT   if frames of wrong KMOS format are provided
 */
/*----------------------------------------------------------------------------*/
static int kmos_level_correct(
        cpl_parameterlist   *   parlist, 
        cpl_frameset        *   frameset)
{
    const cpl_parameter *   par ;
    const char          *   lcmethod ;
    int                     i ;
    const cpl_frame     *   rawframe ;
    cpl_frame           *   bpm_frame ;
    cpl_frame           *   lcal_frame ;
    const char          *   in_fn ;
    char                *   out_fn ;
    char                *   bpm_out_fn ;
    char                *   level_out_fn ;
    cpl_image           *   img_in ;
    cpl_image           *   bpm_im ;
    cpl_image           *   lcal_im ;
    cpl_image           *   ima_corr ;
    cpl_image           *   level_out ;
    cpl_mask            *   bpm_out ;
    cpl_propertylist    *   mh_plist ;
    cpl_propertylist    *   plist ;
    double                  rotangle_found, rotangle ;
    
    /* Initialise */
    bpm_frame = lcal_frame = NULL ;

    /* Check initial Entries */
    if (kmos_check_and_set_groups(frameset) != CPL_ERROR_NONE) {
        return cpl_error_get_code();
    }

    /* Get Parameters */
    par = cpl_parameterlist_find_const(parlist, 
            "kmos.kmos_level_correct.lcmethod");
    lcmethod = cpl_parameter_get_string(par) ;

    if (strcmp(lcmethod, "OSCAN") && strcmp(lcmethod, "SLICES_MEAN") &&
            strcmp(lcmethod, "SLICES_MEDIAN")) {
        cpl_msg_error(__func__,
                "lcmethod must be 'OSCAN', 'SLICES_MEAN' or 'SLICES_MEDIAN'") ;
        cpl_error_set(__func__, CPL_ERROR_ILLEGAL_INPUT) ;
        return -1 ;
    }

    /* Get frames */
    bpm_frame = kmo_dfs_get_frame(frameset, BADPIXEL_DARK) ;
    lcal_frame = kmo_dfs_get_frame(frameset, LCAL) ;

    /* Loop on the raw frames */
    rawframe = kmo_dfs_get_frame(frameset, SCIENCE);
    while (rawframe != NULL) {
        /* Output filename */
        in_fn = cpl_frame_get_filename(rawframe) ;
        out_fn = cpl_sprintf("%s_LC.fits", 
                kmos_get_root_name(kmos_get_base_name(in_fn))) ;
        bpm_out_fn = cpl_sprintf("%s_BPM.fits", 
                kmos_get_root_name(kmos_get_base_name(in_fn))) ;
        level_out_fn = cpl_sprintf("%s_LEVEL.fits", 
                kmos_get_root_name(kmos_get_base_name(in_fn))) ;

		/* Create primary header products */
        mh_plist = cpl_propertylist_load(in_fn, 0) ;

        /* BPM_OUT */
        cpl_propertylist_update_string(mh_plist, CPL_DFS_PRO_CATG,
                "BPM_OUT") ;
        cpl_dfs_save_propertylist(frameset, NULL, parlist, frameset,
                rawframe, cpl_func, mh_plist, NULL, VERSION, bpm_out_fn);
        /* LEVEL_OUT */
        cpl_propertylist_update_string(mh_plist, CPL_DFS_PRO_CATG,
                "LEVEL_OUT") ;
        cpl_dfs_save_propertylist(frameset, NULL, parlist, frameset,
                rawframe, cpl_func, mh_plist, NULL, VERSION, level_out_fn);
        /* LEVEL_CORRECTED */
        cpl_propertylist_update_string(mh_plist, CPL_DFS_PRO_CATG,
                "LEVEL_CORRECTED") ;
        cpl_dfs_save_propertylist(frameset, NULL, parlist, frameset,
                rawframe, cpl_func, mh_plist, NULL, VERSION, out_fn);

        /* Loop on detectors */
        for (i = 1; i <= KMOS_NR_DETECTORS ; i++) {
            cpl_msg_info(cpl_func, "Processing det %d of file %s", i, in_fn) ;

            /* Load the input image */
            img_in = cpl_image_load(in_fn, CPL_TYPE_FLOAT, 0, i) ;

            /* Load the extension header */
            plist = cpl_propertylist_load(in_fn, i) ;

            /* Load the BPM / LCAL if passed */
            bpm_im = NULL ;
            if (bpm_frame != NULL) {
                bpm_im = cpl_image_load(cpl_frame_get_filename(bpm_frame), 
                        CPL_TYPE_INT, 0, i) ;
            }
            lcal_im = NULL ;
            if (lcal_frame != NULL) {
                rotangle = kmo_dfs_get_property_double(mh_plist, ROTANGLE);
                lcal_im = kmo_dfs_load_cal_image_frame(lcal_frame, i, 0, 
                        rotangle, FALSE, NULL, &rotangle_found, -1, 0, 0) ;
            }

			/* Apply correction */
            ima_corr = NULL ;
            level_out = NULL ;
            bpm_out = NULL ;
			if (!strcmp(lcmethod, "OSCAN")) {
				cpl_msg_info(__func__, "Apply Overscan correction") ;
				ima_corr = kmos_oscan_correct(img_in) ;
			} else if (!strcmp(lcmethod, "SLICES_MEAN")) {
				cpl_msg_info(__func__, "Apply Intra Slices correction (mean)") ;
				ima_corr = kmos_intraslices_correct(img_in, bpm_im,
                        lcal_im, 1, &bpm_out, &level_out);
			} else if (!strcmp(lcmethod, "SLICES_MEDIAN")) {
				cpl_msg_info(__func__,"Apply Intra Slices correction (median)");
				ima_corr = kmos_intraslices_correct(img_in, bpm_im,
                        lcal_im, 2, &bpm_out, &level_out);
			}
            cpl_image_delete(img_in) ;
            if (lcal_im != NULL) cpl_image_delete(lcal_im) ;
            if (bpm_im != NULL) cpl_image_delete(bpm_im) ;

            /* Save LEVEL_OUT */
            if (level_out == NULL) {
                cpl_propertylist_save(plist, level_out_fn, CPL_IO_EXTEND);
            } else {
                cpl_image_save(level_out, level_out_fn, CPL_TYPE_DOUBLE, plist,
                        CPL_IO_EXTEND);
                cpl_image_delete(level_out) ;
            }

            /* Save BPM_OUT */
            if (bpm_out == NULL) {
                cpl_propertylist_save(plist, bpm_out_fn, CPL_IO_EXTEND);
            } else {
                cpl_mask_save(bpm_out, bpm_out_fn, plist, CPL_IO_EXTEND);
                cpl_mask_delete(bpm_out) ;
            }

            /* Save LEVEL_CORRECTED */
            if (ima_corr == NULL) {
                cpl_propertylist_save(plist, out_fn, CPL_IO_EXTEND);
            } else {
                cpl_image_save(ima_corr, out_fn, CPL_TYPE_FLOAT, plist,
                        CPL_IO_EXTEND);
                cpl_image_delete(ima_corr) ;
            }
            cpl_propertylist_delete(plist) ;
        }
        cpl_propertylist_delete(mh_plist) ;
        cpl_free(out_fn) ;
        cpl_free(bpm_out_fn) ;
        cpl_free(level_out_fn) ;

		/* Get next RAW frame */
		rawframe = kmo_dfs_get_frame(frameset, NULL);
    }

    /* Free and Return */
    return CPL_ERROR_NONE;
}

/**@}*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Find out the root part of a basename (name without extension).
  @param    filename    File name to scan.
  @return   Pointer to statically allocated string.
 */
/*----------------------------------------------------------------------------*/
static char * kmos_get_root_name(const char * filename)
{
    static char path[4096+1];
    char * lastdot ;
    if (filename == NULL) return NULL;

    if (strlen(filename)>4096) return NULL ;
    memset(path, 0, 4096);
    strcpy(path, filename);
    lastdot = strrchr(path, '.');
    if (lastdot == NULL) return path ;
    if ((!strcmp(lastdot, ".fits")) || (!strcmp(lastdot, ".FITS")) ||
        (!strcmp(lastdot, ".dat")) || (!strcmp(lastdot, ".DAT")) ||
        (!strcmp(lastdot, ".paf")) || (!strcmp(lastdot, ".PAF")) ||
        (!strcmp(lastdot, ".txt")) || (!strcmp(lastdot, ".TXT")) ||
        (!strcmp(lastdot, ".ascii")) || (!strcmp(lastdot, ".ASCII")))
    {
        lastdot[0] = (char)0;
    }
    return path ;
}

