/* 
 * This file is part of the KMOS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*----------------------------------------------------------------------------*/
/**
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

#include "kmos_molecfit.h"

/*----------------------------------------------------------------------------*/
/**
 *                              Defines
 */
/*----------------------------------------------------------------------------*/

#define KMOS_MOLECFIT_CALCULE_COLUMN_NF           "NF"
#define KMOS_MOLECFIT_CALCULE_COLUMN_NTF          "NTF"

#define KMOS_MOLECFIT_RESULTS_ROW_RMS_REL_TO_MEAN "rms_rel_to_mean"
#define KMOS_MOLECFIT_RESULTS_ROW_RMS_STATUS      "status"
#define KMOS_MOLECFIT_RESULTS_ROW_RMS_H2O_COL_MM  "h2o_col_mm"

#define KMOS_MOLECFIT_QC_PARAM_RMS                "ESO QC RMS"
#define KMOS_MOLECFIT_QC_PARAM_STATUS             "ESO QC STATUS"
#define KMOS_MOLECFIT_QC_PARAM_H2O                "ESO QC H2O"
#define KMOS_MOLECFIT_QC_PARAM_IMPROV             "ESO QC IMPROV"

#define KMOS_MOLECFIT_QC_PARAM_RMS_TXT            "Molecfit best_fit parameter rms_rel_to_mean"
#define KMOS_MOLECFIT_QC_PARAM_STATUS_TXT         "Molecfit best_fit parameter status"
#define KMOS_MOLECFIT_QC_PARAM_H2O_TXT            "Molecfit best_fit parameter h2o_col_mm"
#define KMOS_MOLECFIT_QC_PARAM_IMPROV_TXT         "Computed from the Molecfit model columns flux, mscal, mtrans"

/*#define KMOS_MOLECFIT_QC_PARAM_STD_STARS		  "ESO QC NR STD STARS"
#define KMOS_MOLECFIT_QC_PARAM_GDAS				  "ESO QC GDAS"
#define KMOS_MOLECFIT_QC_PARAM_FLAG_GDAS		  "ESO QC FLAG GDAS"
#define KMOS_MOLECFIT_QC_PARAM_ZPT_AVG			  "ESO QC ZPOINT AVG"
#define KMOS_MOLECFIT_QC_PARAM_ZPT_RMS			  "ESO QC ZPOINT RMS"
#define KMOS_MOLECFIT_QC_PARAM_RMS_AVG			  "ESO QC RMS AVG"
#define KMOS_MOLECFIT_QC_PARAM_RMS_RMS			  "ESO QC RMS RMS"
#define KMOS_MOLECFIT_QC_PARAM_H2O_AVG			  "ESO QC H2O AVG"
#define KMOS_MOLECFIT_QC_PARAM_H2O_RMS			  "ESO QC H2O RMS"
#define KMOS_MOLECFIT_QC_PARAM_H2O_RELRMS		  "ESO QC H2O RELRMS"
#define KMOS_MOLECFIT_QC_PARAM_IMPROV_AVG		  "ESO QC IMPROV AVG"
#define KMOS_MOLECFIT_QC_PARAM_IMPROV_RMS		  "ESO QC IMPROV RMS"
#define KMOS_MOLECFIT_QC_PARAM_MEAS_IWV			  "ESO QC MEAS IWV"
#define KMOS_MOLECFIT_QC_PARAM_MEAS_IWV30D		  "ESO QC MEAS IWV30D"
#define KMOS_MOLECFIT_QC_PARAM_H2O_DELTA		  "ESO QC H2O DELTA"

#define KMOS_MOLECFIT_QC_PARAM_STD_STARS_TXT	  "Nr. of standard stars"
#define KMOS_MOLECFIT_QC_PARAM_GDAS_TXT			  "GDAS profile of available"
#define KMOS_MOLECFIT_QC_PARAM_FLAG_GDAS_TXT	  "GDAS flag "
#define KMOS_MOLECFIT_QC_PARAM_ZPT_AVG_TXT		  "Zeropoint average"
#define KMOS_MOLECFIT_QC_PARAM_ZPT_RMS_TXT		  "Zeropoint rms "
#define KMOS_MOLECFIT_QC_PARAM_RMS_AVG_TXT		  "RMS average"
#define KMOS_MOLECFIT_QC_PARAM_RMS_RMS_TXT		  "RMS rms"
#define KMOS_MOLECFIT_QC_PARAM_H2O_AVG_TXT		  "H2O value average"
#define KMOS_MOLECFIT_QC_PARAM_H2O_RMS_TXT		  "H2O value rms"
#define KMOS_MOLECFIT_QC_PARAM_H2O_RELRMS_TXT	  "H2O RMS / H2O AVG "
#define KMOS_MOLECFIT_QC_PARAM_IMPROV_AVG_TXT	  "IMPROV average"
#define KMOS_MOLECFIT_QC_PARAM_IMPROV_RMS_TXT	  "IMPROV rms"
#define KMOS_MOLECFIT_QC_PARAM_MEAS_IWV_TXT		  "Avg of TEL.AMBI.IWV.START and TEL.AMBI.IWV.END"
#define KMOS_MOLECFIT_QC_PARAM_MEAS_IWV30D_TXT	  "Avg of TEL.AMBI.IWV30D.START and TEL.AMBI.IWV30D.END"
#define KMOS_MOLECFIT_QC_PARAM_H2O_DELTA_TXT	  "H2O AVG - MEAS IWV" */

/*----------------------------------------------------------------------------*/
/**
 *                 Typedefs: Structs and enum types
 */
/*----------------------------------------------------------------------------*/

typedef struct {
  cpl_boolean fit;              /* Flag: Fit resolution                                     */
  double      res;              /* Initial value for FWHM                                   */
} mf_kernel;

typedef struct {
  kmos_molecfit_parameter mf;   /* Generic molecfit parameter                                   */
  const char  *process_ifus;    /* IFUs to process. If -1 (default), process all IFUs with data */
  double      ftol;             /* Relative chi-square convergence criterion                    */
  double      xtol;             /* Relative parameter  convergence criterion                    */
  mf_kernel   boxcar;           /* Fit resolution by boxcar LSF                                 */
  mf_kernel   gauss;            /* Fit resolution by Gaussian                                   */
  mf_kernel   lorentz;          /* Fit resolution by Lorentz                                    */
} kmos_molecfit_model_parameter;


/*----------------------------------------------------------------------------*/
/**
 *                              Functions prototypes
 */
/*----------------------------------------------------------------------------*/

/* Check the string variables defined by the user (wave_range and molecules definition) */
static cpl_error_code kmos_molecfit_model_check_wave_molec_conf(
    kmos_grating *grating);

/* Fill the internal KMOS configuration file */
static kmos_molecfit_model_parameter * kmos_molecfit_model_conf(
    const cpl_parameterlist *list);

/* Get the input GDAS profile if the user provide */
static cpl_table * kmos_molecfit_model_get_gdas_profile(
    cpl_frameset *frameset);

/* Get the input ATM_PROFILE_STANDARD if the user provide */
static cpl_table * kmos_molecfit_model_get_atm_profile_standard(
    cpl_frameset *frameset);

/* Fill the molecfit specific recipe configuration file */
static cpl_error_code kmos_molecfit_model_mf_conf(
    kmos_molecfit_model_parameter *conf,
    kmos_spectrum                 *ifu,
    kmos_grating_type             type,
    mf_parameters_config          *config_parameters);

/* Clean variables allocated in the recipe */
static void kmos_molecfit_model_clean(
    kmos_molecfit_model_parameter *conf);

/* Insert QC parameters in the headers of the IFUs with valid data */
static cpl_error_code kmos_molecfit_model_headers_fill_qc_parameters(
    const cpl_table  *res_table,
    const cpl_table  *spec_out,
    cpl_propertylist *pl_BParms,
    cpl_propertylist *pl_BModel,
	cpl_vector 		 *qc_zpt,
	cpl_vector 		 *qc_h2o,
	cpl_vector 		 *qc_rms,
	cpl_vector 		 *qc_improv,
	/*cpl_vector 		 *qc_meas_iwv,
	cpl_vector 		 *qc_meas_iwv30d, */
	int 			 k);

/*----------------------------------------------------------------------------*/
/**
 *                          Static variables
 */
/*----------------------------------------------------------------------------*/

#define RECIPE_NAME      KMOS_MOLECFIT_MODEL
#define CONTEXT          "kmos."RECIPE_NAME

static char kmos_molecfit_model_description[] =
    "This recipe runs molecfit on a 1D spectrum and the input data can have category: \n"
    " - STAR_SPEC (24 DATA plus 24 NOISE extensions)\n"
    " - EXTRACT_SPEC (24 DATA extensions, additional 24 NOISE extensions are optional)\n"
    " - SCIENCE (24 DATA extensions, additional 24 NOISE extensions are optional)\n"
    " - SCI_RECONSTRUCTED (24 DATA extensions, additional 24 NOISE extensions are optional)\n"
    "It is not mandatory that all the DATA extension contains data.\n"
    "Molecfit will be run on all the extension that contain data (not noise extensions). \n"
    "The recipe also accepts as input a kernel library, e.g. the information of the spectral resolution for each IFU and each rotator angle.          \n"
    "When the recipe executes molecfit on the i-th IFU, the kernel in the library that matches the IFU number and rotator angle is used.              \n"
    "If no kernel are provided, each time the recipe runs molecfit, the kernel is considered as a free parameter and it will be determined by molecfit itself and stored into the BEST_FIT_PARM output.                            \n"
    "\n"
    "Input files: (It's mandatory to provide 1 file only of type A) \n"
    "\n"
    "   DO                  KMOS                                                  \n"
    "   category            Type   required   Explanation                         \n"
    "   --------            -----  --------   -----------                         \n"
    "   STAR_SPEC            F1I       A       The spectrum (1D spectrum, IMAGE format).\n"
    "   EXTRACT_SPEC         F1I       A       The spectrum (1D spectrum, IMAGE format).\n"
    "   SCIENCE              F3I       A       The spectrum (3D cube,     IMAGE format).\n"
    "   SCI_RECONSTRUCTED    F3I       A       The spectrum (3D cube,     IMAGE format).\n"
    "   KERNEL_LIBRARY       F2I       N       The kernel library; must be the same grating as the other inputs.\n"
    "   GDAS                 F1L       N       User define GDAS profile (fits table).\n"
    "\n"
    "Output files:                                                                \n"
    "\n"
    "   DO                  KMOS                                                  \n"
    "   category            Type              Explanation                         \n"
    "   --------            -----             -----------                         \n"
    "   GDAS                 F1L               Used GDAS profile                                       (fits table).\n"
    "   GDAS_BEFORE          F1L               If ESO DB GDAS is used, file before the MJD-OBS         (fits table).\n"
    "   GDAS_AFTER           F1L               If ESO DB GDAS is used, file after  the MJD-OBS         (fits table).\n"
    "   ATM_PROF_STANDARD    F1L               Atmospheric profile standard.                           (fits table).\n"
    "   ATM_PROF_COMBINED    F1L               Atmospheric profile combined with GDAS.                 (fits table).\n"
    "   ATMOS_PARM           F1L               Atmospheric parameters                   (multiextension fits table).\n"
    "   BEST_FIT_PARM        F1L               Best fitting parameters                  (multiextension fits table).\n"
    "   BEST_FIT_MODEL       F1L               Best fit model and intermediate products (multiextension fits table).\n"
    "   MODEL_KERNEL_LIBRARY F2I               The kernel library used in Molecfit; if user provide kernel inputs.  \n"
    "\n";

/* Standard CPL recipe definition */
cpl_recipe_define(	kmos_molecfit_model,
                  	KMOS_BINARY_VERSION,
                  	"Jose A. Escartin, Yves Jung",
                  	"https://support.eso.org/",
                  	"2017",
                  	"Run molecfit on set of 1D spectra to compute an atmospheric model.",
                  	kmos_molecfit_model_description);


/*----------------------------------------------------------------------------*/
/**
 * @defgroup kmos_molecfit_model  It runs molecfit on KMOS standard star file
 *                                to compute an atmospheric model.
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/**
 *                              Functions code
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 * @brief    Interpret the command line options and execute the data processing
 *
 * @param    frameset   the frames list
 * @param    parlist    the parameters list
 *
 * @return   CPL_ERROR_NONE if everything is OK or CPL_ERROR_CODE in other case
 *
 */
/*----------------------------------------------------------------------------*/
static int kmos_molecfit_model(
    cpl_frameset *frameset, const cpl_parameterlist *parlist)
{
  /* Check initial Entries */
  if (kmos_check_and_set_groups(frameset) != CPL_ERROR_NONE) {
      return cpl_error_get_code();
  }

  /* Get initial errorstate */
  cpl_errorstate initial_errorstate = cpl_errorstate_get();

  /* Extract and verify data in the parameters of the recipe */
  cpl_msg_info(cpl_func, "Configuring initial parameters in the recipe ...");
  kmos_molecfit_model_parameter *conf = kmos_molecfit_model_conf(parlist);
  cpl_error_ensure(conf, CPL_ERROR_ILLEGAL_INPUT,
                   return (int)CPL_ERROR_ILLEGAL_INPUT, "Problems with the configuration parameters");
  if (!(conf->mf.fit_wavelenght.fit)) cpl_msg_info(cpl_func, "Not fit wavelength!");
  if (!(conf->mf.fit_continuum.fit) ) cpl_msg_info(cpl_func, "Not fit continuum!" );


  /* Loading data spectrums */
  if(kmos_molecfit_load_spectrums(frameset, &(conf->mf), RECIPE_NAME) != CPL_ERROR_NONE) {
      kmos_molecfit_model_clean(conf);
      return (int)cpl_error_set_message(cpl_func, CPL_ERROR_FILE_IO,
                                        "Loading spectrums data in the input frameset failed!");
  }

  /* Loading kernels */
  const cpl_frame *frmKernel = cpl_frameset_find(frameset, KERNEL_LIBRARY);
  cpl_error_code sKernels_e;
  if (frmKernel && conf->mf.use_input_kernel) {

      if(kmos_molecfit_load_kernels(frmKernel, &(conf->mf)) != CPL_ERROR_NONE) {
          kmos_molecfit_model_clean(conf);
          return (int)cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                            "Loading convolution kernel data in the input frameset failed!");
      }

      sKernels_e = kmos_molecfit_save(frameset, frameset, parlist, RECIPE_NAME, conf->mf.parms, MODEL_KERNEL_LIBRARY, conf->mf.grating.name, conf->mf.suppress_extension, NULL);
      if (sKernels_e != CPL_ERROR_NONE){
          kmos_molecfit_model_clean(conf);
          return (int)cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                            "Saving generic multi-extension output fits file ('%s') failed!",
                                            MODEL_KERNEL_LIBRARY);
      }

  } else {

      if (frmKernel) {
          cpl_msg_warning(cpl_func, "Kernel is provided, but use_input_kernel = false !");
      } else if (conf->mf.use_input_kernel){
          cpl_msg_warning(cpl_func, "Kernel isn't provided, but use_input_kernel = true !");
      }

      cpl_msg_info(cpl_func, "Using the default molecfit kernels!");
      cpl_msg_info(cpl_func, "Fit resolution by Boxcar   (--fit_res_box    ) = %d", conf->boxcar.fit );
      cpl_msg_info(cpl_func, "Fit resolution by Gaussian (--fit_res_gauss  ) = %d", conf->gauss.fit  );
      cpl_msg_info(cpl_func, "Fit resolution by Lorentz  (--fit_res_lorentz) = %d", conf->lorentz.fit);
  }
 


  cpl_msg_info(cpl_func, " +++ All input data loaded successfully! +++");

  /*
   * Updating KMOS header with parameter values
   */

  cpl_propertylist_update_string(conf->mf.header_spectrums, MF_PARAMETERS_CONTEX_DEFAULT" "KMOS_MOLECFIT_KEYWORD_PROCESS_IFUS,       conf->process_ifus);

  cpl_propertylist_update_bool(  conf->mf.header_spectrums, MF_PARAMETERS_CONTEX_DEFAULT" "KMOS_MOLECFIT_KEYWORD_SUPPRESS_EXTENSION, conf->mf.suppress_extension);

  cpl_propertylist_update_string(conf->mf.header_spectrums, MF_PARAMETERS_CONTEX_DEFAULT" "KMOS_MOLECFIT_KEYWORD_WRANGE,             conf->mf.grating.wave_range);
  cpl_propertylist_update_string(conf->mf.header_spectrums, MF_PARAMETERS_CONTEX_DEFAULT" "KMOS_MOLECFIT_KEYWORD_LIST,               conf->mf.grating.list_molec);
  cpl_propertylist_update_string(conf->mf.header_spectrums, MF_PARAMETERS_CONTEX_DEFAULT" "KMOS_MOLECFIT_KEYWORD_FIT,                conf->mf.grating.fit_molec);
  cpl_propertylist_update_string(conf->mf.header_spectrums, MF_PARAMETERS_CONTEX_DEFAULT" "KMOS_MOLECFIT_KEYWORD_RELATIVE_VALUE,     conf->mf.grating.rel_col);

  cpl_propertylist_update_double(conf->mf.header_spectrums, MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_FTOL,                       conf->ftol);
  cpl_propertylist_update_double(conf->mf.header_spectrums, MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_XTOL,                       conf->xtol);

  cpl_propertylist_update_bool(  conf->mf.header_spectrums, MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_FIT_CONTINUUM,              conf->mf.fit_continuum.fit);
  cpl_propertylist_update_int(   conf->mf.header_spectrums, MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_CONTINUUM_N,                conf->mf.fit_continuum.n);

  cpl_propertylist_update_bool(  conf->mf.header_spectrums, MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_FIT_WLC,                    conf->mf.fit_wavelenght.fit);
  cpl_propertylist_update_int(   conf->mf.header_spectrums, MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_WLC_N,                      conf->mf.fit_wavelenght.n);
  cpl_propertylist_update_double(conf->mf.header_spectrums, MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_WLC_CONST,                  conf->mf.fit_wavelenght.const_val);

  cpl_propertylist_update_bool(  conf->mf.header_spectrums, MF_PARAMETERS_CONTEX_DEFAULT" "KMOS_MOLECFIT_KEYWORD_USE_INPUT_KERNEL,   conf->mf.use_input_kernel);

  cpl_propertylist_update_bool(  conf->mf.header_spectrums, MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_KERN_MODE,                  conf->mf.kernmode);
  cpl_propertylist_update_double(conf->mf.header_spectrums, MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_KERN_FAC,                   conf->mf.kernfac);
  cpl_propertylist_update_bool(  conf->mf.header_spectrums, MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_VAR_KERN,                   conf->mf.varkern);

  cpl_propertylist_update_bool(  conf->mf.header_spectrums, MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_FIT_RES_BOX,                conf->boxcar.fit);
  cpl_propertylist_update_double(conf->mf.header_spectrums, MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_RES_BOX,                    conf->boxcar.res);

  cpl_propertylist_update_bool(  conf->mf.header_spectrums, MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_FIT_GAUSS,                  conf->gauss.fit);
  cpl_propertylist_update_double(conf->mf.header_spectrums, MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_RES_GAUSS,                  conf->gauss.res);

  cpl_propertylist_update_bool(  conf->mf.header_spectrums, MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_FIT_LORENTZ,                conf->lorentz.fit);
  cpl_propertylist_update_double(conf->mf.header_spectrums, MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_RES_LORENTZ,                conf->lorentz.res);


  /* Saving generic multi-extension output *.fits files */
  cpl_msg_info(cpl_func, "Saving generic multi-extension output fits file ('%s','%s','%s') ...",
               ATMOS_PARM, BEST_FIT_PARM, BEST_FIT_MODEL);
  const char * atmos_file = "atmos_param_pre.fits";
  const char * bfp_file = "best_fit_param_pre.fits";
  const char * bfm_file = "best_fit_model_pre.fits";
  cpl_error_code sAtm_e = cpl_propertylist_save(conf->mf.header_spectrums, atmos_file, CPL_IO_CREATE);
		  //kmos_molecfit_save(frameset, frameset, parlist, RECIPE_NAME, conf->mf.parms, ATMOS_PARM,     conf->mf.grating.name, conf->mf.suppress_extension, atmos_file);
  cpl_error_code sBParms_e = cpl_propertylist_save(conf->mf.header_spectrums, bfp_file, CPL_IO_CREATE);
		  //kmos_molecfit_save(frameset, frameset, parlist, RECIPE_NAME, conf->mf.parms, BEST_FIT_PARM,  conf->mf.grating.name, conf->mf.suppress_extension, bfp_file);
  cpl_error_code sBModel_e = cpl_propertylist_save(conf->mf.header_spectrums, bfm_file, CPL_IO_CREATE);
		  //kmos_molecfit_save(frameset, frameset, parlist, RECIPE_NAME, conf->mf.parms, BEST_FIT_MODEL, conf->mf.grating.name, conf->mf.suppress_extension, bfm_file);
  if (sAtm_e != CPL_ERROR_NONE || sBParms_e != CPL_ERROR_NONE || sBModel_e != CPL_ERROR_NONE) {
      kmos_molecfit_model_clean(conf);
      return (int)cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                        "Saving generic multi-extension output fits files ('%s','%s','%s') failed!",
                                        ATMOS_PARM, BEST_FIT_PARM, BEST_FIT_MODEL);
  }

  /*** General configuration of Molecfit for LBLRTM and LNFL calls ***/
  cpl_error_code err = CPL_ERROR_NONE;

  /* Initialize mf_configuration structure */
  mf_configuration *mf_config = mf_configuration_create();
  
  /* ---------------------------------------------------------------*/
  /* Temporaray solution to fit chips and fit ranges flag mechanism */
  /* ---------------------------------------------------------------*/
  cpl_msg_info(cpl_func,"Sorting the chips and ranges flags");

  /* FIX FOR CHIPS */
  cpl_boolean flag= conf->mf.fit_wavelenght.fit;
  int order = conf->mf.fit_wavelenght.n;
  if ( flag) {
      cpl_msg_info(cpl_func,"Fitting wavlength corrections with polynomial of order %d",order);
      for (int i=0; i<MF_PARAMETERS_MAXIMUM_NO_OF_CHIPS; i++) {
          mf_config->parameters->fitting.fit_chips[i]=CPL_TRUE;
      }
  } 
  
  /* FIX FOR RANGES */
  flag  = conf->mf.fit_continuum.fit;
  order = conf->mf.fit_continuum.n;
  if ( flag) {
      cpl_msg_info(cpl_func,"Fitting contiuum model for each range with polynomials of order %d ", order);
      for (int i=0; i<MF_PARAMETERS_MAXIMUM_NO_OF_RANGES; i++) {
          mf_config->parameters->fitting.fit_ranges[i]=CPL_TRUE;
          mf_config->parameters->fitting.cont_poly_order[i]=order;
      }
  }
 
  /* ---------------------------------------------------------------*/
  /* END OF Temporary fix of chips and ranges flag mechanism        */
  /* ---------------------------------------------------------------*/

  
  if (!mf_config) err = CPL_ERROR_NULL_INPUT;

  /* Update the molecfit configuration with the data primary header */
  err = mf_parameters_config_update_with_header_keywords(mf_config->parameters, conf->mf.header_spectrums);
  if (err != CPL_ERROR_NONE) {
      cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                            "Error updating molecfit parameters with the raw primary header configuration");
  }

  /*** GDAS user provided ***/
  cpl_msg_info(cpl_func, "Checking if %s user profile provided ...", GDAS);
  cpl_boolean gdas_write = CPL_FALSE;
  cpl_table   *gdas_user = kmos_molecfit_model_get_gdas_profile(frameset);

  /*** ATM_PROFILE_STANDARD user provided ***/
  cpl_msg_info(cpl_func, "Checking if %s user provided ...", ATM_PROFILE_STANDARD);
  cpl_boolean atm_profile_standard_write = CPL_FALSE;
  cpl_table   *atm_profile_standard = kmos_molecfit_model_get_atm_profile_standard(frameset);
  cpl_table   *atm_profile_combined = NULL;

  cpl_vector * qc_zpt = cpl_vector_new(N_IFUS);
  cpl_vector * qc_h2o = cpl_vector_new(N_IFUS);
  cpl_vector * qc_rms = cpl_vector_new(N_IFUS);
  cpl_vector * qc_improv = cpl_vector_new(N_IFUS);
  /*cpl_vector * qc_meas_iwv = cpl_vector_new(N_IFUS);
  cpl_vector * qc_meas_iwv30d = cpl_vector_new(N_IFUS); */

  /* Number of IFUs processed with molecfit */
  cpl_size     n_procesed_ifus = 0;
  kmos_grating *grating        = &(conf->mf.grating);
  char * gdas_avail = "F";
  int gdas_flag = 0;

  /* Execute molecfit for each data spectrum (odd extensions) in STAR_SPEC */
  for (cpl_size i = 0; i < N_IFUS; i++) {

      kmos_spectrum *ifu = &(conf->mf.ifus[i]);

      /* Initialize variables for molecfit execution */
      mf_model_results *results = NULL;

      /* Create local headers to fill QC parameters, if they are need */
      cpl_propertylist *pl_BParms = cpl_propertylist_duplicate(ifu->header_ext_data);
      cpl_propertylist *pl_BModel = cpl_propertylist_duplicate(ifu->header_ext_data);

      /* Check if the IFU contains data */
      if (ifu->process && ifu->data) {

    	  /*cpl_msg_info(cpl_func, "Dumping Propertylist BEST FIT PARAM ");
    	  cpl_propertylist_dump(pl_BParms, stdout);
    	  cpl_msg_info(cpl_func, "Dumping Propertylist BEST FIT MODEL ");
    	  cpl_propertylist_dump(pl_BModel, stdout); */

          /* Configuration variables */
          cpl_msg_info(cpl_func, "Building molecfict configuration variable in %s ...", ifu->name);

          /* Building molecfic configuration file */
          if (!err) err = kmos_molecfit_model_mf_conf(conf, ifu, grating->type, mf_config->parameters);

          //mf_config->parameters->inputs.silent_external_bins = CPL_FALSE;

          /* Running molecfit */
          cpl_msg_info(cpl_func,"MNB About to call mf_model, err=%d", err);
          if (!err) {

              /* Create input molecfit spec format */
              cpl_table *spec_telluriccorr_lblrtm = mf_spectrum_create(mf_config->parameters, ifu->data);

              /* CALL MOLECFIT */
              cpl_msg_info(cpl_func, "Call mf_model(...) in %s ...", ifu->name);
              results = mf_model( mf_config,                   /* mf_configuration       *config               */
                                  grating->molecules,          /* cpl_table              *molecules            */
                                  ifu->header_ext_data,        /* const cpl_propertylist *header_spec          */
                                  spec_telluriccorr_lblrtm,    /* const cpl_table        *spec_telluriccorr    */
                                  grating->incl_wave_ranges,   /* cpl_table              *inc_wranges          */
                                  NULL,                        /* cpl_table              *exc_wranges          */
                                  NULL,                        /* cpl_table              *exc_pranges          */
                                  ifu->kernel.header_ext_data, /* const cpl_propertylist *header_kernel        */
                                  ifu->kernel.data,            /* const cpl_matrix       *kernel               */
                                  gdas_user,                   /* const cpl_table        *gdas_user            */
                                  atm_profile_standard,        /* const cpl_table        *atm_profile_standard */
                                  atm_profile_combined);       /* const cpl_table        *atm_profile_combined */

              cpl_table_delete(spec_telluriccorr_lblrtm);

              cpl_msg_info(cpl_func, "GDAS source for IFU 1=  %s", results->gdas_src );
              if (strcmp(results->gdas_src,"REPOSITORY")) {
            	  gdas_avail= "T";
            	  gdas_flag = 0;
              }

              /* Check possible errors */
              err = cpl_error_get_code();

        	  /*cpl_msg_info(cpl_func, "Dumping Propertylist BEST FIT PARAMS ");
        	  cpl_propertylist_dump(pl_BParms, stdout);
        	  cpl_msg_info(cpl_func, "Dumping Propertylist BEST FIT MODEL ");
        	  cpl_propertylist_dump(pl_BModel, stdout); */

              /* Add QC parameters to assess the quality of the telluric correction */
              if (!err) {

                  err = kmos_molecfit_model_headers_fill_qc_parameters(results->res, results->spec, pl_BParms, pl_BModel,
                		  qc_zpt, qc_h2o, qc_rms, qc_improv , n_procesed_ifus);
                  //qc_zpoint = cpl_propertylist_get_double(pl_BParams, );

                  cpl_msg_info(cpl_func, "Molecfit call mf_model(...) run successfully! in %s ...", ifu->name);

                  cpl_msg_info(cpl_func,"Dumping BEST FIT MODEL Propertylist");
                  //cpl_propertylist_dump(pl_BModel,stdout);
                  cpl_msg_info(cpl_func,"Dumping BEST FIT PARAM Propertylist");
                  //cpl_propertylist_dump(pl_BParms,stdout);
                  /*cpl_msg_info(cpl_func,"Dumping ATMOS PARAM Propertylist");
                  cpl_propertylist_dump(ifu->header_ext_data,stdout); */

              } else {

                  cpl_msg_warning(cpl_func, "Molecfit call mf_model(...) run Error! in %s ... %s", ifu->name, cpl_error_get_message());

                  /* Reset error */
                  if (!cpl_errorstate_is_equal(initial_errorstate)) cpl_errorstate_set(initial_errorstate);
                  err = CPL_ERROR_NONE;
              }

              /* Increase process ifu counter */
              n_procesed_ifus++;
          }
      }

      if (!err) {

          /* Saving data */
          if (     !(ifu->data)   ) cpl_msg_info(   cpl_func, "Saving data IFU=%d ...", ifu->num);
          else if (!(ifu->process)) cpl_msg_info(   cpl_func, "Saving data IFU=%d ... (data spectrum -> But not processed)", ifu->num);
          else if (!results)        cpl_msg_warning(cpl_func, "Saving data IFU=%d ... (data spectrum -> But failed Molecfit!)", ifu->num);
          else                      cpl_msg_info(   cpl_func, "Saving data IFU=%d ... (data spectrum -> Molecfit executed)", ifu->num);


          /* Write GDAS files on disk */
          if (results && !gdas_write) {

              if (gdas_user) {

                  if (results->gdas_interpolate) {

                      cpl_msg_info(cpl_func, "Saving Molecfit output user provide fits files ('%s') ... [only first call to Molecfit!]",
                                   GDAS);

                      kmos_molecfit_save(frameset, frameset, parlist, RECIPE_NAME, conf->mf.parms, GDAS, NULL, CPL_TRUE, NULL);
                      cpl_table_save(results->gdas_interpolate, NULL, conf->mf.header_spectrums, GDAS".fits", CPL_IO_EXTEND);


                      gdas_write = CPL_TRUE;

                      if (cpl_error_get_code() != CPL_ERROR_NONE) {
                          cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                                "Saving Molecfit output GDAS user provide fits file ('%s') failed!",
                                                GDAS);
                      }
                  }

              } else if (results->gdas_before && results->gdas_after && results->gdas_interpolate) {

                  cpl_msg_info(cpl_func, "Saving Molecfit output GDAS automatic fits files ('%s','%s','%s') ... [only first call to Molecfit!]",
                               GDAS_BEFORE, GDAS_AFTER, GDAS);

                  kmos_molecfit_save(frameset, frameset, parlist, RECIPE_NAME, conf->mf.parms, GDAS_BEFORE, NULL, CPL_FALSE, NULL);
                  cpl_table_save(results->gdas_before,      NULL, conf->mf.header_spectrums, GDAS_BEFORE".fits", CPL_IO_EXTEND);

                  kmos_molecfit_save(frameset, frameset, parlist, RECIPE_NAME, conf->mf.parms, GDAS_AFTER, NULL, CPL_FALSE, NULL);
                  cpl_table_save(results->gdas_after,       NULL, conf->mf.header_spectrums, GDAS_AFTER".fits",  CPL_IO_EXTEND);

                  kmos_molecfit_save(frameset, frameset, parlist, RECIPE_NAME, conf->mf.parms, GDAS, NULL, CPL_FALSE, NULL);
                  cpl_table_save(results->gdas_interpolate, NULL, conf->mf.header_spectrums, GDAS".fits", CPL_IO_EXTEND);

                  /* Keep time, no access to ESO GDAS DB : Same gdas_interpolate for all the IFUs because all the arms have the same 'date' -> MDJ_OBS in the header */
                  gdas_user = cpl_table_duplicate(results->gdas_interpolate);

                  gdas_write = CPL_TRUE;

                  if (cpl_error_get_code() != CPL_ERROR_NONE) {
                      cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                            "Saving Molecfit output GDAS automatic fits files ('%s','%s','%s') failed!",
                                            GDAS_BEFORE, GDAS_AFTER, GDAS);
                  }
              }
          }

          /* Write ATM_PROFILE_STANDARD files on disk */
          if (results && !atm_profile_standard_write) {

              if (atm_profile_standard) {

                  if (results->atm_profile_standard) {

                      cpl_msg_info(cpl_func, "Saving Molecfit output user provide fits files ('%s') ... [only first call to Molecfit!]",
                                   ATM_PROFILE_STANDARD);

                      kmos_molecfit_save(frameset, frameset, parlist, RECIPE_NAME, conf->mf.parms, ATM_PROFILE_STANDARD, NULL, CPL_FALSE, NULL);
                      cpl_table_save(results->atm_profile_standard, NULL, conf->mf.header_spectrums, ATM_PROFILE_STANDARD".fits", CPL_IO_EXTEND);

                      kmos_molecfit_save(frameset, frameset, parlist, RECIPE_NAME, conf->mf.parms, ATM_PROFILE_COMBINED, NULL, CPL_FALSE, NULL);
                      cpl_table_save(results->atm_profile_combined, NULL, conf->mf.header_spectrums, ATM_PROFILE_COMBINED".fits", CPL_IO_EXTEND);

                      atm_profile_combined = cpl_table_duplicate(results->atm_profile_combined);

                      atm_profile_standard_write = CPL_TRUE;

                      if (cpl_error_get_code() != CPL_ERROR_NONE) {
                          cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                                "Saving Molecfit output user provide fits file ('%s') failed!",
                                                ATM_PROFILE_STANDARD);
                      }
                  }

              } else if (results->atm_profile_standard) {

                  cpl_msg_info(cpl_func, "Saving Molecfit output ATM_PROFILE automatic fits files ('%s','%s') ... [only first call to Molecfit!]",
                               ATM_PROFILE_STANDARD, ATM_PROFILE_COMBINED);

                  kmos_molecfit_save(frameset, frameset, parlist, RECIPE_NAME, conf->mf.parms, ATM_PROFILE_STANDARD, NULL, CPL_FALSE, NULL);
                  cpl_table_save(results->atm_profile_standard, NULL, conf->mf.header_spectrums, ATM_PROFILE_STANDARD".fits", CPL_IO_EXTEND);

                  kmos_molecfit_save(frameset, frameset, parlist, RECIPE_NAME, conf->mf.parms, ATM_PROFILE_COMBINED, NULL, CPL_FALSE, NULL);
                  cpl_table_save(results->atm_profile_combined, NULL, conf->mf.header_spectrums, ATM_PROFILE_COMBINED".fits", CPL_IO_EXTEND);

                  /* Keep time, no access to disk : Same atm_profile_standard for all the IFUs because all the arms have the same 'date' -> MDJ_OBS in the header */
                  atm_profile_standard = cpl_table_duplicate(results->atm_profile_standard);
                  atm_profile_combined = cpl_table_duplicate(results->atm_profile_combined);

                  atm_profile_standard_write = CPL_TRUE;

                  if (cpl_error_get_code() != CPL_ERROR_NONE) {
                      cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                            "Saving Molecfit output GDAS automatic fits files ('%s','%s') failed!",
                                            ATM_PROFILE_STANDARD, ATM_PROFILE_COMBINED);
                  }
              }
          }

          cpl_table *atm_profile_fitted = results ? results->atm_profile_fitted : NULL;
          cpl_table *spec               = results ? results->spec               : NULL;
          cpl_table *res                = results ? results->res                : NULL;

          sAtm_e    = kmos_molecfit_save_mf_results(ifu->header_ext_data, ifu->header_ext_noise, ATMOS_PARM, conf->mf.grating.name, conf->mf.suppress_extension, atmos_file, NULL, atm_profile_fitted, NULL);
        		  //kmos_molecfit_save_mf_results(ifu->header_ext_data, ifu->header_ext_noise, ATMOS_PARM,     conf->mf.grating.name, conf->mf.suppress_extension, atmos_file, NULL, atm_profile_fitted, NULL);
          sBModel_e = kmos_molecfit_save_mf_results(pl_BModel, ifu->header_ext_noise, BEST_FIT_MODEL, conf->mf.grating.name, conf->mf.suppress_extension, bfm_file, NULL, spec,  NULL);
        		  //kmos_molecfit_save_mf_results(pl_BModel, ifu->header_ext_noise, BEST_FIT_MODEL, conf->mf.grating.name, conf->mf.suppress_extension, bfm_file, NULL, spec,  NULL);
          sBParms_e = kmos_molecfit_save_mf_results(pl_BParms, ifu->header_ext_noise, BEST_FIT_PARM,  conf->mf.grating.name, conf->mf.suppress_extension, bfp_file, NULL, res, NULL);
        		  //kmos_molecfit_save_mf_results(pl_BParms, ifu->header_ext_noise, BEST_FIT_PARM,  conf->mf.grating.name, conf->mf.suppress_extension, bfp_file, NULL, res, NULL);

          if (sAtm_e != CPL_ERROR_NONE || sBModel_e != CPL_ERROR_NONE || sBParms_e != CPL_ERROR_NONE || sKernels_e != CPL_ERROR_NONE) {
              err = cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                          "Saving Molecfit output fits files ('%s','%s','%s') failed!",
                                          ATMOS_PARM, BEST_FIT_MODEL, BEST_FIT_PARM);
          }

          if (frmKernel && conf->mf.use_input_kernel) {

              cpl_propertylist *header_kernel_data  = ifu->kernel.header_ext_data  ? ifu->kernel.header_ext_data  : ifu->header_ext_data;
              cpl_propertylist *header_kernel_noise = ifu->kernel.header_ext_noise ? ifu->kernel.header_ext_noise : ifu->header_ext_noise;

              cpl_matrix *kernel = NULL;
              if (results) {
                  if (results->kernel) kernel = results->kernel;
              }

              sKernels_e = kmos_molecfit_save_mf_results(header_kernel_data, header_kernel_noise,
                                                         MODEL_KERNEL_LIBRARY, conf->mf.grating.name, conf->mf.suppress_extension, NULL, kernel, NULL, NULL);

              if (sKernels_e != CPL_ERROR_NONE) {
                  err = cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                              "Saving Molecfit output fits file ('%s') failed!",
                                              MODEL_KERNEL_LIBRARY);
              }
          }
      }

      /* Cleanup MF_MODEL auxiliar variables */
      mf_model_results_delete(results);
      cpl_propertylist_delete(pl_BParms);
      cpl_propertylist_delete(pl_BModel);
  }

  /* ---------------------------- QC PARAM CALCULATION -------------------------------------*/

  cpl_msg_info(cpl_func,"Calculating QC Parameters");

  cpl_propertylist* qc_atmos  =  kmclipm_propertylist_load(atmos_file, 0); //kmo_dfs_load_primary_header(frameset, ATMOS_PARM);
  cpl_propertylist* qc_bfp  = kmclipm_propertylist_load(bfp_file, 0); //kmo_dfs_load_primary_header(frameset, BEST_FIT_PARM);
  cpl_propertylist* qc_bfm  = kmclipm_propertylist_load(bfm_file, 0); //kmo_dfs_load_primary_header(frameset, BEST_FIT_MODEL);

  //cpl_propertylist_dump(qc_atmos,stdout);

  /*if (sAtm_e != CPL_ERROR_NONE || sBParms_e != CPL_ERROR_NONE || sBModel_e != CPL_ERROR_NONE) {
      kmos_molecfit_model_clean(conf);
      return (int)cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                        "Saving generic multi-extension output fits files ('%s','%s','%s') failed!",
                                        ATMOS_PARM_TMP, BEST_FIT_PARM_TMP, BEST_FIT_MODEL_TMP);
  }
  else {
	  cpl_msg_info(cpl_func, "last error %s" ,cpl_error_get_message());
  } */

  cpl_size j= 0;

  qc_zpt = cpl_vector_extract(qc_zpt, 0, n_procesed_ifus-1,1);
  qc_h2o = cpl_vector_extract(qc_h2o, 0, n_procesed_ifus-1,1);
  qc_rms = cpl_vector_extract(qc_rms, 0, n_procesed_ifus-1,1);
  qc_improv = cpl_vector_extract(qc_improv, 0, n_procesed_ifus-1,1);

  double  zpt_avg = cpl_vector_get_mean(qc_zpt);
  double  zpt_rms = cpl_vector_get_stdev(qc_zpt);
  double  rms_avg = cpl_vector_get_mean(qc_rms);
  double  rms_rms = cpl_vector_get_stdev(qc_rms);
  double  h2o_avg = cpl_vector_get_mean(qc_h2o);
  double  h2o_rms = cpl_vector_get_stdev(qc_h2o);
  double  improv_avg = cpl_vector_get_mean(qc_improv);
  double  improv_rms = cpl_vector_get_stdev(qc_improv);
  double  meas_iwv =0.0, meas_iwv30d = 0.0;

  /*GDAS Check for QC parameters */
  //mf_parameters *params = mf_parameters_create(mf_config->parameters, grating->molecules, NULL);
  double lon            = mf_config->parameters->ambient.longitude.value;
  double lat            = mf_config->parameters->ambient.latitude.value;
  char *global_gdas_path    = cpl_sprintf("%s/%s", mf_config->parameters->directories.telluriccorr_data_path, MF_PROFILES_GDAS_PATH);

  /* Name of tarball and paths that contain all of GDAS files and the file to search (i.e. Paranal: gdas_profiles_C-70.4-24.6.tar.gz) */
  char * gdas_tarball_name   = cpl_sprintf("gdas_profiles_C%+.1f%+.1f.tar.gz", lon, lat);
  char * global_gdas_tarball = cpl_sprintf("%s/%s",global_gdas_path, gdas_tarball_name);

  if (access(global_gdas_tarball, F_OK) == CPL_ERROR_NONE) {
	  gdas_avail= "T";
	  gdas_flag = 0;
	  cpl_msg_info(cpl_func,"The GDAS file name - %s ",global_gdas_tarball );
	  cpl_msg_info(cpl_func,"The GDAS Profile was found in the REPO ");
  }
  /*END - put the above section in to kmo_utils.c as a new function*/

  /*qc_meas_iwv = cpl_vector_extract(qc_meas_iwv, 0, n_procesed_ifus-1,1);
  qc_meas_iwv30d = cpl_vector_extract(qc_meas_iwv30d, 0, n_procesed_ifus-1,1); */
  if (cpl_propertylist_has(qc_bfm, "ESO TEL AMBI IWV START") &&
		  cpl_propertylist_has(qc_bfm, "ESO TEL AMBI IWV END")){
	  meas_iwv = (cpl_propertylist_get_double(qc_bfm,"ESO TEL AMBI IWV END")+
					  cpl_propertylist_get_double(qc_bfm,"ESO TEL AMBI IWV START"))/2.0;
	  kmclipm_update_property_double(qc_bfm,QC_PARAM_MEAS_IWV,
			  meas_iwv, QC_PARAM_MEAS_IWV_TXT);
  }

  if (cpl_propertylist_has(qc_bfm, "ESO TEL AMBI IWV30D START") &&
		  cpl_propertylist_has(qc_bfm, "ESO TEL AMBI IWV30D END")){
	  meas_iwv30d =  (cpl_propertylist_get_double(qc_bfm,"ESO TEL AMBI IWV30D END")+
					  cpl_propertylist_get_double(qc_bfm,"ESO TEL AMBI IWV30D START"))/2.0;
	  kmclipm_update_property_double(qc_bfm, QC_PARAM_MEAS_IWV30D,
			  meas_iwv30d, QC_PARAM_MEAS_IWV30D_TXT);
  }

  if (cpl_propertylist_has(kmo_dfs_load_primary_header(frameset, "STAR_SPEC"),
		  QC_NR_STD_STARS)){

	  int nr_std_str= cpl_propertylist_get_int(kmo_dfs_load_primary_header(frameset, "STAR_SPEC"), QC_NR_STD_STARS);

	  kmclipm_update_property_int(qc_bfm, QC_NR_STD_STARS, nr_std_str, QC_NR_STD_STARS_TXT);
	  kmclipm_update_property_int(qc_bfp, QC_NR_STD_STARS, nr_std_str, QC_NR_STD_STARS_TXT);
	  kmclipm_update_property_int(qc_atmos, QC_NR_STD_STARS, nr_std_str, QC_NR_STD_STARS_TXT);

  }

  kmclipm_update_property_double(qc_atmos, QC_PARAM_ZPT_AVG,
			  zpt_avg, QC_PARAM_ZPT_AVG_TXT);
  kmclipm_update_property_double(qc_atmos, QC_PARAM_ZPT_RMS,
			  zpt_rms,  QC_PARAM_ZPT_RMS_TXT);
  cpl_msg_info(cpl_func,"atmos primary header updated");

  kmclipm_update_property_double(qc_bfm, QC_PARAM_ZPT_AVG,
			  zpt_avg,  QC_PARAM_ZPT_AVG_TXT);
  kmclipm_update_property_double(qc_bfm, QC_PARAM_ZPT_RMS,
			  zpt_rms,  QC_PARAM_ZPT_RMS_TXT);
  kmclipm_update_property_double(qc_bfm, QC_PARAM_IMPROV_AVG,
			  improv_avg,  QC_PARAM_IMPROV_AVG_TXT);
  kmclipm_update_property_double(qc_bfm, QC_PARAM_IMPROV_RMS,
			  improv_rms,  QC_PARAM_IMPROV_RMS_TXT);
  cpl_msg_info(cpl_func,"best fit params primary header updated");


  kmclipm_update_property_string(qc_bfp, QC_PARAM_GDAS,
		  gdas_avail,  QC_PARAM_GDAS_TXT);
  kmclipm_update_property_int(qc_bfp, QC_PARAM_FLAG_GDAS,
		  gdas_flag,  QC_PARAM_FLAG_GDAS_TXT);
  kmclipm_update_property_double(qc_bfp, QC_PARAM_ZPT_AVG,
			  zpt_avg,  QC_PARAM_ZPT_AVG_TXT);
  kmclipm_update_property_double(qc_bfp, QC_PARAM_ZPT_RMS,
			  zpt_rms,  QC_PARAM_ZPT_RMS_TXT);
  kmclipm_update_property_double(qc_bfp, QC_PARAM_H2O_AVG,
			  h2o_avg,  QC_PARAM_H2O_AVG_TXT);
  kmclipm_update_property_double(qc_bfp, QC_PARAM_H2O_RMS,
			  h2o_rms,  QC_PARAM_H2O_RMS_TXT);
  kmclipm_update_property_double(qc_bfp, QC_PARAM_RMS_AVG,
			  rms_avg,  QC_PARAM_RMS_AVG_TXT);
  kmclipm_update_property_double(qc_bfp, QC_PARAM_RMS_RMS,
			  rms_rms,  QC_PARAM_RMS_RMS_TXT);

  kmclipm_update_property_double(qc_bfm, QC_PARAM_H2O_RELRMS,
			  (h2o_rms/h2o_avg),  QC_PARAM_H2O_RELRMS_TXT);
  kmclipm_update_property_double(qc_bfp, QC_PARAM_H2O_RELRMS,
		  (h2o_rms/h2o_avg),  QC_PARAM_H2O_RELRMS_TXT);


  if (cpl_propertylist_has(qc_bfm, "ESO TEL AMBI IWV START")){
	  kmclipm_update_property_double(qc_bfm, QC_PARAM_H2O_DELTA,
			  (h2o_avg-meas_iwv),  QC_PARAM_H2O_DELTA_TXT);
	  kmclipm_update_property_double(qc_bfp, QC_PARAM_H2O_DELTA,
			  (h2o_avg-meas_iwv),  QC_PARAM_H2O_DELTA_TXT);
  }

  /*cpl_frame *frame_atmos = kmo_dfs_get_frame(frameset, ATMOS_PARM);
  cpl_frame *frame_bfp = kmo_dfs_get_frame(frameset, BEST_FIT_PARM);
  cpl_frame *frame_bfm = kmo_dfs_get_frame(frameset, BEST_FIT_MODEL);
  cpl_msg_info(cpl_func,"Got atmos frame."); */

  // CHANGE FOR SUPPRESS-EXTENSION

  	  //cpl_sprintf("_%s%s%s", conf->mf.grating.name, conf->mf.grating.name, conf->mf.grating.name);

  //kmo_dfs_save_main_header(frameset, ATMOS_PARM , suffix, NULL, qc_atmos, parlist, cpl_func);
  kmos_molecfit_save(frameset, frameset, parlist, RECIPE_NAME, qc_atmos, ATMOS_PARM, conf->mf.grating.name, conf->mf.suppress_extension, NULL);
  //kmo_dfs_save_main_header(frameset, BEST_FIT_PARM ,suffix, NULL, qc_bfp, parlist, cpl_func);
  kmos_molecfit_save(frameset, frameset, parlist, RECIPE_NAME, qc_bfp, BEST_FIT_PARM,  conf->mf.grating.name, conf->mf.suppress_extension, NULL);
  //kmo_dfs_save_main_header(frameset, BEST_FIT_MODEL , suffix, NULL, qc_bfm, parlist, cpl_func);
  kmos_molecfit_save(frameset, frameset, parlist, RECIPE_NAME, qc_bfm, BEST_FIT_MODEL, conf->mf.grating.name, conf->mf.suppress_extension, NULL);

  const char * suffix= "";

  if (!conf->mf.suppress_extension) {

	  cpl_msg_info(cpl_func, "need to find a suffix ! ");

	  suffix = kmo_dfs_get_suffix(kmo_dfs_get_frame(frameset, ATMOS_PARM), TRUE, FALSE);
  }


  for (cpl_size i = 1; i <= N_IFUS; i++) {

	  j = 2*i-1;

	  cpl_propertylist*  sub_header_atmos = kmclipm_propertylist_load(atmos_file,j);
	  cpl_propertylist*  sub_header_bfp = kmclipm_propertylist_load(bfp_file,j);
	  cpl_propertylist*  sub_header_bfm = kmclipm_propertylist_load(bfm_file,j);

	  cpl_propertylist*  sub_header_atmos_noise =  kmclipm_propertylist_load(atmos_file,j+1);
	  cpl_propertylist*  sub_header_bfp_noise =  kmclipm_propertylist_load(bfp_file,j+1);
	  cpl_propertylist*  sub_header_bfm_noise = kmclipm_propertylist_load(bfm_file,j+1);


	  cpl_table * atmos_tbl = kmclipm_table_load( atmos_file, j, 0 );
	  if (atmos_tbl == NULL){
		  kmos_all_clean_plist(sub_header_atmos) ;
		  kmos_all_clean_plist(sub_header_atmos_noise) ;
	  }
	  cpl_table * bfp_tbl = kmclipm_table_load( bfp_file, j, 0);
	  if (bfp_tbl == NULL){
		  kmos_all_clean_plist(sub_header_bfp) ;
		  kmos_all_clean_plist(sub_header_bfp_noise) ;
	  }
	  cpl_table * bfm_tbl = kmclipm_table_load( bfm_file, j, 0);
	  if (bfm_tbl == NULL){
		  kmos_all_clean_plist(sub_header_bfm) ;
		  kmos_all_clean_plist(sub_header_bfm_noise) ;
	  }

	  cpl_table * atmos_tbl_noise = kmclipm_table_load( atmos_file, j+1, 0 );
	  cpl_table * bfp_tbl_noise = kmclipm_table_load( bfp_file, j+1, 0);
	  cpl_table * bfm_tbl_noise = kmclipm_table_load( bfm_file, j+1, 0);
	  //kmos_all_clean_plist(sub_header_atmos_noise) ;
	  //kmos_all_clean_plist(sub_header_bfp_noise) ;

	  sAtm_e    = kmo_dfs_save_table(atmos_tbl, ATMOS_PARM, suffix ,sub_header_atmos);
	  sBModel_e = kmo_dfs_save_table(bfm_tbl, BEST_FIT_MODEL, suffix ,sub_header_bfm);
	  sBParms_e = kmo_dfs_save_table(bfp_tbl, BEST_FIT_PARM, suffix ,sub_header_bfp);

	  sAtm_e    = kmo_dfs_save_table(atmos_tbl_noise, ATMOS_PARM, suffix ,sub_header_atmos_noise);
	  sBModel_e = kmo_dfs_save_table(bfm_tbl_noise, BEST_FIT_MODEL, suffix ,sub_header_bfm_noise);
	  sBParms_e = kmo_dfs_save_table(bfp_tbl_noise, BEST_FIT_PARM, suffix ,sub_header_bfp_noise);

	  cpl_propertylist_delete(sub_header_atmos);
	  cpl_propertylist_delete(sub_header_bfp);
	  cpl_propertylist_delete(sub_header_bfm);

	  cpl_propertylist_delete(sub_header_atmos_noise);
	  cpl_propertylist_delete(sub_header_bfp_noise);
	  cpl_propertylist_delete(sub_header_bfm_noise);

	  cpl_table_delete(atmos_tbl);
	  cpl_table_delete(bfm_tbl);
	  cpl_table_delete(bfp_tbl);

	  cpl_table_delete(atmos_tbl_noise);
	  cpl_table_delete(bfm_tbl_noise);
	  cpl_table_delete(bfp_tbl_noise);

  }

  /*cpl_frame_delete(frame_atmos);
  cpl_frame_delete(frame_bfp);
  cpl_frame_delete(frame_bfm); */

  unlink(atmos_file);
  unlink(bfp_file);
  unlink(bfm_file);

  /* Cleanup: General configuration of Molecfit */
  if (gdas_user) cpl_table_delete(gdas_user);
  if (atm_profile_standard) cpl_table_delete(atm_profile_standard);
  if (atm_profile_combined) cpl_table_delete(atm_profile_combined);
  if (mf_config) mf_configuration_delete(mf_config);

  /*mf_io_rm(atmos_file);
  mf_io_rm(bfm_file);
  mf_io_rm(bfp_file);*/

  /* Check errors in Molecfit calls */
  if (err != CPL_ERROR_NONE) {
      kmos_molecfit_model_clean(conf);
      return cpl_error_set_message(cpl_func, err,
                                   "Call to Molecfit (MF_MODEL) failed, err_str: %s",
                                   cpl_error_get_message());
  }

  /* Cleanup configuration */
  cpl_msg_info(cpl_func,"Cleaning variables ...");
  kmos_molecfit_model_clean(conf);

  /* Check Recipe status and end */
  cpl_msg_info(cpl_func,"Check errorstate ...");
  if (cpl_errorstate_is_equal(initial_errorstate) && cpl_error_get_code() == CPL_ERROR_NONE ) {
      cpl_msg_info(cpl_func,"Recipe successfully!, Number of IFUs processed (with data): %lld", n_procesed_ifus);
  } else {
      /* Dump the error history since recipe execution start.
       * At this point the recipe cannot recover from the error */
      cpl_errorstate_dump(initial_errorstate, CPL_FALSE, NULL);
      cpl_msg_info(cpl_func,"Recipe failed!, error(%d)=%s", cpl_error_get_code(), cpl_error_get_message());
  }

  cpl_msg_info(cpl_func,"Finish recipe. Returning ...");
  return (int)cpl_error_get_code();
}

/**@}*/

/*----------------------------------------------------------------------------*/
/**
 * @brief Function needed by cpl_recipe_define to fill the input parameters
 *
 * @param  self   parameterlist where you need put parameters
 *
 * @return cpl_error_code
 *
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code kmos_molecfit_model_fill_parameterlist(
    cpl_parameterlist *self)
{
  /* Add the different default parameters to the recipe */
  cpl_errorstate prestate = cpl_errorstate_get();

  /* Fill the parameters list */
  cpl_error_code e;
  cpl_boolean    range     = CPL_TRUE;
  const void     *dummyMin = NULL;
  const void     *dummyMax = NULL;


  /* --KMOS_MOLECFIT_PARAMETER_PROCESS_IFUS */
  const char *process_ifus = "-1";
  e = kmos_molecfit_fill_parameter(RECIPE_NAME, self, KMOS_MOLECFIT_PARAMETER_PROCESS_IFUS,
                                   !range, dummyMin, dummyMax, CPL_TYPE_STRING, (const void *)process_ifus,
                                   "A list of IFUs to process. If set to -1, all the IFUs that have data will be process.", CPL_FALSE);
  if (e != CPL_ERROR_NONE) return (int)e;

  /* --KMOS_MOLECFIT_PARAMETER_SUPPRESS_EXTENSION */
  cpl_boolean suppress_extension = CPL_FALSE;
  e = kmos_molecfit_fill_parameter(RECIPE_NAME, self, KMOS_MOLECFIT_PARAMETER_SUPPRESS_EXTENSION,
                                   !range, dummyMin, dummyMax, CPL_TYPE_BOOL, &suppress_extension,
                                   "Suppress arbitrary filename extension.(TRUE (apply) or FALSE (don't apply).", CPL_FALSE);
  if (e != CPL_ERROR_NONE) return (int)e;

  /* --KMOS_MOLECFIT_PARAMETER_WRANGE */
  const char *wrange_inc = "-1";
  e = kmos_molecfit_fill_parameter(RECIPE_NAME, self, KMOS_MOLECFIT_PARAMETER_WRANGE,
                                   !range, dummyMin, dummyMax, CPL_TYPE_STRING, (const void *)wrange_inc,
                                   "A list of numbers defining the wavelength ranges to fit in the grating. "
                                   "If set to -1, grating dependent default values are used (see manual for reference). ", CPL_FALSE);
  if (e != CPL_ERROR_NONE) return (int)e;

  /* --KMOS_MOLECFIT_PARAMETER_LIST */
  const char *list_molec = "-1";
  e = kmos_molecfit_fill_parameter(RECIPE_NAME, self, KMOS_MOLECFIT_PARAMETER_LIST,
                                   !range, dummyMin, dummyMax, CPL_TYPE_STRING, (const void *)list_molec,
                                   "A list of molecules to fit in grating IZ. "
                                   "If set to -1, grating dependent default values are used (see manual for reference). ", CPL_FALSE);
  if (e != CPL_ERROR_NONE) return (int)e;

  /* --KMOS_MOLECFIT_PARAMETER_FIT */
  const char *fit_molec = "-1";
  e = kmos_molecfit_fill_parameter(RECIPE_NAME, self, KMOS_MOLECFIT_PARAMETER_FIT,
                                   !range, dummyMin, dummyMax, CPL_TYPE_STRING, (const void *)fit_molec,
                                   "Flags to fit the column density of the corresponding list_molec in grating. "
                                   "If set to -1, grating dependent default values are used (see manual for reference). ", CPL_FALSE);
  if (e != CPL_ERROR_NONE) return (int)e;

  /* --KMOS_MOLECFIT_PARAMETER_RELATIVE_VALUE */
  const char *relcol = "-1";
  e = kmos_molecfit_fill_parameter(RECIPE_NAME, self, KMOS_MOLECFIT_PARAMETER_RELATIVE_VALUE,
                                   !range, dummyMin, dummyMax, CPL_TYPE_STRING, (const void *)relcol,
                                   "Column density relative to atmospheric profile of the corresponding list_molec in grating. "
                                   "If set to -1, grating dependent default values are used (see manual for reference). ", CPL_FALSE);
  if (e != CPL_ERROR_NONE) return (int)e;

  /* --MF_PARAMETERS_FTOL */
  double ftol = 0.01; /* Molecfit default: 1e-10 */
  e = kmos_molecfit_fill_parameter(RECIPE_NAME, self, MF_PARAMETERS_FTOL,
                                   !range, dummyMin, dummyMax, CPL_TYPE_DOUBLE, &ftol,
                                   "Relative chi-square convergence criterion.", CPL_FALSE);
  if (e != CPL_ERROR_NONE) return (int)e;

  /* --MF_PARAMETERS_XTOL */
  double xtol = 0.001; /* Molecfit default: 1e-10 */
  e = kmos_molecfit_fill_parameter(RECIPE_NAME, self, MF_PARAMETERS_XTOL,
                                   !range, dummyMin, dummyMax, CPL_TYPE_DOUBLE, &xtol,
                                   "Relative parameter convergence criterion.", CPL_FALSE);
  if (e != CPL_ERROR_NONE) return (int)e;

  /* --MF_PARAMETERS_FIT_CONT */
/*
  cpl_boolean fit_cont = MF_PARAMETERS_FIT_CONTINUUM_INIT;
  e = kmos_molecfit_fill_parameter(RECIPE_NAME, self, MF_PARAMETERS_FIT_CONTINUUM,
                                   !range, dummyMin, dummyMax, CPL_TYPE_BOOL, &fit_cont,
                                   "Flag to enable/disable the polynomial fit of the continuum.", CPL_FALSE);
*/

  char* fit_cont_str_lst = MF_PARAMETERS_FIT_CONTINUUM_INIT;
  e = kmos_molecfit_fill_parameter(RECIPE_NAME, self, MF_PARAMETERS_FIT_CONTINUUM,
                                   !range, dummyMin, dummyMax, CPL_TYPE_BOOL, fit_cont_str_lst,
                                   "Flag to enable/disable the polynomial fit of the continuum.", CPL_FALSE);

  if (e != CPL_ERROR_NONE) return (int)e;

  /* --MF_PARAMETERS_CONT_N */
  int cont_n = 1; /* Molecfit default: 0 */
  e = kmos_molecfit_fill_parameter(RECIPE_NAME, self, MF_PARAMETERS_CONTINUUM_N,
                                   !range, dummyMin, dummyMax, CPL_TYPE_INT, &cont_n,
                                   "Degree of polynomial continuum fit.", CPL_FALSE);
  if (e != CPL_ERROR_NONE) return (int)e;

  /* --MF_PARAMETERS_FIT_WLC */
  cpl_boolean fit_wlc = CPL_TRUE;
  e = kmos_molecfit_fill_parameter(RECIPE_NAME, self, MF_PARAMETERS_FIT_WLC,
                                   !range, dummyMin, dummyMax, CPL_TYPE_BOOL, &fit_wlc,
                                   "Flag to enable/disable the refinement of the wavelength solution.", CPL_FALSE);
  if (e != CPL_ERROR_NONE) return (int)e;

  /* --MF_PARAMETERS_FIT_WLC */
  int wlc_n = 2; /* Molecfit default: 1 */
  e = kmos_molecfit_fill_parameter(RECIPE_NAME, self, MF_PARAMETERS_WLC_N,
                                   !range, dummyMin, dummyMax, CPL_TYPE_INT, &wlc_n,
                                   "Polynomial degree of the refined wavelength solution.", CPL_FALSE);
  if (e != CPL_ERROR_NONE) return (int)e;

  /* --MF_PARAMETERS_WLC_CONST */
  double wlc_const = 0.;
  e = kmos_molecfit_fill_parameter(RECIPE_NAME, self, MF_PARAMETERS_WLC_CONST,
                                   !range, dummyMin, dummyMax, CPL_TYPE_DOUBLE, &wlc_const,
                                   "Initial constant term for wavelength adjustment (shift relative to half wavelength range).", CPL_FALSE);
  if (e != CPL_ERROR_NONE) return (int)e;

  /* --KMOS_MOLECFIT_PARAMETER_USE_INPUT_KERNEL */
  cpl_boolean use_input_kernel = CPL_TRUE;
  e = kmos_molecfit_fill_parameter(RECIPE_NAME, self, KMOS_MOLECFIT_PARAMETER_USE_INPUT_KERNEL,
                                   !range, dummyMin, dummyMax, CPL_TYPE_BOOL, &use_input_kernel,
                                   "The parameters below are ignored if use_input_kernel.", CPL_FALSE);
  if (e != CPL_ERROR_NONE) return (int)e;

  /* --MF_PARAMETERS_FIT_RES_BOX */
  cpl_boolean fit_res_box = CPL_FALSE; /* Molecfit default: CPL_TRUE */
  e = kmos_molecfit_fill_parameter(RECIPE_NAME, self, MF_PARAMETERS_FIT_RES_BOX,
                                   !range, dummyMin, dummyMax, CPL_TYPE_BOOL, &fit_res_box,
                                   "Fit resolution by Boxcar LSF.", CPL_FALSE);
  if (e != CPL_ERROR_NONE) return (int)e;

  /* --MF_PARAMETERS_REL_RES_BOX */
  double relres_box_min = MF_PARAMETERS_RES_BOX_MIN;
  double relres_box_max = MF_PARAMETERS_RES_BOX_MAX;
  double relres_box     = 0.;  /* Molecfit default: 1. */
  e = kmos_molecfit_fill_parameter(RECIPE_NAME, self, MF_PARAMETERS_RES_BOX,
                                   range, &relres_box_min, &relres_box_max, CPL_TYPE_DOUBLE, &relres_box,
                                   "Initial value for FWHM of Boxcar rel. to slit width (Range between 0 and 2).", CPL_FALSE);
  if (e != CPL_ERROR_NONE) return (int)e;

  /* --MF_PARAMETERS_FIT_GAUSS */
  cpl_boolean fit_res_gauss = MF_PARAMETERS_FIT_GAUSS_INIT;
  e = kmos_molecfit_fill_parameter(RECIPE_NAME, self, MF_PARAMETERS_FIT_GAUSS,
                                   !range, dummyMin, dummyMax, CPL_TYPE_BOOL, &fit_res_gauss,
                                   "Fit resolution by Gaussian.", CPL_FALSE);
  if (e != CPL_ERROR_NONE) return (int)e;

  /* --MF_PARAMETERS_RES_GAUSS */
  //double res_gauss_min = MF_PARAMETERS_RES_GAUSS_MIN;
  //double res_gauss_max = MF_PARAMETERS_RES_GAUSS_MAX;
  double res_gauss     = -1.; /* Molecfit default: 1. */
  e = kmos_molecfit_fill_parameter(RECIPE_NAME, self, MF_PARAMETERS_RES_GAUSS,
                                   !range, dummyMin, dummyMax, CPL_TYPE_DOUBLE, &res_gauss,
                                   "Initial value for FWHM of the Gaussian in pixels (Default = -1: IZ=1.84, YJ=1.82, H=1.76, K=1.73, HK=2.06).", CPL_FALSE);
  if (e != CPL_ERROR_NONE) return (int)e;

  /* --MF_PARAMETERS_RES_LORENTZ */
  cpl_boolean fit_res_lorentz = CPL_FALSE; /* Molecfit default: CPL_TRUE */
  e = kmos_molecfit_fill_parameter(RECIPE_NAME, self, MF_PARAMETERS_FIT_LORENTZ,
                                   !range, dummyMin, dummyMax, CPL_TYPE_BOOL, &fit_res_lorentz,
                                   "Fit resolution by Lorentz.", CPL_FALSE);
  if (e != CPL_ERROR_NONE) return (int)e;

  /* --MF_PARAMETERS_FIT_LORENTZ */
  //double res_lorentz_min = MF_PARAMETERS_RES_LORENTZ_MIN;
  //double res_lorentz_max = MF_PARAMETERS_RES_LORENTZ_MAX;
  double res_lorentz     = 0.5; /* Molecfit default: 1. */
  e = kmos_molecfit_fill_parameter(RECIPE_NAME, self, MF_PARAMETERS_RES_LORENTZ,
                                   !range, dummyMin, dummyMax, CPL_TYPE_DOUBLE, &res_lorentz,
                                   "Initial value for FWHM of the Lorentz in pixels.", CPL_FALSE);
  if (e != CPL_ERROR_NONE) return (int)e;

  /* --MF_PARAMETERS_KERN_MODE */
  cpl_boolean kernmode = CPL_FALSE;
  e = kmos_molecfit_fill_parameter(RECIPE_NAME, self, MF_PARAMETERS_KERN_MODE,
                                   !range, dummyMin, dummyMax, CPL_TYPE_BOOL, &kernmode,
                                   "Voigt profile approx. or independent Gauss and Lorentz.", CPL_FALSE);
  if (e != CPL_ERROR_NONE) return (int)e;

  /* --MF_PARAMETERS_KERN_FAC */
  double kernfac = 5.; /* Molecfit default: 3. */
  e = kmos_molecfit_fill_parameter(RECIPE_NAME, self, MF_PARAMETERS_KERN_FAC,
                                   !range, dummyMin, dummyMax, CPL_TYPE_DOUBLE, &kernfac,
                                   "Size of Gaussian/Lorentz/Voigt kernel in FWHM.", CPL_FALSE);
  if (e != CPL_ERROR_NONE) return (int)e;

  /* --MF_PARAMETERS_VAR_KERN */
  cpl_boolean varkern = CPL_FALSE;
  e = kmos_molecfit_fill_parameter(RECIPE_NAME, self, MF_PARAMETERS_VAR_KERN,
                                   !range, dummyMin, dummyMax, CPL_TYPE_BOOL, &varkern,
                                   "Does the kernel size increase linearly with wavelength?.", CPL_FALSE);
  if (e != CPL_ERROR_NONE) return (int)e;


  /* --MF_PARAMETERS_EXPERT_MODE */
  cpl_boolean flag = MF_PARAMETERS_EXPERT_MODE_INIT;
  e = kmos_molecfit_fill_parameter(RECIPE_NAME, self, MF_PARAMETERS_EXPERT_MODE,
                                   !range, dummyMin, dummyMax, CPL_TYPE_BOOL, &flag,
                                   MF_PARAMETERS_EXPERT_MODE_DESC, CPL_FALSE);
  if (e != CPL_ERROR_NONE) return (int)e;



  /* Check possible errors */
  if (!cpl_errorstate_is_equal(prestate)) {
      return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                   "kmos_molecfit_model_fill_parameterlist failed!");
  }

  return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief  Check the string variables defined by the user (wave_range and molecules definition)
 *
 * @param  grating    Struct with the variables to check
 *
 * @return cpl_error_code
 *
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code kmos_molecfit_model_check_wave_molec_conf(
    kmos_grating *grating)
{
  /* Check input */
  cpl_error_ensure(grating, CPL_ERROR_NULL_INPUT,
                   return CPL_ERROR_NULL_INPUT, "grating input is NULL!");

  cpl_error_ensure(grating->wave_range && grating->list_molec && grating->fit_molec && grating->rel_col, CPL_ERROR_NULL_INPUT,
                   return CPL_ERROR_NULL_INPUT, "values of grating input are NULL!");

  /* Init variables */
  char     **tokens;
  cpl_size n_tokens;


  /*** Check wave_range variable ***/
  if (strcmp(grating->wave_range, "-1") != 0) {

      tokens = kmos_molecfit_str_split(grating->wave_range, ",", &n_tokens);

      /* Any token? */
      if (!tokens || n_tokens <= 0) {
          if (tokens) kmos_molecfit_str_array_delete(tokens);
          return cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                                       "Unexpected error getting .wave_ranges!");
      }

      /* At least 2 parameters, Check even number of parameters */
      if (n_tokens % 2 != 0) {
          kmos_molecfit_str_array_delete(tokens);
          return cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                                       ".wave_range haven't a even number of parameters");
      }

      /* Range of wavelenghts and direction */
      for (cpl_size i = 0; i < n_tokens / 2; i++) {

          cpl_size idx   = i * 2;
          double   start = strtod(tokens[idx],     NULL);
          double   end   = strtod(tokens[idx + 1], NULL);

          /* Check the wavelength range - range of KMOS between 0.8 to 2.5 (um) */
          if (start < KMOS_WAVELENGTH_START || start > KMOS_WAVELENGTH_END || end < KMOS_WAVELENGTH_START || end > KMOS_WAVELENGTH_END) {
              kmos_molecfit_str_array_delete(tokens);
              return cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                                           "wavelength in '.wave_range' out of the valid range [%g,%g]", KMOS_WAVELENGTH_START, KMOS_WAVELENGTH_END);
          } else if (start > end) {
              kmos_molecfit_str_array_delete(tokens);
              return cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                                           "wavelength in '.wave_range' starts before end , valid range [%g,%g]", KMOS_WAVELENGTH_START, KMOS_WAVELENGTH_END);
          }
      }
      kmos_molecfit_str_array_delete(tokens);
  }

  /*** Check molec_list variable ***/
  if (strcmp(grating->list_molec, "-1") != 0 ){

      tokens = kmos_molecfit_str_split(grating->list_molec, ",", &n_tokens);

      /* Any token? */
      if (!tokens || n_tokens <= 0) {
          kmos_molecfit_str_array_delete(tokens);
          return cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                                       "It's necessary at least one molecule in .list_molec");
      }

      /* Check if the molecs are valid for Molecfit */
      for (cpl_size i = 0; i < n_tokens; i++) {
          if (mf_molecules_str_check(tokens[i]) != CPL_ERROR_NONE) {
              kmos_molecfit_str_array_delete(tokens);
              return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                           "Molecule non-valid for Molecfit in .list_molec");
          }
      }
      kmos_molecfit_str_array_delete(tokens);
  }

  /*** Check fit_molec variable ***/
  if (strcmp(grating->fit_molec, "-1") != 0) {

      tokens = kmos_molecfit_str_split(grating->fit_molec, ",", &n_tokens);

      /* Any token? */
      if (!tokens || n_tokens <= 0) {
          kmos_molecfit_str_array_delete(tokens);
          return cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                                       "It's necessary the same number of values in .list_molec and .fit_molec");
      }

      /* Boolean value? */
      for (cpl_size i = 0; i < n_tokens; i++) {
          int fit_molec = atoi(tokens[i]);

          if (fit_molec != 0 && fit_molec != 1) {
              kmos_molecfit_str_array_delete(tokens);
              return cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                                           "Unexpected value in .fit_molec it'll be boolean, 0 or 1");
          }
      }
      kmos_molecfit_str_array_delete(tokens);
  }

  /*** Check relcol variable ***/
  if (strcmp(grating->rel_col, "-1") != 0) {

      tokens = kmos_molecfit_str_split(grating->rel_col, ",", &n_tokens);

      /* Any token? */
      if (!tokens || n_tokens <= 0) {
          kmos_molecfit_str_array_delete(tokens);
          return cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                                       "It's necessary the same number of values in .list_molec and .relcol");
      }

      /* Range between 0. to 1. ? */
      for (cpl_size j = 0; j < n_tokens; j++) {
          double relcol = strtod(tokens[j], NULL);
          if (relcol < 0. || relcol > 1.) {
              kmos_molecfit_str_array_delete(tokens);
              return cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                                           "Unexpected value in .relcol, it's a fraction. It'll be < 0. or > 1.");
          }
      }
      kmos_molecfit_str_array_delete(tokens);
  }


  return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief  Generate the internal configuration file for the recipe and check values
 *
 * @param  list   parameterlist with the parameters
 *
 * @return configuration file or NULL if exist an error
 *
 */
/*----------------------------------------------------------------------------*/
static kmos_molecfit_model_parameter * kmos_molecfit_model_conf(
    const cpl_parameterlist *list)
{
  /* Check input */
  cpl_error_ensure(list, CPL_ERROR_NULL_INPUT,
                   return NULL, "list input is NULL!");

  /* Get preState */
  cpl_errorstate preState = cpl_errorstate_get();
  const cpl_parameter *p;


  /* Create the configuration parameter */
  kmos_molecfit_model_parameter *conf = (kmos_molecfit_model_parameter *)cpl_malloc(sizeof(kmos_molecfit_model_parameter));
  kmos_molecfit_nullify(&(conf->mf));

  /* Initialize input parameters propertylist */
  conf->mf.parms = cpl_propertylist_new();


  /*** Mapping IFUs in kmos_molecfit_model recipe, correspondence 1 to 1 ***/
  for (cpl_size n_ifu = 0; n_ifu < N_IFUS; n_ifu++) {
      conf->mf.ifus[n_ifu].map = n_ifu + 1;
  }


  /*** What IFUs Do I need to process? ***/
  p = cpl_parameterlist_find_const(list, KMOS_MOLECFIT_PARAMETER_PROCESS_IFUS);
  conf->process_ifus = cpl_parameter_get_string(p);
  if (strcmp(conf->process_ifus, "-1") == 0) {

      /* Process all IFUs by default */
      cpl_msg_info(cpl_func, "Processing all IFUs (by default) ... ");
      for (cpl_size n_ifu = 0; n_ifu < N_IFUS; n_ifu++) {
          conf->mf.ifus[n_ifu].process = CPL_TRUE;
      }

  } else {

      /* Initialize process IFUs to CPL_FALSE */
      for (cpl_size n_ifu = 0; n_ifu < N_IFUS; n_ifu++) {
          conf->mf.ifus[n_ifu].process = CPL_FALSE;
      }

      /* Init variables */
      cpl_size  n_tokens;
      char **tokens = kmos_molecfit_str_split(conf->process_ifus, ",", &n_tokens);

      /* Any token? */
      if (!tokens || n_tokens <= 0) {
          kmos_molecfit_str_array_delete(tokens);
          cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                                "proccess_IFU empty. It's necessary to process some IFU");
          return NULL;
      }

      /* Set process IFUs */
      for (cpl_size i = 0; i < n_tokens; i++) {
          int n_ifu = atoi(tokens[i]);

          /* Number of IFU correct */
          if (n_ifu < 1 || n_ifu > 24) {
              kmos_molecfit_str_array_delete(tokens);
              cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                                    "Incorrect value in process_ifus. It need to be from 1 to 24");
              return NULL;
          }

          /* Process this IFU */
          cpl_msg_info(cpl_func, "Processing IFU.%02d (assigned by the user) ... ", n_ifu);
          conf->mf.ifus[n_ifu - 1].process = CPL_TRUE;
      }
      kmos_molecfit_str_array_delete(tokens);
  }

  /*** Suppress prefix ***/
  p = cpl_parameterlist_find_const(list, KMOS_MOLECFIT_PARAMETER_SUPPRESS_EXTENSION);
  conf->mf.suppress_extension = cpl_parameter_get_bool(p);

  /*** Wavelenght ranges & molecules: Defined by the user ****/

  /* Get string iwht the list of wavelenght for the grating defined by the user */
  p = cpl_parameterlist_find_const(list, KMOS_MOLECFIT_PARAMETER_WRANGE);
  conf->mf.grating.wave_range = cpl_parameter_get_string(p);

  /* Get string with the list of molecules for the grating defined by the user */
  p = cpl_parameterlist_find_const(list, KMOS_MOLECFIT_PARAMETER_LIST);
  conf->mf.grating.list_molec = cpl_parameter_get_string(p);

  /* Get string with the fit list of molecules for this grating */
  p = cpl_parameterlist_find_const(list, KMOS_MOLECFIT_PARAMETER_FIT);
  conf->mf.grating.fit_molec = cpl_parameter_get_string(p);

  /* Get string with the relcol list of molecules for this grating */
  p = cpl_parameterlist_find_const(list, KMOS_MOLECFIT_PARAMETER_RELATIVE_VALUE);
  conf->mf.grating.rel_col = cpl_parameter_get_string(p);

  /* Check values defined by the user */
  if (kmos_molecfit_model_check_wave_molec_conf(&(conf->mf.grating)) != CPL_ERROR_NONE) {
      kmos_molecfit_model_clean(conf);
      return NULL;
  }


  /*** Convergence criterion ***/

  p = cpl_parameterlist_find_const(list, MF_PARAMETERS_FTOL);
  conf->ftol = cpl_parameter_get_double(p);
  if (conf->ftol < 1.e-10) {
      kmos_molecfit_model_clean(conf);
      cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                            ".ftol out of the valid range");
      return NULL;
  }

  p = cpl_parameterlist_find_const(list, MF_PARAMETERS_XTOL);
  conf->xtol = cpl_parameter_get_double(p);
  if (conf->xtol < 1.e-10) {
      kmos_molecfit_model_clean(conf);
      cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                            ".xtol out of the valid range");
      return NULL;
  }


  /*** Continuum ***/

  p = cpl_parameterlist_find_const(list, MF_PARAMETERS_FIT_CONTINUUM);
  conf->mf.fit_continuum.fit = cpl_parameter_get_bool(p);
  if (cpl_error_get_code() != CPL_ERROR_NONE) {
      kmos_molecfit_model_clean(conf);
      cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                            "Unexpected value in .fit_cont it'll be boolean, 0 or 1");
      return NULL;
  }

  p = cpl_parameterlist_find_const(list, MF_PARAMETERS_CONTINUUM_N);
  conf->mf.fit_continuum.n = cpl_parameter_get_int(p);
  if (conf->mf.fit_continuum.n < 0 || conf->mf.fit_continuum.n > MOLECFIT_MAX_POLY_FIT) {
      kmos_molecfit_model_clean(conf);
      cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                            ".cont_n out of the valid range");
      return NULL;
  }


  /*** Wavelength solution fit/adjustment ***/

  p = cpl_parameterlist_find_const(list, MF_PARAMETERS_FIT_WLC);
  conf->mf.fit_wavelenght.fit = cpl_parameter_get_bool(p);
  if (cpl_error_get_code() != CPL_ERROR_NONE) {
      kmos_molecfit_model_clean(conf);
      cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                            "Unexpected value in .fit_wlc it'll be boolean, 0 or 1");
      return NULL;
  }

  p = cpl_parameterlist_find_const(list, MF_PARAMETERS_WLC_N);
  conf->mf.fit_wavelenght.n = cpl_parameter_get_int(p);
  if (conf->mf.fit_wavelenght.n < 1 || conf->mf.fit_wavelenght.n > MOLECFIT_MAX_POLY_FIT) {
      kmos_molecfit_model_clean(conf);
      cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                            ".wlc_n out of the valid range");
      return NULL;
  }

  p = cpl_parameterlist_find_const(list, MF_PARAMETERS_WLC_CONST);
  conf->mf.fit_wavelenght.const_val = cpl_parameter_get_double(p);
  if (conf->mf.fit_wavelenght.const_val < 0.) {
      kmos_molecfit_model_clean(conf);
      cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                            ".wlc_const out of the valid range");
      return NULL;
  }


  /*** User input kernel ? ***/
  p = cpl_parameterlist_find_const(list, KMOS_MOLECFIT_PARAMETER_USE_INPUT_KERNEL);
  conf->mf.use_input_kernel = cpl_parameter_get_bool(p);
  if (cpl_error_get_code() != CPL_ERROR_NONE) {
      kmos_molecfit_model_clean(conf);
      cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                            "Unexpected value in .use_input_kernel it'll be boolean, 0 or 1");
      return NULL;
  }


  /*** Default kernel: Boxcar kernel ***/
  p = cpl_parameterlist_find_const(list, MF_PARAMETERS_FIT_RES_BOX);
  conf->boxcar.fit = cpl_parameter_get_bool(p);
  if (cpl_error_get_code() != CPL_ERROR_NONE) {
      kmos_molecfit_model_clean(conf);
      cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                            "Unexpected value in .fit_res_box it'll be boolean, 0 or 1");
      return NULL;
  }
  p = cpl_parameterlist_find_const(list, MF_PARAMETERS_RES_BOX);
  conf->boxcar.res = cpl_parameter_get_double(p);
  if (conf->boxcar.res < MF_PARAMETERS_RES_BOX_MIN || conf->boxcar.res > MF_PARAMETERS_RES_BOX_MAX) {
      kmos_molecfit_model_clean(conf);
      cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                            ".relres_box out of the valid range");
      return NULL;
  }


  /*** Default kernel: Gaussian kernel ***/
  p = cpl_parameterlist_find_const(list, MF_PARAMETERS_FIT_GAUSS);
  conf->gauss.fit = cpl_parameter_get_bool(p);
  if (cpl_error_get_code() != CPL_ERROR_NONE) {
      kmos_molecfit_model_clean(conf);
      cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                            "Unexpected value in .fit_res_gauss it'll be boolean, 0 or 1");
      return NULL;
  }
  p = cpl_parameterlist_find_const(list, MF_PARAMETERS_RES_GAUSS);
  conf->gauss.res = cpl_parameter_get_double(p);
  if (conf->gauss.res != -1. && conf->gauss.res < 0.01) {
      kmos_molecfit_model_clean(conf);
      cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                            ".res_gauss out of the valid range");
      return NULL;
  }

  /*** Default kernel: Lorentz kernel ***/
  p = cpl_parameterlist_find_const(list, MF_PARAMETERS_FIT_LORENTZ);
  conf->lorentz.fit = cpl_parameter_get_bool(p);
  if (cpl_error_get_code() != CPL_ERROR_NONE) {
      kmos_molecfit_model_clean(conf);
      cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                            "Unexpected value in .fit_res_lorentz it'll be boolean, 0 or 1");
      return NULL;
  }
  p = cpl_parameterlist_find_const(list, MF_PARAMETERS_RES_LORENTZ);
  conf->lorentz.res = cpl_parameter_get_double(p);
  if (conf->lorentz.res < 0. || conf->lorentz.res > 100.) {
      kmos_molecfit_model_clean(conf);
      cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                            ".res_lorentz out of the valid range");
      return NULL;
  }


  /*** Default kernels: Generic parameters ***/

  p = cpl_parameterlist_find_const(list, MF_PARAMETERS_KERN_MODE);
  conf->mf.kernmode = cpl_parameter_get_bool(p);
  if (cpl_error_get_code() != CPL_ERROR_NONE) {
      kmos_molecfit_model_clean(conf);
      cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                            "Unexpected value in .kernmode it'll be boolean, 0 or 1");
      return NULL;
  }

  p = cpl_parameterlist_find_const(list, MF_PARAMETERS_KERN_FAC);
  conf->mf.kernfac = cpl_parameter_get_double(p);
  if (conf->mf.kernfac < 3. || conf->mf.kernfac > 300.) {
      kmos_molecfit_model_clean(conf);
      cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                            ".kernfac out of the valid range");
      return NULL;
  }

  p = cpl_parameterlist_find_const(list, MF_PARAMETERS_VAR_KERN);
  conf->mf.varkern = cpl_parameter_get_bool(p);
  if (cpl_error_get_code() != CPL_ERROR_NONE) {
      kmos_molecfit_model_clean(conf);
      cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                            "Unexpected value in .varkern it'll be boolean, 0 or 1");
      return NULL;
  }


  /* Save parameter in the output propertylist */

  cpl_propertylist_update_string(conf->mf.parms, MF_PARAMETERS_CONTEX_DEFAULT" "KMOS_MOLECFIT_KEYWORD_PROCESS_IFUS,       conf->process_ifus);

  cpl_propertylist_update_bool(  conf->mf.parms, MF_PARAMETERS_CONTEX_DEFAULT" "KMOS_MOLECFIT_KEYWORD_SUPPRESS_EXTENSION, conf->mf.suppress_extension);

  cpl_propertylist_update_string(conf->mf.parms, MF_PARAMETERS_CONTEX_DEFAULT" "KMOS_MOLECFIT_KEYWORD_WRANGE,             conf->mf.grating.wave_range);
  cpl_propertylist_update_string(conf->mf.parms, MF_PARAMETERS_CONTEX_DEFAULT" "KMOS_MOLECFIT_KEYWORD_LIST,               conf->mf.grating.list_molec);
  cpl_propertylist_update_string(conf->mf.parms, MF_PARAMETERS_CONTEX_DEFAULT" "KMOS_MOLECFIT_KEYWORD_FIT,                conf->mf.grating.fit_molec);
  cpl_propertylist_update_string(conf->mf.parms, MF_PARAMETERS_CONTEX_DEFAULT" "KMOS_MOLECFIT_KEYWORD_RELATIVE_VALUE,     conf->mf.grating.rel_col);

  cpl_propertylist_update_double(conf->mf.parms, MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_FTOL,                       conf->ftol);
  cpl_propertylist_update_double(conf->mf.parms, MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_XTOL,                       conf->xtol);

  cpl_propertylist_update_bool(  conf->mf.parms, MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_FIT_CONTINUUM,              conf->mf.fit_continuum.fit);
  cpl_propertylist_update_int(   conf->mf.parms, MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_CONTINUUM_N,                conf->mf.fit_continuum.n);

  cpl_propertylist_update_bool(  conf->mf.parms, MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_FIT_WLC,                    conf->mf.fit_wavelenght.fit);
  cpl_propertylist_update_int(   conf->mf.parms, MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_WLC_N,                      conf->mf.fit_wavelenght.n);
  cpl_propertylist_update_double(conf->mf.parms, MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_WLC_CONST,                  conf->mf.fit_wavelenght.const_val);

  cpl_propertylist_update_bool(  conf->mf.parms, MF_PARAMETERS_CONTEX_DEFAULT" "KMOS_MOLECFIT_KEYWORD_USE_INPUT_KERNEL,   conf->mf.use_input_kernel);

  cpl_propertylist_update_bool(  conf->mf.parms, MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_KERN_MODE,                  conf->mf.kernmode);
  cpl_propertylist_update_double(conf->mf.parms, MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_KERN_FAC,                   conf->mf.kernfac);
  cpl_propertylist_update_bool(  conf->mf.parms, MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_VAR_KERN,                   conf->mf.varkern);

  cpl_propertylist_update_bool(  conf->mf.parms, MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_FIT_RES_BOX,                conf->boxcar.fit);
  cpl_propertylist_update_double(conf->mf.parms, MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_RES_BOX,                    conf->boxcar.res);

  cpl_propertylist_update_bool(  conf->mf.parms, MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_FIT_GAUSS,                  conf->gauss.fit);
  cpl_propertylist_update_double(conf->mf.parms, MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_RES_GAUSS,                  conf->gauss.res);

  cpl_propertylist_update_bool(  conf->mf.parms, MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_FIT_LORENTZ,                conf->lorentz.fit);
  cpl_propertylist_update_double(conf->mf.parms, MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_RES_LORENTZ,                conf->lorentz.res);


  /* Check status */
  if (!cpl_errorstate_is_equal(preState)) {
      /* Configuration failed */
      kmos_molecfit_model_clean(conf);
      return NULL;
  } else {
      /* Configuration successfully */
      return conf;
  }
}

/*----------------------------------------------------------------------------*/
/**
 * @brief  Get the input GDAS profile if the user provide
 *
 * @param  frameset   the frames list
 *
 * @return cpl_error_code
 *
 */
/*----------------------------------------------------------------------------*/
static cpl_table * kmos_molecfit_model_get_gdas_profile(
    cpl_frameset *frameset)
{
  /* Check input */
  cpl_error_ensure(frameset, CPL_ERROR_NULL_INPUT,
                   return NULL, "kmos_molecfit_model_get_gdas_profile inputs NULL!");

  cpl_table *gdas_user = NULL;

  /* Check if exist cpl_frame GDAS */
  const cpl_frame *frmGDAS = cpl_frameset_find(frameset, GDAS);
  if (!frmGDAS) {

      cpl_msg_info(cpl_func, "Not %s user profile provided.", GDAS);

  } else {

      const char *fileGDAS = cpl_frame_get_filename(frmGDAS);
      cpl_msg_info(cpl_func, "Loading '%s' %s user profile ...", fileGDAS, GDAS);

      gdas_user = cpl_table_load(fileGDAS, 1, 0);
      cpl_error_ensure(gdas_user, cpl_error_get_code(),
                       return NULL, "Cannot load %s data from the file '%s'!",
                       GDAS, fileGDAS);
      cpl_msg_info(cpl_func, "cpl_table *gdas_user profile loaded :");
      cpl_table_dump(gdas_user, 0, cpl_table_get_nrow(gdas_user), NULL);
  }

  return gdas_user;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief  Get the input ATM_PROFILE_STANDARD if the user provide
 *
 * @param  frameset   the frames list
 *
 * @return cpl_error_code
 *
 */
/*----------------------------------------------------------------------------*/
static cpl_table * kmos_molecfit_model_get_atm_profile_standard(
    cpl_frameset *frameset)
{
  /* Check input */
  cpl_error_ensure(frameset, CPL_ERROR_NULL_INPUT,
                   return NULL, "kmos_molecfit_model_get_atm_profile_standard inputs NULL!");

  cpl_table *atm_profile_standard = NULL;

  /* Check if exist cpl_frame ATM_PROFILE_STANDARD */
  const cpl_frame *frm_atm_profile_standard = cpl_frameset_find(frameset, ATM_PROFILE_STANDARD);
  if (!frm_atm_profile_standard) {

      cpl_msg_info(cpl_func, "Not %s user profile provided.", ATM_PROFILE_STANDARD);

  } else {

      const char *file_atm_profile_standard = cpl_frame_get_filename(frm_atm_profile_standard);
      cpl_msg_info(cpl_func, "Loading '%s' %s user atm_profile_standard ...", file_atm_profile_standard, ATM_PROFILE_STANDARD);

      atm_profile_standard = cpl_table_load(file_atm_profile_standard, 1, 0);
      cpl_error_ensure(atm_profile_standard, cpl_error_get_code(),
                       return NULL, "Cannot load %s data from the file '%s'!",
                       ATM_PROFILE_STANDARD, file_atm_profile_standard);
      cpl_msg_info(cpl_func, "cpl_table *atm_profile_standard loaded :");
      cpl_table_dump(atm_profile_standard, 0, cpl_table_get_nrow(atm_profile_standard), NULL);
  }

  return atm_profile_standard;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Function needed to fill the molecfit configuration file
 *
 * @param  conf   Recipe configuration.
 * @param  ifu    Concrete IFU for whitch generate the molecfit configuration
 * @param  type   Concrete grating
 *
 * @return parameterlist with contain the config to molecfit or NULL if error
 *
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code kmos_molecfit_model_mf_conf(
    kmos_molecfit_model_parameter *conf,
    kmos_spectrum                 *ifu,
    kmos_grating_type             type,
    mf_parameters_config          *config_parameters)
{
  /* Check inputs */
  cpl_error_ensure(conf && ifu, CPL_ERROR_NULL_INPUT,
                   return CPL_ERROR_NULL_INPUT, "Any input is NULL!");

  /*** Building generic configuration molecfic file ***/
  cpl_error_code err = kmos_molecfit_conf_generic(&(conf->mf), config_parameters);
  if (err != CPL_ERROR_NONE) return err;


  /*** Set molecfit configuration with recipe parameters ***/

  config_parameters->fitting.ftol                     = conf->ftol;
  config_parameters->fitting.xtol                     = conf->xtol;

  config_parameters->fitting.fit_continuum.fit        = conf->mf.fit_continuum.fit;
  config_parameters->fitting.fit_continuum.n          = conf->mf.fit_continuum.n;
  config_parameters->fitting.fit_continuum.const_val  = ifu->median;
  cpl_msg_info(cpl_func,"--.cont_const = %g", config_parameters->fitting.fit_continuum.const_val);

  config_parameters->fitting.fit_wavelenght.fit       = conf->mf.fit_wavelenght.fit;
  config_parameters->fitting.fit_wavelenght.n         = conf->mf.fit_wavelenght.n;
  config_parameters->fitting.fit_wavelenght.const_val = conf->mf.fit_wavelenght.const_val;

  config_parameters->fitting.fit_res_box.fit          = conf->boxcar.fit;
  config_parameters->fitting.fit_res_box.const_val    = conf->boxcar.res;

  config_parameters->fitting.fit_gauss.fit            = conf->gauss.fit;
  config_parameters->fitting.fit_gauss.const_val      = conf->gauss.res;

  if (config_parameters->fitting.fit_gauss.const_val == -1.) {
      switch(type) {
        case GRATING_IZ: config_parameters->fitting.fit_gauss.const_val = RES_GAUSS_IZ; break;
        case GRATING_YJ: config_parameters->fitting.fit_gauss.const_val = RES_GAUSS_YJ; break;
        case GRATING_H:  config_parameters->fitting.fit_gauss.const_val = RES_GAUSS_H;  break;
        case GRATING_K:  config_parameters->fitting.fit_gauss.const_val = RES_GAUSS_K;  break;
        case GRATING_HK: config_parameters->fitting.fit_gauss.const_val = RES_GAUSS_HK; break;
      }
      cpl_msg_info(cpl_func,"--.res_gauss by default = %g",  config_parameters->fitting.fit_gauss.const_val);
  } else {
      cpl_msg_info(cpl_func,"--.res_gauss by the user = %g", config_parameters->fitting.fit_gauss.const_val);
  }

  config_parameters->fitting.fit_lorentz.fit          = conf->lorentz.fit;
  config_parameters->fitting.fit_lorentz.const_val    = conf->lorentz.res;


  /*** PARAMETERS NOT INCLUDED IN THE RECIPE: HARD-CODED ***/

  config_parameters->inputs.transmission                        = MF_PARAMETERS_TRANSMISSION_TRUE;
  config_parameters->inputs.default_error                       = 0.01;
  config_parameters->fitting.flux_unit                          = MF_PARAMETERS_FLUX_UNIT_NO_CONVERSION;

  config_parameters->fitting.fit_telescope_background.fit       = CPL_FALSE;   /* Molecfit default: CPL_TRUE */
  config_parameters->fitting.fit_telescope_background.const_val = 0.1;

  if (config_parameters->atmospheric.ref_atm) cpl_free(config_parameters->atmospheric.ref_atm);
  config_parameters->atmospheric.ref_atm                        = cpl_sprintf("%s", MF_PARAMETERS_REFERENCE_ATMOSPHERIC_INIT);

  if (config_parameters->atmospheric.gdas_prof) cpl_free(config_parameters->atmospheric.gdas_prof);
  config_parameters->atmospheric.gdas_prof                      = cpl_sprintf("%s", MF_PARAMETERS_GDAS_PROFILE_AUTO);

  config_parameters->atmospheric.layers                         = CPL_TRUE;
  config_parameters->atmospheric.emix                           =  5.;
  config_parameters->atmospheric.pwv                            = -1.;


  return CPL_ERROR_NONE;
}

/*-------------------------------------------------------------
 * ---------------*/
/**
 * @brief    Deallocate the given parameter configuration object and its contents
 *
 * @param    conf       The parameter configuration variable in the recipe.
 */
/*----------------------------------------------------------------------------*/
static void kmos_molecfit_model_clean(
    kmos_molecfit_model_parameter *conf)
{
  if (conf) {

      kmos_molecfit_clean(&(conf->mf));

      cpl_free(conf);
  }
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Insert QC parameters in the headers of the IFUs with valid data
 *
 * @param  res_table   Molecfit result table with the parameters of best-fitting.
 * @param  spec_out    Molecfit best fitting table data
 * @param  pl_BParams  input/output Header to BEST_FIT_PARM  FITS file output
 * @param  pl_BModel   input/output Header to BEST_FIT_MODEL FITS file output
 *
 * @return CPL_ERROR_NONE without error or CPL_ERROR_DATA_NOT_FOUND if it doesn't exist the correct columns
 *
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code kmos_molecfit_model_headers_fill_qc_parameters(
    const cpl_table  *res_table,
    const cpl_table  *spec_out,
    cpl_propertylist *pl_BParms,
    cpl_propertylist *pl_BModel,
	cpl_vector 		 *qc_zpt,
	cpl_vector 		 *qc_h2o,
	cpl_vector 		 *qc_rms,
	cpl_vector 		 *qc_improv,
	/*cpl_vector 		 *qc_meas_iwv,
	cpl_vector 		 *qc_meas_iwv30d,*/
	int 			  k)
{


  /* Check column inputs */
  if (   !cpl_table_has_column(res_table, MF_COL_PARAMETER        )
      || !cpl_table_has_column(res_table, MF_COL_VALUE            )
      || !cpl_table_has_column(spec_out,  MF_COL_IN_FLUX          )
      || !cpl_table_has_column(spec_out,  MF_COL_OUT_TELLURIC_CORR)
      || !cpl_table_has_column(spec_out,  MF_COL_MOD_SCALE        ) ){

      return CPL_ERROR_DATA_NOT_FOUND;
  }


  /*cpl_msg_info(cpl_func, "Result Table structure .");
  cpl_propertylist_dump(pl_BParms,stdout);

  cpl_msg_info(cpl_func, "Best Fitting Table data");
  cpl_propertylist_dump(pl_BModel, stdout); */

  /*** Fill the BEST_FIT_PARM FITS header file with the QC parameters ***/
  /* cpl_table_dump(res_table, 0, cpl_table_get_nrow(res_table), NULL); */

  /* Get parameters and values of the best fit model results table */
  const char   **parameter = cpl_table_get_data_string_const(res_table, MF_COL_PARAMETER);
  const double *value      = cpl_table_get_data_double_const(res_table, MF_COL_VALUE    );

  /* Write the QC BEST_FIT_PARM parameters */
  cpl_size nrows_res_table = cpl_table_get_nrow(res_table);
  for (cpl_size i = 0; i < nrows_res_table; i++) {
      if (parameter[i]) {
          cpl_msg_info(cpl_func, "Parameter: %s, Value = %g", parameter[i], value[i]);

          if (!strcmp(parameter[i], KMOS_MOLECFIT_RESULTS_ROW_RMS_REL_TO_MEAN)) {
              cpl_propertylist_update_double( pl_BParms, KMOS_MOLECFIT_QC_PARAM_RMS,    value[i]                         );
              cpl_propertylist_set_comment(   pl_BParms, KMOS_MOLECFIT_QC_PARAM_RMS,    KMOS_MOLECFIT_QC_PARAM_RMS_TXT   );
              cpl_vector_set(qc_rms, k, value[i]);

          }

          if (!strcmp(parameter[i], KMOS_MOLECFIT_RESULTS_ROW_RMS_STATUS)) {
              cpl_propertylist_update_double( pl_BParms, KMOS_MOLECFIT_QC_PARAM_STATUS, value[i]                         );
              cpl_propertylist_set_comment(   pl_BParms, KMOS_MOLECFIT_QC_PARAM_STATUS, KMOS_MOLECFIT_QC_PARAM_STATUS_TXT);
          }

          if (!strcmp(parameter[i], KMOS_MOLECFIT_RESULTS_ROW_RMS_H2O_COL_MM)) {
              cpl_propertylist_update_double( pl_BParms, KMOS_MOLECFIT_QC_PARAM_H2O,    value[i]                         );
              cpl_propertylist_set_comment(   pl_BParms, KMOS_MOLECFIT_QC_PARAM_H2O,    KMOS_MOLECFIT_QC_PARAM_H2O_TXT   );
              cpl_vector_set(qc_h2o, k ,value[i]);
          }
      }
  }

  /* Check QC BEST_FIT_PARM parameters */
  if (   !cpl_propertylist_has(pl_BParms, KMOS_MOLECFIT_QC_PARAM_RMS   )
      || !cpl_propertylist_has(pl_BParms, KMOS_MOLECFIT_QC_PARAM_STATUS)
      || !cpl_propertylist_has(pl_BParms, KMOS_MOLECFIT_QC_PARAM_H2O   ) ){

      return CPL_ERROR_DATA_NOT_FOUND;
  }

  if (cpl_propertylist_has(pl_BParms, QC_ZEROPOINT) ){
	  cpl_vector_set(qc_zpt, k, cpl_propertylist_get_double(pl_BParms,QC_ZEROPOINT));
  }

  /*** Fill the BEST_FIT_MODEL FITS header file with the QC parameters ***/
  cpl_size nrows_spec_out = cpl_table_get_nrow(spec_out);
  double   RMS_NF         =  1.;
  double   RMS_NTF        = -1.;

  /* Create new columns to the calculate table and select to zero for convert in valid */
  cpl_table *copy_spec_out = cpl_table_duplicate(spec_out);
  cpl_table_new_column(copy_spec_out, KMOS_MOLECFIT_CALCULE_COLUMN_NF,  CPL_TYPE_DOUBLE);
  cpl_table_new_column(copy_spec_out, KMOS_MOLECFIT_CALCULE_COLUMN_NTF, CPL_TYPE_DOUBLE);
  for (cpl_size i = 0; i < nrows_spec_out; i++) {
      cpl_table_set_double(copy_spec_out, KMOS_MOLECFIT_CALCULE_COLUMN_NF,  i, 0.);
      cpl_table_set_double(copy_spec_out, KMOS_MOLECFIT_CALCULE_COLUMN_NTF, i, 0.);
  }

  /* Apply the selection: mtrans > 0.05 and extract columns (Avoiding problems with values very close zero) */
  cpl_table_unselect_all(copy_spec_out);
  cpl_table_or_selected_double(copy_spec_out, MF_COL_OUT_TELLURIC_CORR, CPL_GREATER_THAN, 0.05);
  cpl_msg_info(cpl_func, "Number of select columns (%s > 0) = %lld ", MF_COL_OUT_TELLURIC_CORR, cpl_table_count_selected(copy_spec_out));
  cpl_table *calculate_spec_out = cpl_table_extract_selected(copy_spec_out);

  /* cpl_table_dump(calculate_spec_out, 0, cpl_table_get_nrow(calculate_spec_out), NULL); */

  /* Remove the temporary copy of spec_out table */
  cpl_table_delete(copy_spec_out);

  /* Calculate RMS_NF and RMS_NTF */
  if (cpl_table_get_nrow(calculate_spec_out) > 0) {

      /* Calculate NF column and RMS_NF */
      const double *cflux = cpl_table_get_data_double_const(calculate_spec_out, MF_PARAMETERS_COLUMN_FLUX_DEFAULT);
      cpl_table_copy_data_double(calculate_spec_out, KMOS_MOLECFIT_CALCULE_COLUMN_NF,  cflux);
      cpl_table_divide_columns(  calculate_spec_out, KMOS_MOLECFIT_CALCULE_COLUMN_NF,  MF_COL_MOD_SCALE);

      /* Calculate NTF column and RMS_NTF */
      const double *cNF = cpl_table_get_data_double_const(calculate_spec_out, KMOS_MOLECFIT_CALCULE_COLUMN_NF);
      cpl_table_copy_data_double(calculate_spec_out, KMOS_MOLECFIT_CALCULE_COLUMN_NTF, cNF);
      cpl_table_divide_columns(  calculate_spec_out, KMOS_MOLECFIT_CALCULE_COLUMN_NTF, MF_COL_OUT_TELLURIC_CORR);

      /* Calculate the standard deviation */
      RMS_NF  = cpl_table_get_column_stdev(calculate_spec_out, KMOS_MOLECFIT_CALCULE_COLUMN_NF);
      RMS_NTF = cpl_table_get_column_stdev(calculate_spec_out, KMOS_MOLECFIT_CALCULE_COLUMN_NTF);

      cpl_msg_info(cpl_func, "RMS_NF  = %g", RMS_NF);
      cpl_msg_info(cpl_func, "RMS_NTF = %g", RMS_NTF);

  } else {

      cpl_msg_warning(cpl_func, "Any values in the column MSCAL < 0");
  }

  /* Remove the temporary calculate of spec_out table */
  cpl_table_delete(calculate_spec_out);

  /* Write the QC BEST_FIT_MODEL parameters */
  double improv = RMS_NTF / RMS_NF;
  cpl_propertylist_update_double( pl_BModel, KMOS_MOLECFIT_QC_PARAM_IMPROV, improv);
  cpl_propertylist_set_comment(   pl_BModel, KMOS_MOLECFIT_QC_PARAM_IMPROV, KMOS_MOLECFIT_QC_PARAM_IMPROV_TXT);
  cpl_msg_info(cpl_func, "HIERACH ESO QC IMPROV = %g ", improv);
  cpl_vector_set(qc_improv,k,improv);


  return CPL_ERROR_NONE;
}
