# ========================================================================================================
# Jose A. Escartin
# 2019.04.05
#
# A python interactive window for use in the KMOS Reflex workflow.
# This interactive window shows the results of the kmos_molecfit_calctrans recipe.   
# From this interactive Actor the the user can modify pipeline parameters and re-initiate the processing.
#
#
# Pipeline Parameters for inclusion:
#
#    --USE_INPUT_KERNEL      : In order to use kernel library provide by the user. [TRUE]
#    --IFU_1                 : IFU number in ATMOS_PARM and BEST_FIT_PARM to be used to compute the telluric correction for IFU 1 . [-1]
#    --IFU_2                 : IFU number in ATMOS_PARM and BEST_FIT_PARM to be used to compute the telluric correction for IFU 2 . [-1]
#    --IFU_3                 : IFU number in ATMOS_PARM and BEST_FIT_PARM to be used to compute the telluric correction for IFU 3 . [-1]
#    --IFU_4                 : IFU number in ATMOS_PARM and BEST_FIT_PARM to be used to compute the telluric correction for IFU 4 . [-1]
#    --IFU_5                 : IFU number in ATMOS_PARM and BEST_FIT_PARM to be used to compute the telluric correction for IFU 5 . [-1]
#    --IFU_6                 : IFU number in ATMOS_PARM and BEST_FIT_PARM to be used to compute the telluric correction for IFU 6 . [-1]
#    --IFU_7                 : IFU number in ATMOS_PARM and BEST_FIT_PARM to be used to compute the telluric correction for IFU 7 . [-1]
#    --IFU_8                 : IFU number in ATMOS_PARM and BEST_FIT_PARM to be used to compute the telluric correction for IFU 8 . [-1]
#    --IFU_9                 : IFU number in ATMOS_PARM and BEST_FIT_PARM to be used to compute the telluric correction for IFU 9 . [-1]
#    --IFU_10                : IFU number in ATMOS_PARM and BEST_FIT_PARM to be used to compute the telluric correction for IFU 10. [-1]
#    --IFU_11                : IFU number in ATMOS_PARM and BEST_FIT_PARM to be used to compute the telluric correction for IFU 11. [-1]
#    --IFU_12                : IFU number in ATMOS_PARM and BEST_FIT_PARM to be used to compute the telluric correction for IFU 12. [-1]
#    --IFU_13                : IFU number in ATMOS_PARM and BEST_FIT_PARM to be used to compute the telluric correction for IFU 13. [-1]
#    --IFU_14                : IFU number in ATMOS_PARM and BEST_FIT_PARM to be used to compute the telluric correction for IFU 14. [-1]
#    --IFU_15                : IFU number in ATMOS_PARM and BEST_FIT_PARM to be used to compute the telluric correction for IFU 15. [-1]
#    --IFU_16                : IFU number in ATMOS_PARM and BEST_FIT_PARM to be used to compute the telluric correction for IFU 16. [-1]
#    --IFU_17                : IFU number in ATMOS_PARM and BEST_FIT_PARM to be used to compute the telluric correction for IFU 17. [-1]
#    --IFU_18                : IFU number in ATMOS_PARM and BEST_FIT_PARM to be used to compute the telluric correction for IFU 18. [-1]
#    --IFU_19                : IFU number in ATMOS_PARM and BEST_FIT_PARM to be used to compute the telluric correction for IFU 19. [-1]
#    --IFU_20                : IFU number in ATMOS_PARM and BEST_FIT_PARM to be used to compute the telluric correction for IFU 20. [-1]
#    --IFU_21                : IFU number in ATMOS_PARM and BEST_FIT_PARM to be used to compute the telluric correction for IFU 21. [-1]
#    --IFU_22                : IFU number in ATMOS_PARM and BEST_FIT_PARM to be used to compute the telluric correction for IFU 22. [-1]
#    --IFU_23                : IFU number in ATMOS_PARM and BEST_FIT_PARM to be used to compute the telluric correction for IFU 23. [-1]
#    --IFU_24                : IFU number in ATMOS_PARM and BEST_FIT_PARM to be used to compute the telluric correction for IFU 24. [-1]
#    --suppress_extension    : Suppress arbitrary filename extension.(TRUE (apply) or FALSE (don't apply). [FALSE]
#	 --SCALE_PWV			 : Sscale the Precipitable Water Vapor (PWV). 
#
#
# Images with possible plots (TAG's): 
#
#    * STAR_SPEC/EXTRACT_SPEC/SCIENCE/SCI_RECONSTRUCTED     The input data
#    * ATMOS_PARM                                           kmos_molecfit_model - Atmospheric parameters
#    * BEST_FIT_PARM                                        kmos_molecfit_model - Best fit parameters
#    * TELLURIC_DATA                                        Telluric correction and intermediate products for each IFU. Output fits file with 48 ext. (24-data and 24-error, in TABLE format).
#    * TELLURIC_CORR                                        Telluric correction for each IFU. Output fits file with 48 ext. (24-data and 24-error, in IMAGE format).
#
# ========================================================================================================
from __future__ import with_statement
from __future__ import absolute_import
from __future__ import print_function
import sys

try:
    import numpy
    import re
    import reflex
    from pipeline_product import PipelineProduct
    import pipeline_display
    import reflex_plot_widgets
    import matplotlib.gridspec as gridspec
    import_success = True
except ImportError:
    import_success = False
    print("Error importing modules pyfits, wx, matplotlib, numpy")

def paragraph(text, width=None):
    """ wrap text string into paragraph
       text:  text to format, removes leading space and newlines
       width: if not None, wraps text, not recommended for tooltips as
              they are wrapped by wxWidgets by default
    """
    import textwrap
    if width is None:
        return textwrap.dedent(text).replace('\n', ' ').strip()
    else:
        return textwrap.fill(textwrap.dedent(text), width=width)


class DataPlotterManager(object):
    # static members
    recipe_name = "kmos_molecfit_calctrans"
    telluric_data_cat = "TELLURIC_DATA"

    def setWindowTitle(self):
        return self.recipe_name+"_interactive"

    def setInteractiveParameters(self):
        return [
            
            reflex.RecipeParameter(recipe=self.recipe_name, displayName="USE_INPUT_KERNEL",
                    group="Inputs", description="In order to use kernel library provide by the user"),
            reflex.RecipeParameter(recipe=self.recipe_name, displayName="suppress_extension",
                    group="Inputs", description="Suppress arbitrary filename extension.(TRUE (apply) or FALSE (don't apply)"),
            reflex.RecipeParameter(recipe=self.recipe_name, displayName="SCALE_PWV",
                    group="Inputs", description="Apply transmission function scaling for percipitable water vapor.(AUTO (apply) or  NONE (don't apply)"),

            reflex.RecipeParameter(recipe=self.recipe_name, displayName="IFU_1",
                    group="IFUs-MATCH", description="IFU number in ATMOS_PARM and BEST_FIT_PARM to be used to compute the telluric correction for IFU 1"),
            reflex.RecipeParameter(recipe=self.recipe_name, displayName="IFU_2",
                    group="IFUs-MATCH", description="IFU number in ATMOS_PARM and BEST_FIT_PARM to be used to compute the telluric correction for IFU 2"),
            reflex.RecipeParameter(recipe=self.recipe_name, displayName="IFU_3",
                    group="IFUs-MATCH", description="IFU number in ATMOS_PARM and BEST_FIT_PARM to be used to compute the telluric correction for IFU 3"),
            reflex.RecipeParameter(recipe=self.recipe_name, displayName="IFU_4",
                    group="IFUs-MATCH", description="IFU number in ATMOS_PARM and BEST_FIT_PARM to be used to compute the telluric correction for IFU 4"),
            reflex.RecipeParameter(recipe=self.recipe_name, displayName="IFU_5",
                    group="IFUs-MATCH", description="IFU number in ATMOS_PARM and BEST_FIT_PARM to be used to compute the telluric correction for IFU 5"),
            reflex.RecipeParameter(recipe=self.recipe_name, displayName="IFU_6",
                    group="IFUs-MATCH", description="IFU number in ATMOS_PARM and BEST_FIT_PARM to be used to compute the telluric correction for IFU 6"),
            reflex.RecipeParameter(recipe=self.recipe_name, displayName="IFU_7",
                    group="IFUs-MATCH", description="IFU number in ATMOS_PARM and BEST_FIT_PARM to be used to compute the telluric correction for IFU 7"),
            reflex.RecipeParameter(recipe=self.recipe_name, displayName="IFU_8",
                    group="IFUs-MATCH", description="IFU number in ATMOS_PARM and BEST_FIT_PARM to be used to compute the telluric correction for IFU 8"),
            reflex.RecipeParameter(recipe=self.recipe_name, displayName="IFU_9",
                    group="IFUs-MATCH", description="IFU number in ATMOS_PARM and BEST_FIT_PARM to be used to compute the telluric correction for IFU 9"),
            reflex.RecipeParameter(recipe=self.recipe_name, displayName="IFU_10",
                    group="IFUs-MATCH", description="IFU number in ATMOS_PARM and BEST_FIT_PARM to be used to compute the telluric correction for IFU 10"),
            reflex.RecipeParameter(recipe=self.recipe_name, displayName="IFU_11",
                    group="IFUs-MATCH", description="IFU number in ATMOS_PARM and BEST_FIT_PARM to be used to compute the telluric correction for IFU 11"),
            reflex.RecipeParameter(recipe=self.recipe_name, displayName="IFU_12",
                    group="IFUs-MATCH", description="IFU number in ATMOS_PARM and BEST_FIT_PARM to be used to compute the telluric correction for IFU 12"),
            reflex.RecipeParameter(recipe=self.recipe_name, displayName="IFU_13",
                    group="IFUs-MATCH", description="IFU number in ATMOS_PARM and BEST_FIT_PARM to be used to compute the telluric correction for IFU 13"),
            reflex.RecipeParameter(recipe=self.recipe_name, displayName="IFU_14",
                    group="IFUs-MATCH", description="IFU number in ATMOS_PARM and BEST_FIT_PARM to be used to compute the telluric correction for IFU 14"),
            reflex.RecipeParameter(recipe=self.recipe_name, displayName="IFU_15",
                    group="IFUs-MATCH", description="IFU number in ATMOS_PARM and BEST_FIT_PARM to be used to compute the telluric correction for IFU 15"),
            reflex.RecipeParameter(recipe=self.recipe_name, displayName="IFU_16",
                    group="IFUs-MATCH", description="IFU number in ATMOS_PARM and BEST_FIT_PARM to be used to compute the telluric correction for IFU 16"),
            reflex.RecipeParameter(recipe=self.recipe_name, displayName="IFU_17",
                    group="IFUs-MATCH", description="IFU number in ATMOS_PARM and BEST_FIT_PARM to be used to compute the telluric correction for IFU 17"),
            reflex.RecipeParameter(recipe=self.recipe_name, displayName="IFU_18",
                    group="IFUs-MATCH", description="IFU number in ATMOS_PARM and BEST_FIT_PARM to be used to compute the telluric correction for IFU 18"),
            reflex.RecipeParameter(recipe=self.recipe_name, displayName="IFU_19",
                    group="IFUs-MATCH", description="IFU number in ATMOS_PARM and BEST_FIT_PARM to be used to compute the telluric correction for IFU 19"),
            reflex.RecipeParameter(recipe=self.recipe_name, displayName="IFU_20",
                    group="IFUs-MATCH", description="IFU number in ATMOS_PARM and BEST_FIT_PARM to be used to compute the telluric correction for IFU 20"),
            reflex.RecipeParameter(recipe=self.recipe_name, displayName="IFU_21",
                    group="IFUs-MATCH", description="IFU number in ATMOS_PARM and BEST_FIT_PARM to be used to compute the telluric correction for IFU 21"),
            reflex.RecipeParameter(recipe=self.recipe_name, displayName="IFU_22",
                    group="IFUs-MATCH", description="IFU number in ATMOS_PARM and BEST_FIT_PARM to be used to compute the telluric correction for IFU 22"),
            reflex.RecipeParameter(recipe=self.recipe_name, displayName="IFU_23",
                    group="IFUs-MATCH", description="IFU number in ATMOS_PARM and BEST_FIT_PARM to be used to compute the telluric correction for IFU 23"),
            reflex.RecipeParameter(recipe=self.recipe_name, displayName="IFU_24",
                    group="IFUs-MATCH", description="IFU number in ATMOS_PARM and BEST_FIT_PARM to be used to compute the telluric correction for IFU 24"),
            
        ]

    def readFitsData(self, fitsFiles):
        self.frames = dict()
        for f in fitsFiles:
            self.frames[f.category] = PipelineProduct(f)

        # Two cases: the file category is found or not found.
        # Define the plotting functions in both cases
        if self.telluric_data_cat in self.frames:
            # Get the wished files
            telluric_data = self.frames[self.telluric_data_cat]
           
            # Initialise
            self.telluric_data_extnames = []
            self.spec_data_lambda = dict()
            self.spec_data_flux = dict()
            self.spec_data_cflux = dict()
            self.spec_data_mtrans = dict()
            self.spec_units = dict()
            self.spec_iwv = -1
            if ( 'ESO TEL AMBI IWV START' in telluric_data.all_hdu[0].header): 
            	self.spec_iwv =  telluric_data.all_hdu[0].header['ESO TEL AMBI IWV START']          
            
            
            # Loop on all extensions of std_image to find the std stars extensions
            for telluric_data_ext in telluric_data.all_hdu:
                
                naxis = telluric_data_ext.header['NAXIS']
                
                # NAXIS is 1 if there is an spectrum, 0 otherwise
                if (naxis == 2):
                    
                    # extname is like IFU.3.DATA 
                    extname = telluric_data_ext.header['EXTNAME']
                    self.telluric_data_extnames.append(extname)

                    # Get infos from telluric_data using the extname
                    self.spec_data_lambda[extname] = telluric_data_ext.data.field("lambda")
                    self.spec_data_flux[extname] = telluric_data_ext.data.field("flux")
                    self.spec_data_cflux[extname] = telluric_data_ext.data.field("cflux")
                    self.spec_data_mtrans[extname] = telluric_data_ext.data.field("mtrans")
                    
                    self.spec_units[extname] = telluric_data_ext.header['ESO QC CUBE_UNIT']

            # Set the plotting functions
            self._add_subplots = self._add_subplots
            self._plot = self._data_plot
        else:
            self.telluric_data_extnames = []
            # Set the plotting functions to NODATA ones
            self._add_subplots = self._add_nodata_subplots
            self._plot = self._nodata_plot

    def addSubplots(self, figure):
        self._add_subplots(figure)

    def plotProductsGraphics(self):
        self._plot()

    def plotWidgets(self) :
        widgets = list()
        # Radio button
        self.radiobutton = reflex_plot_widgets.InteractiveRadioButtons(self.axradiobutton, self.setRadioCallback, 
                self.telluric_data_extnames, 0, title='IFU selection')
        widgets.append(self.radiobutton)
        return widgets

    def setRadioCallback(self, label) :
          
        # Plot Spectrum1
        specdisp1 = pipeline_display.SpectrumDisplay()
        self.spec1_plot.clear()
        
        pwv_corr=''         
        #get pwv-scaling information out. 
        if ((self.spec_iwv is not -1)): #(self.get_pcc_Value('SCALE_PWV') is 'auto') and 
            pwv_corr='$\n$not scaled with pwv- check header keywords.'	                 
        
        specdisp1.setLabels(r"$\lambda$["+"$\mu$m]" + " (blue: extracted standard star spectrum, red: corrected spectrum) "+pwv_corr, self._process_label(self.spec_units[label]) )
        specdisp1.display(self.spec1_plot, "Extracted spectrum", self._data_plot1_get_tooltip(label), 
                self.spec_data_lambda[label], self.spec_data_flux[label])
        specdisp1.overplot(self.spec1_plot, self.spec_data_lambda[label], self.spec_data_cflux[label], 'red')

        # Plot Spectrum2
        specdisp2 = pipeline_display.SpectrumDisplay()
        self.spec2_plot.clear()
        specdisp2.setLabels(r"$\lambda$["+"$\mu$m]", "Atmospheric transmission - with" )
        specdisp2.display(self.spec2_plot, "Telluric correction", self._data_plot2_get_tooltip(label), 
                self.spec_data_lambda[label], self.spec_data_mtrans[label])
        

    def _add_subplots(self, figure):
        gs = gridspec.GridSpec(2, 4)
        self.axradiobutton = figure.add_subplot(gs[0:2, 0  ])
        self.spec1_plot    = figure.add_subplot(gs[0,   1:4])
        self.spec2_plot    = figure.add_subplot(gs[1,   1:4])
        
    def _data_plot1_get_tooltip(self, extname):
        # Create the tooltip
        tooltip = " \
            IFU.EXT  : %s \n \
            blue      : Extracted standard star spectrum \n \
            red       : Best fit molecfit model" % (extname)
        return tooltip
        

    def _data_plot2_get_tooltip(self, extname):
        # Create the tooltip
        tooltip = " \
            IFU.EXT  : %s" % (extname)
        return tooltip
        

    def _data_plot(self):
        
        extname = self.telluric_data_extnames[0]
        
        pwv_corr=''         
        #get pwv-scaling information out. 
        if ((self.get_pcc_Value('SCALE_PWV') is 'auto') and (self.spec_iwv is not -1)):
            pwv_corr='not scaled with pwv- check header keywords.'	
          
        
        # Plot Spectrum1
        specdisp1 = pipeline_display.SpectrumDisplay()
        specdisp1.setLabels(r"$\lambda$["+"$\mu$m]" + " (blue: extracted standard star spectrum, red: corrected spectrum) "+pwv_corr, self._process_label(self.spec_units[extname]) )
        specdisp1.display(self.spec1_plot, "Extracted spectrum", self._data_plot1_get_tooltip(extname), 
                         self.spec_data_lambda[extname], self.spec_data_flux[extname])
        specdisp1.overplot(self.spec1_plot, self.spec_data_lambda[extname], self.spec_data_cflux[extname], 'red')

        
        # Plot Spectrum2
        specdisp2 = pipeline_display.SpectrumDisplay()
        specdisp2.setLabels(r"$\lambda$["+"$\mu$m]",  "Atmospheric transmission")
        specdisp2.display(self.spec2_plot, "Telluric correction", self._data_plot2_get_tooltip(extname), 
                         self.spec_data_lambda[extname], self.spec_data_mtrans[extname])
        

    def _process_label(self, in_label):
        # If known, 'pretty print' the label
        if (in_label == "erg.s**(-1).cm**(-2).angstrom**(-1)"):
            return "Flux [erg sec" + r"$^{-1}$"+"cm" + r"$^{-2}$" + r"$\AA^{-1}$]"
        else:
            return in_label

    def _add_nodata_subplots(self, figure):

        gs = gridspec.GridSpec(2, 4)
        self.axradiobutton = figure.add_subplot(gs[0:2, 0  ])
        self.spec1_plot    = figure.add_subplot(gs[0,   1:4])
        self.spec2_plot    = figure.add_subplot(gs[1,   1:4])

    def _nodata_plot(self):
        self.spec1_plot.set_axis_off()
        text_nodata = "Data not found\nMissing type:\n%s" % self.telluric_data_cat
        self.spec1_plot.text(0.1, 0.6, text_nodata, color='#11557c',
                      fontsize=18, ha='left', va='center', alpha=1.0)
        self.spec1_plot.tooltip = 'No data found'
        
 
     # ---------------------------------------------------------------------------------------
    def get_pcc(self, param):
        if hasattr( self, 'recipe_name') and hasattr( self, 'params_by_name') :
            if (self.recipe_name,param) in self.params_by_name.keys() :
                return self.params_by_name[(self.recipe_name,param)].paramChangeCtrl
        return None
    # ---------------------------------------------------------------------------------------
 # ---------------------------------------------------------------------------------------
    def get_pcc_Value(self, param, default_value=None):
        pcc=self.get_pcc(param)
        if pcc is None : return default_value
        valtypes={
            'double': float,
            'int': int,
            'bool': bool,
        }
        if hasattr( self, 'params_by_name') :
            return valtypes.get(self.params_by_name[(self.recipe_name,param)].parameter.valtype,str)(pcc.GetValue())
        else :
            return pcc.GetValue()
    # ---------------------------------------------------------------------------------------

#This is the 'main' function
if __name__ == '__main__':
    from reflex_interactive_app import PipelineInteractiveApp

    # Create interactive application
    interactive_app = PipelineInteractiveApp(enable_init_sop=True)

    # get inputs from the command line
    interactive_app.parse_args()

    #Check if import failed or not
    if not import_success:
        interactive_app.setEnableGUI(False)

    #Open the interactive window if enabled
    if interactive_app.isGUIEnabled():
        #Get the specific functions for this window
        dataPlotManager = DataPlotterManager()

        interactive_app.setPlotManager(dataPlotManager)
        interactive_app.showGUI()
    else:
        interactive_app.set_continue_mode()

    #Print outputs. This is parsed by the Reflex python actor to
    #get the results. Do not remove
    interactive_app.print_outputs()
    sys.exit()
