# ========================================================================================================
# Jose A. Escartin
# 2019.04.05
#
# A python interactive window for use in the KMOS Reflex workflow.
# This interactive window shows the results of the kmos_molecfit_model recipe.   
# From this interactive Actor the the user can modify pipeline parameters and re-initiate the processing.
#
#
# Pipeline Parameters for inclusion:
#
#    --PROCESS_IFUS          : A list of IFUs to process. If set to -1, all the IFUs that have data will be process. [-1]
#    --suppress_extension    : Suppress arbitrary filename extension.(TRUE (apply) or FALSE (don't apply). [FALSE]
#    --WAVE_RANGE            : A list of numbers defining the wavelength ranges to fit in the grating. If set to -1, grating dependent default values are used (see manual for reference). [-1]
#    --LIST_MOLEC            : A list of molecules to fit in grating IZ. If set to -1, grating dependent default values are used (see manual for reference). [-1]
#    --FIT_MOLEC             : Flags to fit the column density of the corresponding list_molec in grating. If set to -1, grating dependent default values are used (see manual for reference). [-1]
#    --REL_COL               : Column density relative to atmospheric profile of the corresponding list_molec in grating. If set to -1, grating dependent default values are used (see manual for reference). [-1]
#    --FTOL                  : Relative chi-square convergence criterion. [0.01]
#    --XTOL                  : Relative parameter convergence criterion. [0.001]
#    --FIT_CONTINUUM         : FFlag to enable/disable the polynomial fit of the continuum. [TRUE]
#    --CONTINUUM_N           : Degree of polynomial continuum fit. [1]
#    --FIT_WLC               : Flag to enable/disable the refinement of the wavelength solution. [TRUE]
#    --WLC_N                 : Polynomial degree of the refined wavelength solution. [2]
#    --WLC_CONST             : Initial constant term for wavelength adjustment (shift relative to half wavelength range). [0.0]
#    --USE_INPUT_KERNEL      : The parameters below are ignored if use_input_kernel. [TRUE]
#    --FIT_RES_BOX           : Fit resolution by Boxcar LSF. [FALSE]
#    --RES_BOX               : Initial value for FWHM of Boxcar rel. to slit width (Range between 0 and 2). [0.0]
#    --FIT_RES_GAUSS         : Fit resolution by Gaussian. [TRUE]
#    --RES_GAUSS             : Initial value for FWHM of the Gaussian in pixels (Default = -1: IZ=1.84, YJ=1.82, H=1.76, K=1.73, HK=2.06). [-1.0]
#    --FIT_RES_LORENTZ       : Fit resolution by Lorentz. [FALSE]
#    --RES_LORENTZ           : Initial value for FWHM of the Lorentz in pixels. [0.5]
#    --KERNMODE              : Voigt profile approx. or independent Gauss and Lorentz. [FALSE]
#    --KERNFAC               : Size of Gaussian/Lorentz/Voigt kernel in FWHM. [5.0]
#    --VARKERN               : Does the kernel size increase linearly with wavelength?. [FALSE]
#
#
# Images with possible plots (TAG's): 
#
#    * STAR_SPEC/EXTRACT_SPEC/SCIENCE/SCI_RECONSTRUCTED     The input data
#    * ATMOS_PARM                                           Atmospheric parameters
#    * BEST_FIT_PARM                                        Best fit parameters 
#    * BEST_FIT_MODEL                                       Best fit model
#
#
# ========================================================================================================

from __future__ import with_statement
from __future__ import absolute_import
from __future__ import print_function
import sys

try:
    import numpy
    import re
    import reflex
    from pipeline_product import PipelineProduct
    import pipeline_display
    import reflex_plot_widgets
    import matplotlib.gridspec as gridspec
    import_success = True
except ImportError:
    import_success = False
    print("Error importing modules pyfits, wx, matplotlib, numpy")

def paragraph(text, width=None):
    """ wrap text string into paragraph
       text:  text to format, removes leading space and newlines
       width: if not None, wraps text, not recommended for tooltips as
              they are wrapped by wxWidgets by default
    """
    import textwrap
    if width is None:
        return textwrap.dedent(text).replace('\n', ' ').strip()
    else:
        return textwrap.fill(textwrap.dedent(text), width=width)


class DataPlotterManager(object):
    # static members
    recipe_name = "kmos_molecfit_model"
    star_spec_cat = "STAR_SPEC"
    extract_spec_cat = "EXTRACT_SPEC"
    best_fit_model_cat = "BEST_FIT_MODEL"
    y_scalefactor = 1000

    def setWindowTitle(self):
        return self.recipe_name+"_interactive"

    def setInteractiveParameters(self):
        return [
            
            reflex.RecipeParameter(recipe=self.recipe_name, displayName="suppress_extension",
                    group="Inputs", description="Suppress arbitrary filename extension.(TRUE (apply) or FALSE (don't apply)"),
            reflex.RecipeParameter(recipe=self.recipe_name, displayName="PROCESS_IFUS",
                    group="Inputs", description="A list of IFUs to process. If set to -1, all the IFUs that have data will be process"),
            reflex.RecipeParameter(recipe=self.recipe_name, displayName="WAVE_RANGE",
                    group="Inputs", description="A list of numbers defining the wavelength ranges to fit in the grating. If set to -1, grating dependent default values are used (see manual for reference)"),
            reflex.RecipeParameter(recipe=self.recipe_name, displayName="LIST_MOLEC",
                    group="Inputs", description="A list of molecules to fit in grating IZ. If set to -1, grating dependent default values are used (see manual for reference)"),
            reflex.RecipeParameter(recipe=self.recipe_name, displayName="FIT_MOLEC",
                    group="Inputs", description="Flags to fit the column density of the corresponding list_molec in grating. If set to -1, grating dependent default values are used (see manual for reference)"),
            reflex.RecipeParameter(recipe=self.recipe_name, displayName="REL_COL",
                    group="Inputs", description="Column density relative to atmospheric profile of the corresponding list_molec in grating. If set to -1, grating dependent default values are used (see manual for reference)"),

            reflex.RecipeParameter(recipe=self.recipe_name, displayName="FTOL",
                    group="Fitting", description="Relative chi-square convergence criterion"),
            reflex.RecipeParameter(recipe=self.recipe_name, displayName="XTOL",
                    group="Fitting", description="Relative parameter convergence criterion"),
            reflex.RecipeParameter(recipe=self.recipe_name, displayName="FIT_CONTINUUM",
                    group="Fitting", description="Flag to enable/disable the polynomial fit of the continuum"),
            reflex.RecipeParameter(recipe=self.recipe_name, displayName="CONTINUUM_N",
                    group="Fitting", description="Degree of polynomial continuum fit"),
            reflex.RecipeParameter(recipe=self.recipe_name, displayName="FIT_WLC",
                    group="Fitting", description="Flag to enable/disable the refinement of the wavelength solution"),
            reflex.RecipeParameter(recipe=self.recipe_name, displayName="WLC_N",
                    group="Fitting", description="Polynomial degree of the refined wavelength solution"),
            reflex.RecipeParameter(recipe=self.recipe_name, displayName="WLC_CONST",
                    group="Fitting", description="Initial constant term for wavelength adjustment (shift relative to half wavelength range)"),

            reflex.RecipeParameter(recipe=self.recipe_name, displayName="USE_INPUT_KERNEL",
                    group="Kernel", description="The parameters below are ignored if use_input_kernel"),
            reflex.RecipeParameter(recipe=self.recipe_name, displayName="FIT_RES_BOX",
                    group="Kernel", description="Fit resolution by Boxcar LSF"),
            reflex.RecipeParameter(recipe=self.recipe_name, displayName="RES_BOX",
                    group="Kernel", description="Initial value for FWHM of Boxcar rel. to slit width (Range between 0 and 2)"),
            reflex.RecipeParameter(recipe=self.recipe_name, displayName="FIT_RES_GAUSS",
                    group="Kernel", description="Fit resolution by Gaussian"),
            reflex.RecipeParameter(recipe=self.recipe_name, displayName="RES_GAUSS",
                    group="Kernel", description="Initial value for FWHM of the Gaussian in pixels (Default = -1: IZ=1.84, YJ=1.82, H=1.76, K=1.73, HK=2.06)"),
            reflex.RecipeParameter(recipe=self.recipe_name, displayName="FIT_RES_LORENTZ",
                    group="Kernel", description="Fit resolution by Lorentz"),
            reflex.RecipeParameter(recipe=self.recipe_name, displayName="RES_LORENTZ",
                    group="Kernel", description="Initial value for FWHM of the Lorentz in pixels"),
            reflex.RecipeParameter(recipe=self.recipe_name, displayName="KERNMODE",
                    group="Kernel", description="Voigt profile approx. or independent Gauss and Lorentz"),
            reflex.RecipeParameter(recipe=self.recipe_name, displayName="KERNFAC",
                    group="Kernel", description="Size of Gaussian/Lorentz/Voigt kernel in FWHM"),
            reflex.RecipeParameter(recipe=self.recipe_name, displayName="VARKERN",
                    group="Kernel", description="Does the kernel size increase linearly with wavelength?"),
            
        ]

    def readFitsData(self, fitsFiles):
        self.frames = dict()
        for f in fitsFiles:
            self.frames[f.category] = PipelineProduct(f)

        # Two cases: the file category is found or not found.
        # Define the plotting functions in both cases
        if ((self.star_spec_cat in self.frames or self.extract_spec_cat in self.frames ) and self.best_fit_model_cat in self.frames):
            
            # Get the wished files
            self.orig_star_spec = False
            self.orig_extract_spec = False
            if (self.star_spec_cat in self.frames) :
                orig_spec = self.frames[self.star_spec_cat]
                self.orig_star_spec = True
            else:
                orig_spec = self.frames[self.extract_spec_cat]
                self.orig_extract_spec = True
            
            best_fit_model = self.frames[self.best_fit_model_cat]
           
            # Initialise
            self.star_data_extnames = []
            self.crpix1 = dict()
            self.crval1 = dict()
            self.cdelt1 = dict()
            self.spec_data = dict()
            self.best_fit_model_data = dict()
            
            # READ data
            self.qc_nr_std_stars = 0.
            self.qc_thruput_mean = 0.
            self.qc_thruput_sdv  = 0.
            if (self.orig_star_spec) :
                self.qc_nr_std_stars = orig_spec.all_hdu[0].header['ESO QC NR STD STARS']
                self.qc_thruput_mean = orig_spec.all_hdu[0].header['ESO QC THRUPUT MEAN']
                self.qc_thruput_sdv  = orig_spec.all_hdu[0].header['ESO QC THRUPUT SDV']


            # Loop on all extensions of best_fit_model
            for best_fit_model_ext in best_fit_model.all_hdu:

                # NAXIS is 2 if there is a table, 0 otherwise
                naxis_best_fit_model = best_fit_model_ext.header['NAXIS']
                if (naxis_best_fit_model == 2):

                    # extname is like IFU.3.DATA 
                    extname = best_fit_model_ext.header['EXTNAME']
                    self.star_data_extnames.append(extname)
                    
                    orig_spec_ext = orig_spec.all_hdu[extname]

                    # NAXIS is 1 if there is a spectrum, 0 otherwise
                    naxis_orig = orig_spec_ext.header['NAXIS']
                    if (naxis_orig == 1):

                        # Get infos from orig_spec using the extname
                        self.crpix1[extname] = orig_spec_ext.header['CRPIX1']
                        self.crval1[extname] = orig_spec_ext.header['CRVAL1']
                        self.cdelt1[extname] = orig_spec_ext.header['CDELT1']
                        self.spec_data[extname] = orig_spec_ext.data
                        
                        self.best_fit_model_data[extname] = best_fit_model_ext.data.field("mflux")
                

            # Set the plotting functions
            self._add_subplots = self._add_subplots
            self._plot = self._data_plot
        else:
            self.star_data_extnames = []
            # Set the plotting functions to NODATA ones
            self._add_subplots = self._add_nodata_subplots
            self._plot = self._nodata_plot

    def addSubplots(self, figure):
        self._add_subplots(figure)

    def plotProductsGraphics(self):
        self._plot()

    def plotWidgets(self) :
        widgets = list()
        # Radio button
        self.radiobutton = reflex_plot_widgets.InteractiveRadioButtons(self.axradiobutton, self.setRadioCallback, 
                self.star_data_extnames, 0, title='IFU selection')
        widgets.append(self.radiobutton)
        return widgets

    def setRadioCallback(self, label) :

        # Define wave
        pix = numpy.arange(len(self.spec_data[label]))
        wave = self.crval1[label] + pix * self.cdelt1[label]  
          
        # Plot Spectrum
        specdisp = pipeline_display.SpectrumDisplay()
        self.spec_plot.clear()
        specdisp.setLabels(r"$\lambda$[" + "$\mu$m]" + " (blue: extracted standard star spectrum, red: best fit molecfit model)", "Flux (ADU)   [x"+str(self.y_scalefactor)+"]" )
        specdisp.display(self.spec_plot, "Standard star spectrum", self._data_plot_get_tooltip(label), 
                wave, self.spec_data[label]/self.y_scalefactor) # TO DO: Unify plotting in one function.
        
        self.spec_plot.fill(wave, self.best_fit_model_data[label]/self.y_scalefactor, 'green', alpha=0.3)
        specdisp.overplot(self.spec_plot, wave, self.best_fit_model_data[label]/self.y_scalefactor, 'red')   # TO DO: Unify plotting in one function.

    def _add_subplots(self, figure):
        gs = gridspec.GridSpec(1, 4)
        self.axradiobutton =    figure.add_subplot(gs[0:1, 0  ])
        self.spec_plot =        figure.add_subplot(gs[0:1, 1:4])

    def _data_plot_get_tooltip(self, extname):
        
        # Create the tooltip
        if self.orig_star_spec :
            tooltip = " \
                ESO QC NR STD STARS : %f \n \
                ESO QC THRUPUT MEAN : %f \n \
                ESO QC THRUPUT SDV  : %f \
                " % (self.qc_nr_std_stars, self.qc_thruput_mean, self.qc_thruput_sdv)
        else :
            tooltip = " \
                "

        return tooltip

    def _data_plot(self):
        
        extname = self.star_data_extnames[0]
        
        # Define wave
        pix = numpy.arange(len(self.spec_data[extname]))
        wave = self.crval1[extname] + pix * self.cdelt1[extname]  
        
        # Plot Spectrum
        specdisp = pipeline_display.SpectrumDisplay()
        specdisp.setLabels(r"$\lambda$[" + "$\mu$m]" + " (blue: extracted standard star spectrum, red: best fit molecfit model)", "Flux (ADU)   [x"+str(self.y_scalefactor)+"]" )
        specdisp.display(self.spec_plot, "Standard star spectrum", self._data_plot_get_tooltip(extname), wave, self.spec_data[extname]/self.y_scalefactor) # TO DO: Unify plotting in one function.
        
        self.spec_plot.fill(wave, self.best_fit_model_data[extname]/self.y_scalefactor, 'green', alpha=0.3)
        specdisp.overplot(self.spec_plot, wave, self.best_fit_model_data[extname]/self.y_scalefactor, 'red')

    def _process_label(self, in_label):
        # If known, 'pretty print' the label
        if (in_label == "erg.s**(-1).cm**(-2).angstrom**(-1)"):
            return "Flux [erg sec" + r"$^{-1}$"+"cm" + r"$^{-2}$" + r"$\AA^{-1}$] (x1000)"
        else:
            return in_label

    def _add_nodata_subplots(self, figure):
        gs = gridspec.GridSpec(1, 4)
        self.axradiobutton =    figure.add_subplot(gs[0:1, 0  ])
        self.spec_plot =        figure.add_subplot(gs[0:1, 1:4])

    def _nodata_plot(self):
        self.spec_plot.set_axis_off()
        text_nodata = "Data not found\nMissing type:\n%s" % self.star_spec_cat
        self.spec_plot.text(0.1, 0.6, text_nodata, color='#11557c',
                      fontsize=18, ha='left', va='center', alpha=1.0)
        self.spec_plot.tooltip = 'No data found'

#This is the 'main' function
if __name__ == '__main__':
    from reflex_interactive_app import PipelineInteractiveApp

    # Create interactive application
    interactive_app = PipelineInteractiveApp(enable_init_sop=True)

    # get inputs from the command line
    interactive_app.parse_args()

    #Check if import failed or not
    if not import_success:
        interactive_app.setEnableGUI(False)

    #Open the interactive window if enabled
    if interactive_app.isGUIEnabled():
        #Get the specific functions for this window
        dataPlotManager = DataPlotterManager()

        interactive_app.setPlotManager(dataPlotManager)
        interactive_app.showGUI()
    else:
        interactive_app.set_continue_mode()

    #Print outputs. This is parsed by the Reflex python actor to
    #get the results. Do not remove
    interactive_app.print_outputs()
    sys.exit()
