# ========================================================================================================
# Jose A. Escartin
# 2019.04.11
#
# A python interactive window for use in the KMOS Reflex workflow.
# This interactive window shows the results of the kmos_extract_spec recipe.   
# From this interactive Actor the the user can modify pipeline parameters and re-initiate the processing.
#
#
# Pipeline Parameters for inclusion:
#
#    --ProcessIFU           : A list of IFUs to process. If set to -1, all the IFUs that have data will be process. [-1]
#
#
# Images to plot (TAG's): 
#
#    * EXTRACT_SPEC
#    * SCI_RECONSTRUCTED
#
# ========================================================================================================
  
from __future__ import with_statement
from __future__ import absolute_import
from __future__ import print_function
import sys
try:
    import inspect
    import numpy
    import os
    import re
    import reflex
    from pipeline_product import PipelineProduct
    import pipeline_display
    import reflex_plot_widgets
    import matplotlib.gridspec as gridspec
    from matplotlib.text import Text
    from matplotlib.pyplot import cm
    from pylab import *
    from tokenize import tokenize

    import_success = True
except ImportError:
    import_success = False
    print("Error importing modules pyfits, wx, matplotlib, numpy")

def paragraph(text, width=None):
    """ wrap text string into paragraph
       text:  text to format, removes leading space and newlines
       width: if not None, wraps text, not recommended for tooltips as
              they are wrapped by wxWidgets by default
    """
    import textwrap
    if width is None:
        return textwrap.dedent(text).replace('\n', ' ').strip()
    else:
        return textwrap.fill(textwrap.dedent(text), width=width)

class DataPlotterManager(object):
    
    # static members
    recipe_name  = "select_reference"
    reference_cat = "REFERENCE"
    extract_spec_cat = "EXTRACT_SPEC"
    
    selected_file_idx = 0

    def setCurrentParameterHelper(self, helper):
        self.param_helper = helper

    def setWindowTitle(self):
        return self.recipe_name+"_interactive"

    def setInteractiveParameters(self):
        return [
            reflex.RecipeParameter(recipe="kmos_molecfit_model", displayName="PROCESS_IFUS",
                    group="Inputs", description="A list of IFUs to process. If set to -1, all the IFUs that have data will be process"),
        ]

    def readFitsData(self, fitsFiles):

        # Get PROCESS_IFUS value 
        parser = reflex.ReflexIOParser()
        params = interactive_app.inputs.in_sop
        values = {}
        for param in params:
            values[param.name]= param.value
        self.PROCESS_IFUS = values['PROCESS_IFUS']

        # Initialize data structures
        self.files_extract_spec = []

        # Loop on all FITS input files
        for f in fitsFiles:

            if f.category == self.reference_cat :
                
                reference_product = PipelineProduct(f)
                
                reference_dict = dict()
                
                self.readFitsData_product(reference_product, reference_dict)
                self.file_reference = reference_dict
                
            elif f.category == self.extract_spec_cat :
                
                extract_spec_product = PipelineProduct(f)
                filename = os.path.basename(f.name)

                
                extract_spec_dict = dict()
                extract_spec_dict["CATEGORY"] = f.category
                extract_spec_dict["FILENAME"] = f.name
                extract_spec_dict["PURPOSES"] = f.purposes
                
                self.readFitsData_product(extract_spec_product, extract_spec_dict)
                self.files_extract_spec.append(extract_spec_dict)
                        
            # Set the plotting functions
            self._add_subplots = self._add_subplots
            self._plot = self._data_plot

    def readFitsData_product(self, product, product_dict):

        product_dict["ARCFILE"]  = product.all_hdu[0].header['ARCFILE']

        # Create Entry for the extension
        product_dict["EXTENSIONS"] = []
        
        # Loop on extensions
        for product_ext in product.all_hdu:

            # EXTNAME is missing in the primary header - Skip it anyway
            try:
                extname = product_ext.header['EXTNAME']
            except KeyError:
                continue

            product_ext_dict = dict()

            # Get the IFU number from extname to get the IFU status
            m = re.search(r"\d+", extname)
            ifu_number = m.group()
            product_ext_dict["IFU_NUMBER"] = int(ifu_number)

            naxis = product_ext.header['NAXIS']
            
            # NAXIS is 1 if there is an spectrum, 0 otherwise
            if (naxis == 1):
                product_ext_dict["IFU_STATUS"] = 'Active'

                # Get infos from extract_spec using the extname
                product_ext_dict['CRPIX1']     = product_ext.header['CRPIX1']
                product_ext_dict['CRVAL1']     = product_ext.header['CRVAL1']
                product_ext_dict['CDELT1']     = product_ext.header['CDELT1']
                product_ext_dict["NAME"]       = extname + "." + product_ext.header['ESO OCS ARM' + ifu_number +' NAME']
                product_ext_dict["DATA"]       = product_ext.data
            else:
                product_ext_dict["IFU_STATUS"] = 'Empty'
                
            product_dict["EXTENSIONS"].append(product_ext_dict)
            
    def addSubplots(self, figure):
        self._add_subplots(figure)

    def plotProductsGraphics(self):
        self._plot()

    def plotWidgets(self) :
        widgets = list()

        labels=[]
        for file_idx in range(0, len(self.files_extract_spec)):
            labels.append(self.files_extract_spec[file_idx]["ARCFILE"])
        
        # Files Selector radiobutton
        self.radiobutton = reflex_plot_widgets.InteractiveRadioButtons(self.files_extract_spec_selector, self.setFSCallback, labels, self.selected_file_idx, title='Files Selection (double-click Left Mouse Button)')
        widgets.append(self.radiobutton)
                
        return widgets
        
    def setFSCallback(self, label) :
        for file_idx in range(0, len(self.files_extract_spec)):
            if (self.files_extract_spec[file_idx]["ARCFILE"] == label):
                self.selected_file_idx = file_idx        

        # Redraw spectra stack
        self._disp_spectra_stack(self.reference_stack, "Exposure to correct (" + self.file_reference["ARCFILE"] + ")\n", self.file_reference["EXTENSIONS"])
        self._disp_spectra_stack(self.extract_spec_stack, "Used to fit Atm. model (" + label + ")\n", self.files_extract_spec[self.selected_file_idx]["EXTENSIONS"])
            
    def _add_subplots(self, figure):
        gs = gridspec.GridSpec(5, 6)
        self.files_extract_spec_selector = figure.add_subplot(gs[0:2, 0:5])
        self.reference_stack = figure.add_subplot(gs[2:5, 0:2])
        self.extract_spec_stack = figure.add_subplot(gs[2:5, 3:5])

    def _data_plot_get_tooltip(self):
        return self.files_extract_spec[self.selected_file_idx]["ARCFILE"]
    
    def _data_plot(self):
        self._disp_spectra_stack(self.reference_stack, "Exposure to correct (" + self.file_reference["ARCFILE"] + ")\n", self.file_reference["EXTENSIONS"])
        self._disp_spectra_stack(self.extract_spec_stack, "Used to fit Atm. model (" + self.files_extract_spec[self.selected_file_idx]["ARCFILE"] + ")\n", self.files_extract_spec[self.selected_file_idx]["EXTENSIONS"])
        
    def _disp_spectra_stack(self, stack, title, extensions):        

        try:
            from __builtin__ import str #Python2
        except:
            from builtins import str #Python3
        # Plot Spectrum
        stack.clear();
        sstackdisp = pipeline_display.SpectrumDisplay()
        
        v_offset = 0.
        lower_ybound = 0.
        upper_ybound = 0.
        for ext in range(0, len(extensions) ):

            ifu_number = str(extensions[ext]["IFU_NUMBER"])                
            
            # Empty line Spectra in Black
            if (extensions[ext]["IFU_STATUS"] == 'Empty') :
                c = 'black'
            else :

                if self.PROCESS_IFUS == "-1" :
                    c = 'red'
                else :
                    process_ifus_split = [x.strip() for x in self.PROCESS_IFUS.split(',')]
                    if ifu_number in process_ifus_split :
                        c = 'red'
                    else:
                        c = 'blue'
                
                spectrum = extensions[ext]["DATA"]
                
                tmp = [x for x in spectrum if (np.isnan(x) == False)]
                spectrum = spectrum/np.median(tmp) +2*ext

                # Define wave
                pix  = numpy.arange(len(spectrum))
                wave = extensions[ext]["CRVAL1"] + pix * extensions[ext]["CDELT1"]
                
                lower_xbound = np.nanmin(wave)
                upper_xbound = np.nanmax(wave)
                
                spec_ = [x for x in spectrum if (np.isnan(x) == False)]
                
                med = np.median(spec_) if not np.isnan(np.median(spec_)) else 0
                mad = np.median(np.abs(spec_ - med)) if not np.isnan(np.median(spec_)) else 0.1  #Once NaN always NaN assumption forced because of isnan()
                
                os_factor = med
                v_offset = 0
                
                lower_ybound = -0.01
                upper_ybound = 50.
                spectrum_str = "ext: %02d" % int(ifu_number)
                sstackdisp.overplot(stack, wave, np.add(spectrum, v_offset), c)
                stack.text(upper_xbound * 1.04411725927, 
                                     v_offset + os_factor, spectrum_str, verticalalignment='top', horizontalalignment='right', color=c, fontsize=10)
 
        if (lower_ybound != 0 and upper_ybound != 0) :
            stack.set_ybound(lower = lower_ybound * 1.15,          upper = upper_ybound * 1.02)
            stack.set_xbound(lower = lower_xbound * 1.00259742743, upper = upper_xbound       )
            
        stack.set_title(title, fontsize=12, fontweight='semibold') 
        stack.set_xlabel(r"$\lambda$ ["+"$\mu$m]" + " (red: selected in PROCESS_IFUS)")
        stack.set_ylabel("Normalized flux.") 


#This is the 'main' function
if __name__ == '__main__':
    from reflex_interactive_app import PipelineInteractiveApp

    # Create interactive application
    interactive_app = PipelineInteractiveApp(enable_init_sop=True)

    # get inputs from the command line
    interactive_app.parse_args()

    #Check if import failed or not
    if not import_success:
        interactive_app.setEnableGUI(False)

    #Open the interactive window if enabled
    if interactive_app.isGUIEnabled():
        #Get the specific functions for this window
        dataPlotManager = DataPlotterManager()

        interactive_app.setPlotManager(dataPlotManager)
        interactive_app.showGUI()

        #Create list with the selected new 'out_sof' file    
        files           = list()
        output_name     = dataPlotManager.files_extract_spec[dataPlotManager.selected_file_idx]["FILENAME"]
        output_category = dataPlotManager.files_extract_spec[dataPlotManager.selected_file_idx]["CATEGORY"]
        output_purpose  = dataPlotManager.files_extract_spec[dataPlotManager.selected_file_idx]["PURPOSES"]
        files.append(reflex.FitsFile(output_name, output_category, None, output_purpose))
    
        # Copy the selected sof file in the in_sof --> interactive_app.print_outputs() will copy '.in_sof' to '.out.sof'     
        interactive_app.inputs.in_sof = reflex.SetOfFiles(interactive_app.inputs.in_sof.datasetName, files)
    else:
        interactive_app.set_continue_mode()

    #Print outputs. This is parsed by the Reflex python actor to
    #get the results. Do not remove
    interactive_app.print_outputs()
    sys.exit()
