import numpy as np
import os
from adari_core.plots.panel import Panel
from adari_core.plots.images import ImagePlot
from adari_core.plots.points import LinePlot
from adari_core.plots.text import TextPlot
from adari_core.data_libs.master_ifu_flat import MasterIfuFlatReport
from .kmos_utils import KmosReportMixin


class KmosIfuFlatReport(KmosReportMixin, MasterIfuFlatReport):

    def __init__(self, name="kmos_ifu_flat"):
        super().__init__(name)
        self.detectors = {"DET.1.DATA": None, "DET.2.DATA": None, "DET.3.DATA": None}
        self.files_needed = {"MASTER_FLAT": None}
        self.center_size = 500
        self.default_angle_entry = 0

    def parse_sof(self):
        """Returns a list of files selected from a set of frames (sof).

        If more than one file fullfills the criteria, the first file
        in the array will be selected.
        """
        file_path, files_category = (
            [elem[0] for elem in self.inputs],
            [elem[1] for elem in self.inputs],
        )
        for required_file in self.files_needed.keys():
            # Check that category matches the requirement
            if required_file in files_category:
                self.files_needed[required_file] = file_path[
                    files_category.index(required_file)
                ]
            else:
                raise IOError("[WARNING] {} file not found".format(required_file))
        return [self.files_needed]

    def get_rotations(self):
        """Get the rotation angles present in a KMOS HDU list.

        Description
        -----------
        There are three detectors labelled as DET.1, DET.2, DET.3
        The expentions containing data (with different rotation angles)
        correspond to multiple DET.***.DATA.
        This function creates a dictionary (detector) that contains the
        three KMOS detectors. Within each dictionary, two additional nested
        dictionaries contain the information of the detector rotation angle
        (angle) and the corresponding index within the HDU list (entry).
        """

        for i, hdu in enumerate(self.hdus[0]["MASTER_FLAT"]):
            if hdu.name in self.detectors.keys():
                # detector = hdu.name.replace(".DATA", "")
                angle = hdu.header.get("HIERARCH ESO PRO ROT NAANGLE")
                if self.detectors[hdu.name] is None:
                    self.detectors[hdu.name] = dict(angle=[angle], entry=[i], nangle=1)
                else:
                    self.detectors[hdu.name]["angle"].append(angle)
                    self.detectors[hdu.name]["entry"].append(i)
                    self.detectors[hdu.name]["nangle"] += 1
            else:
                continue

    def add_text_plot(self, panel, master_flat, extension_idx, vspace=0.3):
        """Add to the single-extension report panel the text plots requirements."""
        px, py = 0, 0
        t1 = TextPlot(columns=1, v_space=vspace)
        fname = os.path.basename(master_flat.filename())

        extname = master_flat[extension_idx].header.get("EXTNAME", "N/A")
        rotangle = master_flat[extension_idx].header.get("HIERARCH ESO PRO ROT NAANGLE")
        col1 = (
            master_flat["PRIMARY"].header.get("INSTRUME"),
            "EXTNAME: " + extname,
            "ROT. ANGLE: {:.1f}".format(rotangle),
            "PRO CATG: " + master_flat["PRIMARY"].header.get("HIERARCH ESO PRO CATG"),
            "FILE NAME: " + fname,
            "RAW NAME: "
            + master_flat["PRIMARY"].header.get(
                "HIERARCH ESO PRO REC1 RAW{} NAME".format(
                    extname.replace(".DATA", "").replace("DET.", "")
                )
            ),
        )
        t1.add_data(col1)

        panel.assign_plot(t1, px, py, xext=2)

        px += 2
        t2 = TextPlot(columns=1, v_space=vspace, xext=1)
        col2 = (
            "INS.GRAT1.NAME: "
            + str(master_flat["PRIMARY"].header.get("HIERARCH ESO INS GRAT1 NAME")),
            "INS.GRAT1.WLEN: "
            + str(master_flat["PRIMARY"].header.get("HIERARCH ESO INS GRAT1 WLEN")),
            "DET.SEQ1.DIT: "
            + str(master_flat["PRIMARY"].header.get("HIERARCH ESO DET SEQ1 DIT")),
        )
        t2.add_data(col2)
        panel.assign_plot(t2, px, py, xext=1)

    def generate_single_ext_panels(self):
        panels = {}
        for hdul_dict in self.hdus:
            master_flat_hdul = hdul_dict["MASTER_FLAT"]
            # Only select the first rotation angle for the
            # three detectors
            extension_numbers = [
                self.detectors["DET.1.DATA"]["entry"][self.default_angle_entry],
                self.detectors["DET.2.DATA"]["entry"][self.default_angle_entry],
                self.detectors["DET.3.DATA"]["entry"][self.default_angle_entry],
            ]
            # Ensure that the rotator angle is the same
            # for the three detectors
            rotation_angle = [
                self.detectors["DET.1.DATA"]["angle"][self.default_angle_entry],
                self.detectors["DET.2.DATA"]["angle"][self.default_angle_entry],
                self.detectors["DET.3.DATA"]["angle"][self.default_angle_entry],
            ]
            if np.unique(rotation_angle).size > 1:
                raise ArithmeticError(
                    "KMOS detector extension have different rotation angles: {}, {}, {}".format(
                        *rotation_angle
                    )
                )

            # TODO: In the interactive case, the user should specify
            # a given rotation angle or instead use all the entries avaialble
            for i in extension_numbers:
                hdr = master_flat_hdul[i].header
                if "EXTNAME" in hdr:
                    extname = hdr.get("EXTNAME")
                    if "DATA" in extname:
                        if master_flat_hdul[i].data is None:
                            continue
                        elif (~np.isfinite(master_flat_hdul[i].data)).all():
                            continue
                        panel = self.single_panel(master_flat_hdul[i])
                        self.add_text_plot(
                            panel=panel, master_flat=master_flat_hdul, extension_idx=i
                        )
                        setup = (
                            master_flat_hdul["PRIMARY"].header.get(
                                "HIERARCH ESO INS GRAT1 NAME"
                            )
                            + master_flat_hdul["PRIMARY"].header.get(
                                "HIERARCH ESO INS GRAT2 NAME"
                            )
                            + master_flat_hdul["PRIMARY"].header.get(
                                "HIERARCH ESO INS GRAT3 NAME"
                            )
                        )
                        input_files = [master_flat_hdul.filename()]
                        panels[panel] = {
                            # "master_im": master_im.filename(),
                            # "master_im_ext": master_im_ext,
                            "report_name": "kmos_master_flat_single_{}_{:03.0f}_{}".format(
                                extname.replace(".DATA", ""),
                                master_flat_hdul[i].header.get(
                                    "HIERARCH ESO PRO ROT NAANGLE"
                                ),
                                setup.lower(),
                            ),
                            "report_description": "Flat single panel",
                            "report_tags": [],
                            "input_files": input_files,
                        }
        return panels

    def multipanel_plot(self, data, title=""):
        if data is not None:
            mplot = ImagePlot(
                title=title, cbar_kwargs={"pad": 0.1}, interpolation="none"
            )
            data = np.nan_to_num(data)
            mplot.add_data(data)
        else:
            # if no image data, setup a placeholder plot
            # TODO: implement empty plot class
            mplot = LinePlot(
                legend=False, y_label="y", x_label="x", title=title + " (NO DATA)"
            )
        return mplot

    def generate_multi_ext_panels(self):
        panels = {}

        for hdu_list in self.hdus:
            number_of_rotations = max(
                [self.detectors[det]["nangle"] for det in self.detectors.keys()]
            )

            p = Panel(number_of_rotations, 4, height_ratios=[1, 4, 4, 4])
            # Text Plot
            px, py = 0, 0
            # which hdul and ext to use
            vspace = 0.3
            t1 = TextPlot(columns=1, v_space=vspace)
            fname = os.path.basename(str(hdu_list["MASTER_FLAT"].filename()))

            col1 = (
                str(hdu_list["MASTER_FLAT"]["PRIMARY"].header.get("INSTRUME")),
                "EXTNAME: "
                + str(hdu_list["MASTER_FLAT"]["PRIMARY"].header.get("EXTNAME", "N/A")),
                "PRO CATG: "
                + str(
                    hdu_list["MASTER_FLAT"]["PRIMARY"].header.get(
                        "HIERARCH ESO PRO CATG"
                    )
                ),
                "FILE NAME: " + fname,
                "RAW1 NAME: "
                + str(
                    hdu_list["MASTER_FLAT"]["PRIMARY"].header.get(
                        "HIERARCH ESO PRO REC1 RAW1 NAME"
                    )
                ),
            )
            t1.add_data(col1)
            p.assign_plot(t1, px, py, xext=2)

            px = px + 2
            t2 = TextPlot(columns=1, v_space=vspace, xext=1)
            col2 = (
                "INS.GRAT1.NAME: "
                + str(
                    hdu_list["MASTER_FLAT"]["PRIMARY"].header.get(
                        "HIERARCH ESO INS GRAT1 NAME"
                    )
                ),
                "INS.GRAT1.WLEN: "
                + str(
                    hdu_list["MASTER_FLAT"]["PRIMARY"].header.get(
                        "HIERARCH ESO INS GRAT1 WLEN"
                    )
                ),
                "DET.SEQ1.DIT: "
                + str(
                    hdu_list["MASTER_FLAT"]["PRIMARY"].header.get(
                        "HIERARCH ESO DET SEQ1 DIT"
                    )
                ),
            )
            t2.add_data(col2)
            p.assign_plot(t2, px, py, xext=1)

            setup = (
                str(
                    hdu_list["MASTER_FLAT"]["PRIMARY"].header.get(
                        "HIERARCH ESO INS GRAT1 NAME"
                    )
                )
                + str(
                    hdu_list["MASTER_FLAT"]["PRIMARY"].header.get(
                        "HIERARCH ESO INS GRAT2 NAME"
                    )
                )
                + str(
                    hdu_list["MASTER_FLAT"]["PRIMARY"].header.get(
                        "HIERARCH ESO INS GRAT3 NAME"
                    )
                )
            )
            # row 0 is for metadata
            py = 1
            hdu_info = []
            hdu_plot_args = []
            for extname in self.detectors.keys():
                px = 0
                detector_info = self.detectors[extname]
                for rotation_angle, entry in zip(
                    detector_info["angle"], detector_info["entry"]
                ):
                    hdu_info.append((0, "MASTER_FLAT", entry, px, py))
                    hdu_plot_args.append(dict(title=f"{extname} {rotation_angle}"))
                    px = px + 1
                py = py + 1

            p = self.multi_panel(panel=p, hdu_info=hdu_info, plot_kwargs=hdu_plot_args)
            addme = {  # "ext": ext,
                "report_name": f"kmos_master_flat_multi_{setup.lower()}",
                "report_description": "KMOS flat multi panel",
                #                      f"{ext})",
                "report_tags": [],
                "input_files": [fname],
            }

            panels[p] = addme
        return panels

    def generate_panels(self, **kwargs):
        """Create both single and multiple extension panels."""
        self.get_rotations()
        panels = {
            **self.generate_single_ext_panels(),
            **self.generate_multi_ext_panels(),
        }
        return panels


rep = KmosIfuFlatReport()
