import os

from adari_core.plots.panel import Panel
from adari_core.plots.images import ImagePlot
from adari_core.plots.points import LinePlot
from adari_core.plots.text import TextPlot
from adari_core.report import AdariReportBase

from .kmos_utils import KmosReportMixin


class KmosIllumCorrReport(KmosReportMixin, AdariReportBase):
    rot_angles = {}
    files_needed = {"ILLUM_CORR": None}
    n_ifu = 24

    def __init__(self):
        super().__init__("kmos_illumination")

    def parse_sof(self):
        """Returns a list of files selected from a set of frames (sof).

        If more than one file fullfills the criteria, the first file
        in the array will be selected.
        """
        file_path, files_category = (
            [elem[0] for elem in self.inputs],
            [elem[1] for elem in self.inputs],
        )
        for required_file in self.files_needed.keys():
            # Check that category matches the requirement
            if required_file in files_category:
                self.files_needed[required_file] = file_path[
                    files_category.index(required_file)
                ]
            else:
                raise IOError("[WARNING] {} file not found".format(required_file))
        return [self.files_needed]

    def get_rotations(self):
        """Get the rotation angles present in a KMOS HDU list."""
        self.number_of_rotations = (len(self.hdus[0]["ILLUM_CORR"]) - 1) // 24

    def create_rotation_panel(self, hdul):
        """Generate the panel of a given rotation from an HDU list"""
        panel = Panel(6, 5, height_ratios=[1, 4, 4, 4, 4])
        # IFU plots
        for i in range(1, self.n_ifu + 1):
            row = (i - 1) // 6
            col = i - 1 - 6 * ((i - 1) // 6)
            ifu_hdu = hdul["IFU.{}.DATA".format(i)]
            if ifu_hdu.data is not None:
                mplot = ImagePlot(
                    title="IFU.{}".format(i),
                    cbar_kwargs={"pad": 0.1},
                    interpolation="none",
                )
                mplot.add_data(ifu_hdu.data)

            else:
                mplot = LinePlot(
                    legend=False,
                    y_label="y",
                    x_label="x",
                    title="IFU.{} (NO DATA)".format(i),
                )
            panel.assign_plot(mplot, col, row + 1)
        return panel

    def generate_panels(self, **kwargs):
        panels = {}
        self.get_rotations()
        for hdu_list in self.hdus:
            hdul_illum_corr = hdu_list["ILLUM_CORR"]
            setup = (
                hdul_illum_corr["PRIMARY"].header.get("HIERARCH ESO INS GRAT1 NAME")
                + hdul_illum_corr["PRIMARY"].header.get("HIERARCH ESO INS GRAT2 NAME")
                + hdul_illum_corr["PRIMARY"].header.get("HIERARCH ESO INS GRAT3 NAME")
            )

            for i in range(0, self.number_of_rotations):
                rot_angle_extensions = slice(
                    i * self.n_ifu + 1, (i + 1) * self.n_ifu + 1
                )
                p = self.create_rotation_panel(hdul_illum_corr[rot_angle_extensions])
                # Metadata
                text_col = (
                    str(hdul_illum_corr["PRIMARY"].header.get("INSTRUME")),
                    "PRO CATG: "
                    + str(
                        hdul_illum_corr["PRIMARY"].header.get("HIERARCH ESO PRO CATG")
                    ),
                    "FILE NAME: " + os.path.basename(hdul_illum_corr.filename()),
                    "RAW1 NAME: "
                    + str(
                        hdul_illum_corr["PRIMARY"].header.get(
                            "HIERARCH ESO PRO REC1 RAW1 NAME"
                        )
                    ),
                )
                text = TextPlot(columns=1, v_space=0.3)
                text.add_data(text_col)
                p.assign_plot(text, 0, 0, xext=3)
                # Instrument setup metadata
                text_col = (
                    "INS.GRAT1.NAME: "
                    + str(
                        hdul_illum_corr["PRIMARY"].header.get(
                            "HIERARCH ESO INS GRAT1 NAME"
                        )
                    ),
                    "INS.GRAT1.WLEN: "
                    + str(
                        hdul_illum_corr["PRIMARY"].header.get(
                            "HIERARCH ESO INS GRAT1 WLEN"
                        )
                    ),
                    "DET.SEQ1.DIT: "
                    + str(
                        hdul_illum_corr["PRIMARY"].header.get(
                            "HIERARCH ESO DET SEQ1 DIT"
                        )
                    ),
                    "PRO.ROT.NAANGLE: {:03.1f}".format(
                        hdul_illum_corr[i * self.n_ifu + 1].header.get(
                            "HIERARCH ESO PRO ROT NAANGLE"
                        )
                    ),
                )
                text = TextPlot(columns=1, v_space=0.3)
                text.add_data(text_col)
                p.assign_plot(text, 2, 0, xext=3)
                input_files = [hdul_illum_corr.filename()]
                addme = {
                    "report_name": "kmos_illumination_{:03.0f}_{}".format(
                        hdul_illum_corr[i * self.n_ifu + 1].header.get(
                            "HIERARCH ESO PRO ROT NAANGLE"
                        ),
                        setup.lower(),
                    ),
                    "report_description": "KMOS Illumination correction panel",
                    "report_tags": [],
                    "input_files": input_files,
                }

                panels[p] = addme
        return panels


rep = KmosIllumCorrReport()
