/*
 * This file is part of the ESO Telluric Correction Library
 * Copyright (C) 2001-2018 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

/*----------------------------------------------------------------------------*/
/**
 *                              Includes
 */
/*----------------------------------------------------------------------------*/


#include "mf_strlst.h"

/*----------------------------------------------------------------------------*/
/**
 * @brief    Checks if given string list is actually "NULL" (case insensitive)
 *           Return 1 if true and 0 if false.
 *
 * @param  strlst                 The string list to check 
 
*/
/*----------------------------------------------------------------------------*/

cpl_boolean  mf_strlst_is_null     (const char* strlst) {

/*
    int const true  = 1;
    int const false = 0;
*/
    /* Check for trivial case that string is empty*/
    if (strlst[0]=='\0') return CPL_FALSE;
    
    
    int string_size = strlen(strlst);
    
    /* Check if string is all blanks and if not where first non blank is*/
    cpl_boolean all_blanks=CPL_TRUE;
    int idx0=-1;
    for (int i=0;i<string_size;i++) {
        if (strlst[i]==' ') continue;
        all_blanks=CPL_FALSE;
        idx0=i;
        break;       
    }
    
    /* If string is all blanks then return false */    
    if (all_blanks) return CPL_FALSE;
    
    
    /* Check that there are enough characters in string to spell null*/
    if (idx0+3>string_size) return CPL_FALSE;
    
    /* Check that this and the next 3 characters spell NULL independant of case*/
    if (strlst[idx0+0]!='N' && strlst[idx0+0]!='n') return CPL_FALSE; 
    if (strlst[idx0+1]!='U' && strlst[idx0+1]!='u') return CPL_FALSE; 
    if (strlst[idx0+2]!='L' && strlst[idx0+2]!='l') return CPL_FALSE; 
    if (strlst[idx0+3]!='L' && strlst[idx0+3]!='l') return CPL_FALSE; 
    
    /* We dont care what comes after the word NULL so return true*/
    return CPL_TRUE;

}

/*----------------------------------------------------------------------------*/
/**
 * @brief    Returns the mumber of elements in the given string list
 *
 * @param  strlst                 The string list to check 
 
*/
/*----------------------------------------------------------------------------*/


int  mf_strlst_get_size    (const char* strlst) {

    /* Check for trivial case that string is empty*/
    if (strlst[0]=='\0') return 0;

    /* String is non empty so count the number of deliminators*/
    int size = strlen(strlst);
    int cnt=0;
    for (int i=0;i<size;i++) {
        if (strlst[i]==',') cnt=cnt+1;
    }
    
    /* No of list elements is the no of deliminators +1*/
    return cnt+1;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief    Checks if given string list is valid given a specific string list type
 *           Return 1 if true and 0 if false.
 *
 * @param  strlst                 The string list to check 
 * @param  strlst_type            The string list type. 
 
*/
/*----------------------------------------------------------------------------*/

cpl_boolean mf_wrap_strlst_is_valid (const char* strlst, enum MF_WRAP_STRLST_TYPE strlst_type) {

    cpl_boolean valid_flag;

    switch (strlst_type) {
    
        case(MF_WRAP_STRLST_STRING):

            valid_flag=mf_wrap_strlst_is_valid_list_of_types(strlst,CPL_TYPE_STRING);
            break;

        case(MF_WRAP_STRLST_DOUBLE):

            valid_flag=mf_wrap_strlst_is_valid_list_of_types(strlst,CPL_TYPE_DOUBLE);
            break;

        case(MF_WRAP_STRLST_INT):

            valid_flag=mf_wrap_strlst_is_valid_list_of_types(strlst,CPL_TYPE_INT);
            break;

        case(MF_WRAP_STRLST_BOOL):

            valid_flag=mf_wrap_strlst_is_valid_list_of_types(strlst,CPL_TYPE_BOOL);
            break;
            
        case(MF_WRAP_STRLST_MF_DRANGE):
            
            /* Check that string is a list is of valid doubles */
            valid_flag=mf_wrap_strlst_is_valid_list_of_types(strlst,CPL_TYPE_DOUBLE);
            
            /* Check that list has an even number of elements */
            if (valid_flag) {
            
                int nelems=mf_strlst_get_size(strlst);
                if (nelems%2 != 0) {
                    cpl_msg_info(cpl_func,"Range list has odd number of elements: %s", strlst);
                    valid_flag=CPL_FALSE;                    
                } 
            }

            /* Check that range pairs [val,val2] is such that val2>val1 */
            if (valid_flag) {
                int nelems=mf_strlst_get_size(strlst);
                for (int i=0;i<=nelems/2;i++) {
                    double val1 = mf_strlst_get_double(strlst,i  );
                    double val2 = mf_strlst_get_double(strlst,i+1);
                    if (val2<val1) {
                        cpl_msg_info(cpl_func,"Range list contains invalid range: [%f,%f]",val1,val2);
                        valid_flag=CPL_FALSE;                    
                    }
                }
            }            
            
            break;
        
        default:
        
            valid_flag = CPL_TRUE;
        
    };/* end switch */
    
    return valid_flag;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief    Checks if given string list is valid given a specific data type
 *           Return 1 if true and 0 if false.
 *
 * @param  strlst                 The string list to check 
 * @param  strlst                 The data type that the list is made of. 
 
*/
/*----------------------------------------------------------------------------*/

cpl_boolean mf_wrap_strlst_is_valid_list_of_types (const char* strlst, cpl_type data_type) {


    /* Check for trivial case that string is empty*/
    if (strlst[0]=='\0') {
        cpl_msg_info(cpl_func,"Invalid empty string list");
        return CPL_FALSE;
    }

    /* Check if NULL */
    if (mf_strlst_is_null(strlst)) return CPL_TRUE;
    
    
    /* Get the no of elements that the string contains*/
    int nelems   = mf_strlst_get_size(strlst);
    
    /* if no elements then return CPL_FALSE*/
    if (nelems==0)  {
        cpl_msg_info(cpl_func,"Invalid empty string list");
        return CPL_FALSE;
    }
    
    /* Check that each element is non empty */
    for (int elem=1; elem<=nelems; elem++) {
        char* elem_str=mf_wrap_strlst_get_str(strlst,elem);
        int elem_size=strlen(elem_str);
        if (elem_size==0) {
            cpl_msg_info(cpl_func,"Invalid list member (empty) %d in list string %s ",elem , strlst);
            return CPL_FALSE;
        }
        free(elem_str);
    }


    /* If data type is a string then there are no more tests*/
    if (data_type==CPL_TYPE_STRING) return CPL_TRUE;
    
    
    /* Check that each element is of correct type */

    cpl_boolean is_valid=CPL_TRUE; /* Assume all is valid until proven otherwise*/

    for (int elem=1; elem<=nelems; elem++) {

        /* Get the element as a string */
        char* elem_str=mf_wrap_strlst_get_str(strlst,elem);

        /* Attempt a string to double conversion using strtod()*/
        char * p;
        double dval0=strtod (elem_str, &p); /* If succesful conversion then *p=='\0]*/
        if (*p != '\0') {
	
            /* Thid is s non numerical string */
	    cpl_msg_info(cpl_func,"Non numerical element: \"%s\" found as item %d in list: \"%s\"",elem_str,elem,strlst);
            is_valid=CPL_FALSE;
        
	} else {
        
            /* truncate and convert to integer the parsed double dval0*/
	    double tdval = trunc(dval0);
            int    ival  = (int) tdval;

            /* Check that this number is valid to the specified data type */
            switch (data_type) {

        	case CPL_TYPE_INT:
        	if (tdval != dval0) {
                    cpl_msg_info(cpl_func,"Non integer element: \"%s\" found as item %d in list: \"%s\"",elem_str,elem,strlst);
                    is_valid=CPL_FALSE;
        	}
        	break;

        	case CPL_TYPE_BOOL:
        	if (ival!=1 && ival!=0) {
                    cpl_msg_info(cpl_func,"Non boolean element: \"%s\" found as item %d in list: \"%s\"",elem_str,elem,strlst);
                    is_valid=CPL_FALSE;
        	}
        	break;
                
                default:
                cpl_msg_info(cpl_func,"do nothing");

             }
         
	 }         
         
	 free(elem_str);
         
    };/* end elem loop*/

     
    return is_valid;
}


/*----------------------------------------------------------------------------*/
/**
 * @brief    Returns the requested element of a string list as a double
 *           
 *
 * @param  strlst                 The string list
 * @param  idx                    The element index 
 
*/
/*----------------------------------------------------------------------------*/

double mf_strlst_get_double (const char* strlst, int idx) {

    /* Get the list element as a string */
    char* elem_str = mf_wrap_strlst_get_str (strlst,idx);

   /* Convert string to a double */
   char *p;
   double dval = strtod (elem_str, &p);
  
   return dval;

}


/*----------------------------------------------------------------------------*/
/**
 * @brief    Returns the requested element of a string list as a float
 *          
 *
 * @param  strlst                 The string list
 * @param  idx                    The element index 
 
*/
/*----------------------------------------------------------------------------*/

float mf_strlst_get_float (const char* strlst, int idx) {

    /* Get the list element as a double */
    double dval= mf_strlst_get_double (strlst,idx);

   /* Convert to a float */
   float fval = (float) dval;
  
   return fval;

}

/*----------------------------------------------------------------------------*/
/**
 * @brief    Returns the requested element of a string list as an integer
 *           
 *
 * @param  strlst                 The string list
 * @param  idx                    The element index 
 
*/
/*----------------------------------------------------------------------------*/


int mf_strlst_get_int (const char* strlst, int idx) {

    /* Get the list element as a double */
    double dval= mf_strlst_get_double (strlst,idx);

   /* Convert to a float */
   int ival = (int) dval;
  
   return ival;

}

/*----------------------------------------------------------------------------*/
/**
 * @brief    Returns the requested element of a string list as a boolean (1 or 0)
 *  
 *
 * @param  strlst                 The string list
 * @param  idx                    The element index 
 
*/
/*----------------------------------------------------------------------------*/

cpl_boolean mf_strlst_get_boolean (const char* strlst, int idx) {

    /* Get the list element as a double */
    double dval= mf_strlst_get_double (strlst,idx);

   /* Convert to an int */
   int ival = (int) dval;
  
   if (ival!=0) return CPL_TRUE;
  
   return CPL_FALSE;

}

/*----------------------------------------------------------------------------*/
/**
 * @brief    Returns the requested element of a string list as a string trimmed
 *           of any leading and trailing whitespaces.
 * 
 *
 * @param  strlst                 The string list
 * @param  idx                    The element index 
 
*/
/*----------------------------------------------------------------------------*/

char* mf_wrap_strlst_get_str (const char* strlst, int idx) {

    int strsize=strlen(strlst);
    int nelems=mf_strlst_get_size(strlst);

    /* Define the first and last deliminator position as -1 and string length*/
    int delim_pos[nelems];
    delim_pos[0]=-1;
    delim_pos[nelems]=strsize;
    
    /* Now locate the deliminator positions in the string itself*/
    int cnt=0;
    for (int i=0;i<strsize;i++) {
        if (strlst[i]!=',') continue;
        cnt=cnt+1;
        delim_pos[cnt]=i;
    }

    /* Define the indecies of interest */
    int idx1=delim_pos[idx-1]+1;
    int idx2=delim_pos[idx-0]-1;
    
    /*Alter first index so as to ignore leading whitespaces*/
    for (int i=idx1;i<idx2; i++) {
        if (strlst[i]==' ') {idx1++;} else {break;}
    }

    /*Alter second index so as to ignore trailing whitespaces*/
    for (int i=idx2;i>idx1; i--) {
        if (strlst[i]==' ') {idx2--;} else {break;}
    }

    /* Determine the character size of this element */
    int char_size=idx2-idx1+1;
    if (char_size<1)  char_size=1;
        
    /* Allocate a return string of appropriate size */
    char * ret_str = (char*) malloc((char_size+1)*sizeof(char));
    
    /* Copy the element string to the return string one character at a time*/
    int j=0;
    for (int i=idx1;i<=idx2;i++) {
        ret_str[j]=strlst[i];
        j++;
    }
    
    /* Append the \0 char*/
    ret_str[j]='\0';
    
    /* Remove leading and trailing whaite spaces */
    /*ret_str=trim_space(ret_str); */
        
    return ret_str;

}

/*----------------------------------------------------------------------------*/
/**
 * @brief    Returns true or false if a string trimmed
 *           of any leading and trailing whitespaces.
 *
 *
 * @param  strlst                 The string list
 * @param  idx                    The element index

*/
/*----------------------------------------------------------------------------*/

cpl_boolean mf_strlst_get_bool_err (const char* strlst, int idx, char* ret_str) {

    int strsize=strlen(strlst);
    int nelems=mf_strlst_get_size(strlst);

    if (idx>nelems) return CPL_FALSE;

    /* Define the first and last deliminator position as -1 and string length*/
    int delim_pos[nelems+1];
    delim_pos[0]=-1;
    delim_pos[nelems]=strsize;

    /* Now locate the deliminator positions in the string itself*/
    int cnt=0;
    for (int i=0;i<strsize;i++) {
        if (strlst[i]!=',') continue;
        cnt=cnt+1;
        delim_pos[cnt]=i;
    }

    /* Define the indecies of interest */
    int idx1=delim_pos[idx-1]+1;
    int idx2=delim_pos[idx-0]-1;

    /*Alter first index so as to ignore leading whitespaces*/
    int new_idx1=idx1;
    for (int i=idx1;i<idx2; i++) {
        if (strlst[i]==' ') {new_idx1++;} else {break;}
    }
    idx1=new_idx1;

    /*Alter second index so as to ignore trailing whitespaces*/
    int new_idx2=idx2;
    for (int i=idx2;i>idx1; i--) {
        if (strlst[i]==' ') {new_idx2--;} else {break;}
    }
    idx2=new_idx2;

    /* Determine the character size of this element */
    int char_size=idx2-idx1+1;
    if (char_size<1)  char_size=1;

    /* Allocate a return string of appropriate size */
   /* char * ret_str = (char*) malloc((char_size+1)*sizeof(char));*/
   /* char * ret_str =  cpl_calloc((char_size+2),sizeof(char));*/

    /* Copy the element string to the return string one character at a time*/
    int j=0;
    for (int i=idx1;i<=idx2;i++) {
        ret_str[j]=strlst[i];
        j++;
    }

    /* Append the \0 char*/
    ret_str[char_size]='\0';

    /* Remove leading and trailing whaite spaces */
    /*ret_str=trim_space(ret_str); */

    return CPL_TRUE;

}

/* ---------------------------------------------------------------------------*/
/**
 * @brief Split a string into pieces at a given delimiter.
 *
 * @param str         The string to split.
 * @param delimiter   String specifying the locations where to split.
 * @param size        (Output) If non NULL, the number of found tokens is returned.
 *
 * @return The function returns a newly allocated, @c NULL terminated array of strings, or @c NULL in case of an error.
 *
 * The function breaks up the string @em string into pieces at the places indicated by @em delimiter.
 * The delimiter string @em delimiter never shows up in any of the resulting strings.
 *
 * As a special case, the result of splitting the empty string "" is an empty vector, not a vector containing a single string.
 *
 * The created result vector can be deallocated using @b mf_string_strfreev().
 *
 */
/* ---------------------------------------------------------------------------*/
char ** mf_str_split(
    const char               *str,
    const char               *delimiter,
    cpl_size                 *size)
{
  /* Check inputs */
  cpl_error_ensure(str && delimiter && size, CPL_ERROR_NULL_INPUT,
                   return NULL, "Not all input data is provided!");
  cpl_error_ensure(*delimiter != '\0', CPL_ERROR_ILLEGAL_INPUT,
                   return NULL, "delimiter is \"\\0\"!");

  /* Get the number of tokens to allocate */
  unsigned int sz        = strlen(delimiter);
  const char   *save_str = str;
  char         *s        = strstr(save_str, delimiter);
  cpl_size     nr        = 1;

  while (s && (strlen(s) > sz)) {
      save_str = s + sz;
      s         = strstr(save_str, delimiter);
      nr++;
  }

  /* Allocate array */
  char **sarray = cpl_calloc(nr + 1, sizeof(char*));
  *size = nr;

  /* Copy tokens into array */
  cpl_size i = 0;
  save_str   = str;
  s          = strstr(save_str, delimiter);

  if (!s || !(strlen(s) > sz)) {

      /* Only one token, copy it */
      sarray[i] = cpl_sprintf("%s", str);

  } else {

      while (s && (strlen(s) >= sz)) {

          cpl_size length = s - save_str;

          /* Initialize token */
          int n = length + 1;
          char *token = cpl_calloc(n, sizeof(char));
          strncpy(token, save_str, length);
          token[length] = '\0';

          /* Put it into array */
          sarray[i] = token;
          i++;

          /* Prepare next token */
          save_str = s + sz;
          if (strlen(save_str) == 0) s = NULL;
          else                       s = strstr(save_str, delimiter);
      }

      /* Copy last token if there is one and if it isn't the delimiter */
      if (save_str && strlen(save_str) > 0 && strcmp(save_str, delimiter) != 0) {
        sarray[i] = cpl_sprintf("%s", save_str);
      } else {
        sarray[i] = NULL;
      }
  }

  /* Initialize last member to NULL */
  sarray[nr] = NULL;

  return sarray;
}

/* ---------------------------------------------------------------------------*/
/**
 * @brief Deallocate a string array.
 *
 * @param strarr  String array to deallocate
 *
 * The function deallocates the array of strings @em strarr and any string it possibly contains.
 *
 */
/* ---------------------------------------------------------------------------*/
void mf_str_array_delete(
    char                     **strarr)
{
  if (strarr) {
      cpl_size i = 0;
      while (strarr[i] != NULL) {
          cpl_free(strarr[i]);
          i++;
      }
      cpl_free(strarr);
  }
}

