/*
 * This file is part of the ESO Telluric Correction Library
 * Copyright (C) 2001-2018 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

/*----------------------------------------------------------------------------*/
/**
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

#include "mf_wrap.h"
#include "mf_wrap_config_corr.h"


/*----------------------------------------------------------------------------*/
/**
 *                 Typedefs: Enumeration types
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Defines
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Global variables
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Macros
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Typedefs: Structured types
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Functions prototypes
 */
/*----------------------------------------------------------------------------*/
/* Fill the internal Molecfit configuration file */
molecfit_correct_parameter * mf_wrap_config_corr_init(cpl_frameset * frameset, const cpl_parameterlist  *list, const char* arm){
  cpl_msg_info(cpl_func,"mf_wrap_config_corr_init");
  /* Check input */
  cpl_error_ensure(frameset && list, CPL_ERROR_NULL_INPUT,
                   return NULL, "list input is NULL!");

  /* Get preState */
  cpl_errorstate preState = cpl_errorstate_get();
  const cpl_parameter *p;

  /* Create the configuration parameter */
  molecfit_correct_parameter *parameters = cpl_malloc(sizeof(molecfit_correct_parameter));
  parameters->mapping_correct_table            = NULL;
  parameters->telluric_corr                    = NULL;
  parameters->pl                               = cpl_propertylist_new();


  /* Use only primary extension in the input FITS file ? */
  p = cpl_parameterlist_find_const(list, MOLECFIT_PARAMETER_USE_ONLY_INPUT_PRI);
  parameters->use_only_input_pri_ext = cpl_parameter_get_bool(p);
  cpl_propertylist_update_bool(parameters->pl, MF_PARAMETERS_CONTEX_DEFAULT" "MOLECFIT_PARAMETER_USE_ONLY_INPUT_PRI, parameters->use_only_input_pri_ext);

  cpl_msg_info(cpl_func,"MOLECFIT_PARAMETER_USE_ONLY_INPUT_PRI %s",cpl_error_get_message());

  /* If parameters->use_only_input_pri_ext == CPL_TRUE, you can provide a error flux extension */
  p = cpl_parameterlist_find_const(list, MOLECFIT_PARAMETER_DFLUX_EXTENSION_DATA);
  parameters->dflux_extension_data = cpl_parameter_get_int(p);
  cpl_propertylist_update_int(parameters->pl, MF_PARAMETERS_CONTEX_DEFAULT" "MOLECFIT_PARAMETER_DFLUX_EXTENSION_DATA, parameters->dflux_extension_data);

  cpl_msg_info(cpl_func,"MOLECFIT_PARAMETER_DFLUX_EXTENSION_DATA %s",cpl_error_get_message());

  /* If parameters->use_only_input_pri_ext == CPL_TRUE, you can provide a mask extension */
  p = cpl_parameterlist_find_const(list, MOLECFIT_PARAMETER_MASK_EXTENSION_DATA);
  parameters->mask_extension_data = cpl_parameter_get_int(p);
  cpl_propertylist_update_int(parameters->pl, MF_PARAMETERS_CONTEX_DEFAULT" "MOLECFIT_PARAMETER_MASK_EXTENSION_DATA, parameters->mask_extension_data);

  cpl_msg_info(cpl_func,"MOLECFIT_PARAMETER_MASK_EXTENSION_DATA %s",cpl_error_get_message());

  /* Mapping correct */
  p = cpl_parameterlist_find_const(list, MOLECFIT_PARAMETER_MAPPING_CORRECT);
  parameters->mapping_correct = cpl_parameter_get_string(p);
  cpl_propertylist_update_string(parameters->pl, MF_PARAMETERS_CONTEX_DEFAULT" "MOLECFIT_PARAMETER_MAPPING_CORRECT, parameters->mapping_correct);
  /* Create mapping correct cpl_table */
  if (strcmp(parameters->mapping_correct, MF_PARAMETERS_NULL)) {
      parameters->mapping_correct_table = mf_wrap_config_table_mapping(parameters->mapping_correct, MOLECFIT_MAPPING_CORRECT_EXT);
  }

  cpl_msg_info(cpl_func,"MOLECFIT_PARAMETER_MAPPING_CORRECT %s",cpl_error_get_message());

  /* Suppress extension in the output recipe FITS files */
  p = cpl_parameterlist_find_const(list, MOLECFIT_PARAMETER_SUPPRESS_EXTENSION);
  parameters->suppress_extension = cpl_parameter_get_bool(p);
  cpl_propertylist_update_bool(parameters->pl, MF_PARAMETERS_CONTEX_DEFAULT" "MOLECFIT_PARAMETER_SUPPRESS_EXTENSION, parameters->suppress_extension);

  cpl_msg_info(cpl_func,"MOLECFIT_PARAMETER_SUPPRESS_EXTENSION %s",cpl_error_get_message());

  /* Load TAG = MOLECFIT_TELLURIC_CORR */
  const char* input_tag = mf_wrap_tag_suffix(MOLECFIT_TELLURIC_CORR,arm,CPL_FALSE);
  cpl_msg_info(cpl_func, "Load %s from molecfit_calctrans(...)", input_tag);
  const cpl_frame *frm_telluric_corr = cpl_frameset_find_const(frameset, input_tag);
  if (frm_telluric_corr) {
      const char *filename = cpl_frame_get_filename(frm_telluric_corr);
      parameters->telluric_corr = mf_wrap_fits_load(filename, CPL_FALSE);
  }


  /* Load COLUMNS in TELLURIC_CORR HEADER */
  if (parameters->telluric_corr) {

      const char *column_str;

      column_str = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_COLUMN_LAMBDA;
      if (cpl_propertylist_has(parameters->telluric_corr->v_ext[0].header, column_str)) {
          parameters->column_lambda = cpl_strdup(cpl_propertylist_get_string(parameters->telluric_corr->v_ext[0].header, column_str));
      }

      column_str = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_COLUMN_FLUX;
      if (cpl_propertylist_has(parameters->telluric_corr->v_ext[0].header, column_str)) {
          parameters->column_flux   = cpl_strdup(cpl_propertylist_get_string(parameters->telluric_corr->v_ext[0].header, column_str));
      }

      column_str = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_COLUMN_DFLUX;
      if (cpl_propertylist_has(parameters->telluric_corr->v_ext[0].header, column_str)) {
          parameters->column_dflux  = cpl_strdup(cpl_propertylist_get_string(parameters->telluric_corr->v_ext[0].header, column_str));
      }

      column_str = MF_PARAMETERS_CONTEX_DEFAULT" "MF_PARAMETERS_COLUMN_MASK;
      if (cpl_propertylist_has(parameters->telluric_corr->v_ext[0].header, column_str)) {
          parameters->column_mask   = cpl_strdup(cpl_propertylist_get_string(parameters->telluric_corr->v_ext[0].header, column_str));
      } else {
          //fixes a bug loading tables from calctrans without mask column
        parameters->column_mask = MF_PARAMETERS_COLUMN_MASK_NULL;//"NULL";
      }
  }


  /* Combine extensions in the FITS file ? */
  p = cpl_parameterlist_find_const(list, MOLECFIT_PARAMETER_CHIP_EXTENSIONS);
  parameters->chip_extensions = cpl_parameter_get_bool(p);
  cpl_propertylist_update_bool(parameters->pl, MF_PARAMETERS_CONTEX_DEFAULT" "MOLECFIT_PARAMETER_CHIP_EXTENSIONS, parameters->chip_extensions);


  /* Check status */
  if (!cpl_errorstate_is_equal(preState)) {
      /* Configuration failed */
      molecfit_correct_parameter_delete(parameters);
      return NULL;
  }

  return parameters;


}
/*----------------------------------------------------------------------------*/
/**
 * @brief    Deallocate the given parameter configuration object and its contents
 *
 * @param    parameters       The parameter configuration variable in the recipe.
 */
/*----------------------------------------------------------------------------*/
void molecfit_correct_parameter_delete(
    molecfit_correct_parameter *parameters)
{
  if (parameters) {

      if (parameters->mapping_correct_table) cpl_table_delete(        parameters->mapping_correct_table);

      if (parameters->telluric_corr        ) mf_wrap_fits_delete(    parameters->telluric_corr        );

      if (parameters->pl                   ) cpl_propertylist_delete( parameters->pl                   );

      cpl_free(parameters);
  }
}




