/*
 * This file is part of the Molecfit Pipeline
 * Copyright (C) 2001-2019 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifdef HAVE_CONFIG_H
    #include <config.h>
#endif

/*----------------------------------------------------------------------------*/
/**
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

#include <string.h>

#include "mf_wrap_fits.h"

/*----------------------------------------------------------------------------*/
/**
 *                 Typedefs: Enumeration types
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Defines
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Global variables
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Macros
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Typedefs: Structured types
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Functions prototypes
 */
/*----------------------------------------------------------------------------*/

/* Identifies the FITS type (table or image) */
/* Removed - already in mf_wrap_fits.h*/
/*mf_wrap_fits_file_format mf_wrap_fits_get_file_format(
    const char                  *filename,
    const cpl_boolean           verbose);
*/


/*----------------------------------------------------------------------------*/
/**
 *                 Functions
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 * @defgroup mf_wrap_fits  .
 *
 * @brief
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/* ---------------------------------------------------------------------------*/
/**
 * @brief Create mf_wrap_fits without data
 *
 * @param filename           Name of the file associate with the structure
 * @param format             Format of the structure: BINTABLE, IMAGE(1D), IMAGE(2D), IMAGE(3D).
 * @param type               Type of the structure: cpl_table, cpl_vector, cpl_matrix, cpl_image, cpl_imagelist
 * @param n_ext              Number of FITS extensions.
 *
 * @return mf_wrap_fits   Created is everything is OK or NULL in other case
 *
 */
/* ---------------------------------------------------------------------------*/
mf_wrap_fits * mf_wrap_fits_create(
    const char                  *filename,
    mf_wrap_fits_file_format   format,
    mf_wrap_fits_type          type,
    const cpl_size              n_ext)
{

  /* Check inputs */
  cpl_ensure(   format == mf_wrap_fits_file_table
             || format == mf_wrap_fits_file_image_1D
             || format == mf_wrap_fits_file_image_2D
             || format == mf_wrap_fits_file_image_3D,
             CPL_ERROR_ILLEGAL_INPUT, NULL);

  cpl_ensure(   type == mf_wrap_fits_table
             || type == mf_wrap_fits_vector
             || type == mf_wrap_fits_matrix
             || type == mf_wrap_fits_image
             || type == mf_wrap_fits_cube,
             CPL_ERROR_ILLEGAL_INPUT, NULL);

  cpl_ensure(n_ext > 0, CPL_ERROR_ILLEGAL_INPUT, NULL);


  /* Reserve memory */
  mf_wrap_fits *data = cpl_malloc(sizeof(mf_wrap_fits));

  /* Initialize */
  data->filename = cpl_strdup(filename);
  data->format   = format;
  data->type     = type;
  data->n_ext    = n_ext;
  data->v_ext    = cpl_calloc(n_ext, sizeof(mf_wrap_fits_ext));

  /* Nullify extension variables */
  for (cpl_size ext = 0; ext < n_ext; ext++) {

      data->v_ext[ext].header        = NULL;
      data->v_ext[ext].table         = NULL;
      data->v_ext[ext].vector        = NULL;
      data->v_ext[ext].matrix        = NULL;
      data->v_ext[ext].image         = NULL;
      data->v_ext[ext].cube          = NULL;

      data->v_ext[ext].spectrum_head = NULL;
      data->v_ext[ext].spectrum_data = NULL;
  }

  /* Check and return */
  if (!cpl_error_get_code()) {
      return data;
  } else {
      mf_wrap_fits_delete(data);
      return NULL;
  }
}

/* ---------------------------------------------------------------------------*/
/**
 * @brief Delete mf_wrap_fits structure.
 *
 * @param data               mf_wrap_data_structure structure for delete.
 *
 */
/* ---------------------------------------------------------------------------*/
void mf_wrap_fits_delete(
    mf_wrap_fits            *data)
{
  if (data) {

      if (data->filename) cpl_free(data->filename);

      /* Free extension variables */
      if (data->v_ext) {

          for (cpl_size ext = 0; ext < data->n_ext; ext++) {

              if (data->v_ext[ext].header       ) cpl_propertylist_delete(data->v_ext[ext].header       );
              if (data->v_ext[ext].table        ) cpl_table_delete(       data->v_ext[ext].table        );
              if (data->v_ext[ext].vector       ) cpl_vector_delete(      data->v_ext[ext].vector       );
              if (data->v_ext[ext].matrix       ) cpl_matrix_delete(      data->v_ext[ext].matrix       );
              if (data->v_ext[ext].image        ) cpl_image_delete(       data->v_ext[ext].image        );
              if (data->v_ext[ext].cube         ) cpl_imagelist_delete(   data->v_ext[ext].cube         );

              if (data->v_ext[ext].spectrum_head) cpl_propertylist_delete(data->v_ext[ext].spectrum_head);
              if (data->v_ext[ext].spectrum_data) cpl_table_delete(       data->v_ext[ext].spectrum_data);
          }

          cpl_free(data->v_ext);
      }

      cpl_free(data);
  }
}

/* ---------------------------------------------------------------------------*/
/**
 * @brief  Reads a FITS table with an arbitrary number of extensions and puts the data into an molecfit_data structure consisting of an array of CPL tables and CPL property lists.
 *
 * @param filename           Path and name of input FITS.
 * @param is_matrix          Boolean to become the load data in cpl_matrix.
 *
 * @return mf_wrap_fits   FITS file loaded is everything is OK or NULL in other case.
 *
 */
/* ---------------------------------------------------------------------------*/
mf_wrap_fits * mf_wrap_fits_load(
    const char                  *filename,
    cpl_boolean                 is_matrix)
{
   return mf_wrap_fits_load_verbose(filename,is_matrix,CPL_TRUE);
}

mf_wrap_fits * mf_wrap_fits_load_verbose(
    const char                  *filename,
    cpl_boolean                 is_matrix,
    cpl_boolean                  verbose )
{
  cpl_msg_info(cpl_func,"mf_wrap_fits_load_verbose");
  /* Check input */
  cpl_ensure(filename, CPL_ERROR_NULL_INPUT, NULL);
  cpl_msg_info(cpl_func,"check_input");

  /* Get FITS file type */
  mf_wrap_fits_file_format format = mf_wrap_fits_get_file_format(filename,verbose);
  cpl_error_code err = cpl_error_get_code();

  /* Local variables */
  mf_wrap_fits *data = NULL;
  cpl_size        n_ext = -1;

  if (!err) {

      cpl_error_ensure(!is_matrix || format == mf_wrap_fits_file_image_2D, cpl_error_get_code(),
                       return NULL, "Bad FITS format in file : %s",
                       filename);

      if (verbose) cpl_msg_info(cpl_func, "File[%s] - Loading FITS header/data ... (all extensions)", filename);

      n_ext = 1 + cpl_fits_count_extensions(filename);

      switch(format) {

          case mf_wrap_fits_file_table :
              data = mf_wrap_fits_create(filename, format, mf_wrap_fits_table, n_ext);
              break;

          case mf_wrap_fits_file_image_1D :
              data = mf_wrap_fits_create(filename, format, mf_wrap_fits_vector, n_ext);
              break;

          case mf_wrap_fits_file_image_2D :
              if (is_matrix) data = mf_wrap_fits_create(filename, format, mf_wrap_fits_matrix, n_ext);
              else           data = mf_wrap_fits_create(filename, format, mf_wrap_fits_image,  n_ext);
              break;

          case mf_wrap_fits_file_image_3D :
              data = mf_wrap_fits_create(filename, format, mf_wrap_fits_cube, n_ext);
              break;

          default :
              /* mf_wrap_fits_file_none || mf_wrap_fits_file_invalid */
              cpl_error_set_message(cpl_func, CPL_ERROR_BAD_FILE_FORMAT,
                                    "Unexpected FITS file format");
              break;
      }

      cpl_error_ensure(data, CPL_ERROR_FILE_IO,
                       return NULL, "File opening failed: %s",
                       filename);

      err = cpl_error_get_code();
  }

  /* Read all extensions in the FITS file : header + data */
  if (!err) {

      for (cpl_size ext = 0; ext < n_ext && !err; ext++) {

          /* Load header */
          data->v_ext[ext].header = cpl_propertylist_load(filename, ext);
          if (verbose) cpl_msg_info(cpl_func, "Loaded cpl_propertylist file: %s (extension=%lld) --> [HEADER   ]", filename, ext);

          /* Get error code */
          err = cpl_error_get_code();
          if (!err) {

              /* Not throw and error if not extension data */
              cpl_errorstate pre_state = cpl_errorstate_get();
              cpl_image      *img_mat  = NULL;

              switch (data->type) {

                case mf_wrap_fits_table :
                    data->v_ext[ext].table = cpl_table_load(filename, ext, 0);
                    if (data->v_ext[ext].table && verbose) cpl_msg_info(cpl_func, "Loaded cpl_table        file: %s (extension=%lld) --> [BINTABLE ]", filename, ext);
                    else cpl_errorstate_set(pre_state);
                    break;

                case mf_wrap_fits_vector:
                    data->v_ext[ext].vector = cpl_vector_load(filename, ext);
                    if (data->v_ext[ext].vector && verbose) cpl_msg_info(cpl_func, "Loaded cpl_vector       file: %s (extension=%lld) --> [IMAGE(1D)]", filename, ext);
                    else cpl_errorstate_set(pre_state);
                    break;

                case mf_wrap_fits_matrix:
                    img_mat = cpl_image_load(filename, CPL_TYPE_DOUBLE, 0, ext);
                    if (img_mat) {
                        if (verbose) cpl_msg_info(cpl_func, "Loaded cpl_matrix       file: %s (extension=%lld) --> [IMAGE(2D)]", filename, ext);
                        cpl_matrix *mat = cpl_matrix_wrap( cpl_image_get_size_y(     img_mat),
                                                           cpl_image_get_size_x(     img_mat),
                                                           cpl_image_get_data_double(img_mat));
                        data->v_ext[ext].matrix = cpl_matrix_duplicate(mat);
                        cpl_matrix_unwrap(mat);
                        cpl_image_delete(img_mat);
                    } else {
                        cpl_errorstate_set(pre_state);
                        img_mat = NULL;
                    }
                    break;

                case mf_wrap_fits_image:
                    data->v_ext[ext].image = cpl_image_load(filename, CPL_TYPE_UNSPECIFIED, 0, ext);
                    if (data->v_ext[ext].image && verbose) cpl_msg_info(cpl_func, "Loaded cpl_image (2D)   file: %s (extension=%lld) --> [IMAGE(2D)]", filename, ext);
                    else cpl_errorstate_set(pre_state);
                    break;

                case mf_wrap_fits_cube:
                    data->v_ext[ext].cube = cpl_imagelist_load(filename, CPL_TYPE_UNSPECIFIED, ext);
                    if (data->v_ext[ext].cube) {
                        cpl_size size = cpl_imagelist_get_size(data->v_ext[ext].cube);
                        if (verbose) cpl_msg_info(cpl_func, "Loaded cpl_image (3D)   file: %s (extension=%lld) --> [IMAGE(3D) - n_img = %lld]", filename, ext, size);
                    } else {
                        cpl_errorstate_set(pre_state);
                    }
                    break;

                default: break;
              }

              err = cpl_error_get_code();
          }
      }
  }
  //cpl_msg_info(cpl_func,"mf_wrap_fits (load): %s",err);

  /* Check and return */
  if (!err) {
      return data;
  } else {
      mf_wrap_fits_delete(data);
      return NULL;
  }
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Writes data of an molecfit_data structure into a FITS table. The number of FITS extensions depends on the content of the molecfit_data structure.
 *
 * @param  all_frameset    Input frameset       in the recipe.
 * @param  used_frameset   frameset with the frame affected in the saving function.
 * @param  parlist         Input parameter list in the recipe.
 * @param  recipe          Name of the recipe
 * @param  tag             Tag in the ESO PRO CATG property.
 * @param  data            mf_wrap_fits to load in the disk.
 * @param  out_filename    Name of the output fits file, if NULL the funtion compose the name with the tag
 *
 * @return cpl_error_code  CPL_ERROR_NONE if everything is OK or cpl_error_code in other case.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code mf_wrap_fits_write(
    cpl_frameset                *all_frameset,
    cpl_frameset                *used_frameset,
    const cpl_parameterlist     *parlist,
    const char                  *recipe,
    const char                  *tag,
    const mf_wrap_fits         *data,
    const char                  *out_filename)
{
  cpl_msg_info(cpl_func,"mf_wrap_fits_write");
  /* Check Filename */
  cpl_ensure(data, CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT);


  /* Local variables */
  cpl_error_code err = CPL_ERROR_NONE;


  /* Get filename: If filename not NULL get this, if NULL create with original data filename */
  char *filename = out_filename ? cpl_strdup(out_filename) : cpl_sprintf("out_%s", data->filename);


  /* Set applist with the input recipe parameters */
  cpl_propertylist *applist = cpl_propertylist_duplicate(data->v_ext[0].header);
  cpl_propertylist_update_string(applist, CPL_DFS_PRO_CATG, tag);


  /* Save base file : Primary extension */
  cpl_boolean primary_extension_write = CPL_FALSE;
  switch (data->type) {

    case mf_wrap_fits_table :
      if (data->v_ext[0].table) {
          cpl_msg_info(cpl_func, "Write output cpl_table     in primary extension --> 'cpl_table *' with cpl_dfs_save_table(...) function in file : %s", filename);
          err = cpl_dfs_save_table(  all_frameset, NULL, parlist, used_frameset, NULL,
                                     data->v_ext[0].table, data->v_ext[0].header,
                                     recipe, applist, NULL, PACKAGE "/" PACKAGE_VERSION,
                                     filename);
          primary_extension_write = CPL_TRUE;
      }
      break;

    case mf_wrap_fits_vector :
      if (data->v_ext[0].vector) {

          cpl_msg_info(cpl_func, "Write output cpl_vector    in primary extension --> 'cpl_vector *' with cpl_dfs_save_image(...) function in file : %s", filename);
          err = mf_wrap_dfs_save_vector(  all_frameset, NULL, parlist, used_frameset, NULL,
                                        data->v_ext[0].vector, CPL_TYPE_DOUBLE,
                                        recipe, applist, NULL, PACKAGE "/" PACKAGE_VERSION,
                                        filename);
          primary_extension_write = CPL_TRUE;
      }
      break;

    case mf_wrap_fits_matrix :
      if (data->v_ext[0].matrix) {
          cpl_image *img_mat = cpl_image_wrap_double(cpl_matrix_get_ncol(data->v_ext[0].matrix),
                                                     cpl_matrix_get_nrow(data->v_ext[0].matrix),
                                                     cpl_matrix_get_data(data->v_ext[0].matrix));
          cpl_msg_info(cpl_func, "Write output cpl_matrix    in primary extension --> 'cpl_matrix *' with cpl_dfs_save_image(...) function in file : %s", filename);
          err = cpl_dfs_save_image(  all_frameset, NULL, parlist, used_frameset, NULL,
                                     img_mat, CPL_TYPE_DOUBLE,
                                     recipe, applist, NULL, PACKAGE "/" PACKAGE_VERSION,
                                     filename);
          cpl_image_unwrap(img_mat);
          primary_extension_write = CPL_TRUE;
      }
      break;

    case mf_wrap_fits_image :
      if (data->v_ext[0].image) {
          cpl_msg_info(cpl_func, "Write output cpl_image(2D) in primary extension --> 'cpl_image *' with cpl_dfs_save_image(...) function in file : %s", filename);
          err = cpl_dfs_save_image(  all_frameset, NULL, parlist, used_frameset, NULL,
                                     data->v_ext[0].image, CPL_TYPE_UNSPECIFIED,
                                     recipe, applist, NULL, PACKAGE "/" PACKAGE_VERSION,
                                     filename);
          primary_extension_write = CPL_TRUE;
      }
      break;

    case mf_wrap_fits_cube :
      if (data->v_ext[0].cube) {
          cpl_msg_info(cpl_func, "Write output cpl_image(3D) in primary extension --> 'cpl_imagelist *' with cpl_dfs_save_imagelist(...) function in file : %s", filename);
          err = cpl_dfs_save_imagelist(  all_frameset, NULL, parlist, used_frameset, NULL,
                                         data->v_ext[0].cube, CPL_TYPE_UNSPECIFIED,
                                         recipe, applist, NULL, PACKAGE "/" PACKAGE_VERSION,
                                         filename);
          primary_extension_write = CPL_TRUE;
      }
      break;

    default : break;
  }

  /* Save only a HEADER in the primary extension : No data in the primary extension */
  if (!primary_extension_write) {
      cpl_msg_info(cpl_func, "Write output cpl_propertylist in primary extension --> 'cpl_propertylist *' with cpl_dfs_save_propertylist(...) function in file : %s", filename);
      err = cpl_dfs_save_propertylist(  all_frameset, NULL, parlist, used_frameset, NULL,
                                        recipe, applist, NULL, PACKAGE "/" PACKAGE_VERSION,
                                        filename);
  }

  /* Save following extension's */
  for (cpl_size ext = 1; ext < data->n_ext && !err; ext++) {

      cpl_boolean extension_write = CPL_FALSE;
      switch (data->type) {

        case mf_wrap_fits_table :
          if (data->v_ext[ext].table) {
              cpl_msg_info(cpl_func, "Write output cpl_table     in extension[%lld]  --> 'cpl_table *' with cpl_dfs_save_table(...) function in file : %s", ext, filename);
              cpl_table_save(data->v_ext[ext].table,  NULL, data->v_ext[ext].header, filename, CPL_IO_EXTEND);
              extension_write = CPL_TRUE;
          }
          break;

        case mf_wrap_fits_vector :
          if (data->v_ext[ext].vector) {
              cpl_msg_info(cpl_func, "Write output cpl_vector    in extension[%lld] --> 'cpl_vector *' with cpl_dfs_save_image(...) function in file : %s", ext, filename);
              cpl_vector_save(data->v_ext[ext].vector, filename, CPL_TYPE_DOUBLE, data->v_ext[ext].header, CPL_IO_EXTEND);
              extension_write = CPL_TRUE;
          }
          break;

        case mf_wrap_fits_matrix :
          if (data->v_ext[ext].matrix) {
              cpl_image *img_mat = cpl_image_wrap_double(cpl_matrix_get_ncol(data->v_ext[ext].matrix),
                                                         cpl_matrix_get_nrow(data->v_ext[ext].matrix),
                                                         cpl_matrix_get_data(data->v_ext[ext].matrix));
              cpl_msg_info(cpl_func, "Write output cpl_matrix    in extension[%lld] --> 'cpl_matrix *' with cpl_dfs_save_image(...) function in file : %s", ext, filename);
              cpl_image_save(img_mat, filename, CPL_TYPE_DOUBLE, data->v_ext[ext].header, CPL_IO_EXTEND);
              cpl_image_unwrap(img_mat);
              extension_write = CPL_TRUE;
          }
          break;

        case mf_wrap_fits_image :
          if (data->v_ext[ext].image) {
              cpl_msg_info(cpl_func, "Write output cpl_image(2D) in extension[%lld] --> 'cpl_image *' with cpl_dfs_save_image(...) function in file : %s", ext, filename);
              cpl_image_save(data->v_ext[ext].image, filename, CPL_TYPE_UNSPECIFIED, data->v_ext[ext].header, CPL_IO_EXTEND);
              extension_write = CPL_TRUE;
          }
          break;

        case mf_wrap_fits_cube :
          if (data->v_ext[ext].cube) {
              cpl_msg_info(cpl_func, "Write output cpl_image(3D) in extension[%lld] --> 'cpl_imagelist *' with cpl_dfs_save_imagelist(...) function in file : %s", ext, filename);
              cpl_imagelist_save(data->v_ext[ext].cube, filename, CPL_TYPE_UNSPECIFIED, data->v_ext[ext].header, CPL_IO_EXTEND);
              extension_write = CPL_TRUE;
          }
          break;

        default : break;
      }

      /* Save only a HEADER in the primary extension : No data in the primary extension */
      if (!extension_write) {
          cpl_msg_info(cpl_func, "Write output cpl_propertylist in extension[%lld] --> 'cpl_propertylist *' with cpl_dfs_save_propertylist(...) function in file : %s", ext, filename);
          cpl_propertylist_save(data->v_ext[ext].header, filename, CPL_IO_EXTEND);
      }

      err = cpl_error_get_code();
  }

  /* Cleanup */
  cpl_free(filename);
  cpl_propertylist_delete(applist);

  return err;
}


/** @cond PRIVATE */

/* ---------------------------------------------------------------------------*/
/**
 * @brief Identifies the FITS type (table or image) by means of the header keyword XTENSION of the first FITS extension.
 *
 * @param  filename                       Input filename to check the format.
 *
 * @return mf_wrap_fits_file_format      Format of the structure.
 *
 * @note   If an extension is not present, the image format is assumed.
 *           The keyword NAXIS is used to distinguish between 1D, 2D, and 3D images.
 *           It is assumed that all extensions have the same FITS type.
 *
 * @note   The routine returns :
 *           #  0: for non-FITS format (e.g. ASCII)
 *           #  1: for FITS table
 *           #  2: for 1D FITS image
 *           #  3: for 2D FITS image
 *           #  4: for 3D FITS image
 *           # -1: in the case of errors.
 *
 */
/* ---------------------------------------------------------------------------*/

mf_wrap_fits_file_format mf_wrap_fits_get_file_format(
    const char                  *filename,
    const cpl_boolean           verbose)
{
    cpl_ensure(filename, CPL_ERROR_NULL_INPUT, mf_wrap_fits_file_none);

    cpl_error_code err = CPL_ERROR_NONE;

    mf_wrap_fits_file_format format = mf_wrap_fits_file_none;

    if (verbose) cpl_msg_info(cpl_func, "File[%s] - Check FITS file format ... (Searching first extension with data)", filename);

    /* Get sum of extension : Without primary, only others */
    cpl_size n_ext = 1 + cpl_fits_count_extensions(filename);
    if (n_ext < 1) {

        err = cpl_error_set_message(cpl_func, CPL_ERROR_BAD_FILE_FORMAT,
                              "Unexpected non-fits file structure: %s (No extensions)",
                              filename);
        format = mf_wrap_fits_file_invalid;

    } else {

        format = mf_wrap_fits_file_no_data;

        for (cpl_size ext = 0; ext < n_ext && format == mf_wrap_fits_file_no_data && !err; ext++) {

            if (verbose) cpl_msg_info(cpl_func, "File[%s] - Load  header in the extension[%lld]", filename, ext);
            cpl_propertylist *header = cpl_propertylist_load(filename, ext);
            if (!header) {

                err = cpl_error_set_message(cpl_func, CPL_ERROR_BAD_FILE_FORMAT,
                                            "Unexpected non-fits file structure: %s (No extensions)",
                                            filename);
                format = mf_wrap_fits_file_invalid;

            } else {

                const cpl_property *prop_naxis = cpl_propertylist_get_property_const(header, MOLECFIT_FITS_KEYWORD_NAXIS);
                int                naxis       = prop_naxis ? cpl_property_get_int(prop_naxis) : -1;

                if (verbose) cpl_msg_info(cpl_func, "File[%s] - Check data   in the extension[%lld] : KEYWORD[%s] = %d", filename, ext, MOLECFIT_FITS_KEYWORD_NAXIS, naxis);

                if (naxis < 0 || naxis > 3) {

                    err = cpl_error_set_message(cpl_func, CPL_ERROR_BAD_FILE_FORMAT,
                                                "Unexpected file structure: %s (keyword %s -> Not exist)",
                                                filename, MOLECFIT_FITS_KEYWORD_NAXIS);

                    format = mf_wrap_fits_file_invalid;

                } else if (naxis > 0) {

                    /* Check header KEYWORD : XTENSION or directly NAXIS */
                    const cpl_property *prop_xtension = cpl_propertylist_get_property_const(header, MOLECFIT_FITS_KEYWORD_XTENSION);
                    cpl_boolean        check_NAXIS    = prop_xtension ? CPL_FALSE : CPL_TRUE;

                    if (prop_xtension) {

                        const char *xtension = cpl_property_get_string(prop_xtension);
                        if (verbose) cpl_msg_info(cpl_func, "File[%s] - Check           in the extension[%lld]: KEYWORD[%s] = %s", filename, ext, MOLECFIT_FITS_KEYWORD_XTENSION, xtension);

                        if (!strcmp(xtension, MOLECFIT_FITS_KEYWORD_BINTABLE)) {

                            format = mf_wrap_fits_file_table;

                        } else if (!strcmp(xtension, MOLECFIT_FITS_KEYWORD_IMAGE)){

                            check_NAXIS = CPL_TRUE;

                        } else {

                            err = cpl_error_set_message(cpl_func, CPL_ERROR_BAD_FILE_FORMAT,
                                                        "Unexpected file structure: %s (keyword %s != %s or %s)",
                                                        filename, MOLECFIT_FITS_KEYWORD_XTENSION, MOLECFIT_FITS_KEYWORD_IMAGE, MOLECFIT_FITS_KEYWORD_BINTABLE);

                            format = mf_wrap_fits_file_invalid;
                        }
                    }

                    /* Check Image FITS extension */
                    if (check_NAXIS) {
                        if (     naxis == 1) format = mf_wrap_fits_file_image_1D;  /* 1D FITS image */
                        else if (naxis == 2) format = mf_wrap_fits_file_image_2D;  /* 2D FITS image */
                        else if (naxis == 3) format = mf_wrap_fits_file_image_3D;  /* 3D FITS image */
                        else {
                            err = cpl_error_set_message(cpl_func, CPL_ERROR_BAD_FILE_FORMAT,
                                                        "Unexpected file structure: %s (keyword NAXIS = 0 in the last extension[%lld])",
                                                        filename, n_ext);

                            format = mf_wrap_fits_file_invalid;
                        }
                    }
                }

                /* Cleanup */
                if (header) cpl_propertylist_delete(header);
            }
        }
    }

    /* Check data */
    if (!err) {
        switch (format) {

            case mf_wrap_fits_file_no_data:
               cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                        "No data extensions, only header extensions in file : %s",filename);
                break;

            case mf_wrap_fits_file_table:
                if (verbose) cpl_msg_info(cpl_func, "File[%s] - Found format = BINTABLE!",  filename);
                break;

            case mf_wrap_fits_file_image_1D:
                if(verbose) cpl_msg_info(cpl_func, "File[%s] - Found format = IMAGE(1D)!", filename);
                break;

            case mf_wrap_fits_file_image_2D:
                if(verbose) cpl_msg_info(cpl_func, "File[%s] - Found format = IMAGE(2D)!", filename);
                break;

            case mf_wrap_fits_file_image_3D:
                if(verbose) cpl_msg_info(cpl_func, "File[%s] - Found format = IMAGE(3D)!", filename);
                break;

            default:                
                cpl_error_set_message(cpl_func, CPL_ERROR_BAD_FILE_FORMAT,
                                  "Unexpected file structure: %s (keyword NAXIS = 0 in the last extension[%lld])",
                                  filename, n_ext);

                format = mf_wrap_fits_file_invalid;

        }; /* End Switch*/
                
    }

    return format;
}

/** @endcond */


/**@}*/

