/*
 * This file is part of the KMOS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef KMOS_MOLECFIT_H
#define KMOS_MOLECFIT_H

/*----------------------------------------------------------------------------*/
/**
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

#include <string.h>
#include <math.h>

#include <cpl.h>

#include <telluriccorr.h>

#include "kmo_constants.h"
#include "kmo_utils.h"
#include "kmo_dfs.h"
#include "kmos_pfits.h"


/*----------------------------------------------------------------------------*/
/**
 *                              Defines
 */
/*----------------------------------------------------------------------------*/

/* KMOS_MOLECFIT RECIPES */
#define KMOS_MOLECFIT_MODEL                       "kmos_molecfit_model"
#define KMOS_MOLECFIT_CALCTRANS                   "kmos_molecfit_calctrans"
#define KMOS_MOLECFIT_CORRECT                     "kmos_molecfit_correct"

/* KMOS_MOLECFIT RECIPE PARAMETERS (MAY BE UPPER OR lower CASE) */
#define KMOS_MOLECFIT_PARAMETER_WRANGE             "WAVE_RANGE"
#define KMOS_MOLECFIT_PARAMETER_PROCESS_IFUS       "PROCESS_IFUS"
#define KMOS_MOLECFIT_PARAMETER_USE_INPUT_KERNEL   "USE_INPUT_KERNEL"
#define KMOS_MOLECFIT_PARAMETER_LIST               MF_COL_LIST_MOLECULES
#define KMOS_MOLECFIT_PARAMETER_FIT                MF_COL_FIT_MOLECULES
#define KMOS_MOLECFIT_PARAMETER_RELATIVE_VALUE     MF_COL_REL_COL
#define KMOS_MOLECFIT_PARAMETER_SUPPRESS_EXTENSION "suppress_extension"
#define KMOS_MOLECFIT_PARAMETER_IFU_1              "IFU_1"
#define KMOS_MOLECFIT_PARAMETER_IFU_2              "IFU_2"
#define KMOS_MOLECFIT_PARAMETER_IFU_3              "IFU_3"
#define KMOS_MOLECFIT_PARAMETER_IFU_4              "IFU_4"
#define KMOS_MOLECFIT_PARAMETER_IFU_5              "IFU_5"
#define KMOS_MOLECFIT_PARAMETER_IFU_6              "IFU_6"
#define KMOS_MOLECFIT_PARAMETER_IFU_7              "IFU_7"
#define KMOS_MOLECFIT_PARAMETER_IFU_8              "IFU_8"
#define KMOS_MOLECFIT_PARAMETER_IFU_9              "IFU_9"
#define KMOS_MOLECFIT_PARAMETER_IFU_10             "IFU_10"
#define KMOS_MOLECFIT_PARAMETER_IFU_11             "IFU_11"
#define KMOS_MOLECFIT_PARAMETER_IFU_12             "IFU_12"
#define KMOS_MOLECFIT_PARAMETER_IFU_13             "IFU_13"
#define KMOS_MOLECFIT_PARAMETER_IFU_14             "IFU_14"
#define KMOS_MOLECFIT_PARAMETER_IFU_15             "IFU_15"
#define KMOS_MOLECFIT_PARAMETER_IFU_16             "IFU_16"
#define KMOS_MOLECFIT_PARAMETER_IFU_17             "IFU_17"
#define KMOS_MOLECFIT_PARAMETER_IFU_18             "IFU_18"
#define KMOS_MOLECFIT_PARAMETER_IFU_19             "IFU_19"
#define KMOS_MOLECFIT_PARAMETER_IFU_20             "IFU_20"
#define KMOS_MOLECFIT_PARAMETER_IFU_21             "IFU_21"
#define KMOS_MOLECFIT_PARAMETER_IFU_22             "IFU_22"
#define KMOS_MOLECFIT_PARAMETER_IFU_23             "IFU_23"
#define KMOS_MOLECFIT_PARAMETER_IFU_24             "IFU_24"
//kmos_molecfit_calctrans scale_pwv for transmission function scaling
#define KMOS_MOLECFIT_PARAMETER_SCALE_PWV          "SCALE_PWV"
/*#define KMOS_MOLECFIT_PARAMETER_SCALE_PWV_DESC     "Value read from science file to scale the precipitable water vapor (PWV) of the telluric spectra.\n"\
                                                      " If set to 'auto', the mean value of TEL AMBI IWV START/END is read from the header of the input science file.\n"\
                                                      " If set to a user-provided numerical value, use this as the scale factor.\n"\
                                                      " If set to a FITS header keyword, read in the scale factor from the header of the input science file using this keyword.\n"\
                                                      " If set to 'none', perform no scaling.\n" */
//#define KMOS_MOLECFIT_PARAMETER_SCALE_PWV_INIT     "auto"

/* KMOS_MOLECFIT HEADER KEYWORDS (MUST BE UPPER CASE) */
#define KMOS_MOLECFIT_KEYWORD_WRANGE             KMOS_MOLECFIT_PARAMETER_WRANGE
#define KMOS_MOLECFIT_KEYWORD_PROCESS_IFUS       KMOS_MOLECFIT_PARAMETER_PROCESS_IFUS
#define KMOS_MOLECFIT_KEYWORD_USE_INPUT_KERNEL   KMOS_MOLECFIT_PARAMETER_USE_INPUT_KERNEL
#define KMOS_MOLECFIT_KEYWORD_LIST               KMOS_MOLECFIT_PARAMETER_LIST
#define KMOS_MOLECFIT_KEYWORD_FIT                KMOS_MOLECFIT_PARAMETER_FIT
#define KMOS_MOLECFIT_KEYWORD_RELATIVE_VALUE     KMOS_MOLECFIT_PARAMETER_RELATIVE_VALUE
#define KMOS_MOLECFIT_KEYWORD_SUPPRESS_EXTENSION "SUPPRESS_EXTENSION"
#define KMOS_MOLECFIT_KEYWORD_IFU_1              KMOS_MOLECFIT_PARAMETER_IFU_1
#define KMOS_MOLECFIT_KEYWORD_IFU_2              KMOS_MOLECFIT_PARAMETER_IFU_2
#define KMOS_MOLECFIT_KEYWORD_IFU_3              KMOS_MOLECFIT_PARAMETER_IFU_3
#define KMOS_MOLECFIT_KEYWORD_IFU_4              KMOS_MOLECFIT_PARAMETER_IFU_4
#define KMOS_MOLECFIT_KEYWORD_IFU_5              KMOS_MOLECFIT_PARAMETER_IFU_5
#define KMOS_MOLECFIT_KEYWORD_IFU_6              KMOS_MOLECFIT_PARAMETER_IFU_6
#define KMOS_MOLECFIT_KEYWORD_IFU_7              KMOS_MOLECFIT_PARAMETER_IFU_7
#define KMOS_MOLECFIT_KEYWORD_IFU_8              KMOS_MOLECFIT_PARAMETER_IFU_8
#define KMOS_MOLECFIT_KEYWORD_IFU_9              KMOS_MOLECFIT_PARAMETER_IFU_9
#define KMOS_MOLECFIT_KEYWORD_IFU_10             KMOS_MOLECFIT_PARAMETER_IFU_10
#define KMOS_MOLECFIT_KEYWORD_IFU_11             KMOS_MOLECFIT_PARAMETER_IFU_11
#define KMOS_MOLECFIT_KEYWORD_IFU_12             KMOS_MOLECFIT_PARAMETER_IFU_12
#define KMOS_MOLECFIT_KEYWORD_IFU_13             KMOS_MOLECFIT_PARAMETER_IFU_13
#define KMOS_MOLECFIT_KEYWORD_IFU_14             KMOS_MOLECFIT_PARAMETER_IFU_14
#define KMOS_MOLECFIT_KEYWORD_IFU_15             KMOS_MOLECFIT_PARAMETER_IFU_15
#define KMOS_MOLECFIT_KEYWORD_IFU_16             KMOS_MOLECFIT_PARAMETER_IFU_16
#define KMOS_MOLECFIT_KEYWORD_IFU_17             KMOS_MOLECFIT_PARAMETER_IFU_17
#define KMOS_MOLECFIT_KEYWORD_IFU_18             KMOS_MOLECFIT_PARAMETER_IFU_18
#define KMOS_MOLECFIT_KEYWORD_IFU_19             KMOS_MOLECFIT_PARAMETER_IFU_19
#define KMOS_MOLECFIT_KEYWORD_IFU_20             KMOS_MOLECFIT_PARAMETER_IFU_20
#define KMOS_MOLECFIT_KEYWORD_IFU_21             KMOS_MOLECFIT_PARAMETER_IFU_21
#define KMOS_MOLECFIT_KEYWORD_IFU_22             KMOS_MOLECFIT_PARAMETER_IFU_22
#define KMOS_MOLECFIT_KEYWORD_IFU_23             KMOS_MOLECFIT_PARAMETER_IFU_23
#define KMOS_MOLECFIT_KEYWORD_IFU_24             KMOS_MOLECFIT_PARAMETER_IFU_24
#define KMOS_MOLECFIT_KEYWORD_SCALE_PWV          KMOS_MOLECFIT_PARAMETER_SCALE_PWV

/* KMOS counters */
#define N_IFUS                         24
#define N_KERNEL_LIBRARY_EXTENSIONS    N_IFUS * 6

/* Wavelength detection limits in KMOS */
#define KMOS_WAVELENGTH_START          0.8
#define KMOS_WAVELENGTH_END            2.5

/* Initial  FWHM of the Gaussian in pixels have Grating dependency */
#define RES_GAUSS_IZ                   1.84
#define RES_GAUSS_YJ                   1.82
#define RES_GAUSS_H                    1.76
#define RES_GAUSS_K                    1.73
#define RES_GAUSS_HK                   2.06

#define MOLECFIT_MAX_POLY_FIT          8


/*----------------------------------------------------------------------------*/
/**
 *                 Typedefs: Structs and enum types
 */
/*----------------------------------------------------------------------------*/

typedef struct {
  char                *name;                  /* Kernel extension name                                                                */
  int                 num;                    /* IFU number                                                                           */
  int                 ext;                    /* IFU.ANGLE num  extension in the kernel library                                       */
  cpl_propertylist    *header_ext_data;       /* Header property list in IFU data  extension of kernel library for the specific angle */
  cpl_propertylist    *header_ext_noise;      /* Header property list in IFU noise extension of kernel library for the specific angle */
  double              header_CRVAL1;          /* RA at ref. pixel [deg]                                                               */
  double              header_CD2_2;           /* Pixel resolution in x [deg]                                                          */
  cpl_matrix          *data;                  /* Convolution kernel values specifics in the grating                                   */
} kmos_kernel;

typedef struct {
  cpl_boolean         process;                /* Flag: Process this IFU or not                                                        */
  int                 map;                    /* Mapping of ifu (in kmos_molecfit_calctrans relation between IFU_X and IFU_Y          */
  char                *name;                  /* IFU extension name   in the raw file                                                 */
  int                 num;                    /* IFU number           in the raw file                                                 */
  int                 ext;                    /* IFU extension number in the raw file                                                 */
  cpl_propertylist    *header_ext_data;       /* Header property list in IFU data  extension of raw file                              */
  cpl_propertylist    *header_ext_noise;      /* Header property list in IFU noise extension of raw file                              */
  cpl_propertylist    *header_1D_data;        /* Header property list convert from 3D DataCube to 1D spectrum, if necessary           */
  double              header_CRVAL1;          /* RA at ref. pixel [deg]                                                               */
  double              header_CDELT1;          /* Pixel resolution in x [deg]                                                          */
  cpl_table           *data;                  /* Wavelength data in the input file. Contain only one column with wavelengths          */
  double              median;                 /* Median of data spectrum                                                              */
  kmos_kernel         kernel;                 /* Associate kernel to one IFU                                                          */
} kmos_spectrum;

typedef enum {
  GRATING_IZ = 0,                            /* KMOS IZ grating                                                                      */
  GRATING_YJ,                                /* KMOS YJ grating                                                                      */
  GRATING_H,                                 /* KMOS H  grating                                                                      */
  GRATING_K,                                 /* KMOS K  grating                                                                      */
  GRATING_HK                                 /* KMOS HK grating                                                                      */
} kmos_grating_type;

typedef struct {
  const char          *name;                  /* Specific unique grating to these spectrums data: 'IZ', 'YJ', 'H', 'K', 'HK'          */
  kmos_grating_type   type;                   /* Type that match with grating_name                                                    */
  const char          *wave_range;            /* wave_ranges{ini1,end1,...,iniN,endN} defined by the user to fit in the grating       */
  cpl_table           *incl_wave_ranges;      /* Wavelength ranges     to fit in the grating. Contains 2 columns [MF_COL_WAVE_RANGE_LOWER, MF_COL_WAVE_RANGE_UPPER] */
  const char          *list_molec;            /* list_molec{H2O,CO,CO2,CH4,O2} str    defined by the user to fit in the grating       */
  const char          *fit_molec;             /* fit_molec {1,  0, 1,  1,  0 } flag   defined by the user to fit in the grating       */
  const char          *rel_col;               /* relDensity{1.,1.,1.06,1.,1. } double defined by the user to fit in the grating       */
  cpl_table           *molecules;             /* Molecules to fit. Contains 3 columns [MF_COL_LIST_MOLECULES, MF_COL_FIT_MOLECULES, MF_COL_REL_COL] */
} kmos_grating;

typedef struct {
  cpl_boolean         fit;                    /* Flag: Polynomial fit                                                                 */
  cpl_size            n;                      /* Polynomial degree                                                                    */
  double              const_val;              /* Initial value of the constant term of the polynomial fit                             */
} mf_fit;

typedef struct {
  cpl_propertylist    *parms;                 /* Input recipe parameters to be save in the output res_out fits file                   */
  kmos_spectrum       ifus[N_IFUS];           /* Data ifu spectrums. Contains name, property list, data and scale of wavelength       */
  cpl_propertylist    *header_spectrums;      /* Primary property list header in the raw STAR_SPEC file                               */
  float               rot_angle;              /* Rotator relative to nasmyth [deg]                                                    */
  kmos_grating        grating;                /* Grating data by user or default values                                               */
  cpl_boolean         use_input_kernel;       /* The parameters below are ignored if use_input_kernel                                 */
  cpl_propertylist    *header_kernels;        /* Primary property list header in the kernel library file                              */
  cpl_boolean         kernmode;               /* Voigt profile approx. or independent Gauss and Lorentz                               */
  double              kernfac;                /* Size of kernel in FWHM                                                               */
  cpl_boolean         varkern;                /* Does the kernel size increase linearly with wavelength?                              */
  mf_fit              fit_continuum;          /* Continuum fit                                                                        */
  mf_fit              fit_wavelenght;         /* Wavelength solution refinement/adjustment                                            */
  cpl_boolean         suppress_extension;     /* Flag to activate/suppress the output grating prefix                                  */
  //const char          *scale_pwv;     		  /* Value to use for scale the precipitable water vapor                                  */
} kmos_molecfit_parameter;


/*----------------------------------------------------------------------------*/
/**
 *                              Functions prototypes
 */
/*----------------------------------------------------------------------------*/

/* Insert parameter in the input parameterlist */
cpl_error_code kmos_molecfit_fill_parameter(
    const char* recipe, cpl_parameterlist *self, const char* param,
    cpl_boolean range, const void *rMin, const void *rMax,
    cpl_type type, const void *value, const char* description,
    cpl_boolean mf_conf);

/* Set the grating type that check with grating string */
cpl_error_code kmos_molecfit_grating_type(
    const char* name, kmos_grating_type *type);

/* Fill type grating with parameters by default */
cpl_error_code kmos_molecfit_grating_by_default(
    kmos_grating *grating, kmos_grating_type type);


/* Split a string into pieces at a given delimiter */
char ** kmos_molecfit_str_split(
    const char               *str,            /* Input string                               */
    const char               *delimiter,      /* Delimiter to split the string              */
    cpl_size                 *size);          /* Out: Number of tokens                      */

/* Deallocate a string array */
void kmos_molecfit_str_array_delete(
    char                     **strarr);       /* Input string array                         */

/* Generate the wave_range and molecules cpl_table's */
cpl_error_code kmos_molecfit_generate_wave_molec_tables(
    kmos_molecfit_parameter *conf);

/* Fill the molecfit generic configuration file */
cpl_error_code kmos_molecfit_conf_generic(
    kmos_molecfit_parameter       *conf,
    mf_parameters_config          *config_parameters);

/* Get frame with input data */
cpl_frame * kmos_molecfit_get_frame_spectrums(
    cpl_frameset *frameset);

/* Load input spectrum in the RAW file for a kmos_std_star execution */
cpl_error_code kmos_molecfit_load_spectrums(
    cpl_frameset *frameset, kmos_molecfit_parameter *conf, const char *recipe_name);

/* Extract a 1D spectrum from a extension datacube data in one FITS file */
cpl_vector * kmos_molecfit_extract_spec_from_datacube(
    const char *filename, int ext, cpl_propertylist *header);

/* Load and setup the convolution kernels provided by the user for one grating */
cpl_error_code kmos_molecfit_load_kernels(
    const cpl_frame *frm, kmos_molecfit_parameter *conf);

/* Load a convolution kernel for one specific extensions in the kernel library */
cpl_error_code kmos_molecfit_load_kernel(
    const char *filename, kmos_kernel *kernel);

/* Save to disk a new *.fits output file */
cpl_error_code kmos_molecfit_save(
    cpl_frameset            *all_frameset,
    cpl_frameset            *used_frameset,
    const cpl_parameterlist *parlist,
    const char              *recipe,
    cpl_propertylist        *list,
    const char              *tag,
    const char              *gratingname,
    cpl_boolean             suppress_extension,
    const char              *filename);

/* Save to disk a cpl_table output for one IFU molecfit execution */
cpl_error_code kmos_molecfit_save_mf_results(
    cpl_propertylist *header_data,
    cpl_propertylist *header_noise,
    const char       *tag,
    const char       *gratingname,
    cpl_boolean      suppress_extension,
    const char       *filename,
    cpl_matrix       *matrix,
    cpl_table        *table,
    cpl_vector       *vec);

/* Clean configuration generic parameter */
void kmos_molecfit_clean(
    kmos_molecfit_parameter *conf);

/* Clean spectrum parameter */
void kmos_molecfit_clean_spectrum(
    kmos_spectrum *ifu);

/* Clean kernel parameter */
void kmos_molecfit_clean_kernel(
    kmos_kernel *kernel);

/* Clean grating parameter */
void kmos_molecfit_clean_graing(
    kmos_grating *grating);

/* Nullify configuration generic parameter */
void kmos_molecfit_nullify(
    kmos_molecfit_parameter *conf);


#endif /* KMOS_MOLECFIT_H */
