# ========================================================================================================
# Jose A. Escartin
# 2019.04.11
#
# A python interactive window for use in the KMOS Reflex workflow.
# This interactive window shows the results of the kmos_extract_spec recipe.   
# From this interactive Actor the the user can modify pipeline parameters and re-initiate the processing.
#
#
# Pipeline Parameters for inclusion:
#
#    --mask_method           : Method used : mask, integrated or optimal. [integrated]
#    --centre                : The centre of the circular mask (pixel). [7.5,7.5]
#    --radius                : The radius of the circular mask (pixel). [3.0]
#    --save_mask             : Flag to save the mask. [FALSE]
#    --extra_outputs         : Flag to save extra outputs. [FALSE]
#    --cmethod               : Apply 'average', 'median', 'sum', 'min_max' or 'ksigma'. [ksigma]
#    --cpos_rej              : The positive rejection threshold for kappa-sigma-clipping (sigma). [3.0]
#    --cneg_rej              : The negative rejection threshold for kappa-sigma-clipping (sigma). [3.0]
#    --citer                 : The number of iterations for kappa-sigma-clipping. [3]
#    --cmax                  : The number of maximum pixel values to clip with min/max-clipping. [1]
#    --cmin                  : The number of minimum pixel values to clip with min/max-clipping. [1]
#
#
# Images to plot (TAG's): 
#
#    * EXTRACT_SPEC                                         The output data
#
# ========================================================================================================
    
from __future__ import with_statement
from __future__ import absolute_import
from __future__ import print_function
import sys
try:
   from wx._aui import AuiPaneInfo_IsLeftDockable
except:
   dummy=0

try:
    import numpy
    import os
    import re
    import reflex
    from pipeline_product import PipelineProduct
    import pipeline_display
    import reflex_plot_widgets
    import matplotlib.gridspec as gridspec
    from matplotlib.text import Text
    from pylab import *

    import_success = True
except ImportError:
    import_success = False
    print("Error importing modules pyfits, wx, matplotlib, numpy")

def paragraph(text, width=None):
    """ wrap text string into paragraph
       text:  text to format, removes leading space and newlines
       width: if not None, wraps text, not recommended for tooltips as
              they are wrapped by wxWidgets by default
    """
    import textwrap
    if width is None:
        return textwrap.dedent(text).replace('\n', ' ').strip()
    else:
        return textwrap.fill(textwrap.dedent(text), width=width)

class DataPlotterManager(object):
    # static members
    recipe_name = "kmos_extract_spec"
    extract_spec_cat = "EXTRACT_SPEC"
    sci_reconstructed_cat = "SCI_RECONSTRUCTED"
    
    def setWindowTitle(self):
        return self.recipe_name+"_interactive"

    def setInteractiveParameters(self):
        return [
            
            reflex.RecipeParameter(recipe=self.recipe_name, displayName="mask_method",
                    group="Inputs", description="Method used : mask, integrated or optimal"),
            reflex.RecipeParameter(recipe=self.recipe_name, displayName="centre",
                    group="Inputs", description="The centre of the circular mask (pixel)"),
            reflex.RecipeParameter(recipe=self.recipe_name, displayName="radius",
                    group="Inputs", description="The radius of the circular mask (pixel)"),
            reflex.RecipeParameter(recipe=self.recipe_name, displayName="save_mask",
                    group="Inputs", description="Flag to save the mask"),
            reflex.RecipeParameter(recipe=self.recipe_name, displayName="extra_outputs",
                    group="Inputs", description="Flag to save extra outputs"),
            reflex.RecipeParameter(recipe=self.recipe_name, displayName="cmethod",
                    group="Inputs", description="Apply 'average', 'median', 'sum', 'min_max' or 'ksigma'."),
            reflex.RecipeParameter(recipe=self.recipe_name, displayName="cpos_rej",
                    group="Inputs", description="The positive rejection threshold for kappa-sigma-clipping (sigma)"),
            reflex.RecipeParameter(recipe=self.recipe_name, displayName="cneg_rej",
                    group="Inputs", description="The negative rejection threshold for kappa-sigma-clipping (sigma)"),
            reflex.RecipeParameter(recipe=self.recipe_name, displayName="citer",
                    group="Inputs", description="The number of iterations for kappa-sigma-clipping"),
            reflex.RecipeParameter(recipe=self.recipe_name, displayName="cmax",
                    group="Inputs", description="The number of maximum pixel values to clip with min/max-clipping"),
            reflex.RecipeParameter(recipe=self.recipe_name, displayName="cmin",
                    group="Inputs", description="The number of minimum pixel values to clip with min/max-clipping"),

        ]

    def readFitsData(self, fitsFiles):
        
        self.frames = dict()
        for f in fitsFiles:
            self.frames[f.category] = PipelineProduct(f)
                  
        if self.extract_spec_cat in self.frames:
            
            # Get the wished files
            extract_spec = self.frames[self.extract_spec_cat]
            
            self.arcfile = extract_spec.all_hdu[0].header['ARCFILE']
           
            # Initialise
            self.extract_spec_extnames = []
            self.crpix1 = dict()
            self.crval1 = dict()
            self.cdelt1 = dict()
            self.cunit1 = dict()
            self.extract_spec_data = dict()
                        
            # Loop on all extensions of std_image to find the std stars extensions
            for extract_spec_ext in extract_spec.all_hdu:
                
                naxis = extract_spec_ext.header['NAXIS']
                
                # NAXIS is 1 if there is an spectrum, 0 otherwise
                if (naxis == 1):
                    
                    # extname is like IFU.3.DATA 
                    extname_keyword = extract_spec_ext.header['EXTNAME']
                    
                    # Get the IFU number from extname to get the IFU status
                    m = re.search(r"\d+", extname_keyword)
                    ifu_number = m.group()
                    
                    #Add name
                    extname = extname_keyword + " Target name: " + extract_spec_ext.header['ESO OCS ARM' + ifu_number +' NAME']
                    self.extract_spec_extnames.append(extname)

                    # Get infos from star_spec using the extname
                    self.crpix1[extname] = extract_spec_ext.header['CRPIX1']
                    self.crval1[extname] = extract_spec_ext.header['CRVAL1']
                    self.cdelt1[extname] = extract_spec_ext.header['CDELT1']
                    self.cunit1[extname] = extract_spec_ext.header['ESO QC CUBE_UNIT']
                    self.extract_spec_data[extname] = extract_spec_ext.data

            # Set the plotting functions
            self._add_subplots = self._add_subplots
            self._plot = self._data_plot
 
    def addSubplots(self, figure):
        self._add_subplots(figure)

    def plotProductsGraphics(self):
        self._plot()

    def plotWidgets(self) :
        widgets = list()
        # Radio button
        self.radiobutton = reflex_plot_widgets.InteractiveRadioButtons(self.axradiobutton, self.setRadioCallback, 
                self.extract_spec_extnames, 0, title = self.arcfile)
        widgets.append(self.radiobutton)
        return widgets

    def setRadioCallback(self, label) :

        # Define wave
        pix = numpy.arange(len(self.extract_spec_data[label]))
        wave = self.crval1[label] + pix * self.cdelt1[label]  
          
        # Plot Spectrum
        specdisp = pipeline_display.SpectrumDisplay()
        self.spec_plot.clear()
        specdisp.setLabels(r"$\lambda$[" + "$\mu$m]", self._process_label(self.cunit1[label]))
        specdisp.display(self.spec_plot, "Extracted spectrum", self._data_plot_get_tooltip(label), 
                wave, self.extract_spec_data[label])

    def _add_subplots(self, figure):
        gs = gridspec.GridSpec(4, 4)
        self.axradiobutton =    figure.add_subplot(gs[0:2, 0:3])
        self.spec_plot =        figure.add_subplot(gs[2:4, 0:4])

    def _data_plot_get_tooltip(self, extname):
        # Create the tooltip
        tooltip = " \
           "
        return tooltip

    def _data_plot(self):
        
        extname = self.extract_spec_extnames[0]
        
        # Define wave
        pix = numpy.arange(len(self.extract_spec_data[extname]))
        wave = self.crval1[extname] + pix * self.cdelt1[extname]  
        
        # Plot Spectrum
        specdisp = pipeline_display.SpectrumDisplay()
        specdisp.setLabels(r"$\lambda$[" + "$\mu$m]", self._process_label(self.cunit1[extname]))
        specdisp.display(self.spec_plot, "Extracted spectrum", self._data_plot_get_tooltip(extname), wave, self.extract_spec_data[extname])

    def _process_label(self, in_label):
        # If known, 'pretty print' the label
        if (in_label == "erg.s**(-1).cm**(-2).angstrom**(-1)"):
            return "Flux [erg sec" + r"$^{-1}$"+"cm" + r"$^{-2}$" + r"$\AA^{-1}$]"
        else:
            return in_label

#This is the 'main' function
if __name__ == '__main__':
    from reflex_interactive_app import PipelineInteractiveApp

    # Create interactive application
    interactive_app = PipelineInteractiveApp(enable_init_sop=True)

    # get inputs from the command line
    interactive_app.parse_args()

    #Check if import failed or not
    if not import_success:
        interactive_app.setEnableGUI(False)

    #Open the interactive window if enabled
    if interactive_app.isGUIEnabled():
        #Get the specific functions for this window
        dataPlotManager = DataPlotterManager()

        interactive_app.setPlotManager(dataPlotManager)
        interactive_app.showGUI()
    else:
        interactive_app.set_continue_mode()

    #Print outputs. This is parsed by the Reflex python actor to
    #get the results. Do not remove
    interactive_app.print_outputs()
    sys.exit()
