import numpy as np
import os
from adari_core.plots.text import TextPlot
from adari_core.data_libs.master_astrometry import MasterAstrometryReport
from adari_core.plots.panel import Panel
from astropy.io import fits

from . import KmosReportMixin


class KmosAstrometryReport(KmosReportMixin, MasterAstrometryReport):
    name = "kmos_astrometry"
    data_extensions = None
    image_category = "FIT_PROFILE"
    panel_kwargs = dict(x=6, y=4, height_ratios=[1, 4, 4, 4, 4])

    def __init__(self):
        super().__init__(self.name)

    def parse_sof(self):
        """Returns a list of files selected from a set of frames (sof).

        If more than one file fullfills the criteria, the first file
        in the array will be selected.
        """
        files_needed = {self.image_category: None}
        for filename, catg in self.inputs:
            if (
                catg == self.image_category
                and files_needed[self.image_category] is None
            ):
                files_needed[self.image_category] = filename

        if files_needed[self.image_category] is None:
            raise IOError("[ERROR] {} file not found".format(self.image_category))

        return [files_needed]

    def generate_panels(self, **kwargs):
        self.data_extensions = [f"IFU.{i}.DATA" for i in range(1, 25)]
        panels = super().generate_panels()
        for i, (panel, panel_descr) in enumerate(panels.items()):
            hdul = self.hdus[i]["FIT_PROFILE"]
            setup = (
                hdul["PRIMARY"].header.get("HIERARCH ESO INS GRAT1 NAME")
                + hdul["PRIMARY"].header.get("HIERARCH ESO INS GRAT2 NAME")
                + hdul["PRIMARY"].header.get("HIERARCH ESO INS GRAT3 NAME")
            )
            panel_descr["report_name"] = "{}_{}_{}".format(
                self.name,
                hdul["PRIMARY"].header.get("HIERARCH ESO OCS ROT NAANGLE"),
                setup.lower(),
            )
            panel_descr["report_description"] = (
                f"{self.name} Panel - " f"{os.path.basename(hdul.filename())}, "
            )
            exts = panel_descr["ext"]

            text_col = (
                "INS.GRAT1.NAME: {}".format(
                    str(hdul["PRIMARY"].header.get("HIERARCH ESO INS GRAT1 NAME"))
                ),
                "INS.GRAT1.WLEN: {}".format(
                    str(hdul["PRIMARY"].header.get("HIERARCH ESO INS GRAT1 WLEN"))
                ),
                "DET.SEQ1.DIT: {}".format(
                    str(hdul["PRIMARY"].header.get("HIERARCH ESO DET SEQ1 DIT"))
                ),
                "OSC.ROT.NAANGLE: {}".format(
                    str(hdul["PRIMARY"].header.get("HIERARCH ESO OCS ROT NAANGLE"))
                ),
            )
            text = TextPlot(columns=1, v_space=0.3)
            text.add_data(text_col)
            panel.assign_plot(text, 2, 0, xext=3)
            # p.assign_plot(text, 2, 0, xext=3)

        return panels


rep = KmosAstrometryReport()
