from edps import data_source, match_rules
from edps.generator.time_range import *

from .kmos_classification import *

# Convention for Data sources Association rule levels:
# Each data source can have several match function which correspond to different
# quality levels for the selected data. The level is specified as a number that
# follows this convention:
#   level < 0: more restrictive than the calibration plan
#   level = 0 follows the calibration plan
#   level = 1 quality sufficient for QC1 certification
#   level = 2 probably still acceptable quality
#   level = 3 significant risk of bad quality results

# standard matching keywords:
setup = [kwd.grat1_name, kwd.grat2_name, kwd.grat3_name]

#################################################################################################
# Raw files data sources
#################################################################################################
# ------ Raw darks -------------------------------------------------------------------------------

raw_dark = (data_source()
            .with_classification_rule(dark_class)
            .with_grouping_keywords([kwd.tpl_start])
            .with_setup_keywords([kwd.dit1, kwd.dit2, kwd.dit3])
            .with_min_group_size(5)
            .with_match_function(rules.is_assoc_dark, time_range=RelativeTimeRange(-0.3, 1), level=0)
            .with_match_function(rules.is_assoc_dark, time_range=TWO_WEEKS, level=1)
            .with_match_function(rules.is_assoc_dark, time_range=UNLIMITED, level=3)
            .build())
# -----------------------------------------------------------------------------------------------


# ------ Raw files for wavelength calibration----------------------------------------------------
raw_arc_on = (data_source()
              .with_classification_rule(arc_on_class)
              .with_grouping_keywords([kwd.tpl_start])
              .with_min_group_size(6)
              .with_setup_keywords(setup)
              .with_match_keywords(setup, time_range=ONE_AND_HALF_HOURS, level=-1)
              .with_match_keywords(setup, time_range=NEXT_DAY, level=0)
              .with_match_keywords(setup, time_range=TWO_DAYS, level=1)
              .with_match_keywords(setup, time_range=TWO_WEEKS, level=2)
              .with_match_keywords(setup, time_range=UNLIMITED, level=3)
              .build())

raw_arc_off = (data_source()
               .with_classification_rule(arc_off_class)
               .with_grouping_keywords([kwd.tpl_start])
               .with_setup_keywords(setup)
               .with_match_keywords(setup, time_range=ONE_AND_HALF_HOURS, level=-1)
               .with_match_keywords(setup, time_range=NEXT_DAY, level=0)
               .with_match_keywords(setup, time_range=TWO_DAYS, level=1)
               .with_match_keywords(setup, time_range=TWO_WEEKS, level=2)
               .with_match_keywords(setup, time_range=UNLIMITED, level=3)
               .build())
# -----------------------------------------------------------------------------------------------


# ---- Raw flat field files   --------------------------------------------------------------------
raw_flat_on = (data_source()
               .with_classification_rule(flat_on_class)
               .with_grouping_keywords([kwd.tpl_start])
               .with_min_group_size(18)
               .with_setup_keywords(setup)
               .with_match_keywords(setup, time_range=ONE_AND_HALF_HOURS, level=-1)
               .with_match_keywords(setup, time_range=RelativeTimeRange(-0.08, 1), level=0)
               .with_match_keywords(setup, time_range=TWO_DAYS, level=1)
               .with_match_keywords(setup, time_range=TWO_WEEKS, level=2)
               .with_match_keywords(setup, time_range=UNLIMITED, level=3)
               .build())

# Selection of flat fields for arcs requires its own rule
PAST_4_HRS = RelativeTimeRange(-0.16, 0)
NEXT_4_HRS = RelativeTimeRange(0, 0.16)
flat_for_arcs = (match_rules()
                 .with_match_keywords(setup, time_range=PAST_4_HRS, level=0)
                 # In the first months of operations, flats were taken AFTER the arcs; therefore I include this rule
                 # with association level = 0
                 .with_match_keywords(setup, time_range=NEXT_4_HRS, level=0)
                 .with_match_keywords(setup, time_range=TWO_DAYS, level=1)
                 .with_match_keywords(setup, time_range=UNLIMITED, level=3))

raw_flat_off = (data_source()
                .with_classification_rule(flat_off_class)
                .with_grouping_keywords([kwd.tpl_start])
                .with_match_keywords(setup + [kwd.tpl_start], level=0)
                .with_match_keywords(setup, time_range=UNLIMITED, level=3)
                .build())

raw_flat_sky = (data_source()
                .with_classification_rule(flat_sky_class)
                .with_grouping_keywords([kwd.tpl_start])
                .with_min_group_size(4)
                .with_setup_keywords(setup)
                .with_match_keywords(setup, time_range=ONE_MONTH, level=0)
                .with_match_keywords(setup, time_range=RelativeTimeRange(-40, 40), level=1)
                .with_match_keywords(setup, time_range=QUARTERLY, level=2)
                .with_match_keywords(setup, time_range=UNLIMITED, level=3)
                .build())
# -------------------------------------------------------------------------------------------------------

# -----  Raw telluric standard star  --------------------------------------------------------------------
TWO_HRS = RelativeTimeRange(-0.083, 0.083)
raw_std = (data_source()
           .with_classification_rule(std_class)
           .with_grouping_keywords([kwd.tpl_start])
           .with_setup_keywords(setup)
           .with_match_function(rules.is_assoc_std_cal, time_range=TWO_HRS, level=-1)
           .with_match_keywords(setup, time_range=SAME_NIGHT, level=0)  # level definition changed from Oct 2023
           .with_match_keywords(setup, time_range=UNLIMITED, level=3)
           .build())
#
# -----------------------------------------------------------------------------------------------


# ----- Raw science exposures  ------------------------------------------------------------------
raw_science = (data_source()
               .with_classification_rule(science_class)
               .with_setup_keywords(setup)
               .with_match_keywords([kwd.grat1_id])
               .with_grouping_keywords([kwd.tpl_start])
               .build())
# ----------------------------------------------------------------------------------------------------------------------

########################################################################################################################
# data sources for static calibrations
########################################################################################################################

# Line Spread Function
static_kernel = (data_source()
                 .with_classification_rule(KERNEL_LIBRARY)
                 .with_grouping_keywords([kwd.date])
                 .with_match_keywords(setup, time_range=IN_THE_PAST, level=0)
                 .build())

# GDAS catalog for atmospheric model
static_gdas = (data_source()
               .with_classification_rule(gdas_class)
               .with_match_function(rules.is_assoc, time_range=UNLIMITED, level=0)
               .build())

# ---- Files with Arc lines and bands for wavelength calibrationn ------------------------------------------------------
static_ref_lines = (data_source()
                    .with_classification_rule(ref_lines_class)
                    .with_grouping_keywords([kwd.date])
                    .with_match_function(rules.is_assoc, time_range=UNLIMITED, level=0)
                    .build())
static_wave_band = (data_source()
                    .with_classification_rule(wave_band_class)
                    .with_match_function(rules.is_assoc, time_range=UNLIMITED, level=0)
                    .build())

static_arc_list = (data_source()
                   .with_classification_rule(arc_list_class)
                   .with_setup_keywords(setup)
                   .with_match_function(rules.is_assoc_filt, time_range=UNLIMITED, level=0)
                   .build())
# ----------------------------------------------------------------------------------------------------------------------

# Template of solar spectrum
static_solar_spec = (data_source()
                     .with_classification_rule(solar_spec_class)
                     .with_setup_keywords(setup)
                     .with_match_function(rules.is_assoc_filt, time_range=UNLIMITED, level=0)
                     .build())

# Average atmospheric transmission
static_atmos_model = (data_source()
                      .with_classification_rule(atmos_model_class)
                      .with_setup_keywords(setup)
                      .with_match_function(rules.is_assoc_filt, time_range=UNLIMITED, level=0)
                      .build())

# Spectral look-up table for generating the model of the telluric standard
static_spec_type_lookup = (data_source()
                           .with_classification_rule(spec_type_lookup_class)
                           .with_match_function(rules.is_assoc, time_range=UNLIMITED, level=0)
                           .build())

# Average response function
static_response = (data_source()
                   .with_classification_rule(response_class)
                   .with_match_keywords(setup, time_range=IN_THE_PAST, level=0)
                   .with_match_keywords(setup, time_range=UNLIMITED, level=3)
                   .build())

# Table OH lines
static_oh_spec = (data_source()
                  .with_classification_rule(oh_spec_class)
                  .with_match_function(rules.is_assoc_filt, time_range=UNLIMITED, level=0)
                  .build())

########################################################################################################################
# other data sources
########################################################################################################################

raw_acquisition = (data_source()
                   .with_classification_rule(acquisition_class)
                   .with_grouping_keywords([kwd.tpl_start])
                   .with_setup_keywords(setup)
                   .build())

raw_astrometry = (data_source()
                  .with_classification_rule(astrometry_class)
                  .with_grouping_keywords([kwd.date])
                  .with_setup_keywords(setup)
                  .build())

reduced = (data_source()
           .with_classification_rule(reduced_class)
           .with_grouping_keywords([kwd.targ_name, kwd.grat1_name, kwd.grat2_name, kwd.grat3_name])
           .build())
