from edps import subworkflow, task, List
from edps.executor.recipe import File, FitsFile, ProductRenamer
from edps.executor.recipe import RecipeInvocationArguments, RecipeInvocationResult, InvokerProvider, RecipeInputs

from .kmos_datasources import *


# -- Auxiliary function ------------------------------------------------------------------------------------------------
# Generic function to run a recipe
def run_recipe(input_file, associated_files, parameters, recipe_name, args, invoker, renamer) -> (RecipeInvocationResult, List):
    # Inputs:
    # input_file:       main input and category. Format: File(string_with_full_path, string_with_category, "")
    # associated_files: calibrations. Format List[file], where files have the format: File(string_with_full_path,
    #                   string_with_category, "")
    # parameters: non default recipe parameters. Format {'parameter_name1': value1, 'parameter_name2': value2}
    # recipe_name: recipe name  Format: string
    # args, invoker: extra stuff provided by the task that calls the function calling run_recipe()

    inputs = RecipeInputs(main_upstream_inputs=[input_file], associated_upstream_inputs=associated_files)
    arguments = RecipeInvocationArguments(inputs=inputs, parameters=parameters,
                                          job_dir=args.job_dir, input_map={},
                                          logging_prefix=args.logging_prefix)

    results = invoker.invoke(recipe_name, arguments, renamer, create_subdir=True)
    output_files = [File(f.name, f.category, "") for f in results.output_files]
    return results, output_files


# Generic function to loop on a recipe
def loop_recipe(recipe, args, invoker_provider, renamer, categories="all") -> RecipeInvocationResult:
    # Inputs:
    # recipe [string] name of the recipe to loop
    # args   [RecipeInvocationArguments] arguments of the recipe. Argument of the function associated to the task
    # invoker_provider [InvokerProvider].                         Argument of the function associated to the task
    # categories: list of categories to loop on. If categories="all" the recipe is looped onto all the inputs.

    results = []
    ret_codes = []
    invoker = invoker_provider.recipe_invoker

    if categories == "all":
        # all files are processed by the recipe in the loop; no calibrations are associated.
        loop_files = [File(f.name, f.category, "") for f in args.inputs.combined]
        calibrations = None
    else:
        # only files of certain categories are processed by the recipe. Other files are treated as calibrations.
        loop_files = [File(f.name, f.category, "") for f in args.inputs.combined if f.category in categories]
        calibrations = [File(f.name, f.category, "") for f in args.inputs.combined if f.category not in categories]

    for input_file in loop_files:
        result_recipe, junk = run_recipe(input_file, calibrations, {}, recipe, args, invoker, renamer)
        ret_codes.extend([result_recipe.return_code])
        results.append(result_recipe)

    ret_code = min(ret_codes)
    output_files = [FitsFile(name=f[0].name, category=f[0].category) for f in ([r.output_files for r in results])]
    final_results = RecipeInvocationResult(return_code=ret_code, output_files=output_files)
    return final_results


# ----------------------------------------------------------------------------------------------------------------------

# This sub-workflow computes the atmospheric transmission by running molecfit on telluric standard star.
@subworkflow("telluric_on_standard", "")
def telluric_on_standard(reference, target):
    # Create an atmospheric model using the telluric standard star as reference spectrum
    model = (task('model_on_standard')
             .with_recipe('kmos_molecfit_model')
             .with_main_input(reference)
             .with_associated_input(static_kernel, min_ret=0, max_ret=1)
             .with_associated_input(static_gdas, min_ret=0, max_ret=1)
             .with_input_filter(STAR_SPEC, KERNEL_LIBRARY, gdas_class)
             .build())

    # Computing the transmission curve using the information from the science target
    # and the atmospheric model computed om the telluric standard stare.
    transmission = (task('transmission_on_standard')
                    .with_recipe('kmos_molecfit_calctrans')
                    .with_main_input(target)
                    .with_associated_input(model)
                    .with_associated_input(static_kernel, min_ret=0, max_ret=1)
                    .with_input_filter(SCI_RECONSTRUCTED, BEST_FIT_PARM, ATMOS_PARM, KERNEL_LIBRARY)
                    .build())

    return transmission


# This sub-workflow computes the atmospheric transmission by running molecfit directly on science data.
@subworkflow("telluric_on_science", "")
def telluric_on_science(target):
    # This runs the extraction recipe kmos_extrac_spec on each individual file separately and collects all the results.
    # This "loop" can be removed once the recipe implements its own internal loop.
    def extraction(args: RecipeInvocationArguments, invoker_provider: InvokerProvider,
                   renamer: ProductRenamer) -> RecipeInvocationResult:
        extracted_spectra = loop_recipe("kmos_extract_spec", args, invoker_provider, renamer, categories='all')
        return extracted_spectra

    # Extract the spectrum of the science target, integrating over the full spatial extension of the IFU
    # The extracted spectrum will be used as reference spectrum.
    extract_spectra = (task('extract_spectra')
                       .with_function(extraction)
                       .with_main_input(target)
                       .with_input_filter(SCI_RECONSTRUCTED)
                       .build())

    # ---- Place holder for a Python interactive window that allows to select the spectrum to be used as reference. ----
    #      By default, the  extracted spectra of the last science exposure are used as reference.
    # Function to copy only the last input file.
    def use_last(parameters: RecipeInvocationArguments, invoker_provider: InvokerProvider,
                 renamer: ProductRenamer) -> RecipeInvocationResult:
        x = parameters.inputs.combined
        return RecipeInvocationResult(return_code=0, output_files=[FitsFile(name=x[-1].name, category=x[-1].category)])

    reference = (task('select_reference')
                 .with_function(use_last)
                 .with_main_input(extract_spectra)
                 .build())
    # ------------------------------------------------------------------------------------------------------------------

    # Create an atmospheric model using the selected science spectrum as reference.
    model = (task('model_on_science')
             .with_recipe('kmos_molecfit_model')
             .with_main_input(reference)
             .with_associated_input(static_kernel, min_ret=0, max_ret=1)
             .with_associated_input(static_gdas, min_ret=0, max_ret=1)
             .build())

    # Computing the transmission curve using the information from the science target
    # and the atmospheric model computed om the telluric standard star.
    transmission = (task('transmission_on_science')
                    .with_recipe('kmos_molecfit_calctrans')
                    .with_main_input(target)
                    .with_associated_input(model)
                    .with_associated_input(static_kernel, min_ret=0, max_ret=1)
                    .with_input_filter(SCI_RECONSTRUCTED, BEST_FIT_PARM, ATMOS_PARM, KERNEL_LIBRARY)
                    .build())

    return transmission
