from edps import FilterMode
from edps import QC1_CALIB, QC0, SCIENCE, CALCHECKER, ReportInput, task

from .kmos_datasources import *
from .kmos_fit_profiles import kmos_fit_profiles
from .kmos_illumination import select_illumination
from .kmos_molecfit import telluric_on_standard, telluric_on_science
from .kmos_task_functions import *

__title__ = "KMOS workflow"

IDP = "idp"

# --- PROCESSING TASKS -------------------------------------------------------------------

# ------ Process dark frames
dark = (task('dark')
        .with_recipe('kmos_dark')
        .with_report('kmos_rawdisp', ReportInput.RECIPE_INPUTS, driver='png')
        .with_report('kmos_master_dark', ReportInput.RECIPE_INPUTS_OUTPUTS, driver='png')
        .with_main_input(raw_dark)
        .with_meta_targets([QC1_CALIB])
        .build())

# ------ Process lamp flats frames
lamp_flat = (task('lamp_flat')
             .with_recipe('kmos_flat')
             .with_report('kmos_rawdisp', ReportInput.RECIPE_INPUTS, driver='png')
             .with_report('kmos_ifu_flat', ReportInput.RECIPE_INPUTS_OUTPUTS, driver='png')
             .with_main_input(raw_flat_on)
             .with_associated_input(raw_flat_off, min_ret=1, max_ret=3)
             .with_associated_input(dark, [BADPIXEL_DARK])
             .with_input_filter(BADPIXEL_DARK, flat_off_class, flat_on_class)
             .with_meta_targets([QC1_CALIB, CALCHECKER])
             .build())

# ------ Process arcs
wavelength = (task('arc')
              .with_recipe('kmos_wave_cal')
              .with_report('kmos_rawdisp', ReportInput.RECIPE_INPUTS, driver='png')
              .with_report('kmos_wave_cal', ReportInput.RECIPE_INPUTS_OUTPUTS, driver='png')
              .with_main_input(raw_arc_on)
              .with_associated_input(raw_arc_off)
              .with_associated_input(lamp_flat, [XCAL, YCAL, FLAT_EDGE], match_rules=flat_for_arcs)
              .with_associated_input(static_ref_lines)
              .with_associated_input(static_wave_band)
              .with_associated_input(static_arc_list)
              .with_input_filter(XCAL, YCAL, arc_off_class, FLAT_EDGE, ref_lines_class, wave_band_class, arc_list_class)
              .with_meta_targets([QC1_CALIB, CALCHECKER])
              .build())

# Subworkflow that returns the illumination correction according to the
# desired reduction strategy, determined by the workflow parameter use_sky_flats.
# use_sky_flats = "false" : illumination correction is obtained from lamp_flats.
# use_sky_flats = "true"  : illumination correction is obtained from sky_flats.
illumination_correction, sky_flat, illumination = select_illumination(dark, lamp_flat, wavelength)

# --- Process standard stars for telluric and response curve correction
standard = (task('standard')
            .with_recipe('kmos_std_star')
            .with_report('kmos_rawdisp', ReportInput.RECIPE_INPUTS, driver='png')
            .with_report('kmos_std_star', ReportInput.RECIPE_INPUTS_OUTPUTS, driver='png')
            .with_main_input(raw_std)
            .with_associated_input(lamp_flat, [XCAL, YCAL, MASTER_FLAT])
            .with_associated_input(wavelength, [LCAL])
            .with_associated_input(static_wave_band)
            .with_associated_input(static_solar_spec, min_ret=0, max_ret=1)
            .with_associated_input(static_spec_type_lookup, min_ret=0, max_ret=1)
            .with_associated_input(static_atmos_model, min_ret=0, max_ret=1)
            .with_alternative_associated_inputs(illumination_correction)
            .with_input_filter(XCAL, YCAL, MASTER_FLAT, LCAL, ILLUM_CORR, wave_band_class, solar_spec_class,
                               spec_type_lookup_class, atmos_model_class)
            .with_meta_targets([QC1_CALIB, CALCHECKER])
            .build())

# ---- Tasks to select the response curve and telluric correction ------------------------------------------------------
# These tasks select the telluric correction and response curve to pass to the science
# tasks, depending on the selected data reduction strategy.
# "molecfit" = "false": the task telluric_and_response is activated. The products of the
#                      task standard, that contain telluric and response corrections,
#                      and flux-to-ADU conversion are directly passed to the science
#                      tasks.
#
# "molecfit" = "standard" or "molecfit" = "science": the task response is activated. A static
#             calibration containing an average response curve is passed to the science
#             tasks. The telluric correction is done by other tasks, depending on the
#             desired strategy (run molecfit on standard stars or directly on the
#             science spectra).

telluric_and_response = (task('telluric_and_response')
                         .with_recipe('kmos_gen_telluric')
                         .with_main_input(standard)
                         .with_condition(use_telluric_standard)  # the task is active if "molecfit"="false"
                         .with_input_filter(TELLURIC, TELLURIC_CORR, response_class)
                         .build())

response = (task('response')
            .with_recipe('kmos_gen_telluric')
            .with_main_input(standard)
            .with_associated_input(static_response)
            .with_condition(use_molecfit)  # the task is active if "molecfit"!="false"
            .with_input_filter(TELLURIC, TELLURIC_CORR, response_class)
            .build())
# ----------------------------------------------------------------------------------------------------------------------


# ---- Processing the science exposures --------------------------------------------------------------------------------

science = (task('object')
           .with_recipe('kmos_sci_red')
           .with_main_input(raw_science)
           .with_associated_input(lamp_flat, [XCAL, YCAL, MASTER_FLAT])
           .with_associated_input(wavelength, [LCAL])
           .with_alternatives(illumination_correction)
           .with_associated_input(dark, [BADPIXEL_DARK])
           .with_associated_input(static_wave_band)
           .with_associated_input(static_oh_spec, min_ret=0, max_ret=1)
           # Products of the task 'object' are telluric corrected and flux calibrated if "molecfit"="false" .
           # Otherwise they are only flux calibrated:
           .with_associated_input(response, [TELLURIC_GEN], min_ret=0, max_ret=1, condition=not_qc0)
           .with_associated_input(telluric_and_response, [TELLURIC_GEN], min_ret=0, max_ret=1, condition=not_qc0)
           .with_input_filter(TELLURIC_GEN, BADPIXEL_DARK, ILLUM_CORR, LCAL, XCAL, YCAL, MASTER_FLAT, wave_band_class,
                              oh_spec_class)
           .with_meta_targets([CALCHECKER, QC0])
           .build())

# ----- Tasks to perform the telluric correction if "molecfit" is not "false" ------------------------------------------

# If the parameter "molecfit"="standard", then the atmospheric transmission is computed on the spectrum
# of the standard star. This is done in the subworkflow telluric_on_standard.
transmission_from_standard = telluric_on_standard(standard, science)

# If the parameter "molecfit"="science", then the atmospheric transmission is directly computed on
# the science spectrum. This is done in the subworkflow telluric_on_science.
transmission_from_science = telluric_on_science(science)

# Correction of science spectra
# The atmospheric transmission curve to be used depends on whether molecfit was run on the telluric standard
# or directly on the science spectra
telluric_correction = (task('telluric_correction')
                       .with_recipe('kmos_molecfit_correct')
                       .with_main_input(science)
                       .with_associated_input(transmission_from_standard, condition=molecfit_on_standard)
                       .with_associated_input(transmission_from_science, condition=molecfit_on_science)
                       .with_grouping_keywords(
    [kwd.targ_name, kwd.grat1_name, kwd.grat2_name, kwd.grat3_name])  # Grouping exposures of the same target and setup
                       .with_input_filter(SCI_RECONSTRUCTED_COLL, CALCTRANS_ATMOS_PARM, mode=FilterMode.REJECT)
                       .with_meta_targets([SCIENCE, CALCHECKER])
                       .build())

# ----------------------------------------------------------------------------------------------------------------------

# Subworkflow to measure the radial profile of acquisition images and astrometric fields.
# This is needed for quality control but not for science reduction.
acquisition_profile, astrometry_profile = kmos_fit_profiles(lamp_flat, wavelength)

# ----------------  Tasks to combine the science exposures -------------------------------------------------------------
# The combination of cubes needs to be treated differently for the following cases, because they corresponding
# tasks have different main_inputs.
#   - Telluric correction done with molecfit. Response curve correction comes from static calibration.
#     Task: object_combination_molecfit.
#   - Telluric correction done by dividing the spectra by the spectrum of the telluric standard. Response
#     curve correction comes from the reduction of the telluric star. Task: object combination.
#   - Data that are pre-processed by previous workflow executions and present in the input data pool.
#     Task: cube combination.


science_combination_molecfit = (task('object_combination_molecfit')  #
                                .with_recipe('kmos_combine')
                                .with_main_input(telluric_correction)
                                .with_associated_input(static_oh_spec, min_ret=0, max_ret=1)
                                .with_condition(use_molecfit)  # condition for triggering the task
                                .with_input_filter(SINGLE_CUBES, oh_spec_class)
                                .with_meta_targets([SCIENCE, IDP, CALCHECKER])
                                .build())

science_combination = (task('object_combination')
                       .with_recipe('kmos_combine')
                       .with_main_input(science)
                       .with_associated_input(static_oh_spec, min_ret=0, max_ret=1)
                       .with_condition(use_telluric_standard)  # condition for triggering the task
                       .with_input_filter(SCI_RECONSTRUCTED, SINGLE_CUBES, oh_spec_class)
                       .with_meta_targets([SCIENCE])
                       .build())

cubes_combination = (task('cubes_combination')
                     .with_recipe('kmos_combine')
                     .with_main_input(reduced)
                     .with_associated_input(static_oh_spec, min_ret=0, max_ret=1)
                     .with_input_filter(SCI_RECONSTRUCTED, SINGLE_CUBES, oh_spec_class)
                     .with_meta_targets([SCIENCE])
                     .build())
# ----------------------------------------------------------------------------------------------------------------------
