/*
 * This file is part of the ESO Telluric Correction Library
 * Copyright (C) 2001-2018 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

/*----------------------------------------------------------------------------*/
/**
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

#include "mf_constants.h"

#include "mf_parameters.h"
#include "mf_model.h"
#include "mf_kernel_user.h"

#include <cpl_test.h>
#define MF_KEYWORD_CRVAL1              "CRVAL1"                                 /*  */
#define MF_KEYWORD_CD1_1              "CD1_1"                                 /*  */
#define MF_KEYWORD_CD2_2               "CD2_2"                                  /*  */

/*----------------------------------------------------------------------------*/
/**
 *                 Typedefs: Enumeration types
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Defines
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Global variables
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Macros
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Typedefs: Structured types
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                 Functions prototypes
 */
/*----------------------------------------------------------------------------*/
/* Convolution of a flux array with given kernel for a fixed range of pixels */
static cpl_error_code mf_kernel_convolve(
    cpl_array                *convflux,
    const cpl_array          *flux,
    const int                range[2],
    const cpl_array          *kernel);

/*----------------------------------------------------------------------------*/
/**
 *                 Program
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 * @defgroup test_mf_model   .
 *
 * @brief
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/* ---------------------------------------------------------------------------*/
/**
 * @brief .
 *
 * @param .                  .
 * @param                    .
 *
 * @return cpl_error_code    CPL_ERROR_NONE is everything is OK.
 *                           If not, these are the errors:
 *                           - .
 *                           - Error in subroutine (see subroutines).
 *
 * @description .
 *
 * @note .
 *
 */
/* ---------------------------------------------------------------------------*/
int main(void)
{
  cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_DEBUG);


  /* Create configuration */
  cpl_test_error(CPL_ERROR_NONE);
 
  cpl_msg_info(cpl_func, "\n");   
  cpl_msg_info(cpl_func, "======================");   
  cpl_msg_info(cpl_func, "Testing mf_kernel_user");   
  cpl_msg_info(cpl_func, "======================");   

 
  cpl_size          spec_n_lambdas;
  cpl_propertylist *header_spec    = cpl_propertylist_new();
  cpl_propertylist *header_kernel  = cpl_propertylist_new();
  cpl_matrix       *kernel;
  cpl_matrix       *matrix;
 
  cpl_error_code err;	
  
  /* Create a dummy header spec property list of keyword value pairs */
  cpl_propertylist_append_double(header_spec,MF_KEYWORD_CRVAL1, 7.244334002981E+03);
  cpl_propertylist_append_double(header_spec,MF_KEYWORD_CD1_1, 1.577355366657E+00);
  cpl_propertylist_append_double(header_spec,MF_KEYWORD_CD2_2, 1.577355366657E+00);
  
  /* Create a dummy header kernel property list of keyword value pairs */
  cpl_propertylist_append_double(header_kernel,MF_KEYWORD_CRVAL1, 7.244334002981E+03);
  cpl_propertylist_append_double(header_kernel,MF_KEYWORD_CD1_1, 1.577355366657E+00);
  cpl_propertylist_append_double(header_kernel,MF_KEYWORD_CD2_2, 1.577355366657E+00);

  err = cpl_error_get_code();
  if (err!=CPL_ERROR_NONE) {
      cpl_msg_info(cpl_func,"Error in setting up dummy header key values!");
      cpl_propertylist_delete(header_spec);
      cpl_propertylist_delete(header_kernel);
      return cpl_test_end(0);
  }


  /* Create a dummy kernel */
  spec_n_lambdas=10;
  int kernel_size=3;
  kernel  = cpl_matrix_new(spec_n_lambdas,kernel_size);
  for (int i=0; i<spec_n_lambdas; i++) {
       for (int j=0; j<kernel_size; j++) {
           cpl_matrix_set(kernel,i,j,1.0);
       }
   }

  /* -------------------------------------- */
  /* Test the mf_kernel_user_create routine */
  /* -------------------------------------- */
  matrix=mf_kernel_user_create(spec_n_lambdas, 
                         header_spec, 
                         header_kernel,
                         kernel);
  if (matrix==NULL) {
      cpl_msg_info(cpl_func, "NULL Matrix");
  } else {
      int m,n;
      m=cpl_matrix_get_nrow(matrix);
      n=cpl_matrix_get_ncol(matrix);
      cpl_msg_info(cpl_func, "Matrix of size %d x %d", m,n);
      double xd;
      int i,j;
      for (i=0; i<m; i++) {
          for (j=0; j<n; j++) {          
              xd=cpl_matrix_get(matrix,i,j);
              cpl_msg_info(cpl_func, "Matrix val at %d  %d = %f", i,j,xd);
	     
          }
      }
      
  }
  
  /* ------------------------------- */
  /* Test the mf_kernel_user routine */
  /* ------------------------------- */
  /* TO BE DONE */
  
  /* ---------------------------------------- */
  /* Test the mf_kernel_user_convolve routine */
  /* ---------------------------------------- */
  /* TO BE DONE */
    /* Convolve spectrum with kernel(s) (pixel by pixel) */
    int m=spec_n_lambdas;
    cpl_array *convflux     = cpl_array_new(m, CPL_TYPE_DOUBLE);
    cpl_array *flux         = cpl_array_new(m, CPL_TYPE_DOUBLE);
    cpl_array *kernel_array = cpl_array_new(m, CPL_TYPE_DOUBLE);
    
    cpl_array_fill_window_double(kernel_array, 0, m, 1.0/m);
    
/*
    for (cpl_size i = 0; i < m; i++) {
       cpl_array_set_double(kernel_array,i,0.1);
    }
*/    
    /*double *kern = cpl_array_get_data_double(kernel_array);*/
    /*const double *matkern = cpl_matrix_get_data_const(kernel);*/
    for (cpl_size i = 0; i < m; i++) {

        /* Set pixel range */
        int range[2] = {i, i};

        /* Copy kernel elements into array of correct length */
        /*
	for (cpl_size j = 0; j < m; j++) {
            cpl_size idx = j ;
            kern[j] = matkern[idx];
            kern[j] =0.1;
        }
        */
	
        /* Convolve pixel with kernel */
        mf_kernel_convolve(convflux, flux, range, kernel_array);
    }
                         
  /* Cleanup*/
  cpl_propertylist_delete(header_spec);
  cpl_propertylist_delete(header_kernel);
  cpl_matrix_delete(kernel);
  cpl_matrix_delete(matrix);
  cpl_array_delete(convflux);
  cpl_array_delete(flux);
  cpl_array_delete(kernel_array);
  
  /* Show errors and return */

  return cpl_test_end(0);
}

/* 
   -------------------------------------------------------
   I DID NOT REALISE UNTIL THE LAST MOMENT THE 
              mf_kernel_convolve
   FUNCTION IS STATIC. LATER IT WILL BE TESTED WHEN WE TEST
              mf_kernel_user
   BUT I WANTED TO TEST mf_kernel_convolve first SO FOR
   TEMPORARY PURPOSES A CUT AND PASTE OF mf_kernel_convolve
   IS PLACED HERE (mikeB)	      
   -------------------------------------------------------   
*/

static cpl_error_code mf_kernel_convolve(
    cpl_array                *convflux,
    const cpl_array          *flux,
    const int                range[2],
    const cpl_array          *kernel)
{
    /*!
     * Convolution of a flux array with given kernel for a fixed range of
     * pixels. The output array has to exist and have the same number of
     * pixels as the input array. Technically, the convolution is carried out
     * by adding the convolved fluxes for each pixel in the selected window.
     *
     * \note The center of the convolution function is shifted by -0.5 pixels
     *       for an even number of kernel pixels.
     *
     * \b INPUT:
     * \param convflux  CPL array of convolved flux values
     * \param flux      CPL array of unconvolved input flux
     * \param range     range of pixels to be considered (minimum and maximum)
     * \param kernel    kernel as CPL array
     *                  (required: sum of all values = 1)
     *
     * \b OUTPUT:
     * \param convflux  output array with added convolved flux from given
     *                  range
     *
     * \b ERRORS:
     * - Invalid object structure
     * - Invalid input parameter(s)
     * - Invalid object value(s)
     */

    /* No data points -> no convolution */
    cpl_size n = cpl_array_get_size(flux);
    if (n <= 0) return CPL_ERROR_NONE;

    /* Check correspondence of data points */
    if (cpl_array_get_size(convflux) != n) {
        return cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                                     "Invalid object structure: cpl_array *convflux != cpl_array *flux (size)");
    }

    /* Check range values */
    if (range[0] > range[1]) {
        return cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                                     "Invalid input parameter(s): range[2] (min. > max.)");
    }

    if (range[0] < 0 || range[1] >= n) {
        return cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                                     "Invalid input parameter(s): range[2] (invalid pixel)");
    }

    /* Initialize output array if not done so far */

    if (cpl_array_has_invalid(convflux) == 1) {
        cpl_array_fill_window_double(convflux, 0, n, 0.);
    }

    /* Get pointers to CPL arrays */
    const double *influx  = cpl_array_get_data_double_const(flux    );
    double       *outflux = cpl_array_get_data_double(      convflux);

    /* No kernel data -> no convolution */
    cpl_size nkpix = cpl_array_get_size(kernel);
    if (nkpix == 0) {
        for (cpl_size i = range[0]; i <= range[1]; i++) {
            outflux[i] += influx[i];
        }
        return CPL_ERROR_NONE;
    }

    /* Get pointer to CPL array */
    const double *kern = cpl_array_get_data_double_const(kernel);

    /* Check kernel */
    double sum = 0.;
    for (cpl_size k = 0; k < nkpix; k++) {
        if (kern[k] < 0 || kern[k] > 1) {
            for (cpl_size i = range[0]; i <= range[1]; i++) {
                outflux[i] += influx[i];
            }
            return cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                                         "Invalid object value(s): cpl_array *kernel (kernel element(s) < 0 or > 1)");
        }
        sum += kern[k];
    }

    if (sum < 1 - MF_TOL || sum > 1 + MF_TOL) {
        for (cpl_size i = range[0]; i <= range[1]; i++) {
            outflux[i] += influx[i];
        }
        return cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                                     "Invalid object value(s): cpl_array *kernel (sum of kernel elements != 1 (%f))",sum);
    }

    /* Skip convolution if number of kernel pixels is one */
    if (nkpix == 1) {
        for (cpl_size i = range[0]; i <= range[1]; i++) {
            outflux[i] += influx[i];
        }
        return CPL_ERROR_NONE;
    }

    /* Kernel with even or odd pixel number? Note: center of kernel at -0.5 pixels for even pixel number */
    cpl_size kmin = (nkpix % 2 == 0) ? - nkpix / 2 : - (nkpix - 1) / 2;
    cpl_size kmax = kmin + nkpix - 1;

    /* Set pixel range (add virtual pixels for marginal ranges) */
    cpl_size jmin = (range[0] == 0    ) ?       - kmax : range[0];
    cpl_size jmax = (range[1] == n - 1) ? n - 1 - kmin : range[1];

    /* Set flux of virtual input pixels */
    double in0   = influx[0];
    double innm1 = influx[n-1];

    /* Convolve array with kernel */
    for (cpl_size j = jmin; j <= jmax; j++) {

        /* Flux of real and virtual input pixels */
        double in;
        if (     j <  0) in = in0;
        else if (j >= n) in = innm1;
        else             in = influx[j];

        /* Calculate output flux for each kernel element and add it to the corresponding output pixel */
        for (cpl_size k = CPL_MAX(kmin, - j); k <= CPL_MIN(kmax, n - 1 - j); k++) {
            cpl_size i = j + k;
            outflux[i] += in * kern[k - kmin];
        }
    }

    return CPL_ERROR_NONE;
}




/** @cond PRIVATE */

/** @endcond */

/**@}*/
