/* $Id: mat_array.c,v0.5 2014-06-15 12:56:21 mheininger Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: mheininger $
 * $Date: 2014/05/21 11:07:00 $
 * $Revision: 0.5 $
 * $Name: mat_array.c $
 */

#include <stdlib.h>

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
  Includes
  -----------------------------------------------------------------------------*/

#include <string.h>
#include <math.h>
#include <float.h>
#include <assert.h>

#include "mat_const.h"
#include "mat_error.h"
#include "mat_array.h"
#include "mat_utils.h"

/*-----------------------------------------------------------------------------
  Define
  -----------------------------------------------------------------------------*/


/*-----------------------------------------------------------------------------
  Functions
  -----------------------------------------------------------------------------*/

/**
   @ingroup oiarr
   @brief Creates a mat_array data structure for a given number of stations.
   @param nbstation   The number of stations for which elements are allocated.
   @returns A new mat_array data structure or NULL.

   This function creates a new mat_arraw data structure. Memory for all requested
   stations (parameter nbstation) is allocated. All string keywords are initialized
   with NULL.
*/
mat_array *mat_array_new(int nbstation)
{
  mat_array   *arr;
  int          i;

  arr = (mat_array *)cpl_calloc(1, sizeof(mat_array));
  if (arr == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for mat_array");
      return NULL;
    }
  arr->list_station = (mat_station **)cpl_calloc(nbstation, sizeof(mat_station *));
  if (arr->list_station == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for mat_array->list_station");
      mat_array_delete(arr);
      return NULL;
    }
  arr->nbstation = nbstation; // needed for mat_array_delete
  for(i = 0; i < nbstation; i++)
    {
      arr->list_station[i] = (mat_station *)cpl_calloc(1, sizeof(mat_station));
      if (arr->list_station[i] == NULL)
	{
	  cpl_msg_error(cpl_func,"could not allocate memory for array->list_station[i]");
	  mat_array_delete(arr);
	  return NULL;
	}
    }
  return arr;
}

/**
   @ingroup oiarr
   @brief Deletes a mat_array data structure including all embedded mat_station data structures.
   @param array  A mat_array data structure.
   @return error code
*/
cpl_error_code mat_array_delete(mat_array *array)
{
  int i=0;
  
  if (array == NULL) return CPL_ERROR_NONE;
  /*
  if (array->origin != NULL)
    {
      cpl_free(array->origin);
      array->origin = NULL;
    }
  */
  if (array->telescope != NULL)
    {
      cpl_free(array->telescope);
      array->telescope = NULL;
    }
  /*
  if (array->date != NULL)
    {
      cpl_free(array->date);
      array->date = NULL;
    }
  */
  if (array->dcsdictionaryid != NULL)
    {
      cpl_free(array->dcsdictionaryid);
      array->dcsdictionaryid = NULL;
    }
  if (array->dcsid != NULL)
    {
      cpl_free(array->dcsid);
      array->dcsid = NULL;
    }
  if (array->arrayname != NULL)
    {
      cpl_free(array->arrayname);
      array->arrayname = NULL;
    }
  if (array->cooframe != NULL)
    {
      cpl_free(array->cooframe);
      array->cooframe = NULL;
    }
  if (array->list_station != NULL)
    {
      for (i = 0; i < array->nbstation; i++)
	{
	  if (array->list_station[i] != NULL)
	    {
	      cpl_free(array->list_station[i]);
	      array->list_station[i] = NULL;
	    }
	}
      cpl_free(array->list_station);
      array->list_station = NULL;
      array->nbstation = 0;
    }

  cpl_free(array);
  /* array=NULL; */
  return cpl_error_get_code();
}

/**
   @ingroup oiarr
   @brief Duplicates a mat_array structure.
   @param arr  A mat_array data structure which is duplicated.
   @return A newly allocated and filled mat_array data structure or NULL.
*/
mat_array * mat_array_duplicate(const mat_array *arr)
{
  mat_array *array=NULL;
  int i=0;

  cpl_errorstate prestate = cpl_errorstate_get();
  if( arr == NULL){
    return NULL;
  }

  // Pointers allocation
  array = mat_array_new(arr->nbstation);
  if (array == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for array");
      return NULL;
    }
  
  // Set data from ARRAY_GEOMETRY header
  //array->origin = cpl_strdup(arr->origin);
  array->telescope = cpl_strdup(arr->telescope);
  //array->date = cpl_strdup(arr->date);
  array->dcsdictionaryid = cpl_strdup(arr->dcsdictionaryid);
  array->dcsid = cpl_strdup(arr->dcsid);
  array->arrayname = cpl_strdup(arr->arrayname);
  array->cooframe = cpl_strdup(arr->cooframe);
  array->arrayx= arr->arrayx;
  array->arrayy= arr->arrayy;
  array->arrayz= arr->arrayz;

  // Copy stations
  for(i=0;i<array->nbstation;i++)
    {
      *(array->list_station[i]) = *(arr->list_station[i]);
    }
  /* Error handling */
  if (!cpl_errorstate_is_equal(prestate)) { 
    cpl_errorstate_dump(prestate, CPL_FALSE, cpl_errorstate_dump_one);
    mat_array_delete(array);
    return NULL;
  }
  return array;
}

/**
   @ingroup oiarr
   @brief Load a mat_array data structure from a CPL table and a propertylist.
   @param plist  This CPL propertylist contais the header keywords.
   @param table  This CPL table contains the OI_ARRAY binary table.
   @return A newly allocated and filled mat_array or NULL on failure.
*/
mat_array *mat_array_from_table(cpl_propertylist *plist, cpl_table *table)
{
  mat_array *array=NULL;
  int        nbstation;
  int        i,j;

  /*
  mat_assert_value((plist != NULL), CPL_ERROR_NULL_INPUT, NULL,
		   "no cpl_propertylist (plist) argument given");
  mat_assert_value((table != NULL), CPL_ERROR_NULL_INPUT, NULL,
		   "no cpl_table (table) argument given");
  */
  if (plist == NULL) return NULL;
  if (table == NULL) return NULL;

  nbstation = (int)cpl_table_get_nrow(table);
  array = mat_array_new(nbstation);
  mat_assert_value((array != NULL), CPL_ERROR_NULL_INPUT, NULL,
		   "could not allocate memory for array");

  array->extver          = mat_propertylist_get_int_default(plist, "EXTVER", 1);
  //array->origin          = mat_propertylist_get_string_default(NULL, SIZE_MAX_KEYWORDS, plist, "ORIGIN",      "empty");
  array->telescope       = mat_propertylist_get_string_default(NULL, SIZE_MAX_KEYWORDS, plist, "TELESCOP",    "empty");
  //array->date            = mat_propertylist_get_string_default(NULL, SIZE_MAX_KEYWORDS, plist, "DATE",        "empty");
  array->dcsdictionaryid = mat_propertylist_get_string_default(NULL, SIZE_MAX_KEYWORDS, plist, "ESO ISS DID", "empty");
  array->dcsid           = mat_propertylist_get_string_default(NULL, SIZE_MAX_KEYWORDS, plist, "ESO ISS ID",  "empty");
  array->arrayname       = mat_propertylist_get_string_default(NULL, SIZE_MAX_KEYWORDS, plist, "ARRNAME",     "empty");
  array->cooframe        = mat_propertylist_get_string_default(NULL, SIZE_MAX_KEYWORDS, plist, "FRAME",       "empty");
  array->arrayx          = mat_propertylist_get_double_default(plist, "ARRAYX", 0.0);
  array->arrayy          = mat_propertylist_get_double_default(plist, "ARRAYY", 0.0);
  array->arrayz          = mat_propertylist_get_double_default(plist, "ARRAYZ", 0.0);

  // Copy stations
  for(i = 0; i < array->nbstation; i++)
    {
      strncpy(array->list_station[i]->telname, 
	      cpl_table_get_string(table,"TEL_NAME", i), 17-1); array->list_station[i]->telname[17-1] = '\0';
      strncpy(array->list_station[i]->staname,
	      cpl_table_get_string(table,"STA_NAME", i), 17-1); array->list_station[i]->staname[17-1] = '\0';
      array->list_station[i]->staindex=
	cpl_table_get_int(table,"STA_INDEX", i, NULL);
      array->list_station[i]->diameter=
	cpl_table_get(table,"DIAMETER", i, NULL);
      for(j = 0; j < 3; j++)
	{
	  array->list_station[i]->staxyz[j]=
	    cpl_array_get_double(cpl_table_get_array(table,"STAXYZ",i), j,NULL);
	}
      if (cpl_table_get_ncol(table)>6) {
	array->list_station[i]->fov=cpl_table_get_double(table,"FOV", i, NULL);
	strncpy(array->list_station[i]->fovtype, cpl_table_get_string(table,"FOVTYPE", i), 7-1);
	array->list_station[i]->fovtype[7-1] = '\0';
      } else {
	array->list_station[i]->fov=0.;
	strncpy(array->list_station[i]->fovtype, "FOO", 7);
      }
    }
  
  /* Error handling */
  /* if (!cpl_errorstate_is_equal(prestate)) */
  /*   {  */
  /*     cpl_msg_info(cpl_func,"mat_array_load error: "); */
  /*     cpl_errorstate_dump(prestate, CPL_FALSE, cpl_errorstate_dump_one); */
  /*     return NULL; */
  /*   } */
  return array;
}

/**
   @ingroup oiarr
   @brief Stores a mat_array data structure into a property list and a table.
   @param array    The OI_ARRAY extension as mat_array data structure.
   @param plist    An optional property list for storing the keywords of the extension.

   This function creates a cpl_table data structure and fills it with the array setup
   given as a mat_array data structure. If a property list is given, the keywords
   of the mat_array data structure are stored in this list.
*/
cpl_table *mat_array_to_table(mat_array *array, cpl_propertylist *plist)
{
  cpl_table  *table;
  int         i;
  cpl_array  *sta_array = NULL;
  int         type = 0;
  const char *str;

  mat_assert_value((array != NULL), CPL_ERROR_NULL_INPUT, NULL,
		   "no mat_array (array) argument given");
  // put all properties of a OI_ARRAY binary table into the property list (if one is given)
  if (plist != NULL)
    {
      if (cpl_propertylist_has(plist,"EXTNAME") == 0)
	{
	  cpl_propertylist_append_string(plist,"EXTNAME", "OI_ARRAY");
	  if ( cpl_propertylist_has(plist,"OI_REVN") )
	    {
	      cpl_propertylist_erase(plist,"OI_REVN");
	    }
	  cpl_propertylist_append_int(plist,"OI_REVN",2);
	  type=2;
	}
      else
	{
	  str=cpl_propertylist_get_string(plist,"EXTNAME");
	  if (strcmp(str,"ARRAY_GEOMETRY") == 0)
	    {
	      type=1;
	    }
	  else
	    {
	      type=2;
	    }
	}
      cpl_propertylist_append_int(plist, "EXTVER", array->extver);
      cpl_propertylist_append_int(plist,"OI_REVN",2);
      /*
      if ((array->origin != NULL) && (strcmp(array->origin, "empty") != 0))
	{
	  cpl_propertylist_append_string(plist,"ORIGIN",array->origin);
	}
      */
      if ((array->telescope != NULL) && (strcmp(array->telescope, "empty") != 0))
	{
	  cpl_propertylist_append_string(plist,"TELESCOP",array->telescope);
	}
      /*
      if ((array->date != NULL) && (strcmp(array->date, "empty") != 0))
	{
	  cpl_propertylist_append_string(plist,"DATE",array->date);
	}
      */
      if ((array->dcsdictionaryid != NULL) && (strcmp(array->dcsdictionaryid, "empty") != 0))
	{
	  cpl_propertylist_append_string(plist,"ESO ISS DID", array->dcsdictionaryid);
	}
      if ((array->dcsid != NULL) && (strcmp(array->dcsid, "empty") != 0))
	{
	  cpl_propertylist_append_string(plist,"ESO ISS ID",array->dcsid);
	}
      if ((array->arrayname != NULL) && (strcmp(array->arrayname,"empty") != 0))
	{
	  cpl_propertylist_append_string(plist,"ARRNAME",array->arrayname);
	}
      if ((array->cooframe != NULL) && (strcmp(array->cooframe,"empty") != 0))
	{
	  cpl_propertylist_append_string(plist,"FRAME",array->cooframe);
	}
      cpl_propertylist_append_double(plist,"ARRAYX",array->arrayx);
      cpl_propertylist_append_double(plist,"ARRAYY",array->arrayy);
      cpl_propertylist_append_double(plist,"ARRAYZ",array->arrayz);
    }
  // Create an OI_ARRAY or ARRAY_GEOMETRY table
  table= cpl_table_new(array->nbstation);
  if (table == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for cpl_table");
      return NULL;
    }
  // Add columns in the table
  cpl_table_new_column(table, "TEL_NAME", CPL_TYPE_STRING);
  cpl_table_new_column(table, "STA_NAME", CPL_TYPE_STRING);
  cpl_table_new_column(table, "STA_INDEX", CPL_TYPE_INT);
  cpl_table_new_column(table, "DIAMETER", CPL_TYPE_FLOAT);
  cpl_table_set_column_unit(table, "DIAMETER", "m");
  cpl_table_new_column_array(table, "STAXYZ", CPL_TYPE_DOUBLE, 3);
  cpl_table_set_column_unit(table, "STAXYZ", "m");
  if (type == 1) {
    cpl_table_new_column(table, "MNTSTA", CPL_TYPE_INT);
  }
  cpl_table_new_column(table, "FOV", CPL_TYPE_DOUBLE);
  cpl_table_set_column_unit(table, "FOV", "arcsec");
  cpl_table_new_column(table, "FOVTYPE", CPL_TYPE_STRING);
  // Set data in the table
  for(i = 0; i < array->nbstation; i++)
    {
      cpl_table_set_string(table,"TEL_NAME", i, array->list_station[i]->telname);
      cpl_table_set_string(table,"STA_NAME", i, array->list_station[i]->staname);
      cpl_table_set_int(table,"STA_INDEX", i, array->list_station[i]->staindex);
      cpl_table_set_float(table,"DIAMETER", i, array->list_station[i]->diameter);
      sta_array = cpl_array_wrap_double(array->list_station[i]->staxyz, 3);
      cpl_table_set_array(table,"STAXYZ", i, sta_array);
      cpl_array_unwrap(sta_array);
      if (type == 1) {
	cpl_table_set_int(table,"MNTSTA", i, 0);
      }
      cpl_table_set_double(table,"FOV", i, array->list_station[i]->fov);
      cpl_table_set_string(table,"FOVTYPE", i, array->list_station[i]->fovtype);
    }
  cpl_table_set_column_savetype(table, "STA_INDEX", CPL_TYPE_SHORT);
  return table;
}

/**
   @ingroup oiarr
   @brief Save a mat_array structure in a FITS file.
   @param array       mat_array structure to save
   @param plist       primary header of the FITS file
   @param filename    FITS file name
   @param type        1: ARRAY_GEOMETRY, 2: OI_ARRAY
   @return cpl_error_code
*/
cpl_error_code mat_array_save(mat_array *array,
			      cpl_propertylist *plist,
			      char* filename,
			      int type)
{

  cpl_table *table=NULL;
  cpl_propertylist *keywords=NULL;
  cpl_frame *frame=NULL;
  int nbExtent=0;
  char *keyExtname=NULL;
  cpl_propertylist *plist_tmp=NULL;
  int i=0;
  FILE *fp=NULL;


  if (array == NULL){
    return CPL_ERROR_UNSPECIFIED;
  }

  if (filename == NULL){
    return CPL_ERROR_UNSPECIFIED;
  }
  frame = cpl_frame_new();
  cpl_frame_set_filename(frame, filename);
  // Return an error if a table ARRAY_GEOMETRY is already there
  fp=fopen(filename,"r");
  if (fp != NULL) {
    fclose(fp);
    nbExtent=cpl_frame_get_nextensions(frame);
    for(i=0;i<nbExtent;i++) {
      plist_tmp=cpl_propertylist_load(cpl_frame_get_filename(frame),i+1);
      
      keyExtname=(char *)cpl_propertylist_get_string(plist_tmp,"EXTNAME");
      if (keyExtname != NULL) {
	if (!strcmp(keyExtname,"ARRAY_GEOMETRY")) {
	  cpl_propertylist_delete(plist_tmp);
	  cpl_frame_delete(frame);
	  return CPL_ERROR_UNSPECIFIED;
	}
	if (!strcmp(keyExtname,"OI_ARRAY")) {
	  cpl_propertylist_delete(plist_tmp);
	  cpl_frame_delete(frame);
	  return CPL_ERROR_UNSPECIFIED;
	}
      }
      cpl_propertylist_delete(plist_tmp);
    }
  }
  // Create the ARRAY_GEOMETRY keywords
  keywords= cpl_propertylist_new();
  if (type == 1) {
    cpl_propertylist_append_string(keywords,"EXTNAME","ARRAY_GEOMETRY");  
  } else {
    cpl_propertylist_append_string(keywords,"EXTNAME","OI_ARRAY"); 
  }
  table = mat_array_to_table(array, keywords);


  // Append the table to the FITS file if it exists
  fp=fopen(filename,"r");
  if (fp == NULL) {
    cpl_table_save(table, plist, keywords, filename, CPL_IO_CREATE);
  } else {
    fclose(fp);
    cpl_table_save(table, NULL, keywords, filename, CPL_IO_EXTEND);
  }

  // Free memory
  if (frame != NULL) cpl_frame_delete(frame);
  if (table != NULL) cpl_table_delete(table);
  if (keywords != NULL) cpl_propertylist_delete(keywords);
  return cpl_error_get_code();
}

/**
   @ingroup oiarr
   @brief Append an OI_ARRAY binary table to an existing OIFITS file.
   @param filename  Name of the FITS file
   @param array     The mat_array data structure containing the informations for filling the binary table.
   @return cpl_error_code
*/
cpl_error_code  mat_array_append(char *filename,
				 mat_array *array)
{
  cpl_propertylist *keywords;
  cpl_table        *table;
  cpl_errorstate    prestate = cpl_errorstate_get();

  /*Check input parameters*/
  mat_assert_value((array!=NULL), CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT,
		   "no mat_array (array) argument given");
  mat_assert_value((filename!=NULL), CPL_ERROR_NULL_INPUT, 
		   CPL_ERROR_NULL_INPUT, "no filename given");
  
  keywords= cpl_propertylist_new();
  table = mat_array_to_table(array, keywords);
  cpl_table_save(table, NULL, keywords, filename, CPL_IO_EXTEND);
  if (table != NULL) cpl_table_delete(table);
  if (keywords != NULL) cpl_propertylist_delete(keywords);
  if (!cpl_errorstate_is_equal(prestate))
    {
      cpl_errorstate_set(prestate);
      return CPL_ERROR_UNSPECIFIED;
    }
  return CPL_ERROR_NONE;
}

/**
   @ingroup oiarr
   @brief Compare two mat_array data structures.
   @param array1  The first mat_array data structure.
   @param array2  The second mat_array data structure.
   @return Returns CPL_TRUE if array1 and array2 are identical.
*/
cpl_boolean mat_array_compare(mat_array *array1, mat_array *array2) {

  int i = 0;
  int j =0;
  int cpt = 0;
  mat_assert_value((array1 != NULL), CPL_ERROR_NULL_INPUT, CPL_FALSE,
		   "no mat_array (array1) argument given");
  mat_assert_value((array2 != NULL), CPL_ERROR_NULL_INPUT, CPL_FALSE,
		   "no mat_array (array2) argument given");
  if (  !strcmp(array1->telescope, array2->telescope)
	&& !strcmp(array1->dcsdictionaryid, array2->dcsdictionaryid)
	&& !strcmp(array1->dcsid, array2->dcsid)
	&& !strcmp(array1->arrayname, array2->arrayname)
	&& !strcmp(array1->cooframe, array2->cooframe)
	&& (array1->arrayx==array2->arrayx) 
	&& (array1->arrayy == array2->arrayy)
	&& (array1->arrayz == array2->arrayz)
	&& (array1->nbstation == array2->nbstation) )
    {
      for (i = 0; i<array1->nbstation; i++)
	{
	  if (   (array1->list_station[i]->staindex == 
		  array2->list_station[i]->staindex)
		 && (array1->list_station[i]->diameter == 
		     array2->list_station[i]->diameter)
		 )
	    {
	      for (j = 0; j<3; j++)
		{
		  if (fabs(array1->list_station[i]->staxyz[j] - 
			  array1->list_station[i]->staxyz[j]) < 1.E-4) {			    		    
		    cpt++;
		  } 
		}
	    } 
	}
      if (cpt == 3*array1->nbstation)
	{
	  return CPL_TRUE;
	}   
      else
	{
	  return CPL_FALSE;
	}
    }
  else
    {
      return CPL_FALSE;
    }  
}

