/* $Id: mat_array.h,v0.5 2014-06-15 12:56:21 mheininger Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: mheininger $
 * $Date: 2014/05/21 11:07:00 $
 * $Revision: 0.5 $
 * $Name: mat_array.h $
 */

#ifndef MAT_ARRAY_H
#define MAT_ARRAY_H

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include <cpl.h>

/*-----------------------------------------------------------------------------
                                   Define
 -----------------------------------------------------------------------------*/

/* Structure mat_station */
/**
   @ingroup oiarr
   @brief Structure containing one station of a OI_ARRAY.

   Each station of a OI_ARRAY binary table (each row) is stored in a mat_station
   data structure.
*/
typedef struct {
    char          telname[17];  /*!< Telescope name. */
    char          staname[17];  /*!< Station name. */
    int           staindex;     /*!< Station number must be greater or equal 1. */
    double        diameter;     /*!< Telescope diameter in meter. */
    double        staxyz[3];    /*!< Station coordinates in meter. */
    double fov;                      /*!< Field of view (arcsec) */
    char fovtype[7];                   /*!< Field of view type : FWHM or RADIUS*/
} mat_station;

/* Structure mat_array */
/**
   @ingroup oiarr
   @brief Structure containing OI_ARRAY.
*/
typedef struct {
    int            extver;           /*!< Extension version (optional). */
  //char          *origin;           /*!< Origin of the data. */
    char          *telescope;        /*!< ESO telescope name */
  //char          *date;             /*!< HDU creation date */
    char          *dcsdictionaryid;  /*!< data dic name */
    char          *dcsid;            /*!< DCS software version strin*/
    char          *arrayname;        /*!< Array name for cross referencing. */
    char          *cooframe;         /*!< Coordinate frame. */
    double         arrayx;           /*!< Array center, X coordinate in meter. */
    double         arrayy;           /*!< Array center, Y coordinate in meter. */
    double         arrayz;           /*!< Array center, Z coordinate in meter. */
    int            nbstation;        /*!< Number of stations (rows in the binary table). */
    mat_station  **list_station;     /*!< Array of stations. */
} mat_array;

/*-----------------------------------------------------------------------------
                                   Functions prototypes
 -----------------------------------------------------------------------------*/
mat_array *mat_array_new(int nbstation);

cpl_error_code mat_array_delete(mat_array *array);

mat_array * mat_array_duplicate(const mat_array * arr);

mat_array *mat_array_from_table(cpl_propertylist *plist, 
				cpl_table *table);

cpl_table *mat_array_to_table(mat_array *array,
			      cpl_propertylist *plist);

cpl_error_code mat_array_save(mat_array *array,
			      cpl_propertylist *plist,
			      char *filename,
			      int type);

cpl_error_code  mat_array_append(char *filename,mat_array *array);

cpl_boolean mat_array_compare(mat_array *array1, mat_array *array2);

#endif
