/* $Id: mat_badpixel.c,v0.5 2014-06-15 12:56:21 mheininger Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: mheininger $
 * $Date: 2014/05/21 11:07:00 $
 * $Revision: 0.5 $
 * $Name: mat_badpixel.c $
 */

#include <stdlib.h>
#include <math.h>
#include <stdio.h>
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include <string.h>

#include "mat_badpixel.h"
#include "mat_frame.h"
#include "mat_imagingdetector.h"
#include "mat_utils.h"

/*-----------------------------------------------------------------------------
                                   Define
 -----------------------------------------------------------------------------*/

#define MAT_LOCAL_WINDOW_SIZE  7

/*-----------------------------------------------------------------------------
                                   Functions
 -----------------------------------------------------------------------------*/

/**
   @ingroup bpm
   @brief Creates an empty badpixel map assocciated with a detector and covering the whole detector or only a set of sub-windows on that detector.
   @param det     This structure describes the detector of the badpixel map.
   @param imgdet  This optional structure describes the sub-window-setup.
   @return This function returns the new badpixel map of NULL on error.

   This function creates a badpixel map data structure using the detector specification
   as a template. If a sub-window setup is given, this setup is used as setup for the
   badpixel map. This means, that for each sub-window an individual cpl_mask is allocated
   and stored in the list_badpixel member.

   Possible cpl_error_code set in this function:
   - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
mat_badpixel *mat_badpixel_new(mat_detector *det,
			       mat_imagingdetector *imgdet)
{
  mat_badpixel *bpm = NULL;
  int           r;

  if (det == NULL)
    {
      cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_detector (det) argument given");
      return NULL;
    }
  bpm = (mat_badpixel *)cpl_calloc(1, sizeof(mat_badpixel));
  if (bpm == NULL)
    {
      cpl_msg_error(cpl_func, "could not allocate memory for mat_badpixel");
      return NULL;
    }
  bpm->det = mat_detector_duplicate(det);
  if (bpm->det == NULL)
    {
      cpl_msg_error(cpl_func, "could not duplicate mat_detector");
      mat_badpixel_delete(bpm);
      return NULL;
    }
  bpm->keywords = cpl_propertylist_new();
  if (bpm->keywords == NULL)
    {
      cpl_msg_error(cpl_func, "could not allocate memory for cpl_propertylist");
      mat_badpixel_delete(bpm);
      return NULL;
    }
  if (imgdet == NULL)
    { /* create an imaging detector table covering the whole detector */
      bpm->imgdet = mat_imagingdetector_new(NULL, 1); /* no propertylist and one sub-window */
      if (bpm->imgdet == NULL)
	{
	  cpl_msg_error(cpl_func, "could not allocate memory for mat_imagingdetector");
	  mat_badpixel_delete(bpm);
	  return NULL;
	}
      /* set the geometry of the sub-window to the whole detector */
      bpm->imgdet->list_region[0]->corner[0] = 1;
      bpm->imgdet->list_region[0]->corner[1] = 1;
      bpm->imgdet->list_region[0]->naxis[0] = bpm->det->nx;
      bpm->imgdet->list_region[0]->naxis[1] = bpm->det->ny;
    }
  else
    {
      bpm->imgdet = mat_imagingdetector_duplicate(imgdet);
      if (bpm->imgdet == NULL)
	{
	  cpl_msg_error(cpl_func, "could not duplicate the mat_imagingdetector");
	  mat_badpixel_delete(bpm);
	  return NULL;
	}
    }
  bpm->list_badpixel = (cpl_mask **)cpl_calloc(bpm->imgdet->nbregion, sizeof(cpl_mask *));
  if (bpm->list_badpixel == NULL)
    {
      cpl_msg_error(cpl_func, "could not create a cpl_mask list");
      mat_badpixel_delete(bpm);
      return NULL;
    }
  bpm->nbtotal = 0;
  for (r = 0; r < bpm->imgdet->nbregion; r++)
    {
      int nx = bpm->imgdet->list_region[r]->naxis[0];
      int ny = bpm->imgdet->list_region[r]->naxis[1];
      bpm->list_badpixel[r] = cpl_mask_new(nx, ny);
      bpm->nbtotal += nx*ny;
      if (bpm->list_badpixel[r] == NULL)
	{
	  cpl_msg_error(cpl_func, "could not create a cpl_mask");
	  mat_badpixel_delete(bpm);
	  return NULL;
	}
    }
  bpm->nbgood = bpm->nbtotal;
  bpm->nbbad = 0;
  bpm->goodpixelratio = 1.0;
  return bpm;
}

/**
   @ingroup bpm
   @brief Resets a bad pixel map.
   @param bpm  The bad pixel map which will be reset.
   @return the cpl_error_code or CPL_ERROR_NONE

   A previously created bad pixel map is reset to an empty bad pixel map.

   Possible cpl_error_code set in this function:
   - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
cpl_error_code mat_badpixel_reset(mat_badpixel *bpm)
{
  if (bpm == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_badpixel (bpm) argument given");
    }
  if (bpm->list_badpixel != NULL)
    {
      if (bpm->imgdet == NULL)
	{
	  if (bpm->list_badpixel[0] != NULL)
	    {
	      int x, nx, y, ny;
	      nx = cpl_mask_get_size_x(bpm->list_badpixel[0]);
	      ny = cpl_mask_get_size_y(bpm->list_badpixel[0]);
	      for (y = 1; y <= ny; y++)
		{
		  for (x = 1; x <= nx; x++)
		    {
		      cpl_mask_set(bpm->list_badpixel[0], x, y, CPL_BINARY_0);
		    }
		}
	    }
	}
      else
	{
	  int r;
	  for (r = 0; r < bpm->imgdet->nbregion; r++)
	    {
	      if (bpm->list_badpixel[r] != NULL)
		{
		  int x, nx, y, ny;
		  nx = cpl_mask_get_size_x(bpm->list_badpixel[r]);
		  ny = cpl_mask_get_size_y(bpm->list_badpixel[r]);
		  for (y = 1; y <= ny; y++)
		    {
		      for (x = 1; x <= nx; x++)
			{
			  cpl_mask_set(bpm->list_badpixel[r], x, y, CPL_BINARY_0);
			}
		    }
		}
	    }
	}
    }
  return CPL_ERROR_NONE;
}

/**
   @ingroup bpm
   @brief Deletes a bad pixel map.
   @param bpm  The bad pixel map which will be deleted.
   @return the cpl_error_code or CPL_ERROR_NONE

   A previously created bad pixel map is deleted from memory. This includes deleting
   the internal pixel masks!

   Possible cpl_error_code set in this function:
   - CPL_ERROR_NULL_INPUT if an input pointer is NULL
 */
cpl_error_code mat_badpixel_delete(mat_badpixel *bpm)
{
  if (bpm == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_badpixel (bpm) argument given");
    }
  if (bpm->keywords != NULL)
    {
      cpl_propertylist_delete(bpm->keywords);
      bpm->keywords = NULL;
    }
  if (bpm->list_badpixel != NULL)
    {
      if (bpm->imgdet == NULL)
	{
	  if (bpm->list_badpixel[0] != NULL)
	    {
	      cpl_mask_delete(bpm->list_badpixel[0]);
	      bpm->list_badpixel[0] = NULL;
	    }
	}
      else
	{
	  int r;
	  for (r = 0; r < bpm->imgdet->nbregion; r++)
	    {
	      if (bpm->list_badpixel[r] != NULL)
		{
		  cpl_mask_delete(bpm->list_badpixel[r]);
		  bpm->list_badpixel[r] = NULL;
		}
	    }
	}
      cpl_free(bpm->list_badpixel);
      bpm->list_badpixel = NULL;
    }
  if (bpm->imgdet != NULL)
    {
      mat_imagingdetector_delete(bpm->imgdet);
      bpm->imgdet = NULL;
    }
  if (bpm->det != NULL)
    {
      mat_detector_delete(bpm->det);
      bpm->det = NULL;
    }
  cpl_free(bpm);
  return CPL_ERROR_NONE;
}

/**
   @ingroup bpm
   @brief Gets the bad pixel status of a specific pixel
   @param bpm the bad pixel map
   @param r   the region number (starting at 0)
   @param x   the x coordinate relative to the region (starting at 1)
   @param y   the y coordinate relative to the region (starting at 1)
   @return 1 if the pixel is a bad pixel, 0 otherwise

   This function can be used to get the bad pixel status of a specific pixel
   in a specific sub-window (region) of a bad pixel map.

   The static bad pixel map always covers the whole detector (only one region).
   After mapping the static bad pixel map to a specific science observation
   with up to 56 sub-windows (photometric, interferometric and reference sub-windows),
   a bad pixel map contains several pixel masks (one for each sub-window).

   Possible cpl_error_code set in this function:
   - CPL_ERROR_NULL_INPUT if an input pointer is NULL
   - CPL_ERROR_UNSPECIFIED if the internal data structure is corrupt (should not happen!)
   - CPL_ERROR_ACCESS_OUT_OF_RANGE if the region number or the coordinates are outside
     the bad pixel map
 */
cpl_binary mat_badpixel_get(mat_badpixel *bpm, int r, int x, int y)
{
  if (bpm == NULL)
    {
      cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_badpixel (bpm) argument given");
      return CPL_BINARY_0;
    }
  if (bpm->list_badpixel == NULL)
    {
      cpl_error_set_message(cpl_func, CPL_ERROR_UNSPECIFIED, "no bad pixel masks in structure");
      return CPL_BINARY_0;
    }
  if (bpm->imgdet == NULL)
    {
      cpl_error_set_message(cpl_func, CPL_ERROR_UNSPECIFIED, "could not get the number of bad pixel masks since bpm->imgdet is empty");
      return CPL_BINARY_0;
    }
  if (r < 0)
    {
      cpl_error_set_message(cpl_func, CPL_ERROR_ACCESS_OUT_OF_RANGE, "region number must be >= 0");
      return CPL_BINARY_0;
    }
  if (r >= bpm->imgdet->nbregion)
    {
      cpl_error_set_message(cpl_func, CPL_ERROR_ACCESS_OUT_OF_RANGE, "not enough regions in bad pixel map");
      return CPL_BINARY_0;
    }
  return cpl_mask_get(bpm->list_badpixel[r], x, y);
}

/**
   @ingroup bpm
   @brief The static bad pixel map is written to a FITS file.
   @param bpm      The mandatory badpixel map.
   @param fname    The mandatory filename for the static badpixel map.
   @param rname    The name of the pipeline recipe.
   @param parlist  The parameterlist of the plugin.
   @param frameset The frameset of the plugin.
   @return the cpl_error_code or CPL_ERROR_NONE

   This function stores the contents of the mat_badpixel data structure as a static badpixel map
   in a FITS file. The badpixel map must cover the whole detector (no sub-window setup is allowed).
   The badpixel map is stored as a binary image in the primary header where all bad
   pixels are marked as 1 and all good pixels are marked as 0. In addition, some QC parameters
   from the data structure are stored as primary header keywords:

   QC DETi GOODPIX  the number of good pixels
   QC DETi BADPIX the number of bad pixels
   QC DETi GOODPIXELRATIO the ratio bad versus good pixels

   Where i is the detector number (HAWAII-2RG -> 1, Aquarius -> 2).

   Possible cpl_error_code set in this function:
   - CPL_ERROR_NULL_INPUT if an input pointer is NULL
   - CPL_ERROR_UNSPECIFIED if it is not possible to fill the DFS header
   or store the bad pixel map in a file
 */
cpl_error_code mat_badpixel_save(mat_badpixel *bpm,
				 const char *fname,
				 const char *rname,
				 cpl_parameterlist *parlist,
				 cpl_frameset *frameset)
{
  cpl_frame        *bpmframe = NULL;
  cpl_propertylist *plist = NULL;
  char              kwd[64];

  cpl_msg_info(cpl_func, "storing the bad pixel map in file %s", fname);
  cpl_error_reset();
  if (bpm == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_badpixel (bpm) argument given");
    }
  /* Create product frame */
  bpmframe = cpl_frame_new();
  cpl_frame_set_filename(bpmframe, fname);
  cpl_frame_set_tag(bpmframe, MATISSE_BPM_PROCATG);
  cpl_frame_set_type(bpmframe, CPL_FRAME_TYPE_IMAGE);
  cpl_frame_set_group(bpmframe, CPL_FRAME_GROUP_PRODUCT);
  cpl_frame_set_level(bpmframe, CPL_FRAME_LEVEL_FINAL);
  if (cpl_error_get_code()) {
    cpl_msg_error(cpl_func, "Error while initialising the product frame, code = %d, message = %s",
		  cpl_error_get_code(),
		  cpl_error_get_message());
    cpl_frame_delete(bpmframe);
    return CPL_ERROR_UNSPECIFIED;
  }
  plist = cpl_propertylist_new();
  /* Add DataFlow keywords */
  /*
    check if the input frameset is empty
    => function used in an artificial unit test (no real input files)
    => the DFS header cannot include the keywords from the first input file
  */
  if (!cpl_frameset_is_empty(frameset))
    {
      if (cpl_dfs_setup_product_header(plist, bpmframe, frameset, parlist,
				       rname, PACKAGE "/" PACKAGE_VERSION, "?Dictionary?", NULL) != CPL_ERROR_NONE) {
	cpl_msg_error(cpl_func, "Problem in the product DFS-compliance, code = %d, message = %s",
		      cpl_error_get_code(),
		      cpl_error_get_message());
	cpl_error_reset();
	cpl_propertylist_delete(plist);
	cpl_frame_delete(bpmframe);
	return CPL_ERROR_UNSPECIFIED;
      }
    }
  /* Add QC parameters in the header */
  strncpy(kwd, "GOODPIX", 64); kwd[63] = '\0';
  cpl_propertylist_append_int(plist, kwd, bpm->nbgood);
  strncpy(kwd, "BADPIX", 64); kwd[63] = '\0';
  cpl_propertylist_append_int(plist, kwd, bpm->nbbad);
  snprintf(kwd, 64, "ESO QC DET%d GOODPIX%d", bpm->det->nr, bpm->det->read_id);
  cpl_propertylist_append_int(plist, kwd, bpm->nbgood);
  snprintf(kwd, 64, "ESO QC DET%d BADPIX%d", bpm->det->nr, bpm->det->read_id);
  cpl_propertylist_append_int(plist, kwd, bpm->nbbad);
  snprintf(kwd, 64, "ESO QC DET%d GOODPIXELRATIO%d", bpm->det->nr, bpm->det->read_id);
  cpl_propertylist_append_double(plist, kwd, bpm->goodpixelratio);
  //(info->det.nx*info->det.ny - info->nbad)/(double)(info->det.nx*info->det.ny));


  // Add Generic QC parameters
  cpl_propertylist *qclist=NULL;
  qclist = cpl_propertylist_new();
  mat_add_generic_qc(plist,qclist);
  cpl_propertylist_append(plist,qclist);
  cpl_propertylist_delete(qclist);


  /* Save the file */
  cpl_propertylist_erase(plist,"RADECSYS");
  if (cpl_mask_save(bpm->list_badpixel[0], fname, plist,
		    CPL_IO_CREATE) != CPL_ERROR_NONE) {
    cpl_msg_error(cpl_func, "Could not save product, code = %d, message = %s",
		  cpl_error_get_code(),
		  cpl_error_get_message());
    cpl_propertylist_delete(plist);
    cpl_frame_delete(bpmframe);
    return CPL_ERROR_UNSPECIFIED;
  }
  cpl_propertylist_delete(plist);
  /* Log the saved file in the input frameset */
  cpl_frameset_insert(frameset, bpmframe);
  //cpl_frame_delete(bpmframe);
  return CPL_ERROR_NONE;
}

/**
   @ingroup bpm
   @brief This function loads the content of a static BADPIXEL map in a mat_badpixel structure.
   @param bpmframe The frame containing the name of the badpixel map.
   @returns The loaded badpixel map or NULL.

   This function reads the static badpixel map and stores the contents in a mat_badpixel
   data structure. The badpixel map covers the whole detector and must be mapped
   to a given sub-window layout by using the mat_badpixel_map function.
*/
mat_badpixel *mat_badpixel_load(cpl_frame *bpmframe)
{
  mat_detector     *det = NULL;
  cpl_propertylist *plist = NULL;
  cpl_mask         *map = NULL;
  mat_badpixel     *bpm = NULL;
  char              kwd[64];

  cpl_msg_info(cpl_func, "load the badpixel map from file %s", cpl_frame_get_filename(bpmframe));
  /* create a mat_detector structure, load the primary keywords and decode the detector */
  det = mat_detector_new();
  if (det == NULL)
    {
      cpl_msg_error(cpl_func, "could not allocate memory for mat_badpixel->det");
      return NULL;
    }
  plist = cpl_propertylist_load(cpl_frame_get_filename(bpmframe), 0);
  if (plist == NULL)
    {
      cpl_msg_error(cpl_func, "could not load the primary keywords");
      mat_detector_delete(det);
      return NULL;
    }
  mat_detector_decode(det, plist);
  /* create and initialize the mat_badpixel structure */
  bpm = mat_badpixel_new(det, NULL); /* no mat_imagingdetector means the whole detector */
  mat_detector_delete(det); /* we can delete the mat_detector, because we have a copy in bpm */
  det = NULL;
  if (bpm == NULL)
    {
      cpl_msg_error(cpl_func, "could not allocate memory for mat_badpixel");
      cpl_propertylist_delete(plist);
      return NULL;
    }
  /* move the propertylist into the bpm */
  cpl_propertylist_delete(bpm->keywords);
  bpm->keywords = plist;
  /* get all important keyword values from the propertylist */
  bpm->nbtotal = bpm->det->nx * bpm->det->ny;
  bpm->nbgood = cpl_propertylist_get_int(bpm->keywords, "GOODPIX");
  bpm->nbbad = cpl_propertylist_get_int(bpm->keywords, "BADPIX");
  snprintf(kwd, 64, "ESO QC DET%d GOODPIXELRATIO%d", bpm->det->nr, bpm->det->read_id);
  bpm->goodpixelratio = cpl_propertylist_get_double(bpm->keywords, kwd);
  /* and now load the real badpixel map from the file */
  map = cpl_mask_load(cpl_frame_get_filename(bpmframe), 0, 0);
  if (map == NULL)
    {
      cpl_msg_error(cpl_func, "could not load the badpixel mask for bpm->list_badpixel[0]");
      mat_badpixel_delete(bpm);
      return NULL;
    }
  else
    {
      cpl_mask_delete(bpm->list_badpixel[0]);
      bpm->list_badpixel[0] = map;
    }
  return bpm;
}

/**
   @ingroup bpm
   @brief This function maps a badpixel map to a given sub-window layout.
   @param bpm    This badpixel map covers the whole detector.
   @param imgdet This structure describes the sub-window layout.
   @returns The mapped badpixel map or NULL.

   This function uses the sub-window layout of the imgdet argument to create
   a new badpixel map (using the function mat_badpixel_new and the layout in imgdet).
   In a second step, the sub-window specific parts of the badpixel mask are extracted
   and stored in the newly created badpixel map. The number of good and bad pixels
   are recalculated for the pixels in the sub-windows.
 */
mat_badpixel *mat_badpixel_map(mat_badpixel *bpm, mat_imagingdetector *imgdet)
{
  mat_badpixel *nbpm = NULL;
  int           r;

  if (bpm == NULL)
    {
      cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_badpixel (bpm) argument given");
      return NULL;
    }
  if (imgdet == NULL)
    {
      cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_imagingdetector (imgdet) argument given");
      return NULL;
    }
  if (imgdet->list_region == NULL)
    {
      cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no valid mat_imagingdetector (imgdet->list_region) argument given");
      return NULL;
    }
  nbpm = mat_badpixel_new(bpm->det, imgdet);
  if (nbpm == NULL)
    {
      cpl_msg_error(cpl_func, "could not allocate memory for mat_badpixel");
      return NULL;
    }
  /* put a copy of the propertylist into the newly created badpixel map */
  if (nbpm->keywords != NULL)
    {
      cpl_propertylist_delete(nbpm->keywords);
      nbpm->keywords = NULL;
      nbpm->keywords = cpl_propertylist_duplicate(bpm->keywords);
    }
  /* extract only the sub-window equivalents of the badpixel map */
  nbpm->nbbad = 0;
  for (r = 0; r < nbpm->imgdet->nbregion; r++)
    {
      if (nbpm->list_badpixel[r] != NULL)
	{
	  cpl_mask_delete(nbpm->list_badpixel[r]);
	  nbpm->list_badpixel[r] = NULL;
	}
      nbpm->list_badpixel[r] = cpl_mask_extract(bpm->list_badpixel[0],
						imgdet->list_region[r]->corner[0],
						imgdet->list_region[r]->corner[1],
						imgdet->list_region[r]->corner[0] + imgdet->list_region[r]->naxis[0] - 1,
						imgdet->list_region[r]->corner[1] + imgdet->list_region[r]->naxis[1] - 1);
      
      nbpm->nbbad += cpl_mask_count(nbpm->list_badpixel[r]);
    }
  nbpm->nbgood = nbpm->nbtotal - nbpm->nbbad;
  nbpm->goodpixelratio = ((double)nbpm->nbgood)/((double)nbpm->nbtotal);
  return nbpm;
}


/**
   @ingroup bpm
   @brief Detect bad pixel using a global criteria.
   @param data       this input frame contains pixel specific values
   @param stdevlimit this sigma factor defines the value range for good pixels
   @param bpm        this bad pixel map is modified
   @return the cpl_error_code or CPL_ERROR_NONE

   This function uses the values in the input frame (data) to calculate their
   median and standard deviation for each sub-window. Using the stdevlimit parameter,
   the range for the good pixels inside a sub-window is calculated. All pixels
   having values outside this range are defined as bad pixels. For calculating
   the range, the pixels already marked as bad pixels are ignored.
   
   Mathematical description:

<pre>BEGIN
   FOR EACH sub-window r DO
      CALL mat_badpixel_detect_global_window for the whole sub-window;
   ENDFOR
END</pre>

   Possible cpl_error_code set in this function:
   - CPL_ERROR_NULL_INPUT if an input pointer (data or bpm) is NULL
*/
cpl_error_code mat_badpixel_detect_global(mat_frame *data,
					  double stdevlimit,
					  mat_badpixel *bpm)
{
  int    r;

  if (data == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_frame (data) argument given");
    }
  if (bpm == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_badpixel (bpm) argument given");
    }
  for (r = 0; r < data->nbsubwin; r++)
    {
      mat_badpixel_detect_global_window(data, stdevlimit, bpm, r, 1, 1, -1, -1);
    }
  return CPL_ERROR_NONE;
}

/**
   @ingroup bpm
   @brief Detect bad pixel using a global criteria inside an area.
   @param data       this input frame contains pixel specific values
   @param stdevlimit this sigma factor defines the value range for good pixels
   @param bpm        this bad pixel map is modified
   @param r          the sub-window/region (starting with 0)
   @param x0         left side of the area inside a sub-window (starting with 1)
   @param y0         bottom side of the area inside a sub-window (starting with 1)
   @param nx         width of the area in pixels
   @param ny         height of the area in pixels
   @return the cpl_error_code or CPL_ERROR_NONE
   
   This function uses the values in the input frame (data) to calculate their
   median and standard deviation for an area (parameters x0, y0, xn and yn) inside
   a specific sub-window (parameter r). Using the stdevlimit parameter, the range
   for the good pixels inside that area is calculated. All pixels having values
   outside this range are defined as bad pixels. For calculating the range, the
   pixels already marked as bad pixels are ignored.

   This function is needed, because the HAWAII-2RG and Aquarius detectors are
   organized as several independent detector channels (32 for the HAWAII-2RG
   and 64 for the Aquarius). The calibration measurement covers the whole
   detector and stores the data in one sub-window. Because the pixels of different
   detector channels may behave slightly different, the range for the valid pixels
   must be calculated for each detector channel and used only for the pixels inside
   that channel. This means, that normally r is equal to 0 (only one sub-window in
   the input data) and x0, y0, nx and ny defines exactly one detector channel.

   Mathematical description:

<pre>BEGIN
    Calculate the global median and stdev for data[r, ...];
    lower_limit := median - stdevlimit*stdev;
    upper_limit := median + stdevlimit*stdev;
    FOR EACH pixel inside data[r, ...] DO
       IF (data[r, pixel] < lower_limit) OR (data[r, pixel] > upper_limit)
          THEN bpm[r, pixel] := is bad;
       ENDIF
    ENDFOR
END</pre>

   Possible cpl_error_code set in this function:
   - CPL_ERROR_NULL_INPUT if an input pointer (data or bpm) is NULL
*/
cpl_error_code mat_badpixel_detect_global_window(mat_frame *data,
						 double stdevlimit,
						 mat_badpixel *bpm,
						 int r,
						 int x0,
						 int y0,
						 int nx,
						 int ny)
{
  mat_imgreg *region = NULL;
  cpl_image  *image  = NULL;
  cpl_mask   *nbpm   = NULL;
  double median, stdev, lower_limit, upper_limit;

  if (data == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_frame (data) argument given");
    }
  if (bpm == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_badpixel (bpm) argument given");
    }
  if ((r < 0) || (r >= data->nbsubwin))
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_ACCESS_OUT_OF_RANGE, "region %d outside of [0 .. %d]", r, data->nbsubwin - 1);
    }
  region = data->list_subwin[r];
  image = region->imgreg[0]; /* exactly one cpl_image for each region! */
  if (nx == -1)
    {
      nx = cpl_image_get_size_x(image);
    }
  if (ny == -1)
    {
      ny = cpl_image_get_size_y(image);
    }
  cpl_image_reject_from_mask(image, bpm->list_badpixel[r]); /* ignore the already found bad pixels */
  median = cpl_image_get_median_dev_window(image, x0, y0, x0 + nx - 1, y0 + ny - 1, &stdev);
  lower_limit = median - stdevlimit*stdev;
  upper_limit = median + stdevlimit*stdev;
  nbpm = cpl_mask_threshold_image_create(image, lower_limit, upper_limit);
  if (nbpm != NULL)
    {
      int count = 0;
      int x;
      cpl_mask_not(nbpm); // the pixels with values outside the range are bad
      for (x = x0; x <= x0 + nx - 1; x++)
	{
	  int y;
	  for (y = y0; y <= y0 + ny - 1; y++)
	    {
	      if ((cpl_mask_get(nbpm, x, y) == CPL_BINARY_1) && (cpl_mask_get(bpm->list_badpixel[r], x, y) != CPL_BINARY_1))
		{
		  count++;
		  /* cpl_msg_info(cpl_func, "pixel at %d, %d, %d added because %lf != [%lf .. %lf]", r, x, y, mat_frame_get(data, r, x, y), lower_limit, upper_limit); */
		  cpl_mask_set(bpm->list_badpixel[r], x, y, CPL_BINARY_1);
		}
	    }
	}
      //cpl_msg_info("mat_badpixel_detect_global_window", "nbad=%d", count);
      cpl_mask_delete(nbpm);
    }
  return CPL_ERROR_NONE;
}

/**
   @ingroup bpm
   @brief Detect bad pixel using a local criteria.
   @param data       this input frame contains pixel specific values
   @param stdevlimit this sigma factor defines the value range for good pixels
   @param bpm        this bad pixel map is modified
   @return the cpl_error_code or CPL_ERROR_NONE

   This function uses the values in the input frame (data) to calculate for each
   pixel the local median and standard deviation. Using the stdevlimit parameter,
   the local range for the good pixels is calculated. A pixel having a value
   outside this range is marked as bad pixel. For calculating the range, the
   pixels already marked as bad pixels are ignored.
   
   Mathematical description:

<pre>BEGIN
   FOR EACH sub-window r DO
      CALL mat_badpixel_detect_local_window for the whole sub-window;
   ENDFOR
END</pre>

   Possible cpl_error_code set in this function:
   - CPL_ERROR_NULL_INPUT if an input pointer (data or bpm) is NULL
*/
cpl_error_code mat_badpixel_detect_local(mat_frame *data,
					 double stdevlimit,
					 mat_badpixel *bpm)
{
  int    r;

  if (data == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_frame (data) argument given");
    }
  if (bpm == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_badpixel (bpm) argument given");
    }
  for (r = 0; r < data->nbsubwin; r++)
    {
      mat_badpixel_detect_local_window(data, stdevlimit, bpm, r, 1, 1, -1, -1);
    }
  //cpl_msg_info("mat_badpixel_detect_local", "nbad=%d", count);
  return CPL_ERROR_NONE;
}

/**
   @ingroup bpm
   @brief Detect bad pixel inside an area using a local criteria.
   @param data This data structure contains the statistical values (median or variance) for a detector.
   @param stdevlimit Defines the standard deviation limit used for the detecting bad pixels.
   @param bpm This data structure contains the current bad pixel map for the same detector.
   @param r          the sub-window/region (starting with 0)
   @param x0         left side of the area inside a sub-window (starting with 1)
   @param y0         bottom side of the area inside a sub-window (starting with 1)
   @param nx         width of the area in pixels
   @param ny         height of the area in pixels
   @return the cpl_error_code or CPL_ERROR_NONE

   This function uses the values in the input frame (data) to calculate their
   median and standard deviation for an area (parameters x0, y0, xn and yn) inside
   a specific sub-window (parameter r). Using the stdevlimit parameter, the range
   for the good pixels inside that area is calculated. All pixels having values
   outside this range are defined as bad pixels. For calculating the range, the
   pixels already marked as bad pixels are ignored.

   This function is needed, because the HAWAII-2RG and Aquarius detectors are
   organized as several independent detector channels (32 for the HAWAII-2RG
   and 64 for the Aquarius). The calibration measurement covers the whole
   detector and stores the data in one sub-window. Because the pixels of different
   detector channels may behave slightly different, the range for the valid pixels
   must be calculated for each detector channel and used only for the pixels inside
   that channel. This means, that normally r is equal to 0 (only one sub-window in
   the input data) and x0, y0, nx and ny defines exactly one detector channel.

   Mathematical description:

<pre>BEGIN
   FOR EACH pixel inside data[r, ...] DO
      Calculate the local median and stdev for the data[r, pixel];
      lower_limit := median - stdevlimit*stdev;
      upper_limit := median + stdevlimit*stdev;
      IF (data[r, pixel] < lower_limit) OR (data[r, pixel] > upper_limit) THEN
         bpm[r, pixel] := is bad;
      ENDIF
   ENDFOR
END</pre>
*/
cpl_error_code mat_badpixel_detect_local_window(mat_frame *data,
						double stdevlimit,
						mat_badpixel *bpm,
						int r,
						int x0,
						int y0,
						int nx,
						int ny)
{
  mat_imgreg *region = NULL;
  cpl_image  *image  = NULL;
  int         x, y, count;

  if (data == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_frame (data) argument given");
    }
  if (bpm == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_badpixel (bpm) argument given");
    }
  if ((r < 0) || (r >= data->nbsubwin))
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_ACCESS_OUT_OF_RANGE, "region %d outside of [0 .. %d]", r, data->nbsubwin - 1);
    }
  region = data->list_subwin[r];
  image = region->imgreg[0]; /* exactly one cpl_image for each region! */
  if (nx == -1)
    {
      nx = cpl_image_get_size_x(image);
    }
  if (ny == -1)
    {
      ny = cpl_image_get_size_y(image);
    }
  cpl_image_reject_from_mask(image, bpm->list_badpixel[r]); /* ignore the already found bad pixels */
  count = 0;
  for (x = x0; x <= x0 + nx - 1; x++)
    {
      int left = x - (MAT_LOCAL_WINDOW_SIZE - 1)/2;
      int right = x + (MAT_LOCAL_WINDOW_SIZE - 1)/2;
      if (left < x0)
	{
	  left = x0;
	}
      if (right > x0 + nx - 1)
	{
	  right = x0 + nx - 1;
	}
      for (y = y0; y <= y0 + ny - 1; y++)
	{
	  int rejected, top, bottom;
	  double median, stdev, lower_limit, upper_limit;
	  double value = cpl_image_get(image, x, y, &rejected);
	  if (rejected) continue; /* this pixel is already marked as bad */
	  bottom = y - (MAT_LOCAL_WINDOW_SIZE - 1)/2;
	  top = y + (MAT_LOCAL_WINDOW_SIZE - 1)/2;
	  if (bottom < y0)
	    {
	      bottom = y0;
	    }
	  if (top > y0 + ny - 1)
	    {
	      top = y0 + ny - 1;
	    }
	  median = cpl_image_get_median_dev_window(image, left, bottom, right, top, &stdev);
	  lower_limit = median - stdevlimit*stdev;
	  upper_limit = median + stdevlimit*stdev;
	  if ((value < lower_limit) || (value > upper_limit))
	    {
	      count++;
	      /* cpl_msg_info(cpl_func, "pixel at %d, %d, %d added because %lf != [%lf .. %lf]", r, x, y, mat_frame_get(data, r, x, y), lower_limit, upper_limit); */
	      cpl_image_reject(image, x, y); /* do not use this pixel for the following calculations */
	      cpl_mask_set(bpm->list_badpixel[r], x, y, CPL_BINARY_1);
	    }
	}
    }
  cpl_msg_info("mat_badpixel_detect_local_window", "nbad=%d", count);
  return CPL_ERROR_NONE;
}

/**
   @ingroup bpm
   @brief Detect bad pixel inside an area using a local criteria.
   @param data This data structure contains the statistical values (median or variance) for a detector.
   @param stdevlimit Defines the standard deviation limit used for the detecting bad pixels.
   @param bpm This data structure contains the current bad pixel map for the same detector.
   @param r          the sub-window/region (starting with 0)
   @param x0         left side of the area inside a sub-window (starting with 1)
   @param y0         bottom side of the area inside a sub-window (starting with 1)
   @param nx         width of the area in pixels
   @param ny         height of the area in pixels
   @param intrange   a map containing the intensity range
   @return the cpl_error_code or CPL_ERROR_NONE

   This function uses the values in the input frame (data) to calculate their
   median and standard deviation for an area (parameters x0, y0, xn and yn) inside
   a specific sub-window (parameter r). Using the stdevlimit parameter, the range
   for the good pixels inside that area is calculated. All pixels having values
   outside this range are defined as bad pixels. For calculating the range, the
   pixels already marked as bad pixels are ignored.

   This function is needed, because the HAWAII-2RG and Aquarius detectors are
   organized as several independent detector channels (32 for the HAWAII-2RG
   and 64 for the Aquarius). The calibration measurement covers the whole
   detector and stores the data in one sub-window. Because the pixels of different
   detector channels may behave slightly different, the range for the valid pixels
   must be calculated for each detector channel and used only for the pixels inside
   that channel. This means, that normally r is equal to 0 (only one sub-window in
   the input data) and x0, y0, nx and ny defines exactly one detector channel.

   Mathematical description:

<pre>BEGIN
   FOR EACH pixel inside data[r, ...] DO
      Calculate the local median and stdev for the data[r, pixel];
      lower_limit := median - stdevlimit*stdev;
      upper_limit := median + stdevlimit*stdev;
      IF (data[r, pixel] < lower_limit) OR (data[r, pixel] > upper_limit) THEN
         bpm[r, pixel] := is bad;
      ENDIF
   ENDFOR
END</pre>
*/
cpl_error_code mat_badpixel_detect_local_window_range(mat_frame *data,
						      double stdevlimit,
						      mat_badpixel *bpm,
						      int r,
						      int x0,
						      int y0,
						      int nx,
						      int ny,
						      mat_frame *intrange)
{
  mat_imgreg *region = NULL;
  cpl_image  *image  = NULL;
  int         x, y, count;

  if (data == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_frame (data) argument given");
    }
  if (bpm == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_badpixel (bpm) argument given");
    }
  if ((r < 0) || (r >= data->nbsubwin))
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_ACCESS_OUT_OF_RANGE, "region %d outside of [0 .. %d]", r, data->nbsubwin - 1);
    }
  region = data->list_subwin[r];
  image = region->imgreg[0]; /* exactly one cpl_image for each region! */
  if (nx == -1)
    {
      nx = cpl_image_get_size_x(image);
    }
  if (ny == -1)
    {
      ny = cpl_image_get_size_y(image);
    }
  cpl_image_reject_from_mask(image, bpm->list_badpixel[r]); /* ignore the already found bad pixels */
  count = 0;
  for (x = x0; x <= x0 + nx - 1; x++)
    {
      int left = x - (MAT_LOCAL_WINDOW_SIZE - 1)/2;
      int right = x + (MAT_LOCAL_WINDOW_SIZE - 1)/2;
      if (left < x0)
	{
	  left = x0;
	}
      if (right > x0 + nx - 1)
	{
	  right = x0 + nx - 1;
	}
      for (y = y0; y <= y0 + ny - 1; y++)
	{
	  int rejected, top, bottom;
	  double median, stdev, lower_limit, upper_limit;
	  double value = cpl_image_get(image, x, y, &rejected);
	  if (rejected) continue; /* this pixel is already marked as bad */
	  if (mat_frame_get(intrange, r, x, y) == MAT_SATURATED_INTENSITY) continue; /* ignore saturated pixels */
	  bottom = y - (MAT_LOCAL_WINDOW_SIZE - 1)/2;
	  top = y + (MAT_LOCAL_WINDOW_SIZE - 1)/2;
	  if (bottom < y0)
	    {
	      bottom = y0;
	    }
	  if (top > y0 + ny - 1)
	    {
	      top = y0 + ny - 1;
	    }
	  median = cpl_image_get_median_dev_window(image, left, bottom, right, top, &stdev);
	  lower_limit = median - stdevlimit*stdev;
	  upper_limit = median + stdevlimit*stdev;
	  if ((value < lower_limit) || (value > upper_limit))
	    {
	      count++;
	      /* cpl_msg_info(cpl_func, "pixel at %d, %d, %d added because %lf != [%lf .. %lf]", r, x, y, mat_frame_get(data, r, x, y), lower_limit, upper_limit); */
	      cpl_image_reject(image, x, y); /* do not use this pixel for the following calculations */
	      cpl_mask_set(bpm->list_badpixel[r], x, y, CPL_BINARY_1);
	    }
	}
    }
  cpl_msg_info("mat_badpixel_detect_local_window", "nbad=%d", count);
  return CPL_ERROR_NONE;
}

static int mat_compare_double(const void *e1, const void *e2)
{
  const double *d1 = (const double *)e1;
  const double *d2 = (const double *)e2;
  if (*d1 < *d2) return -1;
  if (*d1 > *d2) return +1;
  return 0;
}

cpl_error_code mat_badpixel_detect_line(mat_frame *data,
					double stdevlimit,
					mat_badpixel *bpm,
					int r,
					int x0,
					int y0,
					int nx,
					int ny)
{
  mat_imgreg *region = NULL;
  cpl_image  *image  = NULL;
  int         x, y;
  int         count = 0;
  int         npix, i;
  double     *values;

  if (data == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_frame (data) argument given");
    }
  if (bpm == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_badpixel (bpm) argument given");
    }
  if ((r < 0) || (r >= data->nbsubwin))
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_ACCESS_OUT_OF_RANGE, "region %d outside of [0 .. %d]", r, data->nbsubwin - 1);
    }
  region = data->list_subwin[r];
  image = region->imgreg[0]; /* exactly one cpl_image for each region! */
  if (nx == -1)
    {
      nx = cpl_image_get_size_x(image);
    }
  if (ny == -1)
    {
      ny = cpl_image_get_size_y(image);
    }
  values = cpl_malloc(nx*sizeof(double));
  if (values == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "cannot allocate memory for %d intensity values", nx);
    }
  cpl_image_reject_from_mask(image, bpm->list_badpixel[r]); /* ignore the already found bad pixels */
  for (y = y0; y <= y0 + ny - 1; y++)
    {
      double mean = 0.0;
      double var  = 0.0;
      double lower_limit, upper_limit;
      // First copy all pixel intensities (except the bad pixels) from one line into a local array
      npix = 0;
      for (x = x0; x <= x0 + nx - 1; x++)
	{
	  int rejected;
	  double intensity = cpl_image_get(image, x, y, &rejected);
	  if (!rejected)
	    {
	      values[npix++] = intensity;
	    }
	}
      // Sort the pixel intensities
      qsort(values, npix, sizeof(double), mat_compare_double);
      // calculate the mean and variance using values[4] .. values[count - 4]
      for (i = 4; i <= npix - 4; i++)
	{
	  mean += values[i];
	}
      mean /= (double)(npix - 8);
      for (i = 4; i <= npix - 4; i++)
	{
	  var += (mean - values[i])*(mean - values[i]);
	}
      var /= (double)(npix - 9);
      // calculating the lower and upper limit is straightforward
      lower_limit = mean - stdevlimit*sqrt(var);
      upper_limit = mean + stdevlimit*sqrt(var);
      // And now detect the bad pixels in this pixel line
      for (x = x0; x <= x0 + nx - 1; x++)
	{
	  int rejected;
	  double intensity = cpl_image_get(image, x, y, &rejected);
	  if (rejected) continue;
	  if ((intensity < lower_limit) || (intensity > upper_limit))
	    {
	      count++;
	      /* cpl_msg_info(cpl_func, "pixel at %d, %d, %d added because %lf != [%lf .. %lf]", r, x, y, mat_frame_get(data, r, x, y), lower_limit, upper_limit); */
	      cpl_image_reject(image, x, y); /* do not use this pixel for the following calculations */
	      cpl_mask_set(bpm->list_badpixel[r], x, y, CPL_BINARY_1);
	    }
	}
    }
  cpl_free(values);
  cpl_msg_info("mat_badpixel_detect_line", "nbad=%d", count);
  return CPL_ERROR_NONE;
}
