/* $Id: mat_badpixel.h,v0.5 2014-06-15 12:56:21 mheininger Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: mheininger $
 * $Date: 2014/05/21 11:07:00 $
 * $Revision: 0.5 $
 * $Name: mat_badpixel.h $
 */

#ifndef MAT_BADPIXEL_H
#define MAT_BADPIXEL_H

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include <cpl.h>
#include "mat_detector.h"
#include "mat_imagingdetector.h"

/*-----------------------------------------------------------------------------
                                   Define
 -----------------------------------------------------------------------------*/

/**
   @ingroup bpm
   @brief Structure containing the static badpixel map of a detector.

   This structure represents all contents in a BADPIXEL FITS file. This includes
   the primary header keywords (keywords member), the sub-window setup (imgdet member),
   the number of good and bad pixels (nbtotal, nbgood, nbbad and goodpixelratio members)
   and the badpixel map itself.

   After loading a badpixel map from file, the map covers the whole detector. This
   map can be mapped to a certain sub-window setup where each sub-window is
   covered by an individual badpixel map as a mask in the list_badpixel member.
*/
typedef struct {
  mat_detector         *det;            /*!< The detector specification for this badpixel map. */
  cpl_propertylist     *keywords;       /*!< The keywords from the primary header. */
  mat_imagingdetector  *imgdet;         /*!< The sub-window setup defined by the contents of the IMAGING_DETECTOR binary table. */
  int                   nbtotal;        /*!< Number of total pixels. */
  int                   nbgood;         /*!< Number of good pixels (QC1 parameter QC DETi GOODPIX). */
  int                   nbbad;          /*!< Number of bad pixels (QC1 parameter QC DETi BADPIX). */
  double                goodpixelratio; /*!< The ratio good over total pixels (QC1 parameter QC DETi GOODPIXELRATIO). */
  cpl_mask            **list_badpixel;  /*!< Vector containing the badpixel maps for each region. Dimensions given by imgdet->nbregion. */
} mat_badpixel;

/*-----------------------------------------------------------------------------
                                   Functions prototypes
 -----------------------------------------------------------------------------*/

mat_badpixel *mat_badpixel_new(mat_detector *det, mat_imagingdetector *imgdet);

cpl_error_code mat_badpixel_delete(mat_badpixel *bpm);

cpl_error_code mat_badpixel_reset(mat_badpixel *bpm);

cpl_binary  mat_badpixel_get(mat_badpixel *bpm, int r, int x, int y);

cpl_error_code mat_badpixel_save(mat_badpixel *bpm,
				 const char *fname,
				 const char *rname,
				 cpl_parameterlist *parlist,
				 cpl_frameset *frameset);

mat_badpixel *mat_badpixel_load(cpl_frame *bpmframe);

mat_badpixel *mat_badpixel_map(mat_badpixel *bpm, mat_imagingdetector *imgdet);

cpl_error_code mat_badpixel_detect_global(mat_frame *data,
					  double stdevlimit,
					  mat_badpixel *bpm);
cpl_error_code mat_badpixel_detect_global_window(mat_frame *data,
						 double stdevlimit,
						 mat_badpixel *bpm,
						 int r,
						 int x0,
						 int y0,
						 int nx,
						 int ny);

cpl_error_code mat_badpixel_detect_local(mat_frame *data,
					 double stdevlimit,
					 mat_badpixel *bpm);

cpl_error_code mat_badpixel_detect_local_window(mat_frame *data,
						double stdevlimit,
						mat_badpixel *bpm,
						int r,
						int x0,
						int y0,
						int nx,
						int ny);

cpl_error_code mat_badpixel_detect_local_window_range(mat_frame *data,
						      double stdevlimit,
						      mat_badpixel *bpm,
						      int r,
						      int x0,
						      int y0,
						      int nx,
						      int ny,
						      mat_frame *intrange);


cpl_error_code mat_badpixel_detect_line(mat_frame *data,
					double stdevlimit,
					mat_badpixel *bpm,
					int r,
					int x0,
					int y0,
					int nx,
					int ny);

#endif
