/* $Id: mat_cal_image_lib.c,v0.5 2014-06-15 12:56:21 pberio Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: pberio $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_cal_image_lib.c $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#include <string.h>
#endif

/*-----------------------------------------------------------------------------
  Includes
  -----------------------------------------------------------------------------*/

#include "mat_const.h"
#include "mat_cal_image_lib.h"
#include "mat_error.h"

/*-----------------------------------------------------------------------------
  Define
  -----------------------------------------------------------------------------*/
#define MATISSE_DO_OFM         "OBS_FLATFIELD"
#define MATISSE_DO_FFM         "FLATFIELD"
#define MATISSE_DO_BADPIXEL    "BADPIX"
#define MATISSE_DO_NLM         "NONLINEARITY"
#define MATISSE_DO_SHIFT       "SHIFT_MAP"
#define NB_HOT_DARK            10
#define NB_BCD                 2
#define MATISSE_NB_FRAME_SKIP_AQUARIUS 30
/*-----------------------------------------------------------------------------
  Functions prototypes
  -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/

typedef struct {
  cpl_frameset      *frameset;
  cpl_parameterlist *parlist;
  const char        *recname;
  int                use_defaults;
  int                compensate;
  double             gain;
  int                reduce;
  int                ioi_first;
  int                ioi_count;
  int                tartyp;
  int                excess_count_n;
  int                excess_count_lm;
  int                el_nbentries;
  int               *el_expno;
  int               *el_count;
  mat_badpixel      *bpm;
  mat_nonlinearity  *nlm;
  mat_flatfield     *ffm;
  mat_obsflat       *ofm;
  mat_shiftmap      *odm;
  cpl_frame         *hot_dark_frame[NB_HOT_DARK];
  mat_gendata       *hot_dark_data[NB_HOT_DARK];
  cpl_frame         *sky_frame[NB_BCD];
  mat_gendata       *sky_data[NB_BCD];
  int                nbsky;
  cpl_image        **list_sky;
  int                nbmean;
  int                nbttresult;
  mat_tartyp_result *ttresults;
} mat_cal_info;

static void mat_info_init(mat_cal_info *info)
{
  
  info->use_defaults = 0;
  info->compensate = NO_COMPENSATION;
  info->gain = 0.0;
  info->reduce = 0;
  info->el_nbentries = 0;
  info->el_expno = NULL;
  info->el_count = NULL;
  info->bpm = NULL;
  info->nlm = NULL;
  info->ffm = NULL;
  info->ofm = NULL;
  info->odm = NULL;
  for(int i=0;i<NB_HOT_DARK;++i) {
    info->hot_dark_frame[i] = NULL;
    info->hot_dark_data[i]  = NULL;
  }
  for(int i=0;i<NB_BCD;++i) {
    info->sky_frame[i] = NULL;
    info->sky_data[i]  = NULL;
  }
  info->nbsky = 0;
  info->list_sky = NULL;
  info->nbmean = 0;
  info->nbttresult = 0;
  info->ttresults = NULL;
}

static void mat_info_delete(mat_cal_info *info)
{
  int    i;

  if (info->el_expno != NULL)
    {
      cpl_free(info->el_expno);
      info->el_expno = NULL;
    }
  if (info->el_count != NULL)
    {
      cpl_free(info->el_count);
      info->el_count = NULL;
    }
  if (info->bpm != NULL)
    {
      mat_badpixel_delete(info->bpm);
      info->bpm = NULL;
    }
  if (info->nlm != NULL)
    {
      mat_nonlinearity_delete(info->nlm);
      info->nlm = NULL;
    }
  if (info->ffm != NULL)
    {
      mat_flatfield_delete(info->ffm);
      info->ffm = NULL;
    }
  if (info->ofm != NULL)
    {
      mat_obsflat_delete(info->ofm);
      info->ofm = NULL;
    }
  if (info->odm != NULL)
    {
      mat_shiftmap_free(info->odm);
      info->odm = NULL;
    }
  for(i=0; i<NB_HOT_DARK; i++) {
    if (info->hot_dark_data[i] != NULL)
      {
  	mat_gendata_delete(info->hot_dark_data[i]);
  	info->hot_dark_data[i] = NULL;
      }
  }
  for(i=0; i<NB_BCD; i++) {
    if (info->sky_data[i] != NULL)
      {
  	mat_gendata_delete(info->sky_data[i]);
  	info->sky_data[i] = NULL;
      }
  }
  if (info->list_sky != NULL)
    {
      for (i = 0; i < info->nbsky; i++)
  	{
  	  if (info->list_sky[i] != NULL)
  	    {
  	      cpl_image_delete(info->list_sky[i]);
  	      info->list_sky[i] = NULL;
  	    }
  	}
      cpl_free(info->list_sky);
      info->list_sky = NULL;
      info->nbsky = 0;
    }
  if (info->ttresults != NULL)
    {
      cpl_free(info->ttresults);
      info->ttresults = NULL;
    }
}

static void mat_get_parameters(mat_cal_info *info, cpl_parameterlist *parlist)
{
  cpl_parameter     *param;
  const char        *el;

  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_image.compensate");
  if (strcmp(cpl_parameter_get_string(param), "none") == 0) info->compensate = NO_COMPENSATION;
  if (strcmp(cpl_parameter_get_string(param), "all")  == 0) info->compensate = FULL_COMPENSATION;
  if (strcmp(cpl_parameter_get_string(param), "dd")  == 0) info->use_defaults = 1;
  if (strstr(cpl_parameter_get_string(param), "pb") != NULL) info->compensate |= PIXEL_BIAS_COMPENSATION;
  if (strstr(cpl_parameter_get_string(param), "gb") != NULL) info->compensate |= GLOBAL_BIAS_COMPENSATION;
  if (strstr(cpl_parameter_get_string(param), "cb") != NULL) info->compensate |= CHANNEL_BIAS_COMPENSATION;
  if (strstr(cpl_parameter_get_string(param), "rb") != NULL) info->compensate |= HORIZONTAL_BIAS_COMPENSATION;
  if (strstr(cpl_parameter_get_string(param), "ct") != NULL) info->compensate |= CROSSTALK_COMPENSATION;
  if (strstr(cpl_parameter_get_string(param), "rc") != NULL) info->compensate |= RAMP_COMPENSATION;
  if (strstr(cpl_parameter_get_string(param), "nl") != NULL) info->compensate |= NONLINEARITY_COMPENSATION;
  if (strstr(cpl_parameter_get_string(param), "df") != NULL) info->compensate |= DETECTOR_FLATFIELD_COMPENSATION;
  if (strstr(cpl_parameter_get_string(param), "if") != NULL) info->compensate |= INSTRUMENT_FLATFIELD_COMPENSATION;
  if (strstr(cpl_parameter_get_string(param), "bp") != NULL) info->compensate |= BADPIXEL_COMPENSATION;
  if (strstr(cpl_parameter_get_string(param), "od") != NULL) info->compensate |= DISTORTION_COMPENSATION;
  if (strstr(cpl_parameter_get_string(param), "el") != NULL) info->compensate |= CONVERSION_COMPENSATION;
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_image.gain");
  info->gain = cpl_parameter_get_double(param);
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_image.reduce");
  if (cpl_parameter_get_bool(param))
    info->reduce = 1;
  else
    info->reduce = 0;
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_image.ioi");
  mat_parameter_get_int_double(param, "ioi", &(info->ioi_first), &(info->ioi_count));
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_image.excess_count_list");
  el = cpl_parameter_get_string(param);
  if (strcmp(el, "none") != 0)
    { // we have a list of "ESO TPL EXPNO"/excess count pairs
      const char *hel;
      int         count = 0;
      info->el_nbentries = 0;
      hel = el;
      while (*hel != '\0')
	{
	  if (*hel == ',') count++; // count ','
	  hel++;
	}
      count = (count + 1)/2; // after the last number no comma is needed but allowed
      info->el_expno = (int*)cpl_calloc(count, sizeof(int));
      info->el_count = (int*)cpl_calloc(count, sizeof(int));
      if ((info->el_expno == NULL) || (info->el_count == NULL))
	{
	  cpl_msg_error(cpl_func, "cannot allocate memory for the excess count list");
	  // we can continue, but assume the excess count list is empty
	}
      else
	{
	  hel = el;
	  while (*hel != '\0')
	    {
	      char *hend;
	      int expno;
	      expno = (int)strtol(el, &hend, 10);
	      if (hend == hel) break; // not a number
	      hel = hend;             // continue at first non digit
	      if (*hel == ',') hel++; // skip a ',' character
	      count = (int)strtol(el, &hend, 10);
	      if (hend == hel) break; // not a number
	      hel = hend;             // continue at first non digit
	      if (*hel == ',') hel++; // skip a ',' character
	      info->el_expno[info->el_nbentries] = expno;
	      info->el_count[info->el_nbentries] = count;
	      info->el_nbentries++;
	    }
	}
    }
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_image.tartyp");
  info->tartyp = cpl_parameter_get_int(param);
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_image.excess_count_n");
  info->excess_count_n = cpl_parameter_get_int(param);
  param = cpl_parameterlist_find(parlist, "matisse.mat_cal_image.excess_count_lm");
  info->excess_count_lm = cpl_parameter_get_int(param);
  
  cpl_msg_info(cpl_func, "compensations: %d", info->compensate);
  cpl_msg_info(cpl_func, "gain: %g", info->gain);
  cpl_msg_info(cpl_func, "reduce: %d", info->reduce);
  cpl_msg_info(cpl_func, "ioi: %d %d", info->ioi_first, info->ioi_count);
  cpl_msg_info(cpl_func, "tartyp: %d", info->tartyp);
  cpl_msg_info(cpl_func, "excess_count: %d %d", info->excess_count_lm, info->excess_count_n);
}

static cpl_error_code mat_load_bpm(mat_cal_info *info, cpl_frameset *frameset)
{
  int                count;

  // load the static bad pixel map, it is optional, but should be provided when available
  count = cpl_frameset_count_tags(frameset, MATISSE_DO_BADPIXEL);
  if (count == 1)
    {
      cpl_frame *frame = cpl_frameset_find(frameset, MATISSE_DO_BADPIXEL);
      cpl_frame_set_group(frame, CPL_FRAME_GROUP_CALIB);
      info->bpm = mat_badpixel_load(frame);
      return CPL_ERROR_NONE;
    }
  else if (count != 0)
    {
      cpl_msg_error(cpl_func, "only zero or one badpixel map is allowed in the SOF");
      return CPL_ERROR_ILLEGAL_INPUT;
    }
  return CPL_ERROR_NONE;
}

static cpl_error_code mat_load_nlm(mat_cal_info *info, cpl_frameset *frameset)
{
  int                count;

  // load the static nonlinearity map, it is optional, but should be provided when available
  count = cpl_frameset_count_tags(frameset, MATISSE_DO_NLM);
  if (count == 1)
    {
      cpl_frame *frame = cpl_frameset_find(frameset, MATISSE_DO_NLM);
      cpl_frame_set_group(frame, CPL_FRAME_GROUP_CALIB);
      info->nlm = mat_nonlinearity_load(frame);

      return CPL_ERROR_NONE;
    }
  else if (count != 0)
    {
      cpl_msg_error(cpl_func, "only zero or one nonlinearity map is allowed in the SOF");
      return CPL_ERROR_ILLEGAL_INPUT;
    }
  return CPL_ERROR_NONE;
}

static cpl_error_code mat_load_ffm(mat_cal_info *info, cpl_frameset *frameset)
{
  int                count;

  // load the static flatfield map, it is optional
  count = cpl_frameset_count_tags(frameset, MATISSE_DO_FFM);
  if (count == 1)
    {
      cpl_frame *frame = cpl_frameset_find(frameset, MATISSE_DO_FFM); // already checked previously!
      cpl_frame_set_group(frame, CPL_FRAME_GROUP_CALIB);
      info->ffm = mat_flatfield_load(frame);
      if ((info->ffm != NULL) && (info->gain == 0.0))
	{
	  info->gain = info->ffm->detectorgain;
	}
      return CPL_ERROR_NONE;
    }
  else if (count != 0)
    {
      cpl_msg_error(cpl_func, "only one static flatfield map is allowed in the SOF");
      return CPL_ERROR_ILLEGAL_INPUT;
    }
  return CPL_ERROR_NONE;
}

static cpl_error_code mat_load_ofm(mat_cal_info *info, cpl_frameset *frameset)
{
  int                count;

  // load the observation specific flatfield map, it is optional
  count = cpl_frameset_count_tags(frameset, MATISSE_DO_OFM);
  if (count == 1)
    {
      cpl_frame *frame = cpl_frameset_find(frameset, MATISSE_DO_OFM); // already checked previously!
      cpl_frame_set_group(frame, CPL_FRAME_GROUP_CALIB);
      info->ofm = mat_obsflat_load(frame);
      if ((info->ofm != NULL) && (info->gain == 0.0))
	{
	  info->gain = info->ofm->detectorgain;
	}
      return CPL_ERROR_NONE;
    }
  else if (count != 0)
    {
      cpl_msg_error(cpl_func, "only one instrument flatfield map is allowed in the SOF");
      return CPL_ERROR_ILLEGAL_INPUT;
    }
  return CPL_ERROR_NONE;
}

static cpl_error_code mat_load_odm(mat_cal_info *info, cpl_frameset *frameset)
{
  int                count;
  // load the shift map, it is optional, but should be provided when available
  count = cpl_frameset_count_tags(frameset, MATISSE_DO_SHIFT);
  if (count == 1)
    {
      cpl_frame *frame = cpl_frameset_find(frameset, MATISSE_DO_SHIFT);
      cpl_frame_set_group(frame, CPL_FRAME_GROUP_CALIB);
      info->odm = mat_shiftmap_load(frame);
      return CPL_ERROR_NONE;
    }
  else if (count != 0)
    {
      if (info->compensate & DISTORTION_COMPENSATION)
	{
	  cpl_msg_warning(cpl_func, "SHIFT_MAP missing");
	  return CPL_ERROR_INCOMPATIBLE_INPUT;
	}
      else
	{
	  cpl_msg_error(cpl_func, "only zero or one shift map is allowed in the SOF");
	  return CPL_ERROR_INCOMPATIBLE_INPUT;
	}
    }
  return CPL_ERROR_NONE;
}

static mat_gendata *mat_load_and_calibrate(mat_cal_info *info, cpl_frame *frame)
{
  mat_gendata       *rawdata = NULL; // raw data and 1. calibration step (static) due to inplace calibration (including reduce)
  mat_gendata       *caldata = NULL; // 2. calibration step (shift)
  mat_detector       det;
  int                compensate;
  int                skip_count;

  cpl_msg_info(cpl_func, "load and calibrate %s (%d, %d)",
  	       cpl_frame_get_filename(frame), info->ioi_first, info->ioi_count);
  char *detName=NULL;
  cpl_propertylist *header=NULL;
  header=cpl_propertylist_load(cpl_frame_get_filename(frame),0);
  detName=(char *)cpl_propertylist_get_string(header,"ESO DET CHIP NAME");
  if (detName == NULL) {
    cpl_msg_error(cpl_func,"no ESO DET CHIP NAME keyword in frame");
    return NULL;
  }
  
  
  if (info->ioi_count == 0)
    {
      if(strcmp(detName, "AQUARIUS") == 0 && strcmp(cpl_frame_get_tag(frame), "SKY_RAW") != 0 )
	{
	  rawdata = mat_gendata_load_skip(frame, MATISSE_NB_FRAME_SKIP_AQUARIUS, CPL_TYPE_FLOAT);
	  skip_count = MATISSE_NB_FRAME_SKIP_AQUARIUS;
	}
      else
	{
	  rawdata = mat_gendata_load(frame, CPL_TYPE_FLOAT);
	  skip_count = 0;
	}
    }
  else
    {
      rawdata = mat_gendata_load_window(frame, info->ioi_first, info->ioi_count, CPL_TYPE_FLOAT);
      skip_count = info->ioi_first;
    }
  cpl_propertylist_delete(header);

  if (rawdata == NULL)
    {
      cpl_msg_error(cpl_func,"could not load data from file %s", cpl_frame_get_filename(frame));
      return NULL;
    }
  //mat_tartyp_check(rawdata, info->tartyp);
  mat_detector_decode(&det, rawdata->keywords);
  if (info->use_defaults)
    {
      compensate = NO_COMPENSATION;
      if (info->ofm != NULL) compensate |= PIXEL_BIAS_COMPENSATION | CHANNEL_BIAS_COMPENSATION | HORIZONTAL_BIAS_COMPENSATION;
      if (info->bpm != NULL) compensate |= BADPIXEL_COMPENSATION;
      if (info->nlm != NULL) compensate |= NONLINEARITY_COMPENSATION;
      if (info->odm != NULL) compensate |= DISTORTION_COMPENSATION;
      if ((info->ofm != NULL) || (info->ffm != NULL) || (info->gain != 0.0)) compensate |= CONVERSION_COMPENSATION;
      if ((det.type == MAT_AQUARIUS_DET) && (info->ofm != NULL) && (info->bpm != NULL)) compensate |= CROSSTALK_COMPENSATION;
      if ((info->ofm != NULL) && (info->ffm == NULL)) compensate |= INSTRUMENT_FLATFIELD_COMPENSATION;
      if (info->ffm != NULL) compensate |= DETECTOR_FLATFIELD_COMPENSATION;
      compensate |= info->compensate;
    }
  else
    {
      compensate = info->compensate;
    }
  if (mat_calibration_detector(rawdata, info->bpm, info->ffm, info->nlm, info->ofm, NULL, compensate, info->reduce, 1) == NULL)
    {
      cpl_msg_error(cpl_func,"could not calibrate raw data");
      mat_gendata_delete(rawdata);
      return NULL;
    }
  // analyze the TARTYP and store the result for chopped exposures in info->ttresult[]
  if (info->el_nbentries == 0)
    {
      switch (det.type)
	{
	case MAT_HAWAII2RG_DET:
	  mat_tartyp_check(rawdata, info->tartyp, info->excess_count_lm, skip_count, &(info->ttresults[info->nbttresult]));
	  if (info->ttresults[info->nbttresult].chopfreq != 0.0) info->nbttresult++; // this frame uses chopping
	  break;
	case MAT_AQUARIUS_DET:
	  mat_tartyp_check(rawdata, info->tartyp, info->excess_count_n, skip_count, &(info->ttresults[info->nbttresult]));
	  if (info->ttresults[info->nbttresult].chopfreq != 0.0) info->nbttresult++; // this frame uses chopping
	  break;
	default:
	  mat_tartyp_check(rawdata, info->tartyp, 0, skip_count, NULL);
	}
    }
  else
    {
      int i, expno, excess_count = 0;
      expno = mat_propertylist_get_int_default(rawdata->keywords, "ESO TPL EXPNO", 0);
      for (i = 0; i < info->el_nbentries; i++)
	{
	  if (info->el_expno[i] != expno) continue;
	  excess_count = info->el_count[i];
	  break;
	}
      mat_tartyp_check(rawdata, info->tartyp, excess_count, skip_count, &(info->ttresults[info->nbttresult]));
      if (info->ttresults[info->nbttresult].chopfreq != 0.0) info->nbttresult++; // this frame uses chopping
    }
  if ((compensate & DISTORTION_COMPENSATION) && (info->odm != NULL))
    {
      cpl_msg_info(cpl_func,"Applying distortion compensation");
      caldata = mat_apply_shift(rawdata, info->odm, 0);
      if (caldata == NULL)
  	{
  	  cpl_msg_error(cpl_func, "could not apply SHIFT_MAP");
  	  mat_gendata_delete(rawdata);
  	  return NULL;
  	}
      mat_gendata_delete(rawdata);
      rawdata = caldata;
    }
  if (compensate & CONVERSION_COMPENSATION)
    {
      int              i;
      mat_imagingdata *imgdata = rawdata->imgdata;
      for (i = 0; i < imgdata->nbframe; i++)
	{
	  mat_frame_multiply(imgdata->list_frame[i], info->gain);
	}
    }
  cpl_msg_info(cpl_func, "mat_load_and_calibrate finished");
  return rawdata;
}

static int mat_get_index_for_hot_dark(cpl_frame *frame)
{
  /* int index=0; */
  char *filename=NULL;
  cpl_propertylist *plist=NULL;
  char *chipname=NULL;
  char *bcdstatus=NULL;
  int shut1=0;
  int shut2=0;
  int shut3=0;
  int shut4=0;
  int index_local=-10;

  filename = (char *)cpl_frame_get_filename(frame);
  plist = cpl_propertylist_load(filename,0);
  chipname=(char *)cpl_propertylist_get_string(plist,"ESO DET CHIP NAME");
  if (cpl_propertylist_has(plist,"ESO INS BCD1 ID")) {
    bcdstatus = (char *)cpl_propertylist_get_string(plist,"ESO INS BCD1 ID");
  } else {
    cpl_msg_error(cpl_func, "No BCD keywords found");
    return -1;
  }

  if (strcmp(chipname, "AQUARIUS") == 0) {
    shut1 = cpl_propertylist_get_bool(plist,"ESO INS BSN1 ST");
    shut2 = cpl_propertylist_get_bool(plist,"ESO INS BSN2 ST");
    shut3 = cpl_propertylist_get_bool(plist,"ESO INS BSN3 ST");
    shut4 = cpl_propertylist_get_bool(plist,"ESO INS BSN4 ST");
  } else {
    shut1 = cpl_propertylist_get_bool(plist,"ESO INS BSL1 ST");
    shut2 = cpl_propertylist_get_bool(plist,"ESO INS BSL2 ST");
    shut3 = cpl_propertylist_get_bool(plist,"ESO INS BSL3 ST");
    shut4 = cpl_propertylist_get_bool(plist,"ESO INS BSL4 ST");
  }
  /* printf("%s %s %d %d %d %d\n",filename,bcdstatus,shut1,shut2,shut3,shut4); */
  if ( shut1==1 && shut2==1 && shut3==1 && shut4==1 ) {
    index_local=0;
  }
  if ( shut1==1 && shut2==0 && shut3==0 && shut4==0 ) {
    index_local=1;
  }
  if ( shut1==0 && shut2==1 && shut3==0 && shut4==0 ) {
    index_local=2;
  }
  if ( shut1==0 && shut2==0 && shut3==1 && shut4==0 ) {
    index_local=3;
  }
  if ( shut1==0 && shut2==0 && shut3==0 && shut4==1  ) {
    index_local=4;
  }
  if (strstr(bcdstatus, "OUT") != NULL) {
    index_local+=5;
  }
  cpl_propertylist_delete(plist);
  return index_local;
}

static int mat_get_index_for_sky(cpl_frame *frame)
{
  char *filename=NULL;
  cpl_propertylist *plist=NULL;
  char *bcdstatus=NULL;
  int index_local=-1;

  filename = (char *)cpl_frame_get_filename(frame);
  plist = cpl_propertylist_load(filename,0);
  if (cpl_propertylist_has(plist,"ESO INS BCD1 ID")) {
    bcdstatus = (char *)cpl_propertylist_get_string(plist,"ESO INS BCD1 ID");
  } else {
    cpl_msg_error(cpl_func, "No BCD keywords found");
    return -1;
  }

  if (strcmp(bcdstatus, "IN") == 0) {
    index_local=0;
  } 
  if (strcmp(bcdstatus, "OUT") == 0) {
    index_local=1;
  } 
  cpl_propertylist_delete(plist);
  return index_local;
}

static cpl_error_code mat_proc_hot_dark(mat_cal_info *info, cpl_frame *frame)
{
  int index_local=mat_get_index_for_hot_dark(frame);
  if (index_local <0) {
    return CPL_ERROR_UNSPECIFIED;
  }

  if (info->hot_dark_data[index_local] != NULL)
    {
      cpl_msg_warning(cpl_func, "Several HOT_DARK frames found for this configuration (index=%d)",index_local);
      return CPL_ERROR_INCOMPATIBLE_INPUT;
    } 
  info->hot_dark_frame[index_local] = frame;
  info->hot_dark_data[index_local] = mat_load_and_calibrate(info, info->hot_dark_frame[index_local]);
  if (info->hot_dark_data[index_local] != NULL)
    return CPL_ERROR_NONE;
  else
    return CPL_ERROR_UNSPECIFIED;
}

static cpl_error_code mat_proc_sky(mat_cal_info *info, cpl_frame *frame)
{
  //mat_gendata *calsky;
  int          i, j;
  int          start;
  
  int index_local=mat_get_index_for_sky(frame);
  //  printf("%d\n",index_local);
  if (index_local <0) {
    return CPL_ERROR_UNSPECIFIED;
  }
  if (info->sky_data[index_local] == NULL) {
    info->sky_frame[index_local] = frame;
    info->sky_data[index_local] = mat_load_and_calibrate(info, info->sky_frame[index_local]);
    /* if (info->sky_data[index_local] != NULL) */
    /*   return CPL_ERROR_NONE; */
    /* else */
    /*   return CPL_ERROR_UNSPECIFIED; */
  }
  if (info->sky_data[index_local] == NULL)
    {
      cpl_msg_error(cpl_func,"could not load and calibrate sky data");
      return CPL_ERROR_UNSPECIFIED;
    }
  if (info->nbsky == 0)
    { // it is the first sky frame
      // -> create the images for the mean sky from the first data frame
      info->nbsky = info->sky_data[index_local]->imgdet->nbregion;
      info->list_sky = cpl_calloc(info->nbsky, sizeof(cpl_image *));
      for (j = 0; j < info->nbsky; j++)
	{
	  info->list_sky[j] = cpl_image_duplicate(info->sky_data[index_local]->imgdata->list_frame[0]->list_subwin[j]->imgreg[0]);
	}
      start = 1;
    }
  else
    { // we have already a previous mean sky (not scaled)
      start = 0;
    }
  info->nbmean += info->sky_data[index_local]->imgdata->nbframe;
  for (i = start; i < info->sky_data[index_local]->imgdata->nbframe; i++)
    {
      for (j = 0; j < info->nbsky; j++)
	{
	  cpl_image_add(info->list_sky[j], info->sky_data[index_local]->imgdata->list_frame[i]->list_subwin[j]->imgreg[0]);
	}
    }
  return CPL_ERROR_NONE;
}

static void mat_update_sky(mat_cal_info *info)
{
  int   j;

  if (info->list_sky == NULL) return;
  for (j = 0; j < info->nbsky; j++)
    {
      cpl_image_divide_scalar(info->list_sky[j], info->nbmean);
    }
  cpl_msg_info(cpl_func, "Mean Sky computed");
}

static void mat_compensate_sky(mat_cal_info *info, mat_gendata *data)
{
  int   i, j;
  if (info->list_sky == NULL) return;
  // subract the mean sky from the already calibrated data
  for (i = 0; i < data->imgdata->nbframe; i++)
    {
      for (j = 0; j < info->nbsky; j++)
	{
	  cpl_image_subtract(data->imgdata->list_frame[i]->list_subwin[j]->imgreg[0], info->list_sky[j]);
	}
    }
}

static int mat_compare_tartyp_result(const void *r1, const void *r2)
{
  const mat_tartyp_result  *hr1 = (mat_tartyp_result *)r1;
  const mat_tartyp_result  *hr2 = (mat_tartyp_result *)r2;

  if (hr1->delay < hr2->delay) return -1;
  if (hr1->delay > hr2->delay) return +1;
  return 0;
}

static cpl_error_code mat_store_result(mat_cal_info      *info,
				       mat_gendata       *data,
				       cpl_frame         *frame,
				       const char        *pro_cat,
				       const char        *pro_tech,
				       const char        *pro_sci,
				       mat_gendata       *adata,
//				       cpl_frame         *aframe,
				       int               index_input)
{
  char         *output = NULL;
//  cpl_frame    *outframe;
  cpl_frame    *cur_frame;
  /* cpl_frameset *usedframes; */
  const char   *filenameCur = NULL;
//  const char   *tagCur = NULL;
  const char   *filename = NULL;
  cpl_frameset_iterator *it;
  cpl_frameset *selframes;
  cpl_size nb;
  int idxframe=0;
  cpl_frame **framecopy=NULL;
  





  // Save in a Fits file
  cpl_msg_info(cpl_func,"Creating FITS file...");
  // create the FITS file and save IMAGING_DATA
  /* cpl_frame_set_group(frame, CPL_FRAME_GROUP_RAW); */
  /* outframe = cpl_frame_duplicate(frame); */
 
  // Create selframes with the current frame in first position
  nb=cpl_frameset_get_size(info->frameset);
  framecopy=cpl_malloc(nb*sizeof(cpl_frame *));

  filename=cpl_frame_get_filename(frame);
  framecopy[idxframe]=cpl_frame_duplicate(frame);
  cpl_frame_set_group(framecopy[idxframe], CPL_FRAME_GROUP_RAW);
  selframes = cpl_frameset_new();
  cpl_frameset_insert(selframes,framecopy[idxframe]);
  idxframe++;

  it = cpl_frameset_iterator_new(info->frameset);
  cur_frame = cpl_frameset_iterator_get(it);
  while (cur_frame != NULL)
    {    
      filenameCur=cpl_frame_get_filename(cur_frame);
      if (strstr(filename,filenameCur) == NULL)
	{
	  if (strcmp(cpl_frame_get_tag(cur_frame), "TARGET_RAW") == 0 ||
	      strcmp(cpl_frame_get_tag(cur_frame), "CALIB_RAW") == 0 ||
	      strcmp(cpl_frame_get_tag(cur_frame), "CALIB_SRC_RAW") == 0 ||
	      strcmp(cpl_frame_get_tag(cur_frame), "HOT_DARK") == 0 ||
	      strcmp(cpl_frame_get_tag(cur_frame), "SKY_RAW") == 0)
	    {
	      framecopy[idxframe]=cpl_frame_duplicate(cur_frame);
	      cpl_frame_set_group(framecopy[idxframe], CPL_FRAME_GROUP_RAW);
	      cpl_frameset_insert(selframes, framecopy[idxframe]);  
	      idxframe++;
	    } 
	  if (strcmp(cpl_frame_get_tag(cur_frame), "BADPIX") == 0 ||
	      strcmp(cpl_frame_get_tag(cur_frame), "NONLINEARITY") == 0 ||
	      strcmp(cpl_frame_get_tag(cur_frame), "OBS_FLATFIELD") == 0 ||
	      strcmp(cpl_frame_get_tag(cur_frame), "SHIFT_MAP") == 0 ||
	      strcmp(cpl_frame_get_tag(cur_frame), "KAPPA_MATRIX") == 0)
	    {
	      framecopy[idxframe]=cpl_frame_duplicate(cur_frame);
	      cpl_frame_set_group(framecopy[idxframe], CPL_FRAME_GROUP_CALIB);
	      cpl_frameset_insert(selframes, framecopy[idxframe]);  
	      idxframe++;
	    }
	}
      cpl_frameset_iterator_advance(it, 1);
      cur_frame = cpl_frameset_iterator_get(it);
    }
  cpl_frameset_iterator_delete(it);

  // Save in a Fits file
  cpl_msg_info(cpl_func,"Creating FITS file...");
  // create the FITS file and save IMAGING_DATA
  /* cpl_frame_set_group(frame, CPL_FRAME_GROUP_RAW); */
  /* outframe = cpl_frame_duplicate(frame); */
  output = cpl_sprintf("%s_%04d.fits",pro_cat,index_input);
  // Add dispersion coefficients to the primary header
  if ((info->odm != NULL))
    {
      cpl_msg_info(cpl_func,"Adding Dispersion coefficients to primary header...");
      cpl_propertylist_append_double(data->keywords,"PRO DISP COEF0", info->odm->dispcoef[0]);
      cpl_propertylist_append_double(data->keywords,"PRO DISP COEF1", info->odm->dispcoef[1]);
      cpl_propertylist_append_double(data->keywords,"PRO DISP COEF2", info->odm->dispcoef[2]);
      cpl_propertylist_append_double(data->keywords,"PRO DISP COEF3", info->odm->dispcoef[3]);
      cpl_propertylist_append_double(data->keywords,"PRO DISP COEF4", info->odm->dispcoef[4]);
    }
  cpl_msg_info(cpl_func,"Saving file");
  /* usedframes = cpl_frameset_new(); */
  /* cpl_frameset_insert(usedframes, outframe); */
  /* if (aframe != NULL) */
  /*   { */
  /*     cpl_frame_set_group(aframe, CPL_FRAME_GROUP_RAW); */
  /*     cpl_frameset_insert(usedframes, cpl_frame_duplicate(aframe)); */
  /*   } */
  cpl_propertylist_erase(data->keywords,"RADECSYS");
  if (mat_gendata_save(data, info->frameset, selframes, info->parlist, info->recname,
		       pro_cat, pro_tech, pro_sci, output, CPL_TYPE_FLOAT) != CPL_ERROR_NONE) {
    cpl_free(framecopy);
    cpl_free(output);
    cpl_frameset_delete(selframes);
    return CPL_ERROR_UNSPECIFIED;
  }
  if (adata != NULL)
    {
      mat_gendata_append(adata, output, CPL_TYPE_FLOAT);
    }
  /* if (usedframes != NULL) cpl_frameset_delete(usedframes); */
  // Save IMAGING_DETECTOR and ARRAY
  mat_imagingdetector_save(data->imgdet, data->keywords, output);
  mat_array_save(data->array, data->keywords, output, 1);
  cpl_free(output);
  cpl_frameset_delete(selframes);
  cpl_free(framecopy);
  return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
   @brief Transform each raw frame in a calibrated frame (mat_cal_image)
   @param frameset           contains the frames
   @param parlist            contains the recipes parameters
   @param recname            Name of the recipes calling this function
   @return 0 if no error 
*/
int mat_cal_image_lib(cpl_frameset *frameset, 
		      cpl_parameterlist *parlist,
		      const char *recname)
{
  mat_cal_info           info;
  cpl_error_code         ec = CPL_ERROR_NONE;
  cpl_frameset_iterator *it;
  cpl_frame             *frame;
  cpl_errorstate         prestate = cpl_errorstate_get();
  int index_local=0;
  int indexFileCalib=1;
  int indexFileTarget=1;
  char *chipname=NULL;
  char *dprtype=NULL;
  cpl_propertylist *header=NULL;
  mat_gendata * sky_dataSelected =NULL;
  /* mat_gendata * sky_dataBinning =NULL; */
  mat_gendata * caldata = NULL;
  /* cpl_parameter *binning = NULL; */
  /* int nbbin=1; */
  /* double alpha,beta,gamma; */
  /* int c; */
  int i;
  
  mat_assert_value((frameset != NULL), CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT, "no frameset argument given");
  mat_info_init(&info);   // initialize the image calibration context
  info.frameset = frameset;
  info.parlist  = parlist;
  info.recname  = recname;
  info.ttresults = (mat_tartyp_result*)cpl_calloc(cpl_frameset_get_size(frameset), sizeof(mat_tartyp_result));
  if (info.ttresults == NULL)
    {
      cpl_msg_error(cpl_func, "Cannot create TARTYP processing results");
      mat_info_delete(&info);
      return -1;
    }

  /* binning=cpl_parameterlist_find(parlist,"matisse.mat_cal_image.binning"); */
  /* nbbin=cpl_parameter_get_int(binning); */

  mat_get_parameters(&info, parlist);
  if (ec == CPL_ERROR_NONE) ec = mat_load_bpm(&info, frameset);
  if (ec == CPL_ERROR_NONE) ec = mat_load_nlm(&info, frameset);
  if (ec == CPL_ERROR_NONE) ec = mat_load_ffm(&info, frameset);
  if (ec == CPL_ERROR_NONE) ec = mat_load_ofm(&info, frameset);
  if (ec == CPL_ERROR_NONE) ec = mat_load_odm(&info, frameset);

  // iterate over all HOT_DARK frames
  it = cpl_frameset_iterator_new(frameset);
  if (it == NULL)
    {
      cpl_msg_error(cpl_func, "Cannot create a frameset iterator");
      mat_info_delete(&info);
      return -1;
    }
  frame = cpl_frameset_iterator_get(it);
  while (frame != NULL)
    {
      if (strcmp(cpl_frame_get_tag(frame), "HOT_DARK") == 0)
  	{
  	  mat_proc_hot_dark(&info, frame);
  	}
      cpl_frameset_iterator_advance(it, 1);
      frame = cpl_frameset_iterator_get(it);
    }
  cpl_frameset_iterator_delete(it);
  // iterate over all SKY_RAW frames
  it = cpl_frameset_iterator_new(frameset);
  if (it == NULL)
    {
      cpl_msg_error(cpl_func, "Cannot create a frameset iterator");
      mat_info_delete(&info);
      return -1;
    }
  frame = cpl_frameset_iterator_get(it);
  while (frame != NULL)
    {
      if (strcmp(cpl_frame_get_tag(frame), "SKY_RAW") == 0)
  	{
  	  mat_proc_sky(&info, frame);
  	}
      cpl_frameset_iterator_advance(it, 1);
      frame = cpl_frameset_iterator_get(it);
    }
  cpl_frameset_iterator_delete(it);
  mat_update_sky(&info); 
  
  prestate=cpl_errorstate_get();
  // iterate over all raw data frames
  it = cpl_frameset_iterator_new(frameset);
  if (it == NULL)
    {
      cpl_msg_error(cpl_func, "Cannot create a frameset iterator");
      mat_info_delete(&info);
      return -1;
    }
  frame = cpl_frameset_iterator_get(it);
  while (frame != NULL)
    {
      if (strcmp(cpl_frame_get_tag(frame), "TARGET_RAW") == 0)
  	{
	  header = cpl_propertylist_load(cpl_frame_get_filename(frame),0);
	  chipname=(char *)cpl_propertylist_get_string(header,"ESO DET CHIP NAME");
	  if (chipname == NULL) {
	    cpl_msg_error(cpl_func,"No CHIP NAME keyword");
	    return -1;
	  }
	  dprtype=(char *)cpl_propertylist_get_string(header,"ESO DPR TYPE");
	  if (dprtype == NULL) {
	    cpl_msg_error(cpl_func,"No DPR TYPE keyword");
	    return -1;
	  }
	  
  	  caldata = mat_load_and_calibrate(&info, frame);
	  if (caldata != NULL)
	    {
	      index_local=mat_get_index_for_sky(frame);
	      if (index_local < 0) {
	  	return CPL_ERROR_UNSPECIFIED;
	      }
	      
	      if ( (strcmp(chipname, "AQUARIUS") == 0) &&
	  	   (mat_check_photometry(frame) == FALSE) &&
		   (info.odm != NULL) )
	  	{
	  	  cpl_vector * dispCoef = cpl_vector_new(N_DEG_DISPERSION_LAW+1);
	  	  cpl_vector_set(dispCoef,0,info.odm->dispcoef[0]);
	  	  cpl_vector_set(dispCoef,1,info.odm->dispcoef[1]);
	  	  cpl_vector_set(dispCoef,2,info.odm->dispcoef[2]);
	  	  cpl_vector_set(dispCoef,3,info.odm->dispcoef[3]);
	  	  cpl_vector_set(dispCoef,4,info.odm->dispcoef[4]);

	  	  mat_gendata * caldataSelected = mat_gendata_select_band(caldata,dispCoef);
	  	  if (info.sky_data[index_local] == NULL) {
	  	    sky_dataSelected = NULL;
	  	  } else {
		    mat_compensate_sky(&info, info.sky_data[index_local]);
	  	    sky_dataSelected = mat_gendata_select_band(info.sky_data[index_local],dispCoef);
		    // Copy STEPPING_PHASE and LOCAL_OPD from TARGET to SKY
		    int iFrameTarget=0;
		    for(int iFrame=0;iFrame<sky_dataSelected->imgdata->nbframe;iFrame++) {
		      iFrameTarget=iFrame % caldataSelected->imgdata->nbframe;
		      sky_dataSelected->imgdata->list_frame[iFrame]->stepphase=caldataSelected->imgdata->list_frame[iFrameTarget]->stepphase;
		      sky_dataSelected->imgdata->list_frame[iFrame]->localopd[0]=caldataSelected->imgdata->list_frame[iFrameTarget]->localopd[0];
		      sky_dataSelected->imgdata->list_frame[iFrame]->localopd[1]=caldataSelected->imgdata->list_frame[iFrameTarget]->localopd[1];
		      sky_dataSelected->imgdata->list_frame[iFrame]->localopd[2]=caldataSelected->imgdata->list_frame[iFrameTarget]->localopd[2];
		      sky_dataSelected->imgdata->list_frame[iFrame]->localopd[3]=caldataSelected->imgdata->list_frame[iFrameTarget]->localopd[3];
		    }
	  	  }
	  	  cpl_vector_delete(dispCoef);
		  /* if (nbbin == 1) */
		  /*   { */
		  if (caldataSelected != NULL)
		    {
		      //  Flag MATISSE data with GRA4MAT data
		      char *filename_gra=NULL;
		      filename_gra = mat_get_fname_gra(cpl_propertylist_get_int(caldataSelected->keywords,"ESO TPL EXPNO"), frameset);
		      if (filename_gra != NULL)
			{
			  cpl_msg_info(cpl_func,"Tagging MATISSE frames with GRA4MAT data1");
			  mat_flag_frames_N(caldataSelected,filename_gra,1);
			}
		      if (strstr(dprtype,"NOFRINGE") != NULL)
			{
			  mat_store_result(&info, caldataSelected, frame, "TARGET_CAL", "NOFRINGE", "T", sky_dataSelected,indexFileTarget);
			}
		      else
			{
			  mat_store_result(&info, caldataSelected, frame, "TARGET_CAL", "IMAGE", "T", sky_dataSelected,indexFileTarget);
			}
		      indexFileTarget++;
		      mat_gendata_delete(caldataSelected);
		    }
		  mat_gendata_delete(sky_dataSelected);
		  /*   } */
		  /* else */
		  /*   { */
		  /*     mat_gendata * caldataBinning = mat_gendata_binning(caldataSelected,nbbin); */
		  /*     mat_gendata_delete(caldataSelected); */
		  /*     if (sky_dataSelected == NULL) */
		  /* 	{ */
		  /* 	  sky_dataBinning = NULL; */
		  /* 	} */
		  /*     else */
		  /* 	{ */
		  /* 	  sky_dataBinning = mat_gendata_binning(sky_dataSelected,nbbin); */
		  /* 	} */
		  /*     mat_gendata_delete(sky_dataSelected); */
		  /*     if (caldataBinning != NULL) */
		  /* 	{ */
		  /* 	  alpha=info.odm->dispcoef[0]; */
		  /* 	  beta=info.odm->dispcoef[1]; */
		  /* 	  gamma=info.odm->dispcoef[2]; */
		  /* 	  c=caldataBinning->imgdet->list_region[0]->corner[1]; */
		  /* 	  info.odm->dispcoef[2]=gamma*nbbin*nbbin; */
		  /* 	  info.odm->dispcoef[1]=beta*nbbin+2*gamma*c*nbbin-2*c*info.odm->dispcoef[2]; */
		  /* 	  info.odm->dispcoef[0]=alpha+beta*c+gamma*c*c-info.odm->dispcoef[1]*c-info.odm->dispcoef[2]*c*c; */
		  /* 	  if (strstr(dprtype,"NOFRINGE") != NULL) */
		  /* 	    { */
		  /* 	      mat_store_result(&info, caldataBinning, frame, "TARGET_CAL", "NOFRINGE", "T", sky_dataBinning,indexFileTarget); */
		  /* 	    } */
		  /* 	  else */
		  /* 	    { */
		  /* 	      mat_store_result(&info, caldataBinning, frame, "TARGET_CAL", "IMAGE", "T", sky_dataBinning,indexFileTarget); */
		  /* 	    } */
		  /* 	  info.odm->dispcoef[0]=alpha; */
		  /* 	  info.odm->dispcoef[1]=beta; */
		  /* 	  info.odm->dispcoef[2]=gamma; */
			  
		  /* 	  indexFileTarget++; */
		  /* 	  mat_gendata_delete(caldataBinning); */
		  /* 	} */
		  /*     mat_gendata_delete(sky_dataBinning); */
		  /*   }    */
	  	}
	      
	      if ( (strcmp(chipname, "AQUARIUS") == 0) &&
	  	   (mat_check_photometry(frame) == TRUE) &&
		   (info.odm != NULL) )
	  	{
	  	  cpl_vector * dispCoef = cpl_vector_new(N_DEG_DISPERSION_LAW+1);
	  	  cpl_vector_set(dispCoef,0,info.odm->dispcoef[0]);
	  	  cpl_vector_set(dispCoef,1,info.odm->dispcoef[1]);
	  	  cpl_vector_set(dispCoef,2,info.odm->dispcoef[2]);
	  	  cpl_vector_set(dispCoef,3,info.odm->dispcoef[3]);
	  	  cpl_vector_set(dispCoef,4,info.odm->dispcoef[4]);

	  	  mat_gendata * caldataSelected = mat_gendata_select_band(caldata,dispCoef);
	  	  cpl_vector_delete(dispCoef);
		  /* if (nbbin == 1) */
		  /*   { */
		  if (caldataSelected != NULL)
		    {
		      //  Flag MATISSE data with GRA4MAT data
		      char *filename_gra=NULL;
		      filename_gra = mat_get_fname_gra(cpl_propertylist_get_int(caldataSelected->keywords,"ESO TPL EXPNO"), frameset);
		      if (filename_gra != NULL)
			{
			  cpl_msg_info(cpl_func,"Tagging MATISSE frames with GRA4MAT data2");
			  mat_flag_frames_N(caldataSelected,filename_gra,1);
			}
		      if (strstr(dprtype,"NOFRINGE") != NULL)
			{
			  mat_store_result(&info, caldataSelected, frame, "TARGET_CAL", "NOFRINGE", "T", NULL, indexFileTarget);
			}
		      else
			{
			  mat_store_result(&info, caldataSelected, frame, "TARGET_CAL", "IMAGE", "T", NULL, indexFileTarget);
			}
		      indexFileTarget++;
		      mat_gendata_delete(caldataSelected);
		    }
		  /*   } */
		  /* else */
		  /*   { */
		  /*     mat_gendata * caldataBinning = mat_gendata_binning(caldataSelected,nbbin); */
		  /*     mat_gendata_delete(caldataSelected); */
		  /*     if (caldataBinning != NULL) */
		  /* 	{ */
		  /* 	  alpha=info.odm->dispcoef[0]; */
		  /* 	  beta=info.odm->dispcoef[1]; */
		  /* 	  gamma=info.odm->dispcoef[2]; */
		  /* 	  c=caldataBinning->imgdet->list_region[0]->corner[1]; */
		  /* 	  info.odm->dispcoef[2]=gamma*nbbin*nbbin; */
		  /* 	  info.odm->dispcoef[1]=beta*nbbin+2*gamma*c*nbbin-2*c*info.odm->dispcoef[2]; */
		  /* 	  info.odm->dispcoef[0]=alpha+beta*c+gamma*c*c-info.odm->dispcoef[1]*c-info.odm->dispcoef[2]*c*c; */
		  /* 	  if (strstr(dprtype,"NOFRINGE") != NULL) */
		  /* 	    { */
		  /* 	      mat_store_result(&info, caldataBinning, frame, "TARGET_CAL", "NOFRINGE", "T", NULL, indexFileTarget); */
		  /* 	    } */
		  /* 	  else */
		  /* 	    { */
		  /* 	      mat_store_result(&info, caldataBinning, frame, "TARGET_CAL", "IMAGE", "T", NULL, indexFileTarget); */
		  /* 	    } */
		  /* 	  info.odm->dispcoef[0]=alpha; */
		  /* 	  info.odm->dispcoef[1]=beta; */
		  /* 	  info.odm->dispcoef[2]=gamma; */
			  
		  /* 	  indexFileTarget++; */
		  /* 	  mat_gendata_delete(caldataBinning); */
		  /* 	} */
		  /*   } */
	  	}
	      if ( (strcmp(chipname, "AQUARIUS") == 0) && (info.odm == NULL) )
	  	{
		  //  Flag MATISSE data with GRA4MAT data
		  char *filename_gra=NULL;
		  filename_gra = mat_get_fname_gra(cpl_propertylist_get_int(caldata->keywords,"ESO TPL EXPNO"), frameset);
		  if (filename_gra != NULL)
		    {
		      cpl_msg_info(cpl_func,"Tagging MATISSE frames with GRA4MAT data3");
		      mat_flag_frames_N(caldata,filename_gra,1);
		    }
		  if (strstr(dprtype,"NOFRINGE") != NULL)
		    {
		      mat_store_result(&info, caldata, frame, "TARGET_CAL", "NOFRINGE", "T", NULL, indexFileTarget);
		    }
		  else
		    {
		      mat_store_result(&info, caldata, frame, "TARGET_CAL", "IMAGE", "T", NULL, indexFileTarget);
		    }
		  
		  indexFileTarget++;
		}
	      if ( (strcmp(chipname, "HAWAII-2RG") == 0) && (info.odm != NULL) )
	  	{
   	  	  mat_compensate_sky(&info, caldata);
	  	  cpl_vector * dispCoef = cpl_vector_new(N_DEG_DISPERSION_LAW+1);
	  	  cpl_vector_set(dispCoef,0,info.odm->dispcoef[0]);
	  	  cpl_vector_set(dispCoef,1,info.odm->dispcoef[1]);
	  	  cpl_vector_set(dispCoef,2,info.odm->dispcoef[2]);
	  	  cpl_vector_set(dispCoef,3,info.odm->dispcoef[3]);
	  	  cpl_vector_set(dispCoef,4,info.odm->dispcoef[4]);

	  	  mat_gendata * caldataSelected = mat_gendata_select_band(caldata,dispCoef);
	  	  cpl_vector_delete(dispCoef);
		  if (caldataSelected != NULL)
		    {
		      //  Flag MATISSE data with GRA4MAT data
		      char *filename_gra=NULL;
		      filename_gra = mat_get_fname_gra(cpl_propertylist_get_int(caldataSelected->keywords,"ESO TPL EXPNO"), frameset);
		      if (filename_gra != NULL)
			{
			  cpl_msg_info(cpl_func,"Tagging MATISSE frames with GRA4MAT data4");
			  mat_flag_frames_L(caldataSelected,filename_gra,1);
			}
		      if (strstr(dprtype,"NOFRINGE") != NULL)
			{
			  mat_store_result(&info, caldataSelected, frame, "TARGET_CAL", "NOFRINGE", "T", NULL, indexFileTarget);
			}
		      else
			{
			  mat_store_result(&info, caldataSelected, frame, "TARGET_CAL", "IMAGE", "T", NULL, indexFileTarget);
			}
		      
		      indexFileTarget++;
		      mat_gendata_delete(caldataSelected);
		    }
	  	}
	      if ( (strcmp(chipname, "HAWAII-2RG") == 0) && (info.odm == NULL) )
	  	{
   	  	  mat_compensate_sky(&info, caldata);
		  //  Flag MATISSE data with GRA4MAT data
		  char *filename_gra=NULL;
		  filename_gra = mat_get_fname_gra(cpl_propertylist_get_int(caldata->keywords,"ESO TPL EXPNO"), frameset);
		  if (filename_gra != NULL)
		    {
		      cpl_msg_info(cpl_func,"Tagging MATISSE frames with GRA4MAT data5");
		      mat_flag_frames_L(caldata,filename_gra,1);
		    }
		  mat_store_result(&info, caldata, frame, "TARGET_CAL", "IMAGE", "T", NULL, indexFileTarget);
		  indexFileTarget++;
	  	}
	      mat_gendata_delete(caldata);

	    }
	  cpl_propertylist_delete(header);
  	}
      else if (strcmp(cpl_frame_get_tag(frame), "CALIB_RAW" ) == 0)
  	{
	  header = cpl_propertylist_load(cpl_frame_get_filename(frame),0);
	  chipname=(char *)cpl_propertylist_get_string(header,"ESO DET CHIP NAME");
	  if (chipname == NULL) {
	    cpl_msg_error(cpl_func,"No CHIP NAME keyword");
	    return -1;
	  }
	  dprtype=(char *)cpl_propertylist_get_string(header,"ESO DPR TYPE");
	  if (dprtype == NULL) {
	    cpl_msg_error(cpl_func,"No DPR TYPE keyword");
	    return -1;
	  }

  	  caldata = mat_load_and_calibrate(&info, frame);
	  if (caldata != NULL)
	    {
	      index_local=mat_get_index_for_sky(frame);
	      if (index_local < 0)
		{
		  return CPL_ERROR_UNSPECIFIED;
		}
	      
	      if ( (strcmp(chipname, "AQUARIUS") == 0) &&(mat_check_photometry(frame) == FALSE) && (info.odm != NULL) )
	  	{
	  	  cpl_vector * dispCoef = cpl_vector_new(N_DEG_DISPERSION_LAW+1);
	  	  cpl_vector_set(dispCoef,0,info.odm->dispcoef[0]);
	  	  cpl_vector_set(dispCoef,1,info.odm->dispcoef[1]);
	  	  cpl_vector_set(dispCoef,2,info.odm->dispcoef[2]);
	  	  cpl_vector_set(dispCoef,3,info.odm->dispcoef[3]);
	  	  cpl_vector_set(dispCoef,4,info.odm->dispcoef[4]);

	  	  mat_gendata * caldataSelected = mat_gendata_select_band(caldata,dispCoef);
	  	  if (info.sky_data[index_local] == NULL) {
	  	    sky_dataSelected = NULL;
	  	  } else {
		    mat_compensate_sky(&info, info.sky_data[index_local]);
	  	    sky_dataSelected = mat_gendata_select_band(info.sky_data[index_local],dispCoef);
	  	    // Copy STEPPING_PHASE and LOCAL_OPD from TARGET to SKY
	  	    int iFrameTarget=0;
	  	    for(int iFrame=0;iFrame<sky_dataSelected->imgdata->nbframe;iFrame++) {
	  	      iFrameTarget=iFrame % caldataSelected->imgdata->nbframe;
	  	      sky_dataSelected->imgdata->list_frame[iFrame]->stepphase=caldataSelected->imgdata->list_frame[iFrameTarget]->stepphase;
	  	      sky_dataSelected->imgdata->list_frame[iFrame]->localopd[0]=caldataSelected->imgdata->list_frame[iFrameTarget]->localopd[0];
	  	      sky_dataSelected->imgdata->list_frame[iFrame]->localopd[1]=caldataSelected->imgdata->list_frame[iFrameTarget]->localopd[1];
	  	      sky_dataSelected->imgdata->list_frame[iFrame]->localopd[2]=caldataSelected->imgdata->list_frame[iFrameTarget]->localopd[2];
	  	      sky_dataSelected->imgdata->list_frame[iFrame]->localopd[3]=caldataSelected->imgdata->list_frame[iFrameTarget]->localopd[3];
	  	    }
	  	  }
	  	  cpl_vector_delete(dispCoef);
		  
		  //  Flag MATISSE data with GRA4MAT data
		  char *filename_gra=NULL;
		  filename_gra = mat_get_fname_gra(cpl_propertylist_get_int(caldataSelected->keywords,"ESO TPL EXPNO"), frameset);
		  if (filename_gra != NULL)
		    {
		      cpl_msg_info(cpl_func,"Tagging MATISSE frames with GRA4MAT data6");
		      mat_flag_frames_N(caldataSelected,filename_gra,1);
		    }
		  /* if (nbbin == 1) */
		  /*   { */
		  if (caldataSelected != NULL)
		    {
		      if (strstr(dprtype,"NOFRINGE") != NULL)
			{
			  mat_store_result(&info, caldataSelected, frame, "CALIB_CAL", "NOFRINGE", "F", sky_dataSelected,indexFileCalib);
			}
		      else
			{
			  mat_store_result(&info, caldataSelected, frame, "CALIB_CAL", "IMAGE", "F", sky_dataSelected,indexFileCalib);
			}
		      indexFileCalib++;
		      mat_gendata_delete(caldataSelected);
		    }
		  mat_gendata_delete(sky_dataSelected);
		  /*   } */
		  /* else */
		  /*   { */
		  /*     mat_gendata * caldataBinning = mat_gendata_binning(caldataSelected,nbbin); */
		  /*     mat_gendata_delete(caldataSelected); */
		  /*     if (sky_dataSelected == NULL) */
		  /* 	{ */
		  /* 	  sky_dataBinning = NULL; */
		  /* 	} */
		  /*     else */
		  /* 	{ */
		  /* 	  sky_dataBinning = mat_gendata_binning(sky_dataSelected,nbbin); */
		  /* 	} */
		  /*     mat_gendata_delete(sky_dataSelected); */
		  /*     if (caldataBinning != NULL) */
		  /* 	{ */
		  /* 	  alpha=info.odm->dispcoef[0]; */
		  /* 	  beta=info.odm->dispcoef[1]; */
		  /* 	  gamma=info.odm->dispcoef[2]; */
		  /* 	  c=caldataBinning->imgdet->list_region[0]->corner[1]; */
		  /* 	  info.odm->dispcoef[2]=gamma*nbbin*nbbin; */
		  /* 	  info.odm->dispcoef[1]=beta*nbbin+2*gamma*c*nbbin-2*c*info.odm->dispcoef[2]; */
		  /* 	  info.odm->dispcoef[0]=alpha+beta*c+gamma*c*c-info.odm->dispcoef[1]*c-info.odm->dispcoef[2]*c*c; */
		  /* 	  if (strstr(dprtype,"NOFRINGE") != NULL) */
		  /* 	    { */
		  /* 	      mat_store_result(&info, caldataBinning, frame, "CALIB_CAL", "NOFRINGE", "T", sky_dataBinning,indexFileCalib); */
		  /* 	    } */
		  /* 	  else */
		  /* 	    { */
		  /* 	      mat_store_result(&info, caldataBinning, frame, "CALIB_CAL", "IMAGE", "T", sky_dataBinning,indexFileCalib); */
		  /* 	    } */
		  /* 	  info.odm->dispcoef[0]=alpha; */
		  /* 	  info.odm->dispcoef[1]=beta; */
		  /* 	  info.odm->dispcoef[2]=gamma; */
			  
		  /* 	  indexFileCalib++; */
		  /* 	  mat_gendata_delete(caldataBinning); */
		  /* 	} */
		  /*     mat_gendata_delete(sky_dataBinning); */
		  /*   }    */
		      
	  	}
	      
	      if ( (strcmp(chipname, "AQUARIUS") == 0) && (mat_check_photometry(frame) == TRUE) && (info.odm != NULL) )
	  	{
	  	  cpl_vector * dispCoef = cpl_vector_new(N_DEG_DISPERSION_LAW+1);
	  	  cpl_vector_set(dispCoef,0,info.odm->dispcoef[0]);
	  	  cpl_vector_set(dispCoef,1,info.odm->dispcoef[1]);
	  	  cpl_vector_set(dispCoef,2,info.odm->dispcoef[2]);
	  	  cpl_vector_set(dispCoef,3,info.odm->dispcoef[3]);
	  	  cpl_vector_set(dispCoef,4,info.odm->dispcoef[4]);

	  	  mat_gendata * caldataSelected = mat_gendata_select_band(caldata,dispCoef);
	  	  cpl_vector_delete(dispCoef);
		  /* if (nbbin == 1) */
		  /*   { */
		  //  Flag MATISSE data with GRA4MAT data
		  char *filename_gra=NULL;
		  filename_gra = mat_get_fname_gra(cpl_propertylist_get_int(caldataSelected->keywords,"ESO TPL EXPNO"), frameset);
		  if (filename_gra != NULL)
		    {
		      cpl_msg_info(cpl_func,"Tagging MATISSE frames with GRA4MAT data7");
		      mat_flag_frames_N(caldataSelected,filename_gra,1);
		    }


		  if (caldataSelected != NULL)
		    {
		      if (strstr(dprtype,"NOFRINGE") != NULL)
			{
			  mat_store_result(&info, caldataSelected, frame, "CALIB_CAL", "NOFRINGE", "F", NULL, indexFileCalib);
			}
		      else
			{
			  mat_store_result(&info, caldataSelected, frame, "CALIB_CAL", "IMAGE", "F", NULL, indexFileCalib);
			} 
		      indexFileCalib++;
		      mat_gendata_delete(caldataSelected);
		    }
		  /*   } */
		  /* else */
		  /*   { */
		  /*     mat_gendata * caldataBinning = mat_gendata_binning(caldataSelected,nbbin); */
		  /*     mat_gendata_delete(caldataSelected); */
		  /*     if (caldataBinning != NULL) */
		  /* 	{ */
		  /* 	  alpha=info.odm->dispcoef[0]; */
		  /* 	  beta=info.odm->dispcoef[1]; */
		  /* 	  gamma=info.odm->dispcoef[2]; */
		  /* 	  c=caldataBinning->imgdet->list_region[0]->corner[1]; */
		  /* 	  info.odm->dispcoef[2]=gamma*nbbin*nbbin; */
		  /* 	  info.odm->dispcoef[1]=beta*nbbin+2*gamma*c*nbbin-2*c*info.odm->dispcoef[2]; */
		  /* 	  info.odm->dispcoef[0]=alpha+beta*c+gamma*c*c-info.odm->dispcoef[1]*c-info.odm->dispcoef[2]*c*c; */
		  /* 	  if (strstr(dprtype,"NOFRINGE") != NULL) */
		  /* 	    { */
		  /* 	      mat_store_result(&info, caldataBinning, frame, "CALIB_CAL", "NOFRINGE", "T", NULL, indexFileCalib); */
		  /* 	    } */
		  /* 	  else */
		  /* 	    { */
		  /* 	      mat_store_result(&info, caldataBinning, frame, "CALIB_CAL", "IMAGE", "T", NULL, indexFileCalib); */
		  /* 	    } */
		  /* 	  info.odm->dispcoef[0]=alpha; */
		  /* 	  info.odm->dispcoef[1]=beta; */
		  /* 	  info.odm->dispcoef[2]=gamma; */

		  /* 	  indexFileCalib++; */
		  /* 	  mat_gendata_delete(caldataBinning); */
		  /* 	} */
		  /*   } */
	  	}
	      if ( (strcmp(chipname, "AQUARIUS") == 0) && (info.odm == NULL) )
	  	{
		  //  Flag MATISSE data with GRA4MAT data
		  char *filename_gra=NULL;
		  filename_gra = mat_get_fname_gra(cpl_propertylist_get_int(caldata->keywords,"ESO TPL EXPNO"), frameset);
		  if (filename_gra != NULL)
		    {
		      cpl_msg_info(cpl_func,"Tagging MATISSE frames with GRA4MAT data8");
		      mat_flag_frames_N(caldata,filename_gra,1);
		    }



		  if (strstr(dprtype,"NOFRINGE") != NULL)
		    {
		      mat_store_result(&info, caldata, frame, "CALIB_CAL", "NOFRINGE", "T", NULL, indexFileTarget);
		    }
		  else
		    {
		      mat_store_result(&info, caldata, frame, "CALIB_CAL", "IMAGE", "T", NULL, indexFileTarget);
		    }
		  indexFileTarget++;
		}
	      if ( (strcmp(chipname, "HAWAII-2RG") == 0) && (info.odm != NULL) )
	  	{
	  	  mat_compensate_sky(&info, caldata);
	  	  cpl_vector * dispCoef = cpl_vector_new(N_DEG_DISPERSION_LAW+1);
	  	  cpl_vector_set(dispCoef,0,info.odm->dispcoef[0]);
	  	  cpl_vector_set(dispCoef,1,info.odm->dispcoef[1]);
	  	  cpl_vector_set(dispCoef,2,info.odm->dispcoef[2]);
	  	  cpl_vector_set(dispCoef,3,info.odm->dispcoef[3]);
	  	  cpl_vector_set(dispCoef,4,info.odm->dispcoef[4]);

	  	  mat_gendata * caldataSelected = mat_gendata_select_band(caldata,dispCoef);

		  
		  //  Flag MATISSE data with GRA4MAT data
		  char *filename_gra=NULL;
		  filename_gra = mat_get_fname_gra(cpl_propertylist_get_int(caldataSelected->keywords,"ESO TPL EXPNO"), frameset);
		  if (filename_gra != NULL)
		    {
		      cpl_msg_info(cpl_func,"Tagging MATISSE frames with GRA4MAT data9");
		      mat_flag_frames_L(caldataSelected,filename_gra,1);
		    }
	  

		  
	  	  cpl_vector_delete(dispCoef);
		  if (caldataSelected != NULL)
		    {
		      if (strstr(dprtype,"NOFRINGE") != NULL)
			{
			  mat_store_result(&info, caldataSelected, frame, "CALIB_CAL", "NOFRINGE", "F", NULL, indexFileCalib);
			}
		      else
			{
			  mat_store_result(&info, caldataSelected, frame, "CALIB_CAL", "IMAGE", "F", NULL, indexFileCalib);
			}
		      indexFileCalib++;
		      mat_gendata_delete(caldataSelected);
		    }
	  	}
	      if ( (strcmp(chipname, "HAWAII-2RG") == 0) && (info.odm == NULL) )
	  	{
   	  	  mat_compensate_sky(&info, caldata);
		  //  Flag MATISSE data with GRA4MAT data
		  char *filename_gra=NULL;
		  filename_gra = mat_get_fname_gra(cpl_propertylist_get_int(caldata->keywords,"ESO TPL EXPNO"), frameset);
		  if (filename_gra != NULL)
		    {
		      cpl_msg_info(cpl_func,"Tagging MATISSE frames with GRA4MAT data10");
		      mat_flag_frames_L(caldata,filename_gra,1);
		    }
		  mat_store_result(&info, caldata, frame, "TARGET_CAL", "IMAGE", "T", NULL, indexFileTarget);
		  indexFileTarget++;
	  	}
	      mat_gendata_delete(caldata);
	    }
	  cpl_propertylist_delete(header);
  	}
      else if (strcmp(cpl_frame_get_tag(frame), "CALIB_SRC_RAW" ) == 0)
      	{
	  header = cpl_propertylist_load(cpl_frame_get_filename(frame),0);
	  dprtype=(char *)cpl_propertylist_get_string(header,"ESO DPR TYPE");
	  if (dprtype == NULL) {
	    cpl_msg_error(cpl_func,"No DPR TYPE keyword");
	    return -1;
	  }

      	  index_local=mat_get_index_for_hot_dark(frame);
      	  if (index_local < 0) {
      	    return CPL_ERROR_UNSPECIFIED;
      	  }
      	  caldata = mat_load_and_calibrate(&info, frame);


	
      	  cpl_vector * dispCoef = cpl_vector_new(N_DEG_DISPERSION_LAW+1);
      	  cpl_vector_set(dispCoef,0,info.odm->dispcoef[0]);
      	  cpl_vector_set(dispCoef,1,info.odm->dispcoef[1]);
      	  cpl_vector_set(dispCoef,2,info.odm->dispcoef[2]);
      	  cpl_vector_set(dispCoef,3,info.odm->dispcoef[3]);
      	  cpl_vector_set(dispCoef,4,info.odm->dispcoef[4]);

      	  mat_gendata * caldataSelected = mat_gendata_select_band(caldata,dispCoef);
      	  if (info.hot_dark_data[index_local] == NULL) {
      	    cpl_msg_error(cpl_func,"HOT_DARK missing");
      	    cpl_vector_delete(dispCoef);
      	    mat_gendata_delete(caldata);
      	    mat_gendata_delete(caldataSelected);
      	    cpl_frameset_iterator_delete(it);
      	    mat_info_delete(&info);
      	    return -1;
      	  }
      	  mat_gendata * hot_dark_dataSelected = mat_gendata_select_band(info.hot_dark_data[index_local],dispCoef);
      	  cpl_vector_delete(dispCoef);
      	  if (caldataSelected != NULL)
      	    {
      	      //mat_compensate_sky(&info, caldata);
	      if (strstr(dprtype,"NOFRINGE") != NULL)
		{
		  mat_store_result(&info, caldataSelected, frame, "CALIB_CAL", "NOFRINGE", "F", hot_dark_dataSelected,indexFileCalib);
		}
	      else
		{
		  mat_store_result(&info, caldataSelected, frame, "CALIB_CAL", "IMAGE", "F", hot_dark_dataSelected,indexFileCalib);
		}
      	      indexFileCalib++;
      	      mat_gendata_delete(caldataSelected);
      	      mat_gendata_delete(hot_dark_dataSelected);
      	    }
      	  if (caldata != NULL)
      	    {
      	      mat_gendata_delete(caldata);
      	    }
	  cpl_propertylist_delete(header);
      	}
      cpl_frameset_iterator_advance(it, 1);
      frame = cpl_frameset_iterator_get(it);
    }
  cpl_frameset_iterator_delete(it);

  if (!cpl_errorstate_is_equal(prestate)) {
    cpl_errorstate_set(prestate);
  }
  // show the TARTYP processing results in a compact table
  qsort(info.ttresults, info.nbttresult, sizeof(mat_tartyp_result), mat_compare_tartyp_result);
  cpl_msg_info(cpl_func, "# TARTYP processing results");
  cpl_msg_info(cpl_func, "# EXPNO TSTART TPERIOD FSTART CHOPFREQ NSKY NTARGET NDEAD DELAY TSHIFT VSHIFT ISHIFT SRI1 SRI2 SRI3 SRI4 TRI1 TRI2 TRI3 TRI4");
  for (i = 0; i < info.nbttresult; i++)
    {
      cpl_msg_info(cpl_func, "%3d %.3f %.3f %.3f %.6f %2d %2d %2d %8.3f %2d %2d %2d %+.6f %+.6f %+.6f %+.6f %+.6f %+.6f %+.6f %+.6f",
		   info.ttresults[i].expno,
		   info.ttresults[i].timstart, info.ttresults[i].timperiod,
		   info.ttresults[i].framestart, info.ttresults[i].chopfreq,
		   info.ttresults[i].nbsky, info.ttresults[i].nbtarget, info.ttresults[i].nbdead,
		   info.ttresults[i].delay,
		   info.ttresults[i].est_time_shift, info.ttresults[i].est_V_shift, info.ttresults[i].est_N_shift,
		   info.ttresults[i].sky_relint[0], info.ttresults[i].sky_relint[1], info.ttresults[i].sky_relint[2], info.ttresults[i].sky_relint[3],
		   info.ttresults[i].tar_relint[0], info.ttresults[i].tar_relint[1], info.ttresults[i].tar_relint[2], info.ttresults[i].tar_relint[3]
		   );
    }
  mat_info_delete(&info);
  return 0;
}
