/* $Id: mat_cal_vis_lib.c,v0.5 2014-06-15 12:56:21 fguitton Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: fguitton $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_cal_vis_lib.c $
 */


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif


/*-----------------------------------------------------------------------------
  Includes
  -----------------------------------------------------------------------------*/

#include "mat_cal_vis_lib.h"


/*-----------------------------------------------------------------------------
  Functions prototypes
  -----------------------------------------------------------------------------*/



/*----------------------------------------------------------------------------*/
/**
   @brief Calculates the linear regression
   @param fref : reference function, cpl_bivector
   @param fout : result, cpl_bivector
   @return cpl_error_code
*/
/*----------------------------------------------------------------------------*/
cpl_error_code LinReg(cpl_bivector *fref, cpl_bivector *fout)
{
    
    double *xRef = NULL;
    double *yRef = NULL;
    double *xOut = NULL;
    double delta = 0.0;
    double b = 0.0;
    double a = 0.0;
    int refSize = 0;
    int outSize = 0;
    int i = 0;
    double sumX = 0.0;
    double sumY = 0.0;
    double sumXsquare = 0.0;
    double sumXY = 0.0;

    /* x size = y size */
    xRef = cpl_bivector_get_x_data (fref); 
    yRef = cpl_bivector_get_y_data (fref);
    xOut = cpl_bivector_get_x_data(fout);
    refSize = cpl_vector_get_size(cpl_bivector_get_x(fref)); 
    outSize = cpl_vector_get_size(cpl_bivector_get_x(fout)); 
    for (i=0; i<refSize; i++)
    {
	sumX = sumX + xRef[i];
	sumXsquare = sumXsquare + xRef[i]*xRef[i];
	sumY = sumY +yRef[i];
	sumXY = sumXY +xRef[i]*yRef[i];       
    }
    delta = refSize*sumXsquare - sumX*sumX;
    b=(sumXsquare*sumY-sumX*sumXY)/delta;
    a=(refSize*sumXY-sumX*sumY)/delta;
    /* y=ax+b */
    for (i=0; i<outSize; i++)
    {       
	cpl_vector_set(cpl_bivector_get_y(fout), i, a*xOut[i]+b);
    }

    return cpl_error_get_code();
}




/*----------------------------------------------------------------------------*/
/**
   @brief This function is called by the mat_cal_vis recipe
   @param parlist   contains the parameters supported 
   by the recipe
   @param frameset  the list of frames that have been given to
   the recipe for processing
   @param recname   Name teh recipes calling this function
   @return int : 0 if everything is ok
*/
/*----------------------------------------------------------------------------*/

int mat_cal_vis_lib(cpl_parameterlist *parlist, cpl_frameset *frameset,
		    const char *recname) {
    
    cpl_frame *cur_frame=NULL;
    cpl_frame *frame=NULL;
    cpl_frameset *usedframes = NULL;
    cpl_parameter *param = NULL;
    cpl_propertylist *p_list = NULL;
    cpl_propertylist *p_list_array = NULL;
    cpl_propertylist *qclist = NULL;
    cpl_table *table = NULL;
    cpl_vector *xRef = NULL;
    cpl_vector *yRef = NULL;
    cpl_vector *XOut = NULL;
    cpl_vector *YOut = NULL;
    cpl_bivector *fout = NULL;
    cpl_bivector *fref = NULL;
    int tfInterp = 0;
    int tfKeep = 0;
    /* int expert = 0; */
    int nb_extent = 0;
    int nbfile = 0;
    int i = 0;
    int j = 0;
    int k = 0;
    int l = 0;
    int m = 0;
    int cpt = 0;
    int nbBlockTotal = 0;
    int *nbBlock=NULL;
    int nbTel=0;
    int nbBase=0;
    int index1 = 0;
    int index2 = 0;
    //    int flag = 0;
    int nbOifitscal = 0;
    int indexChipName = 0;
    int i_nbchannel = 0;
    /* char filename[200] = " "; */
    char *filename=NULL;
    char *qcname1 = NULL;
    char *qcname2 = NULL;
    char *pszChipName = NULL;
    double *stdev = NULL;
    double **aux = NULL;
    double *data = NULL;
    double *auxErr = NULL;
    double *snrvis2 = NULL;
    double *snrtf2 = NULL;
    double *timet = NULL;
    double *timeI = NULL;
    double mean = 0.0;
    double snr = 0.0;
    char *p_filename = NULL;
    char *p_filetag = NULL;
    char *keyExtname = NULL;
    mat_oifits *oifits = NULL;
    mat_oifits **oifitscal = NULL;
    mat_array *oiarray = NULL;
    mat_array *oiarray_tempo = NULL;
    mat_oiwavelength *oiwave = NULL;
    mat_oiwavelength *oiwave_tempo = NULL;
    /* mat_oifits *p_oifits = NULL; */
    cpl_frameset_iterator *it=NULL;
    cpl_errorstate prestate = cpl_errorstate_get();

    /* tfInterp 0:mean, 1: running average, 2: linear interpolation */
    /* running average non codé*/    
    param = cpl_parameterlist_find(parlist, 
				   "matisse.mat_cal_vis.tfInterp");
    tfInterp = cpl_parameter_get_int(param);
    param = cpl_parameterlist_find(parlist, 
				   "matisse.mat_cal_vis.tfKeep");
    tfKeep = cpl_parameter_get_int(param);
    if (tfKeep == 1)
      {
	cpl_msg_warning(cpl_func,"tfKeep : Option not yet implemented");
      }
   
    nbfile = cpl_frameset_get_size( frameset);
    oifitscal = cpl_calloc(nbfile-1,sizeof(mat_oifits));
    usedframes=cpl_frameset_new();
 
    /*target file, reference */
    /*to compare with calib files */
    prestate=cpl_errorstate_get();
    it= cpl_frameset_iterator_new(frameset);
    do {
	cur_frame = cpl_frameset_iterator_get(it);
	if (cur_frame != NULL) {
	    cpl_frame_set_group(cur_frame,CPL_FRAME_GROUP_RAW);
	    p_filename = (char *)cpl_frame_get_filename(cur_frame);
	    p_filetag  = (char *)cpl_frame_get_tag(cur_frame);
	    if (!strcmp(p_filetag,"TARGET_RAW_INT")){
		nb_extent = cpl_frame_get_nextensions(cur_frame);
		for(i = 0; i<nb_extent; i++)
		{
		    p_list = cpl_propertylist_load(cpl_frame_get_filename(cur_frame),i+1);
		    keyExtname = (char *)cpl_propertylist_get_string(p_list,"EXTNAME");
		    if (!strcmp(keyExtname,"OI_ARRAY")) 
		    {			
			table=cpl_table_load(p_filename,i+1,0);
			oiarray = mat_array_from_table(p_list,table);
			cpl_table_delete(table);
		    }
		    if (!strcmp(keyExtname,"OI_WAVELENGTH") ) 
		    {			  
			oiwave=mat_oiwavelength_load(cur_frame);
		    }
		    cpl_propertylist_delete(p_list);
		}
		break;
	    }
	}
    } while (cpl_frameset_iterator_advance(it, 1) != CPL_ERROR_ACCESS_OUT_OF_RANGE);
    cpl_frameset_iterator_delete(it);
    if (!cpl_errorstate_is_equal(prestate)) {
	cpl_errorstate_set(prestate);
    }
    /*check tag*/
    /*load oifits structure*/
    /*load oifitscal structures if oiarray and oiwavelength are equal to oifits */
    prestate=cpl_errorstate_get();
    it= cpl_frameset_iterator_new(frameset);
    do {
      cur_frame = cpl_frameset_iterator_get(it);
      if (cur_frame != NULL) {
	/* Get Classification Tag */
	p_filename = (char *)cpl_frame_get_filename(cur_frame);
	p_filetag  = (char *)cpl_frame_get_tag(cur_frame);	  
	nb_extent = cpl_frame_get_nextensions(cur_frame);
	for(i = 0; i<nb_extent; i++) {
	  p_list = cpl_propertylist_load(cpl_frame_get_filename(cur_frame),i+1);
	  keyExtname = (char *)cpl_propertylist_get_string(p_list,"EXTNAME");
	  if (!strcmp(keyExtname,"OI_ARRAY")) {  
	    table=cpl_table_load(p_filename,i+1,0);
	    p_list_array = cpl_propertylist_duplicate(p_list);
	    if (!strcmp(p_filetag,"CALIB_RAW_INT")) {
	      oiarray_tempo = mat_array_from_table(p_list_array,table);			
	    }		    
	  }
	  if (!strcmp(keyExtname,"OI_WAVELENGTH") ) {
	    if (!strcmp(p_filetag,"CALIB_RAW_INT")) {
	      oiwave_tempo=mat_oiwavelength_load(cur_frame);
	    }
	  }
	  cpl_propertylist_delete(p_list);
	}
	if (!strcmp(p_filetag,"TARGET_RAW_INT")) {  
	  frame = cpl_frame_duplicate(cur_frame);
	  cpl_frame_set_group(frame,CPL_FRAME_GROUP_RAW);
	  cpl_frameset_insert(usedframes, frame);
	  oifits = mat_oifits_load(cur_frame);//ici la table est un oi_vis2
	  if (oifits->oivis2 == NULL) {
	    cpl_msg_error(cpl_func,"%s does not contain OI_VIS2 binary table",p_filename);
	    return CPL_ERROR_INCOMPATIBLE_INPUT;
	  }
	}
 
	if (!strcmp(p_filetag,"CALIB_RAW_INT")) {
	  if (mat_array_compare(oiarray, oiarray_tempo) == CPL_TRUE &&
	      mat_oiwavelength_compare(oiwave, oiwave_tempo) == CPL_TRUE) {
	    frame = cpl_frame_duplicate(cur_frame);
	    cpl_frame_set_group(frame,CPL_FRAME_GROUP_RAW);
	    cpl_frameset_insert(usedframes, frame);
	    oifitscal[j] = mat_oifits_load(cur_frame);
	    if (oifitscal[j]->oitf2 == NULL) {
	      cpl_msg_error(cpl_func,"%s does not contain TF2 binary table",p_filename);
	      return CPL_ERROR_INCOMPATIBLE_INPUT;
	    }
	    j = j+1;
	  } 
	}
	cpl_propertylist_delete(p_list_array);	
	cpl_table_delete(table);
	if (!strcmp(p_filetag,"CALIB_RAW_INT")) {
	  mat_array_delete(oiarray_tempo);
	  mat_oiwavelength_delete(oiwave_tempo);	  
	}
      }
    } while (cpl_frameset_iterator_advance(it, 1) != CPL_ERROR_ACCESS_OUT_OF_RANGE);
    cpl_frameset_iterator_delete(it);
    if (!cpl_errorstate_is_equal(prestate)) {
	cpl_errorstate_set(prestate);
    }
    mat_array_delete(oiarray);
    mat_oiwavelength_delete(oiwave);   
    nbOifitscal = j;
    nbBlock=cpl_calloc(nbOifitscal,sizeof(int));
    nbBlockTotal=0;
    nbTel=mat_get_nbtel_from_oivis2(oifitscal[0]->oivis2);
    cpl_msg_info(cpl_func,"Number of Telescope: %d",nbTel);
    nbBase=nbTel*(nbTel-1)/2;
    cpl_msg_info(cpl_func,"Number of base: %d",nbBase);

    for(j=0; j<nbOifitscal; j++ )
      {
	nbBlock[j]=(int)(oifitscal[j]->oivis2->nbvis2/nbBase);
	nbBlockTotal+=nbBlock[j];
      }
    cpl_msg_info(cpl_func,"Number of blocks: %d",nbBlockTotal);
    
    if (oifits == NULL)
      {
	cpl_free(nbBlock);
	return CPL_ERROR_INCOMPATIBLE_INPUT;
      }

    i_nbchannel = oifits->oivis2->nbchannel;
    aux = cpl_calloc(i_nbchannel, sizeof(double));
    for(j=0; j< i_nbchannel; j++ )
      {
	aux[j] = cpl_calloc(nbBlockTotal, sizeof(double));
      }

    data = cpl_calloc(i_nbchannel, sizeof(double));
    stdev = cpl_calloc(i_nbchannel, sizeof(double));
    auxErr = cpl_calloc(i_nbchannel, sizeof( double));
    snrvis2 = cpl_calloc(oifitscal[0]->oitf2->nbchannel, sizeof(double));
    snrtf2 = cpl_calloc(oifitscal[0]->oitf2->nbchannel, sizeof(double));
 
    for (l=0; l < i_nbchannel; l++ )
      {
	stdev[l] = 0.0;
	auxErr[l] = 0.0;
	data[l] = 0.0;
	for(j=0; j<nbBlockTotal; j++ )
	  {
	    aux[l][j]=0.0;
	  }
      }


    /*compute vis2 and vis2err*/
    /*"pour chaque ligne du fichier cible:*/
    /* read target file's line */
    for (i=0; i<oifits->oivis2->nbvis2; i++)
      {
	for (l=0; l < i_nbchannel; l++ )
	  {
	    stdev[l] = 0.0;
	    auxErr[l] = 0.0;
	    data[l] = 0.0;
	    for(j=0; j<nbBlockTotal; j++ )
	      {
		aux[l][j]=0.0;
	      }
	  }
	index1 = oifits->oivis2->list_vis2[i]->stationindex[0];
	index2 = oifits->oivis2->list_vis2[i]->stationindex[1];
	/*one calib file */
	/* if (nbBlockTotal == 1) */
	/*   { */
	/*     for(k=0;k<oifitscal[0]->oivis2->nbvis2;k++) */
	/*       { */
	/* 	/\*on cherche la ligne correspondante dans le fichier de calib*\/ */
	/* 	if(oifitscal[0]->oivis2->list_vis2[k]->stationindex[0] == index1 */
	/* 	   && oifitscal[0]->oivis2->list_vis2[k]->stationindex[1] == index2) { */
	/* 	  /\* data and error*\/ */
	/* 	  for (l=0; l < i_nbchannel; l++ ) */
	/* 	    { */
	/* 	      aux[l][0] += oifitscal[0]->oitf2->list_tf2[k]->tf2[l]; */
	/* 	      auxErr[l] = auxErr[l] + pow(oifitscal[0]->oitf2->list_tf2[k]->tf2err[l],2.0); */
	/* 	    } */
	/* 	  break; */
	/* 	} */
	/*       } */
	/*     /\*nbOfitscal calib files*\/ */
	/*   } */
	/* else */
	/*   { */
	    cpt=0;
	    for(j=0;j<nbOifitscal;j++)
	      {
		for(k=0;k<oifitscal[j]->oivis2->nbvis2;k++)
		  {
		    /*on cherche la ligne correspondante 
		      dans le fichier de calib*/
		    if(oifitscal[j]->oivis2->list_vis2[k]->stationindex[0] == index1
		       && oifitscal[j]->oivis2->list_vis2[k]->stationindex[1] == index2)
		      {
			for (l=0;l<i_nbchannel;l++)
			  {      
			    aux[l][cpt] = oifitscal[j]->oitf2->list_tf2[k]->tf2[l];
			    /*somme des carres des erreurs */
			    auxErr[l] = auxErr[l]+ pow(oifitscal[j]->oitf2->list_tf2[k]->tf2err[l],2.0); 
			  }
			cpt++;
		      }
		  }
	      }
	  /* } */

	for (l=0; l < oifitscal[0]->oivis2->nbchannel; l++ )
	  {	 
	    /* mean */
	    if (tfInterp == 0)
	      {
		data[l]=0.;
		cpt=0;
		for(j=0; j<nbOifitscal; j++ )
		  {
		    for(m=0;m<nbBlock[j];m++)
		      {
			data[l]=data[l]+aux[l][cpt];
			cpt++;
		      }
		  }
		data[l] = data[l]/nbBlockTotal;	
	      }
	    
	    
	    /* linear interpolation */
	    /* on est toujours pour une ligne donnee*/
	    if (tfInterp == 2)
	      {
		if (nbBlockTotal >1)
		  {
		    YOut =  cpl_vector_new(1);
		    timet  = cpl_calloc(nbBlockTotal, sizeof(double));
		    timeI  = cpl_calloc(1, sizeof(double));
		    /*pour une ligne donnee, 
		      on prend le time de chaque fichier de calib*/
		    cpt=0;
		    for (k=0;k<nbOifitscal;k++)
		      {
			for(m=0;m<nbBlock[k];m++)
			  {
			    timet[cpt]=oifitscal[k]->oivis2->list_vis2[m*nbBase]->time;
			    cpt++;
			  }
		      }
			
		    xRef = cpl_vector_wrap(nbBlockTotal, timet);
		    /*pour une ligne donnee,  */
		    yRef=cpl_vector_wrap(nbBlockTotal, aux[l]);
		    timeI[0] =  oifits->oivis2->list_vis2[i]->time;
		    XOut = cpl_vector_wrap(1,timeI);
		    fref = cpl_bivector_wrap_vectors(xRef,yRef);
		    fout = cpl_bivector_wrap_vectors(XOut,YOut);
		    LinReg(fref,fout);
		    data[l] =  cpl_bivector_get_y_data(fout)[0];
		    cpl_vector_unwrap(xRef);
		    cpl_vector_unwrap(yRef);
		    cpl_vector_unwrap(XOut);
		    cpl_bivector_unwrap_vectors(fout);
		    cpl_bivector_unwrap_vectors(fref);
		    cpl_vector_delete(YOut);
		    cpl_free(timet);
		    cpl_free(timeI);
		  }
		else
		  {
		    cpl_msg_info(cpl_func,"not enough calibration files");
		    exit(1);
		  }
	      }

	    /*calcul erreurs */
	    snrvis2[l] =  oifits->oivis2->list_vis2[i]->vis2[l]/oifits->oivis2->list_vis2[i]->vis2err[l];
	    stdev[l] = sqrt(auxErr[l]/nbBlockTotal);
	    snrtf2[l]= data[l]/stdev[l];
	    if (data[l] == 0.)
	      {
		oifits->oivis2->list_vis2[i]->vis2err[l] =  -1.; 
		oifits->oivis2->list_vis2[i]->vis2[l]=-1.;
	      }
	    else
	      {
		oifits->oivis2->list_vis2[i]->vis2err[l] =  (oifits->oivis2->list_vis2[i]->vis2[l]/data[l])*pow((1/pow(snrvis2[l],2.0)+1/pow(snrtf2[l],2.0)),0.5); 
		oifits->oivis2->list_vis2[i]->vis2[l] /= data[l];
	      }
	  }
      }


    
    cpl_free(stdev);
    for(j=0; j< i_nbchannel; j++ )
      {
	cpl_free(aux[j]);
      }
    cpl_free(aux);
    cpl_free(snrvis2);
    cpl_free(snrtf2);
    cpl_free(auxErr);  
    cpl_free(data);  

    /*qcparameters*/
    qclist = cpl_propertylist_new();
    if ( !strcmp(oifits->oivis2->insname,"MATISSE"))
      {
	pszChipName=(char *)cpl_propertylist_get_string(oifits->keywords,"ESO DET CHIP NAME");//keywords???
	if (pszChipName == NULL)
	  {
	    cpl_free(nbBlock);
	    cpl_propertylist_delete(qclist);
	    cpl_msg_error(cpl_func,"No ESO DET CHIP NAME keyword");
	    return 1;
	  }
	if(strcmp(pszChipName, "AQUARIUS") == 0)
	  {
	    indexChipName=2;
	  }
      else if(strcmp(pszChipName, "HAWAII2RG") == 0)
	{
	  indexChipName=1;
	}
      }
    else
      {
	indexChipName=3;
      }
    for (i=0; i<oifits->oivis2->nbvis2; i++)
      {
	mean = 0.0;
	snr = 0.0;
	qcname1 = cpl_sprintf("ESO QC DET%d CALVISSQR%d ",indexChipName,i+1);
	qcname2 = cpl_sprintf("ESO QC DET%d CALVISSQR%d SNR ", indexChipName,i+1);
	for(j=0; j<i_nbchannel; j++ )
	  {
	    if (!isnan(oifits->oivis2->list_vis2[i]->vis2[j]))
	      {
		mean = mean +  oifits->oivis2->list_vis2[i]->vis2[j];
	      }
	  }
	mean = mean / i_nbchannel;
	for (j=0; j < i_nbchannel; j++)
	  {
	    snr =snr + pow((oifits->oivis2->list_vis2[i]->vis2[j]-mean),2.0);
	  }
	if (snr < 0) snr=fabs(snr);
	if (snr == 0) snr=1.E-6;
	snr = mean/sqrt(snr/i_nbchannel);
	cpl_propertylist_append_double(qclist, qcname1, mean);
	cpl_propertylist_append_double(qclist, qcname2, snr);
	cpl_free(qcname1);
	cpl_free(qcname2);
      }
    cpl_propertylist_append(oifits->keywords, qclist);
    /*save data*/
    it= cpl_frameset_iterator_new(frameset);
    cur_frame = cpl_frameset_iterator_get(it);
    cpl_frameset_iterator_delete(it);
    filename=cpl_sprintf("CALVIS_%s",basename((char *)cpl_frame_get_filename(cur_frame)));
    mat_oifits_save(oifits, frameset, usedframes, parlist,recname, filename, 6);
    mat_oifits_delete(oifits);
    cpl_free(filename);
    

    cpl_propertylist_delete(qclist);
    cpl_frameset_delete(usedframes);
    for(i=0; i<nbfile-1; i++)
      {
	mat_oifits_delete(oifitscal[i]);
      }
    cpl_free(oifitscal);
    cpl_free(nbBlock);
    return 0;
}

