/* $Id: mat_compute_phidiff.c,v0.5 2014-06-15 12:56:21 fmillour Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software;you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation;either version 2 of the License,	or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY;without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 */

/*
 * $Author: fmillour $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_compute_phidiff.c $
 */
#ifdef HAVE_CONFIG_H
#include <config.h>
#include <string.h>
#endif
#include <stdio.h>
#include <gsl/gsl_rng.h>
/*-----------------------------------------------------------------------------
  Includes
  ----------------------------------------------------------------------------*/

#include "mat_opd_wvpo.h"
#include "mat_compute_phidiff.h"
#include "mat_error.h"

/*-----------------------------------------------------------------------------
  Define
  ----------------------------------------------------------------------------*/
/*-----------------------------------------------------------------------------
  Functions prototypes
  ----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------*/

/**
   @brief Compute the interspectrum and differential phase from
   @brief mat_corrflux structure and a mat_photbeams structure.
   @brief This function fills a mat_oifits with computed differential
   @brief phases in the way described by F. Millour et al's 
   @brief ("DIFFERENTIAL INTERFEROMETRY WITH THE AMBER/VLTI INSTRUMENT:
   @brief				 DESCRIPTION, PERFORMANCES AND ILLUSTRATION", EAS
   @brief Publications			 Series, Vol. 22, pp 379, 2006)
   @param corrFlux correlated flux structure
   @param phot photometry structure 
   @param oifits oifits containing the differential phases 
   @param oiopdwvpo structure containing the OPD values 
   @return 0 if no error
*/
int mat_compute_phidiff(mat_corrflux	*corrFlux,
                        mat_photbeams	*phot,
                        mat_oifits	*oifits,
                        mat_oiopdwvpo	*oiopdwvpo)
{
    /* cpl_parameter *useAvgPhot) */
    /* Ranges */
    int	iReg = 0,iFrame=0,iWlen=0,iWlen0=0,jWlen=0,iPix=0,iBase=0;
    int	nbRegion, nbFrames;
    int nbFramesPhot=0;
    int	nbWlen, newNbWlen, nbFreq, nbPix, nbBases = 6;
    /* Counters */
    int	k;
    int		 cptWave     = 0;
    double	 uPixel	     = 0;
    int		 pis	     = 0;
    /* Intermediate images with the work and reference channels */
    cpl_image	*workChanRe  = NULL;
    cpl_image	*workChanIm  = NULL;
    cpl_image	*workChanRe2 = NULL;
    cpl_image	*workChanIm2 = NULL;
    cpl_image	*refChanRe   = NULL; 
    cpl_image	*refChanIm   = NULL;
    cpl_image	*crossSpecRe = NULL;
    cpl_image	*crossSpecIm = NULL;
    cpl_image	*w1Re	     = NULL;
    cpl_image	*w1Im	     = NULL;
    /* cpl_image	*w1Re_forleo = NULL; */
    /* cpl_image	*w1Im_forleo = NULL; */
    cpl_image	*phi_SUM     = NULL;
    cpl_image	*phi_SUMSQ   = NULL;
    cpl_image	*phi_RMS     = NULL;
    cpl_image	*phi_0	     = NULL;
    cpl_image	*img1	     = NULL;
    cpl_image	*img2	     = NULL;
    cpl_image	*phot1	     = NULL;
    cpl_image	*phot2	     = NULL;
    cpl_image	*phot3	     = NULL;
    cpl_image	*phot4	     = NULL;
    /* intermediate values wavelength by wavelength */
    cpl_image	*refChanReJ  = NULL;
    cpl_image	*refChanImJ  = NULL;
    /* cpl_vector  *fit=NULL;*/
    cpl_vector	*dispCoef    = NULL;
    cpl_vector	*vecw	     = NULL;
    /* cpl_vector  *vecy=NULL;*/
    cpl_vector	*vecrow	     = NULL;
    cpl_image	*vecrow2     = NULL;
    cpl_image	*vecrow3     = NULL;
    cpl_image	*imrow	     = NULL;
    cpl_image	*imrow2	     = NULL;
    cpl_image	*imrow3	     = NULL;
    double	 lambda	     = 0.;
    double	 phi	     = 0.;
    double	 vis	     = 0.;
    double	 phasorIm    = 0.;
    double	 phasorRe    = 0.;

    cpl_image	*tmp = NULL;
    double	 val	    = 0., val1    = 0., val2 = 0.;
    double	 valIm	    = 0., valRe = 0.;
    double	 cflux	    = 0., flux    = 0., sqrtflux=0., avgsqrtflux=0.;
    double	 cfluxIm    = 0., cfluxRe = 0.;
    cpl_image	*phidiff    = NULL;
    cpl_image	*phidifferr = NULL;
    /* cpl_image	*cfxAmp     = NULL; */
    cpl_image	*visdiff    = NULL;
    cpl_image	*visdifferr = NULL;
    cpl_image	*flag       = NULL;

    cpl_image	**prodPhot    = NULL;
    cpl_image	**prodPhotErr = NULL;
    cpl_image	 *foo         = NULL;

    /* char	*insMode      = NULL;*/
    char	*sliderPhot   = NULL;
    char *detName	      = NULL;
    char	 key[15];
    char	 key2[17];
    char	 band;		//= "p";
    float	 lambdamin    = 0.;
    float	 lambdamax = 0.;
  float	  wlenmin	  = 1e99;
  float	  wlenmax	  = 0.;
    char  mode[3]        = " ";
    char	*bcdStatus=NULL;
 
    int   detNum	    = 0;
    int		 resolution = 0;
    char	*specResData = NULL;
    char	 kwd[64];



    /* Check input parameters */
    mat_assert((corrFlux != NULL),CPL_ERROR_NULL_INPUT, "no mat_corrflux	\
argument given");
    mat_assert((oifits != NULL),CPL_ERROR_NULL_INPUT, "no mat_oifits	\
argument given");
    /* mat_assert((phot != NULL),CPL_ERROR_NULL_INPUT, "no mat_phot_beams	\
       argument given"); */

    /* Get the array dimensions */
    nbRegion = corrFlux->imgdet->nbregion;
    nbFrames = corrFlux->nbframe / nbRegion;

    int nbFrameTarget=nbFrames;
    if (cpl_propertylist_has(corrFlux->keywords,"PRO FRAME TARGET")) {
      nbFrameTarget=cpl_propertylist_get_int(corrFlux->keywords,"PRO FRAME TARGET");
    }


    
    if(phot != NULL)
	nbFramesPhot = phot->nbphotxframe / phot->imgdet->nbregion;
    
    /* Read the detector type (and hence band) */
    detName = (char *)cpl_propertylist_get_string(corrFlux->keywords,"ESO DET CHIP NAME");
    if (detName == NULL)
    {
        cpl_msg_error(cpl_func,
                      "no ESO DET CHIP NAME keyword in frame");
        return -1;
    }
    mat_identify_det(detName, &band, &detNum);


    /* Read the spectral resolution */
    sprintf(key, "ESO INS DI%c ID",band);
    specResData = (char *)cpl_propertylist_get_string(corrFlux->keywords,key);
    char *fil = NULL;
    fil = (char *)cpl_propertylist_get_string(corrFlux->keywords,"ESO INS FIL NAME");
    mat_identify_res(specResData, fil, detNum, &resolution);
    

    

    
    if ( cpl_propertylist_has(oifits->keywords,"ESO INS BCD1 ID") )
    {
	bcdStatus = (char *)cpl_propertylist_get_string(oifits->keywords,"ESO INS BCD1 ID");
    }
    
    /* Loading of the dispersion coefficients */
    dispCoef = cpl_vector_new(N_DEG_DISPERSION_LAW+1);
    cpl_vector_set(dispCoef,0,cpl_propertylist_get_double(corrFlux->keywords,"PRO DISP COEF0"));
    cpl_vector_set(dispCoef,1,cpl_propertylist_get_double(corrFlux->keywords,"PRO DISP COEF1"));
    cpl_vector_set(dispCoef,2,cpl_propertylist_get_double(corrFlux->keywords,"PRO DISP COEF2"));
    cpl_vector_set(dispCoef,3,cpl_propertylist_get_double(corrFlux->keywords,"PRO DISP COEF3"));
    cpl_vector_set(dispCoef,4,cpl_propertylist_get_double(corrFlux->keywords,"PRO DISP COEF4"));



      
  /* Set the useful range of wavelengths to compute OPD */
  /* First of all retrieve the actual wavelength range */
  nbRegion = corrFlux->imgdet->nbregion;
  for(iReg = 0;iReg<nbRegion;iReg++)
  {
      cpl_msg_info(cpl_func,"Determining wavelength range");
      nbWlen = corrFlux->imgdet->list_region[iReg]->naxis[1];
        iWlen0 = corrFlux->imgdet->list_region[iReg]->corner[1];
        for(iWlen = 0;iWlen<nbWlen;iWlen++)
	{
            /* Compute wavelength and set wlen interval */
            lambda = mat_get_lambda(dispCoef, iWlen + iWlen0);
	    if(lambda < wlenmin)
		wlenmin = lambda;
	    if(lambda > wlenmax)
		wlenmax = lambda;
	}
  }
  mat_identify_wlrng(detNum, wlenmin, wlenmax, resolution, &lambdamin, &lambdamax);

    

    cpl_msg_info(cpl_func,"Detector %d (%s)",detNum, detName);
    cpl_msg_info(cpl_func,"Band %c, from %1.1f to %1.1f microns", band, lambdamin, lambdamax);
    cpl_msg_info(cpl_func,"Spectral Resolution %d (%s)",resolution, specResData);

    /* Read the instrument mode */
    sprintf(key2, "ESO INS PI%c NAME",band);
    sliderPhot = (char *)cpl_propertylist_get_string(corrFlux->keywords,key2);
    if (!strcmp(sliderPhot, "PHOTO")) {
	cpl_msg_info(cpl_func,"SiPhot mode");
	sprintf(mode,"s");
    } else {
	cpl_msg_info(cpl_func,"HighSens mode");
	sprintf(mode,"h");
    }
    
 
    /*****************************/
    /* First of all correct OPD! */
    /*****************************/
    if (oiopdwvpo != NULL)
    {
        cpl_msg_info(cpl_func,"Correcting OPD");
        mat_corr_opd_wvpo(oiopdwvpo, corrFlux);
    }
 

    cpl_msg_info(cpl_func,"Computing the differential phases");
 
    /* Loop on regions */
    for(iReg = 0;iReg<nbRegion;iReg++)
    {
        cpl_msg_info(cpl_func,"Calculating cross spectrum");

        /* Define nbWlen and nbFreq according to dimensions of the images */
        nbFreq = corrFlux->imgdet->list_region[iReg]->naxis[0];
        nbWlen = corrFlux->imgdet->list_region[iReg]->naxis[1];
        iWlen0 = corrFlux->imgdet->list_region[iReg]->corner[1];
 
        /* Get minimum and maximum wavelength indexes */
        cptWave	= 0;
        for(iWlen = 0;iWlen<nbWlen;iWlen++)
	{
            /* Compute wavelength and peak frequencies */
            lambda = mat_get_lambda(dispCoef, iWlen + iWlen0);
            if (lambda>=lambdamin && lambda<=lambdamax)
                cptWave++;
            /* printf("%f %f\n",lambda, lambdamin); */
	}
        newNbWlen = cptWave;
        cpl_msg_info(cpl_func,"Nb wlen: %d / Nb Wlen for ref. channel: %d",nbWlen, newNbWlen);
            
        /* Initialize work channel, reference channel and cross spectrum */
        workChanIm  = cpl_image_new(nbBases, nbWlen, CPL_TYPE_DOUBLE);
        workChanRe  = cpl_image_new(nbBases, nbWlen, CPL_TYPE_DOUBLE);
        workChanIm2 = cpl_image_new(nbBases, nbWlen, CPL_TYPE_DOUBLE);
        workChanRe2 = cpl_image_new(nbBases, nbWlen, CPL_TYPE_DOUBLE);
        refChanIm   = cpl_image_new(nbBases, nbWlen, CPL_TYPE_DOUBLE);
        refChanRe   = cpl_image_new(nbBases, nbWlen, CPL_TYPE_DOUBLE);
        crossSpecIm = cpl_image_new(nbBases, nbWlen, CPL_TYPE_DOUBLE);
        crossSpecRe = cpl_image_new(nbBases, nbWlen, CPL_TYPE_DOUBLE);
        w1Re	    = cpl_image_new(nbBases, nbWlen, CPL_TYPE_DOUBLE);
        w1Im	    = cpl_image_new(nbBases, nbWlen, CPL_TYPE_DOUBLE);
        /* w1Re_forleo = cpl_image_new(nbBases, nbWlen, CPL_TYPE_DOUBLE); */
        /* w1Im_forleo = cpl_image_new(nbBases, nbWlen, CPL_TYPE_DOUBLE); */

        /* Compute the phase of each fringe peak */
        phidiff    = cpl_image_new(nbBases, nbWlen, CPL_TYPE_DOUBLE);
        phidifferr = cpl_image_new(nbBases, nbWlen, CPL_TYPE_DOUBLE);
	//        cfxAmp     = cpl_image_new(nbBases, nbWlen, CPL_TYPE_DOUBLE);
        visdiff    = cpl_image_new(nbBases, nbWlen, CPL_TYPE_DOUBLE);
        visdifferr = cpl_image_new(nbBases, nbWlen, CPL_TYPE_DOUBLE);
        flag       = cpl_image_new(nbBases, nbWlen, CPL_TYPE_INT);
 
        phi_SUM   = cpl_image_new(nbBases,nbWlen,CPL_TYPE_DOUBLE);
        phi_SUMSQ = cpl_image_new(nbBases,nbWlen,CPL_TYPE_DOUBLE);
        phi_0     = cpl_image_new(nbBases,nbWlen,CPL_TYPE_DOUBLE);
        phi_RMS   = cpl_image_new(nbBases,nbWlen,CPL_TYPE_DOUBLE);


        /* Loop on frames */
        for(iFrame = 0;iFrame<nbFrameTarget;iFrame++)
	{
            if (iFrame % 100 == 0)
	    {
                cpl_msg_info(cpl_func,"Frame %d/%d", iFrame+1, nbFrameTarget);
	    }
            cptWave=0;
            for(iWlen = 0;iWlen<nbWlen;iWlen++)
	    {
                /* Compute wavelength and peak frequencies */
                lambda = mat_get_lambda(dispCoef, iWlen + iWlen0); 

                /* Pick-up phase at the peak frequency and store it */
                for(iBase = 0;iBase<nbBases;iBase++)
		{
                    uPixel = mat_get_pos_fringe_peak(lambda,detNum,resolution,iBase);
		    val    = (int)(uPixel);
                    val1   = uPixel - val;
                    val2   = 1.0 - val1;
 
                    /* if(iBase == 2) */
                    /* if(iFrame == 5) */
                    /* printf("%f %f %f %f %f\n", */
                    /* lambda, uPixel, val, val1, val2);*/

                    /* Interpolate peak value (real part) */
                    tmp	  = corrFlux->list_corr[iReg*nbFrames+iFrame]->imgreg[0];
                    valRe = 
                        val2 * cpl_image_get(tmp, val+1, iWlen+1, &pis) +
                        val1 * cpl_image_get(tmp, val+2, iWlen+1, &pis);

                    /* Interpolate peak value (imaginary part) */
                    tmp	  = corrFlux->list_corr[iReg*nbFrames+iFrame]->imgreg[1];
                    valIm = 
                        val2 * cpl_image_get(tmp, val+1, iWlen+1, &pis) +
                        val1 * cpl_image_get(tmp, val+2, iWlen+1, &pis);
                    
                    /* Store the result in the work channel  */
                    cpl_image_set(workChanRe, iBase+1, iWlen+1, valRe);
                    cpl_image_set(workChanIm, iBase+1, iWlen+1, valIm);
                    /* if (iFrame == 0 && iBase==5)  */
		    /*   printf("%d %f %f %f %f\n",iWlen,lambda,valRe,valIm, atan2(valIm,valRe)); */
                    /* printf("%f %f %f \n", */
                    /*        lambda, lambdamin, lambdamax); */
                    if (lambda	  >= lambdamin && lambda<=lambdamax)
		    {
                        /* Store the result in the work channel for the
                         * reference channel computation */
                        cpl_image_set(workChanRe2, iBase+1, cptWave+1, valRe);
                        cpl_image_set(workChanIm2, iBase+1, cptWave+1, valIm);
                        cpl_image_set(flag, iBase+1, iWlen+1, 0);
                        //printf(" %d %d \n",CPL_FALSE);
                        /* printf("%s %s \n", */
                        /*        0 ? "true" : "false", */
                        /*        cpl_image_get(flag,iBase+1,iWlen+1,&pis) ? "true" : "false"); */
		    }
                    else
		    {
                        cpl_image_set(flag, iBase+1, iWlen+1, 1);
                        /* printf(" %d\n",CPL_TRUE); */
                        /* printf("%s %s \n", */
                        /*        1 ? "true" : "false", */
                        /*        cpl_image_get(flag,iBase+1,iWlen+1,&pis) ? "true" : "false"); */
		    }   
		}
                cptWave++;
	    }
            /* Collapse vector along the wavelength direction */
            imrow2 = cpl_image_collapse_create(workChanRe2, 0);
            imrow3 = cpl_image_collapse_create(workChanIm2, 0);
     
            /* Loop on wavelengths and compute reference channel as */
            /* sum of all channels except the work channel */
	    for(iWlen =	0; iWlen<nbWlen; iWlen++)
	    {
                /* Get current channel */
                vecrow = cpl_vector_new_from_image_row(workChanRe2, iWlen+1);
                imrow      = cpl_image_new(nbBases,1,CPL_TYPE_DOUBLE);
                for(iBase  = 0;iBase<nbBases;iBase++)
                    cpl_image_set(imrow, iBase+1, 1,
                                  cpl_vector_get(vecrow, iBase));
                /* Subtract current channel to the sum of all channels */
                refChanReJ = cpl_image_subtract_create(imrow2, imrow);
                cpl_vector_delete(vecrow);
                cpl_image_delete(imrow);
 
                /* Get current channel */
                vecrow = cpl_vector_new_from_image_row(workChanIm2, iWlen+1);
                imrow  = cpl_image_new(nbBases,1,CPL_TYPE_DOUBLE);
                for(iBase  = 0;iBase<nbBases;iBase++)
                    cpl_image_set(imrow, iBase+1, 1,
                                  cpl_vector_get(vecrow, iBase));
                /* Subtract current channel to the sum of all channels */
                refChanImJ = cpl_image_subtract_create(imrow3, imrow);
                cpl_vector_delete(vecrow);
                cpl_image_delete(imrow);
    
                /* Put back the array into reference channel image */
                for(iBase = 0;iBase<nbBases;iBase++)
		{
                    /* Do that pixel by pixel since it is not possible to put */
                    /* directly a cpl vector in a row or column of an image...*/
		    cpl_image_set(refChanRe,iBase+1,iWlen+1,
				  cpl_image_get(refChanReJ,iBase+1,1,&pis));
		    cpl_image_set(refChanIm,iBase+1,iWlen+1,
				  -cpl_image_get(refChanImJ,iBase+1,1,&pis));
  
		}
                cpl_image_delete(refChanReJ);
                cpl_image_delete(refChanImJ);
	    }
            cpl_image_delete(imrow2);
            cpl_image_delete(imrow3);
     
            /*	cpl_msg_info(cpl_func,"Computing cross product");*/
     
            /* Compute the cross product and add them up for all frames */
            /* First compute real part of the cross product */
            img1 = cpl_image_multiply_create(workChanRe, refChanRe);
            img2 = cpl_image_multiply_create(workChanIm, refChanIm);
            cpl_image_subtract(img1,img2);
            /* Add the values of cross spectrum of all frames */
            cpl_image_add(crossSpecRe, img1);
            /* Cleanup images */
            cpl_image_delete(img1);
            cpl_image_delete(img2);
 
            /* Then compute imaginary part of the cross product */
            img1 = cpl_image_multiply_create(workChanIm, refChanRe);
            img2 = cpl_image_multiply_create(workChanRe, refChanIm);
            cpl_image_add(img1,img2);
            /* Add the values of cross spectrum of all frames */
            cpl_image_add(crossSpecIm,img1);
            /* Cleanup images */
            cpl_image_delete(img1);
            cpl_image_delete(img2);
	}
 
        /* /\* Remove residual phase from interspectrum *\/ */
        /* imrow2 = cpl_image_collapse_create(crossSpecRe, 0); */
        /* imrow3 = cpl_image_collapse_create(crossSpecIm, 0); */
 
	/* for(iWlen=0;iWlen<nbWlen;iWlen++) */
        /* { */
        /*     for(iBase = 0;iBase<nbBases;iBase++) */
        /*     { */
        /*         phi = atan2(cpl_image_get(imrow3,iBase+1,1,&pis), */
        /*                          cpl_image_get(imrow2,iBase+1,1,&pis)); */
        /*         phasorRe = cos(phi); */
        /*         phasorIm = -sin(phi); */
        /*         cfluxRe  = cpl_image_get(crossSpecRe,iBase+1,iWlen+1,&pis); */
        /*         cfluxIm  = cpl_image_get(crossSpecIm,iBase+1,iWlen+1,&pis); */
        /*         valRe    = cfluxRe * phasorRe - cfluxIm * phasorIm; */
        /*         cpl_image_set(crossSpecRe, iBase+1, iWlen+1, valRe); */
        /*         /\* Imaginary part *\/ */
        /*         valIm    = cfluxIm * phasorRe + cfluxRe * phasorIm; */
        /*         cpl_image_set(crossSpecIm,iBase+1,iWlen+1,valIm); */
        /*     } */
        /* } */
        /* cpl_image_delete(imrow2); */
        /* cpl_image_delete(imrow3); */
 
        /* DIFFERENTIAL PHASE COMPUTATION */
        cpl_msg_info(cpl_func,"Getting differential phase");

        for(iWlen=0;iWlen<nbWlen;iWlen++)
	{
            /* Pick-up phase at the peak frequency and store it */
            for(iBase = 0;iBase<nbBases;iBase++)
	    { 
                /* Set the cross spectrum value */
                valRe = cpl_image_get(crossSpecRe,iBase+1,iWlen+1,&pis);
                valIm = cpl_image_get(crossSpecIm,iBase+1,iWlen+1,&pis);
    
                /* DIFFERENTIAL PHASE COMPUTATION ITSELF */
                phi = atan2(valIm,valRe);
                cpl_image_set(phidiff,iBase+1,iWlen+1,phi);
	    }
	}

        /*************************************************************
         *** FIXME: the reset has weirdy side effects to be debugged
         *** Reset tables for later use
         *************************************************************/
        /* cpl_image_delete(crossSpecRe); */
        /* cpl_image_delete(crossSpecIm); */
        
        /* cpl_image_delete(workChanIm2); */
        /* cpl_image_delete(workChanRe2); */
        
        /* cpl_image_delete(workChanIm); */
        /* cpl_image_delete(workChanRe); */
        
        /* crossSpecRe = cpl_image_new(nbBases, nbWlen, CPL_TYPE_DOUBLE); */
        /* crossSpecIm = cpl_image_new(nbBases, nbWlen, CPL_TYPE_DOUBLE); */

        /* workChanIm2 = cpl_image_new(nbBases, nbWlen, CPL_TYPE_DOUBLE); */
        /* workChanRe2 = cpl_image_new(nbBases, nbWlen, CPL_TYPE_DOUBLE); */

        /* workChanIm = cpl_image_new(nbBases, nbWlen, CPL_TYPE_DOUBLE); */
        /* workChanRe = cpl_image_new(nbBases, nbWlen, CPL_TYPE_DOUBLE); */

	
        cpl_msg_info(cpl_func,"Computing differential visibilities");

        /* DIFFERENTIAL VISIBILITY COMPUTATION */
        cpl_msg_info(cpl_func,"Computing cross spectrum");
 
        /* /\* Initialize work channel, reference channel and cross spectrum *\/ */
        /* workChanIm                     = cpl_image_new(nbFreq, nbWlen, CPL_TYPE_FLOAT);*/
        /*                     workChanRe = cpl_image_new(nbFreq, nbWlen, CPL_TYPE_FLOAT);*/

        /* Loop on frames */
        cpl_msg_info(cpl_func,"Calculating correlated fluxes");
        for(iFrame = 0;iFrame<nbFrameTarget;iFrame++)
	{
            if (iFrame % 100 == 0)
	    {
                cpl_msg_info(cpl_func,"Frame %d/%d", iFrame+1, nbFrameTarget);
	    }
     
            /* Get work channel frame */
            for(iWlen = 0;iWlen<nbWlen;iWlen++)
	    {
                /* Compute wavelength and peak frequencies */
                lambda = mat_get_lambda(dispCoef, iWlen + iWlen0);

                /* Pick-up phase at the peak frequency and store it */
                for(iBase=0;iBase<nbBases;iBase++)
		{
                    /*  Get the peak pixel (real number) */
                    uPixel = mat_get_pos_fringe_peak(lambda,detNum,resolution,iBase);
                    val	   = floor(uPixel);
                    val1   = uPixel - val;
                    val2   = 1.0 - val1;

                    /*  Interpolate peak value (real part) */
                    tmp = corrFlux->list_corr[iReg*nbFrames+iFrame]->imgreg[0];
                    valRe = 
                        val2 * cpl_image_get(tmp, val, iWlen+1, &pis) +
                        val1 * cpl_image_get(tmp, val+1, iWlen+1, &pis);

                    /* Interpolate peak value (imaginary part) */
                    tmp = corrFlux->list_corr[iReg*nbFrames+iFrame]->imgreg[1];
                    valIm = 
                        val2 * cpl_image_get(tmp, val, iWlen+1, &pis) +
                        val1 * cpl_image_get(tmp, val+1, iWlen+1, &pis);
 
                    /* Store the result in the work channel  */
                    cpl_image_set(workChanRe,iBase+1,iWlen+1, valRe);
                    cpl_image_set(workChanIm,iBase+1,iWlen+1, valIm);
                    
		}
	    }
            
            cptWave=0;
            /* Subtract differential phase from the correlated flux */
            for(iWlen =	0;iWlen<nbWlen;iWlen++)
	    { 
                for(iBase = 0;iBase<nbBases;iBase++)
		{
                    /* Extract correlated flux (work channel value) */
                    cfluxRe = cpl_image_get(workChanRe,iBase+1,iWlen+1,&pis);
                    cfluxIm = cpl_image_get(workChanIm,iBase+1,iWlen+1,&pis);
 
                    /* Get the differential phase */
                    phi = cpl_image_get(phidiff,iBase+1,iWlen+1,&pis);
                    /* Buildup a phasor with 1 amplitude */
                    /* out of the differential phase */
                    phasorRe =  cos(phi);
                    phasorIm = -sin(phi);
 
                    /* Subtract differential phase phasor from the complex
                       coherent flux to obtain a clean coherent flux w1 */
                    /* Real part */
                    valRe = cfluxRe * phasorRe - cfluxIm * phasorIm;
                    cpl_image_set(w1Re, iBase+1, iWlen+1, valRe);
                    /* Imaginary part */
                    valIm = cfluxIm * phasorRe + cfluxRe * phasorIm;
                    cpl_image_set(w1Im,iBase+1,iWlen+1,valIm);
                    		
                    if (lambda >= lambdamin && lambda<=lambdamax)
		    {
                        /* Store the result in the work channel for the
                         * reference channel computation */
                        cpl_image_set(workChanRe2, iBase+1, cptWave+1, valRe);
                        cpl_image_set(workChanIm2, iBase+1, cptWave+1, valIm);
		    }
		}
                cptWave++;
	    }
	
            /* Remove residual phase from the correlated flux */
            imrow2    = cpl_image_collapse_create(workChanRe2, 0);
            imrow3    = cpl_image_collapse_create(workChanIm2, 0);
            
            cptWave=0;
            for(iWlen = 0;iWlen<nbWlen;iWlen++)
	    { 
                for(iBase = 0;iBase<nbBases;iBase++)
		{
                    phi      = atan2(cpl_image_get(imrow3,iBase+1,1,&pis),
                                     cpl_image_get(imrow2,iBase+1,1,&pis));
                    phasorRe = cos(phi);
                    phasorIm = sin(phi);
                    cfluxRe  = cpl_image_get(w1Re,iBase+1,iWlen+1,&pis);
                    cfluxIm  = cpl_image_get(w1Im,iBase+1,iWlen+1,&pis);
                    valRe    = cfluxRe * phasorRe + cfluxIm * phasorIm;
                    cpl_image_set(w1Re, iBase+1, iWlen+1, valRe);
                    /* Imaginary part */
                    valIm    = cfluxIm * phasorRe - cfluxRe * phasorIm;
                    cpl_image_set(w1Im,iBase+1,iWlen+1,valIm);

                    if (lambda >= lambdamin && lambda<=lambdamax)
		    {
                        /* Store the result in the work channel for the
                         * reference channel computation */
                        cpl_image_set(workChanRe2, iBase+1, cptWave+1, valRe);
                        cpl_image_set(workChanIm2, iBase+1, cptWave+1, valIm);
		    }

                    /**********************************/
                    /* Phase error: store the average */
                    /**********************************/
                    phi = atan2(valIm, valRe);
      
                    if(iFrame == 0)
		    {
                        cpl_image_set(phi_0,   iBase+1, iWlen+1, phi);
		    }
                    else
		    {
                        val1 = cpl_image_get(phi_0,iBase+1,iWlen+1,&pis);
                        val  = cpl_image_get(phi_SUM,iBase+1,iWlen+1,&pis)+
                            phi-val1;
                        cpl_image_set(phi_SUM, iBase+1, iWlen+1, val);
                        val  = cpl_image_get(phi_SUMSQ,iBase+1,iWlen+1,&pis)+
                            (phi-val1)*(phi-val1);
                        cpl_image_set(phi_SUMSQ, iBase+1, iWlen+1, val);
		    }
		}
                cptWave++;
	    }
     
	    /* Here, w1re and w1im are the correlated fluxes according to WJ */
            /* Add the values of cross spectrum of all frames */
	    //            cpl_image_add(cfxAmp, w1Re);
	    
            /* Make some space */
            cpl_image_delete(imrow2);
            cpl_image_delete(imrow3);


            /* compute reference channel as sum of all channels except
               the work channel 
               First collapse interspectrum vector
               along the wavelength direction
            */
            vecrow2 = cpl_image_collapse_create(workChanRe2, 0);
            vecrow3 = cpl_image_collapse_create(workChanIm2, 0);
 
            /* Loop on wavelengths and subtract work channel */
            for(iWlen = 0;iWlen<nbWlen;iWlen++)
	    {
                vecrow	   = cpl_vector_new_from_image_row(w1Re, iWlen+1);
                imrow	   = cpl_image_new(nbBases,1,CPL_TYPE_DOUBLE);
                for(iBase  = 0;iBase<nbBases;iBase++)
                    cpl_image_set(imrow, iBase+1, 1,
                                  cpl_vector_get(vecrow, iBase));
                refChanReJ = cpl_image_subtract_create(vecrow2, imrow);
                cpl_vector_delete(vecrow);
                cpl_image_delete(imrow);
      
                vecrow    = cpl_vector_new_from_image_row(w1Im, iWlen+1);
                imrow     = cpl_image_new(nbBases,1,CPL_TYPE_DOUBLE);
                for(iBase = 0;iBase<nbBases;iBase++)
                    cpl_image_set(imrow, iBase+1, 1,
                                  cpl_vector_get(vecrow, iBase));
                refChanImJ = cpl_image_subtract_create(vecrow3, imrow);
                cpl_vector_delete(vecrow);
                cpl_image_delete(imrow);
 
                /* Put back the array into reference channel image */
                for(iBase = 0;iBase<nbBases;iBase++)
		{
                    valRe = cpl_image_get(refChanReJ, iBase+1, 1, &pis);
                    valIm = cpl_image_get(refChanImJ, iBase+1, 1, &pis);
                    /* Do that pixel by pixel since it is not possible to put */
                    /* directly a cpl vector in a row or column of an image...*/
                    cpl_image_set(refChanRe, iBase+1, iWlen+1,  valRe);
                    cpl_image_set(refChanIm, iBase+1, iWlen+1, -valIm);
      
		}
                cpl_image_delete(refChanReJ);
                cpl_image_delete(refChanImJ);
	    }
            cpl_image_delete(vecrow2);
            cpl_image_delete(vecrow3);
     
            /* Compute the cross product and add them up for all frames */
            img1 = cpl_image_multiply_create(w1Re, refChanRe);
            img2 = cpl_image_multiply_create(w1Im, refChanIm);
            cpl_image_subtract(img1,img2);
  
            /* Add the values of cross spectrum of all frames */
            cpl_image_add(crossSpecRe, img1);
            /* Cleanup images */
            cpl_image_delete(img1);
            cpl_image_delete(img2);
 
            img1 = cpl_image_multiply_create(w1Im, refChanRe);
            img2 = cpl_image_multiply_create(w1Re, refChanIm);
            cpl_image_add(img1,img2);
            /* Add the values of cross spectrum of all frames */
            cpl_image_add(crossSpecIm,img1);
            /* Cleanup images */
            cpl_image_delete(img1);
            cpl_image_delete(img2);
	}
 
        cpl_msg_info(cpl_func,"Computing phase errors");
        /*************************************************/
        /* Get the phase average for the RMS computation */
        /*************************************************/
 
        /* Compute the RMS */
        for(iWlen = 0;iWlen<nbWlen;iWlen++)
	{
            for(iBase = 0;iBase<nbBases;iBase++)
	    {
                val1 = cpl_image_get(phi_SUM,iBase+1,iWlen+1,&pis);
                val2 = cpl_image_get(phi_SUMSQ,iBase+1,iWlen+1,&pis);
                cpl_image_set(phi_RMS, iBase+1, iWlen+1,
                              sqrt((val2 - val1*val1/nbFrameTarget)/(nbFrameTarget-1)));
	    }
	}
        /* Conversion in real phase RMS */
        mat_get_err_phi(phi_RMS, phidifferr);

        /* Divide RMS by square root of number of frames */
	for(iWlen = 0;iWlen<nbWlen;iWlen++)
	{
	    for(iBase=0;iBase<nbBases;iBase++)
	    {     
		cpl_image_set(phidifferr, iBase+1, iWlen+1, 
			      cpl_image_get(phidifferr,iBase+1,iWlen+1,&pis) /
			      sqrt(nbFrameTarget));
	    }
	}
	
	if(phot != NULL)
	{
	    cpl_msg_info(cpl_func,"Getting photometries");
 
	    /* Photometries */
	    /* Define nbWlen and nbFreq according to dimensions of the images */
	    nbPix  = corrFlux->imgdet->list_region[iReg]->naxis[0];
	    nbWlen = corrFlux->imgdet->list_region[iReg]->naxis[1];
 
	    //        if (phot	!= NULL)
	    //        {
	    prodPhot    = cpl_calloc(nbBases*nbFramesPhot,sizeof(cpl_image *));
	    prodPhotErr	 = cpl_calloc(nbBases,sizeof(cpl_image *));
 
	    /* Loop on frames */
	    for(iFrame = 0;iFrame<nbFramesPhot;iFrame++)
	    {

		if (strstr(mode,"s") == NULL && iFrame == 1)
		{

		    /* In the HighSens case, we use the second frame
		       (iFrame=1) which contains the stdev of each pixel
		       of the photometry and we compute the error of the
		       product of 2 photometric channels
		       sig=sqrt(p1**2*sig_p2**2+p2**2*sig_p1**2). We
		       store the results in prodPhotErr. */
		    for(iBase = 0;iBase<nbBases;iBase++)
		    {
			prodPhotErr[iBase] = cpl_image_new(1,nbWlen,CPL_TYPE_DOUBLE);
		    }
		    /* HIGH SENS, only to compute the error */
		    for(iWlen = 0;iWlen<nbWlen;iWlen++)
		    {
			phot1 = phot->list_phot[(iReg*4)*nbFramesPhot]->imgreg[0];
			phot2 = phot->list_phot[(iReg*4+1)*nbFramesPhot]->imgreg[0];
			phot3 = phot->list_phot[(iReg*4+2)*nbFramesPhot]->imgreg[0];
			phot4 = phot->list_phot[(iReg*4+3)*nbFramesPhot]->imgreg[0];
   
			/* 1st peak : T3T4 3D */
			val=0.;
			for(iPix = 0;iPix<nbPix;iPix++)
			{
			    val += 
				pow(cpl_image_get(phot3, iPix+1,iWlen+1,&pis),2.0)*
				pow(cpl_image_get(phot4, iPix+1,iWlen+1,&pis),2.0)+
				pow(cpl_image_get(phot4, iPix+1,iWlen+1,&pis),2.0)*
				pow(cpl_image_get(phot3, iPix+1,iWlen+1,&pis),2.0);
			}
			if(!strcmp(detName, "AQUARIUS"))
			{
			    cpl_image_set(prodPhotErr[0],1,iWlen+1,sqrt(val));
			}
			else
			{
			    cpl_image_set(prodPhotErr[0],1,iWlen+1,sqrt(val));
			}
   
			/* 2nd peak : T1T2 6D */
			val = 0.;
			for(iPix = 0;iPix<nbPix;iPix++)

			{
			    val += 
				pow(cpl_image_get(phot1, iPix+1,iWlen+1,&pis),2.0)*
				pow(cpl_image_get(phot2, iPix+1,iWlen+1,&pis),2.0)+
				pow(cpl_image_get(phot2, iPix+1,iWlen+1,&pis),2.0)*
				pow(cpl_image_get(phot1, iPix+1,iWlen+1,&pis),2.0);
			}
			if(!strcmp(detName, "AQUARIUS"))
			{
			    cpl_image_set(prodPhotErr[1],1,iWlen+1,sqrt(val));
			}
			else
			{
			    cpl_image_set(prodPhotErr[1],1,iWlen+1,sqrt(val));
			}
			/* 3rd peak : T2T3 9D */
			val      = 0.;
			for(iPix = 0;iPix<nbPix;iPix++)
			{
			    val += 
				pow(cpl_image_get(phot2, iPix+1,iWlen+1,&pis),2.0)*
				pow(cpl_image_get(phot3, iPix+1,iWlen+1,&pis),2.0)+
				pow(cpl_image_get(phot3, iPix+1,iWlen+1,&pis),2.0)*
				pow(cpl_image_get(phot2, iPix+1,iWlen+1,&pis),2.0);
			}
			if(!strcmp(detName, "AQUARIUS"))
			{
			    cpl_image_set(prodPhotErr[2],1,iWlen+1,sqrt(val));
			}
			else
			{
			    cpl_image_set(prodPhotErr[3],1,iWlen+1,sqrt(val));
			}
			/* 4th peak : T2T4 12D */
			val=0.;
			for(iPix = 0;iPix<nbPix;iPix++)
			{
			    val += 
				pow(cpl_image_get(phot4, iPix+1,iWlen+1,&pis),2.0)*
				pow(cpl_image_get(phot2, iPix+1,iWlen+1,&pis),2.0)+
				pow(cpl_image_get(phot2, iPix+1,iWlen+1,&pis),2.0)*
				pow(cpl_image_get(phot4, iPix+1,iWlen+1,&pis),2.0);
			}
			if(!strcmp(detName, "AQUARIUS"))
			{
			    cpl_image_set(prodPhotErr[3],1,iWlen+1,sqrt(val));
			}
			else
			{
			    cpl_image_set(prodPhotErr[2],1,iWlen+1,sqrt(val));
			}
			/* 5th peak : T1T3 15D */
			val      = 0.;
			for(iPix = 0;iPix<nbPix;iPix++)
			{
			    val += 
				pow(cpl_image_get(phot1, iPix+1,iWlen+1,&pis),2.0)*
				pow(cpl_image_get(phot3, iPix+1,iWlen+1,&pis),2.0)+
				pow(cpl_image_get(phot3, iPix+1,iWlen+1,&pis),2.0)*
				pow(cpl_image_get(phot1, iPix+1,iWlen+1,&pis),2.0);
			}
			if(!strcmp(detName, "AQUARIUS"))
			{
			    cpl_image_set(prodPhotErr[4],1,iWlen+1,sqrt(val));
			}
			else
			{
			    cpl_image_set(prodPhotErr[5],1,iWlen+1,sqrt(val));
			}
			/* 6th peak : T1T4 19D */
			val      = 0.;
			for(iPix = 0;iPix<nbPix;iPix++)
			{
			    val += 
				pow(cpl_image_get(phot1, iPix+1,iWlen+1,&pis),2.0)*
				pow(cpl_image_get(phot4, iPix+1,iWlen+1,&pis),2.0)+
				pow(cpl_image_get(phot4, iPix+1,iWlen+1,&pis),2.0)*
				pow(cpl_image_get(phot1, iPix+1,iWlen+1,&pis),2.0);
			}
			if(!strcmp(detName, "AQUARIUS"))
			{
			    cpl_image_set(prodPhotErr[5],1,iWlen+1,sqrt(val));
			}
			else
			{
			    cpl_image_set(prodPhotErr[4],1,iWlen+1,sqrt(val));
			}
		    }
		}

		/* }
		   else
		   { */
		/* SCI PHOT for all frames and HIGHSENS for k  = 0 */
		if (strstr(mode,"s") != NULL || iFrame!=1)
		{
		    phot1 = phot->list_phot[(iReg*4)*nbFramesPhot+iFrame]->imgreg[0];
		    phot2 = phot->list_phot[(iReg*4+1)*nbFramesPhot+iFrame]->imgreg[0];
		    phot3 = phot->list_phot[(iReg*4+2)*nbFramesPhot+iFrame]->imgreg[0];
		    phot4 = phot->list_phot[(iReg*4+3)*nbFramesPhot+iFrame]->imgreg[0];
      
		    /* 1st peak : T3T4 3D */
		    foo = cpl_image_multiply_create(phot3, phot4);
		    if(!strcmp(detName, "AQUARIUS"))
		    {
			prodPhot[iFrame*nbBases] = cpl_image_collapse_create(foo,1);
		    }
		    else
		    {
			prodPhot[iFrame*nbBases] = cpl_image_collapse_create(foo,1);
		    }
		    cpl_image_delete(foo);
      
		    /* 2nd peak : T1T2 6D */
		    foo = cpl_image_multiply_create(phot1, phot2);
		    if(!strcmp(detName, "AQUARIUS"))
		    {
			prodPhot[iFrame*nbBases+1] = cpl_image_collapse_create(foo,1);
		    }
		    else
		    {
			prodPhot[iFrame*nbBases+1] = cpl_image_collapse_create(foo,1);
		    }
		    cpl_image_delete(foo);
      
		    /* 3rd peak : T2T3 9D */
		    foo                        = cpl_image_multiply_create(phot2, phot3);
		    if(!strcmp(detName, "AQUARIUS"))
		    {
			prodPhot[iFrame*nbBases+2]   = cpl_image_collapse_create(foo,1);
		    }
		    else
		    {
			prodPhot[iFrame*nbBases+3] = cpl_image_collapse_create(foo,1);
		    }
		    cpl_image_delete(foo);

		    /* 4th peak : T2T4 12D */
		    foo                            = cpl_image_multiply_create(phot2, phot4);
		    if(!strcmp(detName, "AQUARIUS"))
		    {
			prodPhot[iFrame*nbBases+3] = cpl_image_collapse_create(foo,1);
		    }
		    else
		    {
			prodPhot[iFrame*nbBases+2] = cpl_image_collapse_create(foo,1);
		    }
		    cpl_image_delete(foo);

		    /* 5th peak : T1T3 15D */
		    foo = cpl_image_multiply_create(phot1, phot3);
		    if(!strcmp(detName, "AQUARIUS"))
		    {
			prodPhot[iFrame*nbBases+4] = cpl_image_collapse_create(foo,1);
		    }
		    else
		    {
			prodPhot[iFrame*nbBases+5] = cpl_image_collapse_create(foo,1);
		    }
		    cpl_image_delete(foo);

		    /* 6th peak : T1T4 19D */
		    foo = cpl_image_multiply_create(phot1, phot4);
		    if(!strcmp(detName, "AQUARIUS"))
		    {
			prodPhot[iFrame*nbBases+5] = cpl_image_collapse_create(foo,1);
		    }
		    else
		    {
			prodPhot[iFrame*nbBases+4] = cpl_image_collapse_create(foo,1);
		    }
		    cpl_image_delete(foo);
      
		}
	    }
	    //        }
 
 
	    for(iBase = 0;iBase<nbBases;iBase++)
	    {
		//            if (phot != NULL)
		//            {
		/* Compute mean and stdev (in fact error of the mean)
		   of the photometry only in SCIPHOT mode */
		if (strstr(mode,"s") != NULL)
		{
		    prodPhotErr[iBase]=cpl_image_new(1,nbWlen,CPL_TYPE_DOUBLE);
		    vecw      = cpl_vector_new(nbFramesPhot);
		    for(iWlen = 0;iWlen<nbWlen;iWlen++)
		    {
			for(iFrame=0;iFrame<nbFramesPhot;iFrame++)
			{
			    cpl_vector_set(vecw, iFrame, cpl_image_get(prodPhot[iFrame*nbBases+iBase], 1,iWlen+1,&pis));
 
			}
			/* Replace first frame of prodPhot by the mean on frames */
			cpl_image_set(prodPhot[iBase],1,iWlen+1,
				      cpl_vector_get_mean(vecw));
			cpl_image_set(prodPhotErr[iBase],1,iWlen+1,
				      cpl_vector_get_stdev(vecw)/sqrt(nbFramesPhot));
		    }
		    cpl_vector_delete(vecw);
		}
	    }
	}

	// Run this part only if the photometry is present
	if(phot != NULL)
	{
	    cpl_msg_info(cpl_func,"Calculating differential visibilities");
	    for(iWlen = 0;iWlen<nbWlen;iWlen++)
	    {
		/* Compute wavelength and peak frequencies */ 
		for(iBase = 0;iBase<nbBases;iBase++)
		{
		    avgsqrtflux = 0;
		    /* here compute the average sqrt flux */
		    for(jWlen= 0;jWlen<nbWlen;jWlen++)
		    {
			lambda = mat_get_lambda(dispCoef, jWlen + iWlen0);
                    
			if (lambda>=lambdamin && lambda	<= lambdamax)
			    /* This second loop is to integrate reference channel */
			    /* Avoid the wavelength of work channel */
			    if(jWlen != iWlen)
			    {
				flux         = cpl_image_get(prodPhot[iBase],1,jWlen+1,&pis);
				sqrtflux     = ((flux > 0) - (flux < 0))*pow(fabs(flux),0.5);
				avgsqrtflux += sqrtflux;
			    }
		    }
		    flux     = cpl_image_get(prodPhot[iBase],1,iWlen+1,&pis);
		    sqrtflux = ((flux > 0) - (flux < 0))*pow(fabs(flux),0.5);

		    cflux = cpl_image_get(crossSpecRe,iBase+1,iWlen+1,&pis);
		    /* The factor 3.4 is due to the fact that we use
		       the maximum of each fringe peak whereas the
		       photometric correction corresponds to the
		       energy of the fringe peak. We compensate by
		       multiplying the cflux by the integral of a
		       normalized "Chapeau chinois" function
		       (approximatively 3.4). Even if this factor is
		       not exactly the good one, it will disappear
		       during the calibration with calibrator
		       stars. */
		    vis = 3.4 * cflux / sqrtflux / avgsqrtflux / nbFreq / nbFrameTarget;
		    cpl_image_set(visdiff,iBase+1,iWlen+1,vis);
		}
	    }
	}
	else
	{
	    cpl_msg_info(cpl_func,"WARNING: No	photometries = No visibilities");
	}

	
        cpl_msg_info(cpl_func,"Saving differential	phases, correlated fluxes and visibilities");
 
        /* Convert phases to degrees */
        cpl_image_multiply_scalar(phidiff,    CPL_MATH_DEG_RAD);
        cpl_image_multiply_scalar(phidifferr, CPL_MATH_DEG_RAD);
	
        cptWave=0;
        /* /\* Fill oifits with estimated phidiff *\/ */
        /* for(iWlen = 0;iWlen<nbWlen;iWlen++) */
	/* { */
	/*     for(k = 0;k<oifits->oivis->nbvis;k++) */
	/*     { */
	/* 	cpl_msg_info(cpl_func,"cfx value %f",cpl_image_get(cfxAmp,   k+1,iWlen+1,&pis)); */
	/*     } */
	/* } */


	
        /* Fill oifits with estimated phidiff */
        for(iWlen = 0;iWlen<nbWlen;iWlen++)
			{
            /* printf("BCD status %s\n",bcd2Status); */
				if (oifits->oivis->nbvis == nbBases)
					{
						for(k = 0;k<oifits->oivis->nbvis;k++)
							{
							//cpl_msg_info(cpl_func,"Saving diff phi");
							/* printf(" %d\n",oifits->oivis->nbvis); */
							/* Set differential phase for 6 tels */
								oifits->oivis->list_vis[k]->visphi[cptWave] = 
									cpl_image_get(phidiff,   k+1,iWlen+1,&pis);
                    
								oifits->oivis->list_vis[k]->visphierr[cptWave] = 
									fabs(cpl_image_get(phidifferr,k+1,iWlen+1,&pis));

							//cpl_msg_info(cpl_func,"Saving cfx");
							/* oifits->oivis->list_vis[k]->cfxamp[cptWave]    = 
							cpl_image_get(cfxAmp,   k+1,iWlen+1,&pis);
							*/
  								if (detNum == 1)
									{
										if (fabs(oifits->oiwave->effwave[cptWave]*1.e6 - 3.5) <= 0.7 || fabs(oifits->oiwave->effwave[cptWave]*1.e6 - 4.84) <= 0.36)
											{
												oifits->oivis->list_vis[k]->bandflag[cptWave] = CPL_FALSE;
											}
										else
											{
												oifits->oivis->list_vis[k]->bandflag[cptWave] = CPL_TRUE;
											}
									}
								else
									{
										if (fabs(oifits->oiwave->effwave[cptWave]*1.e6 - 8.59) <= 0.59 || fabs(oifits->oiwave->effwave[cptWave]*1.e6 - 11.28) <= 1.72)
											{
												oifits->oivis->list_vis[k]->bandflag[cptWave] = CPL_FALSE;
											}
										else
											{
												oifits->oivis->list_vis[k]->bandflag[cptWave] = CPL_TRUE;
											}
									}
								
								
								
								if(phot != NULL)
									{
									/* Set differential visibility */
									//cpl_msg_info(cpl_func,"Saving visamp");
										oifits->oivis->list_vis[k]->visamp[cptWave] = 
											cpl_image_get(visdiff,   k+1,iWlen+1,&pis);
				
										oifits->oivis->list_vis[k]->visamperr[cptWave] = 
											fabs(cpl_image_get(visdifferr,k+1,iWlen+1,&pis));
				
									//cpl_msg_info(cpl_func,"Saving visamp");
									/*oifits->oivis->list_vis[k]->bandflag[cptWave] = 
									(cpl_image_get(flag, k+1,iWlen+1,&pis)	||
									cpl_image_get(phidifferr,k+1,iWlen+1,&pis) > 60)
									? CPL_TRUE : CPL_FALSE;*/
									}
								else
									cpl_msg_info(cpl_func,"Skipped visamp without photometries");
									// printf(" %d\n",cpl_image_get(flag,k+1,iWlen+1,&pis));
							}
					}
					
				else if (oifits->oivis->nbvis == 3)
	    {
                /* Set differential phase for 3 tels */
                oifits->oivis->list_vis[0]->visphi[cptWave] = 
                    cpl_image_get(phidiff,2,iWlen+1,&pis);
                
                oifits->oivis->list_vis[0]->visphierr[cptWave] = 
		  fabs(cpl_image_get(phidifferr,2,iWlen+1,&pis));

		if(phot != NULL)
		{
		    /* Set differential visibility */
		    oifits->oivis->list_vis[0]->visamp[cptWave]    = 
			cpl_image_get(visdiff,2,iWlen+1,&pis);
		    oifits->oivis->list_vis[0]->visamperr[cptWave] = 
		      fabs(cpl_image_get(visdifferr,2,iWlen+1,&pis));

			oifits->oivis->list_vis[0]->bandflag[cptWave] = CPL_FALSE;
		    /*oifits->oivis->list_vis[0]->bandflag[cptWave] =	
			(cpl_image_get(flag,2,iWlen+1,&pis) ||
			 cpl_image_get(phidifferr,2,iWlen+1,&pis) > 60)
			? CPL_TRUE : CPL_FALSE;*/
		}
		else
		    cpl_msg_info(cpl_func,"Skipped visamp without photometries (3 bases)");

                if (strcmp(bcdStatus,"OUT"))
		{
                    /* Treat the BCD case */
                    oifits->oivis->list_vis[1]->visphi[cptWave]    = 
                        cpl_image_get(phidiff,3,iWlen+1,&pis);
                    oifits->oivis->list_vis[1]->visphierr[cptWave] = 
		      fabs(cpl_image_get(phidifferr,3,iWlen+1,&pis));

		    if(phot != NULL)
		    {
			/* differential visibility */
			oifits->oivis->list_vis[1]->visamp[cptWave] = 
			    cpl_image_get(visdiff,3,iWlen+1,&pis);
			oifits->oivis->list_vis[1]->visamperr[cptWave] = 
			  fabs(cpl_image_get(visdifferr,3,iWlen+1,&pis));

			oifits->oivis->list_vis[1]->bandflag[cptWave] = CPL_FALSE;
			/*oifits->oivis->list_vis[1]->bandflag[cptWave] = 
			    (cpl_image_get(flag,3,iWlen+1,&pis)	||
			     cpl_image_get(phidifferr,3,iWlen+1,&pis) > 60)
			    ? CPL_TRUE : CPL_FALSE;*/
		    }
		    else
			cpl_msg_info(cpl_func,"Skipped visamp without photometries (BCD out)");

                    /* differential phase */
                    oifits->oivis->list_vis[2]->visphi[cptWave]    = 
                        cpl_image_get(phidiff,5,iWlen+1,&pis);
                    oifits->oivis->list_vis[2]->visphierr[cptWave] = 
		      fabs(cpl_image_get(phidifferr,5,iWlen+1,&pis));

		    if(phot != NULL)
		    {
			/* differential visibility */
			oifits->oivis->list_vis[2]->visamp[cptWave]	   = 
			    cpl_image_get(visdiff,5,iWlen+1,&pis);
			oifits->oivis->list_vis[2]->visamperr[cptWave] = 
			  fabs(cpl_image_get(visdifferr,5,iWlen+1,&pis));

			oifits->oivis->list_vis[1]->bandflag[cptWave] = CPL_FALSE;
			/*oifits->oivis->list_vis[1]->bandflag[cptWave] = 
			    (cpl_image_get(flag,5,iWlen+1,&pis)	||
			     cpl_image_get(phidifferr,5,iWlen+1,&pis) > 60)
			    ? CPL_TRUE : CPL_FALSE;*/
		    }
		    else
			cpl_msg_info(cpl_func,"Skipped visamp without photometries (3 bases)");
		}
		else
		{
		    /* Treat the BCD case differential phase */
		    oifits->oivis->list_vis[1]->visphi[cptWave]    = 
			cpl_image_get(phidiff,4,iWlen+1,&pis);
		    oifits->oivis->list_vis[1]->visphierr[cptWave] = 
		      fabs(cpl_image_get(phidifferr,4,iWlen+1,&pis));

		    /* differential visibility */
		    oifits->oivis->list_vis[1]->visamp[cptWave]    = 
			cpl_image_get(visdiff,4,iWlen+1,&pis);
		    oifits->oivis->list_vis[1]->visamperr[cptWave] = 
		      fabs(cpl_image_get(visdifferr,4,iWlen+1,&pis));
                    
		    oifits->oivis->list_vis[1]->bandflag[cptWave] = CPL_FALSE;
		    /*oifits->oivis->list_vis[1]->bandflag[cptWave] = 
			(cpl_image_get(flag,4,iWlen+1,&pis)	||
			 cpl_image_get(phidifferr,4,iWlen+1,&pis) > 60)
			? CPL_TRUE : CPL_FALSE;*/

		    /* differential phase */
		    oifits->oivis->list_vis[2]->visphi[cptWave]    = 
			cpl_image_get(phidiff,6,iWlen+1,&pis);
		    oifits->oivis->list_vis[2]->visphierr[cptWave] = 
		      fabs(cpl_image_get(phidifferr,6,iWlen+1,&pis));

		    if(phot != NULL)
		    {
			/* differential visibility */
			oifits->oivis->list_vis[2]->visamp[cptWave] = 
			    cpl_image_get(visdiff,6,iWlen+1,&pis);
			oifits->oivis->list_vis[2]->visamperr[cptWave] = 
			  fabs(cpl_image_get(visdifferr,6,iWlen+1,&pis));

			oifits->oivis->list_vis[2]->bandflag[cptWave] = CPL_FALSE;
			/*oifits->oivis->list_vis[2]->bandflag[cptWave] = 
			    (cpl_image_get(flag,6,iWlen+1,&pis)	||
			     cpl_image_get(phidifferr,6,iWlen+1,&pis) > 60)
			    ? CPL_TRUE : CPL_FALSE;*/
		    }
		    else
			cpl_msg_info(cpl_func,"Skipped visamp without photometries (BCD prout)");
		}
        
	    }
	    else
	    {
		oifits->oivis->list_vis[0]->visphi[cptWave]    = 
		    cpl_image_get(phidiff,2,iWlen+1,&pis);
		oifits->oivis->list_vis[0]->visphierr[cptWave] = 
		  fabs(cpl_image_get(phidifferr,2,iWlen+1,&pis));
		oifits->oivis->list_vis[0]->bandflag[cptWave] =	CPL_FALSE;
		/*oifits->oivis->list_vis[0]->bandflag[cptWave] =	
		    (cpl_image_get(flag,2,iWlen+1,&pis) ||
		     cpl_image_get(phidifferr,2,iWlen+1,&pis) > 60)
		    ? CPL_TRUE : CPL_FALSE;*/
	    }
	    cptWave++;
	}

	cpl_msg_info(cpl_func,"Delete phi_sum");
	cpl_image_delete(phi_SUM);
	cpl_image_delete(phi_SUMSQ);
	cpl_image_delete(phi_RMS);
	cpl_image_delete(phi_0);
	cpl_image_delete(phidiff);
	cpl_image_delete(phidifferr);
	//cpl_image_delete(cfxAmp);
	cpl_image_delete(flag);
	cpl_msg_info(cpl_func,"Delete vis diff");
	cpl_image_delete(visdiff);
	cpl_image_delete(visdifferr);
	/* images */
	cpl_image_delete(workChanIm);
	cpl_image_delete(workChanRe);
	cpl_image_delete(workChanIm2);
	cpl_image_delete(workChanRe2);
	cpl_msg_info(cpl_func,"Delete ref chan");
	cpl_image_delete(refChanIm);
	cpl_image_delete(refChanRe);
	cpl_image_delete(crossSpecRe);
	cpl_image_delete(crossSpecIm);
	cpl_image_delete(w1Im);
	cpl_image_delete(w1Re);
	/* cpl_image_delete(w1Im_forleo); */
	/* cpl_image_delete(w1Re_forleo); */
 

    }


    cpl_msg_info(cpl_func,"Free structures");
    
    /*  Free all allocated structures */
    /* Vectors */
    cpl_vector_delete(dispCoef);

    if(prodPhot != NULL)
      {
	for(iFrame = 0;iFrame<nbBases*nbFramesPhot;iFrame++)
	  {
	    cpl_image_delete(prodPhot[iFrame]);
	  }
	cpl_free(prodPhot);
      }
    
    if(prodPhotErr != NULL)
      {
	for(iBase = 0;iBase<nbBases;iBase++)
	  {
	    cpl_image_delete(prodPhotErr[iBase]);
	  }
	cpl_free(prodPhotErr);
      }
    
    
    // QC parameter Computation
    for(iBase =	0;iBase<oifits->oivis->nbvis;iBase++)
    {
	val  = 0.;
	val2 = 0.;
	cptWave=0.;
	for(iWlen=0;iWlen<oifits->oivis->nbchannel;iWlen++)
	{
	    if (detNum == 1)
	    {
		if (fabs(oifits->oiwave->effwave[iWlen]*1.e6 - 3.5) <= 0.55 || fabs(oifits->oiwave->effwave[iWlen]*1.e6 - 4.75) <= 0.25)
		  //if (fabs(oifits->oiwave->effwave[iWlen]*1.e6 - 3.5) <= 0.7 || fabs(oifits->oiwave->effwave[iWlen]*1.e6 - 4.84) <= 0.36) 
		{
		    val+=oifits->oivis->list_vis[iBase]->visphi[iWlen];
		    val2+=pow(oifits->oivis->list_vis[iBase]->visphierr[iWlen],2.0);
		    cptWave++;
		}
	    }
	    else if (detNum == 2)
	    {
		if (fabs(oifits->oiwave->effwave[iWlen]*1.e6 - 9.0) <= 1.0)
		  //if (fabs(oifits->oiwave->effwave[iWlen]*1.e6 - 8.59) <= 0.59 || fabs(oifits->oiwave->effwave[iWlen]*1.e6 - 11.28) <= 1.72) 
		{
		    val	 += oifits->oivis->list_vis[iBase]->visphi[iWlen];
		    val2 += pow(oifits->oivis->list_vis[iBase]->visphierr[iWlen],2.0);
		    cptWave++;
		}
	    }
	    else
	    {
		cptWave	= 1;
		val=0.;
		val2=0.;
	    }

	}
	val/=cptWave;
	val2   /= cptWave;
	/* val /= oifits->oivis2->nbchannel; */
	/* val2/=oifits->oivis2->nbchannel; */
	if (val2 < 0) val2=fabs(val2);
	snprintf(kwd, 64, "ESO QC DET%d DPHASE%d",detNum,iBase+1);
	cpl_propertylist_append_double(oifits->keywords, kwd, val);
	snprintf(kwd, 64, "ESO QC DET%d DPHASE%d STDEV",detNum,iBase+1);
	cpl_propertylist_append_double(oifits->keywords, kwd, sqrt(val2));
    }
    
    // Add Generic QC parameters
    cpl_propertylist *qclist=NULL;
    qclist = cpl_propertylist_new();
    mat_add_generic_qc(oifits->keywords,qclist);
    cpl_propertylist_append(oifits->keywords,qclist);
    cpl_propertylist_delete(qclist);
    return(0);
}





/**
   @brief Compute the interspectrum and differential phase from
   @brief mat_corrflux structure and a mat_photbeams structure.
   @brief This function fills a mat_oifits with computed differential
   @brief phases in the way described by F. Millour et al's 
   @brief ("DIFFERENTIAL INTERFEROMETRY WITH THE AMBER/VLTI INSTRUMENT:
   @brief				 DESCRIPTION, PERFORMANCES AND ILLUSTRATION", EAS
   @brief Publications			 Series, Vol. 22, pp 379, 2006)
   @param corrFlux correlated flux structure
   @param phot photometry structure 
   @param oifits oifits containing the differential phases 
   @return 0 if no error
*/
int mat_compute_coherent_observable(mat_corrflux	*corrFlux,
				    mat_photbeams	*phot,
				    mat_oifits	        *oifits,
				    mat_oiopdwvpo	*oiopdwvpo)
{
    /* Ranges */
    int	iReg = 0,iFrame=0,iWlen=0,iWlen0=0,iBase=0;
    //int iPix=0;
    int	nbRegion, nbFrames;
    int	nbWlen;
    
    int	iWave=0,cptWave=0,iSpat=0;
    int	 nbWave, nbSpat;
    int nbFramesPhot=0;
    int	nbBases = 6;
    int		 pis	     = 0;
    cpl_vector	*dispCoef    = NULL;

    /* char	*insMode      = NULL;*/
    char	*sliderPhot   = NULL;
    char *detName	      = NULL;
    char	 key[15];
    char	 key2[17];
    char	 band;		//= "p";
    float	 lambdamin    = 0.;
    float	 lambdamax = 0.;
    float	  wlenmin	  = 1e99;
    float	  wlenmax	  = 0.;
    char  mode[3]        = " ";
 
    int   detNum	    = 0;
    int		 resolution = 0;
    char	*specResData = NULL;
    double lambda=0.;
    //double posFringePeak=0.;
    //double dOverLambda=0.;
    //int uPixelMin=0;
    //int uPixelMax=0;
    cpl_image *vis=NULL;
    cpl_image *viserr=NULL;
    cpl_image *visM=NULL;
    cpl_image *viserrM=NULL;
   cpl_vector **prodPhot=NULL;
    double val,val2;
    int sign;
    //double foore=0.;
    //double fooim=0.;
    //double mur, mui;
    //double phi=0.;
    //double alpha=0.;
    //cpl_vector *vecre=NULL;
    //cpl_vector *vecim=NULL;
    //cpl_vector *vecreNoise=NULL;
    //cpl_vector *vecimNoise=NULL;
    char	 kwd[64];
    
   


    /* Check input parameters */
    mat_assert((corrFlux != NULL),CPL_ERROR_NULL_INPUT, "no mat_corrflux argument given");
    mat_assert((oifits != NULL),CPL_ERROR_NULL_INPUT, "no mat_oifits argument given");

    /* Get the array dimensions */
    nbRegion = corrFlux->imgdet->nbregion;
    nbFrames = corrFlux->nbframe / nbRegion;

    int nbFrameTarget=nbFrames;
    if (cpl_propertylist_has(corrFlux->keywords,"PRO FRAME TARGET")) {
      nbFrameTarget=cpl_propertylist_get_int(corrFlux->keywords,"PRO FRAME TARGET");
    }

    if(phot != NULL)
	nbFramesPhot = phot->nbphotxframe / phot->imgdet->nbregion;
    
    /* Read the detector type (and hence band) */
    detName = (char *)cpl_propertylist_get_string(corrFlux->keywords,"ESO DET CHIP NAME");
    if (detName == NULL)
    {
        cpl_msg_error(cpl_func,
                      "no ESO DET CHIP NAME keyword in frame");
        return -1;
    }
    mat_identify_det(detName, &band, &detNum);


    /* Read the spectral resolution */
    sprintf(key, "ESO INS DI%c ID",band);
    specResData = (char *)cpl_propertylist_get_string(corrFlux->keywords,key);
    char *fil = NULL;
    fil = (char *)cpl_propertylist_get_string(corrFlux->keywords,"ESO INS FIL NAME");
    mat_identify_res(specResData, fil, detNum, &resolution);

      
    
    /* Loading of the dispersion coefficients */
    dispCoef = cpl_vector_new(N_DEG_DISPERSION_LAW+1);
    cpl_vector_set(dispCoef,0,cpl_propertylist_get_double(corrFlux->keywords,"PRO DISP COEF0"));
    cpl_vector_set(dispCoef,1,cpl_propertylist_get_double(corrFlux->keywords,"PRO DISP COEF1"));
    cpl_vector_set(dispCoef,2,cpl_propertylist_get_double(corrFlux->keywords,"PRO DISP COEF2"));
    cpl_vector_set(dispCoef,3,cpl_propertylist_get_double(corrFlux->keywords,"PRO DISP COEF3"));
    cpl_vector_set(dispCoef,4,cpl_propertylist_get_double(corrFlux->keywords,"PRO DISP COEF4"));



      
    /* Set the useful range of wavelengths to compute OPD */
    /* First of all retrieve the actual wavelength range */
    nbRegion = corrFlux->imgdet->nbregion;
    for(iReg = 0;iReg<nbRegion;iReg++)
      {
	cpl_msg_info(cpl_func,"Determining wavelength range");
	nbWlen = corrFlux->imgdet->list_region[iReg]->naxis[1];
        iWlen0 = corrFlux->imgdet->list_region[iReg]->corner[1];
        for(iWlen = 0;iWlen<nbWlen;iWlen++)
	  {
            /* Compute wavelength and set wlen interval */
            lambda = mat_get_lambda(dispCoef, iWlen + iWlen0);
	    if(lambda < wlenmin)
	      wlenmin = lambda;
	    if(lambda > wlenmax)
	      wlenmax = lambda;
	  }
      }
    mat_identify_wlrng(detNum, wlenmin, wlenmax, resolution, &lambdamin, &lambdamax);



    cpl_msg_info(cpl_func,"Detector %d (%s)",detNum, detName);
    cpl_msg_info(cpl_func,"Band %c, from %1.1f to %1.1f microns", band, lambdamin, lambdamax);
    cpl_msg_info(cpl_func,"Spectral Resolution %d (%s)",resolution, specResData);

    /* Read the instrument mode */
    sprintf(key2, "ESO INS PI%c NAME",band);
    sliderPhot = (char *)cpl_propertylist_get_string(corrFlux->keywords,key2); 
    if (strcmp(sliderPhot, "PHOTO"))
    {
	cpl_msg_info(cpl_func,"HighSens mode");
	sprintf(mode,"h");
    }
    else
    {
	cpl_msg_info(cpl_func,"SiPhot mode");
	sprintf(mode,"s");
    }


    cptWave=0;
    for(iReg=0;iReg<nbRegion;iReg++)
      {	
	nbWave = corrFlux->imgdet->list_region[iReg]->naxis[1];
	nbSpat = corrFlux->imgdet->list_region[iReg]->naxis[0];
	vis=cpl_image_new(nbWave,nbBases,CPL_TYPE_DOUBLE);
	viserr=cpl_image_new(nbWave,nbBases,CPL_TYPE_DOUBLE);
	cpl_image ** imgcurrentReal = NULL;
	cpl_image ** imgcurrentImag = NULL;
	imgcurrentReal = cpl_calloc(nbFrameTarget,sizeof(cpl_image *));
	imgcurrentImag = cpl_calloc(nbFrameTarget,sizeof(cpl_image *));
	for(iFrame=0;iFrame<nbFrameTarget;iFrame++)
	  {
	    imgcurrentReal[iFrame] = cpl_image_new(nbWave,14,CPL_TYPE_DOUBLE);
	    imgcurrentImag[iFrame] = cpl_image_new(nbWave,14,CPL_TYPE_DOUBLE);
	  }
	cpl_vector *vecphiStatic[nbBases];
 	cpl_vector *vecphiStaticErr[nbBases];
	cpl_vector *vecphiStaticM[nbBases];
	cpl_vector *vecphiStaticErrM[nbBases];	

	
	// Extract Correlated Flux
	mat_get_correlated_flux(corrFlux,nbBases,nbFrameTarget,dispCoef,detNum,resolution,
				imgcurrentReal,imgcurrentImag,iReg);
	
	// Estimate VISAMP and VISPHI
	// We use the opd estimated in mat_est_opd if provided
	if (oiopdwvpo != NULL)
	  {
	    // Bootstrap Error
	    mat_get_error_vis(oiopdwvpo,nbBases,nbWave,nbFrameTarget,lambdamin,lambdamax,
			      imgcurrentReal,imgcurrentImag,viserr,vecphiStaticErr);
	    mat_get_visamp_visphi(imgcurrentReal,imgcurrentImag,oiopdwvpo,nbWave,nbBases,nbFrameTarget,
				  lambdamin,lambdamax,viserr,vecphiStatic,vis);
	  }
	// otherwise we recompute the opd
	else
	  {
	    // Bootstrap Error
	    mat_get_error_vis(NULL,nbBases,nbWave,nbFrameTarget,lambdamin,lambdamax,
			      imgcurrentReal,imgcurrentImag,viserr,vecphiStaticErr);
	    mat_get_visamp_visphi(imgcurrentReal,imgcurrentImag,NULL,nbWave,nbBases,nbFrameTarget,
				  lambdamin,lambdamax,viserr,vecphiStatic,vis);
	  }
	// For the M band in Low resolution, we compute the opd in M band
	if (resolution == 1 && wlenmax > 4.5 && wlenmin < 3.8)
	  {
	    visM=cpl_image_new(nbWave,nbBases,CPL_TYPE_DOUBLE);
	    viserrM=cpl_image_new(nbWave,nbBases,CPL_TYPE_DOUBLE);
	    // Bootstrap Error
	    mat_get_error_vis(NULL,nbBases,nbWave,nbFrameTarget,4.5,5.0,
			      imgcurrentReal,imgcurrentImag,viserrM,vecphiStaticErrM);
	
	    // Estimate VISAMP and VISPHI
	    mat_get_visamp_visphi(imgcurrentReal,imgcurrentImag,NULL,nbWave,nbBases,nbFrameTarget,
				  4.5,5.0,viserrM,vecphiStaticM,visM);
	    for(iWave=0;iWave<nbWave;iWave++)
	      {
		if (oifits->oiwave->effwave[iWave]*1.e6 >= 4.4)
		  {
		    for(int k = 0;k<nbBases;k++)
		      {
			cpl_image_set(vis,k+1,iWave+1,cpl_image_get(visM,k+1,iWave+1,&pis));
			cpl_image_set(viserr,k+1,iWave+1,cpl_image_get(viserrM,k+1,iWave+1,&pis));
			cpl_vector_set(vecphiStatic[k],iWave,cpl_vector_get(vecphiStaticM[k],iWave));
			cpl_vector_set(vecphiStaticErr[k],iWave,cpl_vector_get(vecphiStaticErrM[k],iWave));
		      }
		  }
	      }
	  }
	
	// Extract Photometry
	if (phot != NULL)
	  {
      	    prodPhot=cpl_calloc(nbBases,sizeof(cpl_vector *));
	    for(iBase=0;iBase<nbBases;iBase++)
	      {
		prodPhot[iBase]=cpl_vector_new(nbWave);
		for(iWave=0;iWave<nbWave;iWave++)
		  {
		    cpl_vector_set(prodPhot[iBase],iWave,0.);
		  }
	      }
	    double p01=0.,p02=0.,p03=0.,p12=0.,p13=0.,p23=0.;
	    for(iWave=0;iWave<nbWave;iWave++)
	      {
		p01=0.;
		p02=0.;
		p03=0.;
		p12=0.;
		p13=0.;
		p23=0.;
		for(iFrame=0;iFrame<nbFramesPhot;iFrame++)
		  {
		    if ((strstr(mode,"s") == NULL && iFrame==0) || (strstr(mode,"s") != NULL))
		      {
			for(iSpat=0;iSpat<nbSpat;iSpat++)
			  {
			    p01+=cpl_image_get(phot->list_phot[(iReg*4+0)*nbFramesPhot+iFrame]->imgreg[0],iSpat+1,iWave+1,&pis)*
			      cpl_image_get(phot->list_phot[(iReg*4+1)*nbFramesPhot+iFrame]->imgreg[0],iSpat+1,iWave+1,&pis);
			    p02+=cpl_image_get(phot->list_phot[(iReg*4+0)*nbFramesPhot+iFrame]->imgreg[0],iSpat+1,iWave+1,&pis)*
			      cpl_image_get(phot->list_phot[(iReg*4+2)*nbFramesPhot+iFrame]->imgreg[0],iSpat+1,iWave+1,&pis);
			    p03+=cpl_image_get(phot->list_phot[(iReg*4+0)*nbFramesPhot+iFrame]->imgreg[0],iSpat+1,iWave+1,&pis)*
			      cpl_image_get(phot->list_phot[(iReg*4+3)*nbFramesPhot+iFrame]->imgreg[0],iSpat+1,iWave+1,&pis);
			    p12+=cpl_image_get(phot->list_phot[(iReg*4+1)*nbFramesPhot+iFrame]->imgreg[0],iSpat+1,iWave+1,&pis)*
			      cpl_image_get(phot->list_phot[(iReg*4+2)*nbFramesPhot+iFrame]->imgreg[0],iSpat+1,iWave+1,&pis);
			    p13+=cpl_image_get(phot->list_phot[(iReg*4+1)*nbFramesPhot+iFrame]->imgreg[0],iSpat+1,iWave+1,&pis)*
			      cpl_image_get(phot->list_phot[(iReg*4+3)*nbFramesPhot+iFrame]->imgreg[0],iSpat+1,iWave+1,&pis);
			    p23+=cpl_image_get(phot->list_phot[(iReg*4+2)*nbFramesPhot+iFrame]->imgreg[0],iSpat+1,iWave+1,&pis)*
			      cpl_image_get(phot->list_phot[(iReg*4+3)*nbFramesPhot+iFrame]->imgreg[0],iSpat+1,iWave+1,&pis);
			  }
		      }
		  }
		if(!strcmp(detName, "AQUARIUS"))
		  {
		    //1st peak : T3T4 3D in L and 6D in N
		    if (p23 < 0)
		      sign=-1;
		    else
		      sign=1;
		    val=cpl_vector_get(prodPhot[1],iWave)+sign*sqrt(fabs(p23));
		    cpl_vector_set(prodPhot[1],iWave,val);
		    //2nd peak : T1T2 6D in L but 3D in N */
		    if (p01 < 0)
		      sign=-1;
		    else
		      sign=1;
		    val=cpl_vector_get(prodPhot[0],iWave)+sign*sqrt(fabs(p01));
		    cpl_vector_set(prodPhot[0],iWave,val);
		    //3rd peak : T2T3 12D
		    if (p12 < 0)
		      sign=-1;
		    else
		      sign=1;
		    val=cpl_vector_get(prodPhot[3],iWave)+sign*sqrt(fabs(p12));
		    cpl_vector_set(prodPhot[3],iWave,val);
		    //4th peak : T2T4 9D in L but 18D in N
		    if (p13 < 0)
		      sign=-1;
		    else
		      sign=1;
		    val=cpl_vector_get(prodPhot[5],iWave)+sign*sqrt(fabs(p13));
		    cpl_vector_set(prodPhot[5],iWave,val);
		    //5th peak : T1T3 18D in L but 9D in N
		    if (p02 < 0)
		      sign=-1;
		    else
		      sign=1;
		    val=cpl_vector_get(prodPhot[2],iWave)+sign*sqrt(fabs(p02));
		    cpl_vector_set(prodPhot[2],iWave,val);
		    //6th peak : T1T4 15D
		    if (p03 < 0)
		      sign=-1;
		    else
		      sign=1;
		    val=cpl_vector_get(prodPhot[4],iWave)+sign*sqrt(fabs(p03));
		    cpl_vector_set(prodPhot[4],iWave,val);
		  }
		else
		  {
		    p01/=nbFramesPhot;
		    p02/=nbFramesPhot;
		    p03/=nbFramesPhot;
		    p12/=nbFramesPhot;
		    p13/=nbFramesPhot;
		    p23/=nbFramesPhot;
		    //1st peak : T3T4 3D in L and 6D in N
		    if (p23 < 0)
		      sign=-1;
		    else
		      sign=1;
		    val=cpl_vector_get(prodPhot[0],iWave)+sign*sqrt(fabs(p23));
		    cpl_vector_set(prodPhot[0],iWave,val);
		    //2nd peak : T1T2 6D in L but 3D in N */
		    if (p01 < 0)
		      sign=-1;
		    else
		      sign=1;
		    val=cpl_vector_get(prodPhot[1],iWave)+sign*sqrt(fabs(p01));
		    cpl_vector_set(prodPhot[1],iWave,val);
		    //3rd peak : T2T3 12D
		    if (p12 < 0)
		      sign=-1;
		    else
		      sign=1;
		    val=cpl_vector_get(prodPhot[3],iWave)+sign*sqrt(fabs(p12));
		    cpl_vector_set(prodPhot[3],iWave,val);
		    //4th peak : T2T4 9D in L but 18D in N
		    if (p13 < 0)
		      sign=-1;
		    else
		      sign=1;
		    val=cpl_vector_get(prodPhot[2],iWave)+sign*sqrt(fabs(p13));
		    cpl_vector_set(prodPhot[2],iWave,val);
		    //5th peak : T1T3 18D in L but 9D in N
		    if (p02 < 0)
		      sign=-1;
		    else
		      sign=1;
		    val=cpl_vector_get(prodPhot[5],iWave)+sign*sqrt(fabs(p02));
		    cpl_vector_set(prodPhot[5],iWave,val);
		    //6th peak : T1T4 15D
		    if (p03 < 0)
		      sign=-1;
		    else
		      sign=1;
		    val=cpl_vector_get(prodPhot[4],iWave)+sign*sqrt(fabs(p03));
		    cpl_vector_set(prodPhot[4],iWave,val);
		  }
	      }
	  }
     // compute vis et viserr
	if (phot != NULL)
	  {
	    for(iBase=0;iBase<nbBases;iBase++)
	      {
		for(iWave=0;iWave<nbWave;iWave++)
		  {
		    if (cpl_vector_get(prodPhot[iBase],iWave) != 0.)
		      {
			
			val=(cpl_image_get(vis,iWave+1,iBase+1,&pis)/cpl_vector_get(prodPhot[iBase],iWave))/sqrt(nbSpat);
			val2=pow(cpl_image_get(viserr,iWave+1,iBase+1,&pis),2.0)/pow(cpl_vector_get(prodPhot[iBase],iWave),2.0)/nbSpat;
			val2=sqrt(val2);
		      }
		    else
		      {
			val=-1.;
			val2=1.E3;
		      }
		    if (isnan(val2))
		      {
			val2=1.E3;
		      }
		    if (isnan(val))
		      {
			val=0.;
		      }

		    cpl_image_set(vis,iWave+1,iBase+1,val);
		    cpl_image_set(viserr,iWave+1,iBase+1,fabs(val2));
		  }
	      }
	  }

	if (prodPhot != NULL)
	  {
	    for(iBase=0;iBase<nbBases;iBase++)
	      {
		cpl_vector_delete(prodPhot[iBase]);
	      }
	    cpl_free(prodPhot);
	  }
	for(iFrame=0;iFrame<nbFrameTarget;iFrame++)
	  {
	    cpl_image_delete(imgcurrentReal[iFrame]);
	    cpl_image_delete(imgcurrentImag[iFrame]);
	  }
	cpl_free(imgcurrentReal);
	cpl_free(imgcurrentImag);

	// Fill oifits with estimated vis
	for(iWave=0;iWave<nbWave;iWave++)
	  {
	    if (oifits->oivis->nbvis == nbBases)
	      {
		for(int k = 0;k<oifits->oivis->nbvis;k++)
		  {
		    oifits->oivis->list_vis[k]->visphi[cptWave]=cpl_vector_get(vecphiStatic[k],iWave)*CPL_MATH_DEG_RAD;
		    oifits->oivis->list_vis[k]->visphierr[cptWave]=fabs(cpl_vector_get(vecphiStaticErr[k],iWave)*CPL_MATH_DEG_RAD);
		    oifits->oivis->list_vis[k]->visamp[cptWave]=cpl_image_get(vis,iWave+1,k+1,&pis);
		    oifits->oivis->list_vis[k]->visamperr[cptWave]=fabs(cpl_image_get(viserr,iWave+1,k+1,&pis));
		    if (detNum == 1)
		      {
			if (fabs(oifits->oiwave->effwave[cptWave]*1.e6 - 3.5) <= 0.7 || fabs(oifits->oiwave->effwave[cptWave]*1.e6 - 4.84) <= 0.36)
			  {
			    oifits->oivis->list_vis[k]->bandflag[cptWave] = CPL_FALSE;
			  }
			else
			  {
			    oifits->oivis->list_vis[k]->bandflag[cptWave] = CPL_TRUE;
			  }
		      }
		    else
		      {
			if (fabs(oifits->oiwave->effwave[cptWave]*1.e6 - 8.59) <= 0.59 || fabs(oifits->oiwave->effwave[cptWave]*1.e6 - 11.28) <= 1.72)
			  {
			    oifits->oivis->list_vis[k]->bandflag[cptWave] = CPL_FALSE;
			  }
			else
			  {
			    oifits->oivis->list_vis[k]->bandflag[cptWave] = CPL_TRUE;
			  }
		      }
		  }
	      }
	    cptWave++;
	  }


	for(iBase=0;iBase<nbBases;iBase++)
	  {
	    cpl_vector_delete(vecphiStatic[iBase]);
	    cpl_vector_delete(vecphiStaticErr[iBase]);
	  }

	cpl_image_delete(vis);
	cpl_image_delete(viserr);

	if (visM != NULL)
	  {
	    for(iBase=0;iBase<nbBases;iBase++)
	      {
		cpl_vector_delete(vecphiStaticM[iBase]);
		cpl_vector_delete(vecphiStaticErrM[iBase]);
	      }
	  }

	cpl_image_delete(visM);
	cpl_image_delete(viserrM);	    
      }
    cpl_vector_delete(dispCoef);




    // QC parameter Computation
    for(iBase =	0;iBase<oifits->oivis->nbvis;iBase++)
      {
	val  = 0.;
	val2 = 0.;
	cptWave=0.;
	for(iWlen=0;iWlen<oifits->oivis->nbchannel;iWlen++)
	  {
	    if (detNum == 1)
	      {
		if (fabs(oifits->oiwave->effwave[iWlen]*1.e6 - 3.5) <= 0.55 || fabs(oifits->oiwave->effwave[iWlen]*1.e6 - 4.75) <= 0.25)
		  //if (fabs(oifits->oiwave->effwave[iWlen]*1.e6 - 3.5) <= 0.7 || fabs(oifits->oiwave->effwave[iWlen]*1.e6 - 4.84) <= 0.36) 
		  {
		    val+=oifits->oivis->list_vis[iBase]->visphi[iWlen];
		    val2+=pow(oifits->oivis->list_vis[iBase]->visphierr[iWlen],2.0);
		    cptWave++;
		  }
	      }
	    else if (detNum == 2)
	      {
		if (fabs(oifits->oiwave->effwave[iWlen]*1.e6 - 9.0) <= 1.0)
		  //if (fabs(oifits->oiwave->effwave[iWlen]*1.e6 - 8.59) <= 0.59 || fabs(oifits->oiwave->effwave[iWlen]*1.e6 - 11.28) <= 1.72) 
		  {
		    val	 += oifits->oivis->list_vis[iBase]->visphi[iWlen];
		    val2 += pow(oifits->oivis->list_vis[iBase]->visphierr[iWlen],2.0);
		    cptWave++;
		  }
	      }
	    else
	      {
		cptWave	= 1;
		val=0.;
		val2=0.;
	      }

	  }
	val/=cptWave;
	val2   /= cptWave;
	/* val /= oifits->oivis2->nbchannel; */
	/* val2/=oifits->oivis2->nbchannel; */
	if (val2 < 0) val2=fabs(val2);
	snprintf(kwd, 64, "ESO QC DET%d DPHASE%d",detNum,iBase+1);
	cpl_propertylist_append_double(oifits->keywords, kwd, val);
	snprintf(kwd, 64, "ESO QC DET%d DPHASE%d STDEV",detNum,iBase+1);
	cpl_propertylist_append_double(oifits->keywords, kwd, sqrt(val2));
      }

    // Add Generic QC parameters
    cpl_propertylist *qclist=NULL;
    qclist = cpl_propertylist_new();
    mat_add_generic_qc(oifits->keywords,qclist);
    cpl_propertylist_append(oifits->keywords,qclist);
    cpl_propertylist_delete(qclist);
    return(0);



    return 0;
}



void mat_get_correlated_flux(mat_corrflux *corrFlux,int nbBases,int nbFrameTarget,
			     cpl_vector *dispCoef,int detNum,int resolution,
			     cpl_image **imgcurrentReal,cpl_image **imgcurrentImag, int iReg)
{
  int iWave, iFrame, iBase, iPix;
  int nbWave, nbSpat, nbFrames, nbRegion;
  int pis = 0;
  double posFringePeak=0.;
  double dOverLambda=0.;
  int uPixelMin=0;
  int uPixelMax=0;
  double valre=0.;
  double valim=0.;
    double lambda=0.;
  cpl_vector *foo=NULL;

  nbWave = corrFlux->imgdet->list_region[iReg]->naxis[1];
  nbSpat = corrFlux->imgdet->list_region[iReg]->naxis[0];
  nbRegion = corrFlux->imgdet->nbregion;
  nbFrames = corrFlux->nbframe / nbRegion;
  foo=cpl_vector_new(10);
	
  for(iWave=0;iWave<nbWave;iWave++)
    {
      lambda=mat_get_lambda(dispCoef,iWave+corrFlux->imgdet->list_region[iReg]->corner[1]);
	    
      for(iFrame=0;iFrame<nbFrameTarget;iFrame++)
	{
	  cpl_image_set(imgcurrentReal[iFrame],iWave+1,1,lambda);
	  cpl_image_set(imgcurrentImag[iFrame],iWave+1,1,lambda);
	  for(iBase=1;iBase<=nbBases;iBase++)
	    {
	      posFringePeak=mat_get_pos_fringe_peak(lambda,detNum,resolution,iBase-1);
	      uPixelMin=(int)posFringePeak;
	      uPixelMax=(int)posFringePeak+1;
	      valre=cpl_image_get(corrFlux->list_corr[iReg*nbFrames+iFrame]->imgreg[0],uPixelMin+1,iWave+1,&pis)*(uPixelMax-posFringePeak)+
		cpl_image_get(corrFlux->list_corr[iReg*nbFrames+iFrame]->imgreg[0],uPixelMax+1,iWave+1,&pis)*(posFringePeak-uPixelMin);
	      valim=cpl_image_get(corrFlux->list_corr[iReg*nbFrames+iFrame]->imgreg[1],uPixelMin+1,iWave+1,&pis)*(uPixelMax-posFringePeak)+
		cpl_image_get(corrFlux->list_corr[iReg*nbFrames+iFrame]->imgreg[1],uPixelMax+1,iWave+1,&pis)*(posFringePeak-uPixelMin);
	      cpl_image_set(imgcurrentReal[iFrame],iWave+1,iBase+1,valre);
	      cpl_image_set(imgcurrentImag[iFrame],iWave+1,iBase+1,valim);


	      dOverLambda=(posFringePeak-nbSpat/2)/(3*iBase);
	      uPixelMin=(int)(posFringePeak+1.5*dOverLambda);
	      valre=cpl_image_get(corrFlux->list_corr[iReg*nbFrames+iFrame]->imgreg[0],uPixelMin+1,iWave+1,&pis);
	      valim=cpl_image_get(corrFlux->list_corr[iReg*nbFrames+iFrame]->imgreg[1],uPixelMin+1,iWave+1,&pis);
	      cpl_image_set(imgcurrentReal[iFrame],iWave+1,iBase+8,valre);
	      cpl_image_set(imgcurrentImag[iFrame],iWave+1,iBase+8,valim);

	    }
	  for(iPix=0;iPix<10;iPix++)
	    {
	      cpl_vector_set(foo,iPix,cpl_image_get(corrFlux->list_corr[iReg*nbFrames+iFrame]->imgreg[0]
						    ,iPix+1,iWave+1,&pis));
	    }
	  cpl_image_set(imgcurrentReal[iFrame],iWave+1,8,cpl_vector_get_stdev(foo));
	  for(iPix=0;iPix<10;iPix++)
	    {
	      cpl_vector_set(foo,iPix,cpl_image_get(corrFlux->list_corr[iReg*nbFrames+iFrame]->imgreg[1]
						    ,iPix+1,iWave+1,&pis));
	    }
	  cpl_image_set(imgcurrentImag[iFrame],iWave+1,8,cpl_vector_get_stdev(foo));
	}
    }
  cpl_vector_delete(foo);
}


void mat_get_error_vis(mat_oiopdwvpo *oiopdwvpo,int nbBases,int nbWave,int nbFrameTarget,double lambdamin, double lambdamax,
		       cpl_image **imgcurrentReal,cpl_image **imgcurrentImag,cpl_image *viserr,cpl_vector *vecphiStaticErr[])
{

  // Do not bootstrap on less than 2 frames
  if(nbFrameTarget < 2)
    {
      cpl_msg_warning(cpl_func, "Cannot bootstrap on less than two frames, setting errors to zero!");
      for(int iBase=0;iBase<nbBases;iBase++)
        {
          vecphiStaticErr[iBase]=cpl_vector_new(nbWave);
          for(int iWave=0;iWave<nbWave;iWave++)
            {
              cpl_image_set(viserr,iWave+1,iBase+1, FLT_MAX);
              cpl_vector_set(vecphiStaticErr[iBase],iWave, FLT_MAX);
            }
        }
      return;
    }

  // BOOTSTRAP FOR ERROR on vecPhiStatic and vis
  int iBase,iFrame,iWave;
  int pis = 0;
  int nBootStrap=5;
  int numFrameBootStrap=0;
  double phi=0.;
  double alpha=0.;
  cpl_vector *vecre=NULL;
  cpl_vector *vecim=NULL;
  cpl_vector *vecreNoise=NULL;
  cpl_vector *vecimNoise=NULL;
  double foore=0.;
  double fooim=0.;
  double mur, mui;
  //double phiBootStrap=0.;
  double reBootStrap[nbWave*nBootStrap*nbBases];
  double imBootStrap[nbWave*nBootStrap*nbBases];
  cpl_image ** imgcurrentRealBootStrap = NULL;
  cpl_image ** imgcurrentImagBootStrap = NULL;
  cpl_vector *vecphiStaticBootStrap[nbBases*nBootStrap];
  cpl_vector *vecphiStatictmp[nbBases];
  const gsl_rng_type * T;
  gsl_rng * r;
  gsl_rng_env_setup();
  T = gsl_rng_default;
  r = gsl_rng_alloc (T);

  cpl_vector *vecalpha[nbBases];
  cpl_vector *vecbeta[nbBases];
  cpl_msg_info(cpl_func,"Bootstrap method running for phase error estimation ...");

  // We use the opd computed in mat_est_opd
  if (oiopdwvpo != NULL)
    {
      int cpt=0;
      for(iBase=0;iBase<nbBases;iBase++)
	{
	  vecalpha[iBase]=cpl_vector_new(nbFrameTarget);
	  vecbeta[iBase]=cpl_vector_new(nbFrameTarget);
	}
      for(iFrame=0;iFrame<nbFrameTarget;iFrame++) {
	for(iBase=0;iBase<nbBases;iBase++) {
	  cpl_vector_set(vecalpha[iBase],iFrame,2.*CPL_MATH_PI*oiopdwvpo->oiopd->list_opd[cpt]->opd);
	  cpl_vector_set(vecbeta[iBase],iFrame,oiopdwvpo->oiopd->list_opd[cpt]->humoff);
	  cpt++;
	}
      }
      imgcurrentRealBootStrap = cpl_calloc(nbFrameTarget,sizeof(cpl_image *));
      imgcurrentImagBootStrap = cpl_calloc(nbFrameTarget,sizeof(cpl_image *));
      cpl_vector *vecalphaBootStrap[nbBases];
      cpl_vector *vecbetaBootStrap[nbBases];
      for(iBase=0;iBase<nbBases;iBase++)
	{
	  vecalphaBootStrap[iBase]=cpl_vector_new(nbFrameTarget);
	  vecbetaBootStrap[iBase]=cpl_vector_new(nbFrameTarget);
	}
      for(iBase=0;iBase<nbBases;iBase++)
	{
	  vecphiStaticErr[iBase]=cpl_vector_new(nbWave);
	  vecphiStatictmp[iBase]=cpl_vector_new(nbWave);
	}
      for(iFrame=0;iFrame<nbFrameTarget;iFrame++)
	{
	  imgcurrentRealBootStrap[iFrame] = cpl_image_new(nbWave,8,CPL_TYPE_DOUBLE);
	  imgcurrentImagBootStrap[iFrame] = cpl_image_new(nbWave,8,CPL_TYPE_DOUBLE);
	}

      for(int iBootStrap=0;iBootStrap<nBootStrap;iBootStrap++)
	{
	  for(iBase=0;iBase<nbBases;iBase++)
	    {
	      vecphiStaticBootStrap[iBase+nbBases*iBootStrap]=cpl_vector_new(nbWave);
	    }
	  for(iFrame=0;iFrame<nbFrameTarget;iFrame++)
	    {
	      numFrameBootStrap=(int)(nbFrameTarget*gsl_rng_uniform(r));
	      cpl_image_copy(imgcurrentRealBootStrap[iFrame],imgcurrentReal[numFrameBootStrap],1,1);
	      cpl_image_copy(imgcurrentImagBootStrap[iFrame],imgcurrentImag[numFrameBootStrap],1,1);
	      for(iBase=0;iBase<nbBases;iBase++)
		{
		  cpl_vector_set(vecalphaBootStrap[iBase],iFrame,cpl_vector_get(vecalpha[iBase],numFrameBootStrap));
		  cpl_vector_set(vecbetaBootStrap[iBase],iFrame,cpl_vector_get(vecbeta[iBase],numFrameBootStrap));
		}
	    }
	  for(iBase=0;iBase<nbBases;iBase++)
	    {
	      for(iWave=0;iWave<nbWave;iWave++)
		{
		  double xval=0.;
		  double yval=0.;

		  for(iFrame=0;iFrame<nbFrameTarget;iFrame++)
		    {
		      double eta=sqrt((cpl_image_get(imgcurrentRealBootStrap[iFrame],iWave+1,iBase+2,&pis))*
				      (cpl_image_get(imgcurrentRealBootStrap[iFrame],iWave+1,iBase+2,&pis))+
				      (cpl_image_get(imgcurrentImagBootStrap[iFrame],iWave+1,iBase+2,&pis))*
				      (cpl_image_get(imgcurrentImagBootStrap[iFrame],iWave+1,iBase+2,&pis)));
		      phi=atan2(cpl_image_get(imgcurrentImagBootStrap[iFrame],iWave+1,iBase+2,&pis),
				cpl_image_get(imgcurrentRealBootStrap[iFrame],iWave+1,iBase+2,&pis));
		      xval+=eta*cos(phi-cpl_vector_get(vecbetaBootStrap[iBase],iFrame)-(cpl_vector_get(vecalphaBootStrap[iBase],iFrame)/
									       cpl_image_get(imgcurrentRealBootStrap[iFrame],iWave+1,1,&pis)));
		      yval+=eta*sin(phi-cpl_vector_get(vecbetaBootStrap[iBase],iFrame)-(cpl_vector_get(vecalphaBootStrap[iBase],iFrame)/
									       cpl_image_get(imgcurrentRealBootStrap[iFrame],iWave+1,1,&pis)));
		    }
		  cpl_vector_set(vecphiStatictmp[iBase],iWave,atan2(yval,xval));
		}
	    }

	  for(iBase=0;iBase<nbBases;iBase++)
	    {
	      for(iWave=0;iWave<nbWave;iWave++)
		{
		  cpl_vector_set(vecphiStaticBootStrap[iBase+nbBases*iBootStrap],iWave,cpl_vector_get(vecphiStatictmp[iBase],iWave));
		}
	    }
	  vecre=cpl_vector_new(nbFrameTarget);
	  vecim=cpl_vector_new(nbFrameTarget);
	  vecreNoise=cpl_vector_new(nbFrameTarget);
	  vecimNoise=cpl_vector_new(nbFrameTarget);
	  for(iBase=0;iBase<nbBases;iBase++)
	    {
	      for(iWave=0;iWave<nbWave;iWave++)
		{
		  for(iFrame=0;iFrame<nbFrameTarget;iFrame++)
		    {
		      phi=(cpl_vector_get(vecalphaBootStrap[iBase],iFrame)/cpl_image_get(imgcurrentImagBootStrap[iFrame],iWave+1,1,&pis))+
			cpl_vector_get(vecbetaBootStrap[iBase],iFrame);
		      foore=(cpl_image_get(imgcurrentRealBootStrap[iFrame],iWave+1,iBase+2,&pis)*cos(phi)+
			     cpl_image_get(imgcurrentImagBootStrap[iFrame],iWave+1,iBase+2,&pis)*sin(phi));
		      fooim=(cpl_image_get(imgcurrentImagBootStrap[iFrame],iWave+1,iBase+2,&pis)*cos(phi)-
			     cpl_image_get(imgcurrentRealBootStrap[iFrame],iWave+1,iBase+2,&pis)*sin(phi));
		      cpl_vector_set(vecre,iFrame,foore);
		      cpl_vector_set(vecim,iFrame,fooim);
		    
		      foore=(cpl_image_get(imgcurrentRealBootStrap[iFrame],iWave+1,iBase+9,&pis)*cos(phi)+
			     cpl_image_get(imgcurrentImagBootStrap[iFrame],iWave+1,iBase+9,&pis)*sin(phi));
		      fooim=(cpl_image_get(imgcurrentImagBootStrap[iFrame],iWave+1,iBase+9,&pis)*cos(phi)-
			     cpl_image_get(imgcurrentRealBootStrap[iFrame],iWave+1,iBase+9,&pis)*sin(phi));
		      cpl_vector_set(vecreNoise,iFrame,foore);
		      cpl_vector_set(vecimNoise,iFrame,fooim);
		    }
		  mur=cpl_vector_get_mean(vecre);
		  mui=cpl_vector_get_mean(vecim);
		  alpha=atan2(mui,mur);
		  reBootStrap[iWave+nbWave*iBase+nbWave*nbBases*iBootStrap]=mur*cos(alpha)+mui*sin(alpha);
		  mur=cpl_vector_get_mean(vecreNoise);
		  mui=cpl_vector_get_mean(vecimNoise);
		  alpha=atan2(mui,mur);
		  reBootStrap[iWave+nbWave*iBase+nbWave*nbBases*iBootStrap]-=mur*cos(alpha)+mui*sin(alpha);
		  imBootStrap[iWave+nbWave*iBase+nbWave*nbBases*iBootStrap]=0.;
		}
	    }
	  cpl_vector_delete(vecre);
	  cpl_vector_delete(vecim);
	  cpl_vector_delete(vecreNoise);
	  cpl_vector_delete(vecimNoise);
	    
	}
      for(iBase=0;iBase<nbBases;iBase++)
      	{
      	  cpl_vector_delete(vecalpha[iBase]);
      	  cpl_vector_delete(vecbeta[iBase]);
      	  cpl_vector_delete(vecphiStatictmp[iBase]);
      	  vecphiStatictmp[iBase]=NULL;
      	}
      for(iBase=0;iBase<nbBases;iBase++)
      	{
      	  cpl_vector_delete(vecalphaBootStrap[iBase]);
      	  cpl_vector_delete(vecbetaBootStrap[iBase]);
      	}
    }
  else
  // We use the opd computed in mat_est_opdrecompute the opd
    {

      imgcurrentRealBootStrap = cpl_calloc(nbFrameTarget,sizeof(cpl_image *));
      imgcurrentImagBootStrap = cpl_calloc(nbFrameTarget,sizeof(cpl_image *));
      for(iBase=0;iBase<nbBases;iBase++)
	{
	  vecphiStaticErr[iBase]=cpl_vector_new(nbWave);
	}
      for(iFrame=0;iFrame<nbFrameTarget;iFrame++)
	{
	  imgcurrentRealBootStrap[iFrame] = cpl_image_new(nbWave,8,CPL_TYPE_DOUBLE);
	  imgcurrentImagBootStrap[iFrame] = cpl_image_new(nbWave,8,CPL_TYPE_DOUBLE);
	}

      for(int iBootStrap=0;iBootStrap<nBootStrap;iBootStrap++)
	{
	  for(iBase=0;iBase<nbBases;iBase++)
	    {
	      vecphiStaticBootStrap[iBase+nbBases*iBootStrap]=cpl_vector_new(nbWave);
	    }
	  for(iFrame=0;iFrame<nbFrameTarget;iFrame++)
	    {
	      numFrameBootStrap=(int)(nbFrameTarget*gsl_rng_uniform(r));
	      cpl_image_copy(imgcurrentRealBootStrap[iFrame],imgcurrentReal[numFrameBootStrap],1,1);
	      cpl_image_copy(imgcurrentImagBootStrap[iFrame],imgcurrentImag[numFrameBootStrap],1,1);
	    }
	  mat_apply_cophasing_algo(imgcurrentRealBootStrap,imgcurrentImagBootStrap,
				   nbWave,nbFrameTarget,vecalpha,vecbeta,vecphiStatictmp,
				   lambdamin,lambdamax);
	    
	  for(iBase=0;iBase<nbBases;iBase++)
	    {
	      for(iWave=0;iWave<nbWave;iWave++)
		{
		  cpl_vector_set(vecphiStaticBootStrap[iBase+nbBases*iBootStrap],iWave,cpl_vector_get(vecphiStatictmp[iBase],iWave));
		}
	    }


	  vecre=cpl_vector_new(nbFrameTarget);
	  vecim=cpl_vector_new(nbFrameTarget);
	  for(iBase=0;iBase<nbBases;iBase++)
	    {
	      for(iWave=0;iWave<nbWave;iWave++)
		{
		  for(iFrame=0;iFrame<nbFrameTarget;iFrame++)
		    {
		      phi=(cpl_vector_get(vecalpha[iBase],iFrame)/cpl_image_get(imgcurrentImagBootStrap[iFrame],iWave+1,1,&pis))+
			cpl_vector_get(vecbeta[iBase],iFrame);
		      foore=(cpl_image_get(imgcurrentRealBootStrap[iFrame],iWave+1,iBase+2,&pis)*cos(phi)+
			     cpl_image_get(imgcurrentImagBootStrap[iFrame],iWave+1,iBase+2,&pis)*sin(phi));
		      fooim=(cpl_image_get(imgcurrentImagBootStrap[iFrame],iWave+1,iBase+2,&pis)*cos(phi)-
			     cpl_image_get(imgcurrentRealBootStrap[iFrame],iWave+1,iBase+2,&pis)*sin(phi));
		      cpl_vector_set(vecre,iFrame,foore);
		      cpl_vector_set(vecim,iFrame,fooim);
		    }
		  reBootStrap[iWave+nbWave*iBase+nbWave*nbBases*iBootStrap]=cpl_vector_get_mean(vecre);
		  imBootStrap[iWave+nbWave*iBase+nbWave*nbBases*iBootStrap]=cpl_vector_get_mean(vecim);
		}
	    }
	  



	  /* vecre=cpl_vector_new(nbFrameTarget); */
	  /* vecim=cpl_vector_new(nbFrameTarget); */
	  /* vecreNoise=cpl_vector_new(nbFrameTarget); */
	  /* vecimNoise=cpl_vector_new(nbFrameTarget); */
	  /* for(iBase=0;iBase<nbBases;iBase++) */
	  /*   { */
	  /*     for(iWave=0;iWave<nbWave;iWave++) */
	  /* 	{ */
	  /* 	  for(iFrame=0;iFrame<nbFrameTarget;iFrame++) */
	  /* 	    { */
	  /* 	      phi=(cpl_vector_get(vecalpha[iBase],iFrame)/cpl_image_get(imgcurrentImagBootStrap[iFrame],iWave+1,1,&pis))+ */
	  /* 		cpl_vector_get(vecbeta[iBase],iFrame); */
	  /* 	      foore=(cpl_image_get(imgcurrentRealBootStrap[iFrame],iWave+1,iBase+2,&pis)*cos(phi)+ */
	  /* 		     cpl_image_get(imgcurrentImagBootStrap[iFrame],iWave+1,iBase+2,&pis)*sin(phi)); */
	  /* 	      fooim=(cpl_image_get(imgcurrentImagBootStrap[iFrame],iWave+1,iBase+2,&pis)*cos(phi)- */
	  /* 		     cpl_image_get(imgcurrentRealBootStrap[iFrame],iWave+1,iBase+2,&pis)*sin(phi)); */
	  /* 	      cpl_vector_set(vecre,iFrame,foore); */
	  /* 	      cpl_vector_set(vecim,iFrame,fooim); */
		    
	  /* 	      foore=(cpl_image_get(imgcurrentRealBootStrap[iFrame],iWave+1,iBase+9,&pis)*cos(phi)+ */
	  /* 		     cpl_image_get(imgcurrentImagBootStrap[iFrame],iWave+1,iBase+9,&pis)*sin(phi)); */
	  /* 	      fooim=(cpl_image_get(imgcurrentImagBootStrap[iFrame],iWave+1,iBase+9,&pis)*cos(phi)- */
	  /* 		     cpl_image_get(imgcurrentRealBootStrap[iFrame],iWave+1,iBase+9,&pis)*sin(phi)); */
	  /* 	      cpl_vector_set(vecreNoise,iFrame,foore); */
	  /* 	      cpl_vector_set(vecimNoise,iFrame,fooim); */
	  /* 	    } */
	  /* 	  mur=cpl_vector_get_mean(vecre); */
	  /* 	  mui=cpl_vector_get_mean(vecim); */
	  /* 	  alpha=atan2(mui,mur); */
	  /* 	  reBootStrap[iWave+nbWave*iBase+nbWave*nbBases*iBootStrap]=mur*cos(alpha)+mui*sin(alpha); */
	  /* 	  mur=cpl_vector_get_mean(vecreNoise); */
	  /* 	  mui=cpl_vector_get_mean(vecimNoise); */
	  /* 	  alpha=atan2(mui,mur); */
	  /* 	  reBootStrap[iWave+nbWave*iBase+nbWave*nbBases*iBootStrap]-=mur*cos(alpha)+mui*sin(alpha); */
	  /* 	} */
	  /*   } */
	  /* cpl_vector_delete(vecre); */
	  /* cpl_vector_delete(vecim); */
	  /* cpl_vector_delete(vecreNoise); */
	  /* cpl_vector_delete(vecimNoise); */

	  for(iBase=0;iBase<nbBases;iBase++)
	    {
	      cpl_vector_delete(vecalpha[iBase]);
	      cpl_vector_delete(vecbeta[iBase]);
	      cpl_vector_delete(vecphiStatictmp[iBase]);
	      vecphiStatictmp[iBase]=NULL;
	    }
	    
	}
    }



  
  double val1BootStrap,val2BootStrap;
  double val3BootStrap,val4BootStrap;
  for(iBase=0;iBase<nbBases;iBase++)
    {
      for(iWave=0;iWave<nbWave;iWave++)
	{
	  val1BootStrap=0.;
	  val2BootStrap=0.;
	  for(int iBootStrap=0;iBootStrap<nBootStrap;iBootStrap++)
	    {
	      val1BootStrap+=pow(cpl_vector_get(vecphiStaticBootStrap[iBase+nbBases*iBootStrap],iWave),2.0);
	      val2BootStrap+=cpl_vector_get(vecphiStaticBootStrap[iBase+nbBases*iBootStrap],iWave);
	    }
	  val2BootStrap/=nBootStrap;
	  val1BootStrap/=nBootStrap;
	  cpl_vector_set(vecphiStaticErr[iBase],iWave,sqrt(val1BootStrap-val2BootStrap*val2BootStrap));

		
	  val1BootStrap=0.;
	  val2BootStrap=0.;
	  val3BootStrap=0.;
	  val4BootStrap=0.;
	  for(int iBootStrap=0;iBootStrap<nBootStrap;iBootStrap++)
	    {
	      val1BootStrap+=pow(reBootStrap[iWave+nbWave*iBase+nbWave*nbBases*iBootStrap],2.0);
	      val2BootStrap+=reBootStrap[iWave+nbWave*iBase+nbWave*nbBases*iBootStrap];
	      val3BootStrap+=pow(imBootStrap[iWave+nbWave*iBase+nbWave*nbBases*iBootStrap],2.0);
	      val4BootStrap+=imBootStrap[iWave+nbWave*iBase+nbWave*nbBases*iBootStrap];
	    }
	  val2BootStrap/=nBootStrap;
	  val1BootStrap/=nBootStrap;
	  val4BootStrap/=nBootStrap;
	  val3BootStrap/=nBootStrap;
	  cpl_image_set(viserr,iWave+1,iBase+1,sqrt(val1BootStrap-val2BootStrap*val2BootStrap+
						    val3BootStrap-val4BootStrap*val4BootStrap));
	}
    }
	
  for(iFrame=0;iFrame<nbFrameTarget;iFrame++)
    {
      cpl_image_delete(imgcurrentRealBootStrap[iFrame]);
      cpl_image_delete(imgcurrentImagBootStrap[iFrame]);
    }
  cpl_free(imgcurrentRealBootStrap);
  cpl_free(imgcurrentImagBootStrap);
  for(int iBootStrap=0;iBootStrap<nBootStrap;iBootStrap++)
    {
      for(iBase=0;iBase<nbBases;iBase++)
	{
	  cpl_vector_delete(vecphiStaticBootStrap[iBase+nbBases*iBootStrap]);
	}
    }
  gsl_rng_free (r);
  // END BOOTSTRAP

  printf("END BOOTSTRAP\n");

}


void mat_get_visamp_visphi(cpl_image **imgcurrentReal,cpl_image **imgcurrentImag,mat_oiopdwvpo *oiopdwvpo,int nbWave,
			   int nbBases,int nbFrameTarget,double lambdamin,double lambdamax,cpl_image *viserr,
			   cpl_vector *vecphiStatic[],cpl_image *vis)
{
  int iBase,iFrame,iWave;
  int pis = 0;
  double phi=0.;
  cpl_vector *vecre=NULL;
  cpl_vector *vecim=NULL;
  double foore=0.;
  double fooim=0.;
  double mur, mui;

  cpl_vector *vecalpha[nbBases];
  cpl_vector *vecbeta[nbBases];

  if (oiopdwvpo != NULL)
    {
      int cpt=0;
      for(iBase=0;iBase<nbBases;iBase++)
	{
	  vecalpha[iBase]=cpl_vector_new(nbFrameTarget);
	  vecbeta[iBase]=cpl_vector_new(nbFrameTarget);
	  vecphiStatic[iBase]=cpl_vector_new(nbWave);
	}
      for(iFrame=0;iFrame<nbFrameTarget;iFrame++) {
	for(iBase=0;iBase<nbBases;iBase++) {
	  cpl_vector_set(vecalpha[iBase],iFrame,2.*CPL_MATH_PI*oiopdwvpo->oiopd->list_opd[cpt]->opd);
	  cpl_vector_set(vecbeta[iBase],iFrame,oiopdwvpo->oiopd->list_opd[cpt]->humoff);
	  cpt++;
	}
      }
      for(iBase=0;iBase<nbBases;iBase++)
	{
	  for(iWave=0;iWave<nbWave;iWave++)
	    {
	      double xval=0.;
	      double yval=0.;

	      for(iFrame=0;iFrame<nbFrameTarget;iFrame++)
	      	{
	      	  double eta=sqrt((cpl_image_get(imgcurrentReal[iFrame],iWave+1,iBase+2,&pis))*
	      			  (cpl_image_get(imgcurrentReal[iFrame],iWave+1,iBase+2,&pis))+
	      			  (cpl_image_get(imgcurrentImag[iFrame],iWave+1,iBase+2,&pis))*
	      			  (cpl_image_get(imgcurrentImag[iFrame],iWave+1,iBase+2,&pis)));
	      	  phi=atan2(cpl_image_get(imgcurrentImag[iFrame],iWave+1,iBase+2,&pis),
			    cpl_image_get(imgcurrentReal[iFrame],iWave+1,iBase+2,&pis));
	      	  xval+=eta*cos(phi-cpl_vector_get(vecbeta[iBase],iFrame)-(cpl_vector_get(vecalpha[iBase],iFrame)/
	      								   cpl_image_get(imgcurrentReal[iFrame],iWave+1,1,&pis)));
	      	  yval+=eta*sin(phi-cpl_vector_get(vecbeta[iBase],iFrame)-(cpl_vector_get(vecalpha[iBase],iFrame)/
	      								   cpl_image_get(imgcurrentReal[iFrame],iWave+1,1,&pis)));
	      	}
	      cpl_vector_set(vecphiStatic[iBase],iWave,atan2(yval,xval));
	    }
	}
    }
  else
    {
      mat_apply_cophasing_algo(imgcurrentReal,imgcurrentImag,
			       nbWave,nbFrameTarget,vecalpha,vecbeta,vecphiStatic,
			       lambdamin,lambdamax);
    }
  // Compute VISAMP and VISPHI
  vecre=cpl_vector_new(nbFrameTarget);
  vecim=cpl_vector_new(nbFrameTarget);
  for(iBase=0;iBase<nbBases;iBase++)
    {
      for(iWave=0;iWave<nbWave;iWave++)
  	{
  	  for(iFrame=0;iFrame<nbFrameTarget;iFrame++)
  	    {
  	      phi=(cpl_vector_get(vecalpha[iBase],iFrame)/cpl_image_get(imgcurrentImag[iFrame],iWave+1,1,&pis))+
  		cpl_vector_get(vecbeta[iBase],iFrame);
  	      foore=(cpl_image_get(imgcurrentReal[iFrame],iWave+1,iBase+2,&pis)*cos(phi)+
  		     cpl_image_get(imgcurrentImag[iFrame],iWave+1,iBase+2,&pis)*sin(phi));
  	      fooim=(cpl_image_get(imgcurrentImag[iFrame],iWave+1,iBase+2,&pis)*cos(phi)-
  		     cpl_image_get(imgcurrentReal[iFrame],iWave+1,iBase+2,&pis)*sin(phi));
  	      cpl_vector_set(vecre,iFrame,foore);
  	      cpl_vector_set(vecim,iFrame,fooim);
  	    }
  	  mur=cpl_vector_get_mean(vecre);
  	  mui=cpl_vector_get_mean(vecim);

  	  cpl_image_set(vis,iWave+1,iBase+1,sqrt(mur*mur+mui*mui)-
  			0.5*pow(cpl_image_get(viserr,iWave+1,iBase+1,&pis),2.)/sqrt(mur*mur+mui*mui));
  	}
    }
  cpl_vector_delete(vecre);
  cpl_vector_delete(vecim);
  /* vecre=cpl_vector_new(nbFrameTarget); */
  /* vecim=cpl_vector_new(nbFrameTarget); */
  /* vecreNoise=cpl_vector_new(nbFrameTarget); */
  /* vecimNoise=cpl_vector_new(nbFrameTarget); */
  /* for(iBase=0;iBase<nbBases;iBase++) */
  /*   { */
  /*     for(iWave=0;iWave<nbWave;iWave++) */
  /* 	{ */
  /* 	  for(iFrame=0;iFrame<nbFrameTarget;iFrame++) */
  /* 	    { */
  /* 	      phi=(cpl_vector_get(vecalpha[iBase],iFrame)/cpl_image_get(imgcurrentImag[iFrame],iWave+1,1,&pis))+ */
  /* 		cpl_vector_get(vecbeta[iBase],iFrame); */
  /* 	      foore=(cpl_image_get(imgcurrentReal[iFrame],iWave+1,iBase+2,&pis)*cos(phi)+ */
  /* 		     cpl_image_get(imgcurrentImag[iFrame],iWave+1,iBase+2,&pis)*sin(phi)); */
  /* 	      fooim=(cpl_image_get(imgcurrentImag[iFrame],iWave+1,iBase+2,&pis)*cos(phi)- */
  /* 		     cpl_image_get(imgcurrentReal[iFrame],iWave+1,iBase+2,&pis)*sin(phi)); */
  /* 	      cpl_vector_set(vecre,iFrame,foore); */
  /* 	      cpl_vector_set(vecim,iFrame,fooim); */
		    
  /* 	      foore=(cpl_image_get(imgcurrentReal[iFrame],iWave+1,iBase+9,&pis)*cos(phi)+ */
  /* 		     cpl_image_get(imgcurrentImag[iFrame],iWave+1,iBase+9,&pis)*sin(phi)); */
  /* 	      fooim=(cpl_image_get(imgcurrentImag[iFrame],iWave+1,iBase+9,&pis)*cos(phi)- */
  /* 		     cpl_image_get(imgcurrentReal[iFrame],iWave+1,iBase+9,&pis)*sin(phi)); */
  /* 	      cpl_vector_set(vecreNoise,iFrame,foore); */
  /* 	      cpl_vector_set(vecimNoise,iFrame,fooim); */
  /* 	    } */
  /* 	  mur=cpl_vector_get_mean(vecre); */
  /* 	  mui=cpl_vector_get_mean(vecim); */
  /* 	  alpha=atan2(mui,mur); */
  /* 	  foore=mur*cos(alpha)+mui*sin(alpha); */
  /* 	  mur=cpl_vector_get_mean(vecreNoise); */
  /* 	  mui=cpl_vector_get_mean(vecimNoise); */
  /* 	  alpha=atan2(mui,mur); */
  /* 	  foore-=(mur*cos(alpha)+mui*sin(alpha)); */

  /* 	  cpl_image_set(vis,iWave+1,iBase+1,foore); */
  /* 	} */
  /*   } */
  /* cpl_vector_delete(vecre); */
  /* cpl_vector_delete(vecim); */
  /* cpl_vector_delete(vecreNoise); */
  /* cpl_vector_delete(vecimNoise); */

  /* // Remove residuals OPD in vecphiStatic */
  /* vecre=cpl_vector_new(nbWave); */
  /* vecim=cpl_vector_new(nbWave); */
  /* double gd=0.; */
  /* double fooR=0.; */
  /* double fooI=0.; */
  /* double lambdamt=0.; */
  /* double resolmt=0.; */
  /* double deltalambdamt=0.; */
  /* double alph=0.; */
  /* for(iBase=0;iBase<nbBases;iBase++) */
  /*   { */
  /*     for(iWave=0;iWave<nbWave;iWave++) */
  /* 	{ */
  /* 	  lambdamt+=cpl_image_get(imgcurrentImag[0],iWave+1,1,&pis); */
  /* 	  cpl_vector_set(vecre,iWave,cos(cpl_vector_get(vecphiStatic[iBase],iWave))); */
  /* 	  cpl_vector_set(vecim,iWave,sin(cpl_vector_get(vecphiStatic[iBase],iWave))); */
  /* 	  if (iWave < nbWave-1) */
  /* 	    { */
  /* 	      deltalambdamt+=cpl_image_get(imgcurrentImag[0],iWave+2,1,&pis)-cpl_image_get(imgcurrentImag[0],iWave+1,1,&pis); */
  /* 	    } */
  /* 	} */
  /*     lambdamt/=nbWave; */
  /*     deltalambdamt/=(nbWave-1); */
  /*     resolmt=lambdamt/deltalambdamt; */
  /*     // Computing Group Delay */
  /*     fooR=0.; */
  /*     fooI=0.; */
  /*     for(iWave=0;iWave<nbWave-1;iWave++) */
  /* 	{ */
  /* 	  fooR+=cpl_vector_get(vecre,iWave)*cpl_vector_get(vecre,iWave+1)+ */
  /* 	    cpl_vector_get(vecim,iWave)*cpl_vector_get(vecim,iWave+1); */
  /* 	  fooI+=cpl_vector_get(vecre,iWave+1)*cpl_vector_get(vecim,iWave)- */
  /* 	    cpl_vector_get(vecre,iWave)*cpl_vector_get(vecim,iWave+1); */
  /* 	} */
  /*     gd=atan2(fooI,fooR)*(lambdamt*resolmt)/(2*CPL_MATH_PI); */
  /*     for(iWave=0;iWave<nbWave;iWave++) */
  /* 	{ */
  /* 	  alph=-(2*CPL_MATH_PI)*gd/cpl_image_get(imgcurrentImag[0],iWave+1,1,&pis); */
  /* 	  fooR=cpl_vector_get(vecre,iWave)*cos(alph)-cpl_vector_get(vecim,iWave)*sin(alph); */
  /* 	  fooI=cpl_vector_get(vecre,iWave)*sin(alph)+cpl_vector_get(vecim,iWave)*cos(alph); */
  /* 	  cpl_vector_set(vecphiStatic[iBase],iWave,atan2(fooI,fooR)); */
  /* 	} */
  /*     mat_vector_unwrap(vecphiStatic[iBase], vecre); */
  /*     for(iWave=0;iWave<nbWave;iWave++) */
  /* 	{ */
  /* 	  cpl_vector_set(vecphiStatic[iBase],iWave,cpl_vector_get(vecre,iWave)); */
  /* 	} */
  /*     alph=cpl_vector_get_mean(vecphiStatic[iBase]); */
  /*     cpl_vector_subtract_scalar(vecphiStatic[iBase],alph); */
  /*   } */
  /* cpl_vector_delete(vecre); */
  /* cpl_vector_delete(vecim); */
  
  for(iBase=0;iBase<nbBases;iBase++)
    {
      cpl_vector_delete(vecalpha[iBase]);
      cpl_vector_delete(vecbeta[iBase]);
    }
}
