/* $Id: mat_compute_vis2.c,v0.5 2014-06-15 12:56:21 pberio Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: pberio $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_compute_vis2.c $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#include <string.h>
#endif
#include <stdio.h>
/*-----------------------------------------------------------------------------
  Includes
  ----------------------------------------------------------------------------*/

#include "mat_compute_vis2.h"
#include "mat_error.h"

/*-----------------------------------------------------------------------------
  Define
  ----------------------------------------------------------------------------*/
/*-----------------------------------------------------------------------------
  Functions prototypes
  ----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
   @defgroup signal Interferometric signal calculation
*/
/**
   @ingroup signal
   @brief Compute the spectral density and the squared visibility
   @brief from a mat_corrflux structure and a mat_photbeams structure. 
   @brief This function fills a mat_oifits with the computed visibilities.
   @param *corrFlux           Correlated Flux
   @param *phot               Photometry
   @param *oifits             mat_oifits structure contaning the results
   @return 0 if no error 
*/
int mat_compute_vis2(mat_corrflux  *corrFlux,
                     mat_photbeams *phot,
                     mat_oifits    *oifits,
                     mat_corrflux  *nofringe)
{
    /* Ranges */
    int iReg=0, iFrame=0, iFreq=0, iWlen=0, iWlen0=0, iPix=0, iBase=0;
    int nbRegion;
    int nbFrameP=0, nbFrameS=0, nbFrameT=0, nbFrameNF=0;
    int nbWlen, nbFreq, nbBases=6;
    
    /* Counters */
    //int k;
    int cptWave=0;
    int pis    =0;
    
    int uPixelMin   =0;
    int uPixelMax   =0;
    int uPixelMinLow=0;
    int uPixelMaxSup=0;
  
    cpl_image   **densitySpec   =NULL;
    cpl_image   **densitySpecSub=NULL;
  
    cpl_image   *foore   =NULL;
    cpl_image   *fooim   =NULL;
    cpl_vector  *fit     =NULL;
    cpl_vector  *dispCoef=NULL;
    cpl_vector  *vecw    =NULL;
    cpl_vector  *vecy    =NULL;
  
    double      lambda     =0.;
    double      dOverLambda=0.;
    double      val        =0.;
    double      val2       =0.;
  
    cpl_image    *vis2       =NULL;
    cpl_image    *vis2err    =NULL;
    cpl_image   **prodPhot   =NULL;
    cpl_image   **prodPhotErr=NULL;
    cpl_image    *foo        =NULL;

    char        *sliderPhot=NULL;
    char        *detName   =NULL;
    char mode[3]           =" ";
    char        *bcd2Status=NULL;
    char         kwd[64];

    double   i1,i2,i3,i4;
    int      nbdetector   =0;
    int      resolution   =0;
    char    *specResData  =NULL;
    char    *fil  =NULL;
    double   posFringePeak=0.;

    double      valX =0.;
    double      valX2=0.;
    double      snr  =0.;
    double      varX =0.;

    cpl_image   *fringePeak=NULL;
    int      nStat     =10;
    int      iStat     =0;
    int      expno = 0;
    char     *output = NULL;
    

    // Check input parameters
    mat_assert((corrFlux!=NULL),CPL_ERROR_NULL_INPUT,
			   "no mat_corrflux argument given");
    mat_assert((oifits!=NULL),CPL_ERROR_NULL_INPUT,
               "no mat_oifits argument given");

    expno = cpl_propertylist_get_int(corrFlux->keywords,"ESO TPL EXPNO");
    
    
    if (nofringe == NULL)
    {
        cpl_msg_info(cpl_func,"NOFRINGE data not provided");
    }
    else
    {
        cpl_msg_info(cpl_func,"NOFRINGE data provided");
    }
      
  
    detName=(char *)
        cpl_propertylist_get_string(corrFlux->keywords,"ESO DET CHIP NAME");
    if (detName == NULL)
    {
        cpl_msg_error(cpl_func,
                      "no ESO DET CHIP NAME keyword in frame");
        return -1;
    }


    if(!strcmp(detName, "AQUARIUS"))
    {
        nbdetector=2;
        sliderPhot=(char *)cpl_propertylist_get_string(corrFlux->keywords,
                                                       "ESO INS PIN NAME");     
        specResData=(char *)cpl_propertylist_get_string(corrFlux->keywords,
                                                        "ESO INS DIN ID");
        if (!strcmp(specResData,"LOW"))
        {
            resolution=1;
        }
        if (!strcmp(specResData,"HIGH"))
        {
            resolution=2;
        }
      
    }
    else
    {
        nbdetector=1;
        sliderPhot=(char *)cpl_propertylist_get_string(corrFlux->keywords,
                                                       "ESO INS PIL NAME");     
        specResData=(char *)cpl_propertylist_get_string(corrFlux->keywords,
                                                        "ESO INS DIL ID");
        fil=(char *)cpl_propertylist_get_string(corrFlux->keywords,
						"ESO INS FIL NAME");
        if (!strcmp(specResData,"LOW"))
        {
            resolution=1;
        }
        if (!strcmp(specResData,"MED"))
        {
            resolution=2;
        }
        if (!strcmp(specResData,"HIGH"))
        {
            resolution=3;
        }
        if (!strcmp(specResData,"HIGH+") && !strcmp(fil,"L"))
        {
            resolution=4;
        }
        if (!strcmp(specResData,"HIGH+") && !strcmp(fil,"M"))
        {
            resolution=5;
        }
    }
    cpl_msg_info(cpl_func,"Detector %d\n",nbdetector);
    cpl_msg_info(cpl_func,"Spectral Resolution %d\n",resolution);

    if (!strcmp(sliderPhot, "PHOTO"))
    {
        sprintf(mode,"s");
    }
    else
    {
        sprintf(mode,"h");
    }

    bcd2Status=(char *)cpl_propertylist_get_string(oifits->keywords,
                                                     "ESO INS BCD2 ID");

    // Loading of the dispersion coefficients
    dispCoef=cpl_vector_new(N_DEG_DISPERSION_LAW+1);
    cpl_vector_set(dispCoef,0,cpl_propertylist_get_double(corrFlux->keywords,"PRO DISP COEF0"));
    cpl_vector_set(dispCoef,1,cpl_propertylist_get_double(corrFlux->keywords,"PRO DISP COEF1"));
    cpl_vector_set(dispCoef,2,cpl_propertylist_get_double(corrFlux->keywords,"PRO DISP COEF2"));
    cpl_vector_set(dispCoef,3,cpl_propertylist_get_double(corrFlux->keywords,"PRO DISP COEF3"));
    cpl_vector_set(dispCoef,4,cpl_propertylist_get_double(corrFlux->keywords,"PRO DISP COEF4"));

    cpl_msg_info(cpl_func,"Computation of the squared visibilities");


    cpl_msg_info(cpl_func,"0");
  
    // NoFringe spectral density computation

    cpl_image **densitySpecNoFringe=NULL;
  	
    if (nofringe != NULL)
    {
  
    cpl_msg_info(cpl_func,"0b");
	
    nbRegion  = nofringe->imgdet->nbregion;
    cpl_msg_info(cpl_func,"0c");
    nbFrameNF = nofringe->nbframe / nbRegion;
  
    cpl_msg_info(cpl_func,"1");

        if (cpl_propertylist_has(nofringe->keywords,"PRO FRAME TARGET"))
        {
            nbFrameNF=cpl_propertylist_get_int(nofringe->keywords,
                                                  "PRO FRAME TARGET");
        }
        densitySpecNoFringe=cpl_calloc(nbRegion,sizeof(cpl_image *));

    cpl_msg_info(cpl_func,"2");
	
        /* Compute the spectral density itself */
        for(iReg=0 ; iReg < nbRegion ; iReg++)
        {
            nbFreq=nofringe->imgdet->list_region[iReg]->naxis[0];
            nbWlen=nofringe->imgdet->list_region[iReg]->naxis[1];
    
    cpl_msg_info(cpl_func,"3");
	          
            foore         =cpl_image_new(nbFreq,nbWlen,CPL_TYPE_FLOAT);
            fooim         =cpl_image_new(nbFreq,nbWlen,CPL_TYPE_FLOAT);
            densitySpecNoFringe[iReg]=cpl_image_new(nbFreq,nbWlen,CPL_TYPE_FLOAT);
    
    cpl_msg_info(cpl_func,"3");
	          
            for(iFrame=5 ; iFrame < nbFrameNF ; iFrame ++)
            {
                /* Real part squared */
                foo   =nofringe->list_corr[iReg*nbFrameNF+iFrame]->imgreg[0];
                cpl_image_copy(foore,foo,1,1);
                cpl_image_multiply(foore,foo);
            
                /* Imaginary part squared */
                foo   =nofringe->list_corr[iReg*nbFrameNF+iFrame]->imgreg[1];
                cpl_image_copy(fooim,foo,1,1);
                cpl_image_multiply(fooim,foo);
            
                /* Sum of Real and Imaginary parts squared */
                cpl_image_add(foore,fooim);
                cpl_image_add(densitySpecNoFringe[iReg],foore); 
            }
            cpl_image_divide_scalar(densitySpecNoFringe[iReg],nbFrameNF-5);
            cpl_image_delete(foore);
            cpl_image_delete(fooim);
        }
        cpl_image_save(densitySpecNoFringe[0],"DSPnofringe.fits",
					   CPL_BPP_IEEE_FLOAT, NULL, CPL_IO_CREATE);
    }


  
  
    // Sky Spectral density computation
    cpl_image **densitySpecSky=NULL;
  
    nbRegion=corrFlux->imgdet->nbregion;  
    nbFrameS=nbFrameT=corrFlux->nbframe / nbRegion;

    // If NoFringe, DSPsky is not necessary.
    // We use DSPnofringe to remove photon bias.
    if (densitySpecNoFringe == NULL)
    {
        if (cpl_propertylist_has(corrFlux->keywords,
                                 "PRO FRAME TARGET"))
        {
            nbFrameT=cpl_propertylist_get_int(corrFlux->keywords,
                                                     "PRO FRAME TARGET");
        }
        if (nbFrameT < nbFrameS && nbdetector == 2)
        {
            densitySpecSky=cpl_calloc(nbRegion,sizeof(cpl_image *));
          
            for(iReg=0 ; iReg < nbRegion ; iReg++)
            {
              
                nbFreq=corrFlux->imgdet->list_region[iReg]->naxis[0];
                nbWlen=corrFlux->imgdet->list_region[iReg]->naxis[1];
              
                foore     =cpl_image_new(nbFreq,nbWlen,CPL_TYPE_FLOAT);
                fooim     =cpl_image_new(nbFreq,nbWlen,CPL_TYPE_FLOAT);
                densitySpecSky[iReg]= cpl_image_new(nbFreq,nbWlen,
													CPL_TYPE_FLOAT);
            
                for(iFrame=nbFrameT ; iFrame < nbFrameS ; iFrame ++)
                {
                    foo=corrFlux->list_corr[iReg*nbFrameS+iFrame]->imgreg[0];
                    cpl_image_copy(foore,foo,1,1);
                    cpl_image_multiply(foore,foo);
                  
                    foo=corrFlux->list_corr[iReg*nbFrameS+iFrame]->imgreg[1];
                    cpl_image_copy(fooim,foo,1,1);
                    cpl_image_multiply(fooim,foo);
                  
                    cpl_image_add(foore,fooim);
                    cpl_image_add(densitySpecSky[iReg],foore);  
                }
                cpl_image_divide_scalar(densitySpecSky[iReg],
                                        nbFrameS-nbFrameT);
                cpl_image_delete(foore);
                cpl_image_delete(fooim);
            }
            cpl_image_save(densitySpecSky[0],"DSPsky.fits",
						   CPL_BPP_IEEE_FLOAT, NULL, CPL_IO_CREATE);
        }
    }
  
  
    // Spectral density computation
    nbFreq  =corrFlux->imgdet->list_region[0]->naxis[0];
    nbWlen  =corrFlux->imgdet->list_region[0]->naxis[1];
  
    fringePeak  =cpl_image_new(nbFreq,nbWlen,CPL_TYPE_FLOAT);
    densitySpec =cpl_calloc(nbRegion,      sizeof(cpl_image *));
    densitySpecSub=cpl_calloc(nStat*nbRegion,sizeof(cpl_image *));
  
    for(iReg=0 ; iReg < nbRegion ; iReg++)
    {
        nbFreq  =corrFlux->imgdet->list_region[iReg]->naxis[0];
        nbWlen  =corrFlux->imgdet->list_region[iReg]->naxis[1];
      
        foore          =cpl_image_new(nbFreq,nbWlen,CPL_TYPE_FLOAT);
        fooim          =cpl_image_new(nbFreq,nbWlen,CPL_TYPE_FLOAT);
        densitySpec[iReg]=cpl_image_new(nbFreq,nbWlen,CPL_TYPE_FLOAT);
      
        for(iFrame =0;iFrame <nbFrameT;iFrame ++)
        {
            foo=corrFlux->list_corr[iReg*nbFrameT+iFrame]->imgreg[0];
            cpl_image_copy(foore,foo,1,1);
            cpl_image_multiply(foore,foo);
          
            foo=corrFlux->list_corr[iReg*nbFrameT+iFrame]->imgreg[1];
            cpl_image_copy(fooim,foo,1,1);
            cpl_image_multiply(fooim,foo);
          
            cpl_image_add(foore,fooim);
            cpl_image_add(densitySpec[iReg],foore);       
        }
        cpl_image_divide_scalar(densitySpec[iReg],nbFrameT);

        for(iStat=0;iStat<nStat;iStat++)
        {
            densitySpecSub[iReg*nStat+iStat]=cpl_image_new(nbFreq,nbWlen,CPL_TYPE_FLOAT);
            for(iFrame=iStat*nbFrameT/(nStat+1.) ; iFrame < (iStat+1)*nbFrameT/(nStat+1.);iFrame ++)
            {
                foo=corrFlux->list_corr[iReg*nbFrameT+iFrame]->imgreg[0];
                cpl_image_copy(foore,foo,1,1);
                cpl_image_multiply(foore,foo);
              
                foo=corrFlux->list_corr[iReg*nbFrameT+iFrame]->imgreg[1];
                cpl_image_copy(fooim,foo,1,1);
                cpl_image_multiply(fooim,foo);
              
                cpl_image_add(foore,fooim);
                cpl_image_add(densitySpecSub[iReg*nStat+iStat],foore);
              
            }
            cpl_image_divide_scalar(densitySpecSub[iReg*nStat+iStat],nbFrameT/(nStat+1.));
        }
        cpl_image_delete(foore);
        cpl_image_delete(fooim);
    }
    cpl_image_save(densitySpec[0],"DSPtarget.fits", CPL_BPP_IEEE_FLOAT,NULL, CPL_IO_CREATE);

 
    if (densitySpecNoFringe != NULL)
    {
        /*    Correction of the photon noise with SKY DSP*/
        for(iReg=0 ; iReg < nbRegion ; iReg++)
        {
            cpl_image_subtract(densitySpec[iReg],densitySpecNoFringe[iReg]);
            for(iStat=0;iStat<nStat;iStat++)
            {
                cpl_image_subtract(densitySpecSub[iReg*nStat+iStat],
                                   densitySpecNoFringe[iReg]);
            }
        }
    }
    else if (densitySpecSky != NULL)
    {
        /*    Correction of the photon noise with SKY DSP*/
        for(iReg=0 ; iReg < nbRegion ; iReg++)
        {
            cpl_image_subtract(densitySpec[iReg],densitySpecSky[iReg]);
            for(iStat=0;iStat<nStat;iStat++)
            {
                cpl_image_subtract(densitySpecSub[iReg*nStat+iStat],
                                   densitySpecSky[iReg]);
            }
        }
    }
    else
    {
        /*    Correction of the photon noise */
        for(iReg=0 ; iReg < nbRegion ; iReg++)
        {
            nbFreq=corrFlux->imgdet->list_region[iReg]->naxis[0];
            nbWlen=corrFlux->imgdet->list_region[iReg]->naxis[1];
            iWlen0=corrFlux->imgdet->list_region[iReg]->corner[1];
        
            vecw  =cpl_vector_new(nbFreq);
            fit   =cpl_vector_new(nbFreq);

            for(iWlen=0 ; iWlen < nbWlen ; iWlen++)
            {
            
                lambda=mat_get_lambda(dispCoef,iWlen+iWlen0);
            
                for(iFreq=0 ; iFreq < nbFreq ; iFreq++)
                {
                    cpl_vector_set(vecw,iFreq,1.0);
                }

                posFringePeak=mat_get_pos_fringe_peak(lambda,nbdetector,resolution,0);
                dOverLambda=(posFringePeak-nbFreq/2)/(3);

                /* Set pixels between which integrate fringe peak */
                uPixelMin=nbFreq/2;
                uPixelMax=nbFreq/2 + (int)(24.5*dOverLambda);
        
                for(iFreq=0 ; iFreq < nbFreq ; iFreq++)
                {
                    if (iFreq >= uPixelMin && iFreq <= uPixelMax)
                    {
                        cpl_vector_set(vecw,iFreq,1.E-20);
                    }
                    if (iFreq >= nbFreq-20 )
                    {
                        cpl_vector_set(vecw,iFreq,1.E-20);
                    }
                }

                uPixelMax=nbFreq/2 - (int)(24.5*dOverLambda);
        
                for(iFreq=0 ; iFreq < nbFreq ; iFreq++)
                {
                    if (iFreq <= uPixelMin && iFreq >= uPixelMax)
                    {
                        cpl_vector_set(vecw,iFreq,1.E-20);
                    }
                    if (iFreq <= 20 )
                    {
                        cpl_vector_set(vecw,iFreq,1.E-20);
                    }
                }
        
                vecy=cpl_vector_new_from_image_row(densitySpec[iReg],iWlen+1);
                mat_polyfit_1d(vecy,vecw,0,fit,1);

                cpl_vector_subtract(vecy,fit);
            
                for(iFreq=0 ; iFreq < nbFreq ; iFreq++)
                {
                    cpl_image_set(densitySpec[iReg],iFreq+1,iWlen+1,
                                  cpl_vector_get(vecy,iFreq));
                }
                cpl_vector_delete(vecy);
            }
            cpl_vector_delete(vecw);
            cpl_vector_delete(fit);
        }
    }



    /* cpl_plot_image_row("set grid;","with lines","", */
    /*                 densitySpec[0],1,1,50); */
    /* cpl_plot_image("set view map;","with pm3d","",densitySpec[0]); */
    /* /\* exit(1); *\/ */
 
    cptWave=0;

    for(iReg=0 ; iReg < nbRegion ; iReg++)
    {
        nbFreq=corrFlux->imgdet->list_region[iReg]->naxis[0];
        nbWlen=corrFlux->imgdet->list_region[iReg]->naxis[1];
        iWlen0=corrFlux->imgdet->list_region[iReg]->corner[1];
      
        // Compute the energy of each fringe peak
        vis2   =      cpl_image_new(nbWlen, 6,CPL_TYPE_DOUBLE);
        vis2err=      cpl_image_new(nbWlen, 6,CPL_TYPE_DOUBLE);
      
        for(iWlen=0;iWlen<nbWlen;iWlen++)
        {
            lambda=mat_get_lambda(dispCoef,iWlen+iWlen0);
      
            for(iBase=1 ; iBase <= nbBases ; iBase++)
            {
                posFringePeak=mat_get_pos_fringe_peak(lambda,nbdetector,resolution,iBase-1);
                dOverLambda=(posFringePeak-nbFreq/2)/(3*iBase);
        
                if (nbdetector==2)
                {
                    uPixelMin=(int)posFringePeak-(int)dOverLambda+1;
                    uPixelMax=(int)posFringePeak+(int)dOverLambda-1;
                    if (iBase > 1)
                    {
                        uPixelMinLow=(int)posFringePeak-(int)(dOverLambda*1.5)+1;
                        uPixelMaxSup=(int)posFringePeak+(int)(dOverLambda*1.5)-1;
                    }
                    else
                    {
                        uPixelMinLow=uPixelMin-1;
                        uPixelMaxSup=(int)posFringePeak+(int)(dOverLambda*1.5);
                    }
          
                    vecw=cpl_vector_new(uPixelMaxSup-uPixelMinLow+1);
                    fit =cpl_vector_new(uPixelMaxSup-uPixelMinLow+1);
                    vecy=cpl_vector_new(uPixelMaxSup-uPixelMinLow+1);
          
                    for(iPix=uPixelMinLow ; iPix <= uPixelMaxSup ; iPix++)
                    {
                        cpl_vector_set(vecw, iPix - uPixelMinLow, 1.0);
                        if (iPix >= uPixelMin && iPix <= uPixelMax)
                        {
                            cpl_vector_set(vecw, iPix-uPixelMinLow, 1.E-20);
                        }
                        cpl_vector_set(vecy, iPix-uPixelMinLow,
                                       cpl_image_get(densitySpec[iReg],
                                                     iPix+1, iWlen+1, &pis));
                    }
                }
                else
                {
                    uPixelMin   =(int)posFringePeak-(int)dOverLambda-1;
                    uPixelMax   =(int)posFringePeak+(int)dOverLambda+1;
                    uPixelMinLow=(int)posFringePeak-(int)(dOverLambda*1.5)-1;
                    uPixelMaxSup=(int)posFringePeak+(int)(dOverLambda*1.5)+1;

                    vecw=cpl_vector_new(uPixelMaxSup-uPixelMinLow+1);
                    fit =cpl_vector_new(uPixelMaxSup-uPixelMinLow+1);
                    vecy=cpl_vector_new(uPixelMaxSup-uPixelMinLow+1);

                    for(iPix=uPixelMinLow ; iPix <= uPixelMaxSup ; iPix++)
                    {
                        cpl_vector_set(vecw,iPix-uPixelMinLow,1.0);
                        if (iPix>=uPixelMin && iPix<=uPixelMax)
                        {
                            cpl_vector_set(vecw,iPix-uPixelMinLow,1.E-20);
                        }
                        cpl_vector_set(vecy,iPix-uPixelMinLow,
                                       cpl_image_get(densitySpec[iReg],
                                                     iPix+1,iWlen+1,&pis));
                    }

                }
        
				val=0.;
				for(iPix=uPixelMinLow;iPix<=uPixelMaxSup;iPix++)
				{
					val+=cpl_vector_get(vecy,iPix-uPixelMinLow);
					cpl_image_set(fringePeak,iPix+1,iWlen+1,cpl_vector_get(vecy,iPix-uPixelMinLow));
				}
				cpl_vector_delete(vecy);
				cpl_vector_delete(vecw);
				cpl_vector_delete(fit);
				cpl_image_set(vis2,iWlen+1,iBase,val);
			}
        
			// Compute stdev of the spectral density
            for(iBase=1 ; iBase <= nbBases ; iBase++)
			{
				posFringePeak=mat_get_pos_fringe_peak(lambda,nbdetector,resolution,iBase-1);
				dOverLambda=(posFringePeak-nbFreq/2)/(3*iBase);

				uPixelMin	=(int)posFringePeak-(int)dOverLambda+1;
				uPixelMax	=(int)posFringePeak+(int)dOverLambda-1;
				uPixelMinLow=(int)posFringePeak-(int)(dOverLambda*1.5)+1;
				uPixelMaxSup=(int)posFringePeak+(int)(dOverLambda*1.5)-1;
					
				vecw=cpl_vector_new(uPixelMaxSup-uPixelMinLow+1);
				fit	=cpl_vector_new(uPixelMaxSup-uPixelMinLow+1);
				vecy=cpl_vector_new(uPixelMaxSup-uPixelMinLow+1);

				valX2 =	0.;
				valX  =	0.;
					
				for(iStat=0;iStat<nStat;iStat++)
				{
					for(iPix=uPixelMinLow;iPix<=uPixelMaxSup;iPix++)
					{
						cpl_vector_set(vecw,iPix-uPixelMinLow,1.0);
						if (iPix>=uPixelMin && iPix<=uPixelMax)
						{
							cpl_vector_set(vecw,iPix-uPixelMinLow,1.E-20);
						}
						cpl_vector_set(vecy,iPix-uPixelMinLow,cpl_image_get(densitySpecSub[iReg*nStat+iStat],iPix+1,iWlen+1,&pis));
					}
					mat_polyfit_1d(vecy,vecw,1,fit,0);
					cpl_vector_subtract(vecy,fit);
					val	= 0.;
					for(iPix=uPixelMinLow;iPix<=uPixelMaxSup;iPix++)
					{
						val+=cpl_vector_get(vecy,iPix-uPixelMinLow);
					}
					valX  += val;
					valX2 += pow(val,2.0);
				}
        
				cpl_vector_delete(vecy);
				cpl_vector_delete(vecw);
				cpl_vector_delete(fit);
        
				valX/=(nStat*1.);
				valX2/=(nStat*1.);
				varX=(valX2-pow(valX,2.0))/(nStat*1.);
				snr=fabs(valX)/sqrt(varX);
				/* printf("%d %d %f\n",n,l,snr); */
				cpl_image_set(vis2err,iWlen+1,iBase,fabs(cpl_image_get(vis2,iWlen+1,iBase,&pis)/snr));

			}
            

		}
	        
	        output=cpl_sprintf("fringePeak_%d.fits",expno);
		cpl_image_save(fringePeak,output, CPL_BPP_IEEE_FLOAT,NULL, CPL_IO_CREATE);
		cpl_free(output);

		if (phot != NULL)
		{
			nbFrameP=phot->nbphotxframe/phot->imgdet->nbregion;

			prodPhot	= cpl_calloc(6*nbFrameP,sizeof(cpl_image *));
			prodPhotErr	= cpl_calloc(6,sizeof(cpl_image *));
				
			for(iFrame=0;iFrame<nbFrameP;iFrame++)
			{
				if (strstr(mode,"s") == NULL && iFrame==1)
				{
					// In the HighSens case, we used the second
					// frame (k=1) which contains the stdev of
					// each pixel of the photometry and we compute
					// the error of the product of 2 photometric
					// channels
					// sig=sqrt(p1**2*sig_p2**2+p2**2*sig_p1**2). We
					// store the results in prodPhotErr.
						
					nbWlen=corrFlux->imgdet->list_region[iReg]->naxis[1];
					for(iBase=0;iBase < nbBases ; iBase++)
					{
						prodPhotErr[iBase]=cpl_image_new(1,nbWlen,CPL_TYPE_DOUBLE);
					}
					for(iWlen=0;iWlen<nbWlen;iWlen++)
					{
						//1st peak : T3T4 3D in L and 6D in N
						val=0.;
						for(iFreq=0 ; iFreq < nbFreq ; iFreq++)
						{
							i1=cpl_image_get(phot->list_phot[(iReg*4+2)*nbFrameP]->imgreg[0],iBase+1,iWlen+1,&pis);
							i2=cpl_image_get(phot->list_phot[(iReg*4+3)*nbFrameP+1]->imgreg[0],iBase+1,iWlen+1,&pis);
							i3=cpl_image_get(phot->list_phot[(iReg*4+3)*nbFrameP]->imgreg[0],iBase+1,iWlen+1,&pis);
							i4=cpl_image_get(phot->list_phot[(iReg*4+2)*nbFrameP+1]->imgreg[0],iBase+1,iWlen+1,&pis);
							if (isnan(i1) == 0 && isnan(i2) == 0 && isnan(i3)==0 && isnan(i4)== 0)
							{
								val+=pow(i1,2.0)*pow(i2,2.0)+pow(i3,2.0)*pow(i4,2.0);
							}
						}
						if(!strcmp(detName, "AQUARIUS"))
						{
							cpl_image_set(prodPhotErr[1],1,iWlen+1,sqrt(val));
						}
						else
						{
							cpl_image_set(prodPhotErr[0],1,iWlen+1,sqrt(val));
						}
						//2nd peak : T1T2 6D in L but 3D in N
						val=0.;
						for(iFreq=0;iFreq<nbFreq;iFreq++)
						{
							i1=cpl_image_get(phot->list_phot[(iReg*4)*nbFrameP]->imgreg[0],iFreq+1,iWlen+1,&pis);
							i2=cpl_image_get(phot->list_phot[(iReg*4+1)*nbFrameP+1]->imgreg[0],iFreq+1,iWlen+1,&pis);
							i3=cpl_image_get(phot->list_phot[(iReg*4+1)*nbFrameP]->imgreg[0],iFreq+1,iWlen+1,&pis);
							i4=cpl_image_get(phot->list_phot[(iReg*4)*nbFrameP+1]->imgreg[0],iFreq+1,iWlen+1,&pis);
							if (isnan(i1) == 0 && isnan(i2) == 0 && isnan(i3)==0 && isnan(i4)== 0)
							{
								val+=pow(i1,2.0)*pow(i2,2.0)+pow(i3,2.0)*pow(i4,2.0);
							}
						}
						if(!strcmp(detName, "AQUARIUS"))
						{
							cpl_image_set(prodPhotErr[0],1,iWlen+1,sqrt(val));
						}
						else
						{
							cpl_image_set(prodPhotErr[1],1,iWlen+1,sqrt(val));
						}
						//3rd peak : T2T3 12D in L and N
						val=0.;
						for(iFreq=0;iFreq<nbFreq;iFreq++)
						{
							i1=cpl_image_get(phot->list_phot[(iReg*4+1)*nbFrameP]->imgreg[0],iFreq+1,iWlen+1,&pis);
							i2=cpl_image_get(phot->list_phot[(iReg*4+2)*nbFrameP+1]->imgreg[0],iFreq+1,iWlen+1,&pis);
							i3=cpl_image_get(phot->list_phot[(iReg*4+2)*nbFrameP]->imgreg[0],iFreq+1,iWlen+1,&pis);
							i4=cpl_image_get(phot->list_phot[(iReg*4+1)*nbFrameP+1]->imgreg[0],iFreq+1,iWlen+1,&pis);
							if (isnan(i1) == 0 && isnan(i2) == 0 && isnan(i3)==0 && isnan(i4)== 0)
							{
								val+=pow(i1,2.0)*pow(i2,2.0)+pow(i3,2.0)*pow(i4,2.0);
							}
						}
						if(!strcmp(detName, "AQUARIUS"))
						{
							cpl_image_set(prodPhotErr[3],1,iWlen+1,sqrt(val));
						}
						else
						{
							cpl_image_set(prodPhotErr[3],1,iWlen+1,sqrt(val));
						}
						//4th peak : T2T4 12D in L but 18D in N
						val=0.;
						for(iFreq=0;iFreq<nbFreq;iFreq++)
						{
							i1=cpl_image_get(phot->list_phot[(iReg*4+3)*nbFrameP]->imgreg[0],iFreq+1,iWlen+1,&pis);
							i2=cpl_image_get(phot->list_phot[(iReg*4+1)*nbFrameP+1]->imgreg[0],iFreq+1,iWlen+1,&pis);
							i3=cpl_image_get(phot->list_phot[(iReg*4+1)*nbFrameP]->imgreg[0],iFreq+1,iWlen+1,&pis);
							i4=cpl_image_get(phot->list_phot[(iReg*4+3)*nbFrameP+1]->imgreg[0],iFreq+1,iWlen+1,&pis);
							if (isnan(i1) == 0 && isnan(i2) == 0 && isnan(i3)==0 && isnan(i4)== 0)
							{
								val+=pow(i1,2.0)*pow(i2,2.0)+pow(i3,2.0)*pow(i4,2.0);
							}
						}
						if(!strcmp(detName, "AQUARIUS"))
						{
							cpl_image_set(prodPhotErr[5],1,iWlen+1,sqrt(val));
						}
						else
						{
							cpl_image_set(prodPhotErr[2],1,iWlen+1,sqrt(val));
						}
						//5th peak : T1T3 18D in L but 12D in N
						val=0.;
						for(iFreq=0;iFreq<nbFreq;iFreq++)
						{
							i1=cpl_image_get(phot->list_phot[(iReg*4)*nbFrameP]->imgreg[0],iFreq+1,iWlen+1,&pis);
							i2=cpl_image_get(phot->list_phot[(iReg*4+2)*nbFrameP+1]->imgreg[0],iFreq+1,iWlen+1,&pis);
							i3=cpl_image_get(phot->list_phot[(iReg*4+2)*nbFrameP]->imgreg[0],iFreq+1,iWlen+1,&pis);
							i4=cpl_image_get(phot->list_phot[(iReg*4)*nbFrameP+1]->imgreg[0],iFreq+1,iWlen+1,&pis);
							if (isnan(i1) == 0 && isnan(i2) == 0 && isnan(i3)==0 && isnan(i4)== 0)
							{
								val+=pow(i1,2.0)*pow(i2,2.0)+pow(i3,2.0)*pow(i4,2.0);
							}
						}
						if(!strcmp(detName, "AQUARIUS"))
						{
							cpl_image_set(prodPhotErr[2],1,iWlen+1,sqrt(val));
						}
						else
						{
							cpl_image_set(prodPhotErr[5],1,iWlen+1,sqrt(val));
						}
						//6th peak : T1T4 15D
						val=0.;
						for(iFreq=0;iFreq<nbFreq;iFreq++)
						{
							i1=cpl_image_get(phot->list_phot[(iReg*4)*nbFrameP]->imgreg[0],iFreq+1,iWlen+1,&pis);
							i2=cpl_image_get(phot->list_phot[(iReg*4+3)*nbFrameP+1]->imgreg[0],iFreq+1,iWlen+1,&pis);
							i3=cpl_image_get(phot->list_phot[(iReg*4+3)*nbFrameP]->imgreg[0],iFreq+1,iWlen+1,&pis);
							i4=cpl_image_get(phot->list_phot[(iReg*4)*nbFrameP+1]->imgreg[0],iFreq+1,iWlen+1,&pis);
							if (isnan(i1) == 0 && isnan(i2) == 0 && isnan(i3)==0 && isnan(i4)== 0)
							{
								val+=pow(i1,2.0)*pow(i2,2.0)+pow(i3,2.0)*pow(i4,2.0);
							}
						}
						if(!strcmp(detName, "AQUARIUS"))
						{
							cpl_image_set(prodPhotErr[4],1,iWlen+1,sqrt(val));
						}
						else
						{
							cpl_image_set(prodPhotErr[4],1,iWlen+1,sqrt(val));
						}
					}
				}
				if (strstr(mode,"s") != NULL || iFrame!=1)
				{
					//1st peak : T3T4 3D in L and 6D in N
					foo=cpl_image_multiply_create(
												  phot->list_phot[(iReg*4+2)*nbFrameP+iFrame]->imgreg[0],
												  phot->list_phot[(iReg*4+3)*nbFrameP+iFrame]->imgreg[0]);
					if(!strcmp(detName, "AQUARIUS"))
					{
						prodPhot[iFrame*6+1]=cpl_image_collapse_create(foo,1);
					}
					else
					{
						prodPhot[iFrame*6]=cpl_image_collapse_create(foo,1);
					}
					cpl_image_delete(foo);
					//2nd peak : T1T2 6D in L but 3D in N
					foo=cpl_image_multiply_create(
												  phot->list_phot[(iReg*4)*nbFrameP+iFrame]->imgreg[0],
												  phot->list_phot[(iReg*4+1)*nbFrameP+iFrame]->imgreg[0]);
					if(!strcmp(detName, "AQUARIUS"))
					{
						prodPhot[iFrame*6+0]=cpl_image_collapse_create(foo,1);
					}
					else
					{
						prodPhot[iFrame*6+1]=cpl_image_collapse_create(foo,1);
					}
					cpl_image_delete(foo);
					//3rd peak : T2T3 12D
					foo=cpl_image_multiply_create(
												  phot->list_phot[(iReg*4+1)*nbFrameP+iFrame]->imgreg[0],
												  phot->list_phot[(iReg*4+2)*nbFrameP+iFrame]->imgreg[0]);
					if(!strcmp(detName, "AQUARIUS"))
					{
						prodPhot[iFrame*6+3]=cpl_image_collapse_create(foo,1);
					}
					else
					{
						prodPhot[iFrame*6+3]=cpl_image_collapse_create(foo,1);
					}
					cpl_image_delete(foo);
					//4th peak : T2T4 9D in L but 18D in N
					foo=cpl_image_multiply_create(
												  phot->list_phot[(iReg*4+1)*nbFrameP+iFrame]->imgreg[0],
												  phot->list_phot[(iReg*4+3)*nbFrameP+iFrame]->imgreg[0]);
					if(!strcmp(detName, "AQUARIUS"))
					{
						prodPhot[iFrame*6+5]=cpl_image_collapse_create(foo,1);
					}
					else
					{
						prodPhot[iFrame*6+2]=cpl_image_collapse_create(foo,1);
					}
					cpl_image_delete(foo);
					//5th peak : T1T3 18D in L but 9D in N
					foo=cpl_image_multiply_create(
												  phot->list_phot[(iReg*4)*nbFrameP+iFrame]->imgreg[0],
												  phot->list_phot[(iReg*4+2)*nbFrameP+iFrame]->imgreg[0]);
					if(!strcmp(detName, "AQUARIUS"))
					{
						prodPhot[iFrame*6+2]=cpl_image_collapse_create(foo,1);
					}
					else
					{
						prodPhot[iFrame*6+5]=cpl_image_collapse_create(foo,1);
					}
					cpl_image_delete(foo);
					//6th peak : T1T4 15D
					foo=cpl_image_multiply_create(
												  phot->list_phot[(iReg*4)*nbFrameP+iFrame]->imgreg[0],
												  phot->list_phot[(iReg*4+3)*nbFrameP+iFrame]->imgreg[0]);
					if(!strcmp(detName, "AQUARIUS"))
					{
						prodPhot[iFrame*6+4]=cpl_image_collapse_create(foo,1);
					}
					else
					{
						prodPhot[iFrame*6+4]=cpl_image_collapse_create(foo,1);
					}
					cpl_image_delete(foo);
				}
			}
		}

    
    
		for(iBase=0;iBase<nbBases;iBase++)
		{
			if (phot != NULL)
			{
				// Compute mean and stdev (in fact error of the mean)
				// of the photometry
				if (strstr(mode,"s") != NULL)
				{
					prodPhotErr[iBase]=cpl_image_new(1,nbWlen, CPL_TYPE_DOUBLE);
					vecw=cpl_vector_new(nbFrameP);
					for(iWlen=0 ; iWlen < nbWlen ; iWlen++)
					{
						for(iFrame=0 ; iFrame < nbFrameP ; iFrame++)
						{
							cpl_vector_set(vecw,iFrame,cpl_image_get(prodPhot[iFrame*6+iBase],1,iWlen+1,&pis));
						}
						cpl_image_set(prodPhot[iBase],1,iWlen+1,cpl_vector_get_mean(vecw));
						cpl_image_set(prodPhotErr[iBase],1,iWlen+1,cpl_vector_get_stdev(vecw)/sqrt(nbFrameP));
					}
					cpl_vector_delete(vecw);
				}
			}
			// compute vis2 et vis2err
			for(iWlen=0 ; iWlen < nbWlen ; iWlen++)
			{
				if (phot != NULL)
				{
					if (cpl_image_get(prodPhot[iBase],1,iWlen+1,&pis) != 0.)
					{
						val=cpl_image_get(vis2,iWlen+1,iBase+1,&pis)/
							cpl_image_get(prodPhot[iBase],1,iWlen+1,&pis)/
							nbFreq;


						val2=(cpl_image_get(vis2err,iWlen+1,iBase+1,&pis)*
							  cpl_image_get(prodPhot[iBase],1,iWlen+1,&pis)+
							  cpl_image_get(vis2,iWlen+1,iBase+1,&pis)*
							  cpl_image_get(prodPhotErr[iBase],1,iWlen+1,&pis))/
							pow(cpl_image_get(prodPhot[iBase],1,iWlen+1,&pis),2.0);
						val2/=nbFreq;



						val2=cpl_image_get(vis2err,iWlen+1,iBase+1,&pis)*cpl_image_get(vis2err,iWlen+1,iBase+1,&pis)+
							cpl_image_get(prodPhotErr[iBase],1,iWlen+1,&pis)*cpl_image_get(prodPhotErr[iBase],1,iWlen+1,&pis)*pow(val,2.0);
						val2/=pow(cpl_image_get(prodPhot[iBase],1,iWlen+1,&pis)*
								  nbFreq,2.0);
						val2=sqrt(val2);

					}
					else
					{
						val=-1.;
						val2=1.E3;
					}
				}
				else
				{
					val=cpl_image_get(vis2,iWlen+1,iBase+1,&pis);
					val2=cpl_image_get(vis2err,iWlen+1,iBase+1,&pis);
				}
				if (isnan(val2))
				{
					val2=1.E3;
				}
				if (isnan(val))
				{
					val=0.;
				}
				//  if (l==0) printf("%f %f %d %f\n",cpl_image_get(vis2,iWlen+1,iBase+1,&pis),cpl_image_get(prodPhot[iBase],1,iWlen+1,&pis),nbFreq,val);

				cpl_image_set(vis2,iWlen+1,iBase+1,val);
				cpl_image_set(vis2err,iWlen+1,iBase+1,fabs(val2));
			}
		}



    
		if (prodPhot != NULL)
		{
			for(iFrame=0;iFrame<6*nbFrameP;iFrame++)
			{
				cpl_image_delete(prodPhot[iFrame]);
			}
			cpl_free(prodPhot);
		}
		if (prodPhotErr != NULL)
		{
			for(iFrame=0;iFrame<6;iFrame++)
			{
				cpl_image_delete(prodPhotErr[iFrame]);
			}
			cpl_free(prodPhotErr);
		}
		// Fill oifits with estimated vis2
		for(iWlen=0;iWlen<nbWlen;iWlen++)
		{
			if (oifits->oivis2->nbvis2 == 6)
			{
				for(iBase=0;iBase<oifits->oivis2->nbvis2;iBase++)
				{
					oifits->oivis2->list_vis2[iBase]->vis2[cptWave]=
						cpl_image_get(vis2,iWlen+1,iBase+1,&pis);
					oifits->oivis2->list_vis2[iBase]->vis2err[cptWave]=
						cpl_image_get(vis2err,iWlen+1,iBase+1,&pis);
					if (nbdetector == 1)
					{
						if (fabs(oifits->oiwave->effwave[cptWave]*1.e6 - 3.5) <= 0.7 || fabs(oifits->oiwave->effwave[cptWave]*1.e6 - 4.84) <= 0.36)
						{
							oifits->oivis2->list_vis2[iBase]->bandflag[cptWave] =
								CPL_FALSE;
						}
						else
						{
							oifits->oivis2->list_vis2[iBase]->bandflag[cptWave] =
								CPL_TRUE;
						}
					}
					else
					{
						if (fabs(oifits->oiwave->effwave[cptWave]*1.e6 - 8.59) <= 0.59 || fabs(oifits->oiwave->effwave[cptWave]*1.e6 - 11.28) <= 1.72)
						{
							oifits->oivis2->list_vis2[iBase]->bandflag[cptWave] =
								CPL_FALSE;
						}
						else
						{
							oifits->oivis2->list_vis2[iBase]->bandflag[cptWave] =
								CPL_TRUE;
						}
					}
				}
			}
			else if (oifits->oivis2->nbvis2 == 3)
			{
				oifits->oivis2->list_vis2[0]->vis2[cptWave]=
					cpl_image_get(vis2,iWlen+1,2,&pis);
				oifits->oivis2->list_vis2[0]->vis2err[cptWave]=
					cpl_image_get(vis2err,iWlen+1,2,&pis);
				//oifits->oivis2->list_vis2[0]->bandflag[cptWave]=CPL_FALSE;
				if (strcmp(bcd2Status,"OUT"))
				{
					oifits->oivis2->list_vis2[1]->vis2[cptWave]=
						cpl_image_get(vis2,iWlen+1,3,&pis);
					oifits->oivis2->list_vis2[1]->vis2err[cptWave]=
						cpl_image_get(vis2err,iWlen+1,3,&pis);
					oifits->oivis2->list_vis2[2]->vis2[cptWave]=
						cpl_image_get(vis2,iWlen+1,5,&pis);
					oifits->oivis2->list_vis2[2]->vis2err[cptWave]=
						cpl_image_get(vis2err,iWlen+1,5,&pis);
				}
				else
				{
					oifits->oivis2->list_vis2[1]->vis2[cptWave]=
						cpl_image_get(vis2,iWlen+1,4,&pis);
					oifits->oivis2->list_vis2[1]->vis2err[cptWave]=
						cpl_image_get(vis2err,iWlen+1,4,&pis);
					oifits->oivis2->list_vis2[2]->vis2[cptWave]=
						cpl_image_get(vis2,iWlen+1,6,&pis);
					oifits->oivis2->list_vis2[2]->vis2err[cptWave]=
						cpl_image_get(vis2err,iWlen+1,6,&pis);
				}
				oifits->oivis2->list_vis2[0]->bandflag[cptWave] = CPL_FALSE;
				oifits->oivis2->list_vis2[1]->bandflag[cptWave] = CPL_FALSE;
				oifits->oivis2->list_vis2[2]->bandflag[cptWave] = CPL_FALSE;
			}
			else
			{
				oifits->oivis2->list_vis2[0]->vis2[cptWave]=
					cpl_image_get(vis2,iWlen+1,2,&pis);
				oifits->oivis2->list_vis2[0]->vis2err[cptWave]=
					cpl_image_get(vis2err,iWlen+1,2,&pis);
				oifits->oivis2->list_vis2[0]->bandflag[cptWave]=CPL_FALSE;
			}
			cptWave++;
		}

		/* cpl_plot_image_row("set grid;set yrange [0.:1.5];","with lines","",vis2,1,6,1);*/
		/* cpl_plot_image_row("set grid;","with lines","",vis2err,1,6,1); */
		cpl_image_delete(vis2);
		cpl_image_delete(vis2err);
	}



	// QC parameter Computation
        cpl_vector *valqc=NULL;
	valqc=cpl_vector_new(oifits->oivis2->nbvis2);
	for(iBase=0;iBase<oifits->oivis2->nbvis2;iBase++)
	{
		val=0.;
		val2=0.;
		cptWave=0;
		for(iWlen=0;iWlen<oifits->oivis2->nbchannel;iWlen++)
		{
			if (nbdetector == 1)
			{
			        if (fabs(oifits->oiwave->effwave[iWlen]*1.e6 - 3.5) <= 0.55 || fabs(oifits->oiwave->effwave[iWlen]*1.e6 - 4.75) <= 0.25)
				  //if (fabs(oifits->oiwave->effwave[iWlen]*1.e6 - 3.5) <= 0.7 || fabs(oifits->oiwave->effwave[iWlen]*1.e6 - 4.84) <= 0.36)
				{
					val+=oifits->oivis2->list_vis2[iBase]->vis2[iWlen];
					val2+=pow(oifits->oivis2->list_vis2[iBase]->vis2err[iWlen],2.0);
					cptWave++;
				}
			}
			else if (nbdetector == 2)
			{
			        if (fabs(oifits->oiwave->effwave[iWlen]*1.e6 - 9.0) <= 1.0)
				  //if (fabs(oifits->oiwave->effwave[iWlen]*1.e6 - 8.59) <= 0.59 || fabs(oifits->oiwave->effwave[iWlen]*1.e6 - 11.28) <= 1.72)
				{
					val+=oifits->oivis2->list_vis2[iBase]->vis2[iWlen];
					val2+=pow(oifits->oivis2->list_vis2[iBase]->vis2err[iWlen],2.0);
					cptWave++;
				}
			}
			else
			{
				cptWave=1;
				val=0.;
				val2=0.;
			}
		}
		val	 /=	cptWave;
		val2 /=	cptWave;

		if (val2 < 0)  val2=fabs(val2);
		if (val2 == 0) val2=1.E-6;
            
		snprintf(kwd, 64, "ESO QC DET%d VISSQR%d",nbdetector,iBase+1);
		cpl_propertylist_append_double(oifits->keywords, kwd, val);
		snprintf(kwd, 64, "ESO QC DET%d VISSQR%d SNR",nbdetector,iBase+1);
		cpl_propertylist_append_double(oifits->keywords, kwd, val/sqrt(val2));
		cpl_vector_set(valqc,iBase,val);
	}
	cpl_propertylist_append_double(oifits->keywords, "ESO QC VIS AVG", cpl_vector_get_mean(valqc));
	cpl_propertylist_append_double(oifits->keywords, "ESO QC VIS STDEV", cpl_vector_get_stdev(valqc));
	cpl_vector_delete(valqc);
	
    
	// Add Generic QC parameters
	cpl_propertylist *qclist=NULL;
	qclist = cpl_propertylist_new();
	mat_add_generic_qc(oifits->keywords,qclist);
	cpl_propertylist_append(oifits->keywords,qclist);
	cpl_propertylist_delete(qclist);
    
	cpl_vector_delete(dispCoef);

	cpl_image_delete(fringePeak);
  
	if (densitySpecSky != NULL)
	{
		for(iReg=0 ; iReg < nbRegion ; iReg++)
		{
			cpl_image_delete(densitySpecSky[iReg]);
		}
		cpl_free(densitySpecSky); 
	}
	if (densitySpecNoFringe != NULL)
	{
		for(iReg=0 ; iReg < nbRegion ; iReg++)
		{
			cpl_image_delete(densitySpecNoFringe[iReg]);
		}
		cpl_free(densitySpecNoFringe); 
	}
  
	for(iReg=0 ; iReg < nbRegion ; iReg++)
	{
		cpl_image_delete(densitySpec[iReg]);
	}
	for(iReg=0;iReg<nbRegion*nStat;iReg++)
	{
		cpl_image_delete(densitySpecSub[iReg]);
	}
	cpl_free(densitySpec);
	cpl_free(densitySpecSub);
	return(0);
}

