/* $Id: mat_corrflux.c,v0.5 2014-06-15 12:56:21 pberio Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: pberio $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_corrflux.c $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
#include <stdio.h>
/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include "mat_corrflux.h"
#include "mat_error.h"

/*-----------------------------------------------------------------------------
                                   Define
 -----------------------------------------------------------------------------*/



/*-----------------------------------------------------------------------------
                                   Functions prototypes
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
/**
  @ingroup corrflux
  @brief Load OBJ_CORR_FLUX data in mat_corrflux structure
  @param frame           current frame
  @return mat_corrflux
 */
/*-----------------------------------------------------------------------------*/
mat_corrflux * mat_corrflux_load(cpl_frame *frame)
{
  mat_corrflux *corrflux=NULL;
  int i=0;
  int j=0;
  int m=0;
  int l=0;
  int index=0;
  int nbExtent=0;
  int flagDet=0;
  int flagCorrFlux=0;
  int dimx=0;
  int dimy=0;
  int cpt=0;
  char str[50];
  char *keyExtname=NULL;
  cpl_type type;
  cpl_array *arr=NULL;

  cpl_propertylist *plist=NULL;
  cpl_table *table=NULL;


 //test if frame is empty
  if (frame == NULL){
    cpl_msg_info(cpl_func,"frame empty");
    return NULL;
  }

  // Allocate the structure and initialize its attributes
  corrflux = cpl_calloc(1, sizeof(mat_corrflux));
  if (corrflux == NULL) {
    cpl_msg_error(cpl_func,"could not allocate memory for corrflux");
    return NULL;
  }
  corrflux->keywords= cpl_propertylist_load(cpl_frame_get_filename(frame),0);
  corrflux->imgdet=NULL;
  corrflux->list_corr=NULL;

  //Load IMAGING_DETECTOR
  nbExtent=cpl_frame_get_nextensions(frame);
  for(i=0;i<nbExtent;i++) {
    plist=cpl_propertylist_load(cpl_frame_get_filename(frame),i+1);
    keyExtname=(char *)cpl_propertylist_get_string(plist,"EXTNAME");

    if (keyExtname != NULL) {
      table=cpl_table_load(cpl_frame_get_filename(frame),i+1,0);
      // Load data in attribute imgdet
      if (!strcmp(keyExtname,"IMAGING_DETECTOR")) {
  	cpl_msg_info(cpl_func, "loading the IMAGING_DETECTOR extension");
  	corrflux->imgdet = mat_imagingdetector_from_table(plist, table);
        /* Error handling */
        if (corrflux->imgdet == NULL) {
          cpl_error_set_message(cpl_func,CPL_ERROR_UNSPECIFIED,
  			                 "Could not load IMAGING_DETECTOR in frame: %s",
  			                 cpl_frame_get_filename(frame));
        }
        else {
  	  flagDet=1;
        }
      }
      cpl_table_delete(table);
    }
    cpl_propertylist_delete(plist);
  }
  if (flagDet == 1) {

    for(i=0;i<nbExtent;i++) {
      plist=cpl_propertylist_load(cpl_frame_get_filename(frame),i+1);
      keyExtname=(char *)cpl_propertylist_get_string(plist,"EXTNAME");
      
      if (keyExtname != NULL) {
    	table=cpl_table_load(cpl_frame_get_filename(frame),i+1,0);
	if (!strcmp(keyExtname,"OBJ_CORR_FLUX")) {
	  cpl_msg_info(cpl_func, "loading the OBJ_CORR_FLUX extension");
	  corrflux->nbframe=cpl_propertylist_get_long(plist,"NAXIS2")*
	    corrflux->imgdet->nbregion;
	  corrflux->list_corr=cpl_malloc(corrflux->nbframe*sizeof(mat_imgcorr *));
	  if (corrflux->list_corr==NULL) {
	    cpl_msg_error(cpl_func,"could not allocate memory for corrflux->list_corr");
	    return NULL;
	  }
	  cpt=0;
	  for(i=0;i<corrflux->imgdet->nbregion;i++) {
	    for(j=0;j<corrflux->nbframe/corrflux->imgdet->nbregion;j++) {
	      corrflux->list_corr[cpt]=cpl_malloc(sizeof(mat_imgcorr));
	      if (corrflux->list_corr[cpt]==NULL) {
	      	cpl_msg_error(cpl_func,"could not allocate memory for corrflux->list_corr[cpt]");
	      	return NULL;
	      }
	      corrflux->list_corr[cpt]->time=
	      	  cpl_table_get_double(table,"TIME",j,NULL);
	      corrflux->list_corr[cpt]->exptime=
	      	  cpl_table_get_float(table,"EXPTIME",j,NULL);

	      corrflux->list_corr[cpt]->numregion=corrflux->imgdet->list_region[i]->numregion;
	      corrflux->list_corr[cpt]->numdetector=corrflux->imgdet->list_region[i]->numdetector;
	      corrflux->list_corr[cpt]->nbimgreg=2;
	      corrflux->list_corr[cpt]->imgreg=cpl_calloc(corrflux->list_corr[cpt]->nbimgreg,
	      						  sizeof(cpl_image *));

	      snprintf(str,50,"CORRFLUXREAL%d",corrflux->list_corr[cpt]->numregion);
	      dimx=corrflux->imgdet->list_region[i]->naxis[0];
	      dimy=corrflux->imgdet->list_region[i]->naxis[1];
	      arr=(cpl_array *)cpl_table_get_array(table,str,j);
	      type=cpl_array_get_type(arr);
	      corrflux->list_corr[cpt]->imgreg[0]=cpl_image_new(dimx,dimy,type);
	      
	      index=0;
	      for(m=0;m<dimy;m++) {
	      	for(l=0;l<dimx;l++) {
	      	  cpl_image_set(corrflux->list_corr[cpt]->imgreg[0],l+1,m+1,
	      			cpl_array_get_float(arr,index,NULL));
	      	  index++;
	      	}
	      }
	      
	      snprintf(str,50,"CORRFLUXIMAG%d",corrflux->list_corr[cpt]->numregion);
	      dimx=corrflux->imgdet->list_region[i]->naxis[0];
	      dimy=corrflux->imgdet->list_region[i]->naxis[1];
	      arr=(cpl_array *)cpl_table_get_array(table,str,j);
	      type=cpl_array_get_type(arr);
	      corrflux->list_corr[cpt]->imgreg[1]=cpl_image_new(dimx,dimy,type);
	      
	      index=0;
	      for(m=0;m<dimy;m++) {
	      	for(l=0;l<dimx;l++) {
	      	  cpl_image_set(corrflux->list_corr[cpt]->imgreg[1],l+1,m+1,
	      			cpl_array_get_float(arr,index,NULL));
	      	  index++;
	      	}
	      }
	      cpt++;
	    }
	  }
	  /* Error handling */
	  if (corrflux->list_corr == NULL) {
	    cpl_error_set_message(cpl_func,CPL_ERROR_UNSPECIFIED,
	  			  "Could not load OBJ_CORR_FLUX in frame: %s",
	  			  cpl_frame_get_filename(frame));
	  } else {
	    flagCorrFlux=1;
	  }

	}
      	cpl_table_delete(table);
      }
      cpl_propertylist_delete(plist);
    }
  }
  if ( flagDet==0 || flagCorrFlux==0) {
    cpl_error_set_message(cpl_func,CPL_ERROR_ILLEGAL_INPUT,
                          "Extension missing in frame: %s",
                          cpl_frame_get_filename(frame));
  }
  
  return corrflux;
}
    





/*----------------------------------------------------------------------------*/
/**
  @ingroup corrflux
  @brief delete structure mat_corrflux
  @param  corr          current structure
  @return error code
 */
/*-----------------------------------------------------------------------------*/
cpl_error_code mat_corrflux_free(mat_corrflux *corr) {
  int i=0;

  // Check input parameters
  mat_assert_value((corr!=NULL),CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT,"no corr argument given");


  if (corr->keywords != NULL)
    {
      cpl_propertylist_delete(corr->keywords);
      corr->keywords=NULL;
    }
  if (corr->imgdet != NULL)
    {
      mat_imagingdetector_delete(corr->imgdet);
      corr->imgdet = NULL;
    }
  for(i=0; i<corr->nbframe; i++)
    {
      if (corr->list_corr[i] != NULL)
	{
	  if (corr->list_corr[i]->imgreg != NULL)
	    {
	      if (corr->list_corr[i]->imgreg[0] != NULL)
		{
		  cpl_image_delete(corr->list_corr[i]->imgreg[0]);
		  corr->list_corr[i]->imgreg[0] = NULL;
		}
	      if (corr->list_corr[i]->imgreg[1] != NULL)
		{
		  cpl_image_delete(corr->list_corr[i]->imgreg[1]);
		  corr->list_corr[i]->imgreg[1] = NULL;
		}
	      cpl_free(corr->list_corr[i]->imgreg);
	      corr->list_corr[i]->imgreg = NULL;
	    }
	  cpl_free(corr->list_corr[i]);
	  corr->list_corr[i] = NULL;
	}
    }
  if (corr->list_corr != NULL)
    {
      cpl_free(corr->list_corr);
      corr->list_corr = NULL;
    }
  cpl_free(corr);
  return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
   @ingroup corrflux
   @brief Save a mat_corrflux structure in a OBJ_CORR_FLUX FITS file
   @param set_in            input frameset
   @param set_sel           selected frameset
   @param parlist           recipe parameter list
   @param recipe_name       name of recipe
   @param pro_cat           PRO.CATG
   @param pro_tech          PRO.TECH
   @param pro_sci           PRO.SCIENCE
   @param corr              mat_corrflux structure to save
   @param pheader           header of the binary table
   @param filename          name of the FITS file
   @return cpl_error_code
*/
/*-----------------------------------------------------------------------------*/

cpl_error_code mat_corrflux_save(
    mat_corrflux *corr,
    cpl_propertylist *pheader, 
    cpl_frameset *set_in,
    cpl_frameset *set_sel,
    const cpl_parameterlist *parlist,
    const char *recipe_name,
    const char *pro_cat,
    const char *pro_tech,
    const char *pro_sci,
    char *filename) {
  
    cpl_table *corr_map=NULL;
    cpl_propertylist *applist=NULL;
    cpl_propertylist *plist=NULL;
    cpl_array *map_array=NULL;
    cpl_array *arrDim=NULL;
    int i=0;
    int j=0;
    int cpt=0;
    char str[50];
    float *map=NULL;
    cpl_errorstate prestate= cpl_errorstate_get();

    (void) pro_sci;
    // Check input parameters
    mat_assert_value((corr!=NULL),CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT, 
		     "no corr argument given");
    mat_assert_value((set_in!=NULL),CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT, 
		     "no cpl_frameset (set_in) argument given");
    mat_assert_value((set_sel!=NULL),CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT, 
		     "no cpl_frameset (set_sel) argument given");
    mat_assert_value((filename!=NULL),CPL_ERROR_NULL_INPUT, 
		     CPL_ERROR_NULL_INPUT, "no filename given");

    cpl_propertylist_erase(pheader,"XTENSION");
    cpl_propertylist_erase(pheader,"BITPIX");
    cpl_propertylist_erase_regexp(pheader,"NAXIS",0);
    cpl_propertylist_erase_regexp(pheader,"COUNT",0);
    cpl_propertylist_erase(pheader,"TFIELDS");
    cpl_propertylist_erase_regexp(pheader,"SIM",0);
    cpl_propertylist_erase_regexp(pheader,"FORM",0);
    cpl_propertylist_erase_regexp(pheader,"TTYPE",0);
    cpl_propertylist_erase_regexp(pheader,"TDIM",0);
    cpl_propertylist_erase_regexp(pheader,"ESO PRO",0);

    // Add the keywords from the PRO categories
    if (pro_tech != NULL) {
	cpl_propertylist_append_string(corr->keywords, CPL_DFS_PRO_TECH,pro_tech);
    }
    if (pro_cat != NULL) {
	cpl_propertylist_append_string(corr->keywords, CPL_DFS_PRO_CATG,pro_cat);
    }
    if (pro_sci != NULL) {
      if (strstr(pro_sci,"F") != NULL)
	{
	  cpl_propertylist_append_bool(corr->keywords, CPL_DFS_PRO_SCIENCE, 0);
	}
      else
	{
	  cpl_propertylist_append_bool(corr->keywords, CPL_DFS_PRO_SCIENCE, 1);
	}
    }
 
    // Prepare the keywords of the CORR_BEAMS binary table
    plist=cpl_propertylist_new();  
    cpl_propertylist_append_string(plist,"EXTNAME","OBJ_CORR_FLUX");
    //cpl_propertylist_append_string(plist,"ORIGIN",corr->imgdet->origin);
    cpl_propertylist_append_string(plist,"INSTRUME",corr->imgdet->instrument);
    cpl_propertylist_append_double(plist,"MJD-OBS",corr->imgdet->dateobsmjd);
    cpl_propertylist_append_string(plist,"DATE-OBS",corr->imgdet->dateobs);
    //cpl_propertylist_append_string(plist,"DATE",corr->imgdet->date);
    cpl_propertylist_append_string(plist,"ESO DET DID",
				   corr->imgdet->dcsdictionaryid);
    cpl_propertylist_append_string(plist,"ESO DET ID",corr->imgdet->dcsid);
    cpl_propertylist_append_int(plist,"NREGION",corr->imgdet->nbregion);
    cpl_propertylist_append_int(plist,"MAXTEL",corr->imgdet->nbtel);


    // Fill a cpl_table with corrflux data that will replace the IMAGING_DATA
    // Add columns
    corr_map = cpl_table_new(corr->nbframe/corr->imgdet->nbregion);
    cpl_table_new_column(corr_map, "TIME", CPL_TYPE_DOUBLE);
    cpl_table_new_column(corr_map, "EXPTIME", CPL_TYPE_FLOAT);
    for(j=0;j<corr->imgdet->nbregion;j++) {
    	snprintf(str,50,"CORRFLUXREAL%d",j+1);
	
    	cpl_table_new_column_array(corr_map, str, CPL_TYPE_FLOAT,
    				   cpl_image_get_size_x
    				   (corr->list_corr[0]->imgreg[0]) *
    				   cpl_image_get_size_y
    				   (corr->list_corr[0]->imgreg[0]));
    	arrDim=cpl_array_new(2,CPL_TYPE_INT);
    	cpl_array_set(arrDim,0,cpl_image_get_size_x
    		      (corr->list_corr[0]->imgreg[0]));
    	cpl_array_set(arrDim,1,cpl_image_get_size_y
    		      (corr->list_corr[0]->imgreg[0]));
    	cpl_table_set_column_dimensions(corr_map,str,arrDim);
    	cpl_array_delete(arrDim);


    	snprintf(str,50,"CORRFLUXIMAG%d",j+1);
    	cpl_table_new_column_array(corr_map, str, CPL_TYPE_FLOAT,
    				   cpl_image_get_size_x
    				   (corr->list_corr[0]->imgreg[1]) *
    				   cpl_image_get_size_y
    				   (corr->list_corr[0]->imgreg[1]));
    	arrDim=cpl_array_new(2,CPL_TYPE_INT);
    	cpl_array_set(arrDim,0,
    		      cpl_image_get_size_x(corr->list_corr[0]->imgreg[0]));
    	cpl_array_set(arrDim,1,
    		      cpl_image_get_size_y(corr->list_corr[0]->imgreg[0]));
    	cpl_table_set_column_dimensions(corr_map,str,arrDim);
    	cpl_array_delete(arrDim);
    }


    // Get the data in a cpl_array
    cpt=0;
    for(i=0; i<corr->nbframe/corr->imgdet->nbregion;i++) {
    	for(j=0;j<corr->imgdet->nbregion;j++) {
    	    if (j==0) {
    		cpl_table_set_double(corr_map,"TIME",i,
    				     corr->list_corr[cpt]->time);
    		cpl_table_set_float(corr_map,"EXPTIME",i,
    				    corr->list_corr[cpt]->exptime);
    	    }
    	    map = (float *)cpl_image_get_data(corr->list_corr[cpt]->imgreg[0]);
    	    map_array = cpl_array_wrap_float
    	    	(map,cpl_image_get_size_x(corr->list_corr[0]->imgreg[0]) *
    	    	 cpl_image_get_size_y(corr->list_corr[0]->imgreg[0]));
    	    snprintf(str,50,"CORRFLUXREAL%d",j+1);
    	    cpl_table_set_array(corr_map,str,i,map_array);
    	    cpl_array_unwrap(map_array);
      
    	    map = (float *)cpl_image_get_data(corr->list_corr[cpt]->imgreg[1]);
    	    map_array = cpl_array_wrap_float
    	    	(map,cpl_image_get_size_x(corr->list_corr[0]->imgreg[1]) *
    	    	 cpl_image_get_size_y(corr->list_corr[0]->imgreg[1]));
    	    snprintf(str,50,"CORRFLUXIMAG%d",j+1);
    	    cpl_table_set_array(corr_map,str,i,map_array);
    	    cpl_array_unwrap(map_array);
    	    cpt++;
    	}
    }


    cpl_propertylist_erase(corr->keywords,"RADECSYS");
    cpl_propertylist_erase(pheader,"RADECSYS");
    // Make a new FITS file with kappa keywords and the created table
    applist=cpl_propertylist_duplicate(corr->keywords);
    cpl_dfs_save_table(set_in, pheader, parlist, set_sel, NULL,
		       corr_map, plist, recipe_name, applist, NULL,
		       PACKAGE "/" PACKAGE_VERSION, filename);
    // Append the IMAGING_DETECTOR table
    mat_imagingdetector_save(corr->imgdet,corr->keywords,filename);
    if (corr_map != NULL) cpl_table_delete(corr_map);
    if (applist != NULL) cpl_propertylist_delete(applist);
    if (plist != NULL) cpl_propertylist_delete(plist);


 
    if (!cpl_errorstate_is_equal(prestate)) { 
    
	cpl_errorstate_set(prestate);
    
	return CPL_ERROR_UNSPECIFIED;
    }
    return CPL_ERROR_NONE;

}



