/* $Id: mat_detector.c,v0.5 2014-06-15 12:56:21 mheininger Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: mheininger $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_detector.c $
 */

#include <stdlib.h>
#include <math.h>

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
#include <stdio.h>
/*-----------------------------------------------------------------------------
  Includes
  -----------------------------------------------------------------------------*/

#include <string.h>
#include <strings.h>

#include "mat_detector.h"

/*-----------------------------------------------------------------------------
  Define
  -----------------------------------------------------------------------------*/

#define MATISSE_DET_CHIP_ID        "ESO DET CHIP ID"
#define MATISSE_DET_CHIP_NAME      "ESO DET CHIP NAME"
#define MATISSE_DET_CHIP_NX        "ESO DET CHIP NX"
#define MATISSE_DET_CHIP_NY        "ESO DET CHIP NY"
#define MATISSE_DET_READ_CURID     "ESO DET READ CURID"
#define MATISSE_DET_READ_CURNAME   "ESO DET READ CURNAME"

const mat_detector DETECTOR_UNKNOWN = {
  MAT_UNKNOWN_DET, 1, "-1", "UNKNOWN",     // detector type, nr, id, name
  0, 0,                                    // nx, ny
  1, "DEFAULT",                            // read mode: CURID, CURNAME
  1, 1, 0, 0,                              // detector channel: nrows, ncols, nx, ny
  0, 0, 0, 0,                              // metal frame: left, right, bottom, top
  1, 0, 0,                                 // sub-windows: n, nx, ny
  {0.0, 1.0, 0.0, 0.0},                    // horizontal pixel timing: [0] + nx*[1] + ny*[2] + nx*ny*[3]
  {0.0, 0.0, 0.0, 1.0},                    // vertical pixel timing: [0] + nx*[1] + ny*[2] + nx*ny*[3]
  65536.0,                                 // top of the linear range for a detector in DU
  "map.fits"
};

const mat_detector DETECTOR_MATISSE_HAWAII = {
  MAT_HAWAII2RG_DET, 1, "X", "HAWAII-2RG", // detector type, nr, id, name
  2048, 2048,                              // nx, ny
  1, "DEFAULT",                            // read mode: CURID, CURNAME
  1, 32, 0, 0,                             // detector channel: nrows, ncols, nx, ny
  128, 128, 40, 56,                        // metal frame: left, right, bottom, top
  1, 0, 0,                                 // sub-windows: n, nx, ny
  {640.0, 0.0, 0.0, 0.0},                  // horizontal pixel timing: [0] + nx*[1] + ny*[2] + nx*ny*[3]
  {0.0, 0.0, 640.0, 0.0},                  // vertical pixel timing: [0] + nx*[1] + ny*[2] + nx*ny*[3]
  25000.0,                                 // top of the linear range for a detector in DU
  "map.fits"
};

const mat_detector DETECTOR_MATISSE_AQUARIUS = {
  MAT_AQUARIUS_DET, 2, "X", "AQUARIUS",    // detector type, nr, id, name
  1024, 1024,                              // nx, ny
  1, "DEFAULT",                            // read mode: CURID, CURNAME
  2, 32, 0, 0,                             // detector channel: nrows, ncols, nx, ny
  32, 32, 36, 36,                          // metal frame: left, right, bottom, top
  1, 0, 0,                                 // sub-windows: n, nx, ny
  {32.0, 0.0, 0.0, 0.0},                   // horizontal pixel timing: [0] + nx*[1] + ny*[2] + nx*ny*[3]
  {0.0, 0.0, 32.0, 0.0},                   // vertical pixel timing: [0] + nx*[1] + ny*[2] + nx*ny*[3]
  15000.0,                                 // top of the linear range for a detector in DU
  "map.fits"
};

/*-----------------------------------------------------------------------------
  Functions prototypes
  -----------------------------------------------------------------------------*/

/**
   @ingroup det
   @brief Creates an empty detector specification.
   @return This function returns a new data structure for a detector specification or NULL.

   This function creates a detector specification data structure. This data structure must
   be freed using the mat_detector_delete function.
*/
mat_detector *mat_detector_new(void)
{
  mat_detector *det = NULL;
  det = (mat_detector *)cpl_calloc(1, sizeof(mat_detector));
  if (det == NULL)
    {
      cpl_msg_error(cpl_func, "could not allocate memory for a mat_detector");
      return NULL;
    }
  det->type = MAT_UNKNOWN_DET;
  det->nr = 0;
  det->id[0] = '\0';
  det->name[0] = '\0';
  det->read_name[0] = '\0';
  det->channel_nrows = 1;
  det->channel_ncolumns = 1;
  det->horizontal_timing[0] = 0.0;
  det->horizontal_timing[1] = 1.0;
  det->horizontal_timing[2] = 0.0;
  det->horizontal_timing[3] = 0.0;
  det->vertical_timing[0] = 0.0;
  det->vertical_timing[1] = 0.0;
  det->vertical_timing[2] = 0.0;
  det->vertical_timing[3] = 1.0;
  det->linear_range = 0.0;
  return det;
}

/**
   @ingroup det
   @brief Copy a detector specification.
   @param det     This structure describes a detector.
   @return This function returns a copy of the detector specification or NULL.

   This function creates a new detector specification data structure and copyies
   the given detector specificatiobn into it. This data structure must
   be freed using the mat_detector_delete function.
*/
mat_detector *mat_detector_duplicate(mat_detector *det)
{
  mat_detector *ndet = NULL;

  if (det == NULL)
    {
      cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_detector (det) argument given");
      return NULL;
    }
  ndet = mat_detector_new();
  if (ndet == NULL)
    {
      return NULL;
    }
  *ndet = *det;
  return ndet;
}

/**
   @ingroup det
   @brief Delete a detector specification.
   @param det     This structure describes a detector.
   @return the cpl_error_code or CPL_ERROR_NONE

   A previously created detector specification is deleted from memory.

   Possible cpl_error_code set in this function:
   - CPL_ERROR_NULL_INPUT if an input pointer is NULL
*/
cpl_error_code mat_detector_delete(mat_detector *det)
{
  if (det == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_detector (det) argument given");
    }
  cpl_free(det);
  return CPL_ERROR_NONE;
}

/**
   @ingroup det
   @brief Fills a detector specification using a property list.
   @param det    This structure will describe a detector (result).
   @param plist  Contains the keywords as a property list
   @return the cpl_error_code or CPL_ERROR_NONE

   This function uses the contents of a property list to fill a mat_detector data structure
   with all necessary information about a detector. This includes the size of a metal frame
   on the detector edges (used for getting dark reference values), the channel structure of
   the detector and the pixel timings.

   The keywords "ESO DET CHIP NX" and "ESO DET CHIP NY" specifies the number of pixels
   of the detector. The keyword "ESO DET CHIP NAME" is used to distinguish between:

   "AQ..."  => the detector is the Aquarius detector, used for the N-band
   "HA..."  => the detector is the HAWAII-2RG detector used fr the L/M-band

   If the detector name does not contain one of the above patterns, a plain detector
   is used (no metal frame, unknown timing, one channel).

   Possible cpl_error_code set in this function:
   - CPL_ERROR_NULL_INPUT if an input pointer is NULL
*/
cpl_error_code mat_detector_decode(mat_detector *det, cpl_propertylist *plist)
{
  const char *detid    = NULL;
  const char *detname  = NULL;
  int         readid   = 0;
  const char *readname = NULL;
  int         nx, ny;

  if (det == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_detector (det) argument given");
    }
  if (plist == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no cpl_propertylist (plist) argument given");
    }
  cpl_error_reset();
  detid = cpl_propertylist_get_string(plist, MATISSE_DET_CHIP_ID);
  if (cpl_error_get_code() == CPL_ERROR_DATA_NOT_FOUND)
    {
      cpl_msg_warning(cpl_func, "property list does not contain the detector chip id ESO DET CHIP ID");
      cpl_error_reset();
      detid = NULL;
    }
  detname = cpl_propertylist_get_string(plist, MATISSE_DET_CHIP_NAME);
  if (cpl_error_get_code() == CPL_ERROR_DATA_NOT_FOUND)
    {
      cpl_msg_warning(cpl_func, "property list does not contain the detector chip name ESO DET CHIP NAME");
      cpl_error_reset();
      detname = NULL;
    }
  readid = cpl_propertylist_get_int(plist, MATISSE_DET_READ_CURID);
  if (cpl_error_get_code() == CPL_ERROR_DATA_NOT_FOUND)
    {
      cpl_msg_warning(cpl_func, "propertylist does not contain the readout mode id ESO DET READ CURID, using 0");
      cpl_error_reset();
      readid = 0;
    }
  readname = cpl_propertylist_get_string(plist, MATISSE_DET_READ_CURNAME);
  if (cpl_error_get_code() == CPL_ERROR_DATA_NOT_FOUND)
    {
      cpl_msg_warning(cpl_func, "property list does not contain the readout mode name ESO DET READ CURNAME");
      cpl_error_reset();
      readname = NULL;
    }
  det->nx = cpl_propertylist_get_int(plist, MATISSE_DET_CHIP_NX);
  det->ny = cpl_propertylist_get_int(plist, MATISSE_DET_CHIP_NY);
  if (cpl_error_get_code() == CPL_ERROR_DATA_NOT_FOUND)
    {
      cpl_msg_warning(cpl_func, "property list does not contain the detector dimensions ESO DET CHIP NX and/or ESO DET CHIP NY");
      cpl_error_reset();
      nx = 0;
      ny = 0;
    }
  
  // use the detector name (ESO DET CHIP NAME) to initialize the data structure
  if (detname == NULL)
    {
      *det = DETECTOR_UNKNOWN;
      det->nx = nx;
      det->ny = ny;
    }
  else if (strncasecmp(detname, "HA", 2) == 0)
    {
      *det = DETECTOR_MATISSE_HAWAII;
    }
  else if (strncasecmp(detname, "AQ", 2) == 0)
    {
      *det = DETECTOR_MATISSE_AQUARIUS;
    }
  else
    {
      cpl_msg_warning(cpl_func, "unknown detector type: %s", detname);
      *det = DETECTOR_UNKNOWN;
      det->nx = nx;
      det->ny = ny;
    }
  // copy all variable values into the stucture
  // Store the unique detector id, because all input file must have the same detector id.
  if (detid != NULL)
    {
      strncpy(det->id, detid, 79);
    }
  // Store the detector name (only the two first characters were compared!)
  if (detname != NULL)
    {
      strncpy(det->name, detname, 79);
    }
  // Store the read id, modify the timing for some HAWAII-2RG modes
  det->read_id = readid;
  if (det->type == MAT_HAWAII2RG_DET)
    {
      if ((det->read_id == 3) || (det->read_id == 4))
	{
	  /*
	   * the third and fourth readmode uses a 1 MHz pixel clock (needs verification)
	   */
	  det->horizontal_timing[0] = 64.0; /* 64 us for all 64 pixels of a detector channel */
	  det->vertical_timing[2] = 64.0;  /* 64 us for all 64 pixels of a detector channel */
	}
    }
  // Store the read name
  if (readname != NULL)
    {
      strncpy(det->read_name, readname, 79);
    }
  // the following values are always calculated and not only for unknown detectors
  det->channel_nx = det->nx/det->channel_ncolumns;
  det->channel_ny = det->ny/det->channel_nrows;
  cpl_msg_debug(cpl_func, "det=%d, %d x %d pixels, %d x %d channels, frame=%d %d %d %d",
		det->type,
		det->nx, det->ny,
		det->channel_ncolumns, det->channel_nrows,
		det->frame_left, det->frame_right, det->frame_bottom, det->frame_top);
  return CPL_ERROR_NONE;
}

/**
   @ingroup det
   @brief Derives the detector specification from raw data.
   @param det   This structure will describe a detector (result).
   @param raw   This structure contains raw data.
   @return the cpl_error_code or CPL_ERROR_NONE

   This function uses a data structure containing raw data to fill a data structure
   specifying a detector. For getting the detector information, the function
   mat_detector_decode is used. In addition, the size of the raw data (number of
   regions and total size) is stored in the mat_detector data structure.

   Possible cpl_error_code set in this function:
   - CPL_ERROR_NULL_INPUT if an input pointer is NULL
*/
cpl_error_code mat_detector_decode_raw(mat_detector *det, mat_gendata *raw)
{
  if (det == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_detector (det) argument given");
    }
  if (raw == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_gendata (raw) argument given");
    }
  if (raw->keywords == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no cpl_propertylist (raw->keywords) argument given");
    }
  if (raw->imgdet == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no imaging detector (raw->imgdet) argument given");
    }
  if (raw->imgdet->list_region == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no regions (raw->imgdet->list_region) argument given");
    }
  if (raw->imgdet->list_region[0] == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no regions (raw->imgdet->list_region[0]) argument given");
    }
  if (raw->imgdata == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no imaging data (raw->imgdata) argument given");
    }
  if (raw->imgdata->list_frame == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no frames in imaging data (raw->imgdata->list_frame) argument given");
    }
  if (raw->imgdata->list_frame[0] == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no frames in imaging data (raw->imgdata->list_frame[0]) argument given");
    }
  if (raw->imgdata->list_frame[0]->list_subwin == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no regions in frames in imaging data (raw->imgdata->list_frame[0]->list_subwin) argument given");
    }
  if (raw->imgdata->list_frame[0]->list_subwin[0] == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no regions in frames in imaging data (raw->imgdata->list_frame[0]->list_subwin[0]) argument given");
    }
  if (raw->imgdata->list_frame[0]->list_subwin[0]->imgreg == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no images in regions in frames in imaging data (raw->imgdata->list_frame[0]->list_subwin[0]->imgreg) argument given");
    }
  if (raw->imgdata->list_frame[0]->list_subwin[0]->imgreg[0] == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no images in regions in frames in imaging data (raw->imgdata->list_frame[0]->list_subwin[0]->imgreg[0]) argument given");
    }
  if (mat_detector_decode(det, raw->keywords) != CPL_ERROR_NONE)
    {
      return cpl_error_get_code();
    }
  det->data_nregions = raw->imgdet->nbregion;
  det->data_nx = raw->imgdet->list_region[0]->naxis[0];
  det->data_ny = raw->imgdet->list_region[0]->naxis[1];
  return CPL_ERROR_NONE;
}

/**
   @ingroup det
   @brief Checks if two detector specifications describe the same detector.
   @param det    This structure describes a detector (calibration map).
   @param rdet   This structure describes a detector of a raw data file.
   @return The comparison result

   This function checks if two detectors are the same. Only the detector related
   information is used, the size of the raw data itself is ignored. This function
   is used to check if a given calibration map fits to a given raw data file. In this
   case, the calibration map dimension is later mapped to the raw data dimension.

   Possible cpl_error_code set in this function:
   - CPL_ERROR_NULL_INPUT: one or both arguments is NULL
   - CPL_ERROR_INCOMPATIBLE_INPUT: both detectors are incompatible
   - CPL_ERROR_NONE: both detectors are identical
*/
cpl_error_code mat_detector_check(mat_detector *det, mat_detector *rdet)
{
  cpl_error_code    retval = CPL_ERROR_NONE;

  cpl_error_reset();
  if (det == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_detector (det) argument given");
    }
  if (rdet == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_detector (rdet) argument given");
    }
  /* check every property of the detector */
  if (det->type != rdet->type)
    {
      cpl_msg_error(cpl_func, "incompatible detector, chip type does not match %d != %d", det->type, rdet->type);
      retval = CPL_ERROR_INCOMPATIBLE_INPUT;
    }
  if (strcmp(det->id, rdet->id) != 0)
    {
      cpl_msg_error(cpl_func, "incompatible detector, chip id does not match %s != %s", det->id, rdet->id);
      retval = CPL_ERROR_INCOMPATIBLE_INPUT;
    }
  if (strcmp(det->name, rdet->name) != 0)
    {
      cpl_msg_error(cpl_func, "incompatible detector, chip name does not match %s != %s", det->name, rdet->name);
      retval = CPL_ERROR_INCOMPATIBLE_INPUT;
    }
  if (det->read_id != rdet->read_id)
    {
      cpl_msg_error(cpl_func, "incompatible detector, readout mode id does not match %d != %d", det->read_id, rdet->read_id);
      retval = CPL_ERROR_INCOMPATIBLE_INPUT;
    }
  if (strcmp(det->read_name, rdet->read_name) != 0)
    {
      cpl_msg_error(cpl_func, "incompatible detector, readout mode name does not match %s != %s", det->read_name, rdet->read_name);
      retval = CPL_ERROR_INCOMPATIBLE_INPUT;
    }
  if (det->nx != rdet->nx)
    {
      cpl_msg_error(cpl_func, "incompatible detector, nx does not match %d != %d", det->nx, rdet->nx);
      retval = CPL_ERROR_INCOMPATIBLE_INPUT;
    }
  if (det->ny != rdet->ny)
    {
      cpl_msg_error(cpl_func, "incompatible detector, ny does not match %d != %d", det->ny, rdet->ny);
      retval = CPL_ERROR_INCOMPATIBLE_INPUT;
    }
  if (det->channel_nrows != rdet->channel_nrows)
    {
      cpl_msg_error(cpl_func, "incompatible detector, channel_nrows does not match %d != %d", det->channel_nrows, rdet->channel_nrows);
      retval = CPL_ERROR_INCOMPATIBLE_INPUT;
    }
  if (det->channel_ncolumns != rdet->channel_ncolumns)
    {
      cpl_msg_error(cpl_func, "incompatible detector, channel_ncolumns does not match %d != %d", det->channel_ncolumns, rdet->channel_ncolumns);
      retval = CPL_ERROR_INCOMPATIBLE_INPUT;
    }
  if (det->frame_left != rdet->frame_left)
    {
      cpl_msg_error(cpl_func, "incompatible detector, frame_left does not match %d != %d", det->frame_left, rdet->frame_left);
      retval = CPL_ERROR_INCOMPATIBLE_INPUT;
    }
  if (det->frame_right != rdet->frame_right)
    {
      cpl_msg_error(cpl_func, "incompatible detector, frame_right does not match %d != %d", det->frame_right, rdet->frame_right);
      retval = CPL_ERROR_INCOMPATIBLE_INPUT;
    }
  if (det->frame_bottom != rdet->frame_bottom)
    {
      cpl_msg_error(cpl_func, "incompatible detector, frame_bottom does not match %d != %d", det->frame_bottom, rdet->frame_bottom);
      retval = CPL_ERROR_INCOMPATIBLE_INPUT;
    }
  if (det->frame_top != rdet->frame_top)
    {
      cpl_msg_error(cpl_func, "incompatible detector, frame_top does not match %d != %d", det->frame_top, rdet->frame_top);
      retval = CPL_ERROR_INCOMPATIBLE_INPUT;
    }
  return retval;
}

/**
   @ingroup det
   @brief Checks if a detector specification and a raw data file are from the same detector.
   @param det   This structure describes a detector (calibration map).
   @param raw   This structure contains raw data.
   @return The comparison result

   This function extracts the detector specification from raw data and compares
   it with a given detector specification. The total dimension of the raw data
   is checked too.

   Possible cpl_error_code set in this function:
   - CPL_ERROR_NULL_INPUT: one or both arguments is NULL
   - CPL_ERROR_INCOMPATIBLE_INPUT: both detectors are incompatible
   - CPL_ERROR_NONE: both detectors are compatible
*/
cpl_error_code mat_detector_check_raw(mat_detector *det, mat_gendata *raw)
{
  cpl_error_code    retval = CPL_ERROR_NONE;
  mat_detector      rdet;

  cpl_error_reset();
  if (det == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_detector (det) argument given");
    }
  if (raw == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_gendata (raw) argument given");
    }
  if (mat_detector_decode_raw(&rdet, raw) != CPL_ERROR_NONE)
    {
      return cpl_error_get_code();
    }
  retval = mat_detector_check(det, &rdet);
  if (det->data_nregions != rdet.data_nregions)
    {
      cpl_msg_error(cpl_func, "incompatible data, data_nregions does not match %d != %d", det->data_nregions, rdet.data_nregions);
      retval = CPL_ERROR_INCOMPATIBLE_INPUT;
    }
  if (det->data_nx != rdet.data_nx)
    {
      cpl_msg_error(cpl_func, "incompatible data, data_nx does not match %d != %d", det->data_nx, rdet.data_nx);
      retval = CPL_ERROR_INCOMPATIBLE_INPUT;
    }
  if (det->data_ny != rdet.data_ny)
    {
      cpl_msg_error(cpl_func, "incompatible data, data_ny does not match %d != %d", det->data_ny, rdet.data_ny);
      retval = CPL_ERROR_INCOMPATIBLE_INPUT;
    }
  return retval;
}

/**
   @ingroup det
   @brief Checks if specific raw data covers the wole detector.
   @param det   This structure describes a detector (calibration map).
   @param raw   This structure contains raw data.
   @return The comparison result

   This function extracts the detector specification from raw data and compares
   it with a given detector specification. The total dimension of the raw data
   must cover the whole detector (full frame).

   Possible cpl_error_code set in this function:
   - CPL_ERROR_NULL_INPUT: one or both arguments is NULL
   - CPL_ERROR_INCOMPATIBLE_INPUT: both detectors are incompatible
   - CPL_ERROR_NONE: both detectors are compatible
*/
cpl_error_code mat_detector_check_raw_full(mat_detector *det, mat_gendata *raw)
{
  cpl_error_code    retval = CPL_ERROR_NONE;
  mat_detector      rdet;

  cpl_error_reset();
  if (det == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_detector (det) argument given");
    }
  if (raw == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_gendata (raw) argument given");
    }
  if (mat_detector_decode_raw(&rdet, raw) != CPL_ERROR_NONE)
    {
      return cpl_error_get_code();
    }
  retval = mat_detector_check(det, &rdet);
  if (rdet.data_nregions != 1)
    {
      cpl_msg_error(cpl_func, "incompatible data, more than one sub-window present (%d)", rdet.data_nregions);
      retval = CPL_ERROR_INCOMPATIBLE_INPUT;
    }
  if (det->nx != rdet.data_nx)
    {
      cpl_msg_error(cpl_func, "incompatible data, data_nx does not match %d != %d", det->nx, rdet.data_nx);
      retval = CPL_ERROR_INCOMPATIBLE_INPUT;
    }
  if (det->ny != rdet.data_ny)
    {
      cpl_msg_error(cpl_func, "incompatible data, data_ny does not match %d != %d", det->ny, rdet.data_ny);
      retval = CPL_ERROR_INCOMPATIBLE_INPUT;
    }
  return retval;
}

/**
   @ingroup det
   @brief Checks if a single frame is compatible with a specific detector specification.
   @param det  This structure specifies a detector (only the data size is used)
   @param data This structure represents a MATISSE frame (including sub-windows)

   This function checks if the layout of a MATISSE frame is compatible with the
   data layout represented by a mat_detector data structure.

   Possible cpl_error_code set in this function:
   - CPL_ERROR_NULL_INPUT: one or both arguments is NULL
   - CPL_ERROR_INCOMPATIBLE_INPUT: both detectors are incompatible
   - CPL_ERROR_NONE: both detectors are compatible
 */
cpl_error_code mat_detector_check_frame(mat_detector *det, mat_frame *data)
{
  cpl_image        *image = NULL;
  cpl_error_code    retval = CPL_ERROR_NONE;

  cpl_error_reset();
  if (det == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_detector (det) argument given");
    }
  if (data == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_frame (data) argument given");
    }
  if (data->list_subwin == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "mat_frame (data->list_subwin) argument has no sub-windows");
    }
  if (data->list_subwin[0] == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "mat_frame (data->list_subwin[0]) argument has no cpl_image in the first sub-window");
    }
  if (data->list_subwin[0]->imgreg == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "mat_frame (data->list_subwin[0]->imgreg) argument has no cpl_image in the first sub-window");
    }
  if (det->data_nregions != data->nbsubwin)
    {
      cpl_msg_error(cpl_func, "incompatible data, data_nregions does not match %d != %d", det->data_nregions, data->nbsubwin);
      retval = CPL_ERROR_INCOMPATIBLE_INPUT;
    }
  image = data->list_subwin[0]->imgreg[0];
  if (det->data_nx != cpl_image_get_size_x(image))
    {
      cpl_msg_error(cpl_func, "incompatible data, data_nx does not match %d != %d", det->data_nx, (int)cpl_image_get_size_x(image));
      retval = CPL_ERROR_INCOMPATIBLE_INPUT;
    }
  if (det->data_nx != cpl_image_get_size_y(image))
    {
      cpl_msg_error(cpl_func, "incompatible data, data_ny does not match %d != %d", det->data_ny, (int)cpl_image_get_size_y(image));
      retval = CPL_ERROR_INCOMPATIBLE_INPUT;
    }
  return retval;
}

/**
   @ingroup det
   @brief Checks if a single frame is compatible with a specific detector specification and covers the whole detector.
   @param det  This structure specifies a detector (only the detector size is used)
   @param data This structure represents a MATISSE frame (including sub-windows)

   This function checks if the layout of a MATISSE frame is a full frame for the
   detector represented by a mat_detector data structure.

   Possible cpl_error_code set in this function:
   - CPL_ERROR_NULL_INPUT: one or both arguments is NULL
   - CPL_ERROR_INCOMPATIBLE_INPUT: both detectors are incompatible
   - CPL_ERROR_NONE: both detectors are compatible
 */
cpl_error_code mat_detector_check_frame_full(mat_detector *det, mat_frame *data)
{
  cpl_image        *image = NULL;
  cpl_error_code    retval = CPL_ERROR_NONE;

  cpl_error_reset();
  if (det == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_detector (det) argument given");
    }
  if (data == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_frame (data) argument given");
    }
  if (data->list_subwin == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "mat_frame (data->list_subwin) argument has no sub-windows");
    }
  if (data->list_subwin[0] == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "mat_frame (data->list_subwin[0]) argument has no cpl_image in the first sub-window");
    }
  if (data->list_subwin[0]->imgreg == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "mat_frame (data->list_subwin[0]->imgreg) argument has no cpl_image in the first sub-window");
    }
  if (data->nbsubwin != 1)
    {
      cpl_msg_error(cpl_func, "incompatible data, more than one sub-window present (%d)", data->nbsubwin);
      retval = CPL_ERROR_INCOMPATIBLE_INPUT;
    }
  image = data->list_subwin[0]->imgreg[0];
  if (det->nx != cpl_image_get_size_x(image))
    {
      cpl_msg_error(cpl_func, "incompatible data, data_nx does not match %d != %d", det->nx, (int)cpl_image_get_size_x(image));
      retval = CPL_ERROR_INCOMPATIBLE_INPUT;
    }
  if (det->ny != cpl_image_get_size_y(image))
    {
      cpl_msg_error(cpl_func, "incompatible data, data_ny does not match %d != %d", det->ny, (int)cpl_image_get_size_y(image));
      retval = CPL_ERROR_INCOMPATIBLE_INPUT;
    }
  return retval;
}

/**
   @ingroup det
   @brief Checks if a specific pixel is a reference pixel
   @param det  This structure specifies a detector (only the detector and frame sizes are used)
   @param x    The x-coordinate of a pixel (1-based).
   @param y    The y-coordinate of a pixel (1-based).
 */
int mat_detector_is_reference(mat_detector *det, int x, int y)
{
  if (det == NULL)
    {
      cpl_msg_error(cpl_func, "no mat_detector (det) argument given");
      return 0;
    }
  if ((det->frame_left != 0)   && (x <= det->frame_left)) return 1;
  if ((det->frame_right != 0)  && (x >= (det->nx - det->frame_right + 1))) return 1;
  if ((det->frame_bottom != 0) && (y <= det->frame_bottom)) return 1;
  if ((det->frame_top != 0)    && (y >= (det->ny - det->frame_top + 1))) return 1;
  return 0;
}

/**
   @ingroup det
   @brief Checks if a sub-window contains reference pixels.
   @param det  This structure specifies a detector (only the detector and frame sizes are used)
   @param x    The leftmost pixel of the sub-window (1-based, from mat_imagingdetector).
   @param y    The bottommost pixel of the sub-window (1-based, from mat_imagingdetector).
   @param nx   The width of the sub-window.
   @param ny   The height of the sub-window.
   @returns A flag describing that the sub-window contains reference pixels.

   This function checks if a given sub-window contains at least one reference pixel.
 */
int mat_detector_contains_reference(mat_detector *det, int x, int y, int nx, int ny)
{
  if (det == NULL)
    {
      cpl_msg_error(cpl_func, "no mat_detector (det) argument given");
      return 0;
    }
  if ((det->frame_left != 0)   && (x <= det->frame_left)) return 1;
  if ((det->frame_right != 0)  && ((x + nx - 1) >= (det->nx - det->frame_right + 1))) return 1;
  if ((det->frame_bottom != 0) && (y <= det->frame_bottom)) return 1;
  if ((det->frame_top != 0)    && ((y + ny - 1) >= (det->ny - det->frame_top + 1))) return 1;
  return 0;
}

/**
   @ingroup det
   @brief Determines the type of reference contained in the sub-window.
   @param det  This structure specifies a detector (only the detector and frame sizes are used)
   @param x    The leftmost pixel of the sub-window (1-based, from mat_imagingdetector).
   @param y    The bottommost pixel of the sub-window (1-based, from mat_imagingdetector).
   @param nx   The width of the sub-window.
   @param ny   The height of the sub-window.
   @returns A set of reference side flags.

   This method determines which kind of reference pixels are in the sub-window. These flags are important
   for the decision which kind of detector effect compensation methods are available.
 */
int mat_detector_decode_reference(mat_detector *det, int x, int y, int nx, int ny)
{
  int flags = MAT_NO_REF;

  if (det == NULL)
    {
      cpl_msg_error(cpl_func, "no mat_detector (det) argument given");
      return flags;
    }
  if ((det->frame_left != 0) && (x <= det->frame_left))
    { // the sub-window contains reference pixels on the left side
      flags |= MAT_LEFT_REF;
      // check if this sub-window spans a whole detector channel
      if ((x == 1) && (nx >= det->channel_nx))
	{ // the sub-window spans at least one detector channel on the left side
	  flags |= MAT_DC_REF;
	}
    }
  if ((det->frame_right != 0) && ((x + nx - 1) >= (det->nx - det->frame_right + 1)))
    { // the sub-window contains reference pixels on the right side
      flags |= MAT_RIGHT_REF;
      // check if this sub-window spans a whole detector channel
      if (((x + nx - 1) == det->nx) && (nx >= det->channel_nx))
	{ // the sub-window spans at least one detector channel on the right side
	  flags |= MAT_DC_REF;
	}
    }
  if ((det->frame_bottom != 0) && (y <= det->frame_bottom))
    { // the sub-window contains reference pixels on the bottom side
      flags |= MAT_BOTTOM_REF;
    }
  if ((det->frame_top != 0) && ((y + ny - 1) >= (det->ny - det->frame_top + 1)))
    {  // the sub-window contains reference pixels on the top side
      flags |= MAT_TOP_REF;
    }
  return flags;
}

/**
   @ingroup det
   @brief Checks if a sub-window contains normal (non-reference) pixels.
   @param det  This structure specifies a detector (only the detector and frame sizes are used)
   @param x    The leftmost pixel of the sub-window (1-based, from mat_imagingdetector).
   @param y    The bottommost pixel of the sub-window (1-based, from mat_imagingdetector).
   @param nx   The width of the sub-window.
   @param ny   The height of the sub-window.
   @returns A flag describing that the sub-window contains normal pixels.

   This function checks if a given sub-window contains at least one normal pixel.
 */
int mat_detector_contains_nonreference(mat_detector *det, int x, int y, int nx, int ny)
{
  if (det == NULL)
    {
      cpl_msg_error(cpl_func, "no mat_detector (det) argument given");
      return 0;
    }
  if ((det->frame_left != 0)   && ((x + nx - 1) <= det->frame_left)) return 0;
  if ((det->frame_right != 0)  && (x >= (det->nx - det->frame_right + 1))) return 0;
  if ((det->frame_bottom != 0) && ((y + ny - 1) <= det->frame_bottom)) return 0;
  if ((det->frame_top != 0)    && (y >= (det->ny - det->frame_top + 1))) return 0;
  return 1;
}

//static char fname[256];

/**
   @ingroup det
   @brief Generates the name of a static bad pixel map for a specific detector.
   @param det  This structure represents a detector
   @return  The name base name of a bad pixel map

   This function uses the detector type and readout mode to create a name for a bad pixel map.
   The generic file name is:

   "BADPIX.fits"
 */
char *mat_detector_get_bpm_name(mat_detector *det)
{
  if (det == NULL)
    {
      return NULL;
    }
  snprintf(det->fname, 255, "%s.fits", MATISSE_BPM_PROCATG);
  return det->fname;
}

/**
   @ingroup det
   @brief Generates the name of a static flatfield map for a specific detector.
   @param det  This structure represents a detector
   @return  The name base name of a flatfield map

   This function uses the detector type and readout mode to create a name for a flatfield map.
   The generic file name is:

   "FLATFIELD.fits"
 */
char *mat_detector_get_ffm_name(mat_detector *det)
{
  if (det == NULL)
    {
      return NULL;
    }
  snprintf(det->fname, 255, "%s.fits", MATISSE_FFM_PROCATG);
  return det->fname;
}

/**
   @ingroup det
   @brief Generates the name of a static nonlinearity map for a specific detector.
   @param det  This structure represents a detector
   @return  The name base name of a nonlinearity map

   This function uses the detector type and readout mode to create a name for a nonlinearity map.
   The generic file name is:

   "NONLINEARITY.fits"
 */
char *mat_detector_get_nlm_name(mat_detector *det)
{
  if (det == NULL)
    {
      return NULL;
    }
  snprintf(det->fname, 255, "%s.fits", MATISSE_NLM_PROCATG);
  return det->fname;
}

/**
   @ingroup det
   @brief Generates the name of a basic detector monitoring map for a specific detector.
   @param det  This structure represents a detector
   @return  The name base name of a basis detector monitoring map

   This function uses the detector type and readout mode to create a name for a basis detector monitoring map.
   The generic file name is:

   "IM_BASIC.fits"
 */
char *mat_detector_get_imb_name(mat_detector *det)
{
  if (det == NULL)
    {
      return NULL;
    }
  snprintf(det->fname, 255, "%s.fits", MATISSE_IMB_PROCATG);
  return det->fname;
}

/**
   @ingroup det
   @brief Generates the name of an extended detector monitoring map for a specific detector.
   @param det  This structure represents a detector
   @return  The name base name of an extended detector monitoring map

   This function uses the detector type and readout mode to create a name for an extended detector monitoring map.
   The generic file name is:

   "IM_EXTENDED.fits"
 */
char *mat_detector_get_ime_name(mat_detector *det)
{
  if (det == NULL)
    {
      return NULL;
    }
  snprintf(det->fname, 255, "%s.fits", MATISSE_IME_PROCATG);
  return det->fname;
}

/**
   @ingroup det
   @brief Generates the name of an detector remanence map for a specific detector.
   @param det  This structure represents a detector
   @return  The name base name of a detector remanence map

   This function uses the detector type and readout mode to create a name for a detector remanence map.
   The generic file name is:

   "IM_REMANENCE.fits"
 */
char *mat_detector_get_imr_name(mat_detector *det)
{
  if (det == NULL)
    {
      return NULL;
    }
  snprintf(det->fname, 255, "%s.fits", MATISSE_IMR_PROCATG);
  return det->fname;
}

/**
   @ingroup det
   @brief Generates the name of an observation specific flatfield map for a specific detector.
   @param det  This structure represents a detector

   @return  The name base name of an observation specific flatfield map

 */
char *mat_detector_get_offm_name(mat_detector *det)
{
  if (det == NULL)
    {
      return NULL;
    }
  snprintf(det->fname, 255, "%s.fits", MATISSE_OFM_PROCATG);
  return det->fname;
}

cpl_error_code mat_detector_calc_timing(mat_detector *det,
					cpl_propertylist *plist,
					double *mindit,
					double *tpixel,
					double *tline,
					double *tframe)
{
  int               nbSkip = 0;
  int               nbRead = 0;
  int               nbRows, iRow;
  int               rowStart, rowHeight;
  double            timefac;
  double            kwPeriod;
  double            kwDelay;
  double            kwDit;
  double            kwNdit;

  cpl_error_reset();
  if (det == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_detector (det, detector specification) argument given");
    }
  if (plist == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no cpl_propertylist (plist, HDU propertylist) argument given");
    }
  nbRows  = cpl_propertylist_get_int(plist, "ESO DET WIN NROW");
  timefac = (double)cpl_propertylist_get_int(plist, "ESO DET SEQ1 TIMEFAC");
  kwPeriod = 1e6*cpl_propertylist_get_double(plist, "ESO DET SEQ1 PERIOD");
  kwDelay  = 1e6*cpl_propertylist_get_double(plist, "ESO DET TDELAY");
  kwDit    = 1e6*cpl_propertylist_get_double(plist, "ESO DET SEQ1 DIT");
  kwNdit   = (double)cpl_propertylist_get_int(plist, "ESO DET NDIT");
  // 1. Calculate the number of skipped and read pixel lines (see document)
  if (det->type == MAT_AQUARIUS_DET)
    {
      int curRow = 512;
      for (iRow = nbRows; iRow >= 1; iRow--)
	{
	  char kw[64];
	  snprintf(kw, 64, "ESO DET WIN MTRS%d", iRow);
	  rowStart = cpl_propertylist_get_int(plist, kw);
	  snprintf(kw, 64, "ESO DET WIN MTRH%d", iRow);
	  rowHeight = cpl_propertylist_get_int(plist, kw);
	  // The sub-window row is on the high side of the Aquarius detector
	  // -> ignore it
	  if (rowStart > 512) continue;
	  if ((rowStart + rowHeight) > 512)
	    {
	      // A sub-window row spans the center part of the detector 
	      // -> use only the position (enforce symmetry) 
	      nbRead += 512 - rowStart + 1;
	    }
	  else
	    {
	      // A sub-window row is completely on the low side of the Aquarius 
	      // -> use the position and height 
	      nbSkip += curRow - (rowStart + rowHeight - 1);
	      nbRead += rowHeight;
	    }
	  curRow = rowStart - 1;
	}
    }
  else
    {
      int curRow = 1;
      for (iRow = 1; iRow <= nbRows; iRow++)
	{
	  char kw[64];
	  snprintf(kw, 64, "ESO DET WIN MTRS%d", iRow);
	  rowStart = cpl_propertylist_get_int(plist, kw);
	  snprintf(kw, 64, "ESO DET WIN MTRH%d", iRow);
	  rowHeight = cpl_propertylist_get_int(plist, kw);
	  nbSkip += rowStart - curRow;
	  nbRead += rowHeight;
	  curRow += rowStart + rowHeight;
	}
    }
  cpl_msg_debug(cpl_func, "detector = %d, readmode = %s", det->type, det->read_name);
  cpl_msg_debug(cpl_func, "nRows = %d nbSkip = %d nbRead = %d", nbRows, nbSkip, nbRead);
  if (det->type == MAT_HAWAII2RG_DET)
    { // HAWAII-2RG detector (LM-Band)
      /*
	double *mindit
	double *tpixel
	double *tline
	double *tframe
      */
      if ((strcmp(det->read_name, "TEC-CDS-RR-SO") == 0)
	  || (strcmp(det->read_name, "SCI-SLOW-SPEED") == 0)
	  )
	{
	  double tTrigger   = 1.5;
	  double tDelay     = 1000.0;
	  double tFirst     = 4.5 + 10.8*(double)nbSkip + (14.8 + 66.0*timefac)*(double)nbRead;
	  double tIntegrate = 1000.0;
	  double tSecond    = 4.5 + 10.8*(double)nbSkip + (14.8 + 66.0*timefac)*(double)nbRead;
	  double nIntegrate;
	  *mindit = tSecond;
	  *tpixel = timefac;
	  *tline  = 14.8 + 66.0*timefac;
	  nIntegrate = ceil((kwDit - tSecond)/tIntegrate);
	  if (kwPeriod != 0.0)
	    {
	      *tframe = kwPeriod;
	    }
	  else
	    {
	      double nDelay = 0.0;
	      if (kwDelay != 0.0)
		{
		  nDelay = round(kwDelay/tDelay);
		}
	      *tframe = tTrigger + nDelay*tDelay + tFirst + nIntegrate*tIntegrate + tSecond;
	    }
	}
      else if ((strcmp(det->read_name, "TEC-CDS-RR-FO") == 0)
	       || (strcmp(det->read_name, "SCI-FAST-SPEED") == 0)
	       )
	{
	  double tTrigger   = 1.5;
	  double tDelay     = 1000.0;
	  double tFirst     = 4.5 + 0.24*timefac + (2.0 + 0.24*timefac)*(double)nbSkip + (4.0 + 6.84*timefac)*(double)nbRead;
	  double tIntegrate = 1000.0;
	  double tSecond    = 4.5 + 0.24*timefac + (2.0 + 0.24*timefac)*(double)nbSkip + (4.0 + 6.84*timefac)*(double)nbRead;
	  double nIntegrate;
	  *mindit = tSecond;
	  *tpixel = 0.1*timefac;
	  *tline  = 4.0 + 6.84*timefac;
	  nIntegrate = ceil((kwDit - tSecond)/tIntegrate);
	  if (kwPeriod != 0.0)
	    {
	      *tframe = kwPeriod;
	    }
	  else
	    {
	      double nDelay = 0.0;
	      if (kwDelay != 0.0)
		{
		  nDelay = round(kwDelay/tDelay);
		}
	      *tframe = tTrigger + nDelay*tDelay + tFirst + nIntegrate*tIntegrate + tSecond;
	    }
	}
      else if (strcmp(det->read_name, "TEC-CDS-GR-SO") == 0)
	{
	  double tTrigger   = 1.5;
	  double tDelay     = 1000.0;
	  double tReset     = 10.0;
	  double tRead      = 4.5 + 10.8*(double)nbSkip + (10.8 + 66.0*timefac)*(double)nbRead;
	  double tIntegrate = 1000.0;
	  double nIntegrate;
	  *mindit = tRead;
	  *tpixel = timefac;
	  *tline  = 10.8 + 66.0*timefac;
	  nIntegrate = ceil((kwDit - tRead)/tIntegrate);
	  if (kwPeriod != 0.0)
	    {
	      *tframe = kwPeriod;
	    }
	  else
	    {
	      double nDelay = 0.0;
	      if (kwDelay != 0.0)
		{
		  nDelay = round((kwDelay - tReset)/tDelay);
		}
	      *tframe = tTrigger + nDelay*tDelay + tReset + tRead + nIntegrate*tIntegrate + tRead;
	    }
	}
      else if (strcmp(det->read_name, "TEC-CDS-GR-FO") == 0)
	{
	  double tTrigger   = 1.5;
	  double tDelay     = 1000.0;
	  double tReset     = 3076.5;
	  double tRead      = 3.5 + 0.24*timefac + 0.24*timefac*(double)nbSkip + 6.84*timefac*(double)nbRead;
	  double tIntegrate = 1000.0;
	  double nIntegrate;
	  *mindit = tRead;
	  *tpixel = 0.1*timefac;
	  *tline  = 6.84*timefac;
	  nIntegrate = ceil((kwDit - tRead)/tIntegrate);
	  if (kwPeriod != 0.0)
	    {
	      *tframe = kwPeriod;
	    }
	  else
	    {
	      double nDelay = 0.0;
	      if (kwDelay != 0.0)
		{
		  nDelay = round((kwDelay - tReset)/tDelay);
		}
	      *tframe = tTrigger + nDelay*tDelay + tReset + tRead + nIntegrate*tIntegrate + tRead;
	    }
	}
      else if (strcmp(det->read_name, "TEC-SR-GR-SO") == 0)
	{
	  double tTrigger   = 1.5;
	  double tDelay     = 1000.0;
	  double tReset     = 10.0;
	  double tRead      = 4.5 + 10.8*(double)nbSkip + (10.8 + 66.0*timefac)*(double)nbRead;
	  double tIntegrate = 1000.0;
	  double nIntegrate;
	  *mindit = tRead;
	  *tpixel = timefac;
	  *tline  = 10.8 + 66.0*timefac;
	  nIntegrate = ceil((kwDit - tRead)/tIntegrate);
	  if (kwPeriod != 0.0)
	    {
	      *tframe = kwPeriod;
	    }
	  else
	    {
	      double nDelay = 0.0;
	      if (kwDelay != 0.0)
		{
		  nDelay = round((kwDelay - tReset)/tDelay);
		}
	      *tframe = tTrigger + nDelay*tDelay + tReset + nIntegrate*tIntegrate + tRead;
	    }
	}
      else if (strcmp(det->read_name, "TEC-SR-GR-FO") == 0)
	{
	  double tTrigger   = 1.5;
	  double tDelay     = 1000.0;
	  double tReset     = 3076.5;
	  double tRead      = 3.5 + 0.24*timefac + 0.24*timefac*(double)nbSkip + 6.84*timefac*(double)nbRead;
	  double tIntegrate = 1000.0;
	  double nIntegrate;
	  *mindit = tRead;
	  *tpixel = 0.1*timefac;
	  *tline  = 6.84*timefac;
	  nIntegrate = ceil((kwDit - tRead)/tIntegrate);
	  if (kwPeriod != 0.0)
	    {
	      *tframe = kwPeriod;
	    }
	  else
	    {
	      double nDelay = 0.0;
	      if (kwDelay != 0.0)
		{
		  nDelay = round((kwDelay - tReset)/tDelay);
		}
	      *tframe = tTrigger + nDelay*tDelay + tReset + nIntegrate*tIntegrate + tRead;
	    }
	}
      else if (strcmp(det->read_name, "TEC-RAMP-SO") == 0)
	{
	  double tTrigger   = 1.5;
	  double tDelay     = 1000.0;
	  double tReset     = 10.0;
	  double tRead      = 4.5 + 10.8*(double)nbSkip + (10.8 + 66.0*timefac)*(double)nbRead;
	  double tIntegrate = 1000.0;
	  double nIntegrate;
	  *mindit = tRead;
	  *tpixel = timefac;
	  *tline  = 10.8 + 66.0*timefac;
	  nIntegrate = ceil((kwDit - tRead)/tIntegrate);
	  if (kwPeriod != 0.0)
	    {
	      *tframe = kwPeriod;
	    }
	  else
	    {
	      double nDelay = 0.0;
	      if (kwDelay != 0.0)
		{
		  nDelay = round((kwDelay - tReset)/tDelay);
		}
	      *tframe = tTrigger + nDelay*tDelay + tReset + kwNdit*(nIntegrate*tIntegrate + tRead);
	    }
	}
      else if (strcmp(det->read_name, "TEC-RAMP-FO") == 0)
	{
	  double tTrigger   = 1.5;
	  double tDelay     = 1000.0;
	  double tReset     = 3076.5;
	  double tRead      = 3.5 + 0.243*timefac + 0.24*(double)nbSkip + 6.84*timefac*(double)nbRead;
	  double tIntegrate = 1000.0;
	  double nIntegrate;
	  *mindit = tRead;
	  *tpixel = timefac;
	  *tline  = 6.84*timefac;
	  nIntegrate = ceil((kwDit - tRead)/tIntegrate);
	  if (kwPeriod != 0.0)
	    {
	      *tframe = kwPeriod;
	    }
	  else
	    {
	      double nDelay = 0.0;
	      if (kwDelay != 0.0)
		{
		  nDelay = round((kwDelay - tReset)/tDelay);
		}
	      *tframe = tTrigger + nDelay*tDelay + tReset + kwNdit*(nIntegrate*tIntegrate + tRead);
	    }
	}
    }
  else if (det->type == MAT_AQUARIUS_DET)
    { // Aquarius detector (N-Band)
      if ((strcmp(det->read_name, "TEC-SR-SH-LG") == 0)
	  || (strcmp(det->read_name, "SCI-LOW-GAIN") == 0)
	  || (strcmp(det->read_name, "TEC-SR-SH-HG") == 0)
	  || (strcmp(det->read_name, "SCI-HIGH-GAIN") == 0))
	{
	  double tTrigger      = 1.5;
	  double tPad          = 10.08;
	  double tRead         = 25.2 + 3.36*(double)nbSkip + (6.56 + 1.28*timefac)*(double)nbRead;
	  double tIntegrateEnd = 20.16;
	  *mindit = 50.0;
	  *tpixel = 0.1 + 0.04*timefac;
	  *tline  = 6.56 + 1.28*timefac;
	  if (kwPeriod != 0.0)
	    {
	      *tframe = kwPeriod;
	    }
	  else
	    {
	      double nPad = 0.0;
	      if (kwDelay != 0.0)
		{
		  nPad = ceil((kwDelay - tRead)/tPad);
		}
	      *tframe = tTrigger + tRead + nPad*tPad + kwDit + tIntegrateEnd;
	    }
	}
      else if ((strcmp(det->read_name, "TEC-CDS-RR-LG") == 0)
	       || (strcmp(det->read_name, "TEC-CDS-RR-HG") == 0))
	{
	  double tTrigger   = 1.5;
	  double tDelay     = 10.08;
	  double tFirst     = 161.92 + 21.76*timefac + (6.56 + 1.28*timefac)*(double)nbSkip + (6.56 + 1.28*timefac)*(double)nbRead;
	  double tIntegrate = 10.08;
	  double tSecond    = 155.36 + 20.48*timefac + (6.56 + 1.28*timefac)*(double)nbSkip + (6.56 + 1.28*timefac)*(double)nbRead;
	  double nIntegrate;
	  *mindit = tSecond;
	  *tpixel = 0.1 + 0.04*timefac;
	  *tline  = 6.56 + 1.28*timefac;
	  nIntegrate = ceil((kwDit - tSecond)/tIntegrate);
	  if (kwPeriod != 0.0)
	    {
	      *tframe = kwPeriod;
	    }
	  else
	    {
	      double nDelay = 0.0;
	      if (kwDelay != 0.0)
		{
		  nDelay = round(kwDelay/tDelay);
		}
	      *tframe = tTrigger + nDelay*tDelay + tFirst + nIntegrate*tIntegrate + tSecond;
	    }
	}
    }
  else
    { // unknown detector
      *mindit = 1e-6*cpl_propertylist_get_double(plist, "ESO DET SEQ1 MINDIT");
      *tpixel = 1.0;
      *tline = 32.0;
      if (kwPeriod == 0.0)
	{
	  *tframe = *mindit + kwDit;
	}
      else
	{
	  *tframe = kwPeriod;
	}
    }
  *mindit *= 1e-6;
  *tpixel *= 1e-6;
  *tline  *= 1e-6;
  *tframe *= 1e-6;
  return CPL_ERROR_NONE;
}
