/* $Id: mat_est_corr_lib.c,v0.5 2014-06-15 12:56:21 pberio Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: pberio $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_est_corr_lib.c $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#include <string.h>
#endif

/*-----------------------------------------------------------------------------
  Includes
  ----------------------------------------------------------------------------*/

#include "mat_est_corr_lib.h"
#include "mat_error.h"

/*-----------------------------------------------------------------------------
  Define
  ----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
  Functions prototypes
  ----------------------------------------------------------------------------*/
/*----------------------------------------------------------------------------*/
/**
   @brief Estimate the correlated flux for each baseline
   @param frameset           contains the frames
   @param parlist            contains the recipes parameters
   @param recname            Name of the recipes calling this function
   @return 0 if no error 
*/
/*----------------------------------------------------------------------------*/
int mat_est_corr_lib(cpl_frameset *frameset, 
		     cpl_parameterlist *parlist,
		     const char *recname) {
  cpl_frame *cur_frame=NULL;
  cpl_frame *cur_frame1=NULL;
  cpl_parameter *useOpdMod=NULL;
  cpl_parameter *coherentIntegTime=NULL;
  char *pszFileName=NULL;
  char *pszFileTag=NULL;
  char *pszFileName1=NULL;
  char *pszFileTag1=NULL;
  char *detName=NULL;
  char *sliderPhot=NULL;
  char mode[3]=" ";
  char modeFirst[3]=" ";
  char *output=NULL;
  char *bcd1Status=NULL;
  char *bcd2Status=NULL;
  char bcd1StatusFirst[100]="";
  char bcd2StatusFirst[100]="";
  cpl_propertylist *plist=NULL;
  cpl_propertylist *plist1=NULL;
  cpl_frameset *usedframes=NULL;
  cpl_frame *sel_frame=NULL;
  int cpt=0;
  int jstep=0;
  int j=0;
  int k=0;
  //  int nbExtent=0;
  int i=0;
  int nbImgInterfCycle=0;
  int *filledInterfCycle=NULL;
  int nbregion=0;
  int nbCorr=0;
  cpl_vector *dispCoef=NULL;
  mat_gendata *gendata=NULL;
  cpl_image **imgInterfCycle=NULL;
  cpl_image **avgSky=NULL;
  cpl_image **avgImg=NULL;
  cpl_vector *localOpd=NULL;
  cpl_matrix *localOpdBcd=NULL;
  mat_corrflux *corrFlux=NULL;
  cpl_frameset_iterator *it=NULL;
  cpl_frameset_iterator *it1=NULL;
  cpl_errorstate prestate = cpl_errorstate_get();
  double shiftMean=0.;
  char *specResData;
  int indexFile=1;
  int nbFrameSky=0;
  int nbdetector=1;
  /* int resolution=1; */
  /* int flagDone=0; */
  cpl_parameter *spectralBinning=NULL;
  int nbBin=0;
  
  // Check input parameters
  mat_assert_value((frameset != NULL),CPL_ERROR_NULL_INPUT, 
		   CPL_ERROR_NULL_INPUT, "no frameset argument given");
  mat_assert_value((parlist != NULL),CPL_ERROR_NULL_INPUT, 
		   CPL_ERROR_NULL_INPUT, "no parlist argument given");

  // 0. Retrieve parameter
  //cpl_parameterlist_dump(parlist,NULL);
  useOpdMod=cpl_parameterlist_find(parlist,"matisse.mat_est_corr.useOpdMod");
  coherentIntegTime=cpl_parameterlist_find(parlist,"matisse.mat_est_corr.coherentIntegTime");
  spectralBinning=cpl_parameterlist_find(parlist,"matisse.mat_raw_estimates.spectralBinning");
  if (spectralBinning != NULL)
    {
      nbBin=cpl_parameter_get_int(spectralBinning);
      if (nbBin % 2 == 0)
	{
	  nbBin++;
	}
    }
  else
    {
      nbBin=1;
    }
  cpl_msg_info(cpl_func,"Spectral Binning in pixel %d",nbBin);

  
  // 1. Check the data type of the input frames 
  cpt=0;
  nbCorr=0;
  prestate=cpl_errorstate_get();
  it= cpl_frameset_iterator_new(frameset);
  do {
    cur_frame = cpl_frameset_iterator_get(it);
    if (cur_frame != NULL) {
      pszFileName = (char*)cpl_frame_get_filename( cur_frame );
      pszFileTag = (char *)cpl_frame_get_tag( cur_frame );
      plist=cpl_propertylist_load(pszFileName,0);
      if ( strcmp(pszFileTag,"TARGET_CAL") == 0 || strcmp(pszFileTag,"CALIB_CAL") == 0 ) {
	nbCorr++;
	detName=
	  (char *)cpl_propertylist_get_string(plist,"ESO DET CHIP NAME");
	if (detName == NULL) {
	  cpl_msg_error(cpl_func,"no ESO DET CHIP NAME keyword in frame");
	  return -1;
	}
	if (cpl_propertylist_has(plist,"ESO INS BCD1 ID") && cpl_propertylist_has(plist,"ESO INS BCD2 ID")) {
	  bcd1Status = (char *)cpl_propertylist_get_string(plist,"ESO INS BCD1 ID");
	  bcd2Status = (char *)cpl_propertylist_get_string(plist,"ESO INS BCD2 ID");
	} else {
	  cpl_msg_error(cpl_func,"no BCD keyword found in frame %s",pszFileName);
	  return -1;
	}
	if (!strcmp(detName, "AQUARIUS")) {
	  sliderPhot=(char *)cpl_propertylist_get_string(plist,"ESO INS PIN NAME");	
	} else {
	  sliderPhot=(char *)cpl_propertylist_get_string(plist,"ESO INS PIL NAME");	
	}
	if (!strcmp(sliderPhot, "PHOTO")) {
	  sprintf(mode,"s");
	} else {
	  sprintf(mode,"h");
	}
	if (cpt == 0 ) {
	  strncpy(bcd1StatusFirst,bcd1Status,100-1); bcd1StatusFirst[100-1] = '\0';
	  strncpy(bcd2StatusFirst,bcd2Status,100-1); bcd2StatusFirst[100-1] = '\0';
	  strncpy(modeFirst,mode,3); 
	} 
	cpt++;
      }
      // Get next frame from SOF
      cpl_propertylist_delete(plist);
    }
  } while (cpl_frameset_iterator_advance(it, 1) != CPL_ERROR_ACCESS_OUT_OF_RANGE);
  cpl_frameset_iterator_delete(it);
  if (!cpl_errorstate_is_equal(prestate)) {
    cpl_errorstate_set(prestate);
  }

  if (strstr(modeFirst,"s") != NULL ) {
    if (cpl_parameter_get_double(coherentIntegTime)>0. ) {
      cpl_msg_warning(cpl_func,"In SiPhot mode, No Coherent Integration");
      cpl_msg_warning(cpl_func,"We fix coherentIntegTime=0.");
      cpl_parameter_set_double(coherentIntegTime,0.);
    }
  } else {
    if (cpl_parameter_get_double(coherentIntegTime)>0. ) {
      if (cpl_parameter_get_bool(useOpdMod)) {
	cpl_msg_warning(cpl_func,"If coherentInetgTime>0, the parameter useOpdMod is disabled");
	cpl_msg_warning(cpl_func,"We fix useOpdMod=False");
	cpl_parameter_set_bool(useOpdMod,CPL_FALSE);
      }
    }
  }
	

  // 3.Processing
  char *protech=NULL;
  cpt=0;
  prestate=cpl_errorstate_get();
  it= cpl_frameset_iterator_new(frameset);
  do {
    cur_frame = cpl_frameset_iterator_get(it);
    if (cur_frame != NULL) {
      pszFileName = (char*)cpl_frame_get_filename( cur_frame );
      pszFileTag = (char *)cpl_frame_get_tag( cur_frame );
      if ((strcmp(pszFileTag,"TARGET_CAL") == 0 || strcmp(pszFileTag,"CALIB_CAL") == 0) && 
	  mat_check_interf(cur_frame) )
	{
	  cpl_msg_info(cpl_func,"Computing an average sky ...");
	  gendata=mat_gendata_load(cur_frame,CPL_TYPE_FLOAT);
	  if (gendata == NULL) {
	    cpl_msg_error(cpl_func,"no gendata loaded");
	    return -1;
	  }
	  protech=(char *)cpl_propertylist_get_string(gendata->keywords,"ESO PRO TECH");
	  if (protech == NULL)
	    {
	      cpl_msg_error(cpl_func,"No PRO TECH keyword");
	      return -1;
	    }
	  cpl_msg_info(cpl_func,"PRO.TECH=%s",protech);
	  j=0;
	  for(i=0;i<gendata->imgdata->nbframe;i++)
	    {
	      j+=gendata->imgdata->list_frame[i]->stepphase;
	    }
	  if (j==gendata->imgdata->nbframe)
	    {
	      cpl_msg_warning(cpl_func,"No OPD modulation in these data");
	      cpl_parameter_set_bool(useOpdMod,0);
	    }
	    
	  detName=(char *) cpl_propertylist_get_string(gendata->keywords,"ESO DET CHIP NAME");
	  if (detName == NULL)
	    {
	      cpl_msg_error(cpl_func,"no ESO DET CHIP NAME keyword in frame");
	      return -1;
	    }
	  if (!strcmp(detName, "AQUARIUS"))
	    {
	      nbdetector=2;
	      specResData=(char *)cpl_propertylist_get_string(gendata->keywords,"ESO INS DIN ID");
	      if (!strcmp(specResData,"LOW"))
		{
		  shiftMean=-7.;
		}
	      if (!strcmp(specResData,"HIGH"))
		{
		  shiftMean=-5.;
		}
	    }
	  else
	    {
	      nbdetector=1;
	      specResData=(char *)cpl_propertylist_get_string(gendata->keywords,"ESO INS DIL ID");
	      if (!strcmp(specResData,"LOW"))
		{
		  shiftMean=50.5;
		}
	      if (!strcmp(specResData,"MED"))
		{
		  shiftMean=40.5;
		}
	      if (!strcmp(specResData,"HIGH"))
		{
		  shiftMean=36.5;
		}
	      if (!strcmp(specResData,"HIGH+"))
		{
		  shiftMean=36.5;
		}
	    }
	  // Compute an average sky
	  if (strstr(modeFirst,"s") != NULL)
	    {
	      nbregion=gendata->imgdet->nbregion/5;
	    }
	  else
	    {
	      nbregion=gendata->imgdet->nbregion;
	    }


	  // Remove the avgSky 
	  avgSky=cpl_calloc(nbregion,sizeof(cpl_image *));
	  mat_compute_avg_sky(gendata,avgSky);
	  mat_remove_avgsky(gendata,avgSky,modeFirst);
	
	  // Compute the avgImg 
	  avgImg=cpl_calloc(nbregion,sizeof(cpl_image *));
	  if (mat_compute_avg_img(gendata,avgImg) != -1)
	    {
	    
	      // Start computing OBJ_CORR_FLUX
	      cpt=-1;
	      if ( cpl_parameter_get_bool(useOpdMod) ) {
		if (strstr(modeFirst,"s") != NULL) {
		  nbImgInterfCycle=(gendata->imgdet->nbregion/5)*gendata->imgdata->maxstep;
		} else {
		  nbImgInterfCycle=(gendata->imgdet->nbregion)*gendata->imgdata->maxstep;
		}
		cpl_msg_info(cpl_func,"Number of image by Interf cycle= %d",nbImgInterfCycle);
		imgInterfCycle=cpl_calloc(nbImgInterfCycle,sizeof(cpl_image *));
		localOpd=cpl_vector_new(4*gendata->imgdata->maxstep);
		filledInterfCycle=cpl_calloc(gendata->imgdata->maxstep,sizeof(int));
	      }
	      // load dispersion coefficients
	      dispCoef=cpl_vector_new(N_DEG_DISPERSION_LAW+1);
	
	      cpl_vector_set(dispCoef,0,cpl_propertylist_get_double(gendata->keywords,"PRO DISP COEF0"));
	      cpl_vector_set(dispCoef,1,cpl_propertylist_get_double(gendata->keywords,"PRO DISP COEF1"));
	      cpl_vector_set(dispCoef,2,cpl_propertylist_get_double(gendata->keywords,"PRO DISP COEF2"));
	      cpl_vector_set(dispCoef,3,cpl_propertylist_get_double(gendata->keywords,"PRO DISP COEF3"));
	      cpl_vector_set(dispCoef,4,cpl_propertylist_get_double(gendata->keywords,"PRO DISP COEF4"));
		       
	      // Initialize mat_corrflux structure
	      corrFlux=mat_init_corrflux(gendata,modeFirst);
	      if (corrFlux == NULL) {
		cpl_msg_error(cpl_func,"Error in mat_corrflux initialisation");
		cpl_vector_delete(dispCoef);
		if ( cpl_parameter_get_bool(useOpdMod) ) {
		  cpl_free(imgInterfCycle);
		  cpl_vector_delete(localOpd);
		  cpl_free(filledInterfCycle);
		}
		return -1;
	      }
	      // Depending on the BCD position, we switch the localopd vector
	      localOpdBcd=mat_switch_localopd_wrt_bcd(gendata);

	      //Remove average image in order to remove the artefacts in the DSP
	      if (avgImg != NULL  && nbdetector == 2) {
		mat_remove_avgimg(gendata,avgImg,modeFirst);
	      }
	  
	      //Determine the origin phase on the first Target image of the first interferomertic region
	      /* j=0; */
	      /* flagDone=0; */
	      /* while(j<gendata->imgdata->nbframe && !flagDone) { */
	      /*   if ( !strcmp(gendata->imgdata->list_frame[j]->tartype,"T") ) { */
	      /*     if (strstr(modeFirst,"s") != NULL) { */
	      /*       // SiPhot */
	      /*       shiftMean=mat_get_origin_phase(gendata->imgdata->list_frame[j]->list_subwin[2]->imgreg[0], */
	      /* 				     dispCoef,gendata->imgdet->list_region[2]->corner[1], */
	      /* 				     nbdetector,resolution); */
	      /*       /\* shiftMean=50.; *\/ */
	      /*     } else { */
	      /*       // HiSens */
	      /*       shiftMean=mat_get_origin_phase(gendata->imgdata->list_frame[j]->list_subwin[0]->imgreg[0], */
	      /* 				     dispCoef,gendata->imgdet->list_region[0]->corner[1], */
	      /* 				     nbdetector,resolution); */
	      /*     } */
	      /*     flagDone=1; */
	      /*   } */
	      /*   j++; */
	      /* } */
	      /* if (!flagDone) { */
	      /*   cpl_msg_error(cpl_func,"No Target frame found"); */
	      /* } else  { */
	      cpl_msg_info(cpl_func,"Origin phase computed: %f",shiftMean);
	      /* } */
	      // Processing of the TARGET frames
	      // Loop on the frames
	      jstep=0;
	      for(j=0;j<gendata->imgdata->nbframe;j++) {
		if ( !strcmp(gendata->imgdata->list_frame[j]->tartype,"T") ) {
		  if ( !cpl_parameter_get_bool(useOpdMod) ) {
		    // if useOpdMod=FALSE, we demodulate each fringe peaks
		    // of each frame from the applied local opd
		    mat_demodulate_fringes(gendata, corrFlux, modeFirst, j, dispCoef, localOpdBcd);
		  } else {
		    // if useOpdMod=TRUE, Process the data following the Opd Mod algorithm
		    // based on the least square method
		    mat_lsq_opdmod(gendata,corrFlux, modeFirst, j, dispCoef, localOpdBcd,
				   filledInterfCycle, imgInterfCycle, localOpd, &cpt, nbImgInterfCycle, &jstep);
		  }
		}
	      }

	      // In the Case of the OpdMod Algorithm we process the last OPD cycle
	      // This step is not necesary for the other algorithms
	      if ( cpl_parameter_get_bool(useOpdMod)) {
		// processing of the last OPD cycle
		printf("Processing of the last OPD cycle ...\n");
		mat_lsq_opdmod_last_frame(gendata,corrFlux, modeFirst, dispCoef,
					  filledInterfCycle, imgInterfCycle,
					  localOpd, nbImgInterfCycle,&jstep);
	      }
	      cpl_propertylist_append_int(corrFlux->keywords,"PRO FRAME TARGET",corrFlux->nbframe);
	      int nbFrameTarget=corrFlux->nbframe;

	      if (strstr(modeFirst,"s") == NULL) {
		// Processing of the SKY frames in case of HIGHSENS AQUARIUS
		cpt=-1;
		for(j=0;j<gendata->imgdata->nbframe;j++) {
		  if ( !strcmp(gendata->imgdata->list_frame[j]->tartype,"S") ) {
		    if ( !cpl_parameter_get_bool(useOpdMod) ) {
		      // if useOpdMod=FALSE, we demodulate each fringe peaks
		      // of each frame from the applied local opd
		      mat_demodulate_fringes(gendata, corrFlux, modeFirst, j, dispCoef, localOpdBcd);
		    } else {
		      // if useOpdMod=TRUE, Process the data following the Opd Mod algorithm
		      // based on the least square method
		      mat_lsq_opdmod(gendata,corrFlux, modeFirst, j, dispCoef, localOpdBcd,
				     filledInterfCycle, imgInterfCycle, localOpd, &cpt, nbImgInterfCycle, &jstep);
		    }
		  }
		}

		// In the Case of the OpdMod Algorithm we process the last OPD cycle
		// This step is not necesary for the other algorithms
		if ( cpl_parameter_get_bool(useOpdMod)) {
		  // processing of the last OPD cycle
		  printf("Processing of the last OPD cycle ...\n");
		  mat_lsq_opdmod_last_frame(gendata,corrFlux, modeFirst, dispCoef,
					    filledInterfCycle, imgInterfCycle,
					    localOpd, nbImgInterfCycle,&jstep);
		}
		nbFrameSky=corrFlux->nbframe-nbFrameTarget;
	      }
	
	      if ( cpl_parameter_get_bool(useOpdMod)) {
		if (localOpd != NULL) cpl_vector_delete(localOpd);
		if (imgInterfCycle != NULL) cpl_free(imgInterfCycle);
		if (filledInterfCycle != NULL)
		  cpl_free(filledInterfCycle);
	      }

	      // Coherent integration
	      if (cpl_parameter_get_double(coherentIntegTime)>0.)
		{
		  // Duplicate corrFlux->list_corr
		  mat_imgcorr **listCorrDup=cpl_malloc(corrFlux->nbframe*sizeof(mat_imgcorr *));
		  for(int iFrame=0;iFrame<corrFlux->nbframe;iFrame++)
		    {
		      listCorrDup[iFrame]=mat_imgcorr_duplicate(corrFlux->list_corr[iFrame]);
		    }
		  // delete corrFlux->list_corr
		  for(int iFrame=0; iFrame<corrFlux->nbframe; iFrame++)
		    {
		      if (corrFlux->list_corr[iFrame] != NULL)
			{
			  if (corrFlux->list_corr[iFrame]->imgreg != NULL)
			    {
			      if (corrFlux->list_corr[iFrame]->imgreg[0] != NULL)
				{
				  cpl_image_delete(corrFlux->list_corr[iFrame]->imgreg[0]);
				  corrFlux->list_corr[iFrame]->imgreg[0] = NULL;
				}
			      if (corrFlux->list_corr[iFrame]->imgreg[1] != NULL)
				{
				  cpl_image_delete(corrFlux->list_corr[iFrame]->imgreg[1]);
				  corrFlux->list_corr[iFrame]->imgreg[1] = NULL;
				}
			      cpl_free(corrFlux->list_corr[iFrame]->imgreg);
			      corrFlux->list_corr[iFrame]->imgreg = NULL;
			    }
			  cpl_free(corrFlux->list_corr[iFrame]);
			  corrFlux->list_corr[iFrame] = NULL;
			}
		    }
		  if (corrFlux->list_corr != NULL)
		    {
		      cpl_free(corrFlux->list_corr);
		      corrFlux->list_corr = NULL;
		    }
		  // Determine the new number of frames
		  cpt=0;
		  int flagNewCycle=1;
		  double t0=0.;
		  int nbFrameNew=0;

		  // TARGET Frames
		  for(int iFrame=0; iFrame<nbFrameTarget;iFrame++)
		    {
		      if (flagNewCycle)
			{
			  t0=listCorrDup[cpt]->time;
			  flagNewCycle=0;
			}
		      else
			{
			  if ((listCorrDup[cpt]->time-t0)>cpl_parameter_get_double(coherentIntegTime)/86400.)
			    {
			      flagNewCycle=1;
			      nbFrameNew++;
			    }
			}
		      cpt++;
		    }
		  cpl_propertylist_set_int(corrFlux->keywords,"PRO FRAME TARGET",nbFrameNew);
	  
		  // SKY Frames
		  flagNewCycle=1;
		  for(int iFrame=nbFrameTarget; iFrame<nbFrameTarget+nbFrameSky;iFrame++)
		    {
		      if (flagNewCycle)
			{
			  t0=listCorrDup[cpt]->time;
			  flagNewCycle=0;
			}
		      else
			{
			  if ((listCorrDup[cpt]->time-t0)>cpl_parameter_get_double(coherentIntegTime)/86400.)
			    {
			      flagNewCycle=1;
			      nbFrameNew++;
			    }
			}
		      cpt++;
		    }

		  // Fill the new corrFlux->list_corr
		  corrFlux->list_corr=cpl_malloc(nbFrameNew*corrFlux->imgdet->nbregion*sizeof(mat_imgcorr *));
		  for(int iFrame=0;iFrame<nbFrameNew*corrFlux->imgdet->nbregion;iFrame++)
		    {
		      corrFlux->list_corr[iFrame]=cpl_malloc(sizeof(mat_imgcorr));
		      corrFlux->list_corr[iFrame]->nbimgreg=2;
		      corrFlux->list_corr[iFrame]->imgreg=cpl_calloc(corrFlux->list_corr[iFrame]->nbimgreg,sizeof(cpl_image *));
		    }
		  cpl_image *imgReal=NULL;
		  cpl_image *imgImag=NULL;
		  // TARGET Frames
		  double tstart=0.;
		  int nbImgperCycle=0;
		  int cptNew=0;
		  for(int iReg=0;iReg<corrFlux->imgdet->nbregion;iReg++)
		    {
		      nbImgperCycle=0;
		      flagNewCycle=1;
		      cptNew=0;
		      for(int iFrame=iReg; iFrame<nbFrameTarget;iFrame+=corrFlux->imgdet->nbregion)
			{
			  if (flagNewCycle)
			    {
			      if (imgReal!=NULL)
				{
				  cpl_image_delete(imgReal);
				}
			      if (imgImag!=NULL)
				{
				  cpl_image_delete(imgImag);
				}
			      t0=listCorrDup[iFrame]->time;
			      flagNewCycle=0;
			      imgReal=cpl_image_duplicate(listCorrDup[iFrame]->imgreg[0]);
			      imgImag=cpl_image_duplicate(listCorrDup[iFrame]->imgreg[1]);
			      tstart=listCorrDup[iFrame]->time;
			    }
			  if ( (listCorrDup[iFrame]->time-t0) <= cpl_parameter_get_double(coherentIntegTime)/86400.)
			    {
			      if ( nbImgperCycle>0 )
				{
				  cpl_image_add(imgReal,listCorrDup[iFrame]->imgreg[0]);
				  cpl_image_add(imgImag,listCorrDup[iFrame]->imgreg[1]);
				}
			      nbImgperCycle++;
			    }
	      
			  if (flagNewCycle==0)
			    {
			      if ((listCorrDup[iFrame]->time-t0)>cpl_parameter_get_double(coherentIntegTime)/86400.)
				{
				  cpl_image_divide_scalar(imgReal,nbImgperCycle);
				  cpl_image_divide_scalar(imgImag,nbImgperCycle);
				  corrFlux->list_corr[cptNew*corrFlux->imgdet->nbregion+iReg]->numregion=listCorrDup[iFrame]->numregion;
				  corrFlux->list_corr[cptNew*corrFlux->imgdet->nbregion+iReg]->numdetector=listCorrDup[iFrame]->numdetector;
				  corrFlux->list_corr[cptNew*corrFlux->imgdet->nbregion+iReg]->time=tstart;
				  corrFlux->list_corr[cptNew*corrFlux->imgdet->nbregion+iReg]->exptime=listCorrDup[iFrame]->exptime;
				  corrFlux->list_corr[cptNew*corrFlux->imgdet->nbregion+iReg]->imgreg[0]=cpl_image_duplicate(imgReal);
				  corrFlux->list_corr[cptNew*corrFlux->imgdet->nbregion+iReg]->imgreg[1]=cpl_image_duplicate(imgImag);
				  nbImgperCycle=0;
				  flagNewCycle=1;
				  cptNew++;
				}
			    }
			}
		    }
		  // SKY Frames
		  tstart=0.;
		  for(int iReg=0;iReg<corrFlux->imgdet->nbregion;iReg++)
		    {
		      nbImgperCycle=0;
		      flagNewCycle=1;
		      for(int iFrame=iReg+nbFrameTarget; iFrame<nbFrameTarget+nbFrameSky;iFrame+=corrFlux->imgdet->nbregion)
			{
			  if (flagNewCycle)
			    {
			      if (imgReal!=NULL)
				{
				  cpl_image_delete(imgReal);
				}
			      if (imgImag!=NULL)
				{
				  cpl_image_delete(imgImag);
				}
			      t0=listCorrDup[iFrame]->time;
			      flagNewCycle=0;
			      imgReal=cpl_image_duplicate(listCorrDup[iFrame]->imgreg[0]);
			      imgImag=cpl_image_duplicate(listCorrDup[iFrame]->imgreg[1]);
			      tstart=listCorrDup[iFrame]->time;
			    }
			  if ( (listCorrDup[iFrame]->time-t0) <= cpl_parameter_get_double(coherentIntegTime)/86400.)
			    {
			      if (nbImgperCycle>0)
				{
				  cpl_image_add(imgReal,listCorrDup[iFrame]->imgreg[0]);
				  cpl_image_add(imgImag,listCorrDup[iFrame]->imgreg[1]);
				}
			      nbImgperCycle++;
			    }
	      
			  if (flagNewCycle==0)
			    {
			      if ((listCorrDup[iFrame]->time-t0)>cpl_parameter_get_double(coherentIntegTime)/86400.)
				{
				  cpl_image_divide_scalar(imgReal,nbImgperCycle);
				  cpl_image_divide_scalar(imgImag,nbImgperCycle);
				  corrFlux->list_corr[cptNew*corrFlux->imgdet->nbregion+iReg]->numregion=listCorrDup[iFrame]->numregion;
				  corrFlux->list_corr[cptNew*corrFlux->imgdet->nbregion+iReg]->numdetector=listCorrDup[iFrame]->numdetector;
				  corrFlux->list_corr[cptNew*corrFlux->imgdet->nbregion+iReg]->time=tstart;
				  corrFlux->list_corr[cptNew*corrFlux->imgdet->nbregion+iReg]->exptime=listCorrDup[iFrame]->exptime;
				  corrFlux->list_corr[cptNew*corrFlux->imgdet->nbregion+iReg]->imgreg[0]=cpl_image_duplicate(imgReal);
				  corrFlux->list_corr[cptNew*corrFlux->imgdet->nbregion+iReg]->imgreg[1]=cpl_image_duplicate(imgImag);
				  nbImgperCycle=0;
				  flagNewCycle=1;
				  cptNew++;
				}
			    }
			}
		    }

		  if (imgReal!=NULL)
		    {
		      cpl_image_delete(imgReal);
		    }
		  if (imgImag!=NULL)
		    {
		      cpl_image_delete(imgImag);
		    }
		  // delete listCorrDup
		  for(int iFrame=0; iFrame<corrFlux->nbframe; iFrame++)
		    {
		      if (listCorrDup[iFrame] != NULL)
			{
			  if (listCorrDup[iFrame]->imgreg != NULL)
			    {
			      if (listCorrDup[iFrame]->imgreg[0] != NULL)
				{
				  cpl_image_delete(listCorrDup[iFrame]->imgreg[0]);
				  listCorrDup[iFrame]->imgreg[0] = NULL;
				}
			      if (listCorrDup[iFrame]->imgreg[1] != NULL)
				{
				  cpl_image_delete(listCorrDup[iFrame]->imgreg[1]);
				  listCorrDup[iFrame]->imgreg[1] = NULL;
				}
			      cpl_free(listCorrDup[iFrame]->imgreg);
			      listCorrDup[iFrame]->imgreg = NULL;
			    }
			  cpl_free(listCorrDup[iFrame]);
			  listCorrDup[iFrame] = NULL;
			}
		    }
		  if (listCorrDup != NULL)
		    {
		      cpl_free(listCorrDup);
		      listCorrDup = NULL;
		    }

		  corrFlux->nbframe=nbFrameNew*corrFlux->imgdet->nbregion;
		}

	      // Remove origin phase
	      mat_remove_origin_phase(corrFlux,shiftMean);


	      // Apply Spectral Binning
	      if (nbBin > 1)
		{
		  int iFrame, iFreq, iReg, nbFreq, nbWlen, iWlen, iPix;
		  cpl_vector *vec2smoothR=NULL;
		  cpl_vector *vec2smoothI=NULL;
		  double valR=0.;
		  double valI=0.;
		  int pis=0;

		  cpt=0;
		  for(iReg=0 ; iReg < corrFlux->imgdet->nbregion ; iReg++)
		    {
		      nbFreq=corrFlux->imgdet->list_region[iReg]->naxis[0];
		      nbWlen=corrFlux->imgdet->list_region[iReg]->naxis[1];
		      for(iFrame=0;iFrame<corrFlux->nbframe/corrFlux->imgdet->nbregion;iFrame++)
			{
			  for(iFreq=0;iFreq<nbFreq;iFreq++)
			    {
			      vec2smoothR=cpl_vector_new(nbWlen);
			      vec2smoothI=cpl_vector_new(nbWlen);
			      for(iWlen=0;iWlen<nbWlen;iWlen++)
				{
				  cpl_vector_set(vec2smoothR,iWlen,cpl_image_get(corrFlux->list_corr[cpt]->imgreg[0],iFreq+1,iWlen+1,&pis));
				  cpl_vector_set(vec2smoothI,iWlen,cpl_image_get(corrFlux->list_corr[cpt]->imgreg[1],iFreq+1,iWlen+1,&pis));
				}
			      for(iWlen=0;iWlen<nbWlen;iWlen++)
				{
				  if (iWlen < (nbBin-1)/2)
				    {
				      valR=0.;
				      valI=0.;
				      for(iPix=0;iPix<=2*iWlen;iPix++)
					{
					  valR+=cpl_vector_get(vec2smoothR,iPix);
					  valI+=cpl_vector_get(vec2smoothI,iPix);
					}
				      valR/=(2*iWlen+1);
				      valI/=(2*iWlen+1);
				      cpl_image_set(corrFlux->list_corr[cpt]->imgreg[0],iFreq+1,iWlen+1,valR);
				      cpl_image_set(corrFlux->list_corr[cpt]->imgreg[1],iFreq+1,iWlen+1,valI);
				    }
				  else if (iWlen >= nbWlen-(nbBin-1)/2)
				    {
				      valR=0.;
				      valI=0.;
				      for(iPix=iWlen-(nbWlen-1-iWlen);iPix<=iWlen+(nbWlen-1-iWlen);iPix++)
					{
					  valR+=cpl_vector_get(vec2smoothR,iPix);
					  valI+=cpl_vector_get(vec2smoothI,iPix);
					}
				      valR/=(2*(nbWlen-1-iWlen)+1);
				      valI/=(2*(nbWlen-1-iWlen)+1);
				      cpl_image_set(corrFlux->list_corr[cpt]->imgreg[0],iFreq+1,iWlen+1,valR);
				      cpl_image_set(corrFlux->list_corr[cpt]->imgreg[1],iFreq+1,iWlen+1,valI);
				    }
				  else
				    {
				      valR=0.;
				      valI=0.;
				      for(iPix=iWlen-(nbBin-1)/2;iPix<=iWlen+(nbBin-1)/2;iPix++)
					{
					  valR+=cpl_vector_get(vec2smoothR,iPix);
					  valI+=cpl_vector_get(vec2smoothI,iPix);
					}
				      valR/=nbBin;
				      valI/=nbBin;
				      cpl_image_set(corrFlux->list_corr[cpt]->imgreg[0],iFreq+1,iWlen+1,valR);
				      cpl_image_set(corrFlux->list_corr[cpt]->imgreg[1],iFreq+1,iWlen+1,valI);
				    }
				}
			      cpl_vector_delete(vec2smoothR);
			      cpl_vector_delete(vec2smoothI);
			    }
			  cpt++;
			}
		    }
		}	
	
	      // Fill usedframes depending on the way mat_ext_beams is called (atomic or master recipes)
	      bcd1Status = (char *)cpl_propertylist_get_string(corrFlux->keywords,"ESO INS BCD1 ID");
	      bcd2Status = (char *)cpl_propertylist_get_string(corrFlux->keywords,"ESO INS BCD2 ID");
	      if (strcmp(bcd1Status, "OUT") == 0 && strcmp(bcd2Status, "OUT") == 0)
		{
		  cpl_propertylist_append_string(corrFlux->keywords,"ESO CFG BCD MODE","OUT-OUT");
		}
	      if (strcmp(bcd1Status, "IN") == 0 && strcmp(bcd2Status, "IN") == 0)
		{
		  cpl_propertylist_append_string(corrFlux->keywords,"ESO CFG BCD MODE","IN-IN");
		}
	      if (strcmp(bcd1Status, "IN") == 0 && strcmp(bcd2Status, "OUT") == 0)
		{
		  cpl_propertylist_append_string(corrFlux->keywords,"ESO CFG BCD MODE","IN-OUT");
		}
	      if (strcmp(bcd1Status, "OUT") == 0 && strcmp(bcd2Status, "IN") == 0)
		{
		  cpl_propertylist_append_string(corrFlux->keywords,"ESO CFG BCD MODE","OUT-IN");
		}

	      if (cpl_frameset_find(frameset,"CALIB_RAW") != NULL ||
		  cpl_frameset_find(frameset,"TARGET_RAW") != NULL ||
		  cpl_frameset_find(frameset,"CALIB_SRC_RAW") != NULL)
		{
		  usedframes=cpl_frameset_new();
		  it1= cpl_frameset_iterator_new(frameset);
		  do
		    {
		      cur_frame1 = cpl_frameset_iterator_get(it1);
		      if (cur_frame1 != NULL)
			{
			  pszFileName1 = (char*)cpl_frame_get_filename( cur_frame1 );
			  pszFileTag1 = (char *)cpl_frame_get_tag( cur_frame1 );
			  if (strcmp(pszFileTag1, "CALIB_RAW") == 0 || strcmp(pszFileTag1, "TARGET_RAW") == 0 || strcmp(pszFileTag1, "CALIB_SRC_RAW") == 0)
			    {
			      plist1=cpl_propertylist_load(pszFileName1,0);
			      if (strcmp(cpl_propertylist_get_string(corrFlux->keywords,"ESO CFG BCD MODE"),"OUT-OUT") == 0)
				{
				  if (strcmp(cpl_propertylist_get_string(plist1,"ESO INS BCD1 NAME"),"OUT") == 0 &&
				      strcmp(cpl_propertylist_get_string(plist1,"ESO INS BCD2 NAME"),"OUT") == 0 &&
				      cpl_propertylist_get_int(plist1,"ESO TPL EXPNO") == cpl_propertylist_get_int(corrFlux->keywords,"ESO TPL EXPNO") )
				    {
				      sel_frame=cpl_frame_duplicate(cur_frame1);
				      cpl_frameset_insert(usedframes,sel_frame);
				    }
				}
			      else if(strcmp(cpl_propertylist_get_string(corrFlux->keywords,"ESO CFG BCD MODE"),"IN-OUT") == 0)
				{
				  if (strcmp(cpl_propertylist_get_string(plist1,"ESO INS BCD1 NAME"),"IN") == 0 &&
				      strcmp(cpl_propertylist_get_string(plist1,"ESO INS BCD2 NAME"),"OUT") == 0 &&
				      cpl_propertylist_get_int(plist1,"ESO TPL EXPNO") == cpl_propertylist_get_int(corrFlux->keywords,"ESO TPL EXPNO") )
				    {
				      sel_frame=cpl_frame_duplicate(cur_frame1);
				      cpl_frameset_insert(usedframes,sel_frame);
				    }
				}
			      else if(strcmp(cpl_propertylist_get_string(corrFlux->keywords,"ESO CFG BCD MODE"),"OUT-IN") == 0)
				{
				  if (strcmp(cpl_propertylist_get_string(plist1,"ESO INS BCD1 NAME"),"OUT") == 0 &&
				      strcmp(cpl_propertylist_get_string(plist1,"ESO INS BCD2 NAME"),"IN") == 0 &&
				      cpl_propertylist_get_int(plist1,"ESO TPL EXPNO") == cpl_propertylist_get_int(corrFlux->keywords,"ESO TPL EXPNO") )
				    {
				      sel_frame=cpl_frame_duplicate(cur_frame1);
				      cpl_frameset_insert(usedframes,sel_frame);
				    }
				}
			      else if(strcmp(cpl_propertylist_get_string(corrFlux->keywords,"ESO CFG BCD MODE"),"IN-IN") == 0)
				{
				  if (strcmp(cpl_propertylist_get_string(plist1,"ESO INS BCD1 NAME"),"IN") == 0 &&
				      strcmp(cpl_propertylist_get_string(plist1,"ESO INS BCD2 NAME"),"IN") == 0 &&
				      cpl_propertylist_get_int(plist1,"ESO TPL EXPNO") == cpl_propertylist_get_int(corrFlux->keywords,"ESO TPL EXPNO") )
				    {
				      sel_frame=cpl_frame_duplicate(cur_frame1);
				      cpl_frameset_insert(usedframes,sel_frame);
				    }
				}
			      cpl_propertylist_delete(plist1);
			    }
			}
		    } while (cpl_frameset_iterator_advance(it1, 1) != CPL_ERROR_ACCESS_OUT_OF_RANGE);
		  cpl_frameset_iterator_delete(it1);
		  it1= cpl_frameset_iterator_new(frameset);
		  do
		    {
		      cur_frame1 = cpl_frameset_iterator_get(it1);
		      if (cur_frame1 != NULL)
			{
			  pszFileName1 = (char*)cpl_frame_get_filename( cur_frame1 );
			  pszFileTag1 = (char *)cpl_frame_get_tag( cur_frame1 );
			  if (strcmp(pszFileTag1, "CALIB_RAW") == 0 || strcmp(pszFileTag1, "TARGET_RAW") == 0 || strcmp(pszFileTag1, "CALIB_SRC_RAW") == 0)
			    {
			      plist1=cpl_propertylist_load(pszFileName1,0);
			      if (strcmp(cpl_propertylist_get_string(corrFlux->keywords,"ESO CFG BCD MODE"),"OUT-OUT") == 0)
				{
				  if (strcmp(cpl_propertylist_get_string(plist1,"ESO INS BCD1 NAME"),"OUT") != 0 ||
				      strcmp(cpl_propertylist_get_string(plist1,"ESO INS BCD2 NAME"),"OUT") != 0 ||
				      cpl_propertylist_get_int(plist1,"ESO TPL EXPNO") != cpl_propertylist_get_int(corrFlux->keywords,"ESO TPL EXPNO") )
				    {
				      sel_frame=cpl_frame_duplicate(cur_frame1);
				      cpl_frameset_insert(usedframes,sel_frame);
				    }
				}
			      else if(strcmp(cpl_propertylist_get_string(corrFlux->keywords,"ESO CFG BCD MODE"),"IN-OUT") == 0)
				{
				  if (strcmp(cpl_propertylist_get_string(plist1,"ESO INS BCD1 NAME"),"IN") != 0 ||
				      strcmp(cpl_propertylist_get_string(plist1,"ESO INS BCD2 NAME"),"OUT") != 0 ||
				      cpl_propertylist_get_int(plist1,"ESO TPL EXPNO") != cpl_propertylist_get_int(corrFlux->keywords,"ESO TPL EXPNO"))
				    {
				      sel_frame=cpl_frame_duplicate(cur_frame1);
				      cpl_frameset_insert(usedframes,sel_frame);
				    }
				}
			      else if(strcmp(cpl_propertylist_get_string(corrFlux->keywords,"ESO CFG BCD MODE"),"OUT-IN") == 0)
				{
				  if (strcmp(cpl_propertylist_get_string(plist1,"ESO INS BCD1 NAME"),"OUT") != 0 ||
				      strcmp(cpl_propertylist_get_string(plist1,"ESO INS BCD2 NAME"),"IN") != 0 ||
				      cpl_propertylist_get_int(plist1,"ESO TPL EXPNO") != cpl_propertylist_get_int(corrFlux->keywords,"ESO TPL EXPNO"))
				    {
				      sel_frame=cpl_frame_duplicate(cur_frame1);
				      cpl_frameset_insert(usedframes,sel_frame);
				    }
				}
			      else if(strcmp(cpl_propertylist_get_string(corrFlux->keywords,"ESO CFG BCD MODE"),"IN-IN") == 0)
				{
				  if (strcmp(cpl_propertylist_get_string(plist1,"ESO INS BCD1 NAME"),"IN") != 0 ||
				      strcmp(cpl_propertylist_get_string(plist1,"ESO INS BCD2 NAME"),"IN") != 0 ||
				      cpl_propertylist_get_int(plist1,"ESO TPL EXPNO") != cpl_propertylist_get_int(corrFlux->keywords,"ESO TPL EXPNO"))
				    {
				      sel_frame=cpl_frame_duplicate(cur_frame1);
				      cpl_frameset_insert(usedframes,sel_frame);
				    }
				}
			      cpl_propertylist_delete(plist1);
			    }
			  else if (strcmp(pszFileTag1, "BADPIX") == 0 || strcmp(pszFileTag1, "NONLINEARITY") == 0 ||
				   strcmp(pszFileTag1, "SHIFT_MAP") == 0 || strcmp(pszFileTag1, "KAPPA_MATRIX") == 0 ||
				   strcmp(pszFileTag1, "OBS_FLATFIELD") == 0 || strcmp(pszFileTag1, "JSDC_CAT") == 0)
			    {
			      sel_frame=cpl_frame_duplicate(cur_frame1);
			      cpl_frameset_insert(usedframes,sel_frame);
			    }
		       
			}
		    } while (cpl_frameset_iterator_advance(it1, 1) != CPL_ERROR_ACCESS_OUT_OF_RANGE);
		  cpl_frameset_iterator_delete(it1);
		  it1= cpl_frameset_iterator_new(frameset);
		  do
		    {
		      cur_frame1 = cpl_frameset_iterator_get(it1);
		      if (cur_frame1 != NULL)
			{
			  pszFileName1 = (char*)cpl_frame_get_filename( cur_frame1 );
			  pszFileTag1 = (char *)cpl_frame_get_tag( cur_frame1 );
			  if (strcmp(pszFileTag1, "SKY_RAW") == 0 || strcmp(pszFileTag1, "HOT_DARK") == 0 )
			    {
			      sel_frame=cpl_frame_duplicate(cur_frame1);
			      cpl_frameset_insert(usedframes,sel_frame);
			    }
			}
		    } while (cpl_frameset_iterator_advance(it1, 1) != CPL_ERROR_ACCESS_OUT_OF_RANGE);
		  cpl_frameset_iterator_delete(it1);
		}
	      else
		{
		  sel_frame=cpl_frame_duplicate(cur_frame);
		  cpl_frame_set_group(sel_frame,CPL_FRAME_GROUP_RAW);
		  usedframes=cpl_frameset_new();
		  cpl_frameset_insert(usedframes,sel_frame);
		}

	      // 180deg Rotating of N band correlated flux
	      if (nbdetector == 2)
	      	{
	      	  for(int iFrame=0;iFrame<corrFlux->nbframe;iFrame++)
	      	    {
	      	      cpl_image_multiply_scalar(corrFlux->list_corr[iFrame]->imgreg[1],-1.0);
	      	    }
	      	}
	      
	      //Save corrFlux in Fits File
	      output=cpl_sprintf("OBJ_CORR_FLUX_%04d.fits",indexFile);
	      if (strcmp(pszFileTag,"TARGET_CAL") == 0) {
		if (strstr(protech,"NOFRINGE") != NULL)
		  {
		    mat_corrflux_save(corrFlux,corrFlux->keywords,frameset,usedframes,parlist,recname,"OBJ_CORR_FLUX","NOFRINGE","T",output);
		  }
		else
		  {
		    mat_corrflux_save(corrFlux,corrFlux->keywords,frameset,usedframes,parlist,recname,"OBJ_CORR_FLUX","IMAGE","T",output);
		  }
	      }
	      if (strcmp(pszFileTag,"CALIB_CAL") == 0)  {
		if (strstr(protech,"NOFRINGE") != NULL)
		  {
		    mat_corrflux_save(corrFlux,corrFlux->keywords,frameset,usedframes,parlist,recname,"OBJ_CORR_FLUX","NOFRINGE","F",output);
		  }
		else
		  {
		    mat_corrflux_save(corrFlux,corrFlux->keywords,frameset,usedframes,parlist,recname,"OBJ_CORR_FLUX","IMAGE","F",output);
		  }
	      }
	      indexFile++;
	      cpl_free(output);
	      if (usedframes != NULL) cpl_frameset_delete(usedframes);
	
	      if (localOpdBcd != NULL) cpl_matrix_delete(localOpdBcd);
	      if (gendata != NULL) mat_gendata_delete(gendata);
	      if (corrFlux != NULL) mat_corrflux_free(corrFlux);
	      if(avgSky != NULL) {
		for(k=0;k<nbregion;k++) {
		  if (avgSky[k] != NULL) {
		    cpl_image_delete(avgSky[k]);
		  }
		}
		cpl_free(avgSky);
		avgSky=NULL;
	      }
	      if(avgImg != NULL) {
		for(k=0;k<nbregion;k++) {
		  if (avgImg[k] != NULL) {
		    cpl_image_delete(avgImg[k]);
		  }
		}
		cpl_free(avgImg);
		avgImg=NULL;
	      }
	      cpl_vector_delete(dispCoef);
	    }
	  else
	    {
	      mat_gendata_delete(gendata);
	    }	  
	}
      // Get next frame from SOF
    }
  } while (cpl_frameset_iterator_advance(it, 1) != CPL_ERROR_ACCESS_OUT_OF_RANGE);
  cpl_frameset_iterator_delete(it);
  if (!cpl_errorstate_is_equal(prestate)) {
    cpl_errorstate_set(prestate);
  }
  return 0;
}
