/* $Id: mat_ext_beams_lib.c,v0.5 2014-06-15 12:56:21 pberio Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: pberio $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_ext_beams_lib.c $
 */
#ifdef HAVE_CONFIG_H
#include <config.h>
#include <string.h>
#endif
#include <stdio.h>
#include <sys/time.h>
/*-----------------------------------------------------------------------------
  Includes
  ----------------------------------------------------------------------------*/

#include "mat_ext_beams_lib.h"
#include "mat_error.h"

/*-----------------------------------------------------------------------------
  Define
  ----------------------------------------------------------------------------*/
#define NB_BCD 4

/* must not be followed by a semicolon!
 * gcc on mac has omp but it doesn't work for nontrivial cases as libc lacks
 * alloca */
#if defined (_OPENMP) && !defined( __APPLE__)
  #define HDRL_OMP(x) _Pragma (#x)
#else
  #define HDRL_OMP(x)
#endif

/*-----------------------------------------------------------------------------
  Functions prototypes
  ----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/





/*----------------------------------------------------------------------------*/
/**
   @brief Extract the photometry
   @param frameset           contains the frames
   @param parlist            contains the recipes parameters
   @param recname            Name of the recipes calling this function
   @return 0 if no error 
*/
/*----------------------------------------------------------------------------*/
int mat_ext_beams_lib(cpl_frameset *frameset, 
		      cpl_parameterlist *parlist,
		      const char *recname) {

  cpl_frame *cur_frame = NULL;
  cpl_frame *cur_frame1 = NULL;
  char *pszFileName    = NULL;
  char *pszFileTag     = NULL;
  char *pszFileName1    = NULL;
  char *pszFileTag1     = NULL;
  char *pszFileTagOutput = NULL;
  char *detName        = NULL;
  //  char *telescope      = NULL;
  char *sliderPhot     = NULL; 
  char detNameFirst[50];
  /* char output[200]=" "; */
  char *output=NULL;
  /* char *p_shutter = NULL; */
  cpl_propertylist *plist=NULL;
  cpl_propertylist *plist1=NULL;
  cpl_frameset *usedframes=NULL;
  cpl_frame *sel_frame=NULL;
  /* cpl_parameter *useAvgSky=NULL; */
  /* cpl_parameter *useAvgShape=NULL; */
  /* cpl_parameter *useKappaMatrix=NULL; */
  cpl_parameter *replaceTel=NULL;
  cpl_parameter *hampelFilterKernel=NULL;
  int cpt=0;
  int *cptPerReg=NULL;
  int j=0;
  /* int indreg=0; */
  int k=0;
  int nbSkyAvg=0;
  int nbSkyAvgFull=0;
  int nbTargetAvgFull=0;
 int nbSky=0;
  /* int expert=0;  */
  int flagNewTarget=0;
  int flagHighSens=0;
  int i=0;
  int *indSkyAvg=NULL;
  int nbTarget=0;
  int nbPhot=0;
  int dimSpatial=0;
  int dimSpatialInterf=0;
  int dimSpectral=0;
  int l=0;
  int m=0;
  int n=0;
  int pis=0;
  int flag_rejected=0;
  int ixPhotChan=0;
  int nbTargetPrev=0;
  int rowDet=0;
  int iJumpInterf=0;
  int ixShut=0;
  int deltapixel=0;
  double val=0.;
  /* double valcur=0.; */
  /* double offset=0.; */
  mat_kappamatrix *kappaMat=NULL;
  mat_gendata *gendata=NULL;
  mat_photbeams *photBeams=NULL;
  cpl_image **sky=NULL;
  cpl_image **skyAvgFull=NULL;
  /* cpl_image **targetAvgFull=NULL; */
  cpl_image *skyAvg=NULL;
  cpl_image *imgPhot=NULL;
  /* cpl_image *imgPhotSky=NULL; */
  cpl_image *foo=NULL;
  cpl_vector *kappaParam=NULL;
  cpl_vector *rowInterp=NULL;
  cpl_vector *rowPhot=NULL;
    
  /* int cptLast=0; */
  /* int ix=0; */
  /* int nexp=0; */
  cpl_frameset_iterator *it=NULL;
  cpl_frameset_iterator *it1=NULL;
  cpl_errorstate prestate = cpl_errorstate_get();
  int *isAlloc=NULL;
  char *bcdStatus=NULL;
  char *bcd1Status=NULL;
  char *bcd2Status=NULL;
  int flagProcess=1;
  int indexFile=1;
  int flagInIn=0;
  int flagOutOut=0;
  int kernel=0;
  cpl_parameter *spectralBinning=NULL;
  int nbBin=0;
  int iFrame, iFreq, iReg, nbFreq, nbWlen, iWlen, iPix;
  cpl_vector *vec2smooth=NULL;
  struct timeval tv1, tv2; // To measure timing of parallel section
  
  // Check input parameters
  mat_assert_value((frameset != NULL),CPL_ERROR_NULL_INPUT, 
		   CPL_ERROR_NULL_INPUT, "no frameset argument given");

  // 0. Retrieve parameter
  /* useAvgSky=cpl_parameterlist_find(parlist,"matisse.mat_ext_beams.useAvgSky"); */
  /* useAvgShape=cpl_parameterlist_find(parlist,"matisse.mat_ext_beams.useAvgShape"); */
  /* useKappaMatrix=cpl_parameterlist_find(parlist,"matisse.mat_ext_beams.useKappaMatrix"); */
  hampelFilterKernel=cpl_parameterlist_find(parlist,"matisse.mat_ext_beams.hampelFilterKernel");
  replaceTel=cpl_parameterlist_find(parlist,"matisse.mat_ext_beams.replaceTel");
  spectralBinning=cpl_parameterlist_find(parlist,"matisse.mat_raw_estimates.spectralBinning");

  kernel=cpl_parameter_get_int(hampelFilterKernel);
  if (spectralBinning != NULL)
    {
      nbBin=cpl_parameter_get_int(spectralBinning);
      if (nbBin % 2 == 0)
	{
	  nbBin++;
	}
    }
  else
    {



      nbBin=1;
    }
  cpl_msg_info(cpl_func,"Spectral Binning in pixel %d",nbBin);
  
  // 1. Check the data type of the input frames 
  cur_frame = cpl_frameset_find(frameset,"TARGET_CAL");
  if (cur_frame == NULL) {
    cur_frame = cpl_frameset_find(frameset,"CALIB_CAL");
  }
  if (cur_frame == NULL) {
    //cpl_msg_info(cpl_func,"NO TARGET_CAL or CALIB_CAL frames found");
    //return 0;
    cpl_msg_error(cpl_func,"NO TARGET_CAL or CALIB_CAL frames found");
    return CPL_ERROR_INCOMPATIBLE_INPUT;
  }
  pszFileName = (char*)cpl_frame_get_filename( cur_frame );
  plist=cpl_propertylist_load(pszFileName,0);
  detName=(char *)cpl_propertylist_get_string(plist,"ESO DET CHIP NAME");
  if (detName == NULL) {
    cpl_msg_error(cpl_func,"no ESO DET CHIP NAME keyword in frame");
    return -1;
  }
  strncpy(detNameFirst,detName,50-1); detNameFirst[50-1] = '\0';
  if(!strcmp(detName, "AQUARIUS")) {
    sliderPhot=
      mat_propertylist_get_string_default(NULL,0,plist,"ESO INS PIN NAME","");
  }
  else {
    sliderPhot=
      mat_propertylist_get_string_default(NULL,0,plist,"ESO INS PIL NAME","");
  }

  int flagAT=0;
  if (strstr(mat_propertylist_get_string_default(NULL,0,plist,"ESO ISS CONF T1NAME",""),"AT") != NULL)
    {
      flagAT=cpl_parameter_get_int(replaceTel) % 10;
      if (flagAT<0 || flagAT>4)
	{
	  flagAT=0;
	}
      cpl_msg_info(cpl_func,"Auxiliary Telescopes - relpaceTel=%d",flagAT);
    }
  else
    {
      flagAT=cpl_parameter_get_int(replaceTel);
      flagAT=(flagAT-(flagAT % 10))/10;
      if (flagAT<0 || flagAT>4)
	{
	  flagAT=0;
	}
      cpl_msg_info(cpl_func,"Unit Telescopes - relpaceTel=%d",flagAT);
    }
 
  /* flagAT=cpl_parameter_get_int(replaceTel); */
  cpl_propertylist_delete(plist);


  // 2.Load KAPPA MATRIX (if SiPhot)
  if (strcmp(sliderPhot,"PHOTO") != 0) {
    cpl_msg_info(cpl_func,"KAPPA_MATRIX not required");
  } else {
    cpl_msg_info(cpl_func,"KAPPA_MATRIX required");
    prestate=cpl_errorstate_get();
    it= cpl_frameset_iterator_new(frameset);
    do {
      cur_frame = cpl_frameset_iterator_get(it);
      if (cur_frame != NULL) {
	pszFileTag = (char *)cpl_frame_get_tag( cur_frame );
	if (strcmp(pszFileTag,"KAPPA_MATRIX") == 0) {
	  // Loading KAPPA_MATRIX frame
	  kappaMat= mat_kappamatrix_load(cur_frame);
	  /* if (kappaMat == NULL &&  */
	  /*     cpl_parameter_get_bool(useKappaMatrix)) { */
	  if (kappaMat == NULL ) {
	    cpl_msg_error(cpl_func,"could not load kappaMat");
	    return CPL_ERROR_INCOMPATIBLE_INPUT;
	  }
	} 
      }
    } while (cpl_frameset_iterator_advance(it, 1) != CPL_ERROR_ACCESS_OUT_OF_RANGE);
    cpl_frameset_iterator_delete(it);
    if (!cpl_errorstate_is_equal(prestate)) {
      cpl_errorstate_set(prestate);
    }
  }



  ////////////////////////////////////////////////////////////////////////////////
  // HighSens Mode
  //
  // While frames
  //   if TARGET_CAL or CALIB_CAL
  //     if Photometry
  //       Determination of the open shutter -> ixShut
  //       Loop on the number of sub-files
  //         if first sub-files of all frames
  //           Allocate and initialize photBeams
  //           Allocate and Initialize cptPerReg
  //           Allocate and initialize skyAvgFull if useAvgSky==TRUE
  //         Loop on the number of frames
  //           Compute the averaged sky per chopping cycle
  //         Loop on the number of region (in fact 4xnumber of spectral bands)
  //           if first time for this region
  //             Allocate and initialize photBeams->list_phot
  //           Loop on the number of frames
  //             Fill photBeams->list_phot
  //             cptPerReg++
  //         Compute skyAvgFull if useAvgSky==TRUE
  // Normalization of skyAvgFull if useAvgSky==TRUE
  // Normalization of photBeams->list_phot by cptPerReg
  // Save
  ////////////////////////////////////////////////////////////////////////////////
  if (strcmp(sliderPhot, "PHOTO") != 0) {
    cpl_msg_info(cpl_func,"HighSens mode detected");


    // Sort the data versus BCD
    for(int ibcd=0;ibcd<NB_BCD;ibcd++) {
      flagProcess=0;
      it= cpl_frameset_iterator_new(frameset);
      do {
	cur_frame = cpl_frameset_iterator_get(it);
	if (cur_frame != NULL) {
	  pszFileName = (char*)cpl_frame_get_filename( cur_frame );
	  pszFileTag = (char *)cpl_frame_get_tag( cur_frame );
	  plist=cpl_propertylist_load(pszFileName,0);
	  if ( strcmp(pszFileTag,"TARGET_CAL") == 0 || strcmp(pszFileTag,"CALIB_CAL") == 0 ) {
	    if (mat_check_photometry(cur_frame)) {
	      if ( cpl_propertylist_has(plist,"ESO INS BCD1 ID") ) {
		bcdStatus = (char *) cpl_propertylist_get_string(plist,"ESO INS BCD1 ID");
		if (strcmp(bcdStatus, "IN") == 0 && ibcd==1) {
		  flagProcess=1;
		}
		if (strcmp(bcdStatus, "OUT") == 0 &&  ibcd==0) {
		  flagProcess=1;
		}
	      }
	    }
	  }
	  cpl_propertylist_delete(plist);
	}
      } while (cpl_frameset_iterator_advance(it, 1) != CPL_ERROR_ACCESS_OUT_OF_RANGE);
      cpl_frameset_iterator_delete(it);      
      if (flagProcess==1) {
	//usedframes=cpl_frameset_new();
	prestate=cpl_errorstate_get();
	it= cpl_frameset_iterator_new(frameset);
	do {
	  cur_frame = cpl_frameset_iterator_get(it);
	  if (cur_frame != NULL) {
	    pszFileName = (char*)cpl_frame_get_filename( cur_frame );
	    pszFileTag = (char *)cpl_frame_get_tag( cur_frame );
	    plist=cpl_propertylist_load(pszFileName,0);
	    if ( cpl_propertylist_has(plist,"ESO INS BCD1 ID") ) {
	      bcdStatus = (char *) cpl_propertylist_get_string(plist,"ESO INS BCD1 ID");
	  
	      if ( (strcmp(pszFileTag,"TARGET_CAL" ) == 0 || strcmp(pszFileTag,"CALIB_CAL") == 0) &&
		   ((strcmp(bcdStatus, "IN") == 0 && ibcd==1) || (strcmp(bcdStatus, "OUT") == 0 && ibcd==0)) ) {
		// Check if frame corresponds to photometry only
		
		if (mat_check_photometry(cur_frame)) {
		  if (output==NULL) {
		    /* output=cpl_sprintf("PHOTBEAMS_%s",basename((char *)cpl_frame_get_filename(cur_frame))); */
		    pszFileTagOutput = (char *)cpl_frame_get_tag( cur_frame );
		  }
		  /* sel_frame=cpl_frame_duplicate(cur_frame); */
		  /* cpl_frame_set_group(sel_frame,CPL_FRAME_GROUP_RAW); */
		  /* cpl_frameset_insert(usedframes,sel_frame); */

		  // Search for the open shutter in the primary header
		  // ixShut contains the shutter number (0...3)
		  ixShut = mat_find_open_shutter(plist,detNameFirst);

		  flagNewTarget=1;

		  gendata=mat_gendata_load(cur_frame,CPL_TYPE_FLOAT);
		  if (gendata == NULL) {
		    cpl_msg_error(cpl_func,"no gendata loaded");
		    /* cpl_free(output); */
		    return -1;
		  }
		  // Allocate the photBeams data structure if first subfile
		  if (!flagHighSens) {
		    photBeams=cpl_malloc(sizeof(mat_photbeams));
		    if (photBeams == NULL) {
		      cpl_msg_error(cpl_func,"could not allocate memory for photBeams");
		      /* cpl_free(output); */
		      return -1;
		    }
		    mat_init_photbeams(gendata,photBeams,1,&nbPhot,0);
		    isAlloc=cpl_calloc(photBeams->nbphotxframe,sizeof(int));
		    flagHighSens=1;
		  }

		  // Loop on nbPhot. In the HighSens mode nbPhot=number of spectral bands
		  for(k=0;k<nbPhot;k++) {
		    // we allocate the list_phot data structure corresponding
		    // to the current open shutter if not already allocated
		    if (isAlloc[k*4+ixShut] == 0) {
		      photBeams->list_phot[k*4+ixShut]=cpl_malloc(sizeof(mat_imgreg));
		      photBeams->list_phot[photBeams->nbphotxframe+k*4+ixShut]=cpl_malloc(sizeof(mat_imgreg));
		      if (photBeams->list_phot[k*4+ixShut] == NULL) {
			cpl_msg_error(cpl_func,"could not allocate memory for photBeams->list_phot[k*4+ixShut]");
			cpl_free(indSkyAvg);
			cpl_free(sky);
			cpl_free(cptPerReg);
			cpl_free(isAlloc);
			/* cpl_free(output); */
			return -1;
		      }
		      photBeams->list_phot[k*4+ixShut]->imgreg=cpl_calloc(1,sizeof(cpl_image *));
		      photBeams->list_phot[photBeams->nbphotxframe+k*4+ixShut]->imgreg=cpl_calloc(1,sizeof(cpl_image *));
		      photBeams->list_phot[k*4+ixShut]->numregion=k*4+ixShut+1;
		      photBeams->list_phot[k*4+ixShut]->numdetector=gendata->imgdet->list_region[k]->numdetector;
		      dimSpatial = gendata->imgdet->list_region[k]->naxis[0];
		      dimSpectral = gendata->imgdet->list_region[k]->naxis[1];
		      photBeams->list_phot[k*4+ixShut]->imgreg[0]=cpl_image_new(dimSpatial,dimSpectral,CPL_TYPE_FLOAT);
		      photBeams->list_phot[photBeams->nbphotxframe+k*4+ixShut]->imgreg[0]=cpl_image_new(dimSpatial,dimSpectral,CPL_TYPE_FLOAT);
		      isAlloc[k*4+ixShut]=1;
		    }
		  }
		    
		  nbSkyAvgFull=0;
		  skyAvgFull=cpl_calloc(gendata->imgdet->nbregion,sizeof(cpl_image *));
		  for(k=0;k<gendata->imgdet->nbregion;k++) {
		    skyAvgFull[k]=cpl_image_new(gendata->imgdet->list_region[k]->naxis[0],
						gendata->imgdet->list_region[k]->naxis[1],CPL_TYPE_FLOAT);
		  }
		  for(j=0;j<gendata->imgdata->nbframe;j++) {
		    if ( !strcmp(gendata->imgdata->list_frame[j]->tartype,"S") ) {
		      for(k=0;k<gendata->imgdet->nbregion;k++) {
			cpl_image_add(skyAvgFull[k],
				      gendata->imgdata->list_frame[j]->list_subwin[k]->imgreg[0]);
			nbSkyAvgFull++;
		      }
		    }
		  }
		  nbSkyAvgFull/=(nbPhot);
		  for(k=0;k<nbPhot;k++) {
		    cpl_image_divide_scalar(skyAvgFull[k],(double)nbSkyAvgFull);
		  }

		    
		  nbTargetAvgFull=0;
		  for(j=0;j<gendata->imgdata->nbframe;j++) {
		    if ( !strcmp(gendata->imgdata->list_frame[j]->tartype,"T") ) {
		      for(k=0;k<gendata->imgdet->nbregion;k++) {
			cpl_image_add(photBeams->list_phot[k*4+ixShut]->imgreg[0],
				      gendata->imgdata->list_frame[j]->list_subwin[k]->imgreg[0]);
			foo=cpl_image_multiply_create(gendata->imgdata->list_frame[j]->list_subwin[k]->imgreg[0],
						      gendata->imgdata->list_frame[j]->list_subwin[k]->imgreg[0]);
			  
			cpl_image_add(photBeams->list_phot[photBeams->nbphotxframe+k*4+ixShut]->imgreg[0],foo);
			cpl_image_delete(foo);
			nbTargetAvgFull++;
		      }
		    }
		  }
		  nbTargetAvgFull/=nbPhot;
		  for(k=0;k<nbPhot;k++) {
		    cpl_image_divide_scalar(photBeams->list_phot[k*4+ixShut]->imgreg[0],(double)nbTargetAvgFull);
		    cpl_image_divide_scalar(photBeams->list_phot[photBeams->nbphotxframe+k*4+ixShut]->imgreg[0],(double)nbTargetAvgFull);
		  }
		  //cpl_image_save(skyAvgFull[0],"sky.fits", CPL_BPP_IEEE_FLOAT,NULL, CPL_IO_CREATE);
		    
		  for(k=0;k<nbPhot;k++) {
		    cpl_image_subtract(photBeams->list_phot[k*4+ixShut]->imgreg[0],skyAvgFull[k]);
		    // Computation of the stdev of the image
		    foo=cpl_image_multiply_create(photBeams->list_phot[k*4+ixShut]->imgreg[0],photBeams->list_phot[k*4+ixShut]->imgreg[0]);
		    cpl_image_subtract(photBeams->list_phot[photBeams->nbphotxframe+k*4+ixShut]->imgreg[0],foo);
		    cpl_image_delete(foo);	
		    for(m=1;m<=dimSpatial;m++) {
		      for(n=1;n<=dimSpectral;n++) {
			val=cpl_image_get(photBeams->list_phot[photBeams->nbphotxframe+k*4+ixShut]->imgreg[0],m,n,&pis);
			cpl_image_set(photBeams->list_phot[photBeams->nbphotxframe+k*4+ixShut]->imgreg[0],m,n,sqrt(val)/nbTargetAvgFull);
		      }
		    }
		  }
		  //cpl_image_save(photBeams->list_phot[ixShut]->imgreg[0],"image.fits", CPL_BPP_IEEE_FLOAT,NULL, CPL_IO_CREATE);
		  //exit(0);
		  for(k=0;k<nbPhot;k++) {
		    cpl_image_delete(skyAvgFull[k]);
		  }
		  if (skyAvgFull != NULL) cpl_free(skyAvgFull);
		  
		  if (gendata != NULL) mat_gendata_delete(gendata);
		}
	      }
	    }
	    if (plist != NULL) cpl_propertylist_delete(plist);
	  }
	} while (cpl_frameset_iterator_advance(it, 1) != CPL_ERROR_ACCESS_OUT_OF_RANGE);
	cpl_frameset_iterator_delete(it);
       	if (!cpl_errorstate_is_equal(prestate)) {
	  cpl_errorstate_set(prestate);
	}

	
	if (photBeams == NULL) {
	  cpl_msg_warning(cpl_func,"No frames corresponding to photometry");
	  cpl_free(sliderPhot);
	  if (kappaMat != NULL) {
	    mat_kappamatrix_free(kappaMat);
	  }
	  /* if (usedframes != NULL) cpl_frameset_delete(usedframes); */
	  
	  return 0;
	}
	if (flagHighSens == 1) {

	  for(k=0;k<nbPhot;k++) {
	    for(ixShut=0;ixShut<4;ixShut++) {
	      if (isAlloc[k*4+ixShut]==0)
		{
		  cpl_msg_warning(cpl_func,"Photometric data for Shutter %d is missing",ixShut);
		  return 0;
		}
	    }
	  }
	  cpl_free(isAlloc);
	}

	// Apply Spectral Binning
	if (nbBin > 1 )
	  {
	    cpt=0;
	    for(iReg=0;iReg<nbPhot;iReg++)
	      {
		nbFreq=photBeams->imgdet->list_region[iReg]->naxis[0];
		nbWlen=photBeams->imgdet->list_region[iReg]->naxis[1];
		for(iFrame=0; iFrame<photBeams->nbphotxframe/nbPhot; iFrame++) {
		  {
		    for(iFreq=0;iFreq<nbFreq;iFreq++)
		      {
			vec2smooth=cpl_vector_new(nbWlen);
			for(iWlen=0;iWlen<nbWlen;iWlen++)
			  {
			    cpl_vector_set(vec2smooth,iWlen,cpl_image_get(photBeams->list_phot[cpt]->imgreg[0],iFreq+1,iWlen+1,&pis));
			  }
			for(iWlen=0;iWlen<nbWlen;iWlen++)
			  {
			    if (iWlen < (nbBin-1)/2)
			      {
				val=0.;
				for(iPix=0;iPix<=2*iWlen;iPix++)
				  {
				    val+=cpl_vector_get(vec2smooth,iPix);
				  }
				val/=(2*iWlen+1);
				cpl_image_set(photBeams->list_phot[cpt]->imgreg[0],iFreq+1,iWlen+1,val);
			      }
			    else if (iWlen >= nbWlen-(nbBin-1)/2)
			      {
				val=0.;
				for(iPix=iWlen-(nbWlen-1-iWlen);iPix<=iWlen+(nbWlen-1-iWlen);iPix++)
				  {
				    val+=cpl_vector_get(vec2smooth,iPix);
				  }
				val/=(2*(nbWlen-1-iWlen)+1);
				cpl_image_set(photBeams->list_phot[cpt]->imgreg[0],iFreq+1,iWlen+1,val);
			      }
			    else
			      {
				val=0.;
				for(iPix=iWlen-(nbBin-1)/2;iPix<=iWlen+(nbBin-1)/2;iPix++)
				  {
				    val+=cpl_vector_get(vec2smooth,iPix);
				  }
				val/=nbBin;
				cpl_image_set(photBeams->list_phot[cpt]->imgreg[0],iFreq+1,iWlen+1,val);
			      }
			  }
			cpl_vector_delete(vec2smooth);
		      }
		    cpt++;
		  }
		}
	      }
	  }




	// Gaussian fit
	double params[5]={10.,0.5,25.,0.1,0.};
	double params_red[3]={10.,0.1,0.};
	gsl_vector *vector;
	double t=0.;
	double cen0=0.;
	double wid0=0.;
	double wave0=0.;
	double wave=0.;
	double amp0=0;;
	cpl_vector *dispCoef;
	int iWlen0;

	dispCoef=cpl_vector_new(N_DEG_DISPERSION_LAW+1);
	cpl_vector_set(dispCoef,0,cpl_propertylist_get_double(photBeams->keywords,"PRO DISP COEF0"));
	cpl_vector_set(dispCoef,1,cpl_propertylist_get_double(photBeams->keywords,"PRO DISP COEF1"));
	cpl_vector_set(dispCoef,2,cpl_propertylist_get_double(photBeams->keywords,"PRO DISP COEF2"));
	cpl_vector_set(dispCoef,3,cpl_propertylist_get_double(photBeams->keywords,"PRO DISP COEF3"));
	cpl_vector_set(dispCoef,4,cpl_propertylist_get_double(photBeams->keywords,"PRO DISP COEF4"));


	vector=gsl_vector_alloc(dimSpatial);
	for(k=0;k<nbPhot;k++) {
	  iWlen0 = photBeams->imgdet->list_region[k]->corner[1];
	  for(ixShut=0;ixShut<4;ixShut++) {
	    // first estimation on a brigth spectral channel
	    int jspectral=15;
	    float vecmax=0.;
	    for(i=0;i<dimSpatial;i++) {
	      gsl_vector_set(vector,i,cpl_image_get(photBeams->list_phot[k*4+ixShut]->imgreg[0],i+1,jspectral+1,&pis));
	      if (cpl_image_get(photBeams->list_phot[k*4+ixShut]->imgreg[0],i+1,jspectral+1,&pis) > vecmax)
		{
		  vecmax=cpl_image_get(photBeams->list_phot[k*4+ixShut]->imgreg[0],i+1,jspectral+1,&pis);
		}
	    }
	    params[0]=vecmax;
	    mat_fit_gauss_linear(dimSpatial, vector, params);
	    amp0=params[0];
	    cen0=params[1];
	    wid0=params[2];
	    wave0=mat_get_lambda(dispCoef,jspectral+iWlen0);
	    if (amp0 > 0.) {
	      for(j=0;j<dimSpectral;j++) {
		for(i=0;i<dimSpatial;i++) {
		  gsl_vector_set(vector,i,cpl_image_get(photBeams->list_phot[k*4+ixShut]->imgreg[0],i+1,j+1,&pis));
		}
		wave=mat_get_lambda(dispCoef,j+iWlen0);
		mat_fit_gauss_linear_red(dimSpatial, vector, params_red, cen0, wid0*pow(wave0/wave,2.));
		for(i=0;i<dimSpatial;i++) {
		  t=i/(1.*dimSpatial);
		  cpl_image_set(photBeams->list_phot[k*4+ixShut]->imgreg[0],i+1,j+1,params_red[0] * exp (-1.*wid0*pow(wave0/wave,2.)*pow(t-cen0, 2)));
		}
	      }
	    } else {
	      cpl_msg_warning(cpl_func,"Unable to fit the photometry shape for shutter %d\n",ixShut);
	    }
	  }
	}
	cpl_vector_delete(dispCoef);




	







	
	// Replace AT1 by (AT2+AT3+AT4)/3
	if (flagAT) {
	  cpl_msg_info(cpl_func,"Replace Telescope %d",flagAT);
	  for(k=0;k<nbPhot;k++) {
	    for(j=0;j<dimSpectral;j++) {
	      for(i=0;i<dimSpatial;i++) {
		if (ibcd==0) {
		  if (flagAT == 3) {
		    cpl_image_set(photBeams->list_phot[k*4]->imgreg[0],i+1,j+1,
				  (cpl_image_get(photBeams->list_phot[k*4+1]->imgreg[0],i+1,j+1,&pis)+
				   cpl_image_get(photBeams->list_phot[k*4+2]->imgreg[0],i+1,j+1,&pis)+
				   cpl_image_get(photBeams->list_phot[k*4+3]->imgreg[0],i+1,j+1,&pis))/3.);
		  } else if (flagAT == 4) {
		    cpl_image_set(photBeams->list_phot[k*4+1]->imgreg[0],i+1,j+1,
				  (cpl_image_get(photBeams->list_phot[k*4]->imgreg[0],i+1,j+1,&pis)+
				   cpl_image_get(photBeams->list_phot[k*4+2]->imgreg[0],i+1,j+1,&pis)+
				   cpl_image_get(photBeams->list_phot[k*4+3]->imgreg[0],i+1,j+1,&pis))/3.);
		  } else if (flagAT == 2) {
		    cpl_image_set(photBeams->list_phot[k*4+2]->imgreg[0],i+1,j+1,
				  (cpl_image_get(photBeams->list_phot[k*4]->imgreg[0],i+1,j+1,&pis)+
				   cpl_image_get(photBeams->list_phot[k*4+1]->imgreg[0],i+1,j+1,&pis)+
				   cpl_image_get(photBeams->list_phot[k*4+3]->imgreg[0],i+1,j+1,&pis))/3.);
		  } else if (flagAT == 1) {
		    cpl_image_set(photBeams->list_phot[k*4+3]->imgreg[0],i+1,j+1,
				  (cpl_image_get(photBeams->list_phot[k*4]->imgreg[0],i+1,j+1,&pis)+
				   cpl_image_get(photBeams->list_phot[k*4+1]->imgreg[0],i+1,j+1,&pis)+
				   cpl_image_get(photBeams->list_phot[k*4+2]->imgreg[0],i+1,j+1,&pis))/3.);
		  }
				
		} else {
		  if (flagAT == 3) {
		    cpl_image_set(photBeams->list_phot[k*4+1]->imgreg[0],i+1,j+1,
				  (cpl_image_get(photBeams->list_phot[k*4+0]->imgreg[0],i+1,j+1,&pis)+
				   cpl_image_get(photBeams->list_phot[k*4+2]->imgreg[0],i+1,j+1,&pis)+
				   cpl_image_get(photBeams->list_phot[k*4+3]->imgreg[0],i+1,j+1,&pis))/3.);
		  } else if (flagAT == 4) {
		    cpl_image_set(photBeams->list_phot[k*4]->imgreg[0],i+1,j+1,
				  (cpl_image_get(photBeams->list_phot[k*4+1]->imgreg[0],i+1,j+1,&pis)+
				   cpl_image_get(photBeams->list_phot[k*4+2]->imgreg[0],i+1,j+1,&pis)+
				   cpl_image_get(photBeams->list_phot[k*4+3]->imgreg[0],i+1,j+1,&pis))/3.);
		  } else if (flagAT == 2) {
		    cpl_image_set(photBeams->list_phot[k*4+3]->imgreg[0],i+1,j+1,
				  (cpl_image_get(photBeams->list_phot[k*4]->imgreg[0],i+1,j+1,&pis)+
				   cpl_image_get(photBeams->list_phot[k*4+1]->imgreg[0],i+1,j+1,&pis)+
				   cpl_image_get(photBeams->list_phot[k*4+2]->imgreg[0],i+1,j+1,&pis))/3.);
		  } else if (flagAT == 1) {
		    cpl_image_set(photBeams->list_phot[k*4+2]->imgreg[0],i+1,j+1,
				  (cpl_image_get(photBeams->list_phot[k*4]->imgreg[0],i+1,j+1,&pis)+
				   cpl_image_get(photBeams->list_phot[k*4+1]->imgreg[0],i+1,j+1,&pis)+
				   cpl_image_get(photBeams->list_phot[k*4+3]->imgreg[0],i+1,j+1,&pis))/3.);
		  }	    
		}
	      }
	    }
	  }
	}

	// Save in a Fits file

	/* * 2 for taking into account of the stdev of the images */
	photBeams->nbphotxframe*=2;
	prestate=cpl_errorstate_get();
	output=cpl_sprintf("PHOT_BEAMS_%04d.fits",indexFile);
	if (indexFile == 1)
	  {
	    flagOutOut = 1;
	    cpl_propertylist_append_string(photBeams->keywords,"ESO CFG BCD MODE","OUT-OUT");
	  }
	if (indexFile == 2)
	  {
	    cpl_propertylist_append_string(photBeams->keywords,"ESO CFG BCD MODE","IN-IN");
	    flagInIn = 1;
	  }
	
	// Fill usedframes depending on the way mat_ext_beams is called (atomic or master recipes)
	if (cpl_frameset_find(frameset,"CALIB_RAW") != NULL ||
	    cpl_frameset_find(frameset,"TARGET_RAW") != NULL ||
	    cpl_frameset_find(frameset,"CALIB_SRC_RAW") != NULL)
	  {
	    usedframes=cpl_frameset_new();
	    it1= cpl_frameset_iterator_new(frameset);
	    do
	      {
		cur_frame1 = cpl_frameset_iterator_get(it1);
		if (cur_frame1 != NULL)
		  {
		    pszFileName1 = (char*)cpl_frame_get_filename( cur_frame1 );
		    pszFileTag1 = (char *)cpl_frame_get_tag( cur_frame1 );
		    if (strcmp(pszFileTag1, "CALIB_RAW") == 0 || strcmp(pszFileTag1, "TARGET_RAW") == 0 || strcmp(pszFileTag1, "CALIB_SRC_RAW") == 0)
		      {
			plist1=cpl_propertylist_load(pszFileName1,0);
			if (strcmp(cpl_propertylist_get_string(photBeams->keywords,"ESO CFG BCD MODE"),"OUT-OUT") == 0)
			  {
			    if (strcmp(cpl_propertylist_get_string(plist1,"ESO INS BCD1 NAME"),"OUT") == 0 &&
				strcmp(cpl_propertylist_get_string(plist1,"ESO INS BCD2 NAME"),"OUT") == 0 )
			      {
				sel_frame=cpl_frame_duplicate(cur_frame1);
				cpl_frameset_insert(usedframes,sel_frame);
			      }
			  }
			else if(strcmp(cpl_propertylist_get_string(photBeams->keywords,"ESO CFG BCD MODE"),"IN-OUT") == 0)
			  {
			    if (strcmp(cpl_propertylist_get_string(plist1,"ESO INS BCD1 NAME"),"IN") == 0 &&
				strcmp(cpl_propertylist_get_string(plist1,"ESO INS BCD2 NAME"),"OUT") == 0 )
			      {
				sel_frame=cpl_frame_duplicate(cur_frame1);
				cpl_frameset_insert(usedframes,sel_frame);
			      }
			  }
			else if(strcmp(cpl_propertylist_get_string(photBeams->keywords,"ESO CFG BCD MODE"),"OUT-IN") == 0)
			  {
			    if (strcmp(cpl_propertylist_get_string(plist1,"ESO INS BCD1 NAME"),"OUT") == 0 &&
				strcmp(cpl_propertylist_get_string(plist1,"ESO INS BCD2 NAME"),"IN") == 0 )
			      {
				sel_frame=cpl_frame_duplicate(cur_frame1);
				cpl_frameset_insert(usedframes,sel_frame);
			      }
			  }
			else if(strcmp(cpl_propertylist_get_string(photBeams->keywords,"ESO CFG BCD MODE"),"IN-IN") == 0)
			  {
			    if (strcmp(cpl_propertylist_get_string(plist1,"ESO INS BCD1 NAME"),"IN") == 0 &&
				strcmp(cpl_propertylist_get_string(plist1,"ESO INS BCD2 NAME"),"IN") == 0 )
			      {
				sel_frame=cpl_frame_duplicate(cur_frame1);
				cpl_frameset_insert(usedframes,sel_frame);
			      }
			  }
			cpl_propertylist_delete(plist1);
		      }
		  }
	      } while (cpl_frameset_iterator_advance(it1, 1) != CPL_ERROR_ACCESS_OUT_OF_RANGE);
	    cpl_frameset_iterator_delete(it1);
	    it1= cpl_frameset_iterator_new(frameset);
	    do
	      {
		cur_frame1 = cpl_frameset_iterator_get(it1);
		if (cur_frame1 != NULL)
		  {
		    pszFileName1 = (char*)cpl_frame_get_filename( cur_frame1 );
		    pszFileTag1 = (char *)cpl_frame_get_tag( cur_frame1 );
		    if (strcmp(pszFileTag1, "CALIB_RAW") == 0 || strcmp(pszFileTag1, "TARGET_RAW") == 0 || strcmp(pszFileTag1, "CALIB_SRC_RAW") == 0)
		      {
			plist1=cpl_propertylist_load(pszFileName1,0);
			if (strcmp(cpl_propertylist_get_string(photBeams->keywords,"ESO CFG BCD MODE"),"OUT-OUT") == 0)
			  {
			    if (strcmp(cpl_propertylist_get_string(plist1,"ESO INS BCD1 NAME"),"OUT") != 0 ||
				strcmp(cpl_propertylist_get_string(plist1,"ESO INS BCD2 NAME"),"OUT") != 0 )
			      {
				sel_frame=cpl_frame_duplicate(cur_frame1);
				cpl_frameset_insert(usedframes,sel_frame);
			      }
			  }
			else if(strcmp(cpl_propertylist_get_string(photBeams->keywords,"ESO CFG BCD MODE"),"IN-OUT") == 0)
			  {
			    if (strcmp(cpl_propertylist_get_string(plist1,"ESO INS BCD1 NAME"),"IN") != 0 ||
				strcmp(cpl_propertylist_get_string(plist1,"ESO INS BCD2 NAME"),"OUT") != 0 )
			      {
				sel_frame=cpl_frame_duplicate(cur_frame1);
				cpl_frameset_insert(usedframes,sel_frame);
			      }
			  }
			else if(strcmp(cpl_propertylist_get_string(photBeams->keywords,"ESO CFG BCD MODE"),"OUT-IN") == 0)
			  {
			    if (strcmp(cpl_propertylist_get_string(plist1,"ESO INS BCD1 NAME"),"OUT") != 0 ||
				strcmp(cpl_propertylist_get_string(plist1,"ESO INS BCD2 NAME"),"IN") != 0 )
			      {
				sel_frame=cpl_frame_duplicate(cur_frame1);
				cpl_frameset_insert(usedframes,sel_frame);
			      }
			  }
			else if(strcmp(cpl_propertylist_get_string(photBeams->keywords,"ESO CFG BCD MODE"),"IN-IN") == 0)
			  {
			    if (strcmp(cpl_propertylist_get_string(plist1,"ESO INS BCD1 NAME"),"IN") != 0 ||
				strcmp(cpl_propertylist_get_string(plist1,"ESO INS BCD2 NAME"),"IN") != 0 )
			      {
				sel_frame=cpl_frame_duplicate(cur_frame1);
				cpl_frameset_insert(usedframes,sel_frame);
			      }
			  }
			cpl_propertylist_delete(plist1);
		      }
		    else if (strcmp(pszFileTag1, "BADPIX") == 0 || strcmp(pszFileTag1, "NONLINEARITY") == 0 ||
			     strcmp(pszFileTag1, "SHIFT_MAP") == 0 || strcmp(pszFileTag1, "KAPPA_MATRIX") == 0 ||
			     strcmp(pszFileTag1, "OBS_FLATFIELD") == 0 || strcmp(pszFileTag1, "JSDC_CAT") == 0)
		      {
			sel_frame=cpl_frame_duplicate(cur_frame1);
			cpl_frameset_insert(usedframes,sel_frame);
		      }
		       
		  }
	      } while (cpl_frameset_iterator_advance(it1, 1) != CPL_ERROR_ACCESS_OUT_OF_RANGE);
	    cpl_frameset_iterator_delete(it1);
	    
	    it1= cpl_frameset_iterator_new(frameset);
	    do
	      {
		cur_frame1 = cpl_frameset_iterator_get(it1);
		if (cur_frame1 != NULL)
		  {
		    pszFileName1 = (char*)cpl_frame_get_filename( cur_frame1 );
		    pszFileTag1 = (char *)cpl_frame_get_tag( cur_frame1 );
		    if (strcmp(pszFileTag1, "SKY_RAW") == 0 || strcmp(pszFileTag1, "HOT_DARK") == 0 )
		      {
			sel_frame=cpl_frame_duplicate(cur_frame1);
			cpl_frameset_insert(usedframes,sel_frame);
		      }
		  }
	      } while (cpl_frameset_iterator_advance(it1, 1) != CPL_ERROR_ACCESS_OUT_OF_RANGE);
	    cpl_frameset_iterator_delete(it1);
	  }
	else
	  {
	    usedframes=cpl_frameset_duplicate(frameset);
	    it= cpl_frameset_iterator_new(usedframes);
	    do {
	      cur_frame = cpl_frameset_iterator_get(it);
	      cpl_frame_set_group(cur_frame,CPL_FRAME_GROUP_RAW);
	    } while (cpl_frameset_iterator_advance(it, 1) != CPL_ERROR_ACCESS_OUT_OF_RANGE);
	    cpl_frameset_iterator_delete(it);
	  }




	if (strcmp(pszFileTagOutput,"TARGET_CAL") == 0) {
	  mat_photbeams_save(photBeams,photBeams->keywords,frameset,usedframes,parlist,recname,"PHOT_BEAMS","IMAGE","T",output);
	}
	if (strcmp(pszFileTagOutput,"CALIB_CAL") == 0)  {
	  mat_photbeams_save(photBeams,photBeams->keywords,frameset,usedframes,parlist,recname,"PHOT_BEAMS","IMAGE","F",output);
	}
	indexFile++;
	mat_imagingdetector_save(photBeams->imgdet,photBeams->keywords,output);
	mat_photbeams_free(photBeams);
	photBeams=NULL;
	cpl_free(output);


	//Create PHOTBEAMS in IN-OUT and OUT-IN from PHOTBEAMS in IN-IN and OUT-OUT
	if ( flagInIn && flagOutOut)
	  {
	    cpl_msg_info(cpl_func,"PHOTBEAMS IN-IN and OUT-OUT exist");
	    cpl_msg_info(cpl_func,"Creation of PHOTBEAMS IN-OUT and OUT-IN");
	    cpl_frame *frameInIn=NULL;
	    cpl_frame *frameOutOut=NULL;
	    mat_photbeams *photInIn=NULL;
	    mat_photbeams *photOutOut=NULL;
	    mat_photbeams *phot=NULL;
	    frameInIn=cpl_frame_new();
	    cpl_frame_set_filename(frameInIn,"PHOT_BEAMS_0002.fits");
	    photInIn=mat_photbeams_load(frameInIn);
	    cpl_frame_delete(frameInIn);

	    frameOutOut=cpl_frame_new();
	    cpl_frame_set_filename(frameOutOut,"PHOT_BEAMS_0001.fits");
	    photOutOut=mat_photbeams_load(frameOutOut);
	    cpl_frame_delete(frameOutOut);
	
	    // IN-OUT
	    phot=mat_photbeams_duplicate(photInIn);
	    cpl_propertylist_append_string(phot->keywords,"ESO CFG BCD MODE","IN-OUT");
	    for(k=0;k<nbPhot;k++) {
	      cpl_image_copy(phot->list_phot[k*4+0]->imgreg[0],
			     photInIn->list_phot[(k*4+0)*phot->nbphotxframe/phot->imgdet->nbregion]->imgreg[0],1,1);
	      cpl_image_copy(phot->list_phot[k*4+1]->imgreg[0],
			     photInIn->list_phot[(k*4+1)*phot->nbphotxframe/phot->imgdet->nbregion]->imgreg[0],1,1);
	      cpl_image_copy(phot->list_phot[phot->nbphotxframe/2+k*4+0]->imgreg[0],
			     photInIn->list_phot[(k*4+0)*phot->nbphotxframe/phot->imgdet->nbregion+1]->imgreg[0],1,1);
	      cpl_image_copy(phot->list_phot[phot->nbphotxframe/2+k*4+1]->imgreg[0],
			     photInIn->list_phot[(k*4+1)*phot->nbphotxframe/phot->imgdet->nbregion+1]->imgreg[0],1,1);

	      cpl_image_copy(phot->list_phot[k*4+2]->imgreg[0],
			     photOutOut->list_phot[(k*4+2)*phot->nbphotxframe/phot->imgdet->nbregion]->imgreg[0],1,1);
	      cpl_image_copy(phot->list_phot[k*4+3]->imgreg[0],
			     photOutOut->list_phot[(k*4+3)*phot->nbphotxframe/phot->imgdet->nbregion]->imgreg[0],1,1);
	      cpl_image_copy(phot->list_phot[phot->nbphotxframe/2+k*4+2]->imgreg[0],
			     photOutOut->list_phot[(k*4+2)*phot->nbphotxframe/phot->imgdet->nbregion+1]->imgreg[0],1,1);
	      cpl_image_copy(phot->list_phot[phot->nbphotxframe/2+k*4+3]->imgreg[0],
			     photOutOut->list_phot[(k*4+3)*phot->nbphotxframe/phot->imgdet->nbregion+1]->imgreg[0],1,1);
	    }
	    
	    output=cpl_sprintf("PHOT_BEAMS_0003.fits");
	    if (strcmp(pszFileTagOutput,"TARGET_CAL") == 0) {
	      mat_photbeams_save(phot,phot->keywords,frameset,usedframes,parlist,recname,"PHOT_BEAMS","IMAGE","T",output);
	    }
	    if (strcmp(pszFileTagOutput,"CALIB_CAL") == 0)  {
	      mat_photbeams_save(phot,phot->keywords,frameset,usedframes,parlist,recname,"PHOT_BEAMS","IMAGE","F",output);
	    }
	    mat_imagingdetector_save(phot->imgdet,phot->keywords,output);
	    mat_photbeams_free(phot);
	    phot=NULL;
	    cpl_free(output);

	    // OUT-IN
	    phot=mat_photbeams_duplicate(photOutOut);
	    cpl_propertylist_append_string(phot->keywords,"ESO CFG BCD MODE","OUT-IN");
	    for(k=0;k<nbPhot;k++) {
	      cpl_image_copy(phot->list_phot[k*4+0]->imgreg[0],
			     photOutOut->list_phot[(k*4+0)*phot->nbphotxframe/phot->imgdet->nbregion]->imgreg[0],1,1);
	      cpl_image_copy(phot->list_phot[k*4+1]->imgreg[0],
			     photOutOut->list_phot[(k*4+1)*phot->nbphotxframe/phot->imgdet->nbregion]->imgreg[0],1,1);
	      cpl_image_copy(phot->list_phot[phot->nbphotxframe/2+k*4+0]->imgreg[0],
			     photOutOut->list_phot[(k*4+0)*phot->nbphotxframe/phot->imgdet->nbregion+1]->imgreg[0],1,1);
	      cpl_image_copy(phot->list_phot[phot->nbphotxframe/2+k*4+1]->imgreg[0],
			     photOutOut->list_phot[(k*4+1)*phot->nbphotxframe/phot->imgdet->nbregion+1]->imgreg[0],1,1);
	      
	      cpl_image_copy(phot->list_phot[k*4+2]->imgreg[0],
			     photInIn->list_phot[(k*4+2)*phot->nbphotxframe/phot->imgdet->nbregion]->imgreg[0],1,1);
	      cpl_image_copy(phot->list_phot[k*4+3]->imgreg[0],
			     photInIn->list_phot[(k*4+3)*phot->nbphotxframe/phot->imgdet->nbregion]->imgreg[0],1,1);
	      cpl_image_copy(phot->list_phot[phot->nbphotxframe/2+k*4+2]->imgreg[0],
			     photInIn->list_phot[(k*4+2)*phot->nbphotxframe/phot->imgdet->nbregion+1]->imgreg[0],1,1);
	      cpl_image_copy(phot->list_phot[phot->nbphotxframe/2+k*4+3]->imgreg[0],
			     photInIn->list_phot[(k*4+3)*phot->nbphotxframe/phot->imgdet->nbregion+1]->imgreg[0],1,1);
	    }

	    output=cpl_sprintf("PHOT_BEAMS_0004.fits");
	    if (strcmp(pszFileTagOutput,"TARGET_CAL") == 0) {
	      mat_photbeams_save(phot,phot->keywords,frameset,usedframes,parlist,recname,"PHOT_BEAMS","IMAGE","T",output);
	    }
	    if (strcmp(pszFileTagOutput,"CALIB_CAL") == 0)  {
	      mat_photbeams_save(phot,phot->keywords,frameset,usedframes,parlist,recname,"PHOT_BEAMS","IMAGE","F",output);
	    }
	    mat_imagingdetector_save(phot->imgdet,phot->keywords,output);
	    mat_photbeams_free(phot);
	    phot=NULL;
	    cpl_free(output);

	    mat_photbeams_free(photInIn);
	    mat_photbeams_free(photOutOut);	
	  }
	
	if (usedframes != NULL) cpl_frameset_delete(usedframes);
	output=NULL;

	flagHighSens=0;
      }
    }
    
    ////////////////////////////////////////////////////////////////////////////////
    // SiPhot Mode
    ////////////////////////////////////////////////////////////////////////////////
  } else {
    char *protech=NULL;
    cpl_msg_info(cpl_func,"SiPhot mode detected");
    if (kappaMat == NULL ) {
      cpl_msg_error(cpl_func,"KAPPA_MATRIX not found in the sof");
      return CPL_ERROR_INCOMPATIBLE_INPUT;
    }

    prestate=cpl_errorstate_get();
    it= cpl_frameset_iterator_new(frameset);
    do {
      cur_frame = cpl_frameset_iterator_get(it);
      if (cur_frame != NULL) {
	pszFileName = (char*)cpl_frame_get_filename( cur_frame );
	pszFileTag = (char *)cpl_frame_get_tag( cur_frame );
	plist=cpl_propertylist_load(pszFileName,0);
	protech=(char *)cpl_propertylist_get_string(plist,"ESO PRO TECH");
	if ( (strcmp(pszFileTag,"TARGET_CAL") == 0 || strcmp(pszFileTag,"CALIB_CAL") == 0 ) &&
	      strstr(protech,"NOFRINGE") == NULL ) {
	  gendata = mat_gendata_load(cur_frame,CPL_TYPE_FLOAT);
	  if (gendata == NULL) {
	    cpl_msg_error(cpl_func,"no gendata loaded");
	    return -1;
	  }

	  //	    cpl_image_save(gendata->imgdata->list_frame[5]->list_subwin[0]->imgreg[0],"target.fits", CPL_BPP_IEEE_FLOAT,NULL, CPL_IO_CREATE);
	  //	    cpl_image_save(skyAvgFull[0],"sky.fits", CPL_BPP_IEEE_FLOAT,NULL, CPL_IO_CREATE);
	  // Loop on the number of subfiles
	  nbTargetPrev=0;
	  flagNewTarget=1;
	  bcd1Status = (char *) cpl_propertylist_get_string(gendata->keywords,"ESO INS BCD1 ID");
	  bcd2Status = (char *) cpl_propertylist_get_string(gendata->keywords,"ESO INS BCD2 ID");

	  // Compute the averaged sky per chopping cycle
	  // indSkyAvg indicates for each frame the index of the
	  // chopping cycle containing the previous averaged sky
	  indSkyAvg=cpl_calloc(gendata->imgdata->nbframe,sizeof(int));
	  sky=cpl_calloc(gendata->imgdet->nbregion*gendata->imgdata->nbframe,sizeof(cpl_image *));
				
	  nbTarget=0;
	  cpt=-1;
	  nbSky=0;
	  cpl_msg_info(cpl_func,"Step: Processing Sky frames");
	  // Loop on the number of frames
	  for(j=0;j<gendata->imgdata->nbframe;j++) {
	    // Sky frames
	    if ( !strcmp(gendata->imgdata->list_frame[j]->tartype,"S") ) {
	      // flagNewTarget=1 indicates that we start a new chopping cycle.
	      // So we fill a new cpl_image (sky) with current raw data
	      // and we start counting the number of sky for the current
	      // chopping cycle. And we increment the cpt counter which indicates
	      // the index of the averaged sky
	      if (flagNewTarget) {
		cpt++;
		flagNewTarget=0;
		for(k=0;k<gendata->imgdet->nbregion;k++) {
		  sky[cpt*gendata->imgdet->nbregion+k]=cpl_image_duplicate(gendata->imgdata->list_frame[j]->list_subwin[k]->imgreg[0]);
		}
		nbSky=1;
		// We are within a chopping cycle on sky. We integrate the frame and
		// we count the number of sky
	      } else {
		for(k=0;k<gendata->imgdet->nbregion;k++) {
		  cpl_image_add(sky[cpt*gendata->imgdet->nbregion+k],gendata->imgdata->list_frame[j]->list_subwin[k]->imgreg[0]);
		}
		nbSky++;
	      }
	      // Sky+Target frames
	      // we test if nbSky>0. if nbSky>0 means that a chopping skycycle on sky has been
	      // processed, so we normalized the results by the number of sky frames (nbSky). This
	      // is done for the first frame on a new Sky+Target cycle.
	      // For all frames, we fill the indSkyAvg vector with the current counter cpt
	    } else if ( !strcmp(gendata->imgdata->list_frame[j]->tartype,"T") ) {
	      nbTarget++;
	      flagNewTarget=1;
	      if (nbSky > 0) {
		for(k=0;k<gendata->imgdet->nbregion;k++) {
		  cpl_image_divide_scalar(sky[cpt*gendata->imgdet->nbregion+k],(double)nbSky);
		}
	      }
	      nbSky=0;
	    }
	    // if the current subfiles start with chopping cycle on Sky+Target, indSkyAvg[j]<0
	    indSkyAvg[j]=cpt;
	  }
	  if (nbTarget != 0)
	    {
	  
	      // if the exposure finish with a chopping cycle on sky, we normalized the results by
	      // the number of sky frames (nbSky)
	      if ( !strcmp(gendata->imgdata->list_frame[gendata->imgdata->nbframe-1]->tartype,"S") ) {
		for(k=0;k<gendata->imgdet->nbregion;k++) {
		  cpl_image_divide_scalar(sky[cpt*gendata->imgdet->nbregion+k],(double)nbSky);
		}
	      }
	      nbSkyAvg=cpt+1;
	      // Allocate the photBeams data structure
	      photBeams=cpl_malloc(sizeof(mat_photbeams));
	      if (photBeams == NULL) {
		cpl_msg_error(cpl_func,"could not allocate memory for photBeams");
		cpl_free(indSkyAvg);
		cpl_free(sky);
		return -1;
	      }
	      mat_init_photbeams(gendata,photBeams,0,&nbPhot,nbTarget);
	      if (strcmp(bcd1Status, "OUT") == 0 && strcmp(bcd2Status, "OUT") == 0)
		{
		  cpl_propertylist_append_string(photBeams->keywords,"ESO CFG BCD MODE","OUT-OUT");
		}
	      if (strcmp(bcd1Status, "IN") == 0 && strcmp(bcd2Status, "IN") == 0)
		{	  
		  cpl_propertylist_append_string(photBeams->keywords,"ESO CFG BCD MODE","IN-IN");
		}
	      if (strcmp(bcd1Status, "IN") == 0 && strcmp(bcd2Status, "OUT") == 0)
		{
		  cpl_propertylist_append_string(photBeams->keywords,"ESO CFG BCD MODE","IN-OUT");
		}
	      if (strcmp(bcd1Status, "OUT") == 0 && strcmp(bcd2Status, "IN") == 0)
		{
		  cpl_propertylist_append_string(photBeams->keywords,"ESO CFG BCD MODE","OUT-IN");
		}
	  

	      cpt=0;
	      // Loop on all frames of the subfiles
	      for(k=0;k<gendata->imgdet->nbregion;k++) {
		// detection of interferometric regions
		if ( gendata->imgdet->list_region[k]->correlation == 2 ) {
		  dimSpatialInterf = gendata->imgdet->list_region[k]->naxis[0];
		}
	      }

	      cpl_msg_info(cpl_func,"Step: Processing Target frames");
	      dimSpatial = gendata->imgdet->list_region[0]->naxis[0];
	      dimSpectral = gendata->imgdet->list_region[0]->naxis[1];
	      rowInterp=cpl_vector_new(dimSpatialInterf);
	      rowPhot=cpl_vector_new(dimSpatial);

	      for(j=0;j<gendata->imgdata->nbframe;j++) {
		if (j%100==0) {
		  cpl_msg_info(cpl_func,"Processing frame number %d",j);
		}
		// detection of Sky+Target frames
		if ( !strcmp(gendata->imgdata->list_frame[j]->tartype,"T") ) {
       
		  //			    printf("Processing frame %d\n",j+1);
		  // Loop on the number of regions
		  for(k=0;k<gendata->imgdet->nbregion;k++) {
		    // detection of photometric regions
		    if ( gendata->imgdet->list_region[k]->correlation == 1 ) {
		      photBeams->list_phot[cpt]=cpl_malloc(sizeof(mat_imgreg));
		      if (photBeams->list_phot[cpt] == NULL) {
			cpl_msg_error(cpl_func,"could not allocate memory for photBeams->list_phot[cpt]");
			cpl_free(indSkyAvg);
			cpl_free(sky);
			return -1;
		      }
		      photBeams->list_phot[cpt]->imgreg=cpl_calloc(1,sizeof(cpl_image *));
		      photBeams->list_phot[cpt]->numregion=k+1;
		      photBeams->list_phot[cpt]->numdetector=gendata->imgdet->list_region[k]->numdetector;
		      photBeams->list_phot[cpt]->imgreg[0]=cpl_image_new(dimSpatialInterf,dimSpectral,CPL_TYPE_FLOAT);
		      /* photBeams->list_phot[cpt]->imgreg[1]=cpl_image_new(dimSpatialInterf,dimSpectral,CPL_TYPE_FLOAT); */

		      //Substract Sky
		      //     1 Looking for the 2 nearest sky (before and after) with indSkyAvg
		      //     2 Avering the 2 nearest sky in SkyAvg
		      //     3 Removing averaged sky to imgPhot
		      //     4 Loading the kappa matrix coefficients
		      //     5 Applying the kappa matrix correction to the photometry with/without sky
		      //     6 storing the results in photBeams data structure
		      /* imgPhotSky=cpl_image_duplicate(gendata->imgdata->list_frame[j]->list_subwin[k]->imgreg[0]); */
		      imgPhot=cpl_image_duplicate(gendata->imgdata->list_frame[j]->list_subwin[k]->imgreg[0]);
		      // handling of the limit case
		      if (indSkyAvg[j] < 0) {
			// No chopping cycle on Sky before. We use the sky after only.
			skyAvg=cpl_image_duplicate(sky[(indSkyAvg[j]+1)*gendata->imgdet->nbregion+k]);
		      } else if (indSkyAvg[j] >= nbSkyAvg-1) {
			// No chopping cycle on Sky after. We use the sky before only.
			skyAvg=cpl_image_duplicate(sky[(nbSkyAvg-1)*gendata->imgdet->nbregion+k]);
			// handling of the standard case
		      } else {
			skyAvg=cpl_image_duplicate(sky[(indSkyAvg[j])*gendata->imgdet->nbregion+k]);
			cpl_image_add(skyAvg,sky[(indSkyAvg[j]+1)*gendata->imgdet->nbregion+k]);
			cpl_image_divide_scalar(skyAvg,(double)2.0);
		      }
		      /* if ( cpl_parameter_get_bool(useAvgSky) ) { */
		      /* 	if (nbSkyAvgFull != 0 ) { */
		      /* 		cpl_image_subtract(imgPhot,skyAvgFull[k]); */
		      /* 	} else { */
		      /* 		if (j==0 && k==0) { */
		      /* 			cpl_msg_info(cpl_func,"No Sky found in the exposure"); */
		      /* 		} */
		      /* 	} */
		      /* } else { */
		      if (skyAvg != NULL) {
			cpl_image_subtract(imgPhot,skyAvg);
		      } else {
			if (j==0 && k==0) {
			  cpl_msg_info(cpl_func,"No Sky found in the exposure 1");
			}
		      }
		      /* } */
		      kappaParam=cpl_vector_new(3);
		      // Load the corresponding Kappa Coefficients
		      ixPhotChan=k-(k/5)*5;
		      if (ixPhotChan > 2) {
			ixPhotChan--;
		      }

		      for (l=0;l<dimSpectral;l++) {
			iJumpInterf=0;
			/* if ( cpl_parameter_get_bool(useKappaMatrix) && kappaMat != NULL) { */
			for(i=0;i<kappaMat->nbbeam/4;i++) 
			  {
			    iJumpInterf++;
			    rowDet=l+gendata->imgdet->list_region[k]->corner[1];
			    if ( (rowDet >= kappaMat->imgdet->list_region[ixPhotChan+i*4+iJumpInterf]->corner[1])
				 && (rowDet < kappaMat->imgdet->list_region[ixPhotChan+i*4+iJumpInterf]->corner[1]+
				     kappaMat->imgdet->list_region[ixPhotChan+i*4+iJumpInterf]->naxis[1]) )
			      {
				deltapixel=gendata->imgdet->list_region[k]->corner[1]-
				  kappaMat->imgdet->list_region[ixPhotChan+i*4+iJumpInterf]->corner[1];
				cpl_vector_set(kappaParam,0,cpl_matrix_get(kappaMat->list_kappacoef[ixPhotChan+i*4]->coef,0,l+deltapixel));
				cpl_vector_set(kappaParam,1,cpl_matrix_get(kappaMat->list_kappacoef[ixPhotChan+i*4]->coef,1,l+deltapixel));
				cpl_vector_set(kappaParam,2,cpl_matrix_get(kappaMat->list_kappacoef[ixPhotChan+i*4]->coef,2,l+deltapixel));
			      }
			  }
			/* } else { */
			/* 	cpl_vector_set(kappaParam,0,2.0); */
			/* 	cpl_vector_set(kappaParam,1,6.0); */
			/* 	cpl_vector_set(kappaParam,2,0.0); */
			/* } */
			for (i=0;i<dimSpatial;i++) {
			  cpl_vector_set(rowPhot,i,cpl_image_get(imgPhot,i+1,l+1,&flag_rejected));
			  //cpl_vector_set(rowPhot,i,cpl_image_get(targetAvgFull[k],i+1,l+1,&flag_rejected));
			}
			mat_apply_kappamatrix(rowPhot,kappaParam,rowInterp);

			for (i=0; i<dimSpatialInterf; i++) {
			  cpl_image_set(photBeams->list_phot[cpt]->imgreg[0],i+1,l+1,cpl_vector_get(rowInterp,i));
			}
		      }
		      // Free Memory
		      if (kappaParam != NULL) cpl_vector_delete(kappaParam);
		      if (skyAvg != NULL) cpl_image_delete(skyAvg);
		      if (imgPhot != NULL) cpl_image_delete(imgPhot);
		      cpt++;
		    }
		  }
		}
	      }
	      if (rowPhot != NULL) cpl_vector_delete(rowPhot);
	      if (rowInterp != NULL) cpl_vector_delete(rowInterp);



	      if (kernel > 0)
		{
		  // Apply Hampel filter to all pixels of photbeams

		  cpl_image *shape=NULL;
		  float flux=0.;
		
		  for(k=0;k<photBeams->imgdet->nbregion;k++)
		    {
		      shape=cpl_image_new(dimSpatialInterf,dimSpectral,CPL_TYPE_FLOAT);

		      /* Start parallel loop */
		      cpl_msg_info(cpl_func, "Starting parallel loops");
		      gettimeofday(&tv1, NULL);
		      HDRL_OMP(omp parallel for)
			for (cpl_size loopl = 0; loopl < dimSpectral ; loopl++)
			  {
			    for (cpl_size loopi = 0; loopi < dimSpatialInterf; loopi++)
			      {
				cpl_vector *foo1 = NULL;
				cpl_vector *foo2 = NULL;
				foo1 = cpl_vector_new(photBeams->nbphotxframe/photBeams->imgdet->nbregion);
				for(cpl_size loopj = 0; loopj < photBeams->nbphotxframe/photBeams->imgdet->nbregion; loopj++)
				  {
				    int loopflag_rejected = 0;
				    //printf("%d %d %d %d %d\n",photBeams->nbphotxframe,photBeams->imgdet->nbregion,j,k,photBeams->imgdet->nbregion*j+k);
				    cpl_vector_set(foo1, loopj, cpl_image_get(photBeams->list_phot[photBeams->imgdet->nbregion*loopj+k]->imgreg[0], loopi + 1, loopl + 1, &loopflag_rejected));
				  }

				foo2 = mat_vector_hampel_filter(foo1, kernel, 2);
				cpl_vector_delete(foo1);
				for( cpl_size loopj = 0; loopj < photBeams->nbphotxframe/photBeams->imgdet->nbregion; loopj++)
				  {
				    cpl_image_set(photBeams->list_phot[photBeams->imgdet->nbregion*loopj+k]->imgreg[0], loopi + 1, loopl + 1, cpl_vector_get(foo2, loopj));
				  }
				cpl_vector_delete(foo2);
			      }
			  }
		      gettimeofday(&tv2, NULL);
		      cpl_msg_info(cpl_func, "Wall time for parallel loop = %f seconds\n",
				   (double) (tv2.tv_usec - tv1.tv_usec) / 1000000 +
				   (double) (tv2.tv_sec - tv1.tv_sec));
		      /* Stop parallel loop */

		      for(j=0;j<photBeams->nbphotxframe/photBeams->imgdet->nbregion;j++)
			{
			  cpl_image_add(shape,photBeams->list_phot[photBeams->imgdet->nbregion*j+k]->imgreg[0]);
			}
		      flux=cpl_image_get_flux(shape);
		      cpl_image_divide_scalar(shape,flux);
		      /* cpl_image_save(shape,"shape.fits", CPL_BPP_IEEE_FLOAT,NULL, CPL_IO_CREATE); */
		      /* exit(1); */
		      for(j=0;j<photBeams->nbphotxframe/photBeams->imgdet->nbregion;j++)
			{
			  flux=cpl_image_get_flux(photBeams->list_phot[photBeams->imgdet->nbregion*j+k]->imgreg[0]);
			  cpl_image_copy(photBeams->list_phot[photBeams->imgdet->nbregion*j+k]->imgreg[0],shape,1,1);
			  cpl_image_multiply_scalar(photBeams->list_phot[photBeams->imgdet->nbregion*j+k]->imgreg[0],flux);
			}
		      cpl_image_delete(shape);
		    }

		}


	      // Apply Spectral Binning
	      if (nbBin > 1 )
		{
		  cpt=0;
		  for(iReg=0;iReg<nbPhot;iReg++)
		    {
		      nbFreq=photBeams->imgdet->list_region[iReg]->naxis[0];
		      nbWlen=photBeams->imgdet->list_region[iReg]->naxis[1];
		      for(iFrame=0; iFrame<photBeams->nbphotxframe/nbPhot; iFrame++) {
			{
			  for(iFreq=0;iFreq<nbFreq;iFreq++)
			    {
			      vec2smooth=cpl_vector_new(nbWlen);
			      for(iWlen=0;iWlen<nbWlen;iWlen++)
				{
				  cpl_vector_set(vec2smooth,iWlen,cpl_image_get(photBeams->list_phot[cpt]->imgreg[0],iFreq+1,iWlen+1,&pis));
				}
			      for(iWlen=0;iWlen<nbWlen;iWlen++)
				{
				  if (iWlen < (nbBin-1)/2)
				    {
				      val=0.;
				      for(iPix=0;iPix<=2*iWlen;iPix++)
					{
					  val+=cpl_vector_get(vec2smooth,iPix);
					}
				      val/=(2*iWlen+1);
				      cpl_image_set(photBeams->list_phot[cpt]->imgreg[0],iFreq+1,iWlen+1,val);
				    }
				  else if (iWlen >= nbWlen-(nbBin-1)/2)
				    {
				      val=0.;
				      for(iPix=iWlen-(nbWlen-1-iWlen);iPix<=iWlen+(nbWlen-1-iWlen);iPix++)
					{
					  val+=cpl_vector_get(vec2smooth,iPix);
					}
				      val/=(2*(nbWlen-1-iWlen)+1);
				      cpl_image_set(photBeams->list_phot[cpt]->imgreg[0],iFreq+1,iWlen+1,val);
				    }
				  else
				    {
				      val=0.;
				      for(iPix=iWlen-(nbBin-1)/2;iPix<=iWlen+(nbBin-1)/2;iPix++)
					{
					  val+=cpl_vector_get(vec2smooth,iPix);
					}
				      val/=nbBin;
				      cpl_image_set(photBeams->list_phot[cpt]->imgreg[0],iFreq+1,iWlen+1,val);
				    }
				}
			      cpl_vector_delete(vec2smooth);
			    }
			  cpt++;
			}
		      }
		    }
		}


	      // Fill usedframes depending on the way mat_ext_beams is called (atomic or master recipes)
	      if (cpl_frameset_find(frameset,"CALIB_RAW") != NULL ||
		  cpl_frameset_find(frameset,"TARGET_RAW") != NULL ||
		  cpl_frameset_find(frameset,"CALIB_SRC_RAW") != NULL)
		{
		  usedframes=cpl_frameset_new();
		  it1= cpl_frameset_iterator_new(frameset);
		  do
		    {
		      cur_frame1 = cpl_frameset_iterator_get(it1);
		      if (cur_frame1 != NULL)
			{
			  pszFileName1 = (char*)cpl_frame_get_filename( cur_frame1 );
			  pszFileTag1 = (char *)cpl_frame_get_tag( cur_frame1 );
			  if (strcmp(pszFileTag1, "CALIB_RAW") == 0 || strcmp(pszFileTag1, "TARGET_RAW") == 0 || strcmp(pszFileTag1, "CALIB_SRC_RAW") == 0)
			    {
			      plist1=cpl_propertylist_load(pszFileName1,0);
			      if (strcmp(cpl_propertylist_get_string(photBeams->keywords,"ESO CFG BCD MODE"),"OUT-OUT") == 0)
				{
				  if (strcmp(cpl_propertylist_get_string(plist1,"ESO INS BCD1 NAME"),"OUT") == 0 &&
				      strcmp(cpl_propertylist_get_string(plist1,"ESO INS BCD2 NAME"),"OUT") == 0 &&
				      cpl_propertylist_get_int(plist1,"ESO TPL EXPNO") == cpl_propertylist_get_int(photBeams->keywords,"ESO TPL EXPNO"))
				    {
				      sel_frame=cpl_frame_duplicate(cur_frame1);
				      cpl_frameset_insert(usedframes,sel_frame);
				    }
				}
			      else if(strcmp(cpl_propertylist_get_string(photBeams->keywords,"ESO CFG BCD MODE"),"IN-OUT") == 0)
				{
				  if (strcmp(cpl_propertylist_get_string(plist1,"ESO INS BCD1 NAME"),"IN") == 0 &&
				      strcmp(cpl_propertylist_get_string(plist1,"ESO INS BCD2 NAME"),"OUT") == 0 &&
				      cpl_propertylist_get_int(plist1,"ESO TPL EXPNO") == cpl_propertylist_get_int(photBeams->keywords,"ESO TPL EXPNO"))
				    {
				      sel_frame=cpl_frame_duplicate(cur_frame1);
				      cpl_frameset_insert(usedframes,sel_frame);
				    }
				}
			      else if(strcmp(cpl_propertylist_get_string(photBeams->keywords,"ESO CFG BCD MODE"),"OUT-IN") == 0)
				{
				  if (strcmp(cpl_propertylist_get_string(plist1,"ESO INS BCD1 NAME"),"OUT") == 0 &&
				      strcmp(cpl_propertylist_get_string(plist1,"ESO INS BCD2 NAME"),"IN") == 0 &&
				      cpl_propertylist_get_int(plist1,"ESO TPL EXPNO") == cpl_propertylist_get_int(photBeams->keywords,"ESO TPL EXPNO"))
				    {
				      sel_frame=cpl_frame_duplicate(cur_frame1);
				      cpl_frameset_insert(usedframes,sel_frame);
				    }
				}
			      else if(strcmp(cpl_propertylist_get_string(photBeams->keywords,"ESO CFG BCD MODE"),"IN-IN") == 0)
				{
				  if (strcmp(cpl_propertylist_get_string(plist1,"ESO INS BCD1 NAME"),"IN") == 0 &&
				      strcmp(cpl_propertylist_get_string(plist1,"ESO INS BCD2 NAME"),"IN") == 0 &&
				      cpl_propertylist_get_int(plist1,"ESO TPL EXPNO") == cpl_propertylist_get_int(photBeams->keywords,"ESO TPL EXPNO"))
				    {
				      sel_frame=cpl_frame_duplicate(cur_frame1);
				      cpl_frameset_insert(usedframes,sel_frame);
				    }
				}
			      cpl_propertylist_delete(plist1);
			    }
			}
		    } while (cpl_frameset_iterator_advance(it1, 1) != CPL_ERROR_ACCESS_OUT_OF_RANGE);
		  cpl_frameset_iterator_delete(it1);
		  it1= cpl_frameset_iterator_new(frameset);
		  do
		    {
		      cur_frame1 = cpl_frameset_iterator_get(it1);
		      if (cur_frame1 != NULL)
			{
			  pszFileName1 = (char*)cpl_frame_get_filename( cur_frame1 );
			  pszFileTag1 = (char *)cpl_frame_get_tag( cur_frame1 );
			  if (strcmp(pszFileTag1, "CALIB_RAW") == 0 || strcmp(pszFileTag1, "TARGET_RAW") == 0 || strcmp(pszFileTag1, "CALIB_SRC_RAW") == 0)
			    {
			      plist1=cpl_propertylist_load(pszFileName1,0);
			      if (strcmp(cpl_propertylist_get_string(photBeams->keywords,"ESO CFG BCD MODE"),"OUT-OUT") == 0)
				{
				  if (strcmp(cpl_propertylist_get_string(plist1,"ESO INS BCD1 NAME"),"OUT") != 0 ||
				      strcmp(cpl_propertylist_get_string(plist1,"ESO INS BCD2 NAME"),"OUT") != 0 ||
				      cpl_propertylist_get_int(plist1,"ESO TPL EXPNO") != cpl_propertylist_get_int(photBeams->keywords,"ESO TPL EXPNO"))
				    {
				      sel_frame=cpl_frame_duplicate(cur_frame1);
				      cpl_frameset_insert(usedframes,sel_frame);
				    }
				}
			      else if(strcmp(cpl_propertylist_get_string(photBeams->keywords,"ESO CFG BCD MODE"),"IN-OUT") == 0)
				{
				  if (strcmp(cpl_propertylist_get_string(plist1,"ESO INS BCD1 NAME"),"IN") != 0 ||
				      strcmp(cpl_propertylist_get_string(plist1,"ESO INS BCD2 NAME"),"OUT") != 0 ||
				      cpl_propertylist_get_int(plist1,"ESO TPL EXPNO") != cpl_propertylist_get_int(photBeams->keywords,"ESO TPL EXPNO"))
				    {
				      sel_frame=cpl_frame_duplicate(cur_frame1);
				      cpl_frameset_insert(usedframes,sel_frame);
				    }
				}
			      else if(strcmp(cpl_propertylist_get_string(photBeams->keywords,"ESO CFG BCD MODE"),"OUT-IN") == 0)
				{
				  if (strcmp(cpl_propertylist_get_string(plist1,"ESO INS BCD1 NAME"),"OUT") != 0 ||
				      strcmp(cpl_propertylist_get_string(plist1,"ESO INS BCD2 NAME"),"IN") != 0 ||
				      cpl_propertylist_get_int(plist1,"ESO TPL EXPNO") != cpl_propertylist_get_int(photBeams->keywords,"ESO TPL EXPNO"))
				    {
				      sel_frame=cpl_frame_duplicate(cur_frame1);
				      cpl_frameset_insert(usedframes,sel_frame);
				    }
				}
			      else if(strcmp(cpl_propertylist_get_string(photBeams->keywords,"ESO CFG BCD MODE"),"IN-IN") == 0)
				{
				  if (strcmp(cpl_propertylist_get_string(plist1,"ESO INS BCD1 NAME"),"IN") != 0 ||
				      strcmp(cpl_propertylist_get_string(plist1,"ESO INS BCD2 NAME"),"IN") != 0 ||
				      cpl_propertylist_get_int(plist1,"ESO TPL EXPNO") != cpl_propertylist_get_int(photBeams->keywords,"ESO TPL EXPNO"))
				    {
				      sel_frame=cpl_frame_duplicate(cur_frame1);
				      cpl_frameset_insert(usedframes,sel_frame);
				    }
				}
			      cpl_propertylist_delete(plist1);
			    }
			  else if (strcmp(pszFileTag1, "BADPIX") == 0 || strcmp(pszFileTag1, "NONLINEARITY") == 0 ||
				   strcmp(pszFileTag1, "SHIFT_MAP") == 0 || strcmp(pszFileTag1, "KAPPA_MATRIX") == 0 ||
				   strcmp(pszFileTag1, "OBS_FLATFIELD") == 0 || strcmp(pszFileTag1, "JSDC_CAT") == 0)
			    {
			      sel_frame=cpl_frame_duplicate(cur_frame1);
			      cpl_frameset_insert(usedframes,sel_frame);
			    }
		       
			}
		    } while (cpl_frameset_iterator_advance(it1, 1) != CPL_ERROR_ACCESS_OUT_OF_RANGE);
		  cpl_frameset_iterator_delete(it1);
		  
		  it1= cpl_frameset_iterator_new(frameset);
		  do
		    {
		      cur_frame1 = cpl_frameset_iterator_get(it1);
		      if (cur_frame1 != NULL)
			{
			  pszFileName1 = (char*)cpl_frame_get_filename( cur_frame1 );
			  pszFileTag1 = (char *)cpl_frame_get_tag( cur_frame1 );
			  if (strcmp(pszFileTag1, "SKY_RAW") == 0 || strcmp(pszFileTag1, "HOT_DARK") == 0 )
			    {
			      sel_frame=cpl_frame_duplicate(cur_frame1);
			      cpl_frameset_insert(usedframes,sel_frame);
			    }
			}
		    } while (cpl_frameset_iterator_advance(it1, 1) != CPL_ERROR_ACCESS_OUT_OF_RANGE);
		  cpl_frameset_iterator_delete(it1);
		}
	      else
		{
		  sel_frame=cpl_frame_duplicate(cur_frame);
		  cpl_frame_set_group(sel_frame,CPL_FRAME_GROUP_RAW);
		  usedframes=cpl_frameset_new();
		  cpl_frameset_insert(usedframes,sel_frame);
		}
	      // Save in a Fits file
	      output=cpl_sprintf("PHOT_BEAMS_%04d.fits",indexFile);
	      if (strcmp(pszFileTag,"TARGET_CAL") == 0) {
		cpl_msg_info(cpl_func,"Saving File with TAG : TARGET_CAL");
		mat_photbeams_save(photBeams,photBeams->keywords,frameset,usedframes,parlist,recname,"PHOT_BEAMS","IMAGE","T",output);
	      }
	      if (strcmp(pszFileTag,"CALIB_CAL") == 0)  {
		cpl_msg_info(cpl_func,"Saving File with TAG : CALIB_CAL");
		mat_photbeams_save(photBeams,photBeams->keywords,frameset,usedframes,parlist,recname,"PHOT_BEAMS","IMAGE","F",output);
	      }
	      indexFile++;
	      nbTargetPrev+=nbTarget;
	      cpl_frameset_delete(usedframes);
	      cpl_msg_info(cpl_func,"Adding IMAGING_DETECTOR");
	      mat_imagingdetector_save(photBeams->imgdet,photBeams->keywords,output);
	      // Keep he last averaged sky in case of absence of sky in the next part 
	      // of the fits file
	      /* if ( cpl_parameter_get_bool(useAvgSky) || cpl_parameter_get_bool(useAvgShape) ) { */
	      /*   if (nbSkyAvgFull != 0 ) { */
	      /*     for(k=0;k<gendata->imgdet->nbregion;k++) { */
	      /*       cpl_image_delete(skyAvgFull[k]); */
	      /*     } */
	      /*     if (skyAvgFull != NULL) cpl_free(skyAvgFull); */
	      /*   } */
	      /* } */
	      /* if ( cpl_parameter_get_bool(useAvgShape) ) { */
	      /*   if (nbTargetAvgFull != 0 ) { */
	      /*     for(k=0;k<gendata->imgdet->nbregion;k++) { */
	      /*       cpl_image_delete(targetAvgFull[k]); */
	      /*     } */
	      /*     if (targetAvgFull != NULL) cpl_free(targetAvgFull); */
	      /*   } */
	      /* } */
	      // Free Memory
	      for(j=0;j<nbSkyAvg;j++) {
		for(k=0;k<gendata->imgdet->nbregion;k++) {
		  cpl_image_delete(sky[j*gendata->imgdet->nbregion+k]);
		}
	      }
	      if (sky != NULL) cpl_free(sky);
	      if (indSkyAvg != NULL) cpl_free(indSkyAvg);
	      if (gendata != NULL) mat_gendata_delete(gendata);
	      mat_photbeams_free(photBeams);
		
	      cpl_free(output);
	    }
	  else
	    {
	      cpl_msg_warning(cpl_func,"No image found");
	      if (sky != NULL) cpl_free(sky);
	      if (indSkyAvg != NULL) cpl_free(indSkyAvg);
	      if (gendata != NULL) mat_gendata_delete(gendata);
	    }
		
	}
	cpl_propertylist_delete(plist);
      }
    } while (cpl_frameset_iterator_advance(it, 1) != CPL_ERROR_ACCESS_OUT_OF_RANGE);
    cpl_frameset_iterator_delete(it);
    if (!cpl_errorstate_is_equal(prestate)) {
      cpl_errorstate_set(prestate);
    }
  }
  cpl_free(sliderPhot);
  if (kappaMat != NULL) {
    mat_kappamatrix_free(kappaMat);
  }
  return 0;
}
