/* $Id: mat_fit_gauss.c,v0.5 2014-06-15 12:56:21 pberio Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: pberio $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_fit_gauss.c $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include "mat_fit_gauss.h"
#include "mat_error.h"
/*-----------------------------------------------------------------------------
                                   Define
 -----------------------------------------------------------------------------*/
#define UNUSED(x) (void)(x)


/*-----------------------------------------------------------------------------
                                   Functions prototypes
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief Fits a gaussian curve
  @param N         number of rows
  @param vector    vector of each maximum indexes
  @param params    parameters list
  @param pos_err   error on the estimate dposition of the gaussian
  @return double position of the gausian
 */
/*----------------------------------------------------------------------------*/
double mat_fit_gauss(int N,gsl_vector *vector, double *params, 
		     double *pos_err) {
  const gsl_multifit_fdfsolver_type *T;
  gsl_multifit_fdfsolver *s;
  int status;
  unsigned int i, iter = 0;
  const size_t n = N;
  const size_t p = 3;
  double res;
  double A;
  double chi;
  double dof;
  double c; 
#if defined GSL_MAJOR_VERSION && GSL_MAJOR_VERSION >= 2
  gsl_matrix *J;
#endif

  gsl_matrix *covar = gsl_matrix_alloc (p, p);
  double y[n], sigma[n];
  struct data d = { n, y, sigma};
  gsl_multifit_function_fdf f;
  gsl_vector_view x = gsl_vector_view_array (params, p);
  const gsl_rng_type * type;
  gsl_rng * r;

// Check input parameters
  mat_assert_value((vector!=NULL),CPL_ERROR_NULL_INPUT, -1,
		   "no vector argument given");
  mat_assert_value((params!=NULL),CPL_ERROR_NULL_INPUT, -1,
		   "no params argument given");


     
  res = 0.0;

  gsl_rng_env_setup();
     
  type = gsl_rng_default;
  r = gsl_rng_alloc (type);
     
  f.f = &expb_f;
  f.df = &expb_df;
  f.fdf = &expb_fdf;
  f.n = n;
  f.p = p;
  f.params = &d;
     
  /* This is the data to be fitted */
  for (i = 0; i < n; i++)
    {
      A= gsl_vector_get(vector, i);
      y[i] = A;
      sigma[i] = 0.1;
      //printf ("data: %u %lf %lf\n", i, y[i], sigma[i]);
    };

  T = gsl_multifit_fdfsolver_lmsder;
  s = gsl_multifit_fdfsolver_alloc (T, n, p);
  gsl_multifit_fdfsolver_set (s, &f, &x.vector);
     
  do {
    iter++;
    status = gsl_multifit_fdfsolver_iterate (s);
     
    if (status)
      break;
     
    status = gsl_multifit_test_delta (s->dx, s->x,1e-4, 1e-4);
  }
  while (status == GSL_CONTINUE && iter < 500);

#if defined GSL_MAJOR_VERSION && GSL_MAJOR_VERSION >= 2
  J = gsl_matrix_alloc(n, p);
  gsl_multifit_fdfsolver_jac(s, J);
  gsl_multifit_covar (J, 0.0, covar);
  gsl_matrix_free (J);
#else
  gsl_multifit_covar (s->J, 0.0, covar);
#endif
     
#define FIT(i) gsl_vector_get(s->x, i)
#define ERR(i) sqrt(gsl_matrix_get(covar,i,i))
   
  
  chi = gsl_blas_dnrm2(s->f);
  dof = n - p;
  c = GSL_MAX_DBL(1, chi / sqrt(dof)); 
     
    //printf("chisq/dof = %g\n",  pow(chi, 2.0) / dof);
     
    //printf ("A     = %.5f +/- %.5f\n", FIT(0), c*ERR(0));
    //printf ("x     = %.5f +/- %.5f\n", FIT(1), c*ERR(1));
    //printf ("c      = %.5f +/- %.5f\n", FIT(2), c*ERR(2));
  
     
  //printf ("status = %s\n", gsl_strerror (status));
  
  res=FIT(1);
  *pos_err=c*ERR(1);
 
  gsl_multifit_fdfsolver_free (s);
  gsl_matrix_free (covar);
  gsl_rng_free (r);

  return res;
  
}

int expb_f (const gsl_vector * x, void *data, 
          gsl_vector * f)
{
  size_t n = ((struct data *)data)->n;
  double *y = ((struct data *)data)->y;
  double *sigma = ((struct data *) data)->sigma;
     
  double A = gsl_vector_get (x, 0);
  double x1 = gsl_vector_get (x, 1);
  double c = gsl_vector_get (x, 2);
     
  size_t i;
     

  for (i = 0; i < n; i++)
  {
    /* Model Yi = A * exp(-lambda * i) + b */
    double t = i; 
    double Yi = A * exp (pow(t-x1, 2)/(-2*pow(c,2)));
    gsl_vector_set (f, i, (Yi - y[i])/sigma[i]);
  }
     
  return GSL_SUCCESS;
}
     
int expb_df (const gsl_vector * x, void *data, 
           gsl_matrix * J)
{
  size_t n = ((struct data *)data)->n;
  double *sigma = ((struct data *) data)->sigma;
     
  double A = gsl_vector_get (x, 0);
  double x1 = gsl_vector_get (x, 1);
  double c = gsl_vector_get (x, 2);
     
  size_t i;
     
  for (i = 0; i < n; i++)
  {
    /* Jacobian matrix J(i,j) = dfi / dxj, */
    /* where fi = (Yi - yi)/sigma[i],      */
    /*       Yi = A * exp(-lambda * i) + b  */
    /* and the xj are the parameters (A,lambda,b) */
    double t = i;
    double s = sigma[i];
    double e = exp (pow(t-x1, 2)/(-2*pow(c,2)));
    gsl_matrix_set (J, i, 0, e/s); 
    gsl_matrix_set (J, i, 1, (A*(t-x1)/pow(c,2))*e/s);
    gsl_matrix_set (J, i, 2, (A*(pow(t-x1, 2)/pow(c,3))*e/s));
  }
  return GSL_SUCCESS;
}
    
int expb_fdf (const gsl_vector * x, void *data,
            gsl_vector * f, gsl_matrix * J)
{
  expb_f (x, data, f);
  expb_df (x, data, J);
     
  return GSL_SUCCESS;
}

/*----------------------------------------------------------------------------*/
/**
  @brief Fits a gaussian curve + a linear trend
  @param N         number of rows
  @param vector    vector of each maximum indexes
  @param params    parameters list
  @param pos_err   error on the estimate dposition of the gaussian
  @return double position of the gausian
 */
/*----------------------------------------------------------------------------*/
int mat_fit_gauss_linear(int N,gsl_vector *vector, double *params) {
  const gsl_multifit_fdfsolver_type *T;
  gsl_multifit_fdfsolver *s;
  int status;
  unsigned int i, iter = 0;
  const size_t n = N;
  const size_t p = 5;
  double A;
  /* double chi; */
  /* double dof; */
  /* double c;  */
#if defined GSL_MAJOR_VERSION && GSL_MAJOR_VERSION >= 2
  gsl_matrix *J;
#endif

  gsl_matrix *covar = gsl_matrix_alloc (p, p);
  double y[n];
  double sigma[n];
  struct data d = { n, y, sigma};
  gsl_multifit_function_fdf f;
  gsl_vector_view x = gsl_vector_view_array (params, p);
  const gsl_rng_type * type;
  gsl_rng * r;

// Check input parameters
  mat_assert_value((vector!=NULL),CPL_ERROR_NULL_INPUT, -1,
		   "no vector argument given");
  mat_assert_value((params!=NULL),CPL_ERROR_NULL_INPUT, -1,
		   "no params argument given");


     

  gsl_rng_env_setup();
     
  type = gsl_rng_default;
  r = gsl_rng_alloc (type);
     
  f.f = &expb_f_linear;
  f.df = &expb_df_linear;
  f.fdf = &expb_fdf_linear;
  f.n = n;
  f.p = p;
  f.params = &d;
     
  /* This is the data to be fitted */
  for (i = 0; i < n; i++)
    {
      A= gsl_vector_get(vector, i);
      y[i] = A;
      //printf ("data: %u %lf %lf\n", i, y[i], sigma[i]);
    };

  T = gsl_multifit_fdfsolver_lmsder;
  s = gsl_multifit_fdfsolver_alloc (T, n, p);
  gsl_multifit_fdfsolver_set (s, &f, &x.vector);
     
  do {
    iter++;
    status = gsl_multifit_fdfsolver_iterate (s);
    if (status)
      break;
     
    status = gsl_multifit_test_delta (s->dx, s->x,1e-4, 1e-4);
  }
  while (status == GSL_CONTINUE && iter < 500);
  
#if defined GSL_MAJOR_VERSION && GSL_MAJOR_VERSION >= 2
  J = gsl_matrix_alloc(n, p);
  gsl_multifit_fdfsolver_jac(s, J);
  gsl_multifit_covar (J, 0.0, covar);
  gsl_matrix_free (J);
#else
  gsl_multifit_covar (s->J, 0.0, covar);
#endif
     
#define FIT(i) gsl_vector_get(s->x, i)
#define ERR(i) sqrt(gsl_matrix_get(covar,i,i))
   
  
  /* chi = gsl_blas_dnrm2(s->f); */
  /* dof = n - p; */
  /* c = GSL_MAX_DBL(1, chi / sqrt(dof));  */
     
    //printf("chisq/dof = %g\n",  pow(chi, 2.0) / dof);
     
    //printf ("A     = %.5f +/- %.5f\n", FIT(0), c*ERR(0));
    //printf ("x     = %.5f +/- %.5f\n", FIT(1), c*ERR(1));
    //printf ("c      = %.5f +/- %.5f\n", FIT(2), c*ERR(2));
  
     
  //printf ("status = %s\n", gsl_strerror (status));

  for (i = 0; i < p; i++) {
    params[i]=FIT(i);
  }
  
 
  gsl_multifit_fdfsolver_free (s);
  gsl_matrix_free (covar);
  gsl_rng_free (r);

  return GSL_SUCCESS;
  
}
int expb_f_linear (const gsl_vector * x, void *data, gsl_vector * f)
{
  size_t n = ((struct data *)data)->n;
  double *y = ((struct data *)data)->y;
     
  double A = gsl_vector_get (x, 0);
  double x1 = gsl_vector_get (x, 1);
  double c = gsl_vector_get (x, 2);
  double ca = gsl_vector_get (x, 3);
  double cb = gsl_vector_get (x, 4);
     
  size_t i;
     

  for (i = 0; i < n; i++)
  {
    /* Model Yi = A * exp(-lambda * i) + b */
    double t = i/(1.*n); 
    double Yi = A * exp (-1.*c*pow(t-x1, 2))+ca+cb*t;
    gsl_vector_set (f, i, (Yi - y[i]));
  }
     
  return GSL_SUCCESS;
}

int expb_df_linear (const gsl_vector * x, void *data,gsl_matrix * J)
{
  size_t n = ((struct data *)data)->n;
     
  double A = gsl_vector_get (x, 0);
  double x1 = gsl_vector_get (x, 1);
  double c = gsl_vector_get (x, 2);
  /* double ca = gsl_vector_get (x, 3); */
  /* double cb = gsl_vector_get (x, 4); */
     
  size_t i;
     
  for (i = 0; i < n; i++)
  {
    /* Jacobian matrix J(i,j) = dfi / dxj, */
    /* where fi = (Yi - yi)/sigma[i],      */
    /*       Yi = A * exp(-lambda * i) + b  */
    /* and the xj are the parameters (A,lambda,b) */
    double t = i/(1.*n);
    double e = exp (-1.*c*pow(t-x1, 2)  );
    gsl_matrix_set (J, i, 0, e); 
    gsl_matrix_set (J, i, 1, 2.*c*(t-x1)*A*e );
    gsl_matrix_set (J, i, 2, -1.*A*pow(t-x1, 2)*e);
    gsl_matrix_set (J, i, 3, 1.);
    gsl_matrix_set (J, i, 4, t);
  }
  return GSL_SUCCESS;
}
int expb_fdf_linear (const gsl_vector * x, void *data,
            gsl_vector * f, gsl_matrix * J)
{
  expb_f_linear (x, data, f);
  expb_df_linear (x, data, J);
     
  return GSL_SUCCESS;
}




/*----------------------------------------------------------------------------*/
/**
  @brief Fits a gaussian curve + a linear trend
  @param N         number of rows
  @param vector    vector of each maximum indexes
  @param params    parameters list
  @param pos_err   error on the estimate dposition of the gaussian
  @return double position of the gausian
 */
/*----------------------------------------------------------------------------*/
int mat_fit_gauss_linear_red(int N,gsl_vector *vector, double *params, double cen0, double wid0) {
  const gsl_multifit_fdfsolver_type *T;
  gsl_multifit_fdfsolver *s;
  int status;
  unsigned int i, iter = 0;
  const size_t n = N;
  const size_t p = 3;
  double cen = cen0;
  double wid = wid0;
  double A;
  /* double chi; */
  /* double dof; */
  /* double c;  */
#if defined GSL_MAJOR_VERSION && GSL_MAJOR_VERSION >= 2
  gsl_matrix *J;
#endif

  gsl_matrix *covar = gsl_matrix_alloc (p, p);
  double y[n];
  struct data_red d = { n, y, cen, wid};
  gsl_multifit_function_fdf f;
  gsl_vector_view x = gsl_vector_view_array (params, p);
  const gsl_rng_type * type;
  gsl_rng * r;

// Check input parameters
  mat_assert_value((vector!=NULL),CPL_ERROR_NULL_INPUT, -1,
		   "no vector argument given");
  mat_assert_value((params!=NULL),CPL_ERROR_NULL_INPUT, -1,
		   "no params argument given");


     

  gsl_rng_env_setup();
     
  type = gsl_rng_default;
  r = gsl_rng_alloc (type);
     
  f.f = &expb_f_linear_red;
  f.df = &expb_df_linear_red;
  f.fdf = &expb_fdf_linear_red;
  f.n = n;
  f.p = p;
  f.params = &d;
     
  /* This is the data to be fitted */
  for (i = 0; i < n; i++)
    {
      A= gsl_vector_get(vector, i);
      y[i] = A;
      //printf ("data: %u %lf %lf\n", i, y[i], sigma[i]);
    };

  T = gsl_multifit_fdfsolver_lmsder;
  s = gsl_multifit_fdfsolver_alloc (T, n, p);
  gsl_multifit_fdfsolver_set (s, &f, &x.vector);
     
  do {
    iter++;
    status = gsl_multifit_fdfsolver_iterate (s);
    if (status)
      break;
     
    status = gsl_multifit_test_delta (s->dx, s->x,1e-4, 1e-4);
  }
  while (status == GSL_CONTINUE && iter < 500);

#if defined GSL_MAJOR_VERSION && GSL_MAJOR_VERSION >= 2
  J = gsl_matrix_alloc(n, p);
  gsl_multifit_fdfsolver_jac(s, J);
  gsl_multifit_covar (J, 0.0, covar);
  gsl_matrix_free (J);
#else
  gsl_multifit_covar (s->J, 0.0, covar);
#endif
     
#define FIT(i) gsl_vector_get(s->x, i)
#define ERR(i) sqrt(gsl_matrix_get(covar,i,i))
   
  
  /* chi = gsl_blas_dnrm2(s->f); */
  /* dof = n - p; */
  /* c = GSL_MAX_DBL(1, chi / sqrt(dof));  */
     
    //printf("chisq/dof = %g\n",  pow(chi, 2.0) / dof);
     
    //printf ("A     = %.5f +/- %.5f\n", FIT(0), c*ERR(0));
    //printf ("x     = %.5f +/- %.5f\n", FIT(1), c*ERR(1));
    //printf ("c      = %.5f +/- %.5f\n", FIT(2), c*ERR(2));
  
     
  //printf ("status = %s\n", gsl_strerror (status));

  for (i = 0; i < p; i++) {
    params[i]=FIT(i);
  }
  
 
  gsl_multifit_fdfsolver_free (s);
  gsl_matrix_free (covar);
  gsl_rng_free (r);

  return GSL_SUCCESS;
  
}
int expb_f_linear_red (const gsl_vector * x, void *data, gsl_vector * f)
{
  size_t n = ((struct data_red *)data)->n;
  double *y = ((struct data_red *)data)->y;
  double cen =  ((struct data_red *)data)->cen;  
  double wid =  ((struct data_red *)data)->wid;  
  double A = gsl_vector_get (x, 0);
  double ca = gsl_vector_get (x, 1);
  double cb = gsl_vector_get (x, 2);
     
  size_t i;
     

  for (i = 0; i < n; i++)
  {
    /* Model Yi = A * exp(-lambda * i) + b */
    double t = i/(1.*n); 
    double Yi = A * exp (-1.*wid*pow(t-cen, 2))+ca+cb*t;
    gsl_vector_set (f, i, (Yi - y[i]));
  }
     
  return GSL_SUCCESS;
}

int expb_df_linear_red (const gsl_vector * x, void *data,gsl_matrix * J)
{
  UNUSED(x);
  size_t n = ((struct data_red *)data)->n;
  double cen =  ((struct data_red *)data)->cen;  
  double wid =  ((struct data_red *)data)->wid;  
     
  /* double A = gsl_vector_get (x, 0); */
  /* double ca = gsl_vector_get (x, 3); */
  /* double cb = gsl_vector_get (x, 4); */
     
  size_t i;
     
  for (i = 0; i < n; i++)
  {
    /* Jacobian matrix J(i,j) = dfi / dxj, */
    /* where fi = (Yi - yi)/sigma[i],      */
    /*       Yi = A * exp(-lambda * i) + b  */
    /* and the xj are the parameters (A,lambda,b) */
    double t = i/(1.*n);
    double e = exp (-1.*wid*pow(t-cen, 2)  );
    gsl_matrix_set (J, i, 0, e); 
    gsl_matrix_set (J, i, 1, 1.);
    gsl_matrix_set (J, i, 2, t);
  }
  return GSL_SUCCESS;
}
int expb_fdf_linear_red (const gsl_vector * x, void *data,
            gsl_vector * f, gsl_matrix * J)
{
  expb_f_linear_red (x, data, f);
  expb_df_linear_red (x, data, J);
     
  return GSL_SUCCESS;
}
