/* $Id: mat_flatfield.c,v0.5 2014-06-15 12:56:21 mheininger Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: mheininger $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_flatfield.c $
 */

#include <stdlib.h>
#include <stdio.h>

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include <string.h>

#include "mat_error.h"
#include "mat_utils.h"
#include "mat_flatfield.h"
#include "mat_frame.h"
#include "mat_imagingdetector.h"
#include "mat_utils.h"

/*-----------------------------------------------------------------------------
                                   Define
 -----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
                                   Functions prototypes
 -----------------------------------------------------------------------------*/

/**
   @ingroup ffm
   @brief Creates a flatfield map data structure assocciated with a detector and covering the whole detector or only a set of sub-windows on that detector.
   @param det     This structure describes the detector of the flatfield map.
   @param imgdet  This optional structure describes the sub-window-setup.
   @returns The new flatfield map of NULL on error.

   This function create a flatfield map data structure using the detector specification as a template. If a sub-window setup is given, this setup is used as setup for the flatfield map. This means, that for each sub-window an individual cpl_image is allocated and stored in the list_flatfield member. The number of detector channels, and therefore the size of the detector channel specific members (channelgain, channeloffset and channelnoise), is determined by the det parameter.
 */
mat_flatfield *mat_flatfield_new(mat_detector *det,
				 mat_imagingdetector *imgdet)
{
  mat_flatfield *ffm = NULL;
  int            r;

  mat_assert_value((det != NULL), CPL_ERROR_NULL_INPUT, NULL, "no mat_detector (det) argument given");
  ffm = (mat_flatfield *)cpl_calloc(1, sizeof(mat_flatfield));
  if (ffm == NULL)
    {
      cpl_msg_error(cpl_func, "could not allocate memory for mat_flatfield");
      return NULL;
    }
  ffm->det = mat_detector_duplicate(det);
  if (ffm->det == NULL)
    {
      cpl_msg_error(cpl_func, "could not duplicate mat_detector");
      mat_flatfield_delete(ffm);
      return NULL;
    }

  ffm->keywords = cpl_propertylist_new();
  if (ffm->keywords == NULL)
    {
      cpl_msg_error(cpl_func, "could not allocate memory for cpl_propertylist");
      mat_flatfield_delete(ffm);
      return NULL;
    }
  if (imgdet == NULL)
    { /* create an imaging detector table covering the whole detector */
      ffm->imgdet = mat_imagingdetector_new(NULL, 1); /* no propertylist and one sub-window */
      if (ffm->imgdet == NULL)
	{
	  cpl_msg_error(cpl_func, "could not allocate memory for mat_imagingdetector");
	  mat_flatfield_delete(ffm);
	  return NULL;
	}
      /* set the geometry of the sub-window to the whole detector */
      ffm->imgdet->list_region[0]->corner[0] = 1;
      ffm->imgdet->list_region[0]->corner[1] = 1;
      ffm->imgdet->list_region[0]->naxis[0] = ffm->det->nx;
      ffm->imgdet->list_region[0]->naxis[1] = ffm->det->ny;
    }
  else
    {
      ffm->imgdet = mat_imagingdetector_duplicate(imgdet);
      if (ffm->imgdet == NULL)
	{
	  cpl_msg_error(cpl_func, "could not duplicate the mat_imagingdetector");
	  mat_flatfield_delete(ffm);
	  return NULL;
	}
    }
  ffm->channelgain = cpl_vector_new(det->channel_ncolumns*det->channel_nrows);
  if (ffm->channelgain == NULL)
    {
      cpl_msg_error(cpl_func, "could not allocate a cpl_vector for the detector channel gain");
      mat_flatfield_delete(ffm);
      return NULL;
    }
  ffm->channeloffset = cpl_vector_new(det->channel_ncolumns*det->channel_nrows);
  if (ffm->channeloffset == NULL)
    {
      cpl_msg_error(cpl_func, "could not allocate a cpl_vector for the detector channel offset");
      mat_flatfield_delete(ffm);
      return NULL;
    }
  ffm->channelnoise = cpl_vector_new(det->channel_ncolumns*det->channel_nrows);
  if (ffm->channelnoise == NULL)
    {
      cpl_msg_error(cpl_func, "could not allocate a cpl_vector for the detector channel noise");
      mat_flatfield_delete(ffm);
      return NULL;
    }
  ffm->list_flatfield = (cpl_image **)cpl_calloc(ffm->imgdet->nbregion, sizeof(cpl_image *));
  if (ffm->list_flatfield == NULL)
    {
      cpl_msg_error(cpl_func, "could not create a cpl_image list (flatfield)");
      mat_flatfield_delete(ffm);
      return NULL;
    }
  ffm->has_stdev = 0; // no valid stdev map, because it is still empty!
  ffm->list_stdev = (cpl_image **)cpl_calloc(ffm->imgdet->nbregion, sizeof(cpl_image *));
  if (ffm->list_stdev == NULL)
    {
      cpl_msg_error(cpl_func, "could not create a cpl_image list (error)");
      mat_flatfield_delete(ffm);
      return NULL;
    }
  for (r = 0; r < ffm->imgdet->nbregion; r++)
    {
      int nx = ffm->imgdet->list_region[r]->naxis[0];
      int ny = ffm->imgdet->list_region[r]->naxis[1];
      ffm->list_flatfield[r] = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
      if (ffm->list_flatfield[r] == NULL)
	{
	  cpl_msg_error(cpl_func, "could not create a cpl_image for a flatfield region");
	  mat_flatfield_delete(ffm);
	  return NULL;
	}
      ffm->list_stdev[r] = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
      if (ffm->list_stdev[r] == NULL)
	{
	  cpl_msg_error(cpl_func, "could not create a cpl_image for a stdev region");
	  mat_flatfield_delete(ffm);
	  return NULL;
	}
    }
  ffm->detectorgain = 1.0;
  ffm->detectornoise = 0.0;
  ffm->horizontalcorrelation = 0.0;
  ffm->verticalcorrelation = 0.0;
  ffm->flatfieldstdev = 0.0;
  return ffm;
}

/**
   @ingroup ffm
   @brief This function deletes a mat_flatfield structure.
   @param ffm  The flatfield map which will be deleted.

   Deleting a flatfield data structure includes all members. If an
   error (for example during the creation of a flatfield data structure)
   leads to an incomplete data structure (some members are NULL) this
   function detects this and deletes only the valid members.
*/
cpl_error_code mat_flatfield_delete(mat_flatfield *ffm)
{
  int         r;

  mat_assert((ffm != NULL), CPL_ERROR_NULL_INPUT, "no mat_flatfield (ffm) argument given");
  mat_assert(((ffm->list_flatfield == NULL) || (ffm->imgdet != NULL)), CPL_ERROR_UNSPECIFIED, "could not get the number of flatfield masks since ffm->imgdet is empty -> memory leak!");
  if (ffm->keywords != NULL)
    {
      cpl_propertylist_delete(ffm->keywords);
      ffm->keywords = NULL;
    }
  if (ffm->list_flatfield != NULL)
    {
      for (r = 0; r < ffm->imgdet->nbregion; r++)
	{
	  if (ffm->list_flatfield[r] != NULL)
	    {
	      cpl_image_delete(ffm->list_flatfield[r]);
	      ffm->list_flatfield[r] = NULL;
	    }
	}
      cpl_free(ffm->list_flatfield);
      ffm->list_flatfield = NULL;
    }
  if (ffm->list_stdev != NULL)
    {
      for (r = 0; r < ffm->imgdet->nbregion; r++)
	{
	  if (ffm->list_stdev[r] != NULL)
	    {
	      cpl_image_delete(ffm->list_stdev[r]);
	      ffm->list_stdev[r] = NULL;
	    }
	}
      cpl_free(ffm->list_stdev);
      ffm->list_stdev = NULL;
    }
  if (ffm->channelgain != NULL)
    {
      cpl_vector_delete(ffm->channelgain);
      ffm->channelgain = NULL;
    }
  if (ffm->channeloffset != NULL)
    {
      cpl_vector_delete(ffm->channeloffset);
      ffm->channeloffset = NULL;
    }
  if (ffm->channelnoise != NULL)
    {
      cpl_vector_delete(ffm->channelnoise);
      ffm->channelnoise = NULL;
    }
  if (ffm->imgdet != NULL)
    {
      mat_imagingdetector_delete(ffm->imgdet);
      ffm->imgdet = NULL;
    }
  if (ffm->det != NULL)
    {
      mat_detector_delete(ffm->det);
      ffm->det = NULL;
    }
  cpl_free(ffm);
  return CPL_ERROR_NONE;
}

/**
   @ingroup ffm
   @brief This function stores a static flatfield map in a FITS file.
   @param ffm The static flatfield map of a detector.
   @param fname The name of the static flatfield map FITS file.
   @param rname The name of the pipeline recipe.
   @param parlist The plugin parameter list.
   @param frameset The input frameset of the plugin (optional).

   This function stores the contents of the mat_flatfield data structure as a static flatfield map
   in a FITS file. The flatfield map must cover the whole detector (no sub-window setup is allowed).
   The flatfield map is stored as a floating point image in the primary header. The error map is
   stored in an image extension (FFSTDERR). In addition, some QC parameters
   from the data structure are stored as primary header keywords:

   - QC DETi GAIN The global detector gain (in electrons per ADU) will be used for scientific purposes.
   - QC DETi RON The global detector readout noise (in electrons) will be used for monitoring purposes.
   - QC DETi CHANNELj GAIN The detector channel specific gain (in electrons per ADU) will be used to detect detector and read-out electronics problems.
   - QC DETi CHANNELj OFFSET The detector channel specific offset (in ADU) will be used to detect detector and read-out electronics problems.
   - QC DETi CHANNELj RON The detector channel specific readout noise (in electrons) will be used to detect detector and read-out electronics problems.
   - QC DETi FFM STDEV The standard deviation of the computed flatfield map (the mean is normalized to 1.0).

   Where i is the detector number (HAWAII-2RG -> 1, Aquarius -> 2) and j is the detector channel number.
 */
cpl_error_code mat_flatfield_save(mat_flatfield *ffm,
				  const char *fname,
				  const char *rname,
				  cpl_parameterlist *parlist,
				  cpl_frameset *frameset)
{
  int               c, cr;
  cpl_frame        *ffmframe = NULL;
  cpl_propertylist *plist = NULL;
  char              kwd[64];

  cpl_msg_info(cpl_func, "storing the flatfield map in file %s", fname);
  cpl_error_reset();
  cpl_ensure_code((ffm != NULL), CPL_ERROR_NULL_INPUT);
  /* Create product frame */
  ffmframe = cpl_frame_new();
  cpl_frame_set_filename(ffmframe, fname);
  cpl_frame_set_tag(ffmframe, MATISSE_FFM_PROCATG);
  cpl_frame_set_type(ffmframe, CPL_FRAME_TYPE_IMAGE);
  cpl_frame_set_group(ffmframe, CPL_FRAME_GROUP_PRODUCT);
  cpl_frame_set_level(ffmframe, CPL_FRAME_LEVEL_FINAL);
  if (cpl_error_get_code()) {
    cpl_msg_error(cpl_func, "Error while initialising the product frame, code = %d, message = %s",
		  cpl_error_get_code(),
		  cpl_error_get_message());
    cpl_frame_delete(ffmframe);
    return CPL_ERROR_UNSPECIFIED;
  }
  plist = cpl_propertylist_new();
  /* Add DataFlow keywords */
  /*
    check if the input frameset is empty
    => function used in an artificial unit test (no real input files)
    => the DFS header cannot include the keywords from the first input file
  */
  if (!cpl_frameset_is_empty(frameset))
    {
      if (cpl_dfs_setup_product_header(plist, ffmframe, frameset, parlist,
				       rname, PACKAGE "/" PACKAGE_VERSION, "?Dictionary?", NULL) != CPL_ERROR_NONE) {
	cpl_msg_error(cpl_func, "Problem in the product DFS-compliance, code = %d, message = %s",
		      cpl_error_get_code(),
		      cpl_error_get_message());
	cpl_error_reset();
	cpl_propertylist_delete(plist);
	cpl_frame_delete(ffmframe);
	return CPL_ERROR_UNSPECIFIED;
      }
    }
  /* Add QC parameters in the header */
  /* readout mode is missing */
  snprintf(kwd, 64, "ESO QC DET%d GAIN%d", ffm->det->nr, ffm->det->read_id);
  cpl_propertylist_append_double(plist, kwd, mat_round(ffm->detectorgain, MAT_PREC_GAIN));
  snprintf(kwd, 64, "ESO QC DET%d RON%d", ffm->det->nr, ffm->det->read_id);
  cpl_propertylist_append_double(plist, kwd, mat_round(ffm->detectornoise, MAT_PREC_NOISE));
  snprintf(kwd, 64, "ESO QC DET%d HACORR%d", ffm->det->nr, ffm->det->read_id);
  cpl_propertylist_append_double(plist, kwd, mat_round(ffm->horizontalcorrelation, MAT_PREC_CORRELATION));
  snprintf(kwd, 64, "ESO QC DET%d VACORR%d", ffm->det->nr, ffm->det->read_id);
  cpl_propertylist_append_double(plist, kwd, mat_round(ffm->verticalcorrelation, MAT_PREC_CORRELATION));
  snprintf(kwd, 64, "ESO QC DET%d FFM STDEV%d", ffm->det->nr, ffm->det->read_id);
  cpl_propertylist_append_double(plist, kwd, mat_round(ffm->flatfieldstdev, MAT_PREC_VARIANCE));
  cr = cpl_vector_get_size(ffm->channelgain);
  for (c = 0; c < cr; c++)
    {
      snprintf(kwd, 64, "ESO QC DET%d CHANNEL%d GAIN%d", ffm->det->nr, c + 1, ffm->det->read_id);
      cpl_propertylist_append_double(plist, kwd, mat_round(cpl_vector_get(ffm->channelgain, c), MAT_PREC_GAIN));
      snprintf(kwd, 64, "ESO QC DET%d CHANNEL%d OFFSET%d", ffm->det->nr, c + 1, ffm->det->read_id);
      cpl_propertylist_append_double(plist, kwd, mat_round(cpl_vector_get(ffm->channeloffset, c), MAT_PREC_INTENSITY));
      snprintf(kwd, 64, "ESO QC DET%d CHANNEL%d RON%d", ffm->det->nr, c + 1, ffm->det->read_id);
      cpl_propertylist_append_double(plist, kwd, mat_round(cpl_vector_get(ffm->channelnoise, c), MAT_PREC_NOISE));
    }

  // Add Generic QC parameters
  cpl_propertylist *qclist=NULL;
  qclist = cpl_propertylist_new();
  mat_add_generic_qc(plist,qclist);
  cpl_propertylist_append(plist,qclist);
  cpl_propertylist_delete(qclist);

  /* Save the file */
  cpl_propertylist_erase(plist,"RADECSYS");
  if (cpl_image_save(ffm->list_flatfield[0], fname, CPL_TYPE_FLOAT, plist,
		     CPL_IO_CREATE) != CPL_ERROR_NONE) {
    cpl_msg_error(cpl_func, "Could not save product, code = %d, message = %s",
		  cpl_error_get_code(),
		  cpl_error_get_message());
    cpl_propertylist_delete(plist);
    cpl_frame_delete(ffmframe);
    return CPL_ERROR_UNSPECIFIED;
  }
  cpl_propertylist_delete(plist);
  if (ffm->has_stdev)
    { // Store the stdev map only if it is valid (created from the detector calibration plugin)
      plist = cpl_propertylist_new();
      cpl_propertylist_append_string(plist, "EXTNAME", MATISSE_FFM_STDERR_EXT);
      if (cpl_image_save(ffm->list_stdev[0], fname, CPL_TYPE_FLOAT, plist,
			 CPL_IO_EXTEND) != CPL_ERROR_NONE) {
	cpl_msg_error(cpl_func, "Could not save product, code = %d, message = %s",
		      cpl_error_get_code(),
		      cpl_error_get_message());
	cpl_propertylist_delete(plist);
	cpl_frame_delete(ffmframe);
	return CPL_ERROR_UNSPECIFIED;
      }
      cpl_propertylist_delete(plist);
    }
  /* Log the saved file in the input frameset */
  cpl_frameset_insert(frameset, ffmframe);
  //cpl_frame_delete(ffmframe);
  return CPL_ERROR_NONE;
}

/**
   @ingroup ffm
   @brief This function loads the content of FLATFIELD map in a mat_flatfield structure.
   @param ffmframe The frame containing the name of the flatfield map.
   @returns The loaded flatfield map or NULL.

   This function reads the static flatfield map and stores the contents in a mat_flatfield
   data structure. The flatfield map covers the whole detector and must be mapped
   to a given sub-window layout by using the mat_flatfield_map function.
 */
mat_flatfield *mat_flatfield_load(cpl_frame *ffmframe)
{
  mat_detector     *det = NULL;
  cpl_propertylist *plist = NULL;
  cpl_image        *map = NULL;
  int               extnr = -1;
  mat_flatfield    *ffm = NULL;
  int               nc, c;
  char              kwd[64];

  cpl_msg_info(cpl_func, "load the flatfield map from file %s", cpl_frame_get_filename(ffmframe));
  /* create a mat_detector structure, load the primary keywords and decode the detector */
  det = mat_detector_new();
  if (det == NULL)
    {
      cpl_msg_error(cpl_func, "could not allocate memory for mat_flatfield->det");
      return NULL;
    }
  plist = cpl_propertylist_load(cpl_frame_get_filename(ffmframe), 0);
  if (plist == NULL)
    {
      cpl_msg_error(cpl_func, "could not load the primary keywords");
      mat_detector_delete(det);
      return NULL;
    }
  mat_detector_decode(det, plist);
  /* create and initialize the mat_flatfield structure */
  ffm = mat_flatfield_new(det, NULL); /* no mat_imagingdetector means the whole detector */
  mat_detector_delete(det); /* we can delete the mat_detector, because we have a copy in ffm */
  det = NULL;
  if (ffm == NULL)
    {
      cpl_msg_error(cpl_func, "could not allocate memory for mat_flatfield");
      cpl_propertylist_delete(plist);
      return NULL;
    }
  /* move the propertylist into the ffm */
  cpl_propertylist_delete(ffm->keywords);
  ffm->keywords = plist;
  /* cpl_propertylist_dump(ffm->keywords, stdout); */
  /* get all important keyword values from the propertylist */
  snprintf(kwd, 64, "ESO QC DET%d GAIN%d", ffm->det->nr, ffm->det->read_id);
  ffm->detectorgain = cpl_propertylist_get_double(ffm->keywords, kwd);
  snprintf(kwd, 64, "ESO QC DET%d RON%d", ffm->det->nr, ffm->det->read_id);
  ffm->detectornoise = cpl_propertylist_get_double(ffm->keywords, kwd);
  snprintf(kwd, 64, "ESO QC DET%d HACORR%d", ffm->det->nr, ffm->det->read_id);
  ffm->horizontalcorrelation = cpl_propertylist_get_double(ffm->keywords, kwd);
  snprintf(kwd, 64, "ESO QC DET%d VACORR%d", ffm->det->nr, ffm->det->read_id);
  ffm->verticalcorrelation = cpl_propertylist_get_double(ffm->keywords, kwd);
  snprintf(kwd, 64, "ESO QC DET%d FFM STDEV%d", ffm->det->nr, ffm->det->read_id);
  ffm->flatfieldstdev = cpl_propertylist_get_double(ffm->keywords, kwd);
  nc = cpl_vector_get_size(ffm->channelgain);
  for (c = 0; c < nc; c++)
    {
      snprintf(kwd, 64, "ESO QC DET%d CHANNEL%d GAIN%d", ffm->det->nr, c + 1, ffm->det->read_id);
      cpl_vector_set(ffm->channelgain, c, cpl_propertylist_get_double(ffm->keywords, kwd));
      snprintf(kwd, 64, "ESO QC DET%d CHANNEL%d OFFSET%d", ffm->det->nr, c + 1, ffm->det->read_id);
      cpl_vector_set(ffm->channeloffset, c, cpl_propertylist_get_double(ffm->keywords, kwd));
      snprintf(kwd, 64, "ESO QC DET%d CHANNEL%d RON%d", ffm->det->nr, c + 1, ffm->det->read_id);
      cpl_vector_set(ffm->channelnoise, c, cpl_propertylist_get_double(ffm->keywords, kwd));
      cpl_msg_info(cpl_func, "channel[%d]: gain=%.2f e-/DU, offset=%.2f DU, ron=%.2f e-",
		   c + 1,
		   cpl_vector_get(ffm->channelgain, c),
		   cpl_vector_get(ffm->channeloffset, c),
		   cpl_vector_get(ffm->channelnoise, c));
    }
  /* and now load the real flatfield map from the file */
  map = cpl_image_load(cpl_frame_get_filename(ffmframe), CPL_TYPE_DOUBLE, 0, 0);
  if (map == NULL)
    {
      cpl_msg_error(cpl_func, "could not load the flatfield map for ffm->list_flatfield[0]");
      mat_flatfield_delete(ffm);
      return NULL;
    }
  else
    {
      cpl_image_delete(ffm->list_flatfield[0]);
      ffm->list_flatfield[0] = map;
    }
  extnr = cpl_fits_find_extension(cpl_frame_get_filename(ffmframe), MATISSE_FFM_STDERR_EXT);
  if (extnr != -1)
    {
      /* we have an error map and we should load it */
      map = cpl_image_load(cpl_frame_get_filename(ffmframe), CPL_TYPE_DOUBLE, 0, extnr);
      if (map == NULL)
	{ // the extension does exist, but the image could not be loaded -> has_stdev := 0
	  ffm->has_stdev = 0;
	  cpl_msg_error(cpl_func, "could not load the error map for ffm->list_stdev[0]");
	}
      else
	{ // the extension does exist and the image could be loaded -> has_stdev := 1
	  ffm->has_stdev = 1;
	  cpl_image_delete(ffm->list_stdev[0]);
	  ffm->list_stdev[0] = map;
	}
    }
  else
    { // the extension does not exist -> has_stdev := 0
      ffm->has_stdev = 0;
      cpl_msg_info(cpl_func, "no error map in the flatfield map in file %s", cpl_frame_get_filename(ffmframe));
    }
  return ffm;
}

/**
   @ingroup ffm
   @brief This function maps a flatfield map to a given sub-window layout.
   @param ffm    This flatfield map covers the whole detector.
   @param imgdet This structure describes the sub-window layout.
   @returns The mapped flatfield map or NULL.

   This function uses the sub-window layout of the imgdet argument to create
   a new flatfield map (using the function mat_flatfield_new and the layout in imgdet).
   In a second step, the sub-window specific parts of the flatfield mask are extracted
   and stored in the newly created flatfield map.
 */
mat_flatfield *mat_flatfield_map(mat_flatfield *ffm, mat_imagingdetector *imgdet)
{
  mat_flatfield *nffm = NULL;
  int           r;

  mat_assert_value((ffm != NULL), CPL_ERROR_NULL_INPUT, NULL, "no mat_flatfield (ffm) argument given");
  mat_assert_value((imgdet != NULL), CPL_ERROR_NULL_INPUT, NULL, "no mat_imagingdetector (imgdet) argument given");
  mat_assert_value((imgdet->list_region != NULL), CPL_ERROR_NULL_INPUT, NULL, "no valid mat_imagingdetector (imgdet->list_region) argument given");
  nffm = mat_flatfield_new(ffm->det, imgdet);
  if (nffm == NULL)
    {
      cpl_msg_error(cpl_func, "could not allocate memory for mat_flatfield");
      return NULL;
    }
  /* put a copy of the propertylist into the newly created flatfield map */
  if (nffm->keywords != NULL)
    {
      cpl_propertylist_delete(nffm->keywords);
      nffm->keywords = NULL;
      nffm->keywords = cpl_propertylist_duplicate(ffm->keywords);
    }
  /* copy the QC parameters into the new flatfield map */
  nffm->detectorgain          = ffm->detectorgain;
  nffm->detectornoise         = ffm->detectornoise;
  nffm->horizontalcorrelation = ffm->horizontalcorrelation;
  nffm->verticalcorrelation   = ffm->verticalcorrelation;
  nffm->flatfieldstdev        = ffm->flatfieldstdev;
  cpl_vector_copy(nffm->channelgain, ffm->channelgain);
  cpl_vector_copy(nffm->channeloffset, ffm->channeloffset);
  cpl_vector_copy(nffm->channelnoise, ffm->channelnoise);
  /* extract only the sub-window equivalents of the flatfield map */
  nffm->has_stdev = ffm->has_stdev;
  for (r = 0; r < nffm->imgdet->nbregion; r++)
    {
      if (nffm->list_flatfield[r] != NULL)
	{
	  cpl_image_delete(nffm->list_flatfield[r]);
	  nffm->list_flatfield[r] = NULL;
	}
      nffm->list_flatfield[r] = cpl_image_extract(ffm->list_flatfield[0],
						  imgdet->list_region[r]->corner[0],
						  imgdet->list_region[r]->corner[1],
						  imgdet->list_region[r]->corner[0] + imgdet->list_region[r]->naxis[0] - 1,
						  imgdet->list_region[r]->corner[1] + imgdet->list_region[r]->naxis[1] - 1);
      if (nffm->list_stdev[r] != NULL)
	{
	  cpl_image_delete(nffm->list_stdev[r]);
	  nffm->list_stdev[r] = NULL;
	}
      nffm->list_stdev[r] = cpl_image_extract(ffm->list_stdev[0],
					      imgdet->list_region[r]->corner[0],
					      imgdet->list_region[r]->corner[1],
					      imgdet->list_region[r]->corner[0] + imgdet->list_region[r]->naxis[0] - 1,
					      imgdet->list_region[r]->corner[1] + imgdet->list_region[r]->naxis[1] - 1);
    }
  return nffm;
}
