/* $Id: mat_flatfield.h,v0.5 2014-06-15 12:56:21 mheininger Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: mheininger $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_flatfield.h $
 */

#ifndef MAT_FLATFIELD_H
#define MAT_FLATFIELD_H

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include <stdio.h>
#include <cpl.h>
#include "mat_detector.h"
#include "mat_imagingdetector.h"

/*-----------------------------------------------------------------------------
                                   Define
 -----------------------------------------------------------------------------*/

/**
   @ingroup ffm
   @brief Structure containing the static flatfield map of a detector.

   This structure represents all contents in a FLATFIELD FITS file. This includes
   the primary header keywords (keywords member), the sub-window setup (imgdet member),
   the conversion factor (detectorgain and channelgain members), the readout noise
   (detectornoise and channelnoise members) and the flatfield map itself.

   After loading a flatfield map from file, the map covers the whole detector. This
   map can be mapped to a certain sub-window setup where each sub-window is
   covered by an individual flatfield map as an image in the list_flatfield member.
 */
typedef struct {
  mat_detector         *det;            /*!< The detector specification for this flatfield map. */
  cpl_propertylist     *keywords;       /*!< The keywords from the primary header. */
  mat_imagingdetector  *imgdet;         /*!< The sub-window setup defined by the contents of the IMAGING_DETECTOR binary table. */
  double                detectorgain;   /*!< The detector specific gain in electrons per ADU (QC1 parameter QC DETi GAIN). */
  double                detectornoise;  /*!< The detector specific readout noise in electrons (QC1 parameter QC DETi RON). */
  double                horizontalcorrelation; /*!< The autocorrelation coefficient between two horizontally adjacent pixels. This value will be stored as a QC1 parameter (QC DETi HACORR). */
  double                verticalcorrelation;   /*!< The autocorrelation coefficient between two vertically adjacent pixels. This value will be stored as a QC1 parameter (QC DETi VACORR). */
  double                flatfieldstdev; /*!< The standard deviation of the computed flatfield map (QC1 parameter QC DETi FFM STDEV). */
  cpl_vector           *channelgain;    /*!< The detector channel specific gain in electrons per ADU (QC1 parameter QC DETi CHANNELj GAIN). */
  cpl_vector           *channeloffset;  /*!< The detector channel specific offset in ADU (QC1 parameter QC DETi CHANNELj OFFSET). */
  cpl_vector           *channelnoise;   /*!< The detector channel specific readout noise in electrons (QC1 parameter QC DETi CHANNELj NOISE). */
  cpl_image           **list_flatfield; /*!< Vector containing the flatfield maps for each region. Dimensions given by imgdet->nbregion. */
  int                   has_stdev;      /*!< Flag, if the loaded flatfield map has a valid stdev map. */
  cpl_image           **list_stdev;     /*!< Vector containing the standard error of the flatfield map for each region. Dimensions given by imgdet->nbregion.  */
} mat_flatfield;

/*-----------------------------------------------------------------------------
                                   Functions prototypes
 -----------------------------------------------------------------------------*/

mat_flatfield *mat_flatfield_new(mat_detector *det, mat_imagingdetector *imgdet);
cpl_error_code mat_flatfield_delete(mat_flatfield *ffm);

cpl_error_code mat_flatfield_save(mat_flatfield *ffm,
				  const char *fname,
				  const char *rname,
				  cpl_parameterlist *parlist,
				  cpl_frameset *frameset);

mat_flatfield *mat_flatfield_load(cpl_frame *ffmframe);

mat_flatfield *mat_flatfield_map(mat_flatfield *ffm, mat_imagingdetector *imgdet);

#endif
