/* $Id: mat_gendata.c,v0.5 2014-06-15 12:56:21 mheininger Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: mheininger $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_gendata.c $
 */

#include <stdlib.h>
#include <stdio.h>

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include <cpl.h>
#include <string.h>
#include <math.h>
#include <float.h>
#include <assert.h>

#include "mat_drl.h"
#include "mat_error.h"
#include "mat_utils.h"
#include "mat_imagingdata.h"
#include "mat_array.h"
#include "mat_gendata.h"

/*-----------------------------------------------------------------------------
                                   Define
 -----------------------------------------------------------------------------*/


/*-----------------------------------------------------------------------------
                                   Functions
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
   @ingroup gendata
   @brief delete structure mat_gendata
   @param  genData          current structure
   @return error code
 */
/*-----------------------------------------------------------------------------*/
cpl_error_code mat_gendata_delete(mat_gendata *genData) {
  // Check input parameters
  mat_assert_value((genData != NULL),CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT, "no genData given in argument");
  if (genData->keywords != NULL)
    {
      cpl_propertylist_delete(genData->keywords);
      genData->keywords = NULL;
    }
  if (genData->imgdata != NULL)
    {
      mat_imagingdata_delete(genData->imgdata);
      genData->imgdata = NULL;
    }
  if (genData->imgdet != NULL)
    {

      mat_imagingdetector_delete(genData->imgdet);
      genData->imgdet = NULL;
    }
  if (genData->array != NULL)
    {
      mat_array_delete(genData->array);
      genData->array = NULL;
    }
  cpl_free(genData);
  return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
   @ingroup gendata
   @brief Duplicate a mat_gendata structure
   @param gen  mat_gendata structure to duplicate
   @param type   The cpl_type used for the new CPL images.
   @return mat_gendata : duplicated structure
 */
/*-----------------------------------------------------------------------------*/
mat_gendata *mat_gendata_duplicate(const mat_gendata *gen, cpl_type type)
{
  mat_gendata *genData = NULL;
  cpl_errorstate prestate = cpl_errorstate_get();

  //test if mat_gendata is empty
  if (gen == NULL)
    {
      return NULL;
    }
  // Allocate the structure and initialize its attributes
  genData = cpl_calloc(1, sizeof(mat_gendata));
  if (genData == NULL) {
    cpl_msg_error(cpl_func,"could not allocate memory for genData");
    return NULL;
  }
  genData->keywords= cpl_propertylist_duplicate(gen->keywords);
  genData->imgdet= mat_imagingdetector_duplicate(gen->imgdet);
  genData->array= mat_array_duplicate(gen->array);
  genData->imgdata= mat_imagingdata_duplicate(gen->imgdata, type);
  /* Error handling */
  if (!cpl_errorstate_is_equal(prestate))
    {
      cpl_errorstate_dump(prestate, CPL_FALSE, cpl_errorstate_dump_one);
      // Unload the mat_gendata structure
      mat_gendata_delete(genData);
      return NULL;
    }
  return genData;
}

/*----------------------------------------------------------------------------*/
/**
   @ingroup gendata
   @brief Load a sub-part of raw data in mat_gendata structure. Only a sub-part 
   (length rows) of the IMAGING_DATA binary table is loaded (starting from row 
   start)
   @param frame           current frame
   @param start           starting row of the imaging data binary table
   @param length          number of row to read
   @param type            The cpl_type used for the loaded CPL images.
   @return mat_gendata
 */
/*-----------------------------------------------------------------------------*/
mat_gendata * mat_gendata_load_window(cpl_frame *frame, int start, int length, cpl_type type)
{
  mat_gendata * genData=NULL;
  int i=0;
  int j=0;
  int nbExtent=0;
  int nbrow_opti=0;
  int flagDet=0;
  int flagArray=0;
  int flagData=0;
  int flagOpti=0;
  int *staindex=NULL;
  int *armindex=NULL;
  char **telname_array=NULL;
  char **telname_opti=NULL;
  char *keyExtname=NULL;

  cpl_propertylist *plist=NULL;
  cpl_table *table=NULL;
  cpl_table *tableopti=NULL;

  /* cpl_errorstate prestate = cpl_errorstate_get(); */


  // Check input parameters
  mat_assert_value((start>=0),CPL_ERROR_NULL_INPUT, NULL, 
		   "start should be greater than or equal to 0");
  mat_assert_value((length > 0),CPL_ERROR_NULL_INPUT, NULL, 
		   "length should be greater than 0");
 //test if frame is empty
  if (frame == NULL){
    cpl_msg_info(cpl_func,"frame empty");
    return NULL;
  }
  
  // Allocate the structure and initialize its attributes
  genData = cpl_calloc(1, sizeof(mat_gendata));
  if (genData == NULL) {
    cpl_msg_error(cpl_func,"could not allocate memory for genData");
    return NULL;
  }
  genData->keywords= cpl_propertylist_load(cpl_frame_get_filename(frame),0);
  genData->imgdet=NULL;
  genData->array=NULL;
  genData->imgdata=NULL;

  if (genData->keywords == NULL)
    {
      /*
	there is something wrong with the file if we cannot load the property list
	-> dump the error, deallocate the structure and return NULL
      */
      // Unload the mat_gendata structure
      mat_gendata_delete(genData);
      return NULL;
    }

  // Find all extensions before IMAGING_DATA
  nbExtent=cpl_frame_get_nextensions(frame);
  for(i=0; i<nbExtent; i++)
    {
      plist=cpl_propertylist_load(cpl_frame_get_filename(frame),i+1);
      keyExtname=(char *)cpl_propertylist_get_string(plist,"EXTNAME");
      if (keyExtname != NULL)
	{
	  table=cpl_table_load(cpl_frame_get_filename(frame),i+1,0);
	  if (!strcmp(keyExtname,"IMAGING_DETECTOR"))
	    { // Load data in attribute imgdet
	      //cpl_msg_info(cpl_func, "loading the IMAGING_DETECTOR extension");
	      genData->imgdet = mat_imagingdetector_from_table(plist, table);
	      /* Error handling */
	      if (genData->imgdet == NULL)
		{ 
		  cpl_msg_warning(cpl_func, "Could not load IMAGING_DETECTOR in frame: %s",
				  cpl_frame_get_filename(frame));
		  cpl_error_set_message(cpl_func,CPL_ERROR_UNSPECIFIED,
					"Could not load IMAGING_DETECTOR in frame: %s",
					cpl_frame_get_filename(frame));
		}
	      else
		{
		  flagDet=1;
		}
	    }
	  else if (!strcmp(keyExtname,"ARRAY_GEOMETRY"))
	    { // Load data in attribute array
	      //cpl_msg_info(cpl_func, "loading the ARRAY_GEOMETRY extension");
	      genData->array = mat_array_from_table(plist, table);
	      /* Error handling */
	      if (genData->array == NULL)
		{ 
		  cpl_error_set_message(cpl_func,CPL_ERROR_UNSPECIFIED,
					"Could not load ARRAY_GEOMETRY in frame: %s",
					cpl_frame_get_filename(frame));
		}
	      else
		{ // Get info for IMAGING_DATA
		  staindex= cpl_calloc(genData->array->nbstation, sizeof(int));
		  telname_array= cpl_calloc(genData->array->nbstation, sizeof(char *));
		  for(j=0; j<genData->array->nbstation; j++)
		    {
		      staindex[j]= genData->array->list_station[j]->staindex;
		      telname_array[j]= cpl_calloc(17, sizeof(char));
		      strncpy(telname_array[j], genData->array->list_station[j]->telname,17);
		    }
		  flagArray=1;
		}
	    }
	  else if (!strcmp(keyExtname,"OPTICAL_TRAIN"))
	    { // Get info for IMAGING_DATA
	      //cpl_msg_info(cpl_func, "loading the OPTICAL_TRAIN extension");
	      tableopti = cpl_table_load(cpl_frame_get_filename(frame),i+1,0);
	      if (tableopti == NULL)
		{ 
		  cpl_error_set_message(cpl_func,CPL_ERROR_UNSPECIFIED,
					"Could not read OPTICAL_TRAIN in frame: %s",
					cpl_frame_get_filename(frame));
		}
	      else
		{
		  nbrow_opti= cpl_table_get_nrow(tableopti);
		  telname_opti= cpl_calloc(nbrow_opti, sizeof(char *));
		  for(j=0;j<nbrow_opti;j++)
		    {
		      telname_opti[j]= cpl_calloc(17, sizeof(char));
		      strncpy(telname_opti[j], cpl_table_get_string(tableopti, "TEL_NAME", j),17);
		    }
		  flagOpti=1;
		}
	    }
	  if (table != NULL) cpl_table_delete(table);
	}
      if (plist != NULL) cpl_propertylist_delete(plist);
    }

  // Set armindex parameter
  if (flagOpti) {
    armindex=cpl_calloc(nbrow_opti, sizeof(int));
    for(i=0;i<nbrow_opti;i++) {
      for(j=0;j<nbrow_opti;j++) {
	if (!strcmp(telname_opti[i], telname_array[j])) {
	  armindex[i] = cpl_table_get_int(tableopti, "INDEX", j, NULL);
	}
      }
    }
  } else {
    armindex=cpl_calloc(4, sizeof(int));
  }

  // Now find  IMAGING_DATA
  for(i=0;i<nbExtent;i++)
    {
      plist=cpl_propertylist_load(cpl_frame_get_filename(frame),i+1);
      keyExtname=(char *)cpl_propertylist_get_string(plist,"EXTNAME");
      if (keyExtname != NULL)
	{ // Load data in attribute imgdata 
	  if (!strcmp(keyExtname,"IMAGING_DATA"))
	    {
	      table=cpl_table_load_window(cpl_frame_get_filename(frame),
					  i+1,0,NULL,start,length);
	      genData->imgdata = mat_imagingdata_load
		(plist, table, genData->imgdet, staindex, armindex, type);
	      /* Error handling */
	      if (genData->imgdata == NULL)
		{
		  cpl_error_set_message(cpl_func,CPL_ERROR_UNSPECIFIED,
					"Could not load IMAGING_DATA in frame: %s",
					cpl_frame_get_filename(frame));
		}
	      else
		{
		  flagData=1;
		}
	      /* heininger:
		 modification for AMBER raw data:
		 since no EXPTIME column is present in some data, the EXPTIME from
		 the primary header keywords is used instead
	      */
	      if (genData->imgdata->list_frame[0]->exptime == 0.0)
	      	{
	      	  double exptime = cpl_propertylist_get_double(genData->keywords, "EXPTIME");
	      	  if (exptime != 0.0)
	      	    {
	      	      int k;
	      	      for (k = 0; k < genData->imgdata->nbframe; k++)
	      		{
	      		  genData->imgdata->list_frame[k]->exptime = exptime;
	      		}
	      	    }
	      	}
	      if (flagOpti)
		{        
		  if (tableopti != NULL) cpl_table_delete(tableopti);
		}
	      if (table != NULL) cpl_table_delete(table);
	    }
	}
      if (plist != NULL) cpl_propertylist_delete(plist);
    }
  // Check the flags
  if ( flagArray==0 || flagDet==0 || flagData==0) {
    cpl_error_set_message(cpl_func,CPL_ERROR_ILLEGAL_INPUT,
                          "Extension missing in frame: %s",
                          cpl_frame_get_filename(frame));
  }
  
  // Free memory
  if (staindex != NULL) cpl_free(staindex);
  if (armindex != NULL) cpl_free(armindex);
  if (telname_array != NULL)
    {
      for(i=0; i<genData->array->nbstation; i++)
	{
	  if (telname_array[i] != NULL) cpl_free(telname_array[i]);
	}
      if (telname_array != NULL) cpl_free(telname_array);
    }
  if (telname_opti != NULL)
    {
      for(i=0; i<nbrow_opti; i++) {
	if (telname_opti[i] != NULL) cpl_free(telname_opti[i]);
      }
      if (telname_opti != NULL) cpl_free(telname_opti);
    }

  return genData;
}

/*----------------------------------------------------------------------------*/
/**
   @ingroup gendata
   @brief Load raw data in mat_gendata structure skipping the nditskip 
   first frames
   @param frame : current frame
   @param nditskip : number of frames to skip
   @param type            The cpl_type used for the loaded CPL images.
   @return mat_gendata : loaded structure
 */
/*-----------------------------------------------------------------------------*/
mat_gendata * mat_gendata_load_skip(cpl_frame *frame, int nditskip, cpl_type type)
{
  mat_gendata * genData=NULL;
  int i=0;
  int j=0;
  int nbExtent=0;
  int nbrow_opti=0;
  int flagDet=0;
//  int flagArray=0;
  int flagData=0;
  int flagOpti=0;
  int *staindex = NULL;
  int *armindex = NULL;
  char **telname_array = NULL;
  char **telname_opti  = NULL;
  char  *keyExtname    =NULL;

  cpl_propertylist *plist=NULL;
  cpl_table *table=NULL;
  cpl_table *tableopti=NULL;

  cpl_errorstate prestate = cpl_errorstate_get();

 //test if frame is empty
  if (frame == NULL){
    cpl_msg_info(cpl_func,"frame empty");
    return NULL;
  }

  //cpl_msg_info(cpl_func,"load data from file %s", cpl_frame_get_filename(frame));
  // Allocate the structure and initialize its attributes
  genData = cpl_calloc(1, sizeof(mat_gendata));
  if (genData == NULL) {
    cpl_msg_error(cpl_func,"could not allocate memory for genData");
    return NULL;
  }
  char *filename = (char *)cpl_frame_get_filename(frame);
  genData->keywords= cpl_propertylist_load(filename,0);
  genData->imgdet=NULL;
  genData->array=NULL;
  genData->imgdata=NULL;

  if (genData->keywords == NULL)
    {
      /*
	there is something wrong with the file if we cannot load the property list
	-> dump the error, deallocate the structure and return NULL
      */
      cpl_msg_error(cpl_func,"could not load property list from file");
      cpl_errorstate_dump(prestate, CPL_FALSE, cpl_errorstate_dump_one);
      // Unload the mat_gendata structure
      mat_gendata_delete(genData);
      return NULL;
    }
  // Find all extensions before IMAGING_DATA
  nbExtent=cpl_frame_get_nextensions(frame);
  for(i=0;i<nbExtent;i++) {
    plist=cpl_propertylist_load(cpl_frame_get_filename(frame),i+1);
    keyExtname=(char *)cpl_propertylist_get_string(plist,"EXTNAME");

    if (keyExtname != NULL) {
      table=cpl_table_load(cpl_frame_get_filename(frame),i+1,0);
      // Load data in attribute imgdet
      if (!strcmp(keyExtname,"IMAGING_DETECTOR")) {
	cpl_msg_debug(cpl_func, "loading the IMAGING_DETECTOR extension");
	genData->imgdet = mat_imagingdetector_from_table(plist, table);
        /* Error handling */
        if (genData->imgdet == NULL) { 
	  cpl_msg_warning(cpl_func, "Could not load IMAGING_DETECTOR in frame: %s",
			                 cpl_frame_get_filename(frame));
          cpl_error_set_message(cpl_func,CPL_ERROR_UNSPECIFIED,
			                 "Could not load IMAGING_DETECTOR in frame: %s",
			                 cpl_frame_get_filename(frame));
        }
        else {
	  flagDet=1;
        }
      }

      // Load data in attribute array
      if (!strcmp(keyExtname,"ARRAY_GEOMETRY")) {
	//cpl_msg_info(cpl_func, "loading the ARRAY_GEOMETRY extension");
	genData->array = mat_array_from_table(plist, table);
	
        /* Error handling */
        if (genData->array == NULL)
	  { 
	    cpl_error_set_message(cpl_func,CPL_ERROR_UNSPECIFIED,
				  "Could not load ARRAY_GEOMETRY in frame: %s",
				  cpl_frame_get_filename(frame));
	  }
        else
	  {
	    // Get info for IMAGING_DATA
	    staindex= cpl_calloc(genData->array->nbstation, sizeof(int));
	    telname_array= cpl_calloc(genData->array->nbstation, sizeof(char *));
	    for(j=0; j<genData->array->nbstation; j++) {
	      staindex[j]= genData->array->list_station[j]->staindex;
	      telname_array[j]= cpl_calloc(17, sizeof(char));
	      strncpy(telname_array[j], genData->array->list_station[j]->telname,17);
	    }
//	    flagArray=1;
	  }
      }

      if (!strcmp(keyExtname,"OPTICAL_TRAIN")) {
        // Get info for IMAGING_DATA
	//cpl_msg_info(cpl_func, "loading the OPTICAL_TRAIN extension");
        tableopti = cpl_table_load(cpl_frame_get_filename(frame),i+1,0);
        if (tableopti == NULL) { 
          cpl_error_set_message(cpl_func,CPL_ERROR_UNSPECIFIED,
				"Could not read OPTICAL_TRAIN in frame: %s",
				cpl_frame_get_filename(frame));
        } else {
          nbrow_opti= cpl_table_get_nrow(tableopti);
          telname_opti= cpl_calloc(nbrow_opti, sizeof(char *));
          for(j=0;j<nbrow_opti;j++) {
            telname_opti[j]= cpl_calloc(17, sizeof(char));
            strncpy(telname_opti[j], cpl_table_get_string(tableopti, "TEL_NAME", j),17);
          }
	  flagOpti=1;
        }
      }
      if (table != NULL) cpl_table_delete(table);
    }
    if (plist != NULL) cpl_propertylist_delete(plist);
  }

  // Set armindex parameter
  if (flagOpti) {
    armindex=cpl_calloc(nbrow_opti, sizeof(int));
    for(i=0;i<nbrow_opti;i++) {
      for(j=0;j<nbrow_opti;j++) {
	if (!strcmp(telname_opti[i], telname_array[j])) {
	  armindex[i] = cpl_table_get_int(tableopti, "INDEX", j, NULL);
	}
      }
    }
  } else {
    armindex=cpl_calloc(4, sizeof(int));
  }

  // Now find  IMAGING_DATA
  for(i=0;i<nbExtent;i++) {
    plist=cpl_propertylist_load(cpl_frame_get_filename(frame),i+1);
    keyExtname=(char *)cpl_propertylist_get_string(plist,"EXTNAME");
    if (keyExtname != NULL) {
      table=cpl_table_load(cpl_frame_get_filename(frame),i+1,0);
      // Load data in attribute imgdata 
      if (!strcmp(keyExtname,"IMAGING_DATA")) {
	cpl_msg_debug(cpl_func, "loading the IMAGING_DATA extension");
	genData->imgdata = mat_imagingdata_load_skip(plist, table, genData->imgdet, staindex, armindex, nditskip, type);
	cpl_msg_debug(cpl_func, "IMAGING_DATA extension loaded");
	/* Error handling */
        if (genData->imgdata == NULL) { 
          cpl_error_set_message(cpl_func,CPL_ERROR_UNSPECIFIED,
				"Could not load IMAGING_DATA in frame: %s",
				cpl_frame_get_filename(frame));
        }
        else {
	  flagData=1;
        }
	/* heininger:
	   modification for AMBER raw data:
	   since no EXPTIME column is present in some data, the EXPTIME from
	   the promary header keywords is used instead
	*/
	if (genData->imgdata->list_frame[0]->exptime == 0.0)
	  {
	    double exptime = cpl_propertylist_get_double(genData->keywords, "EXPTIME");
	    if (exptime != 0.0)
	      {
		int k;
		for (k = 0; k < genData->imgdata->nbframe; k++)
		  {
		    genData->imgdata->list_frame[k]->exptime = exptime;
		  }
	      }
	  }
	if (flagOpti) {        
	  if (tableopti != NULL) cpl_table_delete(tableopti);
	}
      }
      if (table != NULL) cpl_table_delete(table);
    }
    if (plist != NULL) cpl_propertylist_delete(plist);
  }

  // Check the flags
  /* heininger:
     modification for AMBER raw data:
     the ARRAY_GEOMETRY does not exist
  */
  if ( flagDet==0 || flagData==0) {
    cpl_error_set_message(cpl_func,CPL_ERROR_ILLEGAL_INPUT,
                          "Extension missing in frame: %s",
                          cpl_frame_get_filename(frame));
  }
  
  // Free memory
  if (staindex != NULL)
    {
      cpl_free(staindex);
    }
  if (armindex != NULL)
    {
      cpl_free(armindex);
    }
  if (telname_array != NULL)
    {
      for(i=0; i<genData->array->nbstation; i++) {
	if (telname_array[i] != NULL) cpl_free(telname_array[i]);
      }
      if (telname_array != NULL) cpl_free(telname_array);
    }
  if (telname_opti != NULL)
    {
      for(i=0; i<nbrow_opti; i++) {
	if (telname_opti[i] != NULL) cpl_free(telname_opti[i]);
      }
      if (telname_opti != NULL) cpl_free(telname_opti);
    }
  /* Error handling */
  if (!cpl_errorstate_is_equal(prestate)) { 
    
    cpl_errorstate_dump(prestate, CPL_FALSE, cpl_errorstate_dump_one);
    // Unload the mat_gendata structure
    // TODO MHE: remove the following two comment tokens
    //mat_gendata_free(genData);
    //return NULL;
  }

  return genData;
}

/*----------------------------------------------------------------------------*/
/**
   @ingroup gendata
   @brief Load raw data in mat_gendata structure
   @param frame : current frame
   @param type            The cpl_type used for the loaded CPL images.
   @return mat_gendata : loaded structure
*/
/*-----------------------------------------------------------------------------*/
mat_gendata * mat_gendata_load(cpl_frame *frame, cpl_type type)
{
  return mat_gendata_load_skip(frame, 0, type);
}

/*----------------------------------------------------------------------------*/
/**
   @ingroup gendata
   @brief Save a mat_gendata structure in a FITS file
   @param set_in            input frameset
   @param set_sel           selected frameset
   @param parlist           recipe parameter list
   @param recipe_name       name of recipe
   @param pro_cat           PRO.CATG
   @param pro_tech          PRO.TECH
   @param pro_sci           PRO.SCIENCE
   @param gendata           mat_gendata structure to save
   @param filename          name of the FITS file
   @param type            The cpl_type used for the stored CPL images (converted into a FITS data type).
   @return cpl_error_code
 */
/*-----------------------------------------------------------------------------*/
cpl_error_code mat_gendata_save(
				mat_gendata *gendata,
				cpl_frameset *set_in,
				cpl_frameset *set_sel,
				const cpl_parameterlist *parlist,
				const char *recipe_name,
				const char *pro_cat,
				const char *pro_tech,
				const char *pro_sci,
				char *filename,
				cpl_type type)
{

  cpl_table *gendata_tab=NULL;
  int i=0;
  int j=0;
  cpl_propertylist *plist=NULL;
  cpl_propertylist *applist=NULL;
  cpl_array *map_array=NULL;
  cpl_array *arrDim=NULL;
  char str[50];
  //cpl_errorstate prestate= cpl_errorstate_get();

  (void) pro_sci;
  // Check input parameters
  mat_assert_value((set_in!=NULL),CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT, 
		   "no cpl_frameset (set_in) argument given");
  mat_assert_value((set_sel!=NULL),CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT, 
		   "no cpl_frameset (set_sel) argument given");
  mat_assert_value((filename!=NULL),CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT, 
		   "no filename given");
 if (gendata == NULL){
    return CPL_ERROR_UNSPECIFIED;
  }

  // Remove the keywords from the DPR categories
  cpl_propertylist_erase_regexp(gendata->keywords, "DPR", 0);
  
  // Add the keywords from the PRO categories
  if (pro_tech != NULL) {
    cpl_propertylist_append_string(gendata->keywords,CPL_DFS_PRO_TECH,pro_tech);
  }
  if (pro_cat != NULL) {
    cpl_propertylist_append_string(gendata->keywords,CPL_DFS_PRO_CATG,pro_cat);
  }
  if (pro_sci != NULL) {
    if (strstr(pro_sci,"F") != NULL)
      {
	cpl_propertylist_append_bool(gendata->keywords, CPL_DFS_PRO_SCIENCE, 0);
      }
    else
      {
	cpl_propertylist_append_bool(gendata->keywords, CPL_DFS_PRO_SCIENCE, 1);
      }
  }
  /* if (pro_sci != NULL) { */
  /*     cpl_propertylist_append_string(gendata->keywords,CPL_DFS_PRO_SCIENCE,pro_sci); */
  /* } */
  // Prepare the keywords of the IMAGING_DATA binary table
  plist=cpl_propertylist_new();  
  cpl_propertylist_append_string(plist,"EXTNAME","IMAGING_DATA");
  //cpl_propertylist_append_string(plist,"ORIGIN",gendata->imgdata->origin);
  cpl_propertylist_append_string(plist,"INSTRUME",gendata->imgdata->instrument);
  cpl_propertylist_append_double(plist,"MJD-OBS",gendata->imgdata->dateobsmjd);
  cpl_propertylist_append_string(plist,"DATE-OBS",gendata->imgdata->dateobs);
  //cpl_propertylist_append_string(plist,"DATE",gendata->imgdata->date);
  cpl_propertylist_append_string(plist,"ESO DET DID",
				 gendata->imgdata->dcsdictionaryid);
  cpl_propertylist_append_string(plist,"ESO DET ID",gendata->imgdata->dcsid);
  cpl_propertylist_append_int(plist,"NREGION",gendata->imgdet->nbregion);
  cpl_propertylist_append_int(plist,"MAXTEL",gendata->imgdata->nbtel);
  cpl_propertylist_append_int(plist,"MAXSTEP",gendata->imgdata->maxstep);
  cpl_propertylist_append_int(plist,"MAXINS",gendata->imgdata->maxins);
  // Fill a cpl_table with the gendata.imgdata structure
  gendata_tab = cpl_table_new(gendata->imgdata->nbframe);


  cpl_table_new_column(gendata_tab, "TIME", CPL_TYPE_DOUBLE);
  for(i=0;i<gendata->imgdet->nbregion;i++) {
    snprintf(str,50,"DATA%d",i+1);
    cpl_table_new_column_array(gendata_tab,str, type,
			       gendata->imgdet->list_region[i]->naxis[0]*
			       gendata->imgdet->list_region[i]->naxis[1]);
    arrDim=cpl_array_new(2,CPL_TYPE_INT);
    cpl_array_set(arrDim,0,gendata->imgdet->list_region[i]->naxis[0]);
    cpl_array_set(arrDim,1,gendata->imgdet->list_region[i]->naxis[1]);
    cpl_table_set_column_dimensions(gendata_tab,str,arrDim);
    cpl_array_delete(arrDim);
  }
  cpl_table_new_column(gendata_tab, "EXPTIME", CPL_TYPE_FLOAT);
  cpl_table_new_column_array(gendata_tab,"OPD", CPL_TYPE_DOUBLE,
			       gendata->imgdet->nbtel);
  cpl_table_new_column_array(gendata_tab,"LOCALOPD", CPL_TYPE_DOUBLE,
			       gendata->imgdet->nbtel);
  cpl_table_new_column(gendata_tab,"INS_TRAIN",CPL_TYPE_INT);
  cpl_table_new_column(gendata_tab, "ROTATION", CPL_TYPE_DOUBLE);
  cpl_table_new_column_array(gendata_tab,"OFFSET", CPL_TYPE_INT,2);
  cpl_table_new_column(gendata_tab, "STEPPING_PHASE", CPL_TYPE_INT);
  cpl_table_new_column(gendata_tab, "TARTYP", CPL_TYPE_STRING);
  cpl_table_new_column(gendata_tab, "REFERENCE", CPL_TYPE_INT);
  cpl_table_new_column(gendata_tab, "TARGET", CPL_TYPE_INT);

  for(i=0;i<gendata->imgdata->nbframe;i++) {
    cpl_table_set_double(gendata_tab,"TIME",i,
			 gendata->imgdata->list_frame[i]->time);
    for(j=0;j<gendata->imgdet->nbregion;j++) {
      cpl_image *img = gendata->imgdata->list_frame[i]->list_subwin[j]->imgreg[0];
      int        nx = cpl_image_get_size_x(img);
     int        ny = cpl_image_get_size_y(img);
      switch (cpl_image_get_type(img))
	{
	case CPL_TYPE_INT:
	  map_array = cpl_array_wrap_int((int*)cpl_image_get_data(img), nx*ny);
	  break;
	case CPL_TYPE_FLOAT:
	  map_array = cpl_array_wrap_float((float*)cpl_image_get_data(img), nx*ny);
	  break;
	case CPL_TYPE_DOUBLE:
	  map_array = cpl_array_wrap_double((double*)cpl_image_get_data(img), nx*ny);
	  break;
	default:;
	  cpl_msg_error(cpl_func,"Invalid data type");
	  return CPL_ERROR_TYPE_MISMATCH;
	}
      /*
      map = (double *)cpl_image_get_data(gendata->imgdata->list_frame[i]
					  ->list_subwin[j]->imgreg[0]);

      map_array = cpl_array_wrap_double
	(map,gendata->imgdet->list_region[j]->naxis[0]*
	 gendata->imgdet->list_region[j]->naxis[1]);
      */
      snprintf(str,50,"DATA%d",j+1);
      cpl_table_set_array(gendata_tab,str,i,map_array);
      cpl_array_unwrap(map_array);
    }
    cpl_table_set_float(gendata_tab,"EXPTIME",i,
			 gendata->imgdata->list_frame[i]->exptime); 
    
    map_array = cpl_array_wrap_double
      (gendata->imgdata->list_frame[i]->opd,
       gendata->imgdata->nbtel);
    cpl_table_set_array(gendata_tab,"OPD",i,map_array);
    cpl_array_unwrap(map_array);
    map_array = cpl_array_wrap_double
      (gendata->imgdata->list_frame[i]->localopd,
       gendata->imgdata->nbtel);
   cpl_table_set_array(gendata_tab,"LOCALOPD",i,map_array);
    cpl_array_unwrap(map_array);
    
    cpl_table_set_int(gendata_tab,"INS_TRAIN",i,0);

   
    cpl_table_set_double(gendata_tab,"ROTATION",i,0.0); 
    map_array = cpl_array_new(2,CPL_TYPE_INT);
    cpl_array_set(map_array,0,0.);
    cpl_array_set(map_array,1,0.);
    cpl_table_set_array(gendata_tab,"OFFSET",i,map_array);
    cpl_array_delete(map_array);
    
    cpl_table_set_int(gendata_tab,"STEPPING_PHASE",i,
		      gendata->imgdata->list_frame[i]->stepphase);   
   cpl_table_set_string(gendata_tab,"TARTYP",i,
			 gendata->imgdata->list_frame[i]->tartype);  
   cpl_table_set_int(gendata_tab,"REFERENCE",i,0); 
    cpl_table_set_int(gendata_tab,"TARGET",i,0);
  }
  // Make a new FITS file with gendata keywords and the created table
  applist=cpl_propertylist_duplicate(gendata->keywords);

  cpl_propertylist_erase(gendata->keywords,"RADECSYS");
  cpl_propertylist_erase(plist,"RADECSYS");
  cpl_dfs_save_table(set_in, gendata->keywords, parlist, set_sel, NULL,
  		     gendata_tab, plist, recipe_name, applist, NULL,
		     PACKAGE "/" PACKAGE_VERSION, filename);
  //cpl_table_save(gendata_tab, gendata->keywords, plist, filename, CPL_IO_CREATE);
  // Append the IMAGING_DETECTOR table
  
  /* mat_imagingdetector_save(gendata->imgdet,gendata->keywords,filename); */
  /* // Append the ARRAY table */
  /* mat_array_save(gendata->array,gendata->keywords,filename,1); */
  // Free memory
  if (gendata_tab != NULL) cpl_table_delete(gendata_tab);
  if (plist != NULL) cpl_propertylist_delete(plist);
  if (applist != NULL) cpl_propertylist_delete(applist);

  /*if (!cpl_errorstate_is_equal(prestate)) { 
    cpl_errorstate_set(prestate);
    
    return CPL_ERROR_UNSPECIFIED;
    }*/

  return CPL_ERROR_NONE;

}



/*----------------------------------------------------------------------------*/
/**
   @ingroup gendata
   @brief Add the imaging data of a gendata at the end of existing FITS file
   @param gendata           mat_gendata structure to save
   @param filename          name of the FITS file
   @param type              type of the DATA column
   @return cpl_error_code
 */
/*-----------------------------------------------------------------------------*/
cpl_error_code mat_gendata_append(mat_gendata *gendata,
				  char *filename,
				  cpl_type type)
{
  cpl_propertylist *plist=NULL;
  cpl_table *gendata_tab=NULL;
  int i=0;
  int j=0;
  cpl_array *map_array=NULL;
  cpl_array *arrDim=NULL;
  char str[50];
  //cpl_errorstate prestate= cpl_errorstate_get();

  // Prepare the keywords of the IMAGING_DATA binary table
  plist=cpl_propertylist_new();  
  cpl_propertylist_append_string(plist,"EXTNAME ","IMAGING_DATA");
  // Fill a cpl_table with the gendata.imgdata structure
  gendata_tab = cpl_table_new(gendata->imgdata->nbframe);


  cpl_table_new_column(gendata_tab, "TIME", CPL_TYPE_DOUBLE);
  for(i=0;i<gendata->imgdet->nbregion;i++) {
    snprintf(str,50,"DATA%d",i+1);
    cpl_table_new_column_array(gendata_tab,str, type,
			       gendata->imgdet->list_region[i]->naxis[0]*
			       gendata->imgdet->list_region[i]->naxis[1]);
    arrDim=cpl_array_new(2,CPL_TYPE_INT);
    cpl_array_set(arrDim,0,gendata->imgdet->list_region[i]->naxis[0]);
    cpl_array_set(arrDim,1,gendata->imgdet->list_region[i]->naxis[1]);
    cpl_table_set_column_dimensions(gendata_tab,str,arrDim);
    cpl_array_delete(arrDim);
  }
  cpl_table_new_column(gendata_tab, "EXPTIME", CPL_TYPE_FLOAT);
  cpl_table_new_column_array(gendata_tab,"OPD", CPL_TYPE_DOUBLE,
			       gendata->imgdet->nbtel);
  cpl_table_new_column_array(gendata_tab,"LOCALOPD", CPL_TYPE_DOUBLE,
			       gendata->imgdet->nbtel);
  cpl_table_new_column(gendata_tab,"INS_TRAIN",CPL_TYPE_INT);
  cpl_table_new_column(gendata_tab, "ROTATION", CPL_TYPE_DOUBLE);
  cpl_table_new_column_array(gendata_tab,"OFFSET", CPL_TYPE_INT,2);
  cpl_table_new_column(gendata_tab, "STEPPING_PHASE", CPL_TYPE_INT);
  cpl_table_new_column(gendata_tab, "TARTYP", CPL_TYPE_STRING);
  cpl_table_new_column(gendata_tab, "REFERENCE", CPL_TYPE_INT);
  cpl_table_new_column(gendata_tab, "TARGET", CPL_TYPE_INT);


  for(i=0;i<gendata->imgdata->nbframe;i++) {
    cpl_table_set_double(gendata_tab,"TIME",i,
			 gendata->imgdata->list_frame[i]->time);
    for(j=0;j<gendata->imgdet->nbregion;j++) {
      cpl_image *img = gendata->imgdata->list_frame[i]->list_subwin[j]->imgreg[0];
      int        nx = cpl_image_get_size_x(img);
      int        ny = cpl_image_get_size_y(img);
      switch (cpl_image_get_type(img))
	{
	case CPL_TYPE_INT:
	  map_array = cpl_array_wrap_int((int*)cpl_image_get_data(img), nx*ny);
	  break;
	case CPL_TYPE_FLOAT:
	  map_array = cpl_array_wrap_float((float*)cpl_image_get_data(img), nx*ny);
	  break;
	case CPL_TYPE_DOUBLE:
	  map_array = cpl_array_wrap_double((double*)cpl_image_get_data(img), nx*ny);
	  break;
	default:;
	}
      snprintf(str,50,"DATA%d",j+1);
      cpl_table_set_array(gendata_tab,str,i,map_array);
      cpl_array_unwrap(map_array);
    }
    cpl_table_set_float(gendata_tab,"EXPTIME",i,
			 gendata->imgdata->list_frame[i]->exptime); 
    
    map_array = cpl_array_wrap_double
      (gendata->imgdata->list_frame[i]->opd,
       gendata->imgdata->nbtel);
    cpl_table_set_array(gendata_tab,"OPD",i,map_array);
    cpl_array_unwrap(map_array);
    map_array = cpl_array_wrap_double
      (gendata->imgdata->list_frame[i]->localopd,
       gendata->imgdata->nbtel);
    cpl_table_set_array(gendata_tab,"LOCALOPD",i,map_array);
    cpl_array_unwrap(map_array);
    
    cpl_table_set_int(gendata_tab,"INS_TRAIN",i,0);

    
    cpl_table_set_double(gendata_tab,"ROTATION",i,0.0); 
    map_array = cpl_array_new(2,CPL_TYPE_INT);
    cpl_array_set(map_array,0,0.);
    cpl_array_set(map_array,1,0.);
    cpl_table_set_array(gendata_tab,"OFFSET",i,map_array);
    cpl_array_delete(map_array);
    
    cpl_table_set_int(gendata_tab,"STEPPING_PHASE",i,
		      gendata->imgdata->list_frame[i]->stepphase);   
    cpl_table_set_string(gendata_tab,"TARTYP",i,
			 gendata->imgdata->list_frame[i]->tartype);  
    cpl_table_set_int(gendata_tab,"REFERENCE",i,0); 
    cpl_table_set_int(gendata_tab,"TARGET",i,0);
  }

  cpl_propertylist_erase(plist,"RADECSYS");
  cpl_table_save(gendata_tab,NULL,plist,filename,CPL_IO_APPEND);

  if (gendata_tab != NULL) cpl_table_delete(gendata_tab);
  if (plist != NULL) cpl_propertylist_delete(plist);

  /*if (!cpl_errorstate_is_equal(prestate)) { 
    
    cpl_errorstate_set(prestate);
    
    return CPL_ERROR_UNSPECIFIED;
    }*/


  return CPL_ERROR_NONE;

}



