/* $Id: mat_imagingdata.c,v0.5 2014-06-15 12:56:21 mheininger Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: mheininger $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_imagingdata.c $
 */

#include <stdlib.h>
#include <stdio.h>

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include <string.h>

#include "mat_error.h"
#include "mat_utils.h"
#include "mat_imagingdata.h"

/*-----------------------------------------------------------------------------
                                   Define
 -----------------------------------------------------------------------------*/


/*-----------------------------------------------------------------------------
                                   Functions
 -----------------------------------------------------------------------------*/

/**
   @ingroup imgdata
   @brief Creates a new mat_inagingdata data structure.
   @param imgdet    The sub-window setup as mat_imagingdetector data structure.
   @param nbframe   The number of frames.
   @param type      The cpl_type for the new images.

   This function creates a new mat_imagingdata data structure which follows a certain
   sub-window setup specified as parameter(imgdet). The new data structure will contain
   a given number of frames, each following the sub-window setup. All CPL images allocated
   for all frames and sub-windows will have the specified data type.
 */
mat_imagingdata *mat_imagingdata_new(const mat_imagingdetector *imgdet, int nbframe, cpl_type type)
{
  mat_imagingdata   *imgdat = NULL;
  int                i;

  mat_assert_value((imgdet != NULL), CPL_ERROR_NULL_INPUT, NULL, "no mat_imagingdetector (imgdet) argument given");
  imgdat = (mat_imagingdata*)cpl_calloc(1, sizeof(mat_imagingdata));
  if (imgdat == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for mat_imagingdata");
      return NULL;
    }
  //imgdat->origin = cpl_strdup(imgdet->origin);
  imgdat->instrument = cpl_strdup(imgdet->instrument);
  imgdat->dateobsmjd = imgdet->dateobsmjd;
  imgdat->dateobs = cpl_strdup(imgdet->dateobs);
  //imgdat->date = cpl_strdup(imgdet->date);
  imgdat->dcsdictionaryid = cpl_strdup(imgdet->dcsdictionaryid);
  imgdat->dcsid = cpl_strdup(imgdet->dcsid);
  imgdat->nbtel = imgdet->nbtel;
  for (i = 0; i < MAT_MAXTEL; i++)
    {
      imgdat->staindex[i] = 0;
      imgdat->armindex[i] = 0;
    }
  imgdat->maxstep = 0;
  imgdat->maxins = 0;
  imgdat->nbframe = nbframe;
  imgdat->list_frame = (mat_frame **)cpl_calloc(nbframe, sizeof(mat_frame *));
  if (imgdat->list_frame == NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for mat_imagingdata->list_frame");
      mat_imagingdata_delete(imgdat);
      return NULL;
    }
  for (i = 0; i < nbframe; i++)
    {
      imgdat->list_frame[i] = mat_frame_new(imgdet, type);
    }
  return imgdat;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup imgdata
  @brief delete structure mat_imagingdata
  @param imgdat           current structure
  @return error code
 */
/*----------------------------------------------------------------------------*/
cpl_error_code mat_imagingdata_delete(mat_imagingdata *imgdat)
{
  int i;

  // Check input parameters
  mat_assert_value((imgdat!=NULL),CPL_ERROR_NULL_INPUT, CPL_ERROR_NULL_INPUT,"no mat_imagingdata (imgdat) argument given");
  mat_free(imgdat->instrument);
  mat_free(imgdat->dateobs);
  mat_free(imgdat->dcsdictionaryid);
  mat_free(imgdat->dcsid);
  if (imgdat->list_frame != NULL)
    {
      for (i = 0; i < imgdat->nbframe; i++)
  	{
  	  if (imgdat->list_frame[i] != NULL)
  	    {
  	      mat_frame_delete(imgdat->list_frame[i]);
  	      imgdat->list_frame[i] = NULL;
  	    }
  	}
      mat_free(imgdat->list_frame);
    }
  imgdat->nbframe = 0;
  cpl_free(imgdat);
  return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
   @ingroup imgdata
   @brief Duplicate a mat_imagingdata structure
   @param imgdata    mat_imagingdata structure to duplicate
   @param type      The cpl_type for the new images.
   @return mat_imagingdata : duplicated structure
 */
/*----------------------------------------------------------------------------*/
mat_imagingdata * mat_imagingdata_duplicate(const mat_imagingdata *imgdata, cpl_type type)
{
  mat_imagingdata * imagingData=NULL;
  int i=0;
/*   int j=0; */
/*   int k=0; */
/*   int l=0; */
/*   int m=0; */
/*   int index=0; */
/*   int nbsubwin=0; */
/*   int dimX=0; */
/*   int dimY=0; */
/*   cpl_array* columnNames=NULL; */

  cpl_errorstate prestate = cpl_errorstate_get();

  if (imgdata == NULL){
    return NULL;
  }

// Pointers allocation
  imagingData=cpl_calloc(1, sizeof(mat_imagingdata));
  if (imagingData==NULL) {
    cpl_msg_error(cpl_func,"could not allocate memory for imagingData");
    return NULL;
  }
  //imagingData->origin=cpl_strdup(imgdata->origin);
  imagingData->instrument=cpl_strdup(imgdata->instrument);
  imagingData->dateobs=cpl_strdup(imgdata->dateobs);
  //imagingData->date=cpl_strdup(imgdata->date);
  imagingData->dcsdictionaryid=cpl_strdup(imgdata->dcsdictionaryid);
  imagingData->dcsid=cpl_strdup(imgdata->dcsid);
  (*imagingData).dateobsmjd= (*imgdata).dateobsmjd;
  (*imagingData).nbtel= (*imgdata).nbtel;
  (*imagingData).maxins= (*imgdata).maxins;
  (*imagingData).maxstep= (*imgdata).maxstep;
  (*imagingData).nbframe= (*imgdata).nbframe;

  for(i=0;i<(*imagingData).nbtel;i++)
    {
      (*imagingData).staindex[i]= (*imgdata).staindex[i];
      (*imagingData).armindex[i] = (*imgdata).armindex[i];
    }

  // Copy frames
  (*imagingData).list_frame=cpl_calloc(imagingData->nbframe, sizeof(mat_frame *));
  if (imagingData->list_frame==NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for imagingData->list_frame");
      mat_imagingdata_delete(imagingData);
      return NULL;
    }
  for (i=0;i<(*imagingData).nbframe;i++)
    {
      (*imagingData).list_frame[i]= 
	mat_frame_duplicate((*imgdata).list_frame[i], type);
    }

  /* Error handling */
  if (!cpl_errorstate_is_equal(prestate))
    { 
      cpl_errorstate_dump(prestate, CPL_FALSE, cpl_errorstate_dump_one);
      mat_imagingdata_delete(imagingData);
      return NULL;
    }

  return imagingData;
}

/*----------------------------------------------------------------------------*/
/**
   @ingroup imgdata
   @brief Load raw data into mat_imagingdata structure skipping the nditskip first frames.
   @param plist     current cpl_propertylist
   @param table     current cpl_table
   @param imgdet    mat_imagingdetector structure
   @param staindex  array of int representing the telescopes index coming from the ARRAY binary table
   @param armindex  array of int representing teh VLTI arms coming from the OPTICAL_TRAIN binary table
   @param nditskip  number of frames to skip
   @param type      The cpl_type for the new images.
   @return mat_imagingdata
 */
/*----------------------------------------------------------------------------*/
mat_imagingdata * mat_imagingdata_load_skip(cpl_propertylist *plist,
					    cpl_table *table,
					    mat_imagingdetector *imgdet,
					    int *staindex,
					    int *armindex,
					    int nditskip,
					    cpl_type type)
{
  mat_imagingdata *imagingData=NULL;
  int i=0;
  int j=0;
  int k=0;
  int l=0;
  int m=0;
  int idx=0;
  int nbsubwin=0;
  int *nbimgreg=NULL;
  int dimX=0;
  int dimY=0;
  int has_exptime=0;
  int has_opd=0;
  int has_localopd=0;
  int has_stepping_phase=0;
  int has_tartyp=0;
  char str[32];
  cpl_array* columnNames=NULL;
  cpl_array *arr=NULL;
  cpl_type ttype;

  cpl_errorstate prestate = cpl_errorstate_get();
  //si table vide
  if (table == NULL)
    {
      cpl_msg_info(cpl_func, "mat_imagingdata_load_skip() no binary table");
      return NULL;
    }
  if (imgdet == NULL)
    {
      cpl_msg_info(cpl_func, "mat_imagingdata_load_skip() no imaging detector table");
      return NULL;
    } 
  if (plist == NULL)
    {
      cpl_msg_info(cpl_func, "mat_imagingdata_load_skip() no property list");
      return NULL;
    }
  // Pointers allocation
  imagingData=cpl_calloc(1, sizeof(mat_imagingdata));
  if (imagingData==NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for imagingData");
      return NULL;
    }
  // Set data from IMAGING_DATA header
  //cpl_msg_info(cpl_func, "  reading the header keywords");
  //(*imagingData).origin = mat_propertylist_copy_string(plist, "ORIGIN");
  (*imagingData).instrument = mat_propertylist_copy_string(plist, "INSTRUME");
  (*imagingData).dateobsmjd=cpl_propertylist_get_double(plist,"MJD-OBS");
  (*imagingData).dateobs = mat_propertylist_copy_string(plist, "DATE-OBS");
  //(*imagingData).date = mat_propertylist_copy_string(plist, "DATE");
  (*imagingData).dcsdictionaryid = mat_propertylist_copy_string(plist, "ESO DET DID");
  (*imagingData).dcsid = mat_propertylist_copy_string(plist, "ESO DET ID");
  (*imagingData).nbtel=cpl_propertylist_get_int(plist,"MAXTEL");
  (*imagingData).maxins=cpl_propertylist_get_int(plist,"MAXINS");
  (*imagingData).maxstep=cpl_propertylist_get_int(plist,"MAXSTEP");
  // Set data from mat_array parameter
  //cpl_msg_info(cpl_func, "  using the station index array");
  /* heininger:
     modification for AMBER raw data:
     the station index needs the ARRAY_GEOMETRY extension which is not
     present in some AMBER data (calibration measurements) and therefore
     the staindex pointer is NULL.
  */

  if (staindex == NULL)
    {
      for(i=0;i<(*imagingData).nbtel;i++) {
	(*imagingData).staindex[i]= i;
      }
    }
  else
    {
      for(i=0;i<(*imagingData).nbtel;i++) {
	(*imagingData).staindex[i]= staindex[i];
      }
    }
  // Set data from OPTICAL_TRAIN table
  //cpl_msg_info(cpl_func, "  using the array index array");
  if (armindex == NULL)
    {
      for(i=0;i<(*imagingData).nbtel;i++)
	{
	  (*imagingData).armindex[i] = i;
	}
    }
  else
    {
      for(i=0;i<(*imagingData).nbtel;i++)
	{
	  (*imagingData).armindex[i] = armindex[i];
	}
    }
  // Set data from IMAGING_DATA table
  //cpl_msg_info(cpl_func, "  sarching for DATA columns");
  if (nditskip < cpl_table_get_nrow(table))
    {
      (*imagingData).nbframe=cpl_table_get_nrow(table) - nditskip;
    }
  else
    {
      (*imagingData).nbframe=cpl_table_get_nrow(table);
    }


  /* (*imagingData).nbframe=10; */






  columnNames = cpl_table_get_column_names(table);
  for(i=0;i<cpl_table_get_ncol(table);i++)
    {
      if (strstr(cpl_array_get_string(columnNames, i), "DATA") !=NULL)
	{
	  nbsubwin++;
	}
    }
  cpl_array_delete(columnNames);
  // Find the number of imgreg in each subwindow
  //cpl_msg_info(cpl_func, "  sarching for the regions for each DATA column (must be 1!)");
  nbimgreg= cpl_calloc(nbsubwin, sizeof(int));
  for(i=0; i<nbsubwin; i++)
    {
      int ret = snprintf(str,sizeof(str), "DATA%d", i+1);
      if (ret < 0) {
	cpl_msg_error(cpl_func,"could not write column name");
	mat_imagingdata_delete(imagingData);
	mat_free(nbimgreg);
	return NULL;
      }
      dimX= (*imgdet).list_region[i]->naxis[0];
      dimY= (*imgdet).list_region[i]->naxis[1];
      /* REMARK: nbimgreg[i] must be exactly 1 or the data format is wrong! */
      nbimgreg[i]= cpl_array_get_size(cpl_table_get_array(table, str, 0)) / (dimX*dimY);
    }
  // Copy frames
  has_exptime        = cpl_table_has_column(table, "EXPTIME");
  has_opd            = cpl_table_has_column(table, "OPD");
  has_localopd       = cpl_table_has_column(table, "LOCALOPD");
  has_stepping_phase = cpl_table_has_column(table, "STEPPING_PHASE");
  has_tartyp         = cpl_table_has_column(table, "TARTYP");
  cpl_msg_debug(cpl_func, "try to allocate memory for %d mat_frame (%zu)", imagingData->nbframe, sizeof(mat_frame *));
  (*imagingData).list_frame=cpl_calloc(imagingData->nbframe, sizeof(mat_frame *));
  if (imagingData->list_frame==NULL)
    {
      cpl_msg_error(cpl_func,"could not allocate memory for imagingDatalist_frame");
      mat_imagingdata_delete(imagingData);
      mat_free(nbimgreg);
      return NULL;
    }
  /* REMARK:
     i is the index into the data structure
     the row number of the table is i + nditskip
  */
  for (i=0;i<(*imagingData).nbframe;i++)
    {
      int row = i + nditskip;
      /*
      if (i%100 == 0)
	{
	  cpl_msg_info(cpl_func, " frame %d, row %d", i, row);
	}
      */
      (*imagingData).list_frame[i]=cpl_calloc(1, sizeof(mat_frame));
      if (imagingData->list_frame[i]==NULL)
	{
	  cpl_msg_error(cpl_func,"could not allocate memory for imagingDatalist_frame[i]");
	  mat_imagingdata_delete(imagingData);
	  mat_free(nbimgreg);
	  return NULL;
	}
      (*imagingData).list_frame[i]->time=cpl_table_get(table,"TIME",row,NULL);
      if (has_exptime)
	{
	  imagingData->list_frame[i]->exptime=cpl_table_get_float(table,"EXPTIME",row,NULL);
	}
      else
	{
	  imagingData->list_frame[i]->exptime = 0.0;
	}
      /*
	(*imagingData).list_frame[i]->opd=cpl_calloc((*imagingData).nbtel,sizeof(double));
	(*imagingData).list_frame[i]->localopd=cpl_calloc((*imagingData).nbtel,sizeof(double));
      */
      /* heininger:
	 modification for AMBER raw data:
	 some AMBER data (calibration measurements) do not have a OPD, LOCALOPD, ... column
      */
      for (j=0;j<(*imagingData).nbtel;j++)
	{
	  if (has_opd)
	    {
	      (*imagingData).list_frame[i]->opd[j]=
		cpl_array_get_double(cpl_table_get_array(table,"OPD",row),j,NULL);
	    }
	  else
	    {
	      imagingData->list_frame[i]->opd[j] = 0.0;
	    }
	  if (has_localopd)
	    {
	      (*imagingData).list_frame[i]->localopd[j]=
		cpl_array_get_double(cpl_table_get_array(table,"LOCALOPD",row),j,NULL);
	    }
	  else
	    {
	      imagingData->list_frame[i]->localopd[j] = 0.0;
	    }
	}
     if (has_stepping_phase)
	{
	  (*imagingData).list_frame[i]->stepphase=
	    cpl_table_get_int(table,"STEPPING_PHASE",row,NULL);
	}
      else
	{
	  imagingData->list_frame[i]->stepphase = 0;
	}
      if (has_tartyp)
	{
	  char* tartype = (char *)cpl_table_get_string(table,"TARTYP",row);
	  strncpy((*imagingData).list_frame[i]->tartype , tartype,2-1);(*imagingData).list_frame[i]->tartype[2-1] = '\0';
	}
      else
	{
	  strncpy((*imagingData).list_frame[i]->tartype, "B", 2);
	}
      (*imagingData).list_frame[i]->nbsubwin=nbsubwin;
      // Copy sub windows
      (*imagingData).list_frame[i]->list_subwin=
	cpl_calloc(imagingData->list_frame[i]->nbsubwin, sizeof(mat_imgreg *));
      if (imagingData->list_frame[i]->list_subwin==NULL)
	{
	  cpl_msg_error(cpl_func,"could not allocate memory for imagingDatalist_frame[i]->list_subwin");
	  mat_imagingdata_delete(imagingData);
	  mat_free(nbimgreg);
	  return NULL;
	}
      for(j=0;j<(*imagingData).list_frame[i]->nbsubwin;j++)
	{
	  int ret;
	  (*imagingData).list_frame[i]->list_subwin[j]=cpl_calloc(1, sizeof(mat_imgreg));
	  if (imagingData->list_frame[i]->list_subwin[j]==NULL)
	    {
	      cpl_msg_error(cpl_func,"could not allocate memory for imagingDatalist_frame[i]->list_subwin[j]");
	      mat_imagingdata_delete(imagingData);
	      mat_free(nbimgreg);
	      return NULL;
	    }
	  ret = snprintf(str,sizeof(str), "DATA%d", j+1);
	  if (ret < 0) {
	    cpl_msg_error(cpl_func,"could not write column name");
	    mat_imagingdata_delete(imagingData);
	    mat_free(nbimgreg);
	    return NULL;
	  }
	  (*imagingData).list_frame[i]->list_subwin[j]->numregion=j+1;
	  (*imagingData).list_frame[i]->list_subwin[j]->numdetector= 
	    (*imgdet).list_region[j]->numdetector;
	  (*imagingData).list_frame[i]->list_subwin[j]->nbimgreg=nbimgreg[j];
	  dimX= (*imgdet).list_region[j]->naxis[0];
	  dimY= (*imgdet).list_region[j]->naxis[1];
	  (*imagingData).list_frame[i]->list_subwin[j]->imgreg= cpl_calloc((*imagingData).list_frame[i]->list_subwin[j]->nbimgreg, sizeof(cpl_image*));
	  // Copy images
	  for(k=0;k<(*imagingData).list_frame[i]->list_subwin[j]->nbimgreg;k++)
	    {
	      idx=0;
	      arr=(cpl_array *)cpl_table_get_array(table, str, row);
	      ttype=cpl_array_get_type(arr);
	      imagingData->list_frame[i]->list_subwin[j]->imgreg[k] = cpl_image_new(dimX, dimY, type);
	      for(m=0;m<dimY;m++)
		{
		  for(l=0;l<dimX;l++)
		    {
		      if (ttype == CPL_TYPE_INT)
			{
			  cpl_image_set((*imagingData).list_frame[i]->list_subwin[j]->imgreg[k],l+1,m+1,
					(double)cpl_array_get_int(arr,idx,NULL));
			}
		      else if (ttype == CPL_TYPE_FLOAT)
			{
			  cpl_image_set((*imagingData).list_frame[i]->list_subwin[j]->imgreg[k],l+1,m+1,
					(double)cpl_array_get_float(arr,idx,NULL));
			}
		      else if (ttype == CPL_TYPE_DOUBLE)
			{
			  cpl_image_set((*imagingData).list_frame[i]->list_subwin[j]->imgreg[k],l+1,m+1,
					cpl_array_get_double(arr,idx,NULL));
			}
		      idx++;
		    }
		}
	    }
	}
    }

  /*cpl_image_save((*imagingData).list_frame[0]->list_subwin[2]->imgreg[0],
    "test_save.fits", CPL_BPP_IEEE_FLOAT, 
    plist, CPL_IO_CREATE); */
  mat_free(nbimgreg);
  /* Error handling */ 
  if (!cpl_errorstate_is_equal(prestate))
    { 
      cpl_errorstate_dump(prestate, CPL_FALSE, cpl_errorstate_dump_one);
      // TODO MHE: remove the following two comment tokens
      //cpl_free(imagingData);
      //return NULL;
    }
  return imagingData;
}

/*----------------------------------------------------------------------------*/
/**
  @ingroup imgdata
  @brief Load raw data into mat_imagingdata structure
  @param plist           current cpl_propertylist
  @param table           current cpl_table
  @param imgdet          mat_imagingdetector structure
  @param staindex        array of int representing the telescopes index coming 
  from the ARRAY binary table
  @param armindex        array of int representing teh VLTI arms coming from the
  OPTICAL_TRAIN binary table
   @param type      The cpl_type for the new images.
  @return mat_imagingdata
 */
/*----------------------------------------------------------------------------*/
mat_imagingdata * mat_imagingdata_load(cpl_propertylist *plist,
                                       cpl_table *table,
                                       mat_imagingdetector *imgdet,
                                       int *staindex,
                                       int *armindex,
				       cpl_type type)
{
  return mat_imagingdata_load_skip(plist, table, imgdet, staindex, armindex, 0, type);
}
 
/*----------------------------------------------------------------------------*/
/**
  @ingroup imgdata
  @brief This function sums all frames of each sub-window of a IMAGING_DATA
  binary table and saves the results ina  new mat_imaging data structure.
  @param imgdata          mat_imagingdata structure to sum
   @param type      The cpl_type for the new images.
  @return mat_imagingdata
 */
/*----------------------------------------------------------------------------*/
mat_imagingdata * mat_imagingdata_sum(const mat_imagingdata * imgdata, cpl_type type)
{
  mat_imagingdata * imagingData=NULL;
  int i=0;
  int j=0;
  int k=0;

  cpl_errorstate prestate = cpl_errorstate_get();

  if(imgdata == NULL){
    return NULL;
  }

  imagingData=cpl_calloc(1, sizeof(mat_imagingdata));
  if (imagingData==NULL) {
    cpl_msg_error(cpl_func,"could not allocate memory for imagingData");
    return NULL;
  }

  //imagingData->origin=cpl_strdup(imgdata->origin);
  imagingData->instrument=cpl_strdup(imgdata->instrument);
  imagingData->dateobs=cpl_strdup(imgdata->dateobs);
  //imagingData->date=cpl_strdup(imgdata->date);
  imagingData->dcsdictionaryid=cpl_strdup(imgdata->dcsdictionaryid);
  imagingData->dcsid=cpl_strdup(imgdata->dcsid);

  (*imagingData).nbframe=1;
  /* strcpy((*imagingData).origin,(*imgdata).origin); */
  /* strcpy((*imagingData).instrument,(*imgdata).instrument); */
  (*imagingData).dateobsmjd=(*imgdata).dateobsmjd;
  /* strcpy((*imagingData).dateobs,(*imgdata).dateobs); */
  /* strcpy((*imagingData).date,(*imgdata).date); */
  /* strcpy((*imagingData).dcsdictionaryid,(*imgdata).dcsdictionaryid); */
  /* strcpy((*imagingData).dcsid,(*imgdata).dcsid); */
  (*imagingData).nbtel=(*imgdata).nbtel;
  (*imagingData).maxstep=(*imgdata).maxstep;
  (*imagingData).maxins=(*imgdata).maxins;
  for(j=0;j<(*imagingData).nbtel;j++) {
    (*imagingData).staindex[j]= (*imgdata).staindex[j];
  }
  for(j=0;j<(*imagingData).nbtel;j++) {
    (*imagingData).armindex[j] = (*imgdata).armindex[j];
  }
  (*imagingData).list_frame=cpl_calloc(1, sizeof(mat_frame *));
  if (imagingData->list_frame==NULL) {
    cpl_msg_error(cpl_func,"could not allocate memory for imagingData->list_frame");
    mat_imagingdata_delete(imagingData);
    return NULL;
  }
  (*imagingData).list_frame[0]=cpl_calloc(1, sizeof(mat_frame));
  if (imagingData->list_frame[0]==NULL) {
    cpl_msg_error(cpl_func,"could not allocate memory for imagingData->list_frame[0]");
    mat_imagingdata_delete(imagingData);
    return NULL;
  }
  (*imagingData).list_frame[0]->time=(*imgdata).list_frame[0]->time;
  (*imagingData).list_frame[0]->exptime=(*imgdata).list_frame[0]->exptime;
  /*
  (*imagingData).list_frame[0]->opd=cpl_calloc((*imagingData).nbtel,sizeof(double));
  (*imagingData).list_frame[0]->localopd=cpl_calloc((*imagingData).nbtel,sizeof(double));
  */
  for (j=0;j<(*imagingData).nbtel;j++) {
    (*imagingData).list_frame[0]->opd[j]=(*imgdata).list_frame[0]->opd[j];
    (*imagingData).list_frame[0]->localopd[j]=(*imgdata).list_frame[0]->localopd[j];
  }
  (*imagingData).list_frame[0]->stepphase=(*imgdata).list_frame[0]->stepphase;
  strncpy((*imagingData).list_frame[0]->tartype,(*imgdata).list_frame[0]->tartype,2);
  (*imagingData).list_frame[0]->nbsubwin=(*imgdata).list_frame[0]->nbsubwin;
  (*imagingData).list_frame[0]->list_subwin=
    cpl_calloc(imagingData->list_frame[0]->nbsubwin, sizeof(mat_imgreg *));
  if (imagingData->list_frame[0]->list_subwin==NULL) {
    cpl_msg_error(cpl_func,"could not allocate memory for imagingData->list_frame[0]->list_subwin");
    mat_imagingdata_delete(imagingData);
    return NULL;
  }

  for(j=0;j<(*imagingData).list_frame[0]->nbsubwin;j++) {
    (*imagingData).list_frame[0]->list_subwin[j]=cpl_calloc(1, sizeof(mat_imgreg));
    if (imagingData->list_frame[0]->list_subwin[j]==NULL) {
      mat_imagingdata_delete(imagingData);
      cpl_msg_error(cpl_func,"could not allocate memory for imagingData->list_frame[0]->list_subwin[j]");
      return NULL;
    }
    (*imagingData).list_frame[0]->list_subwin[j]->numregion=j+1;
    (*imagingData).list_frame[0]->list_subwin[j]->numdetector= 
      (*imgdata).list_frame[0]->list_subwin[j]->numdetector;
    (*imagingData).list_frame[0]->list_subwin[j]->nbimgreg=
      (*imgdata).list_frame[0]->list_subwin[j]->nbimgreg;
    (*imagingData).list_frame[0]->list_subwin[j]->imgreg= 
      cpl_calloc((*imagingData).list_frame[0]->list_subwin[j]->nbimgreg,sizeof(cpl_image*));
    for(k=0;k<(*imagingData).list_frame[0]->list_subwin[j]->nbimgreg;k++) {
      (*imagingData).list_frame[0]->list_subwin[j]->imgreg[k]=
	cpl_image_new(cpl_image_get_size_x((*imgdata).list_frame[0]->list_subwin[j]->imgreg[k]),
		      cpl_image_get_size_y((*imgdata).list_frame[0]->list_subwin[j]->imgreg[k]),
		      type);
      for (i=0;i<(*imgdata).nbframe;i++) {
	cpl_image_add((*imagingData).list_frame[0]->list_subwin[j]->imgreg[k],
		      (*imgdata).list_frame[i]->list_subwin[j]->imgreg[k]);
      }
      cpl_image_multiply_scalar((*imagingData).list_frame[0]->list_subwin[j]->imgreg[k],
				1./(*imgdata).nbframe);
    }
  }
  
/* Error handling */
  if (!cpl_errorstate_is_equal(prestate)) { 
    cpl_errorstate_dump(prestate, CPL_FALSE, cpl_errorstate_dump_one);
    mat_imagingdata_delete(imagingData);
    return NULL;
  }

  return imagingData;
}

