/* $Id: mat_imbasic.c,v0.5 2014-06-15 12:56:21 mheininger Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: mheininger $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_imbasic.c $
 */

#include <stdlib.h>
#include <stdio.h>
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include <string.h>

#include "mat_error.h"
#include "mat_utils.h"
#include "mat_imbasic.h"
#include "mat_frame.h"
#include "mat_utils.h"

/*-----------------------------------------------------------------------------
                                   Define
 -----------------------------------------------------------------------------*/

/**
   @ingroup imb
   @brief Creates a basic detector monitoring map data structure assocciated with a detector and covering the whole detector or only a set of sub-windows on that detector.
   @param det    This structure describes the detector of the basic detector monitoring map.
   @param imgdet This optional structure describes the sub-window-setup.
   @returns The new basic detector monitoring map of NULL on error.

   This function create a basic detector monitoring map data structure using the detector
   specification as a template. If a sub-window setup is given, this setup is used as setup
   for the map. The number of detector channels, and therefore the size of the detector
   channel specific members (all members except the power spectrum peaks), is determined
   by the det parameter.
 */
mat_imbasic *mat_imbasic_new(mat_detector *det,
			     mat_imagingdetector *imgdet)
{
  mat_imbasic *imb = NULL;
  int          c, i;

  mat_assert_value((det != NULL), CPL_ERROR_NULL_INPUT, NULL, "no mat_detector (det) argument given");
  imb = (mat_imbasic *)cpl_calloc(1, sizeof(mat_imbasic));
  if (imb == NULL)
    {
      cpl_msg_error(cpl_func, "could not allocate memory for mat_imbasic");
      return NULL;
    }
  imb->det = mat_detector_duplicate(det);
  if (imb->det == NULL)
    {
      cpl_msg_error(cpl_func, "could not duplicate mat_detector");
      mat_imbasic_delete(imb);
      return NULL;
    }

  imb->keywords = cpl_propertylist_new();
  if (imb->keywords == NULL)
    {
      cpl_msg_error(cpl_func, "could not allocate memory for cpl_propertylist");
      mat_imbasic_delete(imb);
      return NULL;
    }
  if (imgdet == NULL)
    { /* create an imaging detector table covering the whole detector */
      imb->imgdet = mat_imagingdetector_new(NULL, 1); /* no propertylist and one sub-window */
      if (imb->imgdet == NULL)
	{
	  cpl_msg_error(cpl_func, "could not allocate memory for mat_imagingdetector");
	  mat_imbasic_delete(imb);
	  return NULL;
	}
      /* set the geometry of the sub-window to the whole detector */
      imb->imgdet->list_region[0]->corner[0] = 1;
      imb->imgdet->list_region[0]->corner[1] = 1;
      imb->imgdet->list_region[0]->naxis[0] = imb->det->nx;
      imb->imgdet->list_region[0]->naxis[1] = imb->det->ny;
    }
  else
    {
      imb->imgdet = mat_imagingdetector_duplicate(imgdet);
      if (imb->imgdet == NULL)
	{
	  cpl_msg_error(cpl_func, "could not duplicate the mat_imagingdetector");
	  mat_imbasic_delete(imb);
	  return NULL;
	}
    }

  imb->list_intensity = cpl_imagelist_new();
  if (imb->list_intensity == NULL)
    {
      cpl_msg_error(cpl_func, "could not allocate a cpl_imagelist for the detector channel intensity images");
      mat_imbasic_delete(imb);
      return NULL;
    }
  imb->intensitymedian = cpl_vector_new(det->channel_ncolumns*det->channel_nrows);
  if (imb->intensitymedian == NULL)
    {
      cpl_msg_error(cpl_func, "could not allocate a cpl_vector for the detector channel intensity median");
      mat_imbasic_delete(imb);
      return NULL;
    }
  imb->intensitystdev = cpl_vector_new(det->channel_ncolumns*det->channel_nrows);
  if (imb->intensitystdev == NULL)
    {
      cpl_msg_error(cpl_func, "could not allocate a cpl_vector for the detector channel intensity stdev");
      mat_imbasic_delete(imb);
      return NULL;
    }

  imb->list_variance = cpl_imagelist_new();
  if (imb->list_variance == NULL)
    {
      cpl_msg_error(cpl_func, "could not allocate a cpl_imagelist for the detector channel variance images");
      mat_imbasic_delete(imb);
      return NULL;
    }
  imb->variancemedian = cpl_vector_new(det->channel_ncolumns*det->channel_nrows);
  if (imb->variancemedian == NULL)
    {
      cpl_msg_error(cpl_func, "could not allocate a cpl_vector for the detector channel variance median");
      mat_imbasic_delete(imb);
      return NULL;
    }
  imb->variancestdev = cpl_vector_new(det->channel_ncolumns*det->channel_nrows);
  if (imb->variancestdev == NULL)
    {
      cpl_msg_error(cpl_func, "could not allocate a cpl_vector for the detector channel variance stdev");
      mat_imbasic_delete(imb);
      return NULL;
    }

  imb->list_ron = cpl_imagelist_new();
  if (imb->list_ron == NULL)
    {
      cpl_msg_error(cpl_func, "could not allocate a cpl_imagelist for the detector channel noise images");
      mat_imbasic_delete(imb);
      return NULL;
    }
  imb->ronmedian = cpl_vector_new(det->channel_ncolumns*det->channel_nrows);
  if (imb->ronmedian == NULL)
    {
      cpl_msg_error(cpl_func, "could not allocate a cpl_vector for the detector channel noise median");
      mat_imbasic_delete(imb);
      return NULL;
    }
  imb->ronstdev = cpl_vector_new(det->channel_ncolumns*det->channel_nrows);
  if (imb->ronstdev == NULL)
    {
      cpl_msg_error(cpl_func, "could not allocate a cpl_vector for the detector channel noise stdev");
      mat_imbasic_delete(imb);
      return NULL;
    }

  imb->list_fastpowerspectra = (cpl_vector **)cpl_calloc(det->channel_ncolumns*det->channel_nrows, sizeof(cpl_vector *));
  if (imb->list_fastpowerspectra == NULL)
    {
      cpl_msg_error(cpl_func, "could not create a list of the fast power spectra");
      mat_imbasic_delete(imb);
      return NULL;
    }
  imb->fastpowerspectrapeaks = cpl_bivector_new(MAT_PSPEAKCOUNT);
  if (imb->fastpowerspectrapeaks == NULL)
    {
      cpl_msg_error(cpl_func, "could not allocate a cpl_bivector for the fast power spectra peaks");
      mat_imbasic_delete(imb);
      return NULL;
    }

  imb->list_slowpowerspectra = (cpl_vector **)cpl_calloc(det->channel_ncolumns*det->channel_nrows, sizeof(cpl_vector *));
  if (imb->list_slowpowerspectra == NULL)
    {
      cpl_msg_error(cpl_func, "could not create a list for the slow power spectra");
      mat_imbasic_delete(imb);
      return NULL;
    }
  imb->slowpowerspectrapeaks = cpl_bivector_new(MAT_PSPEAKCOUNT);
  if (imb->slowpowerspectrapeaks == NULL)
    {
      cpl_msg_error(cpl_func, "could not allocate a cpl_bivector for the slow power spectra peaks");
      mat_imbasic_delete(imb);
      return NULL;
    }
  
  for (i = 0; i < MAT_PSPEAKCOUNT; i++)
    {
      cpl_vector_set(cpl_bivector_get_x(imb->fastpowerspectrapeaks), i, 0.0);
      cpl_vector_set(cpl_bivector_get_y(imb->fastpowerspectrapeaks), i, 0.0);
      cpl_vector_set(cpl_bivector_get_x(imb->slowpowerspectrapeaks), i, 0.0);
      cpl_vector_set(cpl_bivector_get_y(imb->slowpowerspectrapeaks), i, 0.0);
    }

  for (c = 0; c < det->channel_ncolumns*det->channel_nrows; c++)
    {
      imb->list_fastpowerspectra[c] = cpl_vector_new(det->channel_nx/2);
      if (imb->list_fastpowerspectra[c] == NULL)
	{
	  cpl_msg_error(cpl_func, "could not create a cpl_vector for the fast power spectrum for detector channel %d", c);
	  mat_imbasic_delete(imb);
	  return NULL;
	}
      imb->list_slowpowerspectra[c] = cpl_vector_new(det->channel_ny);
      if (imb->list_slowpowerspectra[c] == NULL)
	{
	  cpl_msg_error(cpl_func, "could not create a cpl_vector for the slow power spectrum for detector channel %d", c);
	  mat_imbasic_delete(imb);
	  return NULL;
	}
    }
  return imb;
}

/**
   @ingroup imb
   @brief This function deletes a mat_imbasic structure.
   @param imb The basic detector monitoring map which will be deleted.
   @returns CPL_ERROR_NONE or another cpl_error_code.

   This function deletes a mat_imbasic structure.
   Deleting a basic detector monitoring data structure includes all members. If an
   error (for example during the creation of a man_imbasic data structure)
   leads to an incomplete data structure (some members are NULL) this
   function detects this and deletes only the valid members.
*/
cpl_error_code mat_imbasic_delete(mat_imbasic *imb)
{
  int         nc, c;

  mat_assert((imb != NULL), CPL_ERROR_NULL_INPUT, "no mat_imbasic (imb) argument given");
  nc = imb->det->channel_ncolumns*imb->det->channel_nrows;
  if (imb->list_fastpowerspectra != NULL)
    {
      for (c = 0; c < nc; c++)
	{
	  if (imb->list_fastpowerspectra[c] != NULL)
	    {
	      cpl_vector_delete(imb->list_fastpowerspectra[c]);
	      imb->list_fastpowerspectra[c] = NULL;
	    }
	}
      cpl_free(imb->list_fastpowerspectra);
      imb->list_fastpowerspectra = NULL;
    }
  if (imb->fastpowerspectrapeaks != NULL)
    {
      cpl_bivector_delete(imb->fastpowerspectrapeaks);
      imb->fastpowerspectrapeaks = NULL;
    }
  if (imb->list_slowpowerspectra != NULL)
    {
      for (c = 0; c < nc; c++)
	{
	  if (imb->list_slowpowerspectra[c] != NULL)
	    {
	      cpl_vector_delete(imb->list_slowpowerspectra[c]);
	      imb->list_slowpowerspectra[c] = NULL;
	    }
	}
      cpl_free(imb->list_slowpowerspectra);
      imb->list_slowpowerspectra = NULL;
    }
  if (imb->slowpowerspectrapeaks != NULL)
    {
      cpl_bivector_delete(imb->slowpowerspectrapeaks);
      imb->slowpowerspectrapeaks = NULL;
    }

  if (imb->list_intensity != NULL)
    {
      cpl_imagelist_delete(imb->list_intensity);
      imb->list_intensity = NULL;
    }
  if (imb->intensitymedian != NULL)
    {
      cpl_vector_delete(imb->intensitymedian);
      imb->intensitymedian = NULL;
    }
  if (imb->intensitystdev != NULL)
    {
      cpl_vector_delete(imb->intensitystdev);
      imb->intensitystdev = NULL;
    }

  if (imb->list_variance != NULL)
    {
      cpl_imagelist_delete(imb->list_variance);
      imb->list_variance = NULL;
    }
  if (imb->variancemedian != NULL)
    {
      cpl_vector_delete(imb->variancemedian);
      imb->variancemedian = NULL;
    }
  if (imb->variancestdev != NULL)
    {
      cpl_vector_delete(imb->variancestdev);
      imb->variancestdev = NULL;
    }

  if (imb->list_ron != NULL)
    {
      cpl_imagelist_delete(imb->list_ron);
      imb->list_ron = NULL;
    }
  if (imb->ronmedian != NULL)
    {
      cpl_vector_delete(imb->ronmedian);
      imb->ronmedian = NULL;
    }
  if (imb->ronstdev != NULL)
    {
      cpl_vector_delete(imb->ronstdev);
      imb->ronstdev = NULL;
    }

  if (imb->imgdet != NULL)
    {
      mat_imagingdetector_delete(imb->imgdet);
      imb->imgdet = NULL;
    }
  if (imb->keywords != NULL)
    {
      cpl_propertylist_delete(imb->keywords);
      imb->keywords = NULL;
    }
  if (imb->det != NULL)
    {
      mat_detector_delete(imb->det);
      imb->det = NULL;
    }
  cpl_free(imb);
  return CPL_ERROR_NONE;
}

/**
   @ingroup imb
   @brief This function stores a basic detector monitoring map in a FITS file.
   @param imb The basic detector monitoring map of a detector.
   @param fname The name of the detector monitoring map FITS file.
   @param rname The name of the pipeline recipe.
   @param parlist The plugin parameter list.
   @param frameset The input frameset of the plugin (optional).

   This function stores the contents of the mat_imbasic data structure as a basic
   detector monitoring map in a FITS file. All QC1 parameters are stored in an
   empty primary header. All statistical data (median, variance and noise for
   each detector channel) is stored in an IM_STATISTICS binary table extension.
   The fast (horizontal) 1-d power spectrum for each detector channel is stored
   in an IM_FAST_POWERSPECTRUM binary table and the slow 1-d power spectra are
   stored in an IM_SLOW_POWERSPECTRUM binary table.

   The following QC1 parameters are stored in the primary header:

   - QC DETi CHANNELj OFFSETp The offset for the detector channels.
   - QC DETi CHANNELj RONp The read-out noise is given for each detector channel.
   - QC DETi PS FAST PEAKk FREQ The frequency for the k-th strongest peak in the fast (horizontal) 1-d power spectrum.
   - QC DETi PS FAST PEAKk POWER The power for the k-th strongest peak in the fast (horizontal) 1-d power spectrum.
   - QC DETi PS SLOW PEAKk FREQ The frequency for the k-th strongest peak in the slow (vertical) 1-d power spectrum.
   - QC DETi PS SLOW PEAKk POWER The power for the k-th strongest peak in the slow (vertical) 1-d power spectrum.

   Where i is the detector number (1 = L/M-Band, 2 = N-Band), j is the detector
   channel number (1 .. 32 for the L/M-Band, 1 .. 64 for the N-Band), k is the
   k-th power spectrum peak (1 .. 5), and p is the readout mode
   (HAWAII-2RG: 1 = slow readout, 2 = fast readout, Aquarius:
   1 = low gain mode, 2 = high gain mode). The detector number (used for DETi)
   will be always 1 for the L/M-Band detector and 2 for the N-Band detector.
   In addition, the DETi CHIP1 ID keyword allows to distinguish between
   different detector chips. This is necessary, to deal with a detector
   change after, for example, an instrument repair.
 */
cpl_error_code mat_imbasic_save(mat_imbasic *imb,
				const char *fname,
				const char *rname,
				cpl_parameterlist *parlist,
				cpl_frameset *frameset)
{
  int               c, nc;
  int               r;
  cpl_frame        *imbframe = NULL;
  cpl_propertylist *plist = NULL;
  cpl_table        *table = NULL;
  cpl_image        *img = NULL;
  cpl_array        *arr = NULL;
  int               nx, ny;
  int               x, y;
  char              kwd[64];

  cpl_msg_info(cpl_func, "storing the basic detector monitoring information in file %s", fname);
  cpl_error_reset();
  cpl_ensure_code((imb != NULL), CPL_ERROR_NULL_INPUT);
  /* Create product frame */
  imbframe = cpl_frame_new();
  cpl_frame_set_filename(imbframe, fname);
  cpl_frame_set_tag(imbframe, MATISSE_IMB_PROCATG);
  cpl_frame_set_type(imbframe, CPL_FRAME_TYPE_IMAGE);
  cpl_frame_set_group(imbframe, CPL_FRAME_GROUP_PRODUCT);
  cpl_frame_set_level(imbframe, CPL_FRAME_LEVEL_FINAL);
  if (cpl_error_get_code()) {
    cpl_msg_error(cpl_func, "Error while initialising the product frame, code = %d, message = %s",
		  cpl_error_get_code(),
		  cpl_error_get_message());
    cpl_frame_delete(imbframe);
    return CPL_ERROR_UNSPECIFIED;
  }
  plist = cpl_propertylist_new();
  /* Add DataFlow keywords */
  if (cpl_dfs_setup_product_header(plist, imbframe, frameset, parlist,
				   rname, PACKAGE "/" PACKAGE_VERSION, "?Dictionary?", NULL) != CPL_ERROR_NONE) {
    cpl_msg_error(cpl_func, "Problem in the product DFS-compliance, code = %d, message = %s",
		  cpl_error_get_code(),
		  cpl_error_get_message());
    cpl_error_reset();
    cpl_propertylist_delete(plist);
    cpl_frame_delete(imbframe);
    return CPL_ERROR_UNSPECIFIED;
  }
  /* Add QC parameters in the header */
  /* readout mode is missing */
  nc = cpl_vector_get_size(imb->intensitymedian);
  for (c = 0; c < nc; c++)
    {
      snprintf(kwd, 64, "ESO QC DET%d CHANNEL%d OFFSET%d", imb->det->nr, c + 1, imb->det->read_id);
      cpl_propertylist_append_double(plist, kwd, mat_round(cpl_vector_get(imb->intensitymedian, c), MAT_PREC_INTENSITY));
      snprintf(kwd, 64, "ESO QC DET%d CHANNEL%d RON%d", imb->det->nr, c + 1, imb->det->read_id);
      cpl_propertylist_append_double(plist, kwd, mat_round(cpl_vector_get(imb->ronmedian, c), MAT_PREC_NOISE));
    }
  for (c = 0; c < MAT_PSPEAKCOUNT; c++)
    {
      snprintf(kwd, 64, "ESO QC DET%d PS FAST PEAK%d FREQ%d", imb->det->nr, c + 1, imb->det->read_id);
      cpl_propertylist_append_double(plist, kwd, mat_round(cpl_vector_get(cpl_bivector_get_x(imb->fastpowerspectrapeaks), c), MAT_PREC_FREQUENCY));
      snprintf(kwd, 64, "ESO QC DET%d PS FAST PEAK%d POWER%d", imb->det->nr, c + 1, imb->det->read_id);
      cpl_propertylist_append_double(plist, kwd, mat_round(cpl_vector_get(cpl_bivector_get_y(imb->fastpowerspectrapeaks), c), MAT_PREC_POWER));
    }
  for (c = 0; c < MAT_PSPEAKCOUNT; c++)
    {
      snprintf(kwd, 64, "ESO QC DET%d PS SLOW PEAK%d FREQ%d", imb->det->nr, c + 1, imb->det->read_id);
      cpl_propertylist_append_double(plist, kwd, mat_round(cpl_vector_get(cpl_bivector_get_x(imb->slowpowerspectrapeaks), c), MAT_PREC_FREQUENCY));
      snprintf(kwd, 64, "ESO QC DET%d PS SLOW PEAK%d POWER%d", imb->det->nr, c + 1, imb->det->read_id);
      cpl_propertylist_append_double(plist, kwd, mat_round(cpl_vector_get(cpl_bivector_get_y(imb->slowpowerspectrapeaks), c), MAT_PREC_POWER));
    }


  // Add Generic QC parameters
  cpl_propertylist *qclist=NULL;
  qclist = cpl_propertylist_new();
  mat_add_generic_qc(plist,qclist);
  cpl_propertylist_append(plist,qclist);
  cpl_propertylist_delete(qclist);

  
  /* Save the file */
  cpl_propertylist_erase(plist,"RADECSYS");
  if (cpl_propertylist_save(plist, fname, CPL_IO_CREATE) != CPL_ERROR_NONE) {
    cpl_msg_error(cpl_func, "Could not save product, code = %d, message = %s",
		  cpl_error_get_code(),
		  cpl_error_get_message());
    cpl_propertylist_delete(plist);
    cpl_frame_delete(imbframe);
    return CPL_ERROR_UNSPECIFIED;
  }
  cpl_propertylist_delete(plist);
  /* create the binary table extension containing the statistical data */
  img = cpl_imagelist_get(imb->list_intensity, 0);
  nx = cpl_image_get_size_x(img);
  ny = cpl_image_get_size_y(img);
  table = cpl_table_new(3*nc);
  if (table == NULL)
    {
      cpl_error_code ec = cpl_error_get_code();
      cpl_msg_error(cpl_func,
		    "cannot allocate the table for the statistical data, code = %d, message = %s",
		    ec,
		    cpl_error_get_message());
      return ec;
    }
  if (cpl_table_new_column(table, "DETECTOR", CPL_TYPE_INT) != CPL_ERROR_NONE)
    {
      cpl_error_code ec = cpl_error_get_code();
      cpl_msg_error(cpl_func,
		    "cannot allocate the detector column, code = %d, message = %s",
		    ec,
		    cpl_error_get_message());
      cpl_table_delete(table);
      return ec;
    }
  if (cpl_table_new_column(table, "CHANNEL", CPL_TYPE_INT) != CPL_ERROR_NONE)
    {
      cpl_error_code ec = cpl_error_get_code();
      cpl_msg_error(cpl_func,
		    "cannot allocate the channel column, code = %d, message = %s",
		    ec,
		    cpl_error_get_message());
      cpl_table_delete(table);
      return ec;
    }
  if (cpl_table_new_column(table, "TYPE", CPL_TYPE_STRING) != CPL_ERROR_NONE)
    {
      cpl_error_code ec = cpl_error_get_code();
      cpl_msg_error(cpl_func,
		    "cannot allocate the type column, code = %d, message = %s",
		    ec,
		    cpl_error_get_message());
      cpl_table_delete(table);
      return ec;
    }
  if (cpl_table_new_column(table, "MEDIAN", CPL_TYPE_FLOAT) != CPL_ERROR_NONE)
    {
      cpl_error_code ec = cpl_error_get_code();
      cpl_msg_error(cpl_func,
		    "cannot allocate the type column, code = %d, message = %s",
		    ec,
		    cpl_error_get_message());
      cpl_table_delete(table);
      return ec;
    }
  if (cpl_table_new_column(table, "STDEV", CPL_TYPE_FLOAT) != CPL_ERROR_NONE)
    {
      cpl_error_code ec = cpl_error_get_code();
      cpl_msg_error(cpl_func,
		    "cannot allocate the type column, code = %d, message = %s",
		    ec,
		    cpl_error_get_message());
      cpl_table_delete(table);
      return ec;
    }
  if (cpl_table_new_column_array(table, "DATA", CPL_TYPE_FLOAT, nx*ny) != CPL_ERROR_NONE)
    {
      cpl_error_code ec = cpl_error_get_code();
      cpl_msg_error(cpl_func,
		    "cannot allocate the data column, code = %d, message = %s",
		    ec,
		    cpl_error_get_message());
      cpl_table_delete(table);
      return ec;
    }
  arr = cpl_array_new(2, CPL_TYPE_INT);
  if (arr != NULL)
    {
      cpl_array_set(arr, 0, nx);
      cpl_array_set(arr, 1, ny);
      cpl_table_set_column_dimensions(table, "DATA", arr);
      cpl_array_delete(arr);
      arr = NULL;
    }
  else
    {
      cpl_msg_error(cpl_func, "cannot set the dimension of the data column");
    }
  /* fill the binary table extension containing the statistical data */
  r = 0;
  arr = cpl_array_new(nx*ny, CPL_TYPE_FLOAT);
  if (arr == NULL)
    {
      cpl_error_code ec = cpl_error_get_code();
      cpl_msg_error(cpl_func,
		    "cannot allocate the data column array, code = %d, message = %s",
		    ec,
		    cpl_error_get_message());
      cpl_table_delete(table);
      return ec;
    }
  for (c = 0; c < nc; c++)
    {
      /* put the intensity information into the table */
      cpl_table_set_int(table, "DETECTOR", r, imb->det->nr);
      cpl_table_set_int(table, "CHANNEL", r, c+1);
      cpl_table_set_string(table, "TYPE", r, "I");
      cpl_table_set_float(table, "MEDIAN", r, (float)(cpl_vector_get(imb->intensitymedian, c)));
      cpl_table_set_float(table, "STDEV", r, (float)(cpl_vector_get(imb->intensitystdev, c)));
      img = cpl_imagelist_get(imb->list_intensity, c);
      for (x = 0; x < nx; x++)
	{
	  for (y = 0; y < ny; y++)
	    {
	      int rejected;
	      cpl_array_set(arr, y*nx + x, cpl_image_get(img, x + 1, y + 1, &rejected));
	    }
	}
      cpl_table_set_array(table, "DATA", r, arr);
      r++;
      /* put the variance information into the table */
      cpl_table_set_int(table, "DETECTOR", r, imb->det->nr);
      cpl_table_set_int(table, "CHANNEL", r, c+1);
      cpl_table_set_string(table, "TYPE", r, "V");
      cpl_table_set_float(table, "MEDIAN", r, (float)(cpl_vector_get(imb->variancemedian, c)));
      cpl_table_set_float(table, "STDEV", r, (float)(cpl_vector_get(imb->variancestdev, c)));
      img = cpl_imagelist_get(imb->list_variance, c);
      for (x = 0; x < nx; x++)
	{
	  for (y = 0; y < ny; y++)
	    {
	      int rejected;
	      cpl_array_set(arr, y*nx + x, cpl_image_get(img, x + 1, y + 1, &rejected));
	    }
	}
      cpl_table_set_array(table, "DATA", r, arr);
      r++;
      /* put the noise information into the table */
      cpl_table_set_int(table, "DETECTOR", r, imb->det->nr);
      cpl_table_set_int(table, "CHANNEL", r, c+1);
      cpl_table_set_string(table, "TYPE", r, "N");
      cpl_table_set_float(table, "MEDIAN", r, (float)(cpl_vector_get(imb->ronmedian, c)));
      cpl_table_set_float(table, "STDEV", r, (float)(cpl_vector_get(imb->ronstdev, c)));
      img = cpl_imagelist_get(imb->list_ron, c);
      for (x = 0; x < nx; x++)
	{
	  for (y = 0; y < ny; y++)
	    {
	      int rejected;
	      cpl_array_set(arr, y*nx + x, cpl_image_get(img, x + 1, y + 1, &rejected));
	    }
	}
      cpl_table_set_array(table, "DATA", r, arr);
      r++;
    }
  cpl_array_delete(arr);
  /* store the binary table extension containing the statistical data */
  plist = cpl_propertylist_new();
  cpl_propertylist_append_string(plist, "EXTNAME", MATISSE_IMB_STAT_EXT);
  if (cpl_table_save(table, NULL, plist, fname, CPL_IO_EXTEND) != CPL_ERROR_NONE) {
    cpl_msg_error(cpl_func, "Could not save product, code = %d, message = %s",
		  cpl_error_get_code(),
		  cpl_error_get_message());
    cpl_propertylist_delete(plist);
    cpl_frame_delete(imbframe);
    return CPL_ERROR_UNSPECIFIED;
  }
  cpl_propertylist_delete(plist);
  cpl_table_delete(table);
  /* create the binary table extension containing the average fast 1-d power spectrum for each detector channel */
  nx = cpl_vector_get_size(imb->list_fastpowerspectra[0]);
  table = cpl_table_new(nc);
  if (table == NULL)
    {
      cpl_error_code ec = cpl_error_get_code();
      cpl_msg_error(cpl_func,
		    "cannot allocate the table for the average fast 1-d power spectrum, code = %d, message = %s",
		    ec,
		    cpl_error_get_message());
      return ec;
    }
  if (cpl_table_new_column(table, "DETECTOR", CPL_TYPE_INT) != CPL_ERROR_NONE)
    {
      cpl_error_code ec = cpl_error_get_code();
      cpl_msg_error(cpl_func,
		    "cannot allocate the detector column, code = %d, message = %s",
		    ec,
		    cpl_error_get_message());
      cpl_table_delete(table);
      return ec;
    }
  if (cpl_table_new_column(table, "CHANNEL", CPL_TYPE_INT) != CPL_ERROR_NONE)
    {
      cpl_error_code ec = cpl_error_get_code();
      cpl_msg_error(cpl_func,
		    "cannot allocate the channel column, code = %d, message = %s",
		    ec,
		    cpl_error_get_message());
      cpl_table_delete(table);
      return ec;
    }
  if (cpl_table_new_column_array(table, "DATA", CPL_TYPE_FLOAT, nx) != CPL_ERROR_NONE)
    {
      cpl_error_code ec = cpl_error_get_code();
      cpl_msg_error(cpl_func,
		    "cannot allocate the data column, code = %d, message = %s",
		    ec,
		    cpl_error_get_message());
      cpl_table_delete(table);
      return ec;
    }
  /* fill the binary table extension containing the average fast 1-d power spectrum */
  r = 0;
  arr = cpl_array_new(nx, CPL_TYPE_FLOAT);
  if (arr == NULL)
    {
      cpl_error_code ec = cpl_error_get_code();
      cpl_msg_error(cpl_func,
		    "cannot allocate the data column array, code = %d, message = %s",
		    ec,
		    cpl_error_get_message());
      cpl_table_delete(table);
      return ec;
    }
  for (c = 0; c < nc; c++)
    {
      /* put the intensity information into the table */
      cpl_table_set_int(table, "DETECTOR", r, imb->det->nr);
      cpl_table_set_int(table, "CHANNEL", r, c+1);
      for (x = 0; x < nx; x++)
	{
	  cpl_array_set(arr, x, cpl_vector_get(imb->list_fastpowerspectra[c], x));
	}
      cpl_table_set_array(table, "DATA", r, arr);
      r++;
    }
  cpl_array_delete(arr);
  /* store the binary table extension containing the average fast 1-d power spectrum */
  plist = cpl_propertylist_new();
  cpl_propertylist_append_string(plist, "EXTNAME", MATISSE_IMB_FASTPS_EXT);
  if (cpl_table_save(table, NULL, plist, fname, CPL_IO_EXTEND) != CPL_ERROR_NONE) {
    cpl_msg_error(cpl_func, "Could not save product, code = %d, message = %s",
		  cpl_error_get_code(),
		  cpl_error_get_message());
    cpl_propertylist_delete(plist);
    cpl_frame_delete(imbframe);
    return CPL_ERROR_UNSPECIFIED;
  }
  cpl_propertylist_delete(plist);
  cpl_table_delete(table);
  /* create the binary table extension containing the average slow 1-d power spectrum for each detector channel */
  ny = cpl_vector_get_size(imb->list_slowpowerspectra[0]);
  table = cpl_table_new(nc);
  if (table == NULL)
    {
      cpl_error_code ec = cpl_error_get_code();
      cpl_msg_error(cpl_func,
		    "cannot allocate the table for the average slow 1-d power spectrum, code = %d, message = %s",
		    ec,
		    cpl_error_get_message());
      return ec;
    }
  if (cpl_table_new_column(table, "DETECTOR", CPL_TYPE_INT) != CPL_ERROR_NONE)
    {
      cpl_error_code ec = cpl_error_get_code();
      cpl_msg_error(cpl_func,
		    "cannot allocate the detector column, code = %d, message = %s",
		    ec,
		    cpl_error_get_message());
      cpl_table_delete(table);
      return ec;
    }
  if (cpl_table_new_column(table, "CHANNEL", CPL_TYPE_INT) != CPL_ERROR_NONE)
    {
      cpl_error_code ec = cpl_error_get_code();
      cpl_msg_error(cpl_func,
		    "cannot allocate the channel column, code = %d, message = %s",
		    ec,
		    cpl_error_get_message());
      cpl_table_delete(table);
      return ec;
    }
  if (cpl_table_new_column_array(table, "DATA", CPL_TYPE_FLOAT, ny) != CPL_ERROR_NONE)
    {
      cpl_error_code ec = cpl_error_get_code();
      cpl_msg_error(cpl_func,
		    "cannot allocate the data column, code = %d, message = %s",
		    ec,
		    cpl_error_get_message());
      cpl_table_delete(table);
      return ec;
    }
  /* fill the binary table extension containing the average slow 1-d power spectrum */
  r = 0;
  arr = cpl_array_new(ny, CPL_TYPE_FLOAT);
  if (arr == NULL)
    {
      cpl_error_code ec = cpl_error_get_code();
      cpl_msg_error(cpl_func,
		    "cannot allocate the data column array, code = %d, message = %s",
		    ec,
		    cpl_error_get_message());
      cpl_table_delete(table);
      return ec;
    }
  for (c = 0; c < nc; c++)
    {
      /* put the intensity information into the table */
      cpl_table_set_int(table, "DETECTOR", r, imb->det->nr);
      cpl_table_set_int(table, "CHANNEL", r, c+1);
      for (y = 0; y < ny; y++)
	{
	  cpl_array_set(arr, y, cpl_vector_get(imb->list_slowpowerspectra[c], y));
	}
      cpl_table_set_array(table, "DATA", r, arr);
      r++;
    }
  cpl_array_delete(arr);
  /* store the binary table extension containing the average slow 1-d power spectrum */
  plist = cpl_propertylist_new();
  cpl_propertylist_append_string(plist, "EXTNAME", MATISSE_IMB_SLOWPS_EXT);
  if (cpl_table_save(table, NULL, plist, fname, CPL_IO_EXTEND) != CPL_ERROR_NONE) {
    cpl_msg_error(cpl_func, "Could not save product, code = %d, message = %s",
		  cpl_error_get_code(),
		  cpl_error_get_message());
    cpl_propertylist_delete(plist);
    cpl_frame_delete(imbframe);
    return CPL_ERROR_UNSPECIFIED;
  }
  cpl_propertylist_delete(plist);
  cpl_table_delete(table);
  /* Log the saved file in the input frameset */
  cpl_frameset_insert(frameset, imbframe);
  //cpl_frame_delete(imbframe);
  return CPL_ERROR_NONE;
}

