/* $Id: mat_imremanence.c,v0.5 2014-06-15 12:56:21 mheininger Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: mheininger $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_imremanence.c $
 */

#include <stdlib.h>
#include <stdio.h>

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include <string.h>

#include "mat_utils.h"
#include "mat_imremanence.h"
#include "mat_frame.h"
#include "mat_utils.h"

/*-----------------------------------------------------------------------------
                                   Define
 -----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
                                   Functions prototypes
 -----------------------------------------------------------------------------*/

/**
   @ingroup imr
   @brief Creates a detector remanence data structure assocciated with a detector.
   @param det This structure describes the detector of the detector remanence information.
   @returns The new detector remanence data structure or NULL on error.

   This function creates a detector remanence data structure using the detector specification as a template.
*/
mat_imremanence *mat_imremanence_new(mat_detector *det)
{
  mat_imremanence *imr = NULL;

  if (det == NULL)
    {
      cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_detector (det) argument given");
      return NULL;
    }
  imr = (mat_imremanence *)cpl_calloc(1, sizeof(mat_imremanence));
  if (imr == NULL)
    {
      cpl_msg_error(cpl_func, "could not allocate memory for mat_imremanence");
      return NULL;
    }
  imr->det = mat_detector_duplicate(det);
  if (imr->det == NULL)
    {
      cpl_msg_error(cpl_func, "could not duplicate mat_detector");
      mat_imremanence_delete(imr);
      return NULL;
    }

  imr->keywords = cpl_propertylist_new();
  if (imr->keywords == NULL)
    {
      cpl_msg_error(cpl_func, "could not allocate memory for cpl_propertylist");
      mat_imremanence_delete(imr);
      return NULL;
    }
  imr->todarkremanence = 0.0;
  imr->tobrightremanence = 0.0;
  return imr;
}

/**
   @ingroup imr
   @brief This function deletes a mat_imremanence structure.
   @param imr The detector remanence data structure which will be deleted.
   @returns A cpl error code

   This function deletes a mat_imremanence structure.
   Deleting a detector remanence data structure includes all members. If an
   error (for example during the creation of a man_imremanence data structure)
   leads to an incomplete data structure (some members are NULL) this
   function detects this and deletes only the valid members.
 */
cpl_error_code mat_imremanence_delete(mat_imremanence *imr)
{
  if (imr == NULL)
    {
      return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT, "no mat_imremanence (imr) argument given");
    }
  if (imr->keywords != NULL)
    {
      cpl_propertylist_delete(imr->keywords);
      imr->keywords = NULL;
    }
  if (imr->det != NULL)
    {
      mat_detector_delete(imr->det);
      imr->det = NULL;
    }
  cpl_free(imr);
  return CPL_ERROR_NONE;
}

/**
   @ingroup imr
   @brief This function stores a mat_imremanence structure in a FITS file.
   @param imr The detector remanence information of a detector.
   @param fname The name of the detector remanence FITS file.
   @param rname The name of the pipeline recipe.
   @param parlist The plugin parameter list.
   @param frameset The input frameset of the plugin (optional).
   @returns A cpl error code

   This function stores the contents of the mat_imremanence data structure as a detector
   remanence in a FITS file. All QC1 parameters are stored in an empty primary header.

   The following QC1 parameters are stored in the primary header:

   - QC DETi REM TODARKp The remanence value for switching from a bright to a dark frame.
   - QC DETi REM TOBRIGHTp The remanence value for switching from dark to bright frames.

   Where i is the detector number (1 = L/M-Band, 2 = N-Band) and p is the readout mode
   (HAWAII-2RG: 1 = slow readout, 2 = fast readout, Aquarius: 1 = low gain mode,
   2 = high gain mode). The detector number
   (used for DETi) will be always 1 for the L/M-Band detector and 2 for
   the N-Band detector. In addition, the DETi CHIP1 ID keyword allows to
   distinguish between different detector chips. This is necessary, to deal
   with a detector change after, for example, an instrument repair.
 */
cpl_error_code mat_imremanence_save(mat_imremanence *imr,
				    const char *fname,
				    const char *rname,
				    cpl_parameterlist *parlist,
				    cpl_frameset *frameset)
{
  cpl_frame        *imrframe = NULL;
  cpl_propertylist *plist = NULL;
  char              kwd[64];

  cpl_msg_info(cpl_func, "storing the detector remanence information in file %s", fname);
  cpl_error_reset();
  cpl_ensure_code((imr != NULL), CPL_ERROR_NULL_INPUT);
  /* Create product frame */
  imrframe = cpl_frame_new();
  cpl_frame_set_filename(imrframe, fname);
  cpl_frame_set_tag(imrframe, MATISSE_IMR_PROCATG);
  cpl_frame_set_type(imrframe, CPL_FRAME_TYPE_IMAGE);
  cpl_frame_set_group(imrframe, CPL_FRAME_GROUP_PRODUCT);
  cpl_frame_set_level(imrframe, CPL_FRAME_LEVEL_FINAL);
  if (cpl_error_get_code()) {
    cpl_msg_error(cpl_func, "Error while initialising the product frame, code = %d, message = %s",
		  cpl_error_get_code(),
		  cpl_error_get_message());
    cpl_frame_delete(imrframe);
    return CPL_ERROR_UNSPECIFIED;
  }
  plist = cpl_propertylist_new();
  /* Add DataFlow keywords */
  /*
    check if the input frameset is empty
    => function used in an artificial unit test (no real input files)
    => the DFS header cannot include the keywords from the first input file
  */
  if (!cpl_frameset_is_empty(frameset))
    {
      if (cpl_dfs_setup_product_header(plist, imrframe, frameset, parlist,
				       rname, PACKAGE "/" PACKAGE_VERSION, "?Dictionary?", NULL) != CPL_ERROR_NONE) {
	cpl_msg_error(cpl_func, "Problem in the product DFS-compliance, code = %d, message = %s",
		      cpl_error_get_code(),
		      cpl_error_get_message());
	cpl_error_reset();
	cpl_propertylist_delete(plist);
	cpl_frame_delete(imrframe);
	return CPL_ERROR_UNSPECIFIED;
      }
    }
  /* Add QC parameters in the header */
  snprintf(kwd, 64, "ESO QC DET%d REM TODARK%d", imr->det->nr, imr->det->read_id);
  cpl_propertylist_append_double(plist, kwd, mat_round(imr->todarkremanence, MAT_PREC_REMANENCE));
  snprintf(kwd, 64, "ESO QC DET%d REM TOBRIGHT%d", imr->det->nr, imr->det->read_id);
  cpl_propertylist_append_double(plist, kwd, mat_round(imr->tobrightremanence, MAT_PREC_REMANENCE));
  //(info->det.nx*info->det.ny - info->nbad)/(double)(info->det.nx*info->det.ny));


  // Add Generic QC parameters
  cpl_propertylist *qclist=NULL;
  qclist = cpl_propertylist_new();
  mat_add_generic_qc(plist,qclist);
  cpl_propertylist_append(plist,qclist);
  cpl_propertylist_delete(qclist);



  /* Save the file */
  cpl_propertylist_erase(plist,"RADECSYS");
  if (cpl_propertylist_save(plist, fname, CPL_IO_CREATE) != CPL_ERROR_NONE)
    {
    cpl_msg_error(cpl_func, "Could not save product, code = %d, message = %s",
		  cpl_error_get_code(),
		  cpl_error_get_message());
    cpl_propertylist_delete(plist);
    cpl_frame_delete(imrframe);
    return CPL_ERROR_UNSPECIFIED;
  }
  cpl_propertylist_delete(plist);
  /* Log the saved file in the input frameset */
  cpl_frameset_insert(frameset, imrframe);
  //cpl_frame_delete(imrframe);
  return CPL_ERROR_NONE;
}

