/* $Id: mat_nonlinearity.c,v0.5 2014-06-15 12:56:21 mheininger Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: mheininger $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_nonlinearity.c $
 */

#include <stdlib.h>
#include <stdio.h>

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include <string.h>

#include "mat_error.h"
#include "mat_utils.h"
#include "mat_nonlinearity.h"
#include "mat_frame.h"
#include "mat_utils.h"

/*-----------------------------------------------------------------------------
                                   Define
 -----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
                                   Functions prototypes
 -----------------------------------------------------------------------------*/

/**
   @ingroup nlm
   @brief Creates a nonlinearity map data structure assocciated with a detector and covering the whole detector or only a set of sub-windows on that detector.
   @param det     This structure describes the detector of the nonlinearity map.
   @param imgdet  This optional structure describes the sub-window-setup.
   @param nbcoeff The number of coefficients for the polynomial.
   @returns The new nonlinearity map of NULL on error.

   This function creates a nonlinearity map data structure using the detector specification as a template. If a sub-window setup is given, this setup is used as setup for the nonlinearity map. This means, that for each sub-window an individual cpl_imagelist is allocated and stored in the list_coeff member. The number of detector channels, and therefore the size of the detector channel specific members (channelgain, channeloffset and channelnoise), is determined by the det parameter.
 */
mat_nonlinearity *mat_nonlinearity_new(mat_detector *det,
				       mat_imagingdetector *imgdet,
				       int nbcoeff)
{
  mat_nonlinearity *nlm = NULL;
  int               r, c, i;

  mat_assert_value((det != NULL), CPL_ERROR_NULL_INPUT, NULL, "no mat_detector (det) argument given");
  /*
  mat_assert_value((imgdet != NULL), CPL_ERROR_NULL_INPUT, NULL, "no mat_imagingdetector (imgdet) argument given");
  mat_assert_value((imgdet->list_region != NULL), CPL_ERROR_NULL_INPUT, NULL, "no valid mat_imagingdetector (imgdet->list_region) argument given");
  */
  nlm = (mat_nonlinearity *)cpl_calloc(1, sizeof(mat_nonlinearity));
  if (nlm == NULL)
    {
      cpl_msg_error(cpl_func, "could not allocate memory for mat_nonlinearity");
      return NULL;
    }
  nlm->det = mat_detector_duplicate(det);
  if (nlm->det == NULL)
    {
      cpl_msg_error(cpl_func, "could not duplicate mat_detector");
      mat_nonlinearity_delete(nlm);
      return NULL;
    }

  nlm->keywords = cpl_propertylist_new();
  if (nlm->keywords == NULL)
    {
      cpl_msg_error(cpl_func, "could not allocate memory for cpl_propertylist");
      mat_nonlinearity_delete(nlm);
      return NULL;
    }
  if (imgdet == NULL)
    {
      /* create an imaging detector table covering the whole detector */
      nlm->imgdet = mat_imagingdetector_new(NULL, 1); /* no propertylist and one sub-window */
      if (nlm->imgdet == NULL)
	{
	  cpl_msg_error(cpl_func, "could not allocate memory for mat_imagingdetector");
	  mat_nonlinearity_delete(nlm);
	  return NULL;
	}
      /* set the geometry of the sub-window to the whole detector */
      nlm->imgdet->list_region[0]->corner[0] = 1;
      nlm->imgdet->list_region[0]->corner[1] = 1;
      nlm->imgdet->list_region[0]->naxis[0] = nlm->det->nx;
      nlm->imgdet->list_region[0]->naxis[1] = nlm->det->ny;
    }
  else
    {
      nlm->imgdet = mat_imagingdetector_duplicate(imgdet);
      if (nlm->imgdet == NULL)
	{
	  cpl_msg_error(cpl_func, "could not duplicate the mat_imagingdetector");
	  mat_nonlinearity_delete(nlm);
	  return NULL;
	}
    }
  nlm->detnllaw[0] = '\0';
  nlm->detnlncoeff = 0;
  nlm->detnlcoeff[0] = 1.0;
  for (i = 1; i < MAT_NONLIN_MAX_NCOEFF; i++)
    {
      nlm->detnlcoeff[i] = 0.0;
    }
  nlm->channelgain = cpl_vector_new(det->channel_ncolumns*det->channel_nrows);
  if (nlm->channelgain == NULL)
    {
      cpl_msg_error(cpl_func, "could not allocate a cpl_vector for the detector channel gain");
      mat_nonlinearity_delete(nlm);
      return NULL;
    }
  nlm->channeloffset = cpl_vector_new(det->channel_ncolumns*det->channel_nrows);
  if (nlm->channeloffset == NULL)
    {
      cpl_msg_error(cpl_func, "could not allocate a cpl_vector for the detector channel offset");
      mat_nonlinearity_delete(nlm);
      return NULL;
    }
  nlm->channelnoise = cpl_vector_new(det->channel_ncolumns*det->channel_nrows);
  if (nlm->channelnoise == NULL)
    {
      cpl_msg_error(cpl_func, "could not allocate a cpl_vector for the detector channel noise");
      mat_nonlinearity_delete(nlm);
      return NULL;
    }
  nlm->list_limit = (cpl_image **)cpl_calloc(nlm->imgdet->nbregion, sizeof(cpl_image *));
  if (nlm->list_limit == NULL)
    {
      cpl_msg_error(cpl_func, "could not create a cpl_image list (limit)");
      mat_nonlinearity_delete(nlm);
      return NULL;
    }
  for (r = 0; r < nlm->imgdet->nbregion; r++)
    {
      int nx = nlm->imgdet->list_region[r]->naxis[0];
      int ny = nlm->imgdet->list_region[r]->naxis[1];
      nlm->list_limit[r] = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
      if (nlm->list_limit[r] == NULL)
	{
	  cpl_msg_error(cpl_func, "could not create a cpl_image for a limit region");
	  mat_nonlinearity_delete(nlm);
	  return NULL;
	}
    }
  nlm->nbcoeff = nbcoeff;
  nlm->has_coeff = 0; // no valid coefficients map, because it is still empty!
  nlm->list_coeff = (cpl_imagelist **)cpl_calloc(nlm->imgdet->nbregion, sizeof(cpl_imagelist *));
  if (nlm->list_coeff == NULL)
    {
      cpl_msg_error(cpl_func, "could not create a cpl_image list (nonlinearity)");
      mat_nonlinearity_delete(nlm);
      return NULL;
    }
  for (r = 0; r < nlm->imgdet->nbregion; r++)
    {
      int nx = nlm->imgdet->list_region[r]->naxis[0];
      int ny = nlm->imgdet->list_region[r]->naxis[1];
      nlm->list_coeff[r] = cpl_imagelist_new();
      if (nlm->list_coeff[r] == NULL)
	{
	  cpl_msg_error(cpl_func, "could not create a cpl_imagelist (nonlinearity region)");
	  mat_nonlinearity_delete(nlm);
	  return NULL;
	}
      for (c = 0; c < nlm->nbcoeff; c++)
	{
	  cpl_image *img = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
	  if (img == NULL)
	    {
	      cpl_msg_error(cpl_func, "could not create a cpl_image (nonlinearity region)");
	      mat_nonlinearity_delete(nlm);
	      return NULL;
	    }
	  cpl_imagelist_set(nlm->list_coeff[r], img, c);
	}
    }
  nlm->has_stdev = 0; // no valid stdev map, because it is still empty!
  nlm->list_stdev = (cpl_image **)cpl_calloc(nlm->imgdet->nbregion, sizeof(cpl_image *));
  if (nlm->list_stdev == NULL)
    {
      cpl_msg_error(cpl_func, "could not create a cpl_image list (error)");
      mat_nonlinearity_delete(nlm);
      return NULL;
    }
  for (r = 0; r < nlm->imgdet->nbregion; r++)
    {
      int nx = nlm->imgdet->list_region[r]->naxis[0];
      int ny = nlm->imgdet->list_region[r]->naxis[1];
      nlm->list_stdev[r] = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
      if (nlm->list_stdev[r] == NULL)
	{
	  cpl_msg_error(cpl_func, "could not create a cpl_image for a stdev region");
	  mat_nonlinearity_delete(nlm);
	  return NULL;
	}
    }
  nlm->detectorgain = 1.0;
  nlm->detectornoise = 0.0;
  nlm->horizontalcorrelation = 0.0;
  nlm->verticalcorrelation = 0.0;
  nlm->flatfieldstdev = 0.0;
  nlm->has_rcmap = 0;
  nlm->rclimit   = 65536.0;
  nlm->rcmap     = cpl_vector_new(1025);
  if (nlm->rcmap == NULL)
    {
      cpl_msg_error(cpl_func, "could not allocate a cpl_vector for the raw -> cal mapping");
      mat_nonlinearity_delete(nlm);
      return NULL;
    }
  for (i = 0; i < cpl_vector_get_size(nlm->rcmap); i++)
    {
      cpl_vector_set(nlm->rcmap, i, (double)((65536/(cpl_vector_get_size(nlm->rcmap) - 1))*i));
    }
  nlm->has_hist = 0;
  nlm->hist = cpl_image_new(det->nx, det->ny, CPL_TYPE_DOUBLE);
  if (nlm->hist == NULL)
    {
      cpl_msg_error(cpl_func, "could not allocate a cpl_image for the 2-d nonlinearity histogram");
      mat_nonlinearity_delete(nlm);
      return NULL;
    }
  nlm->nltype = MAT_UNKNOWN_NL;
  return nlm;
}

/**
   @ingroup nlm
   @brief This function deletes a mat_nonlinearity structure.
   @param nlm This parameter contains the nonlinearity map.

   Deleting a nonlinearity data structure includes all members. If an
   error (for example during the creation of a nonlinearity data structure)
   leads to an incomplete data structure (some members are NULL) this
   function detects this and deletes only the valid members.
*/
cpl_error_code mat_nonlinearity_delete(mat_nonlinearity *nlm)
{
  int         r;

  mat_assert((nlm != NULL), CPL_ERROR_NULL_INPUT, "no mat_nonlinearity (nlm) argument given");
  mat_assert(((nlm->list_coeff == NULL) || (nlm->imgdet != NULL)), CPL_ERROR_UNSPECIFIED, "could not get the number of nonlinearity regions since nlm->imgdet is empty -> memory leak!");
  if (nlm->keywords != NULL)
    {
      cpl_propertylist_delete(nlm->keywords);
      nlm->keywords = NULL;
    }
  if (nlm->list_coeff != NULL)
    {
      for (r = 0; r < nlm->imgdet->nbregion; r++)
	{
	  if (nlm->list_coeff[r] != NULL)
	    {
	      cpl_imagelist_delete(nlm->list_coeff[r]);
	      nlm->list_coeff[r] = NULL;
	    }
	}
      cpl_free(nlm->list_coeff);
      nlm->list_coeff = NULL;
    }
  if (nlm->list_stdev != NULL)
    {
      for (r = 0; r < nlm->imgdet->nbregion; r++)
	{
	  if (nlm->list_stdev[r] != NULL)
	    {
	      cpl_image_delete(nlm->list_stdev[r]);
	      nlm->list_stdev[r] = NULL;
	    }
	}
      cpl_free(nlm->list_stdev);
      nlm->list_stdev = NULL;
    }
  if (nlm->list_limit != NULL)
    {
      for (r = 0; r < nlm->imgdet->nbregion; r++)
	{
	  if (nlm->list_limit[r] != NULL)
	    {
	      cpl_image_delete(nlm->list_limit[r]);
	      nlm->list_limit[r] = NULL;
	    }
	}
      cpl_free(nlm->list_limit);
      nlm->list_limit = NULL;
    }
  if (nlm->channelgain != NULL)
    {
      cpl_vector_delete(nlm->channelgain);
      nlm->channelgain = NULL;
    }
  if (nlm->channeloffset != NULL)
    {
      cpl_vector_delete(nlm->channeloffset);
      nlm->channeloffset = NULL;
    }
  if (nlm->channelnoise != NULL)
    {
      cpl_vector_delete(nlm->channelnoise);
      nlm->channelnoise = NULL;
    }
  if (nlm->imgdet != NULL)
    {
      mat_imagingdetector_delete(nlm->imgdet);
      nlm->imgdet = NULL;
    }
  if (nlm->det != NULL)
    {
      mat_detector_delete(nlm->det);
      nlm->det = NULL;
    }
  if (nlm->rcmap != NULL)
    {
      cpl_vector_delete(nlm->rcmap);
      nlm->rcmap = NULL;
      nlm->has_rcmap = 0;
    }
  if (nlm->hist != NULL)
    {
      cpl_image_delete(nlm->hist);
      nlm->hist = NULL;
      nlm->has_hist = 0;
    }
  cpl_free(nlm);
  return CPL_ERROR_NONE;
}

/**
   @ingroup nlm
   @brief This function stores a static nonlinearity map in a FITS file.
   @param nlm The static nonlinearity map of a detector.
   @param fname The name of the static nonlinearity map FITS file.
   @param rname The name of the pipeline recipe.
   @param parlist The plugin parameter list.
   @param frameset The input frameset of the plugin (optional).

   This function stores the contents of the mat_nonlinearity data structure as a static nonlinearity map
   in a FITS file. The nonlinearity map must cover the whole detector (no sub-window setup is allowed).
   The nonlinearity map is stored as a floating point image cube (one image for each coefficient) in the
   primary header. The error map is stored in an image extension (NLSTDERR). In addition, some QC parameters
   from the data structure are stored as primary header keywords:

   - QC DETi GAIN The global detector gain (in electrons per ADU) will be used for scientific purposes.
   - QC DETi RON The global detector readout noise (in electrons) will be used for monitoring purposes.
   - QC DETi CHANNELj GAIN The detector channel specific gain (in electrons per ADU) will be used to detect detector and read-out electronics problems.
   - QC DETi CHANNELj OFFSET The detector channel specific offset (in ADU) will be used to detect detector and read-out electronics problems.
   - QC DETi CHANNELj RON The detector channel specific readout noise (in electrons) will be used to detect detector and read-out electronics problems.
   - QC DETi FFM STDEV The standard deviation of the computed nonlinearity map (the mean is normalized to 1.0).

   Where i is the detector number (HAWAII-2RG -> 1, Aquarius -> 2) and j is the detector channel number.
 */
cpl_error_code mat_nonlinearity_save(mat_nonlinearity *nlm,
				     const char *fname,
				     const char *rname,
				     cpl_parameterlist *parlist,
				     cpl_frameset *frameset)
{
  int               c, cr, i;
  cpl_frame        *nlmframe = NULL;
  cpl_propertylist *plist = NULL;
  char              kwd[64];

  cpl_msg_info(cpl_func, "storing the nonlinearity map in file %s", fname);
  cpl_error_reset();
  cpl_ensure_code((nlm != NULL), CPL_ERROR_NULL_INPUT);
  /* Create product frame */
  nlmframe = cpl_frame_new();
  cpl_frame_set_filename(nlmframe, fname);
  cpl_frame_set_tag(nlmframe, MATISSE_NLM_PROCATG);
  cpl_frame_set_type(nlmframe, CPL_FRAME_TYPE_IMAGE);
  cpl_frame_set_group(nlmframe, CPL_FRAME_GROUP_PRODUCT);
  cpl_frame_set_level(nlmframe, CPL_FRAME_LEVEL_FINAL);
  if (cpl_error_get_code()) {
    cpl_msg_error(cpl_func, "Error while initialising the product frame, code = %d, message = %s",
		  cpl_error_get_code(),
		  cpl_error_get_message());
    cpl_frame_delete(nlmframe);
    return CPL_ERROR_UNSPECIFIED;
  }
  plist = cpl_propertylist_new();
  /* Add DataFlow keywords */
  if (cpl_dfs_setup_product_header(plist, nlmframe, frameset, parlist,
				   rname, PACKAGE "/" PACKAGE_VERSION, "?Dictionary?", NULL) != CPL_ERROR_NONE) {
    cpl_msg_error(cpl_func, "Problem in the product DFS-compliance, code = %d, message = %s",
		  cpl_error_get_code(),
		  cpl_error_get_message());
    cpl_error_reset();
    cpl_propertylist_delete(plist);
    cpl_frame_delete(nlmframe);
    return CPL_ERROR_UNSPECIFIED;
  }
  /* Add QC parameters in the header */
  if (nlm->detnllaw[0] != '\0')
    {
      snprintf(kwd, 64, "ESO QC DET%d NL LAW", nlm->det->nr);
      cpl_propertylist_append_string(plist, kwd, nlm->detnllaw);
      snprintf(kwd, 64, "ESO QC DET%d NL NCOEFF", nlm->det->nr);
      cpl_propertylist_append_double(plist, kwd, nlm->detnlncoeff);
      for (i = 0; i < nlm->detnlncoeff; i++)
	{
	  snprintf(kwd, 64, "ESO QC DET%d NL %c", nlm->det->nr, 'A' + i);
	  cpl_propertylist_append_double(plist, kwd, nlm->detnlcoeff[i]);
	}
    }
  /* readout mode is missing */
  snprintf(kwd, 64, "ESO QC DET%d GAIN%d", nlm->det->nr, nlm->det->read_id);
  cpl_propertylist_append_double(plist, kwd, mat_round(nlm->detectorgain, MAT_PREC_GAIN));
  /* cpl_msg_info(cpl_func, "  %s = %f", kwd, nlm->detectorgain); */
  snprintf(kwd, 64, "ESO QC DET%d RON%d", nlm->det->nr, nlm->det->read_id);
  cpl_propertylist_append_double(plist, kwd, mat_round(nlm->detectornoise, MAT_PREC_NOISE));
  /* cpl_msg_info(cpl_func, "  %s = %f", kwd, nlm->detectornoise); */
  snprintf(kwd, 64, "ESO QC DET%d HACORR%d", nlm->det->nr, nlm->det->read_id);
  cpl_propertylist_append_double(plist, kwd, mat_round(nlm->horizontalcorrelation, MAT_PREC_CORRELATION));
  snprintf(kwd, 64, "ESO QC DET%d VACORR%d", nlm->det->nr, nlm->det->read_id);
  cpl_propertylist_append_double(plist, kwd, mat_round(nlm->verticalcorrelation, MAT_PREC_CORRELATION));
  cr = cpl_vector_get_size(nlm->channelgain);
  for (c = 0; c < cr; c++)
    {
      snprintf(kwd, 64, "ESO QC DET%d CHANNEL%d GAIN%d", nlm->det->nr, c + 1, nlm->det->read_id);
      cpl_propertylist_append_double(plist, kwd, mat_round(cpl_vector_get(nlm->channelgain, c), MAT_PREC_GAIN));
      /* cpl_msg_info(cpl_func, "  %s = %f", kwd, cpl_vector_get(nlm->channelgain, c)); */
      snprintf(kwd, 64, "ESO QC DET%d CHANNEL%d OFFSET%d", nlm->det->nr, c + 1, nlm->det->read_id);
      cpl_propertylist_append_double(plist, kwd, mat_round(cpl_vector_get(nlm->channeloffset, c), MAT_PREC_INTENSITY));
      /* cpl_msg_info(cpl_func, "  %s = %f", kwd, cpl_vector_get(nlm->channeloffset, c)); */
      snprintf(kwd, 64, "ESO QC DET%d CHANNEL%d RON%d", nlm->det->nr, c + 1, nlm->det->read_id);
      cpl_propertylist_append_double(plist, kwd, mat_round(cpl_vector_get(nlm->channelnoise, c), MAT_PREC_NOISE));
      /* cpl_msg_info(cpl_func, "  %s = %f", kwd, cpl_vector_get(nlm->channelnoise, c)); */
    }
  snprintf(kwd, 64, "ESO QC DET%d FFM STDEV%d", nlm->det->nr, nlm->det->read_id);
  cpl_propertylist_append_double(plist, kwd, mat_round(nlm->flatfieldstdev, MAT_PREC_VARIANCE));
  /* cpl_msg_info(cpl_func, "  %s = %f", kwd, nlm->flatfieldstdev); */

  // Add Generic QC parameters
  cpl_propertylist *qclist=NULL;
  qclist = cpl_propertylist_new();
  mat_add_generic_qc(plist,qclist);
  cpl_propertylist_append(plist,qclist);
  cpl_propertylist_delete(qclist);



  /* Save the file */
  cpl_propertylist_erase(plist,"RADECSYS");
  cpl_msg_debug(cpl_func, "storing the nonlinearity limits in file %s", fname);
  if (cpl_image_save(nlm->list_limit[0], fname, CPL_TYPE_FLOAT, plist,
		     CPL_IO_CREATE) != CPL_ERROR_NONE) {
    cpl_msg_error(cpl_func, "Could not save product, code = %d, message = %s",
		  cpl_error_get_code(),
		  cpl_error_get_message());
    cpl_propertylist_delete(plist);
    cpl_frame_delete(nlmframe);
    return CPL_ERROR_UNSPECIFIED;
  }
  cpl_propertylist_delete(plist);
  ///*
  if (nlm->has_coeff)
    {
      cpl_msg_debug(cpl_func, "storing the nonlinearity coefficients in file %s", fname);
      plist = cpl_propertylist_new();
      cpl_propertylist_append_string(plist, "EXTNAME", MATISSE_NLM_COEFF_EXT);
      if (cpl_imagelist_save(nlm->list_coeff[0], fname, CPL_TYPE_FLOAT, plist,
			     CPL_IO_EXTEND) != CPL_ERROR_NONE) {
	cpl_msg_error(cpl_func, "Could not save product, code = %d, message = %s",
		      cpl_error_get_code(),
		      cpl_error_get_message());
	cpl_propertylist_delete(plist);
	cpl_frame_delete(nlmframe);
	return CPL_ERROR_UNSPECIFIED;
      }
      cpl_propertylist_delete(plist);
    }
  //*/
  ///*
  if (nlm->has_stdev)
    { // Store the stdev map only if it is valid (created from the detector calibration plugin)
      cpl_msg_debug(cpl_func, "storing the nonlinearity stdev in file %s", fname);
      plist = cpl_propertylist_new();
      cpl_propertylist_append_string(plist, "EXTNAME", MATISSE_NLM_STDERR_EXT);
      if (cpl_image_save(nlm->list_stdev[0], fname, CPL_TYPE_FLOAT, plist,
			 CPL_IO_EXTEND) != CPL_ERROR_NONE) {
	cpl_msg_error(cpl_func, "Could not save product, code = %d, message = %s",
		      cpl_error_get_code(),
		      cpl_error_get_message());
	cpl_propertylist_delete(plist);
	cpl_frame_delete(nlmframe);
	return CPL_ERROR_UNSPECIFIED;
      }
      cpl_propertylist_delete(plist);
    }
  //*/
  ///*
  if (nlm->has_rcmap)
    { // Store the raw -> cal map only if it is valid (created from the detector calibration plugin)
      cpl_msg_debug(cpl_func, "storing the nonlinearity law in file %s", fname);
      plist = cpl_propertylist_new();
      cpl_propertylist_append_string(plist, "EXTNAME", MATISSE_NLM_RCMAP_EXT);
      if (cpl_vector_save(nlm->rcmap, fname, CPL_TYPE_FLOAT, plist,
			 CPL_IO_EXTEND) != CPL_ERROR_NONE) {
	cpl_msg_error(cpl_func, "Could not save product, code = %d, message = %s",
		      cpl_error_get_code(),
		      cpl_error_get_message());
	cpl_propertylist_delete(plist);
	cpl_frame_delete(nlmframe);
	return CPL_ERROR_UNSPECIFIED;
      }
      cpl_propertylist_delete(plist);
    }
  //*/
  ///*
  if (nlm->has_hist)
    { // Store the stdev map only if it is valid (created from the detector calibration plugin)
      cpl_msg_debug(cpl_func, "storing the nonlinearity histogram in file %s", fname);
      plist = cpl_propertylist_new();
      cpl_propertylist_append_string(plist, "EXTNAME", MATISSE_NLM_HIST_EXT);
      if (cpl_image_save(nlm->hist, fname, CPL_TYPE_INT, plist,
			 CPL_IO_EXTEND) != CPL_ERROR_NONE) {
	cpl_msg_error(cpl_func, "Could not save product, code = %d, message = %s",
		      cpl_error_get_code(),
		      cpl_error_get_message());
	cpl_propertylist_delete(plist);
	cpl_frame_delete(nlmframe);
	return CPL_ERROR_UNSPECIFIED;
      }
      cpl_propertylist_delete(plist);
    }
  //*/
  /* Log the saved file in the input frameset */
  cpl_frameset_insert(frameset, nlmframe);
  //cpl_frame_delete(nlmframe);
  return CPL_ERROR_NONE;
}

/**
   @ingroup nlm
   @brief This function loads the content of NONLINEARITY map in a mat_nonlinearity structure.
   @param nlmframe The frame containing the name of the nonlinearity map.
   @returns The loaded nonlinearity map or NULL.

   This function reads the static nonlinearity map and stores the contents in a mat_nonlinearity
   data structure. The nonlinearity map covers the whole detector and must be mapped
   to a given sub-window layout by using the mat_nonlinearity_map function.
 */
mat_nonlinearity *mat_nonlinearity_load(cpl_frame *nlmframe)
{
  mat_detector     *det   = NULL;
  cpl_propertylist *plist = NULL;
  cpl_image        *limit = NULL;
  cpl_imagelist    *coeff = NULL;
  cpl_image        *stdev = NULL;
  cpl_vector       *rcmap = NULL;
  cpl_image        *hist  = NULL;
  int               extnr = -1;
  mat_nonlinearity *nlm = NULL;
  int               cr, c, i;
  char              kwd[64];

  cpl_msg_info(cpl_func, "load the nonlinearity map from file %s", cpl_frame_get_filename(nlmframe));
  /* create a mat_detector structure, load the primary keywords and decode the detector */
  det = mat_detector_new();
  if (det == NULL)
    {
      cpl_msg_error(cpl_func, "could not allocate memory for mat_nonlinearity->det");
      return NULL;
    }
  plist = cpl_propertylist_load(cpl_frame_get_filename(nlmframe), 0);
  if (plist == NULL)
    {
      cpl_msg_error(cpl_func, "could not load the primary keywords");
      mat_detector_delete(det);
      return NULL;
    }
  mat_detector_decode(det, plist);
  /* load the nonlinearity limits from the file */
  limit = cpl_image_load(cpl_frame_get_filename(nlmframe), CPL_TYPE_DOUBLE, 0, 0);
  if (limit == NULL)
    {
      cpl_msg_error(cpl_func, "could not load the nonlinearity limits");
      mat_detector_delete(det);
      cpl_propertylist_delete(plist);
      return NULL;
    }
  /* load the real nonlinearity coefficients from the file (we need the mumber of coefficients) */
  extnr = cpl_fits_find_extension(cpl_frame_get_filename(nlmframe), MATISSE_NLM_COEFF_EXT);
  if (extnr != -1)
    {
      coeff = cpl_imagelist_load(cpl_frame_get_filename(nlmframe), CPL_TYPE_DOUBLE, extnr);
      if (coeff == NULL)
	{
	  cpl_msg_error(cpl_func, "could not load the nonlinearity map for nlm->list_coeff[0]");
	}
    }
  /* load the stdev image */
  extnr = cpl_fits_find_extension(cpl_frame_get_filename(nlmframe), MATISSE_NLM_STDERR_EXT);
  if (extnr != -1)
    {
      stdev = cpl_image_load(cpl_frame_get_filename(nlmframe), CPL_TYPE_DOUBLE, 0, extnr);
      if (stdev == NULL)
	{ // the extension does exist, but the image could not be loaded -> has_stdev := 0
	  cpl_msg_error(cpl_func, "could not load the error map for nlm->list_stdev[0]");
	}
    }
  /* load the raw -> cal map, if it does not exist, give a warning */
  extnr = cpl_fits_find_extension(cpl_frame_get_filename(nlmframe), MATISSE_NLM_RCMAP_EXT);
  if (extnr != -1)
    {
      rcmap = cpl_vector_load(cpl_frame_get_filename(nlmframe), extnr);
      if (rcmap == NULL)
	{ // the extension does exist, but the vector could not be loaded -> has_rcmap := 0
	  cpl_msg_warning(cpl_func, "could not load the raw -> cal map for nlm->list_rcmap");
	}
    }
  /* load the histigram image */
  extnr = cpl_fits_find_extension(cpl_frame_get_filename(nlmframe), MATISSE_NLM_HIST_EXT);
  if (extnr != -1)
    {
      hist = cpl_image_load(cpl_frame_get_filename(nlmframe), CPL_TYPE_DOUBLE, 0, extnr);
      if (hist == NULL)
	{ // the extension does exist, but the image could not be loaded -> has_hist := 0
	  cpl_msg_warning(cpl_func, "could not load the 2-d histogram for nlm->hist");
	}
    }
  /* create and initialize the mat_nonlinearity structure */
  if (coeff == NULL)
    {
      nlm = mat_nonlinearity_new(det, NULL, 0);
    }
  else
    {
      nlm = mat_nonlinearity_new(det, NULL, cpl_imagelist_get_size(coeff));
    }
  mat_detector_delete(det); /* we can delete the mat_detector, because we have a copy in nlm */
  det = NULL;
  if (nlm == NULL)
    {
      cpl_msg_error(cpl_func, "could not allocate memory for mat_nonlinearity");
      cpl_propertylist_delete(plist);
      if (limit != NULL) cpl_image_delete(limit);
      if (stdev != NULL) cpl_image_delete(stdev);
      if (hist != NULL) cpl_image_delete(hist);
      if (coeff != NULL) cpl_imagelist_delete(coeff);
      if (rcmap != NULL) cpl_vector_delete(rcmap);
      return NULL;
    }
  /* move the propertylist into the nlm */
  cpl_propertylist_delete(nlm->keywords);
  nlm->keywords = plist;
  snprintf(kwd, 64, "ESO QC DET%d NL LAW", nlm->det->nr);
  mat_propertylist_get_string_default(nlm->detnllaw, sizeof(nlm->detnllaw), nlm->keywords, kwd, "");
  snprintf(kwd, 64, "ESO QC DET%d NL NCOEFF", nlm->det->nr);
  nlm->detnlncoeff = mat_propertylist_get_int_default(nlm->keywords, kwd, 0);
  for (i = 0; i < nlm->detnlncoeff; i++)
    {
      snprintf(kwd, 64, "ESO QC DET%d NL %c", nlm->det->nr, 'A' + i);
      if (i == 0)
	{
	  nlm->detnlcoeff[i] = mat_propertylist_get_double_default(nlm->keywords, kwd, 1.0);
	}
      else
	{
	  nlm->detnlcoeff[i] = mat_propertylist_get_double_default(nlm->keywords, kwd, 0.0);
	}
    }
  nlm->has_nllaw = (nlm->detnlncoeff != 0);
  /* get all important keyword values from the propertylist */
  snprintf(kwd, 64, "ESO QC DET%d GAIN%d", nlm->det->nr, nlm->det->read_id);
  nlm->detectorgain = mat_propertylist_get_double_default(nlm->keywords, kwd, 1.0);
  snprintf(kwd, 64, "ESO QC DET%d RON%d", nlm->det->nr, nlm->det->read_id);
  nlm->detectornoise = mat_propertylist_get_double_default(nlm->keywords, kwd, 0.0);
  snprintf(kwd, 64, "ESO QC DET%d HACORR%d", nlm->det->nr, nlm->det->read_id);
  nlm->horizontalcorrelation = cpl_propertylist_get_double(nlm->keywords, kwd);
  snprintf(kwd, 64, "ESO QC DET%d VACORR%d", nlm->det->nr, nlm->det->read_id);
  nlm->verticalcorrelation = cpl_propertylist_get_double(nlm->keywords, kwd);
  snprintf(kwd, 64, "ESO QC DET%d FFM STDEV%d", nlm->det->nr, nlm->det->read_id);
  nlm->flatfieldstdev = mat_propertylist_get_double_default(nlm->keywords, kwd, 0.0);
  cr = cpl_vector_get_size(nlm->channelgain);
  for (c = 0; c < cr; c++)
    {
      snprintf(kwd, 64, "ESO QC DET%d CHANNEL%d GAIN%d", nlm->det->nr, c + 1, nlm->det->read_id);
      cpl_vector_set(nlm->channelgain, c, mat_propertylist_get_double_default(nlm->keywords, kwd, 1.0));
      snprintf(kwd, 64, "ESO QC DET%d CHANNEL%d OFFSET%d", nlm->det->nr, c + 1, nlm->det->read_id);
      cpl_vector_set(nlm->channeloffset, c, mat_propertylist_get_double_default(nlm->keywords, kwd, 0.0));
      snprintf(kwd, 64, "ESO QC DET%d CHANNEL%d RON%d", nlm->det->nr, c + 1, nlm->det->read_id);
      cpl_vector_set(nlm->channelnoise, c, mat_propertylist_get_double_default(nlm->keywords, kwd, 0.0));
    }
  // replace the coefficients in the data structure by the FITS file content
  if (limit != NULL)
    {
      cpl_image_delete(nlm->list_limit[0]);
      nlm->list_limit[0] = limit;
    }
  if (coeff != NULL)
    { // the extension does exist and the image list could be loaded -> has_coeff := 1
      nlm->has_coeff = 1;
      cpl_imagelist_delete(nlm->list_coeff[0]);
      nlm->list_coeff[0] = coeff;
    }
  if (stdev != NULL)
    { // the extension does exist and the image could be loaded -> has_stdev := 1
      nlm->has_stdev = 1;
      cpl_image_delete(nlm->list_stdev[0]);
      nlm->list_stdev[0] = stdev;
    }
  if (rcmap != NULL)
    { // the extension does exist and the vector could be loaded -> has_rcmap := 1
      nlm->has_rcmap = 1;
      cpl_vector_delete(nlm->rcmap);
      nlm->rcmap = rcmap;
    }
  if (hist != NULL)
    { // the extension does exist and the image could be loaded -> has_hist := 1
      nlm->has_hist = 1;
      cpl_image_delete(nlm->hist);
      nlm->hist = hist;
    }
  nlm->nltype = MAT_UNKNOWN_NL;
  if (nlm->has_coeff)
    { // coefficients do exist -> check if they are linear for pixel[1,1]
      int    rejected;
      double a0, a1, a2, a3;
      if (nlm->nbcoeff >= 1)
	a0 = cpl_image_get(cpl_imagelist_get(nlm->list_coeff[0], 0), 1, 1, &rejected);
      else
	a0 = 1.0;
      if (nlm->nbcoeff >= 2)
	a1 = cpl_image_get(cpl_imagelist_get(nlm->list_coeff[0], 1), 1, 1, &rejected);
      else
	a1 = 0.0;
      if (nlm->nbcoeff >= 3)
	a2 = cpl_image_get(cpl_imagelist_get(nlm->list_coeff[0], 2), 1, 1, &rejected);
      else
	a2 = 0.0;
      if (nlm->nbcoeff >= 4)
	a3 = cpl_image_get(cpl_imagelist_get(nlm->list_coeff[0], 3), 1, 1, &rejected);
      else
	a3 = 0.0;
      if ((a0 == 1.0) && (a1 == 0.0) && (a2 == 0.0) && (a3 == 0.0))
	{
	  nlm->nltype = MAT_INDIVIDUAL_NL;
	  nlm->rclimit = 65536.0;
	}
      else if (nlm->has_nllaw)
	{
	  nlm->nltype = MAT_GLOBAL_LAW_NL;
	  nlm->rclimit = cpl_image_get(limit, 1, 1, &rejected);
	}
      else if (nlm->has_rcmap)
	{
	  nlm->nltype = MAT_GLOBAL_MAP_NL;
	  nlm->rclimit = cpl_image_get(limit, 1, 1, &rejected);
	}
    }
  else if (nlm->has_nllaw)
    {
      int    rejected;
      nlm->nltype = MAT_GLOBAL_LAW_NL;
      nlm->rclimit = cpl_image_get(limit, 1, 1, &rejected);
    }
  else if (nlm->has_rcmap)
    {
      int    rejected;
      nlm->nltype = MAT_GLOBAL_MAP_NL;
      nlm->rclimit = cpl_image_get(limit, 1, 1, &rejected);
    }
  if (nlm->nltype == MAT_UNKNOWN_NL)
    {
      cpl_msg_warning(cpl_func, "unknown nonlinearity type");
    }
  return nlm;
}

/**
   @ingroup nlm
   @brief This function maps a nonlinearity map to a given sub-window layout.
   @param nlm    This nonlinearity map covers the whole detector.
   @param imgdet This structure describes the sub-window layout.
   @returns The mapped nonlinearity map or NULL.

   This function uses the sub-window layout of the imgdet argument to create
   a new nonlinearity map (using the function mat_nonlinearity_new and the layout in imgdet).
   In a second step, the sub-window specific parts of the nonlinearity mask are extracted
   and stored in the newly created nonlinearity map.
 */
mat_nonlinearity *mat_nonlinearity_map(mat_nonlinearity *nlm, mat_imagingdetector *imgdet)
{
  mat_nonlinearity *nnlm = NULL;
  int           r, c, i;

  mat_assert_value((nlm != NULL), CPL_ERROR_NULL_INPUT, NULL, "no mat_nonlinearity (nlm) argument given");
  mat_assert_value((imgdet != NULL), CPL_ERROR_NULL_INPUT, NULL, "no mat_imagingdetector (imgdet) argument given");
  mat_assert_value((imgdet->list_region != NULL), CPL_ERROR_NULL_INPUT, NULL, "no valid mat_imagingdetector (imgdet->list_region) argument given");
  nnlm = mat_nonlinearity_new(nlm->det, imgdet, nlm->nbcoeff);
  if (nnlm == NULL)
    {
      cpl_msg_error(cpl_func, "could not allocate memory for mat_nonlinearity");
      return NULL;
    }
  /* put a copy of the propertylist into the newly created nonlinearity map */
  if (nnlm->keywords != NULL)
    {
      cpl_propertylist_delete(nnlm->keywords);
      nnlm->keywords = NULL;
      nnlm->keywords = cpl_propertylist_duplicate(nlm->keywords);
    }
  /* copy the QC parameters into the new nonlinearity map */
  nnlm->detectorgain          = nlm->detectorgain;
  nnlm->detectornoise         = nlm->detectornoise;
  nnlm->horizontalcorrelation = nlm->horizontalcorrelation;
  nnlm->verticalcorrelation   = nlm->verticalcorrelation;
  nnlm->flatfieldstdev        = nlm->flatfieldstdev;
  strncpy(nnlm->detnllaw, nlm->detnllaw, sizeof(nnlm->detnllaw));
  nnlm->detnllaw[sizeof(nnlm->detnllaw)-1] = '\0';
  nnlm->detnlncoeff           = nlm->detnlncoeff;
  for (i = 0; i < MAT_NONLIN_MAX_NCOEFF; i++) {
    nnlm->detnlcoeff[i]       = nlm->detnlcoeff[i];
  }
  cpl_vector_copy(nnlm->channelgain, nlm->channelgain);
  cpl_vector_copy(nnlm->channeloffset, nlm->channeloffset);
  cpl_vector_copy(nnlm->channelnoise, nlm->channelnoise);
  /* extract only the sub-window equivalents of the nonlinearity map */
  for (r = 0; r < nnlm->imgdet->nbregion; r++)
    {
      if (nnlm->list_limit[r] != NULL)
	{
	  cpl_image_delete(nnlm->list_limit[r]);
	  nnlm->list_limit[r] = NULL;
	}
      nnlm->list_limit[r] = cpl_image_extract(nlm->list_limit[0],
					      imgdet->list_region[r]->corner[0],
					      imgdet->list_region[r]->corner[1],
					      imgdet->list_region[r]->corner[0] + imgdet->list_region[r]->naxis[0] - 1,
					      imgdet->list_region[r]->corner[1] + imgdet->list_region[r]->naxis[1] - 1);
    }
  nnlm->has_coeff = nlm->has_coeff;
  if (nlm->has_coeff)
    {
      for (r = 0; r < nnlm->imgdet->nbregion; r++)
	{
	  if (nnlm->list_coeff[r] != NULL)
	    {
	      cpl_imagelist_delete(nnlm->list_coeff[r]);
	      nnlm->list_coeff[r] = NULL;
	    }
	  nnlm->list_coeff[r] = cpl_imagelist_new();
	  if (nnlm->list_coeff[r] == NULL)
	    {
	      cpl_msg_error(cpl_func, "could not create a cpl_imagelist (nonlinearity region)");
	      mat_nonlinearity_delete(nnlm);
	      return NULL;
	    }
	  for (c = 0; c < nnlm->nbcoeff; c++)
	    {
	      cpl_imagelist_set(nnlm->list_coeff[r],
				cpl_image_extract(cpl_imagelist_get(nlm->list_coeff[0], c),
						  imgdet->list_region[r]->corner[0],
						  imgdet->list_region[r]->corner[1],
						  imgdet->list_region[r]->corner[0] + imgdet->list_region[r]->naxis[0] - 1,
						  imgdet->list_region[r]->corner[1] + imgdet->list_region[r]->naxis[1] - 1),
				c);
	    }
	}
    }
  nnlm->has_stdev = nlm->has_stdev;
  if (nlm->has_stdev)
    {
      for (r = 0; r < nnlm->imgdet->nbregion; r++)
	{
	  if (nnlm->list_stdev[r] != NULL)
	    {
	      cpl_image_delete(nnlm->list_stdev[r]);
	      nnlm->list_stdev[r] = NULL;
	    }
	  nnlm->list_stdev[r] = cpl_image_extract(nlm->list_stdev[0],
						  imgdet->list_region[r]->corner[0],
						  imgdet->list_region[r]->corner[1],
						  imgdet->list_region[r]->corner[0] + imgdet->list_region[r]->naxis[0] - 1,
						  imgdet->list_region[r]->corner[1] + imgdet->list_region[r]->naxis[1] - 1);
	}
    }
  nnlm->has_rcmap = nlm->has_rcmap;
  nnlm->rclimit   = nlm->rclimit;
  cpl_vector_copy(nnlm->rcmap, nlm->rcmap);
  nnlm->has_hist = nlm->has_hist;
  cpl_image_copy(nnlm->hist, nlm->hist, 1, 1);
  nnlm->nltype = nlm->nltype;
  return nnlm;
}
