/* $Id: mat_nonlinearity.h,v0.5 2014-06-15 12:56:21 mheininger Exp $
 *
 * This file is part of the ESO Matisse pipeline
 * Copyright (C) 2012-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: mheininger $
 * $Date: 2012/06/26 16:52:00 $
 * $Revision: 0.5 $
 * $Name: mat_nonlinearity.h $
 */

#ifndef MAT_NONLINEARITY_H
#define MAT_NONLINEARITY_H

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include <stdio.h>
#include <cpl.h>
#include "mat_detector.h"
#include "mat_imagingdetector.h"
/*-----------------------------------------------------------------------------
                                   Define
 -----------------------------------------------------------------------------*/

#define MAT_NONLIN_MAX_NCOEFF    8

#define MAT_UNKNOWN_NL           0
#define MAT_INDIVIDUAL_NL        1
#define MAT_GLOBAL_MAP_NL        2
#define MAT_GLOBAL_LAW_NL        3

/**
   @ingroup nlm
   @brief Structure containing the static nonlinearity map of a detector.

   This structure represents all contents in a NONLINEARITY FITS file. This includes
   the primary header keywords (keywords member), the sub-window setup (imgdet member),
   the conversion factor (detectorgain and channelgain members), the readout noise
   (detectornoise and channelnoise members) and the nonlinearity map itself.

   After loading a nonlinearity map from file, the map covers the whole detector. This
   map can be mapped to a certain sub-window setup where each sub-window is
   covered by an individual nonlinearity map as an imagelist in the list_coeff member.
 */
typedef struct {
  mat_detector         *det;            /*!< The detector specification for this nonlinearity map. */
  cpl_propertylist     *keywords;       /*!< The keywords from the primary header. */
  mat_imagingdetector  *imgdet;         /*!< The contents of the IMAGING_DETECTOR binary table. */
  double                detectorgain;   /*!< The detector specific gain in electrons per ADU (QC1 parameter QC DETi GAIN). */
  double                detectornoise;  /*!< The detector specific readout noise in eectrons (QC1 parameter QC DETi RON). */
  double                horizontalcorrelation; /*!< The autocorrelation coefficient between two horizontally adjacent pixels. This value will be stored as a QC1 parameter (QC DETi HACORR). */
  double                verticalcorrelation;   /*!< The autocorrelation coefficient between two vertically adjacent pixels. This value will be stored as a QC1 parameter (QC DETi VACORR). */
  double                flatfieldstdev; /*!< The standard deviation of the computed flatfield map (QC1 parameter QC DETi FFM STDEV). */
  int                   has_nllaw;      /*!< The nonlinearity map contains an explicite function for the whole detector */
  char                  detnllaw[64];   /*!< Nonlinearity function used, H2: "a*x+b*x*exp((x-c)*d)", AQ: "a*x+b*(1-exp(-c*x**d))+e*(exp(f*x)-1)". */
  int                   detnlncoeff;    /*!< Number of coefficients in nonlinearity law. */
  double                detnlcoeff[MAT_NONLIN_MAX_NCOEFF];  /*!< Nonlinearity coefficients for the whole detector, a, , c, d */
  cpl_vector           *channelgain;    /*!< The detector channel specific gain in electrons per ADU (QC1 parameter QC DETi CHANNELj GAIN). */
  cpl_vector           *channeloffset;  /*!< The detector channel specific offset in ADU (QC1 parameter QC DETi CHANNELj OFFSET). */
  cpl_vector           *channelnoise;   /*!< The detector channel specific readout noise in electrons (QC1 parameter QC DETi CHANNELj RON). */
  cpl_image           **list_limit;     /*!< Vector containing the intensity limit of the nonlinearity map for each region. Dimensions given by imgdet->nbregion.*/
  int                   has_coeff;      /*!< Flag if the loaded nonlinearity map has valid function coefficients. */
  int                   nbcoeff;        /*!< The number of function coefficients. */
  cpl_imagelist       **list_coeff;     /*!< Imagelist vector containing the nonlinearity function coefficients for each region and pixel. Dimension of the vector given by imgdet->nbregion, dimension of the image list given by nbcoeff. */
  int                   has_stdev;      /*!< Flag, if the loaded nonlinearity map has a valid stdev map. */
  cpl_image           **list_stdev;     /*!< Vector containing the standard error of the nonlinearity map for each region. Dimensions given by imgdet->nbregion.*/
  int                   has_rcmap;      /*!< Flag, if the loaded nonlinearity map has a valid raw -> cal map. */
  double                rclimit;        /*!< Global nonlinearity compensation limit (65536.0 if individual behaviour is used). */
  cpl_vector           *rcmap;          /*!< Vector representing the global raw -> calibrate intensity mapping (1025 values). */
  int                   has_hist;       /*!< Flag, is the loaded nonlinearity map has a valid 2-d nonlinearity histogram. */
  cpl_image            *hist;           /*!< 2-d nonlinearity histogram, depends on method DIT or flux. */
  int                   nltype       ;  /*!< enumeration (MAT_*_NL) for nonlinearity type */
} mat_nonlinearity;

/*-----------------------------------------------------------------------------
                                   Functions prototypes
 -----------------------------------------------------------------------------*/

mat_nonlinearity *mat_nonlinearity_new(mat_detector *det,
				       mat_imagingdetector *imgdet,
				       int nbcoeff);

cpl_error_code mat_nonlinearity_delete(mat_nonlinearity *nlm);

cpl_error_code mat_nonlinearity_save(mat_nonlinearity *nlm,
				     const char *fname,
				     const char *rname,
				     cpl_parameterlist *parlist,
				     cpl_frameset *frameset);

mat_nonlinearity *mat_nonlinearity_load(cpl_frame *nlmframe);

mat_nonlinearity *mat_nonlinearity_map(mat_nonlinearity *nlm, mat_imagingdetector *imgdet);


#endif
